﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Linq;
using App.ConfigData;
using App.PropertyEdit;
using App.Utility;
using ConfigCommon;
using nw.g3d.iflib;
using nw.g3d.nw4f_3dif;

namespace App.Data
{
    public class SkeletalAnimation : AnimationDocument, IHasUserData
    {
        public skeletal_animType Data{ get; private set; }
        public override object nw4f_3difItem
        {
            get { return Data; }
        }

        public override int FrameCount
        {
            get{ return Data.skeletal_anim_info.frame_count; }
            set{ Data.skeletal_anim_info.frame_count = value; }
        }

        public override bool Loop
        {
            get { return Data.skeletal_anim_info.loop; }
            set { Data.skeletal_anim_info.loop = value; }
        }

        public override bool IsFrameCountModified
        {
            get { return savedData.frame_count != Data.skeletal_anim_info.frame_count; }
        }

        public override bool IsLoopModified
        {
            get { return savedData.loop != Data.skeletal_anim_info.loop; }
        }

        public override bool HasAnimation
        {
            get
            {
                return BoneAnims != null &&
                       BoneAnims.Where(x => x.BoneAnimTargets != null)
                                .Any(x => x.BoneAnimTargets.Any());
            }
        }

        public UserDataArray UserDataArray { get; set; }
        public UserDataArray SavedUserDataArray { get; set; }
        public bool UserDataArrayChanged { get; set; }

        public override process_log_arrayType process_log_array
        {
            get { return Data.process_log_array; }
        }

        [Serializable]
        public class BoneAnim
        {
            public BoneAnim()
            {
                //QuantizationInfo = new QuantizationInfo();
                bone_name = string.Empty;
                parent_name		= string.Empty;
                matrix_index	= new[]{-1, -1};
                BoneAnimTargets	= new List<BoneAnimTarget>();
            }

            public string	bone_name			{ get; set; }
            public string	parent_name			{ get; set; }
            public bool		rigid_body			{ get; set; }
            public int[]	matrix_index		{ get; set; }
            public bool		scale_compensate	{ get; set; }
            public bool		binarize_scale		{ get; set; }
            public bool		binarize_rotate		{ get; set; }
            public bool		binarize_translate	{ get; set; }
            public bool		compress_enable		{ get; set; }

            public List<BoneAnimTarget>	BoneAnimTargets{ get; set; }

            public bool IsSame(BoneAnim anim)
            {
                //var anim = animTarget as BoneAnim;
                return
                    bone_name == anim.bone_name &&
                    parent_name == anim.parent_name &&
                    rigid_body == anim.rigid_body &&
                    matrix_index.SequenceEqual(anim.matrix_index) &&
                    scale_compensate == anim.scale_compensate &&
                    binarize_scale == anim.binarize_scale &&
                    binarize_rotate == anim.binarize_rotate &&
                    binarize_translate == anim.binarize_translate &&
                    compress_enable == anim.compress_enable &&
                    BoneAnimTargets.Count == anim.BoneAnimTargets.Count &&
                    BoneAnimTargets.Zip(anim.BoneAnimTargets, (x, y) => x.IsSame(y)).All(x => x);
            }

            public AnimTarget GetTarget(bone_anim_target_targetType targetType)
            {
                return BoneAnimTargets.FirstOrDefault(x => x.target == targetType);
            }
        }

        // BoneAnimTarget
        //TODO: AnimTarget を継承
        [Serializable]
        public class BoneAnimTarget : AnimTarget
        {
            public BoneAnimTarget()
            {
                KeyFrames = new List<KeyFrame>();
            }

            public bone_anim_target_targetType	target		{ get; set; }
            public float						base_value	{ get; set; } // 編集しないのでそのまま残しておく

            //public curve_frame_typeType			frame_type		{ get; set; }			// [hermite|linear|step]_curveType.frame_type;
            //public curve_key_typeType			key_type		{ get; set; }			// [hermite|linear|step]_curveType.key_type;
            //public float						scale			{ get; set; }			// [hermite|linear|step]_curveType.scale
            //public float						offset			{ get; set; }			// [hermite|linear|step]_curveType.offset

            //public curve_wrapType				pre_wrap		{ get; set; }			// [hermite|linear|step]_curveType.pre_wrap;
            //public curve_wrapType				post_wrap		{ get; set; }			// [hermite|linear|step]_curveType.post_wrap;
            //public bool				 			baked			{ get; set; }			// [hermite|linear|step]_curveType.baked;

            //public InterpolationType			CurveInterpolationType { get; set; }
            //public List<KeyFrame>				KeyFrames{	get; set; }

            public bool	IsScale		{ get { return (target == bone_anim_target_targetType.scale_x) ||
                                                   (target == bone_anim_target_targetType.scale_y) ||
                                                   (target == bone_anim_target_targetType.scale_z);			} }

            public bool	IsRotate	{ get { return (target == bone_anim_target_targetType.rotate_x) ||
                                                   (target == bone_anim_target_targetType.rotate_y) ||
                                                   (target == bone_anim_target_targetType.rotate_z) ||
                                                   (target == bone_anim_target_targetType.rotate_z);		} }

            public bool	IsTranslate	{ get { return (target == bone_anim_target_targetType.translate_x) ||
                                                   (target == bone_anim_target_targetType.translate_y) ||
                                                   (target == bone_anim_target_targetType.translate_z);		} }

        }
        public List<BoneAnim>	BoneAnims{ get; set; }

        #region 圧縮率
        public bool IsAllFrameCurveConstant     { get { return IsAllScaleCurveConstant && IsAllRotateCurveConstant && IsAllTranslateCurveConstant; } }
        public bool IsAllScaleCurveConstant     { get { return BoneAnims.SelectMany(x => x.BoneAnimTargets).Any(x => x.IsScale     && x.KeyFrames.Any()) == false; } }
        public bool IsAllRotateCurveConstant    { get { return BoneAnims.SelectMany(x => x.BoneAnimTargets).Any(x => x.IsRotate    && x.KeyFrames.Any()) == false; } }
        public bool IsAllTranslateCurveConstant { get { return BoneAnims.SelectMany(x => x.BoneAnimTargets).Any(x => x.IsTranslate && x.KeyFrames.Any()) == false; } }

        public int FrameCompressSize           { get { return GetFrameDataSize(false);     } }
        public int FrameUncompressSize         { get { return GetFrameDataSize(true);      } }

        public int ScaleCompressDataSize       { get { return GetScaleDataSize(false);     } }
        public int ScaleUncompressDataSize     { get { return GetScaleDataSize(true);      } }

        public int RotateCompressDataSize      { get { return GetRotateDataSize(false);    } }
        public int RotateUncompressDataSize    { get { return GetRotateDataSize(true);     } }

        public int TranslateCompressDataSize   { get { return GetTranslateDataSize(false); } }
        public int TranslateUncompressDataSize { get { return GetTranslateDataSize(true);  } }
        #endregion

        // 編集コマンド用
        [Serializable]
        public class QuantizeToleranceData
        {
            public List<BoneAnim>		BoneAnims{ get; set; }

            public float	QuantizeToleranceScale		{ get; set; }
            public float	QuantizeToleranceRotate		{ get; set; }
            public float	QuantizeToleranceTranslate	{ get; set; }
        }

        public SkeletalAnimation(skeletal_animType animation, List<G3dStream> streams) : base(GuiObjectID.SkeletalAnimation, streams)
        {
            Data = animation;
            Initialize();
            MotionMirroring = false;
        }

        public void Initialize()
        {
            ToolData.Load(Data.tool_data);
            MakeComment(Data.comment);

            // ユーザーデータ作成
            this.MakeUserData(Data.user_data_array, BinaryStreams);

            MakeBoneAnims();
        }

        #region Bridge
        public class SkeletalAnimSwapData : SwapData
        {
            public ToolData ToolData;
            public List<G3dStream> streams;
            public string Comment;
            public RgbaColor? EditColor;
            public skeletal_animType Data;
            public List<BoneAnim> BoneAnims;
            public UserDataArray userData;
        }

        public override SwapData CreateSwapData(object data, List<G3dStream> streams)
        {
            var current = GetSwapData();
            Data = (skeletal_animType)data;
            BinaryStreams = streams;
            Initialize();

            return Swap(current, false);
        }

        public override SwapData GetSwapData()
        {
            return new SkeletalAnimSwapData()
            {
                Data = Data,
                streams = BinaryStreams,
                Comment = Comment,
                EditColor = EditColor,
                BoneAnims = BoneAnims,
                userData = UserDataArray,
            };
        }

        public override SwapData Swap(SwapData swapData, bool copyHistory)
        {
            var current = (SkeletalAnimSwapData)GetSwapData();
            var data = (SkeletalAnimSwapData)swapData;
            Data = data.Data;
            BinaryStreams = data.streams;
            Comment = data.Comment;
            EditColor = data.EditColor;
            BoneAnims = data.BoneAnims;
            UserDataArray = data.userData;

            if (copyHistory)
            {
                UserDataArrayChanged = !SavedUserDataArray.IsSame(UserDataArray);
            }

            return current;
        }
        #endregion

        public override nw4f_3difType Create_nw4f_3difType(bool viewer)
        {
            lock (this)
            {
                UpdateData();

                //			nw4f_3difType nw4f_3dif = new nw4f_3difType();
                nw4f_3dif_.Item = Data;
                nw4f_3dif_.file_info = null;

                return nw4f_3dif_;
            }
        }

        public override IEnumerable<Tuple<AnimTarget, IfQuantizationAnalysisResult>> QuantizeAnalyse()
        {
            // 実装しない
            throw new NotImplementedException();
        }

        public void Optimize()
        {
            // ストリームを更新する。
            UpdateData();

            // <hermite_curve>、<linear_curve>、<step_curve> の frame_type と key_type を
            // none に設定しないと、再分析が行われません。
            if (Data.bone_anim_array != null)
            {
                foreach (var anim in Data.bone_anim_array.bone_anim)
                {
                    foreach (var target in anim.bone_anim_target)
                    {
                        if (target.Curve != null)
                        {
                            target.Curve.frame_type = curve_frame_typeType.none;
                            target.Curve.key_type = curve_key_typeType.none;
                        }
                    }
                }
            }

            try
            {
                // アニメーションに合わせたAnimQuantizationAnalysisOptimizerを呼び出す。
                var optimizer = new IfSkeletalAnimQuantizationAnalysisOptimizer();
                if (ApplicationConfig.DefaultValue.DisableAnimationQuantize)
                {
                    var animinfo = Data.skeletal_anim_info;
                    var tolS = animinfo.quantize_tolerance_scale;
                    var tolR = animinfo.quantize_tolerance_rotate;
                    var tolT = animinfo.quantize_tolerance_translate;
                    animinfo.quantize_tolerance_scale = 0.0f;
                    animinfo.quantize_tolerance_rotate = 0.0f;
                    animinfo.quantize_tolerance_translate = 0.0f;

                    optimizer.Optimize(Data, BinaryStreams);

                    animinfo.quantize_tolerance_scale = tolS;
                    animinfo.quantize_tolerance_rotate = tolR;
                    animinfo.quantize_tolerance_translate = tolT;
                }
                else
                {
                    optimizer.Optimize(Data, BinaryStreams);
                }
            }
            catch(Exception e)
            {
                DebugConsole.WriteLine("Exception Catch : IfSkeletalAnimQuantizationAnalysisOptimizer.Optimize() : {0}", e.Message);
            }

            // 最適化後に、データを再作成する。
            this.MakeUserData(Data.user_data_array, BinaryStreams);
            MakeBoneAnims();
        }

        private void MakeBoneAnims()
        {
            BoneAnims = new List<BoneAnim>();

            if ((Data.bone_anim_array == null) ||
                (Data.bone_anim_array.bone_anim == null))
            {
                return;
            }

            foreach(var boneAnim in Data.bone_anim_array.bone_anim)
            {
                var newBoneAnim = new BoneAnim()
                {
                    bone_name			= ObjectUtility.Clone(boneAnim.bone_name),
                    parent_name			= ObjectUtility.Clone(boneAnim.parent_name),
                    rigid_body			= boneAnim.rigid_body,
                    matrix_index		= ObjectUtility.Clone(boneAnim.matrix_index),
                    scale_compensate	= boneAnim.scale_compensate,
                    binarize_scale		= boneAnim.binarize_scale,
                    binarize_rotate		= boneAnim.binarize_rotate,
                    binarize_translate	= boneAnim.binarize_translate,
                    compress_enable		= boneAnim.compress_enable
                };

                if (boneAnim.bone_anim_target != null)
                {
                    foreach(var boneAnimTarget in boneAnim.bone_anim_target)
                    {
                        var newBoneAnimTarget =  new BoneAnimTarget()
                        {
                            target					= boneAnimTarget.target,
                            base_value				= boneAnimTarget.base_value,
                            CurveInterpolationType	= (boneAnimTarget.Item is hermite_curveType) ? InterpolationType.Hermite :
                                                      (boneAnimTarget.Item is linear_curveType)  ? InterpolationType.Linear :
                                                      (boneAnimTarget.Item is step_curveType)	? InterpolationType.Step :
                                                                                                   InterpolationType.Linear		// null の時
                        };

                        if (boneAnimTarget.Item != null)
                        {
                            switch(newBoneAnimTarget.CurveInterpolationType)
                            {
                                case InterpolationType.Hermite:
                                {
                                    const int dataElementCount = 4;		// frame, value, in-slope, out-slope

                                    var curve = boneAnimTarget.Item as hermite_curveType;
                                    Debug.Assert(curve != null);

                                    newBoneAnimTarget.QuantizationInfo = new QuantizationInfo()
                                    {
                                        frame_type = curve.frame_type,
                                        key_type = curve.key_type,
                                        scale = curve.scale,
                                        offset = curve.offset,
                                    };

                                    newBoneAnimTarget.pre_wrap		= curve.pre_wrap;
                                    newBoneAnimTarget.post_wrap		= curve.post_wrap;
                                    newBoneAnimTarget.baked			= curve.baked;

                                    var stream = BinaryStreams[curve.stream_index];
                                    var data = stream.FloatData;
                                    Debug.Assert(data != null);
                                    Debug.Assert(data.Count == curve.count * dataElementCount);

                                    for(int i = 0;i != curve.count;++ i)
                                    {
                                        newBoneAnimTarget.KeyFrames.Add(
                                            new KeyFrame()
                                            {
                                                Frame	= data[i * dataElementCount + 0],
                                                Value	= data[i * dataElementCount + 1],
                                                InSlope  = data[i * dataElementCount + 2],
                                                OutSlope = data[i * dataElementCount + 3]
                                            }
                                        );
                                    }

                                    break;
                                }

                                case InterpolationType.Linear:
                                case InterpolationType.Step:
                                {
                                    const int dataElementCount = 2;		// frame, value

                                    int stream_index = -1;
                                    int count		 = -1;
                                    {
                                        if (newBoneAnimTarget.CurveInterpolationType == InterpolationType.Linear)
                                        {
                                            var curve = boneAnimTarget.Item as linear_curveType;
                                            Debug.Assert(curve != null);

                                            stream_index = curve.stream_index;
                                            count		 = curve.count;

                                            newBoneAnimTarget.QuantizationInfo = new QuantizationInfo()
                                            {
                                                frame_type = curve.frame_type,
                                                key_type = curve.key_type,
                                                scale = curve.scale,
                                                offset = curve.offset,
                                            };
                                            newBoneAnimTarget.pre_wrap		= curve.pre_wrap;
                                            newBoneAnimTarget.post_wrap		= curve.post_wrap;
                                            newBoneAnimTarget.baked			= curve.baked;
                                        }
                                        else
                                        if (newBoneAnimTarget.CurveInterpolationType == InterpolationType.Step)
                                        {
                                            var curve = boneAnimTarget.Item as step_curveType;
                                            Debug.Assert(curve != null);

                                            stream_index = curve.stream_index;
                                            count		 = curve.count;

                                            newBoneAnimTarget.QuantizationInfo = new QuantizationInfo()
                                            {
                                                frame_type = curve.frame_type,
                                                key_type = curve.key_type,
                                                scale = curve.scale,
                                                offset = curve.offset,
                                            };
                                            newBoneAnimTarget.pre_wrap		= curve.pre_wrap;
                                            newBoneAnimTarget.post_wrap		= curve.post_wrap;
                                            newBoneAnimTarget.baked			= curve.baked;
                                        }

                                        Debug.Assert(stream_index != -1);
                                        Debug.Assert(count != -1);
                                    }

                                    var stream = BinaryStreams[stream_index];
                                    var data = stream.FloatData;
                                    Debug.Assert(data != null);
                                    Debug.Assert(data.Count == count * dataElementCount);

                                    for(int i = 0;i != count;++ i)
                                    {
                                        newBoneAnimTarget.KeyFrames.Add(
                                            new KeyFrame()
                                            {
                                                Frame	= data[i * dataElementCount + 0],
                                                Value	= data[i * dataElementCount + 1]
                                            }
                                        );
                                    }

                                    break;
                                }
                            }
                        }

                        newBoneAnim.BoneAnimTargets.Add(newBoneAnimTarget);
                    }
                }

                BoneAnims.Add(newBoneAnim);
            }
        }

        private int GetFrameDataSize(bool isBase)
        {
            int size = 0;
            {
                foreach (var boneAnims in BoneAnims)
                {
                    foreach (var target in boneAnims.BoneAnimTargets)
                    {
                        size += target.KeyFrames.Count * (isBase ? 4 : IfUtility.GetByteSizeFromCurveFrameType(target.QuantizationInfo.frame_type));
                    }
                }
            }
            return size;
        }

        private int GetScaleDataSize(bool isBase)
        {
            return GetDataSize(isBase)[0];
        }

        private int GetRotateDataSize(bool isBase)
        {
            return GetDataSize(isBase)[1];
        }

        private int GetTranslateDataSize(bool isBase)
        {
            return GetDataSize(isBase)[2];
        }

        // 0:Scale
        // 1:Rotate
        // 2:Translate
        private int[] GetDataSize(bool isBase)
        {
            var size = new int[]{0, 0, 0};
            {
                foreach (var boneAnims in BoneAnims)
                {
                    foreach (var target in boneAnims.BoneAnimTargets)
                    {
                        int kind = 0;
                        {
                                 if (target.IsScale){		kind = 0;				}
                            else if (target.IsRotate){		kind = 1;				}
                            else if (target.IsTranslate){	kind = 2;				}
                            else{							Debug.Assert(false);	}
                        }

                        size[kind] += target.KeyFrames.Count * (isBase ? 4 : IfUtility.GetByteSizeFromCurveKeyType(target.QuantizationInfo.key_type));
                    }
                }
            }
            return size;
        }

        private void UpdateData()
        {
            ToolData.Load(Data.tool_data);
            Data.comment = GetComment();

            if (Data.bone_anim_array != null)
            {
                Data.bone_anim_array = new bone_anim_arrayType();
            }

            var tempBoneAnims = new List<bone_animType>();
            var tempStreams = new List<G3dStream>();
            int targetIndex = 0;    // stream_indexとして使用します。
            foreach (var BoneAnim in BoneAnims)
            {
                // データ
                {
                    var boneAnim = new bone_animType()
                    {
                        bone_name = BoneAnim.bone_name,
                        parent_name = BoneAnim.parent_name,
                        rigid_body = BoneAnim.rigid_body,
                        matrix_index = ObjectUtility.Clone(BoneAnim.matrix_index),
                        scale_compensate = BoneAnim.scale_compensate,
                        binarize_scale = BoneAnim.binarize_scale,
                        binarize_rotate = BoneAnim.binarize_rotate,
                        binarize_translate = BoneAnim.binarize_translate,
                        compress_enable = BoneAnim.compress_enable,
                    };

                    var tempBoneAnimTargets = new List<bone_anim_targetType>();
                    foreach (var BoneAnimTarget in BoneAnim.BoneAnimTargets)
                    {
                        var boneAnimtarget = new bone_anim_targetType()
                        {
                            target = BoneAnimTarget.target,
                            base_value = BoneAnimTarget.base_value, // 編集可能にする場合はカーブから求める必要がある
                        };

                        switch (BoneAnimTarget.CurveInterpolationType)
                        {
                            case InterpolationType.Hermite:
                                {
                                    boneAnimtarget.Item = new hermite_curveType()
                                    {
                                        count = BoneAnimTarget.KeyFrames.Count,
                                        frame_type = BoneAnimTarget.QuantizationInfo.frame_type,
                                        key_type = BoneAnimTarget.QuantizationInfo.key_type,
                                        scale = BoneAnimTarget.QuantizationInfo.scale,
                                        offset = BoneAnimTarget.QuantizationInfo.offset,
                                        pre_wrap = BoneAnimTarget.pre_wrap,
                                        post_wrap = BoneAnimTarget.post_wrap,
                                        baked = BoneAnimTarget.baked,
                                        stream_index = targetIndex,
                                    };
                                    break;
                                }
                            case InterpolationType.Linear:
                                {
                                    boneAnimtarget.Item = new linear_curveType()
                                    {
                                        count = BoneAnimTarget.KeyFrames.Count,
                                        frame_type = BoneAnimTarget.QuantizationInfo.frame_type,
                                        key_type = BoneAnimTarget.QuantizationInfo.key_type,
                                        scale = BoneAnimTarget.QuantizationInfo.scale,
                                        offset = BoneAnimTarget.QuantizationInfo.offset,
                                        pre_wrap = BoneAnimTarget.pre_wrap,
                                        post_wrap = BoneAnimTarget.post_wrap,
                                        baked = BoneAnimTarget.baked,
                                        stream_index = targetIndex,
                                    };
                                    break;
                                }
                            case InterpolationType.Step:
                                {
                                    boneAnimtarget.Item = new step_curveType()
                                    {
                                        count = BoneAnimTarget.KeyFrames.Count,
                                        frame_type = BoneAnimTarget.QuantizationInfo.frame_type,
                                        key_type = BoneAnimTarget.QuantizationInfo.key_type,
                                        scale = BoneAnimTarget.QuantizationInfo.scale,
                                        offset = BoneAnimTarget.QuantizationInfo.offset,
                                        pre_wrap = BoneAnimTarget.pre_wrap,
                                        post_wrap = BoneAnimTarget.post_wrap,
                                        baked = BoneAnimTarget.baked,
                                        stream_index = targetIndex,
                                    };
                                    break;
                                }
                        }

                        tempBoneAnimTargets.Add(boneAnimtarget);

                        if (BoneAnimTarget.KeyFrames.Any())
                        {
                            // コンスタントカーブかどうかをチェックする
                            {
                                float base_value = 0.0f;
                                if (BoneAnimTarget.KeyFrames.IsConstantCurve(out base_value))
                                {
                                    // コンスタントカーブなのでベース値を保存し、カーブを削除する
                                    boneAnimtarget.Item = null;
                                    boneAnimtarget.base_value = base_value;
                                    continue;
                                }
                            }

                            var tempStream = new G3dStream();
                            switch (BoneAnimTarget.CurveInterpolationType)
                            {
                                case InterpolationType.Hermite:
                                    {
                                        tempStream.type = stream_typeType.@float;
                                        tempStream.column = 4;
                                        foreach (var keyFrame in BoneAnimTarget.KeyFrames)
                                        {
                                            tempStream.FloatData.Add(keyFrame.Frame);
                                            tempStream.FloatData.Add(keyFrame.Value);
                                            tempStream.FloatData.Add(keyFrame.InSlope);
                                            tempStream.FloatData.Add(keyFrame.OutSlope);
                                        }
                                        break;
                                    }
                                case InterpolationType.Linear:
                                case InterpolationType.Step:
                                    {
                                        tempStream.type = stream_typeType.@float;
                                        tempStream.column = 2;
                                        foreach (var keyFrame in BoneAnimTarget.KeyFrames)
                                        {
                                            tempStream.FloatData.Add(keyFrame.Frame);
                                            tempStream.FloatData.Add(keyFrame.Value);
                                        }
                                        break;
                                    }
                            }
                            tempStreams.Add(tempStream);
                            targetIndex++;
                        }
                        else
                        {
                            // コンスタントカーブなのでベース値を保存し、カーブを削除する
                            boneAnimtarget.Item = null;
                        }
                    }
                    boneAnim.bone_anim_target = tempBoneAnimTargets.ToArray();

                    tempBoneAnims.Add(boneAnim);
                }

            }

            //  リストを配列に変換して設定
            {
                Data.bone_anim_array.length = tempBoneAnims.Count;
                Data.bone_anim_array.bone_anim = tempBoneAnims.ToArray();
                if (Data.bone_anim_array.length == 0)
                {
                    Data.bone_anim_array = null;
                }
            }

            // ユーザーデータのシリアライズ
            if (UserDataArray.Data == null ||
                UserDataArray.Data.Count == 0)
            {
                Data.user_data_array = null;
            }
            else
            {
                Data.user_data_array = new user_data_arrayType();
                this.MakeSerializeData(Data.user_data_array, UserDataArray, tempStreams);
            }

            // ストリームデータの設定
            BinaryStreams = tempStreams;

            // ストリームのソート(や削除等)を行う。
            nw.g3d.iflib.StreamUtility.SortStream(Data, BinaryStreams);

            Data.stream_array = null;
        }

        #region savedData
        private skeletal_anim_infoType savedData;
        public List<BoneAnim> savedBoneAnims;

        /// <summary>
        /// 空以外の保存されたカーブ
        /// </summary>
        public int SavedCurveCount { get; private set; }

        public override void UpdateSavedData()
        {
            base.UpdateSavedData();
            savedData = ObjectUtility.Clone(Data.skeletal_anim_info);
            SavedUserDataArray = ObjectUtility.Clone(UserDataArray);
            UserDataArrayChanged = false;
            SavedCurveCount = 0;
            foreach (var boneAnim in BoneAnims)
            {
                foreach (var animTarget in boneAnim.BoneAnimTargets)
                {
                    animTarget.IsModified = false;
                    if (NotEmpty(animTarget))
                    {
                        animTarget.IsSaved = true;
                        SavedCurveCount++;
                    }
                    else
                    {
                        animTarget.IsSaved = false;
                    }
                }
            }

            savedBoneAnims = ObjectUtility.Clone(BoneAnims);
        }

        public void CopySavedData(SkeletalAnimation source)
        {
            CopyDocumentSavedData(source);
            savedData = source.savedData;
            SavedUserDataArray = source.SavedUserDataArray;
            UserDataArrayChanged = !source.SavedUserDataArray.IsSame(UserDataArray);
            SavedCurveCount = source.SavedCurveCount;
        }

        public override bool EqualsToSavedData()
        {
            if (!base.EqualsToSavedData())
            {
                return false;
            }

            return !SkeletalAnimationGeneralPage.IsModified(this) &&
                !IsCurveChanged() &&
                !UserDataPage.IsModified(this);
        }

        public bool IsCurveChanged()
        {
            return BoneAnims.Count != savedBoneAnims.Count || !BoneAnims.Zip(savedBoneAnims, (x, y) => x.IsSame(y)).All(x => x);
        }

        public bool IsValueChanged<T>(Func<skeletal_anim_infoType, T> select) where T : struct
        {
            return !select(savedData).Equals(select(Data.skeletal_anim_info));
        }

        public bool IsStringChanged(Func<skeletal_anim_infoType, string> select)
        {
            return select(savedData) != select(Data.skeletal_anim_info);
        }

        public bool IsPreviewRetargetingHostModelNameChanged()
        {
            return RetargetingHostModelName != DocumentManager.ProjectDocument.GetSavedSkeletalAnimationHostModelName(this);
        }

        public bool IsPreviewMotionMirroringChanged()
        {
            // セーブされずデフォルトが無効なので、有効にされている場合は編集されている
            return MotionMirroring;
        }

        #endregion

        public override void OnDocumentSwapAdd(Document old)
        {
            CopySavedData((SkeletalAnimation)old);
            base.OnDocumentSwapAdd(old);
        }

        /// <summary>
        /// バインド先のアニメーションと設定の比較を行う。
        /// </summary>
        public IEnumerable<string> CheckBind(Model target)
        {
            DebugConsole.WriteLine("CheckBind");

            // スケールモードのチェック
            var animScaleMode = ConvertScaleMode(Data.skeletal_anim_info.scale_mode);
            if (target.Data.skeleton.skeleton_info.scale_mode != animScaleMode)
            {
                yield return string.Format(res.Strings.SkeletalAnimation_ScaleModeError,
                    UIText.EnumValue(target.Data.skeleton.skeleton_info.scale_mode),
                    UIText.EnumValue(animScaleMode));
            }

            // スケール補正のチェック
            foreach (var boneAnim in BoneAnims)
            {
                var bone = target.Data.skeleton.bone_array.GetItems().FirstOrDefault(x => x.name == boneAnim.bone_name);
                if (bone != null)
                {
                    if (bone.scale_compensate != boneAnim.scale_compensate)
                    {
                        if (bone.scale_compensate)
                        {
                            yield return string.Format(res.Strings.SkeletalAnimation_ScaleCompensateErrorModel,
                                bone.name,
                                boneAnim.bone_name);
                        }
                        else
                        {
                            yield return string.Format(res.Strings.SkeletalAnimation_ScaleCompensateErrorAnimation,
                                bone.name,
                                boneAnim.bone_name);
                        }
                    }
                    else
                    {
                        // TODO: 別のエラー?
                    }
                }
            }
        }

        private skeleton_info_scale_modeType ConvertScaleMode(skeletal_anim_info_scale_modeType skeletal_anim_info_scale_mode)
        {
            switch (skeletal_anim_info_scale_mode)
            {
                case skeletal_anim_info_scale_modeType.standard:
                    return skeleton_info_scale_modeType.standard;
                case skeletal_anim_info_scale_modeType.maya:
                    return skeleton_info_scale_modeType.maya;
                case skeletal_anim_info_scale_modeType.softimage:
                    return skeleton_info_scale_modeType.softimage;
            }

            throw new NotImplementedException();
        }

        public BoneAnim GetBoneAnim(string boneName)
        {
            return BoneAnims.FirstOrDefault(x => x.bone_name == boneName);
        }

        public void UpdateIsModifiedAnimTargetAll()
        {
            foreach (var boneAnim in BoneAnims)
            {
                var savedBoneAnim = savedBoneAnims.FirstOrDefault(x => x.bone_name == boneAnim.bone_name);
                foreach (var animTarget in boneAnim.BoneAnimTargets)
                {
                    var targetType = animTarget.target;
                    var savedAnimTarget = savedBoneAnim != null ? savedBoneAnim.BoneAnimTargets.FirstOrDefault(x => x.target == targetType) : null; //savedBoneAnim.BoneAnimTargets.FirstOrDefault(x => x.sampler_name == animTarget.) : null;
                    UpdateIsModifiedAnimTarget(animTarget, savedAnimTarget);
                }
            }
        }

        public bool NotEmpty(BoneAnimTarget current)
        {
            return current.KeyFrames.Count > 0;
        }
        /// <summary>
        /// カーブの変更フラグを更新する
        /// </summary>
        public void UpdateIsModifiedAnimTarget(BoneAnimTarget current, BoneAnimTarget saved)
        {
            if (saved == null)
            {
                current.IsModified = NotEmpty(current);
                current.IsSaved = false;
            }
            else
            {
                current.IsModified = !current.IsSame(saved);
                current.IsSaved = saved.IsSaved;
            }
        }

        public bool IsCurvesModified()
        {
            int savedCount = 0;
            foreach (var boneAnim in BoneAnims)
            {
                foreach (var animTarget in boneAnim.BoneAnimTargets)
                {
                    if (animTarget.IsModified)
                    {
                        return true;
                    }
                    if (animTarget.IsSaved)
                    {
                        savedCount++;
                    }
                }
            }

            return savedCount != SavedCurveCount;
        }

        //private string retargetHostModelName = null;
        public override string RetargetingHostModelName
        {
            get;
            set;
        }

        // ミラーリング再生を有効にする
        public bool MotionMirroring
        {
            get;
            set;
        }
    }
}
