﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Linq;
using App.Command;
using App.ConfigData;
using App.PropertyEdit;
using App.Utility;
using ConfigCommon;
using nw.g3d.iflib;
using nw.g3d.nw4f_3dif;

namespace App.Data
{
    public sealed class ShapeAnimation : AnimationDocument, IHasUserData
    {
        public shape_animType Data{ get; private set; }
        public override object nw4f_3difItem
        {
            get { return Data; }
        }

        public override int FrameCount
        {
            get{ return Data.shape_anim_info.frame_count; }
            set{ Data.shape_anim_info.frame_count = value; }
        }

        public override bool Loop
        {
            get { return Data.shape_anim_info.loop; }
            set { Data.shape_anim_info.loop = value; }
        }

        public override bool IsFrameCountModified
        {
            get { return savedData.frame_count != Data.shape_anim_info.frame_count; }
        }

        public override bool IsLoopModified
        {
            get { return savedData.loop != Data.shape_anim_info.loop; }
        }


        public override bool IsEditable
        {
            get
            {
                return false;
            }
        }

        public override bool HasAnimation
        {
            get
            {
                return VertexShapeAnims != null &&
                       VertexShapeAnims
                            .Where(x => x.ShapeAnims != null)
                            .Any(x => x.ShapeAnims.Any());
            }
        }


        public UserDataArray UserDataArray { get; set; }
        public UserDataArray SavedUserDataArray { get; set; }
        public bool UserDataArrayChanged { get; set; }

        public override process_log_arrayType process_log_array
        {
            get { return Data.process_log_array; }
        }

        // マージできるかどうか
        public override bool CanMerge
        {
            get
            {
                return false;
            }
        }

        [Serializable]
        public class VertexShapeAnim
        {
            public VertexShapeAnim()
            {
                shape_name = string.Empty;
                ShapeAnims = new List<ShapeAnim>();
            }

            public string			shape_name{	get; set; }
            public string base_name { get; set; }
            public List<ShapeAnim>	ShapeAnims{	get; set; }
        }

        [Serializable]
        public class ShapeAnim : AnimTarget
        {
            public ShapeAnim()
            {
                key_shape_name = string.Empty;
            }

            public string		key_shape_name{	get; set; }

            public shape_anim_targetType ConvertTo_shape_anim_target(out G3dStream stream, bool viewer = false)
            {
                IG3dQuantizedCurve curve;
                Convert(out curve, out stream, viewer);
                var target = new shape_anim_targetType()
                {
                    base_value = GetBaseValue(),
                    Curve = curve,
                    key_shape_name = key_shape_name,
                };
                return target;
            }
        }

        public List<VertexShapeAnim>	VertexShapeAnims{ get; set; }

        public ShapeAnim GetAnimTarget(List<VertexShapeAnim> vertexShapeAnims, string shapeName, string keyShapeName)
        {
            return (from vertexShapeAnim in vertexShapeAnims
                    where vertexShapeAnim.shape_name == shapeName
                    from shapeAnim in vertexShapeAnim.ShapeAnims
                    where shapeAnim.key_shape_name == keyShapeName
                    select shapeAnim).FirstOrDefault();
        }

        #region 圧縮率
        public bool	IsAllFrameCurveConstant { get { return GetFrameDataSize(true) == 0; } }
        public int	FrameCompressDataSize   { get { return GetFrameDataSize(false); } }
        public int	FrameUncompressDataSize { get { return GetFrameDataSize(true);  } }
        #endregion

        public ShapeAnimation(shape_animType animation, List<G3dStream> streams) : base(GuiObjectID.ShapeAnimation, streams)
        {
            Data = animation;
            Initialize();
        }

        public void Initialize()
        {
            ToolData.Load(Data.tool_data);
            MakeComment(Data.comment);

            // ユーザーデータ作成
            this.MakeUserData(Data.user_data_array, BinaryStreams);

            MakeVertexShapeAnims();
        }

        #region Bridge
        public class ShapeAnimSwapData : SwapData
        {
            public ToolData ToolData;
            public List<G3dStream> streams;
            public string Comment;
            public RgbaColor? EditColor;
            public shape_animType Data;
            public List<VertexShapeAnim> VertexShapeAnims;
            public UserDataArray userData;
        }

        public override SwapData CreateSwapData(object data, List<G3dStream> streams)
        {
            var current = GetSwapData();
            Data = (shape_animType)data;
            BinaryStreams = streams;
            Initialize();

            return Swap(current, false);
        }

        public override SwapData GetSwapData()
        {
            return new ShapeAnimSwapData()
            {
                Data = Data,
                streams = BinaryStreams,
                Comment = Comment,
                EditColor = EditColor,
                VertexShapeAnims = VertexShapeAnims,
                userData = UserDataArray,
            };
        }

        public override SwapData Swap(SwapData swapData, bool copyHistory)
        {
            var current = (ShapeAnimSwapData)GetSwapData();
            var data = (ShapeAnimSwapData)swapData;
            Data = data.Data;
            BinaryStreams = data.streams;
            Comment = data.Comment;
            EditColor = data.EditColor;
            VertexShapeAnims = data.VertexShapeAnims;
            UserDataArray = data.userData;

            if (copyHistory)
            {
                UserDataArrayChanged = !SavedUserDataArray.IsSame(UserDataArray);
                UpdateIsModifiedAnimTargetAll();
            }

            return current;
        }
        #endregion
        public override nw4f_3difType Create_nw4f_3difType(bool viewer)
        {
            lock (this)
            {
                UpdateData(viewer);

                //			nw4f_3difType nw4f_3dif = new nw4f_3difType();
                nw4f_3dif_.Item = Data;
                nw4f_3dif_.file_info = null;

                return nw4f_3dif_;
            }
        }

        private void UpdateData(bool viewer = false)
        {
            ToolData.Load(Data.tool_data);
            Data.comment = GetComment();

            List<G3dStream> streams = new List<G3dStream>();
            List<vertex_shape_animType> vertex_shape_anims = new List<vertex_shape_animType>();
            foreach (var shape in VertexShapeAnims)
            {
                List<shape_anim_targetType> shape_anim_targets = new List<shape_anim_targetType>();
                foreach (var shapeAnim in shape.ShapeAnims)
                {
                    if (shapeAnim.KeyFrames.Any() == false)
                    {
                        // 保存しない
                        continue;
                    }

                    G3dStream stream;
                    var target = shapeAnim.ConvertTo_shape_anim_target(out stream, viewer);

                    if (stream != null)
                    {
                        target.Curve.stream_index = streams.Count;
                        streams.Add(stream);
                    }

                    shape_anim_targets.Add(target);
                }

                if (shape_anim_targets.Any())
                {
                    vertex_shape_anims.Add(new vertex_shape_animType()
                    {
                        shape_name = shape.shape_name,
                        base_name = shape.base_name,
                        shape_anim_target = shape_anim_targets.ToArray(),
                    });
                }
            }

            if (vertex_shape_anims.Any())
            {
                Data.vertex_shape_anim_array = new vertex_shape_anim_arrayType()
                {
                    vertex_shape_anim = vertex_shape_anims.ToArray(),
                };
            }
            else
            {
                Data.vertex_shape_anim_array = null;
            }

            // ユーザーデータのシリアライズ
            if (UserDataArray.Data == null ||
                UserDataArray.Data.Count == 0)
            {
                Data.user_data_array = null;
            }
            else
            {
                Data.user_data_array = new user_data_arrayType();
                this.MakeSerializeData(Data.user_data_array, UserDataArray, streams);
            }

            // ストリームの設定
            {
                BinaryStreams = streams;

                // ストリームのソート(や削除等)を行う。
                nw.g3d.iflib.StreamUtility.SortStream(Data, BinaryStreams);

                Data.stream_array = null;
            }
        }

        public override IEnumerable<Tuple<AnimTarget, IfQuantizationAnalysisResult>> QuantizeAnalyse()
        {
            foreach (var vertexShapeAnim in VertexShapeAnims)
            {
                foreach (var shapeAnim in vertexShapeAnim.ShapeAnims)
                {
                    var curve = new ShapeAnimationCurveTreeNodeInfo(this, vertexShapeAnim.shape_name, shapeAnim.key_shape_name);

                    IfQuantizationAnalysisResult result = null;
                    try
                    {
                        result = AnimationCurveEditCommand.MakeQuantizationAnalysis(this, ObjectID, curve, false);
                    }
                    catch (Exception)
                    {
                    }
                    yield return new Tuple<AnimTarget, IfQuantizationAnalysisResult>(shapeAnim, result);
                }
            }

        }

        public void Optimize()
        {
            // ストリームを更新する。
            UpdateData();

            // <hermite_curve>、<linear_curve>、<step_curve> の frame_type と key_type を
            // none に設定しないと、再分析が行われません。
            if (Data.vertex_shape_anim_array != null)
            {
                foreach (var anim in Data.vertex_shape_anim_array.vertex_shape_anim)
                {
                    if (anim.shape_anim_target != null)
                    {
                        foreach (var target in anim.shape_anim_target.Where(x => x.Curve != null))
                        {
                            target.Curve.frame_type = curve_frame_typeType.none;
                            target.Curve.key_type = curve_key_typeType.none;
                        }
                    }
                }
            }

            try
            {
                // アニメーションに合わせたAnimQuantizationAnalysisOptimizerを呼び出す。
                IfShapeAnimQuantizationAnalysisOptimizer optimizer = new IfShapeAnimQuantizationAnalysisOptimizer();
                optimizer.Optimize(Data, BinaryStreams);
            }
            catch(Exception e)
            {
                DebugConsole.WriteLine("Exception Catch : IfShapeAnimQuantizationAnalysisOptimizer.Optimize() : {0}", e.Message);
            }

            // 最適化後に、データを再作成する。
            this.MakeUserData(Data.user_data_array, BinaryStreams);
            MakeVertexShapeAnims();
        }

        private int GetFrameDataSize(bool isBase)
        {
            int size = 0;
            {
                foreach (var vertexShapeAnim in VertexShapeAnims)
                {
                    foreach (var target in vertexShapeAnim.ShapeAnims)
                    {
                        size += target.KeyFrames.Count * (isBase ? 4 : IfUtility.GetByteSizeFromCurveFrameType(target.QuantizationInfo.frame_type));
                    }
                }
            }
            return size;
        }

        private void MakeVertexShapeAnims()
        {
            VertexShapeAnims = new List<VertexShapeAnim>();

            if (Data.vertex_shape_anim_array != null)
            {
                foreach (var vsa in Data.vertex_shape_anim_array.vertex_shape_anim)
                {
                    var newVsa = new VertexShapeAnim()
                    {
                        shape_name = vsa.shape_name,
                        base_name = vsa.base_name
                    };

                    foreach (var sa in vsa.shape_anim_target)
                    {
                        var newSa = new ShapeAnim()
                        {
                            key_shape_name = sa.key_shape_name
                        };

                        newSa.Initialize(sa.Curve, sa.base_value, BinaryStreams, InterpolationType.Hermite);

                        newVsa.ShapeAnims.Add(newSa);
                    }

                    VertexShapeAnims.Add(newVsa);
                }
            }
        }

        public override EditCommand CreateUpdateBindCommand()
        {
            bool updateVertexShapeAnims = false;
            var newVertexShapeAnims = new List<VertexShapeAnim>();

            // バインドされたマテリアルのグループ
            var shapes = (from model in DocumentManager.Models
                             where model.AllAnimations.Contains(this)
                             from shape in model.Shapes
                             where shape.Data.key_shape_array != null
                             select shape).ToArray();

            var shapeTable = new Dictionary<string, List<Shape>>();
            var shapeNames = new List<string>();
            foreach (var shape in shapes)
            {
                List<Shape> list;
                if (shapeTable.TryGetValue(shape.Name, out list))
                {
                    list.Add(shape);
                }
                else
                {
                    shapeTable.Add(shape.Name, new List<Shape>() { shape });
                    shapeNames.Add(shape.Name);
                }
            }

            // 既存のVertexShapeAnim
            foreach (var vertexShapeAnim in VertexShapeAnims)
            {
                if (shapeTable.ContainsKey(vertexShapeAnim.shape_name) ||
                    vertexShapeAnim.ShapeAnims.Any(x => x.ExportType != CurveExportType.Ignored))
                {
                    newVertexShapeAnims.Add(vertexShapeAnim);
                }
                else
                {
                    updateVertexShapeAnims = true;
                }
            }

            // 新たなVertexShapeAnim を追加
            foreach (var shapeName in shapeNames)
            {
                if (!newVertexShapeAnims.Any(x => x.shape_name == shapeName))
                {
                    newVertexShapeAnims.Add(new VertexShapeAnim
                    {
                        shape_name = shapeName,
                    });
                    updateVertexShapeAnims = true;
                }
            }

            // ParamAnims の更新
            var shapeAnims = new List<ShapeAnimUpdateBindData>();
            foreach (var vertexShapeAnim in newVertexShapeAnims)
            {
                List<Shape> shapeList;
                var keyShapes = new List<string>();
                HashSet<string> nameSet = null;
                if (shapeTable.TryGetValue(vertexShapeAnim.shape_name, out shapeList))
                {
                    keyShapes.AddRange(shapeList.SelectMany(x => x.Data.key_shape_array.key_shape).Select(x => x.name).Distinct());
                    nameSet = new HashSet<string>(keyShapes);
                }
                else
                {
                    nameSet = new HashSet<string>();
                }

                bool updateShapeAnimTargets = false;
                var newShapeAnimTargets = new List<ShapeAnim>();
                // 既存のもの
                foreach (var shapeAnimTarget in vertexShapeAnim.ShapeAnims)
                {
                    if (shapeAnimTarget.ExportType != CurveExportType.Ignored)
                    {
                        newShapeAnimTargets.Add(shapeAnimTarget);
                        nameSet.Remove(shapeAnimTarget.key_shape_name);
                    }
                    else
                    {
                        if (nameSet.Remove(shapeAnimTarget.key_shape_name))
                        {
                            newShapeAnimTargets.Add(shapeAnimTarget);
                        }
                        else
                        {
                            updateShapeAnimTargets = true;
                        }
                    }
                }

                // 新たなもの
                foreach (var paramName in keyShapes)
                {
                    if (nameSet.Contains(paramName))
                    {
                        newShapeAnimTargets.Add(new ShapeAnim()
                        {
                            key_shape_name = paramName,
                        });
                        updateShapeAnimTargets = true;
                    }
                }

                if (updateShapeAnimTargets)
                {
                    shapeAnims.Add(new ShapeAnimUpdateBindData()
                    {
                        vertexShapeAnim = vertexShapeAnim,
                        baseName = string.IsNullOrEmpty(vertexShapeAnim.base_name) ?
                            newShapeAnimTargets.First().key_shape_name:
                            vertexShapeAnim.base_name,
                        ShapeAnims = newShapeAnimTargets,
                    });
                }
            }

            if (updateVertexShapeAnims || shapeAnims.Count > 0)
            {
                // 編集フラグを更新しない
                return new GeneralGroupReferenceEditCommand<DummyObject>(
                    DummyObject.TheDummyObjectGroup,
                    GuiObjectID.DummyObject,
                    Enumerable.Repeat<DummyObject>(DummyObject.TheDummyObject, 1),
                    delegate(ref GuiObject target, ref object data, ref object swap)
                    {
                        if (updateVertexShapeAnims)
                        {
                            var temp = VertexShapeAnims;
                            VertexShapeAnims = newVertexShapeAnims;
                            newVertexShapeAnims = temp;
                        }

                        foreach (var swapData in shapeAnims)
                        {
                            {
                                var temp = swapData.vertexShapeAnim.ShapeAnims;
                                swapData.vertexShapeAnim.ShapeAnims = swapData.ShapeAnims;
                                swapData.ShapeAnims = temp;
                            }

                            {
                                var temp = swapData.vertexShapeAnim.base_name;
                                swapData.vertexShapeAnim.base_name = swapData.baseName;
                                swapData.baseName = temp;
                            }
                        }

                        // 変更マークを更新する
                        UpdateIsModifiedAnimTargetAll();

                        // シリアライズされるデータに変更はないので Viewer に再転送する必要はない
                    });
            }

            return null;
        }

        private class ShapeAnimUpdateBindData
        {
            public VertexShapeAnim vertexShapeAnim;
            public string baseName;
            public List<ShapeAnim> ShapeAnims;
        }

        #region savedData
        private shape_anim_infoType savedData;
        public List<VertexShapeAnim> savedVertexShapeAnims;

        /// <summary>
        /// 空以外の保存されたカーブ
        /// </summary>
        public int SavedCurveCount { get; private set; }

        public override void UpdateSavedData()
        {
            base.UpdateSavedData();
            savedData = ObjectUtility.Clone(Data.shape_anim_info);
            SavedUserDataArray = ObjectUtility.Clone(UserDataArray);
            UserDataArrayChanged = false;

            SavedCurveCount = 0;
            foreach (var vertexShapeAnim in VertexShapeAnims)
            {
                foreach (var shapeAnim in vertexShapeAnim.ShapeAnims)
                {
                    shapeAnim.IsModified = false;
                    if (NotEmpty(shapeAnim))
                    {
                        shapeAnim.IsSaved = true;
                        SavedCurveCount++;
                    }
                    else
                    {
                        shapeAnim.IsSaved = false;
                    }
                }
            }
            savedVertexShapeAnims = ObjectUtility.Clone(VertexShapeAnims);
        }

        public void CopySavedData(ShapeAnimation source)
        {
            CopyDocumentSavedData(source);
            savedData = source.savedData;

            SavedUserDataArray = source.SavedUserDataArray;
            UserDataArrayChanged = !SavedUserDataArray.IsSame(UserDataArray);

            SavedCurveCount = source.SavedCurveCount;
            savedVertexShapeAnims = source.savedVertexShapeAnims;

            UpdateIsModifiedAnimTargetAll();
        }

        public override bool EqualsToSavedData()
        {
            if (!base.EqualsToSavedData())
            {
                return false;
            }

            return !ShapeAnimationGeneralPage.IsModified(this) &&
                !IsCurvesModified() &&
                !UserDataPage.IsModified(this);
        }

        public void UpdateIsModifiedAnimTargetAll()
        {
            foreach (var vertexShapeAnim in VertexShapeAnims)
            {
                var savedVertexShapeAnim = savedVertexShapeAnims.FirstOrDefault(x => x.shape_name == vertexShapeAnim.shape_name);
                foreach (var shapeAnim in vertexShapeAnim.ShapeAnims)
                {
                    var savedShapeAnim = savedVertexShapeAnim != null ? savedVertexShapeAnim.ShapeAnims.FirstOrDefault(x => x.key_shape_name == shapeAnim.key_shape_name) : null;
                    UpdateIsModifiedAnimTarget(shapeAnim, savedShapeAnim);
                }
            }
        }

        public bool NotEmpty(ShapeAnim current)
        {
            return current.KeyFrames.Count > 0;
        }

        /// <summary>
        /// カーブの変更フラグを更新する
        /// </summary>
        public void UpdateIsModifiedAnimTarget(ShapeAnim current, ShapeAnim saved)
        {
            if (saved == null)
            {
                current.IsModified = NotEmpty(current);
                current.IsSaved = false;
            }
            else
            {
                current.IsModified = !current.IsSame(saved);
                current.IsSaved = saved.IsSaved;
            }
        }

        public bool IsValueChanged<T>(Func<shape_anim_infoType, T> select) where T : struct
        {
            return !select(savedData).Equals(select(Data.shape_anim_info));
        }

        public bool IsStringChanged(Func<shape_anim_infoType, string> select)
        {
            return select(savedData) != select(Data.shape_anim_info);
        }

        public bool IsCurvesModified()
        {
            int savedCount = 0;
            foreach (var vertexShapeAnim in VertexShapeAnims)
            {
                foreach (var shapeAnim in vertexShapeAnim.ShapeAnims)
                {
                    if (shapeAnim.IsModified)
                    {
                        return true;
                    }
                    if (shapeAnim.IsSaved)
                    {
                        savedCount++;
                    }
                }
            }

            return savedCount != SavedCurveCount;
        }
        #endregion

        public override void OnDocumentSwapAdd(Document old)
        {
            CopySavedData((ShapeAnimation)old);
            base.OnDocumentSwapAdd(old);
        }
    }
}
