﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.ComponentModel;
using System.Diagnostics;
using System.Drawing;
using System.Windows.Forms;
using App.Utility;

namespace App.Controls
{
    /// <summary>
    /// ＵＩトラックバークラス。
    /// </summary>
    [ToolboxBitmap(typeof(TrackBar))]
    [DefaultEvent("SequentialValueChanged")]
    public class UITrackBar : TrackBar
    {
        // 値変更中フラグ
        private bool _valueChanging = false;
        // 値ツールチップ表示フラグ
        private bool _showValueHint = true;
        // 値表示用ツールチップ
        private readonly UIToolTip _ttpValue = new UIToolTip();

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public UITrackBar()
        {
            base.AutoSize = false;
            base.TickStyle = TickStyle.Both;
        }

        #region プロパティ
        /// <summary>
        /// 値ツールチップ表示フラグ。
        /// </summary>
        [DefaultValue(true)]
        [Category(UIControlHelper.OriginalPropertyCategoryName)]
        [Description("マウスポイント時に値をツールチップ表示するかどうかを示します。")]
        public bool ShowValueHint
        {
            get { return _showValueHint; }
            set { _showValueHint = value; }
        }
        #endregion

        #region イベント
        //---------------------------------------------------------------------
        private static readonly object EVENT_SEQUENTIALVALUECHANGED = new object();

        /// <summary>
        /// 値変更イベント。
        /// </summary>
        [Category(UIControlHelper.OriginalEventCategoryName)]
        [Description("値が変更された時に発生します。")]
        public event SequentialValueChangedEventHandler SequentialValueChanged
        {
            add { base.Events.AddHandler(EVENT_SEQUENTIALVALUECHANGED, value); }
            remove { base.Events.RemoveHandler(EVENT_SEQUENTIALVALUECHANGED, value); }
        }

        /// <summary>
        /// 値変更ハンドラ。
        /// </summary>
        protected virtual void OnSequentialValueChanged(SequentialValueChangedEventArgs e)
        {
            SequentialValueChangedEventHandler handler = (SequentialValueChangedEventHandler)base.Events[EVENT_SEQUENTIALVALUECHANGED];
            if (handler != null) { handler(this, e); }
        }
        #endregion

        /// <summary>
        /// 値ヒント用テキストを取得。
        /// </summary>
        protected virtual string GetValueHintText(int value)
        {
            return value.ToString();
        }

        /// <summary>
        /// 指定した位置に対応する値を取得。
        /// </summary>
        private int ValueAt(Point point)
        {
            Rectangle rcChannel = TrackBarUtility.GetChannelRect(this);
            Rectangle rcThumb   = TrackBarUtility.GetThumbRect(this);

            // チャンネル領域より手前
            if (point.X < rcChannel.Left)
            {
                return base.Minimum;
            }
            // チャンネル領域より奥
            else if (point.X >= rcChannel.Right)
            {
                return base.Maximum;
            }
            // つまみ領域内
            else if (rcThumb.Contains(point))
            {
                return base.Value;
            }

            // 位置に対応した値を取得
            int result = (int)MathUtility.Interpolate(
                rcChannel.Left,
                rcChannel.Right,
                point.X,
                base.Minimum,
                base.Maximum + 1
            );
            if (result > base.Maximum)
            {
                result = base.Maximum;
            }
            return result;
        }

        /// <summary>
        /// 値を増減させる。
        /// </summary>
        private void IncrementValue(bool up, int increment)
        {
            int value  = base.Value;
            int result = value;
            if (increment > 0)
            {
                // 増加
                if (up)
                {
                    result = Convert.ToInt32(Math.Floor((value + increment) / (double)increment) * increment);
                    if (result > base.Maximum)
                    {
                        result = base.Maximum;
                    }
                }
                // 減少
                else
                {
                    result = Convert.ToInt32(Math.Ceiling((value - increment) / (double)increment) * increment);
                    if (result < base.Minimum)
                    {
                        result = base.Minimum;
                    }
                }
                if (result != value)
                {
                    _valueChanging = true;
                    base.Value = result;
                }
            }
        }

        #region オーバーライド
        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override bool CanRaiseEvents
        {
            get
            {
                if (UIControlEventSuppressBlock.Enabled)
                {
                    return false;
                }
                return base.CanRaiseEvents;
            }
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override CreateParams CreateParams
        {
            get
            {
                // つまみを常に四角形にする
                CreateParams cp = base.CreateParams;
                cp.Style |= Win32.TBS.TBS_NOTICKS;
                return cp;
            }
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override Size DefaultSize
        {
            get { return new Size(base.DefaultSize.Width, 24); }
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void Dispose(bool disposing)
        {
            if (disposing)
            {
                _ttpValue.Dispose();
            }
            base.Dispose(disposing);
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void WndProc(ref Message m)
        {
            switch (m.Msg)
            {
                // チラつきを軽減
                case Win32.WM.WM_ERASEBKGND:
                    return;

                // 増減キー入力
                case Win32.WM.WM_KEYDOWN:
                    switch ((int)m.WParam)
                    {
                        // 小増加
                        case Win32.VK.VK_UP:
                        case Win32.VK.VK_RIGHT:
                            IncrementValue(true, base.SmallChange);
                            return;
                        // 小減少
                        case Win32.VK.VK_DOWN:
                        case Win32.VK.VK_LEFT:
                            IncrementValue(false, base.SmallChange);
                            return;
                        // 大増加
                        case Win32.VK.VK_PRIOR:
                            IncrementValue(true, base.LargeChange);
                            return;
                        // 大減少
                        case Win32.VK.VK_NEXT:
                            IncrementValue(false, base.LargeChange);
                            return;
                        default:
                            break;
                    }
                    break;

                // スクロール
                case Win32.WM.WM_REFLECT|Win32.WM.WM_HSCROLL:
                case Win32.WM.WM_REFLECT|Win32.WM.WM_VSCROLL:
                    {
                        int code = Win32.Utility.LOWORD(m.WParam);

                        // スクロール終了
                        if (code == Win32.TB.TB_ENDTRACK)
                        {
                            if (_valueChanging)
                            {
                                _valueChanging = false;
                                OnSequentialValueChanged(new SequentialValueChangedEventArgs(false));
                            }
                        }
                        // スクロール中
                        else
                        {
                            _valueChanging = true;
                        }
                    }
                    break;

                // 左クリック
                case Win32.WM.WM_LBUTTONDOWN:
                    {
                        Point     point     = Win32.Utility.LParamToPoint(m.LParam);
                        Rectangle rcChannel = TrackBarUtility.GetChannelRect(this);
                        Rectangle rcThumb   = TrackBarUtility.GetThumbRect(this);

                        // チャンネル領域より手前
                        if (point.X < rcChannel.Left)
                        {
                            base.Value = base.Minimum;
                        }
                        // チャンネル領域より奥
                        else if (point.X >= rcChannel.Right)
                        {
                            base.Value = base.Maximum;
                        }
                        // チャンネル領域内
                        else
                        {
                            // つまみ領域外なら
                            if (!rcThumb.Contains(point))
                            {
                                // 位置に対応した値を取得
                                int value = (int)MathUtility.Interpolate(
                                    rcChannel.Left,
                                    rcChannel.Right,
                                    point.X,
                                    base.Minimum,
                                    base.Maximum + 1
                                );
                                if (value > base.Maximum)
                                {
                                    value = base.Maximum;
                                }

                                // つまみの位置を変更
                                TrackBarUtility.SetPos(this, value);

                                // つまみ領域を再取得
                                Rectangle rcNewThumb = TrackBarUtility.GetThumbRect(this);

                                // TB_PAGEUP、TB_PAGEDOWN が発生するのを防ぐため
                                // つまみをポイントしていなければ基本処理をさせない
                                if (!rcNewThumb.Contains(point))
                                {
                                    // フォーカスを忘れずに設定
                                    base.Focus();
                                    // 値を更新
                                    base.Value = value;
                                    return;
                                }
                            }
                        }
                    }
                    break;
                // マウスホイール処理させない
                case Win32.WM.WM_MOUSEWHEEL:
                    {
                        if (!Focused)
                        {
                            // 親に委譲する
                            Win32.NativeMethods.SendMessage(Parent.Handle, m.Msg, m.WParam, m.LParam);
                            return;
                        }
                        else
                        {
                            // 領域外でホイール処理をしたときに親をたどってフォーカスを移す
                            var parent = Parent;
                            while (parent != null && !parent.Focus())
                            {
                                parent = parent.Parent;
                            }
                            return;
                        }
                    }
                default: break;
            }
            base.WndProc(ref m);
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void OnValueChanged(EventArgs e)
        {
            // 値変更イベント発行
            OnSequentialValueChanged(new SequentialValueChangedEventArgs(_valueChanging));
            base.OnValueChanged(e);
        }

        /// <summary>
        /// 直前のマウス位置
        /// </summary>
        Point lastMouseLocation = new Point(int.MinValue, int.MinValue);

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void OnMouseMove(MouseEventArgs e)
        {
            // ツールチップ処理
            if (e.Location != lastMouseLocation)
            {
                if (_showValueHint)
                {
                    // 位置
                    Point point = new Point(e.X, e.Y + Cursor.Current.Size.Height);

                    // 通常時
                    if (e.Button == MouseButtons.None)
                    {
                        string text = GetValueHintText(ValueAt(e.Location));
                        _ttpValue.BackColor = SystemColors.Info;
                        _ttpValue.Show(text, this, point);
                    }
                    // つまみドラッグ時
                    else if (e.Button == MouseButtons.Left && _valueChanging)
                    {
                        string text = GetValueHintText(Value);
                        _ttpValue.BackColor = Color.Pink;
                        _ttpValue.Show(text, this, point);
                    }
                }
                lastMouseLocation = e.Location;
            }
            base.OnMouseMove(e);
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void OnMouseLeave(EventArgs e)
        {
            // ツールチップ処理
            if (_showValueHint)
            {
                _ttpValue.Hide(this);
            }
            base.OnMouseLeave(e);
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void OnMouseDown(MouseEventArgs e)
        {
            // ツールチップ処理
            if (_showValueHint)
            {
                _ttpValue.Hide(this);
            }
            base.OnMouseDown(e);
        }
        #endregion

        #region デザイナ制御
        /// <summary>
        /// 再定義。
        /// </summary>
        [DefaultValue(false)]
        [ReadOnly(true)]
        public new bool AutoSize
        {
            get { return base.AutoSize;  }
            set { base.AutoSize = value; }
        }

        /// <summary>
        /// 再定義。
        /// </summary>
        [ReadOnly(true)]
        public new Orientation Orientation
        {
            get { return base.Orientation;  }
            set { base.Orientation = value; }
        }

        /// <summary>
        /// 再定義。
        /// </summary>
        [ReadOnly(true)]
        public new int TickFrequency
        {
            get { return base.TickFrequency;  }
            set { base.TickFrequency = value; }
        }

        /// <summary>
        /// 再定義。
        /// </summary>
        [DefaultValue(TickStyle.Both)]
        [ReadOnly(true)]
        public new TickStyle TickStyle
        {
            get { return base.TickStyle;  }
            set { base.TickStyle = value; }
        }

        /// <summary>
        /// 再定義。
        /// </summary>
        [ReadOnly(true)]
        public override RightToLeft RightToLeft
        {
            get { return base.RightToLeft;  }
            set { base.RightToLeft = value; }
        }

        /// <summary>
        /// 再定義。
        /// </summary>
        [ReadOnly(true)]
        public override bool RightToLeftLayout
        {
            get { return base.RightToLeftLayout;  }
            set { base.RightToLeftLayout = value; }
        }
        #endregion

        private bool drawRedThumb;

        /// <summary>
        /// つまみの枠を赤く描画するかどうか
        /// </summary>
        public bool DrawRedThumb
        {
            set
            {
                if (drawRedThumb != value)
                {
                    drawRedThumb = value;

                    // OnPaint が呼ばれるように設定
                    // UserPaint が true だとスライド中にちらつく問題がある
                    SetStyle(ControlStyles.UserPaint, drawRedThumb);

                    Invalidate();
                }
            }
        }

        protected override void OnPaint(PaintEventArgs e)
        {
            // TODO: スライド中に Thumb がちらつく問題がある
            Debug.Assert(drawRedThumb);
            base.OnPaint(e);

            var thumbRect = TrackBarUtility.GetThumbRect(this);
            var channelRect = TrackBarUtility.GetChannelRect(this);
            TrackBarRenderer.DrawHorizontalTrack(e.Graphics, channelRect);

            // マウス位置でスタイルを決める
            var point = PointToClient(MousePosition);
            System.Windows.Forms.VisualStyles.TrackBarThumbState style;
            if (!Enabled)
            {
                style = System.Windows.Forms.VisualStyles.TrackBarThumbState.Disabled;
            }
            else if (this.Focused)
            {
                style = System.Windows.Forms.VisualStyles.TrackBarThumbState.Pressed;
            }
            else if (thumbRect.Contains(point))
            {
                style = System.Windows.Forms.VisualStyles.TrackBarThumbState.Hot;
            }
            else
            {
                style = System.Windows.Forms.VisualStyles.TrackBarThumbState.Normal;
            }

            TrackBarRenderer.DrawHorizontalThumb(e.Graphics, thumbRect, style);
            if (Enabled)
            {
                e.Graphics.DrawRectangle(Pens.Red, thumbRect.Left, thumbRect.Top, thumbRect.Width - 1, thumbRect.Height - 1);
            }
        }
    }

    //----------------------------------------------------------------------------
    // 値編集用
    #region IntTrackBar
    /// <summary>
    /// 整数値トラックバークラス。
    /// </summary>
    public sealed class IntTrackBar : UITrackBar
    {
        // 連動イベント処理中フラグ
        private bool _buddyEventProcessing = false;
        // 連動入力ボックス
        private IntInputBox _buddyInputBox = null;
        // 連動アップダウン
        private IntUpDown _buddyUpDown = null;

        public double precision = 1;
        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public IntTrackBar()
        {
        }

        private bool baseValueChanged;
        private int lastValue;
        private int lastInternalValue;
        /// <summary>
        /// 値。
        /// </summary>
        [Category()]
        [DefaultValue(0)]
        [Description("値。再定義しています。")]
        public new int Value
        {
            get {
                if (lastInternalValue != base.Value || (baseValueChanged && (lastValue < Minimum || Maximum < lastValue)))
                {
                    baseValueChanged = true;
                    return Convert.ToInt32(base.Value / precision);
                }
                return lastValue;
            }
            set {
                lastValue = value;
                base.Value = ToInternalValue(value);
                baseValueChanged = false;
                lastInternalValue = base.Value;
            }
        }

        int _minimum;
        /// <summary>
        /// 最小値。
        /// </summary>
        [Category()]
        [DefaultValue(0)]
        [Description("最小値。再定義しています。")]
        public new int Minimum
        {
            get { return _minimum; }
            set {
                _minimum = value;
                SetRange();
            }
        }

        int _maximum;
        /// <summary>
        /// 最大値。
        /// </summary>
        [Category()]
        [DefaultValue(10)]
        [Description("最大値。再定義しています。")]
        public new int Maximum
        {
            get { return _maximum; }
            set {
                _maximum = value;
                SetRange();
            }
        }

        private void SetRange()
        {
            using (UIControlEventSuppressBlock block = new UIControlEventSuppressBlock())
            {
                precision = 1;
                int abs = Math.Max(_minimum, _maximum);
                int Base = 1 << 14;
                if (abs >= Base)
                {
                    precision = Base / (double)abs;
                }
                int v = Value;
                base.Maximum = ToInternalValue(_maximum, false);
                base.Minimum = ToInternalValue(_minimum, false);
                base.Value = ToInternalValue(v);
                lastInternalValue = base.Value;
            }
        }

        private int ToInternalValue(int value, bool round = true)
        {
            var dv = precision * value;
            int v;
            if (dv < Int32.MinValue) v = Int32.MinValue;
            else if (dv > Int32.MaxValue) v = Int32.MaxValue;
            else v = Convert.ToInt32(dv);

            if (round)
            {
                if (v < base.Minimum)
                {
                    v = base.Minimum;
                }
                else if (base.Maximum < v)
                {
                    v = base.Maximum;
                }
            }
            return v;
        }
        /// <summary>
        /// 小変化量。
        /// </summary>
        [Category()]
        [DefaultValue(1)]
        [Description("小変化量。再定義しています。")]
        public new int SmallChange
        {
            get { return base.SmallChange; }
            set { base.SmallChange = value; }
        }

        /// <summary>
        /// 大変化量。
        /// </summary>
        [Category()]
        [DefaultValue(5)]
        [Description("大変化量。再定義しています。")]
        public new int LargeChange
        {
            get { return base.LargeChange; }
            set { base.LargeChange = value; }
        }

        /// <summary>
        /// 連動入力ボックス。
        /// </summary>
        [DefaultValue(null)]
        [Description("連動する入力ボックス。入力ボックスからの値変更イベントをフックします。")]
        [RefreshProperties(RefreshProperties.All)]
        public IntInputBox BuddyInputBox
        {
            get { return _buddyInputBox; }
            set
            {
                if (_buddyInputBox != value)
                {
                    SetBuddyInputBox(value);
                    SetBuddyUpDown(null);
                }
            }
        }

        /// <summary>
        /// 連動アップダウン。
        /// </summary>
        [DefaultValue(null)]
        [Description("連動するアップダウン。アップダウンからの値変更イベントをフックします。")]
        [RefreshProperties(RefreshProperties.All)]
        public IntUpDown BuddyUpDown
        {
            get { return _buddyUpDown; }
            set
            {
                if (_buddyUpDown != value)
                {
                    SetBuddyUpDown(value);
                    SetBuddyInputBox(null);
                }
            }
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void Dispose(bool disposing)
        {
            if (disposing)
            {
                SetBuddyInputBox(null);
                SetBuddyUpDown(null);
            }
            base.Dispose(disposing);
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void OnSequentialValueChanged(SequentialValueChangedEventArgs e)
        {
            // 連動イベント処理中は何もしない
            if (_buddyEventProcessing) { return; }

            // 連動コントロールへ通知
            if (_buddyInputBox != null)
            {
                using (UIControlEventSuppressBlock block = new UIControlEventSuppressBlock())
                {
                    _buddyInputBox.Value = Value;
                }
            }
            if (_buddyUpDown != null)
            {
                using (UIControlEventSuppressBlock block = new UIControlEventSuppressBlock())
                {
                    _buddyUpDown.Value = Value;
                }
            }
            base.OnSequentialValueChanged(e);
        }

        /// <summary>
        /// 連動入力ボックスを設定。
        /// </summary>
        private void SetBuddyInputBox(IntInputBox inputBox)
        {
            if (_buddyInputBox != null)
            {
                _buddyInputBox.HandleDestroyed -= buddyInputBox_HandleDestroyed;
                _buddyInputBox.ValueChanged    -= buddyInputBox_ValueChanged;
            }

            _buddyInputBox = inputBox;

            if (_buddyInputBox != null)
            {
                _buddyInputBox.HandleDestroyed += buddyInputBox_HandleDestroyed;
                _buddyInputBox.ValueChanged    += buddyInputBox_ValueChanged;
            }
        }

        /// <summary>
        /// 連動アップダウンを設定。
        /// </summary>
        private void SetBuddyUpDown(IntUpDown upDown)
        {
            if (_buddyUpDown != null)
            {
                _buddyUpDown.HandleDestroyed        -= buddyInputBox_HandleDestroyed;
                _buddyUpDown.SequentialValueChanged -= buddyUpDown_SequentialValueChanged;
            }

            _buddyUpDown = upDown;

            if (_buddyUpDown != null)
            {
                _buddyUpDown.HandleDestroyed        += buddyUpDown_HandleDestroyed;
                _buddyUpDown.SequentialValueChanged += buddyUpDown_SequentialValueChanged;
            }
        }

        #region イベントハンドラ
        //---------------------------------------------------------------------
        // 連動入力ボックス
        private void buddyInputBox_HandleDestroyed(object sender, EventArgs e)
        {
            _buddyInputBox = null;
        }

        private void buddyInputBox_ValueChanged(object sender, EventArgs e)
        {
            // 自身の値を更新
            _buddyEventProcessing = true;
            {
                Value = _buddyInputBox.Value;
            }
            _buddyEventProcessing = false;

            // イベント発行
            OnSequentialValueChanged(new SequentialValueChangedEventArgs(false));
        }

        //---------------------------------------------------------------------
        // 連動アップダウン
        private void buddyUpDown_HandleDestroyed(object sender, EventArgs e)
        {
            _buddyUpDown = null;
        }

        private void buddyUpDown_SequentialValueChanged(object sender, SequentialValueChangedEventArgs e)
        {
            // 自身の値を更新
            _buddyEventProcessing = true;
            {
                Value = _buddyUpDown.Value;
            }
            _buddyEventProcessing = false;

            // イベント発行
            OnSequentialValueChanged(e);
        }
        #endregion

        protected override string GetValueHintText(int value)
        {
            return Convert.ToInt32(value / precision).ToString();
        }
    }
    #endregion

    #region UintTrackBar
    /// <summary>
    /// 整数値トラックバークラス。
    /// </summary>
    public sealed class UintTrackBar : UITrackBar
    {
        // 連動イベント処理中フラグ
        private bool _buddyEventProcessing = false;
        // 連動入力ボックス
        private IntInputBox _buddyInputBox = null;
        // 連動アップダウン
        private UintUpDown _buddyUpDown = null;

        public double precision = 1;
        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public UintTrackBar()
        {
        }

        private bool baseValueChanged;
        private uint lastValue;
        private int lastInternalValue;
        /// <summary>
        /// 値。
        /// </summary>
        [Category()]
        [DefaultValue(0)]
        [Description("値。再定義しています。")]
        public new uint Value
        {
            get
            {
                if (lastInternalValue != base.Value || (baseValueChanged && (lastValue < Minimum || Maximum < lastValue)))
                {
                    baseValueChanged = true;
                    return Convert.ToUInt32(base.Value / precision);
                }
                return lastValue;
            }
            set
            {
                lastValue = value;
                base.Value = ToInternalValue(value);
                baseValueChanged = false;
                lastInternalValue = base.Value;
            }
        }

        uint _minimum;
        /// <summary>
        /// 最小値。
        /// </summary>
        [Category()]
        [DefaultValue(0)]
        [Description("最小値。再定義しています。")]
        public new uint Minimum
        {
            get { return _minimum; }
            set
            {
                _minimum = value;
                SetRange();
            }
        }

        uint _maximum;
        /// <summary>
        /// 最大値。
        /// </summary>
        [Category()]
        [DefaultValue(10)]
        [Description("最大値。再定義しています。")]
        public new uint Maximum
        {
            get { return _maximum; }
            set
            {
                _maximum = value;
                SetRange();
            }
        }

        private void SetRange()
        {
            using (UIControlEventSuppressBlock block = new UIControlEventSuppressBlock())
            {
                precision = 1;
                uint abs = Math.Max(_minimum, _maximum);
                uint Base = 1 << 14;
                if (abs >= Base)
                {
                    precision = Base / (double)abs;
                }
                uint v = Value;
                base.Maximum = ToInternalValue(_maximum, false);
                base.Minimum = ToInternalValue(_minimum, false);
                base.Value = ToInternalValue(v);
                lastInternalValue = base.Value;
            }
        }

        private int ToInternalValue(uint value, bool round = true)
        {
            var dv = precision * value;
            int v;
            if (dv < Int32.MinValue) v = Int32.MinValue;
            else if (dv > Int32.MaxValue) v = Int32.MaxValue;
            else v = Convert.ToInt32(dv);

            if (round)
            {
                if (v < base.Minimum)
                {
                    v = base.Minimum;
                }
                else if (base.Maximum < v)
                {
                    v = base.Maximum;
                }
            }
            return v;
        }
        /// <summary>
        /// 小変化量。
        /// </summary>
        [Category()]
        [DefaultValue(1)]
        [Description("小変化量。再定義しています。")]
        public new uint SmallChange
        {
            get { return (uint)base.SmallChange; }
            set { base.SmallChange = (int)value; }
        }

        /// <summary>
        /// 大変化量。
        /// </summary>
        [Category()]
        [DefaultValue(5)]
        [Description("大変化量。再定義しています。")]
        public new uint LargeChange
        {
            get { return (uint)base.LargeChange; }
            set { base.LargeChange = Convert.ToInt32(value); }
        }

        /// <summary>
        /// 連動入力ボックス。
        /// </summary>
        [DefaultValue(null)]
        [Description("連動する入力ボックス。入力ボックスからの値変更イベントをフックします。")]
        [RefreshProperties(RefreshProperties.All)]
        public IntInputBox BuddyInputBox
        {
            get { return _buddyInputBox; }
            set
            {
                if (_buddyInputBox != value)
                {
                    SetBuddyInputBox(value);
                    SetBuddyUpDown(null);
                }
            }
        }

        /// <summary>
        /// 連動アップダウン。
        /// </summary>
        [DefaultValue(null)]
        [Description("連動するアップダウン。アップダウンからの値変更イベントをフックします。")]
        [RefreshProperties(RefreshProperties.All)]
        public UintUpDown BuddyUpDown
        {
            get { return _buddyUpDown; }
            set
            {
                if (_buddyUpDown != value)
                {
                    SetBuddyUpDown(value);
                    SetBuddyInputBox(null);
                }
            }
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void Dispose(bool disposing)
        {
            if (disposing)
            {
                SetBuddyInputBox(null);
                SetBuddyUpDown(null);
            }
            base.Dispose(disposing);
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void OnSequentialValueChanged(SequentialValueChangedEventArgs e)
        {
            // 連動イベント処理中は何もしない
            if (_buddyEventProcessing) { return; }

            // 連動コントロールへ通知
            if (_buddyInputBox != null)
            {
                using (UIControlEventSuppressBlock block = new UIControlEventSuppressBlock())
                {
                    _buddyInputBox.Value = (int)Value;
                }
            }
            if (_buddyUpDown != null)
            {
                using (UIControlEventSuppressBlock block = new UIControlEventSuppressBlock())
                {
                    _buddyUpDown.Value = Value;
                }
            }
            base.OnSequentialValueChanged(e);
        }

        /// <summary>
        /// 連動入力ボックスを設定。
        /// </summary>
        private void SetBuddyInputBox(IntInputBox inputBox)
        {
            if (_buddyInputBox != null)
            {
                _buddyInputBox.HandleDestroyed -= buddyInputBox_HandleDestroyed;
                _buddyInputBox.ValueChanged -= buddyInputBox_ValueChanged;
            }

            _buddyInputBox = inputBox;

            if (_buddyInputBox != null)
            {
                _buddyInputBox.HandleDestroyed += buddyInputBox_HandleDestroyed;
                _buddyInputBox.ValueChanged += buddyInputBox_ValueChanged;
            }
        }

        /// <summary>
        /// 連動アップダウンを設定。
        /// </summary>
        private void SetBuddyUpDown(UintUpDown upDown)
        {
            if (_buddyUpDown != null)
            {
                _buddyUpDown.HandleDestroyed -= buddyInputBox_HandleDestroyed;
                _buddyUpDown.SequentialValueChanged -= buddyUpDown_SequentialValueChanged;
            }

            _buddyUpDown = upDown;

            if (_buddyUpDown != null)
            {
                _buddyUpDown.HandleDestroyed += buddyUpDown_HandleDestroyed;
                _buddyUpDown.SequentialValueChanged += buddyUpDown_SequentialValueChanged;
            }
        }

        #region イベントハンドラ
        //---------------------------------------------------------------------
        // 連動入力ボックス
        private void buddyInputBox_HandleDestroyed(object sender, EventArgs e)
        {
            _buddyInputBox = null;
        }

        private void buddyInputBox_ValueChanged(object sender, EventArgs e)
        {
            // 自身の値を更新
            _buddyEventProcessing = true;
            {
                Value = (uint)_buddyInputBox.Value;
            }
            _buddyEventProcessing = false;

            // イベント発行
            OnSequentialValueChanged(new SequentialValueChangedEventArgs(false));
        }

        //---------------------------------------------------------------------
        // 連動アップダウン
        private void buddyUpDown_HandleDestroyed(object sender, EventArgs e)
        {
            _buddyUpDown = null;
        }

        private void buddyUpDown_SequentialValueChanged(object sender, SequentialValueChangedEventArgs e)
        {
            // 自身の値を更新
            _buddyEventProcessing = true;
            {
                Value = _buddyUpDown.Value;
            }
            _buddyEventProcessing = false;

            // イベント発行
            OnSequentialValueChanged(e);
        }
        #endregion

        protected override string GetValueHintText(int value)
        {
            return Convert.ToUInt32(value / precision).ToString();
        }
    }
    #endregion

    #region FloatTrackBar
    /// <summary>
    /// 実数値トラックバークラス。
    /// </summary>
    public sealed class FloatTrackBar : UITrackBar
    {
        // 内部精度
        private double _precision = 100;
        // 連動イベント処理中フラグ
        private bool _buddyEventProcessing = false;
        // 連動入力ボックス
        private FloatInputBox _buddyInputBox = null;
        // 連動アップダウン
        private FloatUpDown _buddyUpDown = null;

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public FloatTrackBar()
        {
            // IntTrackBar と初期値を合わせる
            base.Minimum     = (int)(base.Minimum * _precision);
            base.Maximum = (int)(base.Maximum * _precision);
            base.Value       = (int)(base.Value * _precision);
            base.SmallChange = (int)(base.SmallChange * _precision);
            base.LargeChange = (int)(base.LargeChange * _precision);
        }

        private bool baseValueChanged;
        private int lastBaseValue;
        private float lastValue;

        /// <summary>
        /// 値。
        /// </summary>
        [DefaultValue(0.0f)]
        [Description("値。float型で再定義しています。")]
        public new float Value
        {
            get {
                if (base.Value != lastBaseValue || (baseValueChanged && (lastValue < Minimum || Maximum < lastValue)))
                {
                    var v = (float)ToDisplayValue(base.Value);
                    baseValueChanged = true;
                    if (v < Minimum)
                    {
                        return Minimum;
                    }
                    else if (Maximum < v)
                    {
                        return Maximum;
                    }
                    return v;
                }

                return lastValue;
            }
            set {
                var v = ToInternalValue(value);
                Debug.Assert(base.Minimum <= v && v <= base.Maximum);
                base.Value = ToInternalValue(value);
                baseValueChanged = false;
                lastBaseValue = base.Value;
                lastValue = value;
            }
        }

        public float _minimum;
        /// <summary>
        /// 最小値。
        /// </summary>
        [DefaultValue(0.0f)]
        [Description("最小値。float型で再定義しています。")]
        public new float Minimum
        {
            get { return _minimum; }
            set {
                _minimum = value;
                SetRange(value, Maximum);
            }
        }

        public float _maximum;
        /// <summary>
        /// 最大値。
        /// </summary>
        [DefaultValue(10.0f)]
        [Description("最大値。float型で再定義しています。")]
        public new float Maximum
        {
            get { return _maximum; }
            set {
                _maximum = value;
                SetRange(Minimum, value);
            }
        }

        public double _smallChange;
        /// <summary>
        /// 小変化量。
        /// </summary>
        [DefaultValue(1.0f)]
        [Description("小変化量。float型で再定義しています。")]
        public new double SmallChange
        {
            get { return _smallChange; }
            set {
                Debug.Assert(value >= 0);
                _smallChange = value;
                base.SmallChange = ToInternalValue(value, false);
                if (_smallChange > 0 && base.SmallChange == 0)
                {
                    base.SmallChange = 1;
                }
            }
        }

        public double _largeChange;
        /// <summary>
        /// 大変化量。
        /// </summary>
        [DefaultValue(5.0f)]
        [Description("大変化量。float型で再定義しています。")]
        public new double LargeChange
        {
            get { return _largeChange; }
            set {
                Debug.Assert(value >= 0);
                _largeChange = value;
                base.LargeChange = ToInternalValue(value, false);
                if (_largeChange > 0 && base.LargeChange == 0)
                {
                    base.LargeChange = 1;
                }
            }
        }

        /// <summary>
        /// 連動入力ボックス。
        /// </summary>
        [DefaultValue(null)]
        [Description("連動する入力ボックス。入力ボックスからの値変更イベントをフックします。")]
        [RefreshProperties(RefreshProperties.All)]
        public FloatInputBox BuddyInputBox
        {
            get { return _buddyInputBox; }
            set
            {
                if (_buddyInputBox != value)
                {
                    SetBuddyInputBox(value);
                    SetBuddyUpDown(null);
                }
            }
        }

        /// <summary>
        /// 連動アップダウン。
        /// </summary>
        [DefaultValue(null)]
        [Description("連動するアップダウン。アップダウンからの値変更イベントをフックします。")]
        [RefreshProperties(RefreshProperties.All)]
        public FloatUpDown BuddyUpDown
        {
            get { return _buddyUpDown; }
            set
            {
                if (_buddyUpDown != value)
                {
                    SetBuddyUpDown(value);
                    SetBuddyInputBox(null);
                }
            }
        }

        /// <summary>
        /// 範囲を設定。float型で設定できます。
        /// </summary>
        public void SetRange(float minimum, float maximum)
        {
            // 双方の絶対値の大きい方を取得
            double range = Math.Max(Math.Abs(minimum), Math.Abs(maximum));
            range = Math.Max(range, Math.Abs(SmallChange));
            range = Math.Max(range, Math.Abs(LargeChange));

            // 桁数に応じた精度を求める
            double precision = 1;
            if (0 < range)
            {
                precision = Math.Pow(10, 2-Math.Floor(Math.Log10(range) + 1e-6)); // 誤差対策
                if (precision <= 0.1)
                {
                    precision *= 10;
                }
                //DebugConsole.WriteLine(range.ToString() + " " + precision);
            }

            // 精度が変わる場合も変わらない場合も
            //if (_precision != precision)
            {
                float value       = Value;
                double smallChange = SmallChange;
                double largeChange = LargeChange;

                // 変更
                _precision = precision;
                using (UIControlEventSuppressBlock block = new UIControlEventSuppressBlock())
                {
                    base.SetRange(ToInternalValue(minimum, false), ToInternalValue(maximum, false));
                    base.SmallChange = ToInternalValue(smallChange, false);
                    base.LargeChange = ToInternalValue(largeChange, false);
                    base.Value       = ToInternalValue(value);
                    lastBaseValue = base.Value;
                }
            }
        }

        /// <summary>
        /// 値書式指定フォーマットを取得。
        /// </summary>
        public string GetValueFormat()
        {
            // 精度に応じた表示桁数にする
            double precision   = _precision;
            int placeNumber = 1;
            while (precision > 10)
            {
                precision /= 10;
                placeNumber++;
            }
            return string.Format("f{0}", placeNumber);
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void Dispose(bool disposing)
        {
            if (disposing)
            {
                SetBuddyInputBox(null);
                SetBuddyUpDown(null);
            }
            base.Dispose(disposing);
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void OnSequentialValueChanged(SequentialValueChangedEventArgs e)
        {
            // 連動イベント処理中は何もしない
            if (_buddyEventProcessing) { return; }

            // 連動コントロールへ通知
            if (_buddyInputBox != null)
            {
                using (UIControlEventSuppressBlock block = new UIControlEventSuppressBlock())
                {
                    _buddyInputBox.Value = Value;
                }
            }
            if (_buddyUpDown != null)
            {
                using (UIControlEventSuppressBlock block = new UIControlEventSuppressBlock())
                {
                    _buddyUpDown.Value = Value;
                }
            }
            base.OnSequentialValueChanged(e);
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override string GetValueHintText(int value)
        {
            return ToDisplayValue(value).ToString(GetValueFormat());
        }

        /// <summary>
        /// 表示型を内部型に変換。
        /// </summary>
        private int ToInternalValue(double displayValue, bool round = true)
        {
            var dv = displayValue * _precision;
            int v;
            if (dv < Int32.MinValue) v = Int32.MinValue;
            else if (dv > Int32.MaxValue) v = Int32.MaxValue;
            else v = Convert.ToInt32(displayValue * _precision);

            if (round)
            {
                if (v < base.Minimum)
                {
                    v = base.Minimum;
                }
                else if (base.Maximum < v)
                {
                    v = base.Maximum;
                }
            }
            return v;
        }

        /// <summary>
        /// 内部型を表示型に変換。
        /// </summary>
        private double ToDisplayValue(int internalValue)
        {
            return (double)(internalValue / _precision);
        }

        /// <summary>
        /// 連動入力ボックスを設定。
        /// </summary>
        private void SetBuddyInputBox(FloatInputBox inputBox)
        {
            if (_buddyInputBox != null)
            {
                _buddyInputBox.HandleDestroyed -= buddyInputBox_HandleDestroyed;
                _buddyInputBox.ValueChanged    -= buddyInputBox_ValueChanged;
            }

            _buddyInputBox = inputBox;

            if (_buddyInputBox != null)
            {
                _buddyInputBox.HandleDestroyed += buddyInputBox_HandleDestroyed;
                _buddyInputBox.ValueChanged    += buddyInputBox_ValueChanged;
            }
        }

        /// <summary>
        /// 連動アップダウンを設定。
        /// </summary>
        private void SetBuddyUpDown(FloatUpDown upDown)
        {
            if (_buddyUpDown != null)
            {
                _buddyUpDown.HandleDestroyed        -= buddyInputBox_HandleDestroyed;
                _buddyUpDown.SequentialValueChanged -= buddyUpDown_SequentialValueChanged;
            }

            _buddyUpDown = upDown;

            if (_buddyUpDown != null)
            {
                _buddyUpDown.HandleDestroyed        += buddyUpDown_HandleDestroyed;
                _buddyUpDown.SequentialValueChanged += buddyUpDown_SequentialValueChanged;
            }
        }

        #region イベントハンドラ
        //---------------------------------------------------------------------
        // 連動入力ボックス
        private void buddyInputBox_HandleDestroyed(object sender, EventArgs e)
        {
            _buddyInputBox = null;
        }

        private void buddyInputBox_ValueChanged(object sender, EventArgs e)
        {
            // 自身の値を更新
            _buddyEventProcessing = true;
            {
                Value = _buddyInputBox.Value;
            }
            _buddyEventProcessing = false;

            // イベント発行
            OnSequentialValueChanged(new SequentialValueChangedEventArgs(false));
        }

        //---------------------------------------------------------------------
        // 連動アップダウン
        private void buddyUpDown_HandleDestroyed(object sender, EventArgs e)
        {
            _buddyUpDown = null;
        }

        private void buddyUpDown_SequentialValueChanged(object sender, SequentialValueChangedEventArgs e)
        {
            // 自身の値を更新
            _buddyEventProcessing = true;
            {
                Value = _buddyUpDown.Value;
            }
            _buddyEventProcessing = false;

            // イベント発行
            OnSequentialValueChanged(e);
        }
        #endregion
    }
    #endregion

    //----------------------------------------------------------------------------
    // イベントデータ
    #region SequentialValueChangedEvent
    /// <summary>
    /// 連続的な値変更イベントハンドラデリゲート。
    /// </summary>
    public delegate void SequentialValueChangedEventHandler(object sender, SequentialValueChangedEventArgs e);

    /// <summary>
    /// 連続的な値変更イベントデータクラス。
    /// </summary>
    public sealed class SequentialValueChangedEventArgs : EventArgs
    {
        // 変更中
        private readonly bool _changing;

        /// <summary>
        /// 要素インデックス
        /// </summary>
        public uint ElementBits{	get; set;	}

        public MouseEventArgs MouseEventArgs{	get; set;	}

        // このイベントが発生する元になったイベント
        // nullの場合あり
        public EventArgs SourceEventArgs{ get; set; }

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public SequentialValueChangedEventArgs(bool changing)
        {
            MouseEventArgs = null;

            _changing = changing;
        }

        /// <summary>
        /// 変更中かどうか。
        /// </summary>
        public bool Changing
        {
            get { return _changing; }
        }
    }
    #endregion
}
