﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Drawing;
using System.Drawing.Drawing2D;
using System.Drawing.Imaging;
using System.Windows.Forms;
using App.Data;
using App.Utility;
using App.res;
using nw.g3d.nw4f_3dif;

namespace App.Controls
{
    /// <summary>
    /// テクスチャ/パレットビットマップイメージビュークラス。
    /// </summary>
    [ToolboxItem(false)]
    public sealed class TextureImageView : MouseOperatableControl
 	{
        #region フィールド
        //-----------------------------------------------------------------
        // ターゲット
        private readonly Texture target_;
        // チャンネル
        private ChannelFlags channel_;
        // 表示する深さ。-1で全部表示
        private int depthIndex_;
        // 表示倍率
        private float magnify_ = 1.0f;
        // ミップマップ部表示フラグ
        private bool isShowMipmap_ = false;
        // ミップマップ方向
        private ConfigCommon.MipmapOrientationType mipmapOrientation_ = ConfigCommon.MipmapOrientationType.Vertical;

        //-----------------------------------------------------------------
        // カラーイメージ
        private Bitmap colorImage_;
        // カラーイメージ（ミップマップ付き）
        private readonly Dictionary<ConfigCommon.MipmapOrientationType, Bitmap> colorImageMM_ = new Dictionary<ConfigCommon.MipmapOrientationType, Bitmap>();
        // アルファイメージ
        private Bitmap alphaImage_;
        // アルファイメージ（ミップマップ付き）
        private readonly Dictionary<ConfigCommon.MipmapOrientationType, Bitmap> alphaImageMM_ = new Dictionary<ConfigCommon.MipmapOrientationType, Bitmap>();

        //-----------------------------------------------------------------
        // 操作モード
        private OperationMode operationMode_ = OperationMode.Default;
        // ドラッグ時のスクロール値
        private Point scrollOnDrag_ = new Point();
        // 最後にポイントしたピクセル
        private Point lastPointedPixel_ = new Point(-2, -2);

        //-----------------------------------------------------------------
        // 表示倍率テーブル
        private static readonly float[] magnifyTable_ = {
              0.0100f,  0.0200f,  0.0300f,  0.0400f,
              0.0500f,  0.0625f,  0.0833f,  0.1250f,
              0.1660f,  0.2500f,  0.3330f,  0.5000f,
              0.6660f,  1.0000f,  2.0000f,  3.0000f,
              4.0000f,  5.0000f,  6.0000f,  7.0000f,
              8.0000f, 12.0000f, 16.0000f,
        };
        #endregion

        #region イベント
        //---------------------------------------------------------------------
        private static readonly object EVENT_MAGNIFYCHANGED = new object();

        /// <summary>
        /// 表示倍率変更イベント。
        /// </summary>
        public event EventHandler MagnifyChanged
        {
            add { base.Events.AddHandler(EVENT_MAGNIFYCHANGED, value); }
            remove { base.Events.RemoveHandler(EVENT_MAGNIFYCHANGED, value); }
        }

        /// <summary>
        /// 表示倍率変更ハンドラ。
        /// </summary>
        private void OnMagnifyChanged(EventArgs e)
        {
            EventHandler handler = (EventHandler)base.Events[EVENT_MAGNIFYCHANGED];
            if (handler != null) { handler(this, e); }
        }

        //---------------------------------------------------------------------
        private static readonly object EVENT_CHANNELCHANGED = new object();

        /// <summary>
        /// チャンネル変更イベント。
        /// </summary>
        public event EventHandler ChannelChanged
        {
            add { base.Events.AddHandler(EVENT_CHANNELCHANGED, value); }
            remove { base.Events.RemoveHandler(EVENT_CHANNELCHANGED, value); }
        }

        /// <summary>
        /// チャンネル変更ハンドラ。
        /// </summary>
        private void OnChannelChanged(EventArgs e)
        {
            EventHandler handler = (EventHandler)base.Events[EVENT_CHANNELCHANGED];
            if (handler != null) { handler(this, e); }
        }

        //---------------------------------------------------------------------
        private static readonly object EVENT_PIXELHINT = new object();

        /// <summary>
        /// ピクセルヒントイベント。
        /// </summary>
        public event TPBitmapImagePixelHintEventHandler PixelHint
        {
            add { base.Events.AddHandler(EVENT_PIXELHINT, value); }
            remove { base.Events.RemoveHandler(EVENT_PIXELHINT, value); }
        }

        /// <summary>
        /// ピクセルヒントハンドラ。
        /// </summary>
        private void OnPixelHint(TPBitmapImagePixelHintEventArgs e)
        {
            TPBitmapImagePixelHintEventHandler handler = (TPBitmapImagePixelHintEventHandler)base.Events[EVENT_PIXELHINT];
            if (handler != null) { handler(this, e); }
        }
        #endregion

        // depthIndex : 表示する深さ。-1で全部表示
        public TextureImageView(Form owner, Texture target, ChannelFlags channel, int depthIndex)
        {
            target_ = target;
            channel_ = channel;
            depthIndex_ = depthIndex;

            // イメージ作成
            UpdateImages();

            // 初期表示倍率
            Size imageSize = RenderingColorImage.Size;
            {
                int size = Math.Max(imageSize.Width, imageSize.Height);

                // 512を超える場合のみ拡大、それ以下では等倍
                if (size > 512)
                {
                    magnify_ = 0.5f;
                }
            }

            // 初期設定
            AutoScroll = true;
            AutoScrollMinSize = RenderingSize;
            EnableAltInput = true;
            RegisterAltModifiedKeys(Keys.Space);

            // オーナーへのイベント登録
            owner.Deactivate += new EventHandler(owner_Deactivate);
        }

        /// <summary>
        /// ミップマップ部表示フラグ。
        /// </summary>
        public bool ShowMipmap
        {
            get { return isShowMipmap_; }
            set
            {
                if (isShowMipmap_ != value)
                {
                    isShowMipmap_ = value;

                    // スクロール関連の再設定
                    // Magnify の変更時と同じような処理になる
                    Invalidate();
                    AutoScrollMinSize = RenderingSize;
                    AutoScrollPosition = Point.Empty;
                }
            }
        }

        public int DepthIndex
        {
            get { return depthIndex_; }
            set
            {
                if (depthIndex_ != value)
                {
                    depthIndex_ = value;

                    UpdateImages();

                    // スクロール関連の再設定
                    // Magnify の変更時と同じような処理になる
                    Invalidate();
                    AutoScrollMinSize = RenderingSize;
                    AutoScrollPosition = Point.Empty;
                }
            }
        }

        /// <summary>
        /// ミップマップ方向。
        /// </summary>
        public ConfigCommon.MipmapOrientationType MipmapOrientation
        {
            get { return mipmapOrientation_; }
            set
            {
                if (mipmapOrientation_ != value)
                {
                    mipmapOrientation_ = value;

                    // スクロール関連の再設定
                    // Magnify の変更時と同じような処理になる
                    Invalidate();
                    AutoScrollMinSize = RenderingSize;
                    AutoScrollPosition = Point.Empty;
                }
            }
        }

        /// <summary>
        /// ピクセル位置を取得。
        /// </summary>
        private Point GetPixelLocation(Point point)
        {
            // 描画領域
            Rectangle rcImage = RenderingBounds;

            // 領域内を指している場合のみ
            Point ptPixel = new Point(-1, -1);
            if (rcImage.Contains(point))
            {
                Size szBaseImage = RenderingColorImage.Size;
                ptPixel.X = (int)((point.X - rcImage.X) * szBaseImage.Width  / rcImage.Width);
                ptPixel.Y = (int)((point.Y - rcImage.Y) * szBaseImage.Height / rcImage.Height);
            }
            return ptPixel;
        }

        /// <summary>
        /// ミップマップイメージ内でのピクセル位置を取得。
        /// </summary>
        private static Point GetPixelLocationInMipmap(Point point, Size topLevelSize, int mipmapCount, Size mipmapMinSize, ConfigCommon.MipmapOrientationType mipmapOrientation, out int mipmapLevel)
        {
            int imageW = topLevelSize.Width;
            int imageH = topLevelSize.Height;

            if (mipmapOrientation == ConfigCommon.MipmapOrientationType.Vertical)
            {
                int heightPrev = 0;
                int heightNext = imageH;

                // ミップマップを追う
                for (int i = 0; i < mipmapCount; i++)
                {
                    // 縦方向にイメージ内
                    if (point.Y < heightNext)
                    {
                        // 横方向にイメージ内
                        if (point.X < imageW)
                        {
                            mipmapLevel = i;
                            return new Point(point.X, point.Y - heightPrev);
                        }

                        // 終了
                        break;
                    }

                    // 次のレベルへ
                    heightPrev += imageH;
                    {
                        imageW = Math.Max(mipmapMinSize.Width,  imageW >> 1);
                        imageH = Math.Max(mipmapMinSize.Height, imageH >> 1);
                    }
                    heightNext += imageH;
                }
            }
            else
            {
                int widthPrev = 0;
                int widthNext = imageW;

                // ミップマップを追う
                for (int i = 0; i < mipmapCount; i++)
                {
                    // 横方向にイメージ内
                    if (point.X < widthNext)
                    {
                        // 縦方向にイメージ内
                        if (point.Y < imageH)
                        {
                            mipmapLevel = i;
                            return new Point(point.X - widthPrev, point.Y);
                        }

                        // 終了
                        break;
                    }

                    // 次のレベルへ
                    widthPrev += imageW;
                    {
                        imageW /= 2;
                        imageH /= 2;
                    }
                    widthNext += imageW;
                }
            }

            // 無効
            mipmapLevel = -1;
            return new Point(-1, -1);
        }

        // ピクセル位置をテクスチャの次元で補正する
        private static Point CorrectPixelLocation(Point point, Size topLevelSize, int mipmapLevel, Size mipmapMinSize, texture_info_dimensionType dimension, int depth, int depthIndex)
        {
            switch(dimension)
            {
                case texture_info_dimensionType.Item3d:			point = CorrectPixelLocation_Item3d(      point, topLevelSize, mipmapLevel, mipmapMinSize, depth);	break;
                case texture_info_dimensionType.cube:			point = CorrectPixelLocation_cube(        point, topLevelSize, mipmapLevel);	break;
                case texture_info_dimensionType.Item1d_array:	point = CorrectPixelLocation_Item1d_array(point, topLevelSize, mipmapLevel, mipmapMinSize, depth, depthIndex);	break;
                case texture_info_dimensionType.Item2d_array:	point = CorrectPixelLocation_Item2d_array(point, topLevelSize, mipmapLevel, mipmapMinSize, depth, depthIndex);	break;
                case texture_info_dimensionType.cube_array:		point = CorrectPixelLocation_cube_array(  point, topLevelSize, mipmapLevel, depth);	break;

                // 上記４つの次元のもの以外は補正する必要がないのでなにもしない
            }

            return point;
        }

        private static Point CorrectPixelLocation_Item3d(Point point, Size topLevelSize, int mipmapLevel, Size mipmapMinSize, int depth)
        {
            int width  = Math.Max(1, topLevelSize.Width  >> mipmapLevel);
            int height = Math.Max(1, topLevelSize.Height >> mipmapLevel);

            int size = (int)Math.Ceiling(Math.Sqrt(depth));

            int cellWidth  = Math.Max(mipmapMinSize.Width,  width  / size);
            int cellHeight = Math.Max(mipmapMinSize.Height, height / size);

            int allWidth  = size;
            int allHeight = size;
            {
                for(int i = 0;i != mipmapLevel;++ i)
                {
                    if (allHeight > 1)
                    {
                        allHeight >>= 1;
                    }
                    else if (allWidth > 1)
                    {
                        allWidth >>= 1;
                    }
                }

                allWidth  *= cellWidth;
                allHeight *= cellHeight;
            }

            if ((point.X < allWidth) && (point.Y < allHeight))
            {
                int x = (cellWidth  == 0) ? 0 : (point.X % cellWidth);
                int y = (cellHeight == 0) ? 0 : (point.Y % cellHeight);

                return new Point(x, y);
            }
            else
            {
                return new Point(-1, -1);
            }
        }

        private static Point CorrectPixelLocation_cube(Point point, Size topLevelSize, int mipmapLevel)
        {
            // □■□□		■:画像あり
            // ■■■■		□:画像なし
            // □■□□

            int width  = Math.Max(1, topLevelSize.Width  >> mipmapLevel);
            int height = Math.Max(1, topLevelSize.Height >> mipmapLevel);

            int cellWidth  = width  / 4;
            int cellHeight = height / 3;

            int cellX  = (cellWidth  == 0) ? 0 : (point.X / cellWidth);
            int cellY  = (cellHeight == 0) ? 0 : (point.Y / cellHeight);

            if (((cellX == 0) && (cellY == 0)) ||
                ((cellX == 2) && (cellY == 0)) ||
                ((cellX == 3) && (cellY == 0)) ||
                ((cellX == 0) && (cellY == 2)) ||
                ((cellX == 2) && (cellY == 2)) ||
                ((cellX == 3) && (cellY == 2)))
            {
                // 画像のない部分は-1を返す
                return new Point(-1, -1);
            }
            else
            {
                int x = (cellWidth  == 0) ? 0 : (point.X % cellWidth);
                int y = (cellHeight == 0) ? 0 : (point.Y % cellHeight);

                return new Point(x, y);
            }
        }

        private static Point CorrectPixelLocation_Item1d_array(Point point, Size topLevelSize, int mipmapLevel, Size mipmapMinSize, int depth, int depthIndex)
        {
            return CorrectPixelLocation_Item2d_array(point, topLevelSize, mipmapLevel, mipmapMinSize, depth, depthIndex);
        }

        private static Point CorrectPixelLocation_Item2d_array(Point point, Size topLevelSize, int mipmapLevel, Size mipmapMinSize, int depth, int depthIndex)
        {
            int width  = Math.Max(1, topLevelSize.Width  >> mipmapLevel);
            int height = Math.Max(1, topLevelSize.Height >> mipmapLevel);

            int size = (depthIndex == -1) ? 1 : (int)Math.Ceiling(Math.Sqrt(depth));

            int cellWidth  = Math.Max(mipmapMinSize.Width,  width  / size);
            int cellHeight = Math.Max(mipmapMinSize.Height, height / size);

            int x = (cellWidth  == 0) ? 0 : (point.X % cellWidth);
            int y = (cellHeight == 0) ? 0 : (point.Y % cellHeight);

            return new Point(x, y);
        }

        private static Point CorrectPixelLocation_cube_array(Point point, Size topLevelSize, int mipmapLevel, int depth)
        {
            // 一枚あたり単位に調整する
            depth /= 6;

            int size = (int)Math.Ceiling(Math.Sqrt(depth));

            int width  = Math.Max(1, topLevelSize.Width  >> mipmapLevel) / size;
            int height = Math.Max(1, topLevelSize.Height >> mipmapLevel) / size;

            return
                CorrectPixelLocation_cube(
                    new Point(point.X % width, point.Y % height),
                    new Size(topLevelSize.Width / size, topLevelSize.Height / size),
                    mipmapLevel
                );
        }

        private void UpdateImages()
        {
            using (var waitCursor = new WaitCursor())
            {
                var colorImages = new Bitmap[] { };
                var alphaImages = new Bitmap[] { };
                target_.CreateImages(ref colorImages, ref alphaImages);

                if (DepthIndex == -1)
                {
                    // カラーイメージ
                    colorImage_ = colorImages[0];
                    colorImageMM_[ConfigCommon.MipmapOrientationType.Horizontal] = ImageBitmapCreator.CombineMipmapBitmaps(colorImages, target_.MipmapMinSize, ConfigCommon.MipmapOrientationType.Horizontal);
                    colorImageMM_[ConfigCommon.MipmapOrientationType.Vertical] = ImageBitmapCreator.CombineMipmapBitmaps(colorImages, target_.MipmapMinSize, ConfigCommon.MipmapOrientationType.Vertical);

                    // アルファイメージ
                    alphaImage_ = alphaImages[0];
                    alphaImageMM_[ConfigCommon.MipmapOrientationType.Horizontal] = ImageBitmapCreator.CombineMipmapBitmaps(alphaImages, target_.MipmapMinSize, ConfigCommon.MipmapOrientationType.Horizontal);
                    alphaImageMM_[ConfigCommon.MipmapOrientationType.Vertical] = ImageBitmapCreator.CombineMipmapBitmaps(alphaImages, target_.MipmapMinSize, ConfigCommon.MipmapOrientationType.Vertical);
                }
                else
                {
                    var tempColorImages = new Bitmap[colorImages.Length];
                    for(int i = 0;i != tempColorImages.Length;++ i)
                    {
                        tempColorImages[i] = MakeDepthImage(colorImages[i], i);
                    }

                    var tempAlphaImages = new Bitmap[alphaImages.Length];
                    for(int i = 0;i != tempAlphaImages.Length;++ i)
                    {
                        tempAlphaImages[i] = MakeDepthImage(alphaImages[i], i);
                    }

                    // カラーイメージ
                    colorImage_ = MakeDepthImage(colorImages[0], 0);
                    colorImageMM_[ConfigCommon.MipmapOrientationType.Horizontal]	= ImageBitmapCreator.CombineMipmapBitmaps(tempColorImages, target_.MipmapMinSize, ConfigCommon.MipmapOrientationType.Horizontal);
                    colorImageMM_[ConfigCommon.MipmapOrientationType.Vertical]		= ImageBitmapCreator.CombineMipmapBitmaps(tempColorImages, target_.MipmapMinSize, ConfigCommon.MipmapOrientationType.Vertical);

                    // アルファイメージ
                    alphaImage_ = MakeDepthImage(alphaImages[0], 0);
                    alphaImageMM_[ConfigCommon.MipmapOrientationType.Horizontal]	= ImageBitmapCreator.CombineMipmapBitmaps(tempAlphaImages, target_.MipmapMinSize, ConfigCommon.MipmapOrientationType.Horizontal);
                    alphaImageMM_[ConfigCommon.MipmapOrientationType.Vertical]		= ImageBitmapCreator.CombineMipmapBitmaps(tempAlphaImages, target_.MipmapMinSize, ConfigCommon.MipmapOrientationType.Vertical);
                }
            }
        }

        private Bitmap MakeDepthImage(Bitmap srcBitmap, int mipmapLevel)
        {
            var width  = Math.Max(1, target_.Data.texture_info.width  >> mipmapLevel);
            var height = Math.Max(1, target_.Data.texture_info.height >> mipmapLevel);

            // キューブテクスチャの時は、展開図として扱う
            if (target_.IsCube)
            {
                if (target_.IsArray)
                {
                    width *= 3;
                    height *= 2;
                }
                else
                {
                    width *= 4;
                    height *= 3;
                }
            }

            var bitmap = new Bitmap(width, height);
            {
                var allWidth  = srcBitmap.Width;
                var allHeight = srcBitmap.Height;

                var xCount = (int)(allWidth  / width);
                var yCount = (int)(allHeight / height);

                var x = (DepthIndex % xCount) * width;
                var y = (DepthIndex / yCount) * height;

                using (var g = Graphics.FromImage(bitmap))
                {
                    g.DrawImage(
                        srcBitmap,
                        new Rectangle(0, 0, width, height),
                        x, y, width, height, GraphicsUnit.Pixel
                    );
                }
            }
            return bitmap;
        }

        #region 描画情報
        /// <summary>
        /// 描画サイズ。
        /// </summary>
        public Size RenderingSize
        {
            get
            {
                // 基本サイズ
                Size size = RenderingColorImage.Size;

                // 表示倍率をかける（最低でも１ピクセル）
                size.Width  = (int)(size.Width  * magnify_);
                size.Height = (int)(size.Height * magnify_);
                if (size.Width  <= 1) { size.Width  = 1; }
                if (size.Height <= 1) { size.Height = 1; }
                return size;
            }
        }

        /// <summary>
        /// 描画領域。
        /// </summary>
        private Rectangle RenderingBounds
        {
            get
            {
                // 基本領域
                Rectangle rcClient = ClientRectangle;
                Rectangle rcBounds = new Rectangle(AutoScrollPosition, RenderingSize);

                // 非スクロール時は領域中央に描画
                if (!HScroll)
                {
                    rcBounds.X = (rcClient.Width - rcBounds.Width) / 2;
                }
                if (!VScroll)
                {
                    rcBounds.Y = (rcClient.Height - rcBounds.Height) / 2;
                }
                return rcBounds;
            }
        }

        /// <summary>
        /// 描画カラーイメージ。
        /// </summary>
        private Bitmap RenderingColorImage
        {
            get { return isShowMipmap_ ? colorImageMM_[MipmapOrientation] : colorImage_; }
        }

        /// <summary>
        /// 描画アルファイメージ。
        /// </summary>
        private Bitmap RenderingAlphaImage
        {
            get { return isShowMipmap_ ? alphaImageMM_[MipmapOrientation] : alphaImage_; }
        }
        #endregion

        #region チャンネル
        /// <summary>
        /// チャンネル。
        /// </summary>
        public ChannelFlags Channel
        {
            get { return channel_; }
            set
            {
                if (channel_ != value)
                {
                    channel_ = value;
                    Invalidate();
                    OnChannelChanged(EventArgs.Empty);
                }
            }
        }

        /// <summary>
        /// チャンネルを全て含んでいるか。
        /// </summary>
        public bool Contains(ChannelFlags channel)
        {
            return (channel & ~channel_) == 0;
        }

        /// <summary>
        /// チャンネルの切り替え。
        /// </summary>
        public void ChangeChannel(ChannelFlags channel)
        {
            if (channel == channel_)
            {
                // 一致するときは切り替えない
                return;
            }

            if (channel == ChannelFlags.RGB)
            {
                if (Contains(ChannelFlags.RGB))
                {
                    // OFF
                    channel_ &= ~ChannelFlags.RGB;
                }
                else
                {
                    // ON
                    channel_ |= ChannelFlags.RGB;
                }
            }
            else
            {
                // 反転
                channel_ ^= channel;
            }
            Invalidate();
            OnChannelChanged(EventArgs.Empty);
        }
        #endregion

        #region 表示倍率
        /// <summary>
        /// 表示倍率。
        /// </summary>
        public float Magnify
        {
            get { return magnify_; }
        }

        /// <summary>
        /// 表示倍率を変更。
        /// </summary>
        public void ChangeMagnify(float magnify)
        {
            // 倍率テーブルの範囲内のみ設定可能
            if (magnify >= magnifyTable_[0] && magnify <= magnifyTable_[magnifyTable_.Length - 1])
            {
                // 変更
                if (magnify_ != magnify)
                {
                    ChangeMagnifyCore(magnify, CenterOfClient);
                }
            }
        }

        /// <summary>
        /// 表示倍率を変更（内部処理）。
        /// </summary>
        private void ChangeMagnifyCore(float magnify, Point point)
        {
            //-----------------------------------------------------------------
            // 倍率変更前に行う処理

            // 指定点をクライアント中心からと描画領域中心からの位置に変換
            PointF ptFromImageCenter  = new PointF(0.0f, 0.0f);
            PointF ptFromClientCenter = new PointF(0.0f, 0.0f);
            {
                Rectangle rcImage  = RenderingBounds;
                Point     ptCenter = CenterOfClient;
                if (rcImage.X <= point.X && point.X < rcImage.Right)
                {
                    ptFromImageCenter.X  = point.X - ((rcImage.X + rcImage.Right) / 2.0f);
                    ptFromClientCenter.X = point.X - ptCenter.X;
                }
                if (rcImage.Y <= point.Y && point.Y < rcImage.Bottom)
                {
                    ptFromImageCenter.Y  = point.Y - ((rcImage.Y + rcImage.Bottom) / 2.0f);
                    ptFromClientCenter.Y = point.Y - ptCenter.Y;
                }
            }

            // 描画領域中心からの距離は変更率に応じてスケーリングする
            ptFromImageCenter.X *= (magnify / magnify_);
            ptFromImageCenter.Y *= (magnify / magnify_);

            //-----------------------------------------------------------------
            // 倍率変更
            magnify_ = magnify;

            //-----------------------------------------------------------------
            // 倍率変更後に行う処理

            // 先に全領域を無効化
            // スクロール設定変更後に呼ぶとちらつく...
            Invalidate();

            // スクロール最小サイズを変更してスクロール位置を調整
            AutoScrollMinSize = RenderingSize;
            {
                Size szScrollMaxNew = AutoScrollMaxSize;

                // 水平方向
                int scrollX = 0;
                if (szScrollMaxNew.Width > 0)
                {
                    scrollX = (int)(szScrollMaxNew.Width * 0.5f + (ptFromImageCenter.X - ptFromClientCenter.X));
                    if (scrollX < 0)
                    {
                        scrollX = 0;
                    }
                    else if (scrollX > szScrollMaxNew.Width)
                    {
                        scrollX = szScrollMaxNew.Width;
                    }
                }

                // 垂直方向
                int scrollY = 0;
                if (szScrollMaxNew.Height > 0)
                {
                    scrollY = (int)(szScrollMaxNew.Height * 0.5f + (ptFromImageCenter.Y - ptFromClientCenter.Y));
                    if (scrollY < 0)
                    {
                        scrollY = 0;
                    }
                    else if (scrollY > szScrollMaxNew.Height)
                    {
                        scrollY = szScrollMaxNew.Height;
                    }
                }

                // スクロール位置
                AutoScrollPosition = new Point(scrollX, scrollY);
            }

            OnMagnifyChanged(EventArgs.Empty);
        }

        /// <summary>
        /// 自動スクロール最大サイズ。
        /// </summary>
        private Size AutoScrollMaxSize
        {
            get
            {
                Size szScrollMin = AutoScrollMinSize;
                Size szClient    = ClientSize;

                // 幅
                int width = szScrollMin.Width - szClient.Width;
                if (width < 0)
                {
                    width = 0;
                }

                // 高さ
                int height = szScrollMin.Height - szClient.Height;
                if (height < 0)
                {
                    height = 0;
                }

                return new Size(width, height);
            }
        }

        /// <summary>
        /// ズームイン。
        /// </summary>
        public void ZoomIn()
        {
            ZoomInCore(CenterOfClient);
        }

        /// <summary>
        /// ズームアウト。
        /// </summary>
        public void ZoomOut()
        {
            ZoomOutCore(CenterOfClient);
        }

        /// <summary>
        /// ズームイン（内部処理）。
        /// </summary>
        private void ZoomInCore(Point point)
        {
            for (int i = 0; i < magnifyTable_.Length; i++)
            {
                if (magnify_ < magnifyTable_[i])
                {
                    float magnify = magnifyTable_[i];
                    ChangeMagnifyCore(magnify, point);
                    break;
                }
            }
        }

        /// <summary>
        /// ズームアウト（内部処理）。
        /// </summary>
        private void ZoomOutCore(Point point)
        {
            for (int i = magnifyTable_.Length - 1; i >= 0; i--)
            {
                if (magnify_ > magnifyTable_[i])
                {
                    float magnify = magnifyTable_[i];
                    ChangeMagnifyCore(magnify, point);
                    break;
                }
            }
        }

        /// <summary>
        /// クライアント領域の中央。
        /// </summary>
        private Point CenterOfClient
        {
            get
            {
                Size szClient = ClientSize;
                return new Point(szClient.Width / 2, szClient.Height / 2);
            }
        }
        #endregion

        #region 描画
        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void OnPaint(PaintEventArgs e)
        {
            //-------------------------------------------------------------
            // 背景
            e.Graphics.Clear(Color.Silver);

            //-------------------------------------------------------------
            // 描画領域
            Rectangle rcImage = RenderingBounds;

            //-------------------------------------------------------------
            // 枠線
            Rectangle rcFrame = rcImage;
            rcFrame.Inflate(1, 1);
            GraphicsUtility.DrawRectangle(e.Graphics, Pens.Black, rcFrame);

            //-------------------------------------------------------------
            // イメージ
            Bitmap colorImage = RenderingColorImage;
            Bitmap alphaImage = RenderingAlphaImage;
            {
                // ミップマップまたはキューブマップ描画時
                if (isShowMipmap_
                    || target_.Data.texture_info.dimension == texture_info_dimensionType.cube
                    || target_.Data.texture_info.dimension == texture_info_dimensionType.cube_array
                    )
                {
                    // チェック模様の下地
                    using (Brush brush = new HatchBrush(HatchStyle.LargeCheckerBoard, Color.White, Color.LightGray))
                    {
                        e.Graphics.RenderingOrigin = AutoScrollPosition;
                        e.Graphics.FillRectangle(brush, rcImage);
                    }
                }

                // 常にポイントサンプリング
                InterpolationMode imBack = e.Graphics.InterpolationMode;
                e.Graphics.InterpolationMode = InterpolationMode.NearestNeighbor;

                // 拡大時はサンプリングポイントをずらす
                PointF ptSampling = new PointF(0.0f, 0.0f);
                if (rcImage.Width > colorImage.Width || rcImage.Height > colorImage.Height)
                {
                    ptSampling.X = -0.5f;
                    ptSampling.Y = -0.5f;
                }

                // アルファのみ
                if (channel_ == ChannelFlags.A)
                {
                    e.Graphics.DrawImage(
                        alphaImage, rcImage,
                        ptSampling.X, ptSampling.Y,
                        alphaImage.Width, alphaImage.Height,
                        GraphicsUnit.Pixel
                    );
                }
                // カラー
                else
                {
                    // カラー要素別に行列作成
                    using (ImageAttributes ia = new ImageAttributes())
                    {
                        ia.SetColorMatrix(GetColorChannelMatrix());
                        e.Graphics.DrawImage(
                            colorImage, rcImage,
                            ptSampling.X, ptSampling.Y,
                            colorImage.Width, colorImage.Height,
                            GraphicsUnit.Pixel, ia
                        );
                    }

                    // アルファ合成
                    if (Contains(ChannelFlags.A))
                    {
                        ColorMatrix cm = new ColorMatrix();
                        {
                            cm.Matrix00 = 0.0f;
                            cm.Matrix10 = 0.0f;
                            cm.Matrix20 = 0.0f;
                            cm.Matrix30 = 0.0f;
                            cm.Matrix40 = 1.0f;

                            cm.Matrix01 = 0.0f;
                            cm.Matrix11 = 0.0f;
                            cm.Matrix21 = 0.0f;
                            cm.Matrix31 = 0.0f;
                            cm.Matrix41 = 0.0f;

                            cm.Matrix02 = 0.0f;
                            cm.Matrix12 = 0.0f;
                            cm.Matrix22 = 0.0f;
                            cm.Matrix32 = 0.0f;
                            cm.Matrix42 = 0.0f;

                            cm.Matrix03 = -0.5f;
                            cm.Matrix13 = 0.0f;
                            cm.Matrix23 = 0.0f;
                            cm.Matrix33 = 0.5f;
                            cm.Matrix43 = 0.0f;
                        }
                        using (ImageAttributes ia = new ImageAttributes())
                        {
                            ia.SetColorMatrix(cm);
                            e.Graphics.DrawImage(
                                alphaImage,
                                rcImage, ptSampling.X, ptSampling.Y,
                                alphaImage.Width, alphaImage.Height,
                                GraphicsUnit.Pixel, ia
                            );
                        }
                    }
                }

                // 補間モード復帰
                e.Graphics.InterpolationMode = imBack;

                // デコードできなくてオリジナルイメージが存在しない場合はメッセージを表示
                if (target_.IsOriginalImage && target_.OrigianlImageCount == 0)
                {
                    GraphicsUtility.DrawText(e.Graphics, Strings.TextureImagePanel_OnPaint_NoOriginalImage, this.Font, Color.Black, rcImage, HorizontalAlignment.Center);
                    rcImage.Offset(-1, -1);
                    GraphicsUtility.DrawText(e.Graphics, Strings.TextureImagePanel_OnPaint_NoOriginalImage, this.Font, Color.White, rcImage, HorizontalAlignment.Center);
                }

            }
            base.OnPaint(e);
        }

        /// <summary>
        /// カラーチャンネル行列を取得。
        /// </summary>
        private ColorMatrix GetColorChannelMatrix()
        {
            ColorMatrix cm = new ColorMatrix();

            // ＲＧＢ合成
            if (Contains(ChannelFlags.RGB))
            {
                // 単位行列
                cm.Matrix00 = 1.0f;
                cm.Matrix10 = 0.0f;
                cm.Matrix20 = 0.0f;
                cm.Matrix30 = 0.0f;
                cm.Matrix40 = 0.0f;

                cm.Matrix01 = 0.0f;
                cm.Matrix11 = 1.0f;
                cm.Matrix21 = 0.0f;
                cm.Matrix31 = 0.0f;
                cm.Matrix41 = 0.0f;

                cm.Matrix02 = 0.0f;
                cm.Matrix12 = 0.0f;
                cm.Matrix22 = 1.0f;
                cm.Matrix32 = 0.0f;
                cm.Matrix42 = 0.0f;
            }
            // 各成分の組み合わせ
            else
            {
                // 一旦ゼロ行列にする
                cm.Matrix00 = 0.0f;
                cm.Matrix10 = 0.0f;
                cm.Matrix20 = 0.0f;
                cm.Matrix30 = 0.0f;
                cm.Matrix40 = 0.0f;

                cm.Matrix01 = 0.0f;
                cm.Matrix11 = 0.0f;
                cm.Matrix21 = 0.0f;
                cm.Matrix31 = 0.0f;
                cm.Matrix41 = 0.0f;

                cm.Matrix02 = 0.0f;
                cm.Matrix12 = 0.0f;
                cm.Matrix22 = 0.0f;
                cm.Matrix32 = 0.0f;
                cm.Matrix42 = 0.0f;

                // Ｒ成分
                if (Contains(ChannelFlags.R))
                {
                    cm.Matrix00 = 1.0f;
                    cm.Matrix10 = 0.0f;
                    cm.Matrix20 = 0.0f;
                    cm.Matrix30 = 0.0f;
                    cm.Matrix40 = 0.0f;

                    // Ｒ成分のみ
                    if ((channel_ & ChannelFlags.RGB) == ChannelFlags.R)
                    {
                        cm.Matrix01 = 1.0f;
                        cm.Matrix11 = 0.0f;
                        cm.Matrix21 = 0.0f;
                        cm.Matrix31 = 0.0f;
                        cm.Matrix41 = 0.0f;

                        cm.Matrix02 = 1.0f;
                        cm.Matrix12 = 0.0f;
                        cm.Matrix22 = 0.0f;
                        cm.Matrix32 = 0.0f;
                        cm.Matrix42 = 0.0f;
                    }
                }
                // Ｇ成分
                if (Contains(ChannelFlags.G))
                {
                    cm.Matrix01 = 0.0f;
                    cm.Matrix11 = 1.0f;
                    cm.Matrix21 = 0.0f;
                    cm.Matrix31 = 0.0f;
                    cm.Matrix41 = 0.0f;

                    // Ｇ成分のみ
                    if ((channel_ & ChannelFlags.RGB) == ChannelFlags.G)
                    {
                        cm.Matrix00 = 0.0f;
                        cm.Matrix10 = 1.0f;
                        cm.Matrix20 = 0.0f;
                        cm.Matrix30 = 0.0f;
                        cm.Matrix40 = 0.0f;

                        cm.Matrix02 = 0.0f;
                        cm.Matrix12 = 1.0f;
                        cm.Matrix22 = 0.0f;
                        cm.Matrix32 = 0.0f;
                        cm.Matrix42 = 0.0f;
                    }
                }
                // Ｂ成分
                if (Contains(ChannelFlags.B))
                {
                    cm.Matrix02 = 0.0f;
                    cm.Matrix12 = 0.0f;
                    cm.Matrix22 = 1.0f;
                    cm.Matrix32 = 0.0f;
                    cm.Matrix42 = 0.0f;

                    // Ｂ成分のみ
                    if ((channel_ & ChannelFlags.RGB) == ChannelFlags.B)
                    {
                        cm.Matrix00 = 0.0f;
                        cm.Matrix10 = 0.0f;
                        cm.Matrix20 = 1.0f;
                        cm.Matrix30 = 0.0f;
                        cm.Matrix40 = 0.0f;

                        cm.Matrix01 = 0.0f;
                        cm.Matrix11 = 0.0f;
                        cm.Matrix21 = 1.0f;
                        cm.Matrix31 = 0.0f;
                        cm.Matrix41 = 0.0f;
                    }
                }
            }
            return cm;
        }
        #endregion

        #region マウス操作
        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void OnBeginMouseDrag(MouseBeginDragEventArgs e)
        {
            bool begin = false;

            // 拡大表示中
            if (HScroll || VScroll)
            {
                // 手のひらモード中
                if (operationMode_ == OperationMode.Hand)
                {
                    // 左ボタンのみ
                    if (e.Button == MouseButtons.Left)
                    {
                        // スクロール位置を保存
                        scrollOnDrag_ = AutoScrollPosition;
                        scrollOnDrag_.X *= -1;
                        scrollOnDrag_.Y *= -1;

                        // ドラッグ開始ＯＫ
                        begin = true;
                    }
                }
            }

            // 条件を満たさなければ常にキャンセル
            e.Cancel = !begin;
            base.OnBeginMouseDrag(e);
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void OnMouseDrag(MouseEventArgs e)
        {
            // スクロール位置を変更
            Size  szDrag   = MouseDragSize;
            Point ptScroll = new Point(
                Math.Max(0, scrollOnDrag_.X - szDrag.Width),
                Math.Max(0, scrollOnDrag_.Y - szDrag.Height)
            );
            AutoScrollPosition = ptScroll;
            base.OnMouseDrag(e);
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void OnMouseDown(MouseEventArgs e)
        {
            if (e.Button == MouseButtons.Left)
            {
                // 拡大
                if (operationMode_ == OperationMode.ZoomIn)
                {
                    ZoomInCore(e.Location);
                }
                // 縮小
                else if (operationMode_ == OperationMode.ZoomOut)
                {
                    ZoomOutCore(e.Location);
                }
            }
            base.OnMouseDown(e);
        }
        #endregion

        #region ピクセルヒント
        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void OnMouseMove(MouseEventArgs e)
        {
            // ピクセルヒント処理
            // フォームがアクティブな時のみ行う
            if (FindForm() == Form.ActiveForm)
            {
                Point pixel = GetPixelLocation(e.Location);
                if (pixel != lastPointedPixel_)
                {
                    // 領域内
                    if (pixel.X != -1 && pixel.Y != -1)
                    {
                        int mipmapLevel = 0;

                        // ミップマップ表示時の位置補正
                        Point location = pixel;
                        if (target_.MipLevel > 1 && isShowMipmap_)
                        {
                            location = GetPixelLocationInMipmap(
                                pixel,
                                colorImage_.Size,
                                target_.MipLevel,
                                target_.MipmapMinSize,
                                mipmapOrientation_,
                                out mipmapLevel
                            );
                        }

                        if (mipmapLevel != -1)
                        {
                            // ピクセル位置をテクスチャの次元で補正する
                            location =
                                CorrectPixelLocation(
                                    location,
                                    target_.Size,
                                    mipmapLevel,
                                    target_.MipmapMinSize,
                                    target_.Data.texture_info.dimension,
                                    target_.Data.texture_info.depth,
                                    DepthIndex
                                );
                        }

                        // 補正後領域内
                        if (location.X != -1 && location.Y != -1)
                        {
                            Bitmap colorImage = RenderingColorImage;
                            Color colorC = colorImage.GetPixel(pixel.X, pixel.Y);

                            Bitmap alphaImage = RenderingAlphaImage;
                            Color colorA = alphaImage.GetPixel(pixel.X, pixel.Y);

                            // ピクセルカラー
                            Color color = Color.FromArgb(colorA.R, colorC.R, colorC.G, colorC.B);

                            OnPixelHint(new TPBitmapImagePixelHintEventArgs(location, color));
                        }
                        // 補正後領域外（無効）
                        else
                        {
                            OnPixelHint(TPBitmapImagePixelHintEventArgs.Empty);
                        }
                    }
                    // 領域外（無効）
                    else
                    {
                        OnPixelHint(TPBitmapImagePixelHintEventArgs.Empty);
                    }
                }
                lastPointedPixel_ = pixel;
            }
            base.OnMouseMove(e);
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void OnMouseLeave(EventArgs e)
        {
            // ピクセルヒントリセット
            ResetPixelHint();
            base.OnMouseLeave(e);
        }

        /// <summary>
        /// ピクセルヒントをリセット。
        /// </summary>
        private void ResetPixelHint()
        {
            // 保存値を既定値に戻す
            lastPointedPixel_ = new Point(-2, -2);

            // 無効ヒントイベント発行
            OnPixelHint(TPBitmapImagePixelHintEventArgs.Empty);
        }

        //---------------------------------------------------------------------
        private void owner_Deactivate(object sender, EventArgs e)
        {
            // ピクセルヒントリセット
            ResetPixelHint();
        }
        #endregion

        #region スクロール
        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void OnScroll(ScrollEventArgs e)
        {
            // トラッキング中のスクロール
            ControlUtility.ApplyTrackingAutoScroll(this, e);
            base.OnScroll(e);
        }
        #endregion

        #region 操作モード
        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void OnPressKeysChanged(EventArgs e)
        {
            // 操作モード変更
            OperationMode mode = OperationMode.Default;
            {
                // 手のひら（Space）
                if (IsPressKeys(Keys.Space))
                {
                    mode = OperationMode.Hand;
                }
                // ズームイン（Ctrl+Space）
                else if (IsPressKeys(Keys.ControlKey, Keys.Space))
                {
                    mode = OperationMode.ZoomIn;
                }
                // ズームアウト（Alt+Space）
                else if (IsPressKeys(Keys.Menu, Keys.Space))
                {
                    mode = OperationMode.ZoomOut;
                }
            }
            // 変更された場合のみ
            if (operationMode_ != mode)
            {
                operationMode_ = mode;

                // カーソル変更
                switch (operationMode_)
                {
                    case OperationMode.Hand:
                        Cursor = ResCursors.Hand;
                        break;
                    case OperationMode.ZoomIn:
                        Cursor = ResCursors.MagnifyPlus;
                        break;
                    case OperationMode.ZoomOut:
                        Cursor = ResCursors.MagnifyMinus;
                        break;
                    default:
                        Cursor = Cursors.Default;
                        break;
                }

                // ドラッグ処理は解除させる
                CancelMouseDragging();
            }
            base.OnPressKeysChanged(e);
        }

        /// <summary>
        /// 操作モード。
        /// </summary>
        private enum OperationMode
        {
            /// <summary>通常。</summary>
            Default,
            /// <summary>手のひら。</summary>
            Hand,
            /// <summary>ズームイン。</summary>
            ZoomIn,
            /// <summary>ズームアウト。</summary>
            ZoomOut
        }
        #endregion
    }

    #region TPBitmapImagePixelHintEvent
    /// <summary>
    /// テクスチャ/パレットビットマップイメージピクセルヒントイベントハンドラデリゲート。
    /// </summary>
    public delegate void TPBitmapImagePixelHintEventHandler(object sender, TPBitmapImagePixelHintEventArgs e);

    /// <summary>
    /// テクスチャ/パレットビットマップイメージピクセルヒントイベントデータ。
    /// </summary>
    public sealed class TPBitmapImagePixelHintEventArgs : EventArgs
    {
        // ピクセルヒント
        private readonly TPBitmapImagePixelHint _pixelHint = TPBitmapImagePixelHint.None;
        // ピクセル位置
        private readonly Point _pixelLocation = new Point(-1, -1);
        // パレット位置
        private readonly int _paletteLocation = -1;
        // カラー
        private readonly Color _color = Color.FromArgb(0, 0, 0, 0);
        // カラーインデクス
        private readonly int _colorIndex = 0;

        /// <summary>
        /// 既定のイベントデータ。
        /// </summary>
        public new static readonly TPBitmapImagePixelHintEventArgs Empty = new TPBitmapImagePixelHintEventArgs();

        /// <summary>
        /// コンストラクタ。
        /// 無効情報用。
        /// </summary>
        private TPBitmapImagePixelHintEventArgs()
        {
        }

        /// <summary>
        /// コンストラクタ。
        /// ピクセルカラー情報用。
        /// </summary>
        public TPBitmapImagePixelHintEventArgs(Point pixelLocation, Color color)
        {
            _pixelHint = TPBitmapImagePixelHint.PixelColor;
            _pixelLocation = pixelLocation;
            _color = color;
        }

        /// <summary>
        /// ピクセルヒント。
        /// </summary>
        public TPBitmapImagePixelHint PixelHint
        {
            get { return _pixelHint; }
        }

        /// <summary>
        /// ピクセル位置。
        /// </summary>
        public Point PixelLocation
        {
            get { return _pixelLocation; }
        }

        /// <summary>
        /// パレット位置。
        /// </summary>
        public int PaletteLocation
        {
            get { return _paletteLocation; }
        }

        /// <summary>
        /// カラー。
        /// </summary>
        public Color Color
        {
            get { return _color; }
        }

        /// <summary>
        /// カラーインデクス。
        /// </summary>
        public int ColorIndex
        {
            get { return _colorIndex; }
        }
    }

    /// <summary>
    /// テクスチャ/パレットビットマップイメージピクセルヒント。
    /// </summary>
    public enum TPBitmapImagePixelHint
    {
        /// <summary>無効。</summary>
        None,
        /// <summary>ピクセルカラー。</summary>
        PixelColor,
    }
    #endregion
}
