﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Diagnostics;
using System.Drawing;
using System.Drawing.Drawing2D;
using System.Linq;
using System.Text;
using App.Data;
using App.ObjectView.List;
using App.Utility;
using App.res;
using nw.g3d.nw4f_3dif;

namespace App.Controls
{
    public static class MaterialTemplatePropertyGridItem
    {
        public static void CreateMaterialItem(MaterialTemplate.Item item)
        {
            Debug.Assert(item.guiObject is Material);
            Material material = (Material)item.guiObject;

            // TODO: 生成でいいか?
            ToolData.nw3de_CustomUI customUI = material.CreateNw3de_CustomUI();

            var mtMaterial = new MtMaterial()
            {
                general = new MtMaterialGeneral()
                {
                    material = material,
                },
                renderState = new MtMaterialRenderState() { material = material },
                sampler = new MtMaterialSampler()
                {
                    samplers = material.sampler_array == null ?
                    new MtSampler[0] :
                    Enumerable.Range(0, material.sampler_array.sampler.Length)
                    .Select(x => new MtSampler()
                    {
                        material = material,
                        index = x
                    }).ToArray(),
                },
                shaderAssign = new MtShaderAssign()
                {
                    material = material,
                    RenderInfos = new MtRenderInfo()
                    {
                        Items = Enumerable.Range(0, material.MaterialShaderAssign.RenderInfos.Count)
                        .Select(x => new MtRenderInfoItem()
                        {
                            material = material,
                            index = x,
                        }).ToArray(),
                    },
                    Options = new MtShaderOption()
                    {
                        Items = Enumerable.Range(0, material.MaterialShaderAssign.ShaderOptions.Count)
                        .Select(x => new MtShaderOptionItem()
                        {
                            material = material,
                            index = x,
                        }).ToArray(),
                    },
                    Params = new MtShaderParameter()
                    {
                        Items = Enumerable.Range(0, material.MaterialShaderAssign.ShaderParams.Count)
                        .Select(x => new MtShaderParameterItem()
                        {
                            material = material,
                            index = x,
                        }).ToArray(),
                    },
                    SamplerAssigns = new MtSamplerAssign()
                    {
                        Items = Enumerable.Range(0, material.MaterialShaderAssign.SamplerAssigns.Count)
                        .Select(x => new MtSamplerAssignItem()
                        {
                            material = material,
                            index = x,
                        }).ToArray(),
                    },
                    AttributeAssigns = new MtAttributeAssign()
                    {
                        Items = Enumerable.Range(0, material.MaterialShaderAssign.AttribAssigns.Count)
                        .Select(x => new MtAttributeAssignItem()
                        {
                            material = material,
                            index = x,
                        }).ToArray(),
                    },
                },
                userData = new MtUserData(material),
                workData = new MtWorkData(material),
                customLabelData = new MtCustomLabelData()
                {
                    RenderInfos = new MtCustomLabel()
                    {
                        Items = customUI.render_info_slots.Select(x => new MtCustomLabelItem() { item = x }).ToArray(),
                    },
                    Options = new MtCustomLabel()
                    {
                        Items = customUI.option_vars.Select(x => new MtCustomLabelItem() { item = x }).ToArray(),
                    },
                    Params = new MtCustomLabel()
                    {
                        Items = customUI.uniform_vars.Select(x => new MtCustomLabelItem() { item = x }).ToArray(),
                    },
                    SamplerAssigns = new MtCustomLabel()
                    {
                        Items = customUI.sampler_vars.Select(x => new MtCustomLabelItem() { item = x }).ToArray(),
                    },
                    AttributeAssigns = new MtCustomLabel()
                    {
                        Items = customUI.attrib_vars.Select(x => new MtCustomLabelItem() { item = x }).ToArray(),
                    },
                    Groups = new MtCustomLabel()
                    {
                        Items = customUI.groups.Select(x => new MtCustomLabelItem() { item = x }).ToArray(),
                    },
                },
            };
            item.PropertyGridItem = mtMaterial;
            foreach (var child in item.children)
            {
                switch (child.category)
                {
                    case MaterialTemplate.Category.MaterialGeneral:
                        child.PropertyGridItem = mtMaterial.general;
                        break;
                    case MaterialTemplate.Category.MaterialRenderState:
                        child.PropertyGridItem = mtMaterial.renderState;
                        break;
                    case MaterialTemplate.Category.MaterialSampler:
                        child.PropertyGridItem = mtMaterial.sampler;
                        break;
                    case MaterialTemplate.Category.MaterialShader:
                        child.PropertyGridItem = mtMaterial.shaderAssign;
                        break;
                    case MaterialTemplate.Category.UserData:
                        child.PropertyGridItem = mtMaterial.userData;
                        break;
                    case MaterialTemplate.Category.WorkData:
                        child.PropertyGridItem = mtMaterial.workData;
                        break;
                    case MaterialTemplate.Category.CustomLabel:
                        child.PropertyGridItem = mtMaterial.customLabelData;
                        break;
                }
            }
        }

        public static void CreateSamplerItem(MaterialTemplate.Item item)
        {
            item.PropertyGridItem =
                new MtSampler()
                {
                    material = (Material)item.guiObject,
                    index = item.index
                };
        }

        public static void CreateShaderSettingItem(MaterialTemplate.Item item)
        {
            item.PropertyGridItem = new MtShaderSetting()
            {
                material = (Material)item.guiObject,
            };
        }

        public static void CreateShaderAssignGroupItem(MaterialTemplate.Item item)
        {
            var copyData = PropertyEdit.ShaderParamControls.ShaderParamControlGroup.CreateGroupCopyData((Material)item.guiObject, item.name, true, true);
            {
                var shaderAssignGruop = new MtShaderAssignGroup()
                {
                    AttributeAssigns = new MtShaderAssignGroupAttributeAssign()
                    {
                        Items = Enumerable.Range(0, copyData.AttribAssigns.Count)
                        .Select(x => new MtShaderAssignGroupAttributeAssignItem()
                        {
                            AttribAssigns = copyData.AttribAssigns,
                            index = x,
                        }).ToArray(),
                    },
                    ShaderParameter = new MtShaderAssignGroupShaderParameter()
                    {
                        Items = Enumerable.Range(0, copyData.ShaderParams.Count)
                        .Select(x => new MtShaderAssignGroupShaderParameterItem()
                        {
                            ShaderParams = copyData.ShaderParams,
                            index = x,
                        }).ToArray(),
                    },
                    SamplerAssigns = new MtShaderAssignGroupSamplerAssign()
                    {
                        Items = Enumerable.Range(0, copyData.SamplerAssigns.Count)
                        .Select(x => new MtShaderAssignGroupSamplerAssignItem()
                        {
                            SamplerAssigns = copyData.SamplerAssigns,
                            index = x,
                        }).ToArray(),
                    },
                    Options = new MtShaderAssignGroupShaderOption()
                    {
                        Items = Enumerable.Range(0, copyData.ShaderOptions.Count)
                        .Select(x => new MtShaderAssignGroupShaderOptionItem()
                        {
                            ShaderOptions = copyData.ShaderOptions,
                            index = x,
                        }).ToArray(),
                    },
                    RenderInfos = new MtShaderAssignGroupRenderInfo()
                    {
                        Items = Enumerable.Range(0, copyData.RenderInfos.Count)
                        .Select(x => new MtShaderAssignGroupRenderInfoItem()
                        {
                            RenderInfos = copyData.RenderInfos,
                            index = x,
                        }).ToArray(),
                    },
                };

                item.PropertyGridItem = shaderAssignGruop;
            }
        }

        public static void CreateTextureItem(MaterialTemplate.Item item)
        {
            item.PropertyGridItem = new MtTexture() { texture = (Texture)item.guiObject, workData = new MtWorkData(item.guiObject) };
        }

        public static void CreateShaderParameterAnimationItem(MaterialTemplate.Item item, Material material)
        {
            var animation = (ShaderParameterAnimation)item.guiObject;
            var ParamAnim = animation.ShaderParamAnims.FirstOrDefault(x => x.mat_name == material.Name);
            var items = new List<MtParamAnimItem>();
            var uniforms = material.MaterialShaderAssign.ShadingModel != null ?
                material.MaterialShaderAssign.ShadingModel.MaterialUniforms().ToDictionary(x => x.id) :
                new Dictionary<string, uniform_varType>();

            foreach (var child in item.children)
            {
                var paramAnimTarget = ParamAnim.ParamAnims.FirstOrDefault(x => x.id == child.name);
                var paramAnimItem = new MtParamAnimItem(paramAnimTarget);
                child.PropertyGridItem = paramAnimItem;
                items.Add(paramAnimItem);

                uniform_varType uniform_var;
                if (uniforms.TryGetValue(paramAnimTarget.id, out uniform_var))
                {
                    if (paramAnimTarget.type == uniform_var.type &&
                        !string.IsNullOrEmpty(uniform_var.Label()))
                    {
                        child.label = uniform_var.Label();
                    }
                }
            }

            item.PropertyGridItem = new MtShaderParameterAnimation() { animation = animation, anims = new MtParamAnims() { Items = items.ToArray() }, workData = new MtWorkData(item.guiObject), };
        }

        public static void CreateTexturePatternAnimationItem(MaterialTemplate.Item item, Material material)
        {
            var animation = (TexturePatternAnimation)item.guiObject;
            var PatternAnim = animation.TexPatternMatAnims.FirstOrDefault(x => x.mat_name == material.Name);
            var items = new List<MtTexturePatternCurveItem>();
            foreach (var child in item.children)
            {
                var patternAnimItem = new MtTexturePatternCurveItem() { curve = PatternAnim.PatternAnimTargets.FirstOrDefault(x => x.sampler_name == child.name) };
                child.PropertyGridItem = patternAnimItem;
                items.Add(patternAnimItem);
            }
            item.PropertyGridItem = new MtTexturePatternAnimation((TexturePatternAnimation)item.guiObject) { anims = new MtPatternAnims() { Items = items.ToArray() }, workData = new MtWorkData(item.guiObject), };
        }

        public static void CreateMaterialVisibilityAnimationItem(MaterialTemplate.Item item, Material material)
        {
            var animation = (MaterialVisibilityAnimation)item.guiObject;
            var visibilityAnim = animation.MaterialVisibilityMatAnims.FirstOrDefault(x => x.mat_name == material.Name);
            item.PropertyGridItem = new MtMaterialVisibilityAnimation() { animation = animation, curve = new MtMaterialVisibilityCurveItem() { curve = visibilityAnim }, workData = new MtWorkData(item.guiObject), };
        }

        public static void CreateMaterialAnimationItem(MaterialTemplate.Item item, Material material)
        {
            // 全般
            var animation = (MaterialAnimation)item.guiObject;
            var materialAnim = animation.PerMaterialAnims.FirstOrDefault(x => x.mat_name == material.Name);

            // シェーダパラメータ
            var paramAnimItems = new List<MtParamAnimItem>();
            foreach (var anim in materialAnim.ParamAnims)
            {
                var paramAnimItem = new MtParamAnimItem(anim);
                paramAnimItems.Add(paramAnimItem);
            }

            // テクスチャパターン
            var patternCurveItems = new List<MtTexturePatternCurveItem>();
            foreach (var anim in materialAnim.PatternAnimTargets)
            {
                var patternAnimItem = new MtTexturePatternCurveItem();
                patternAnimItem.curve = anim;
                patternCurveItems.Add(patternAnimItem);
            }

            // ビジビリティ
            var visibilityCurveItem = new MtMaterialVisibilityCurveItem();
            visibilityCurveItem.curve = materialAnim.MaterialVisibilityMatAnim;

            item.PropertyGridItem = new MtMaterialAnimation()
            {
                animation = animation,
                paramAnims = new MtParamAnims() { Items = paramAnimItems.ToArray() },
                patternAnims = new MtPatternAnims() { Items = patternCurveItems.ToArray() },
                visibilityCurveItem = visibilityCurveItem,
                workData = new MtWorkData(item.guiObject),
            };
        }
    }

    [TypeConverter(typeof(LocalizedPropertyGridConverter))]
    public class MtMaterial
    {
        public override string ToString()
        {
            return string.Empty;
        }

        [ReadOnly(true)]
        [LocalizedPropertyName("ObjectPropertyPanel_Material_General")]
        public MtMaterialGeneral general { get; set; }

        [ReadOnly(true)]
        [LocalizedPropertyName("ObjectPropertyPanel_Material_RenderState")]
        public MtMaterialRenderState renderState { get; set; }

        [ReadOnly(true)]
        [LocalizedPropertyName("ObjectPropertyPanel_Material_Sampler")]
        public MtMaterialSampler sampler { get; set; }


        [ReadOnly(true)]
        [LocalizedPropertyName("ObjectPropertyPanel_Material_Shader")]
        public MtShaderAssign shaderAssign { get; set; }

        [ReadOnly(true)]
        [LocalizedPropertyName("ObjectPropertyPanel_UserData")]
        public MtUserData userData { get; set; }

        [ReadOnly(true)]
        [LocalizedPropertyName("MaterialTemplate_WorkData")]
        public MtWorkData workData { get; set; }

        [ReadOnly(true)]
        [LocalizedPropertyName("MaterialTemplate_CustomLabel")]
        public MtCustomLabelData customLabelData { get; set; }
    }

    [TypeConverter(typeof(LocalizedPropertyGridConverter))]
    public class MtWorkData
    {
        public MtWorkData(GuiObject obj)
        {
            this.obj = obj;
            if (obj.EditColor.HasValue)
            {
                color = new MtWorkColor()
                {
                    color = obj.EditColor.Value.ToColor(),
                };
            }
        }
        public override string ToString()
        {
            return string.Empty;
        }

        public GuiObject obj;

        [ReadOnly(true)]
        [LocalizedPropertyName("MaterialTemplate_EditColor")]
        public MtWorkColor color { get; set; }

        [LocalizedPropertyName("MaterialTemplate_Comment")]
        public string comment { get { return obj.Comment; } }
    }

    [Editor(typeof(MtColorEditor), typeof(System.Drawing.Design.UITypeEditor))]
    [TypeConverter(typeof(LocalizedPropertyGridConverter))]
    public class MtWorkColor : IMtColor
    {
        public Color color;
        public Color ToColor()
        {
            return color;
        }

        public bool HasAlpha()
        {
            return false;
        }

        public override string ToString()
        {
            return color.R + ", " + color.G + ", " + color.B;
        }
    }

    [TypeConverter(typeof(LocalizedPropertyGridConverter))]
    public class MtMaterialGeneral
    {
        public override string ToString()
        {
            return string.Empty;
        }

        public Material material;

        [LocalizedPropertyName("ListView_Material_Column_Visibility")]
        public string visiblity { get { return UIText.FlagYesNo(material.Data.material_info.visibility); } }

        [LocalizedPropertyName("ListView_Material_Column_Compressable")]
        public string compress_enable { get { return UIText.FlagYesNo(material.Data.material_info.compress_enable); } }

        [LocalizedPropertyName("ListView_Material_Column_DisplayFace")]
        public string display_face { get { return UIText.EnumValue(material.Data.render_state.display_face); } }
    }

    [TypeConverter(typeof(LocalizedPropertyGridConverter))]
    public class MtRenderInfo : ArrayTypeDescriptor<MtRenderInfoItem, MtRenderInfo>
    {
        public MtRenderInfoItem[] Items;

        public override IEnumerable<MtRenderInfoItem> items
        {
            get { return Items; }
        }
    }

    [TypeConverter(typeof(LocalizedPropertyGridConverter))]
    public class MtRenderInfoItem
    {
        public int index;
        public Material material;

        public override string ToString()
        {
            return Name;
        }

        private RenderInfo render_info { get { return material.MaterialShaderAssign.RenderInfos[index]; } }

        [LocalizedPropertyName("MaterialTemplate_RenderInfoName")]
        public string Name { get { return render_info.name; } }

        [LocalizedPropertyName("MaterialTemplate_RenderInfoValue")]
        public string Value
        {
            get
            {
                return string.Concat(render_info.values.Select(x => x + " "));
            }
        }
    }

    [TypeConverter(typeof(LocalizedPropertyGridConverter))]
    public class MtMaterialRenderState
    {
        public override string ToString()
        {
            return string.Empty;
        }

        public Material material;

        [LocalizedPropertyName("ListView_Material_Column_RenderStateMode")]
        public string RenderStateMode { get { return UIText.EnumValue(material.Data.render_state.mode); } }

        [LocalizedPropertyName("ListView_Material_Column_DepthTest")]
        public string DepthTest { get { return UIText.FlagEnableDisable(material.Data.render_state.depth_test.enable); } }

        [LocalizedPropertyName("ListView_Material_Column_DepthTestFunc")]
        public string DepthTestFunc { get { return UIText.EnumValue(material.Data.render_state.depth_test.func); } }

        [LocalizedPropertyName("ListView_Material_Column_DepthTestWrite")]
        public string DepthTestWrite { get { return UIText.FlagYesNo(material.Data.render_state.depth_test.write); } }

        [LocalizedPropertyName("ListView_Material_Column_AlphaTest")]
        public string AlphaTest { get { return UIText.FlagEnableDisable(material.Data.render_state.alpha_test.enable); } }

        [LocalizedPropertyName("ListView_Material_Column_AlphaTestFunc")]
        public string AlphaTestFunc { get { return UIText.EnumValue(material.Data.render_state.alpha_test.func); } }

        [LocalizedPropertyName("ListView_Material_Column_AlphaTestValue")]
        public float AlphaTestWrite { get { return material.Data.render_state.alpha_test.value; } }

        [LocalizedPropertyName("ListView_Material_Column_Blend")]
        public string Blend { get { return UIText.EnumValue(material.Data.render_state.blend_mode); } }

        [LocalizedPropertyName("ListView_Material_Column_BlendRgbOp")]
        public string BlendRgbOp { get { return UIText.EnumValue(material.Data.render_state.color_blend.rgb_op); } }

        [LocalizedPropertyName("ListView_Material_Column_BlendRgbSrcFunc")]
        public string BlendRgbSrcFunc { get { return UIText.EnumValue(material.Data.render_state.color_blend.rgb_src_func); } }

        [LocalizedPropertyName("ListView_Material_Column_BlendRgbDstFunc")]
        public string BlendRgbDstFunc { get { return UIText.EnumValue(material.Data.render_state.color_blend.rgb_dst_func); } }

        [LocalizedPropertyName("ListView_Material_Column_BlendAlphaOp")]
        public string BlendAlphaOp { get { return UIText.EnumValue(material.Data.render_state.color_blend.alpha_op); } }

        [LocalizedPropertyName("ListView_Material_Column_BlendRgbSrcFunc")]
        public string BlendAlphaSrcFunc { get { return UIText.EnumValue(material.Data.render_state.color_blend.alpha_src_func); } }

        [LocalizedPropertyName("ListView_Material_Column_BlendRgbDstFunc")]
        public string BlendAlphaDstFunc { get { return UIText.EnumValue(material.Data.render_state.color_blend.alpha_dst_func); } }

        [LocalizedPropertyName("ListView_Material_Column_BlendColor")]
        public MtColor BlendColor { get { return new MtColor(){color = material.Data.render_state.color_blend.const_color }; } }

        [LocalizedPropertyName("ListView_Material_Column_LogicalBlend")]
        public string LogicalBlend { get { return UIText.EnumValue(material.Data.render_state.logical_blend.op); } }
    }

    [TypeConverter(typeof(LocalizedPropertyGridConverter))]
    public class MtMaterialSampler : ArrayTypeDescriptor<MtSampler, MtMaterialSampler>
    {
        public MtSampler[] samplers;

        public override IEnumerable<MtSampler> items
        {
            get { return samplers; }
        }
    }

    [TypeConverter(typeof(LocalizedPropertyGridConverter))]
    public class MtSampler
    {
        public int index;
        public Material material;
        public override string ToString()
        {
            return string.Format(Strings.MaterialTemplate_Sampler, index);
        }

        private samplerType sampler { get { return material.sampler_array.sampler[index]; } }

        [LocalizedPropertyName("MaterialTemplate_SamplerName")]
        public string Name { get { return sampler.name; } }

        [LocalizedPropertyName("MaterialTemplate_SamplerHint")]
        public string Hint { get { return sampler.hint; } }

        [LocalizedPropertyName("MaterialTemplate_SamplerTexture")]
        public string Texture { get { return sampler.tex_name; } }

        [LocalizedPropertyName("MaterialTemplate_SamplerWrapU")]
        public string WrapU { get { return UIText.EnumValue(sampler.wrap.u); } }

        [LocalizedPropertyName("MaterialTemplate_SamplerWrapV")]
        public string WrapV { get { return UIText.EnumValue(sampler.wrap.v); } }

        [LocalizedPropertyName("MaterialTemplate_SamplerWrapW")]
        public string WrapW { get { return UIText.EnumValue(sampler.wrap.w); } }

        [LocalizedPropertyName("MaterialTemplate_SamplerFilterMag")]
        public string FilterMag { get { return UIText.EnumValue(sampler.filter.mag); } }

        [LocalizedPropertyName("MaterialTemplate_SamplerFilterMin")]
        public string FilterMin { get { return UIText.EnumValue(sampler.filter.min); } }

        [LocalizedPropertyName("MaterialTemplate_SamplerFilterMaxAniso")]
        public string FilterMaxAniso { get { return UIText.EnumValue(sampler.filter.max_aniso); } }

        [LocalizedPropertyName("MaterialTemplate_SamplerLodMin")]
        public float LodMin { get { return sampler.lod.min; } }

        [LocalizedPropertyName("MaterialTemplate_SamplerLodMax")]
        public float LodMax { get { return sampler.lod.max; } }

        [LocalizedPropertyName("MaterialTemplate_SamplerLodBias")]
        public float LodBias { get { return sampler.lod.bias; } }
    }

    [TypeConverter(typeof(LocalizedPropertyGridConverter))]
    public class MtShaderSetting
    {
        public override string ToString()
        {
            return string.Empty;
        }
        public Material material;

        [LocalizedPropertyName("ListView_Material_Column_ShaderArchive")]
        public string ShaderArchive { get { return material.MaterialShaderAssign.ShaderDefinitionFileName; } }

        [LocalizedPropertyName("ListView_Material_Column_Program")]
        public string Program { get { return material.MaterialShaderAssign.ShaderName; } }
    }

    [TypeConverter(typeof(LocalizedPropertyGridConverter))]
    public class MtShaderAssign
    {
        public override string ToString()
        {
            return string.Empty;
        }

        public Material material;

        [LocalizedPropertyName("ListView_Material_Column_ShaderArchive")]
        public string ShaderArchive { get { return material.MaterialShaderAssign.ShaderDefinitionFileName; } }

        [LocalizedPropertyName("ListView_Material_Column_Program")]
        public string Program { get { return material.MaterialShaderAssign.ShaderName; } }

        [ReadOnly(true)]
        [LocalizedPropertyName("MaterialTemplate_RenderInfo")]
        public MtRenderInfo RenderInfos { get; set; }

        [ReadOnly(true)]
        [LocalizedPropertyName("MaterialTemplate_ShaderOption")]
        public MtShaderOption Options { get; set; }

        [ReadOnly(true)]
        [LocalizedPropertyName("MaterialTemplate_ShaderParameter")]
        public MtShaderParameter Params { get; set; }

        [ReadOnly(true)]
        [LocalizedPropertyName("MaterialTemplate_ShaderSamplerAssign")]
        public MtSamplerAssign SamplerAssigns { get; set; }

        [ReadOnly(true)]
        [LocalizedPropertyName("MaterialTemplate_AttributeAssign")]
        public MtAttributeAssign AttributeAssigns { get; set; }
    }

    [TypeConverter(typeof(LocalizedPropertyGridConverter))]
    public class MtCustomLabelData
    {
        public override string ToString()
        {
            return string.Empty;
        }

        public Material material;

        [ReadOnly(true)]
        [LocalizedPropertyName("MaterialTemplate_AttributeAssign")]
        public MtCustomLabel AttributeAssigns { get; set; }

        [ReadOnly(true)]
        [LocalizedPropertyName("MaterialTemplate_ShaderOption")]
        public MtCustomLabel Options { get; set; }

        [ReadOnly(true)]
        [LocalizedPropertyName("MaterialTemplate_ShaderParameter")]
        public MtCustomLabel Params { get; set; }

        [ReadOnly(true)]
        [LocalizedPropertyName("MaterialTemplate_ShaderSamplerAssign")]
        public MtCustomLabel SamplerAssigns { get; set; }

        [ReadOnly(true)]
        [LocalizedPropertyName("MaterialTemplate_RenderInfo")]
        public MtCustomLabel RenderInfos { get; set; }

        [ReadOnly(true)]
        [LocalizedPropertyName("MaterialTemplate_Group")]
        public MtCustomLabel Groups { get; set; }
    }

    [TypeConverter(typeof(LocalizedPropertyGridConverter))]
    public class MtAttributeAssign : ArrayTypeDescriptor<MtAttributeAssignItem, MtAttributeAssign>
    {
        public MtAttributeAssignItem[] Items;

        public override IEnumerable<MtAttributeAssignItem> items
        {
            get { return Items; }
        }
    }

    [TypeConverter(typeof(LocalizedPropertyGridConverter))]
    public class MtAttributeAssignItem
    {
        public override string ToString()
        {
            return id;
        }

        public int index;
        public Material material;
        private attrib_assignType attribAssign { get { return material.MaterialShaderAssign.AttribAssigns[index]; } }

        [LocalizedPropertyName("MaterialTemplate_AttributeAssignId")]
        public string id { get { return attribAssign.id; } }

        [LocalizedPropertyName("MaterialTemplate_AttributeAssignAttributeName")]
        public string sampler_name { get { return attribAssign.attrib_name; } }
    }

    [TypeConverter(typeof(LocalizedPropertyGridConverter))]
    public class MtSamplerAssign : ArrayTypeDescriptor<MtSamplerAssignItem, MtSamplerAssign>
    {
        public MtSamplerAssignItem[] Items;

        public override IEnumerable<MtSamplerAssignItem> items
        {
            get { return Items; }
        }
    }

    [TypeConverter(typeof(LocalizedPropertyGridConverter))]
    public class MtSamplerAssignItem
    {
        public override string ToString()
        {
            return id;
        }

        public int index;
        public Material material;
        private sampler_assignType samplerAssign { get { return material.MaterialShaderAssign.SamplerAssigns[index]; } }

        [LocalizedPropertyName("MaterialTemplate_SamplerAssignId")]
        public string id { get { return samplerAssign.id; } }

        [LocalizedPropertyName("MaterialTemplate_SamplerAssignSamplerName")]
        public string sampler_name { get { return samplerAssign.sampler_name; } }
    }

    [TypeConverter(typeof(LocalizedPropertyGridConverter))]
    public class MtShaderParameter : ArrayTypeDescriptor<MtShaderParameterItem, MtShaderParameter>
    {
        public MtShaderParameterItem[] Items;

        public override IEnumerable<MtShaderParameterItem> items
        {
            get { return Items; }
        }
    }

    [TypeConverter(typeof(LocalizedPropertyGridConverter))]
    public class MtShaderParameterItem
    {
        public override string ToString()
        {
            return id;
        }

        public int index;
        public Material material;
        private shader_paramType shaderParam { get { return material.MaterialShaderAssign.ShaderParams[index]; } }

        [LocalizedPropertyName("MaterialTemplate_ShaderParamId")]
        public string id { get { return shaderParam.id; } }

        [LocalizedPropertyName("MaterialTemplate_ShaderParamType")]
        public shader_param_typeType type { get { return shaderParam.type; } }

        [LocalizedPropertyName("MaterialTemplate_ShaderParamValue")]
        public string value { get { return MaterialListView.valueString(shaderParam.type, shaderParam.Value); } }
    }

    [TypeConverter(typeof(LocalizedPropertyGridConverter))]
    public class MtShaderOption : ArrayTypeDescriptor<MtShaderOptionItem, MtShaderOption>
    {
        public MtShaderOptionItem[] Items;

        public override IEnumerable<MtShaderOptionItem> items
        {
            get { return Items; }
        }
    }

    [TypeConverter(typeof(LocalizedPropertyGridConverter))]
    public class MtShaderOptionItem
    {
        public override string ToString()
        {
            return id;
        }

        public int index;
        public Material material;
        private shader_optionType shaderOption { get { return material.MaterialShaderAssign.ShaderOptions[index]; } }

        [LocalizedPropertyName("MaterialTemplate_ShaderOptionId")]
        public string id { get { return shaderOption.id; } }

        [LocalizedPropertyName("MaterialTemplate_ShaderOptionValue")]
        public string value { get { return shaderOption.value; } }
    }

    [TypeConverter(typeof(LocalizedPropertyGridConverter))]
    public class MtCustomLabel : ArrayTypeDescriptor<MtCustomLabelItem, MtCustomLabel>
    {
        public MtCustomLabelItem[] Items;

        public override IEnumerable<MtCustomLabelItem> items
        {
            get { return Items; }
        }
    }

    [TypeConverter(typeof(LocalizedPropertyGridConverter))]
    public class MtCustomLabelItem
    {
        public override string ToString()
        {
            return string.Format("{0}({1})", value, id);
        }

        public CustomItem item;
        //private shader_optionType shaderOption { get { return material.MaterialShaderAssign.ShaderOptions[index]; } }

        [LocalizedPropertyName("MaterialTemplate_CustomLabelId")]
        public string id { get { return item.id; } }

        [LocalizedPropertyName("MaterialTemplate_CustomLabelValue")]
        public string value { get { return item.label; } }
    }

    [TypeConverter(typeof(LocalizedPropertyGridConverter))]
    public class MtUserData : ArrayTypeDescriptor<MtUserDataItem, MtUserData>
    {
        public MtUserData(Material material)
        {
            if (material.UserDataArray == null)
            {
                Items = new MtUserDataItem[] { };
            }
            else
            {
                Items = material.UserDataArray.Data.Select<UserDataArray.UserData, MtUserDataItem>(
                    x =>
                    {
                        if (x.GetType() == typeof(UserDataArray.UserDataInt))
                        {
                            return new MtUserDataInt() { userData = (UserDataArray.UserDataInt)x };
                        }
                        else if (x.GetType() == typeof(UserDataArray.UserDataFloat))
                        {
                            return new MtUserDataFloat() { userData = (UserDataArray.UserDataFloat)x };
                        }
                        else if (x.GetType() == typeof(UserDataArray.UserDataString))
                        {
                            return new MtUserDataString() { userData = (UserDataArray.UserDataString)x };
                        }
                        else if (x.GetType() == typeof(UserDataArray.UserDataWString))
                        {
                            return new MtUserDataWString() { userData = (UserDataArray.UserDataWString)x };
                        }
                        Debug.Assert(false);
                        return null;
                    }).ToArray();
            }
        }

        public MtUserDataItem[] Items;

        public override IEnumerable<MtUserDataItem> items
        {
            get { return Items; }
        }
    }

    public abstract class MtUserDataItem
    {
    }

    [TypeConverter(typeof(LocalizedPropertyGridConverter))]
    public class MtUserDataInt : MtUserDataItem
    {
        public override string ToString()
        {
            return name;
        }

        public UserDataArray.UserDataInt userData;

        [LocalizedPropertyName("MaterialTemplate_UserDataName")]
        public string name { get { return userData.Name; } }

        [LocalizedPropertyName("MaterialTemplate_UserDataType")]
        public string type { get { return "int"; } }

        [LocalizedPropertyName("MaterialTemplate_UserDataValue")]
        public int[] value { get { return userData.Item.ToArray(); } }
    }

    [TypeConverter(typeof(LocalizedPropertyGridConverter))]
    public class MtUserDataFloat : MtUserDataItem
    {
        public override string ToString()
        {
            return name;
        }

        public UserDataArray.UserDataFloat userData;

        [LocalizedPropertyName("MaterialTemplate_UserDataName")]
        public string name { get { return userData.Name; } }

        [LocalizedPropertyName("MaterialTemplate_UserDataType")]
        public string type { get { return "float"; } }

        [LocalizedPropertyName("MaterialTemplate_UserDataValue")]
        public float[] value { get { return userData.Item.ToArray(); } }
    }

    [TypeConverter(typeof(LocalizedPropertyGridConverter))]
    public class MtUserDataString : MtUserDataItem
    {
        public override string ToString()
        {
            return name;
        }

        public UserDataArray.UserDataString userData;

        [LocalizedPropertyName("MaterialTemplate_UserDataName")]
        public string name { get { return userData.Name; } }

        [LocalizedPropertyName("MaterialTemplate_UserDataType")]
        public string type { get { return "string"; } }

        [LocalizedPropertyName("MaterialTemplate_UserDataValue")]
        public string[] value { get { return userData.Item.ToArray(); } }
    }

    [TypeConverter(typeof(LocalizedPropertyGridConverter))]
    public class MtUserDataWString : MtUserDataItem
    {
        public override string ToString()
        {
            return name;
        }

        public UserDataArray.UserDataWString userData;

        [LocalizedPropertyName("MaterialTemplate_UserDataName")]
        public string name { get { return userData.Name; } }

        [LocalizedPropertyName("MaterialTemplate_UserDataType")]
        public string type { get { return "wstring"; } }

        [LocalizedPropertyName("MaterialTemplate_UserDataValue")]
        public string[] value { get { return userData.Item.ToArray(); } }
    }

    [TypeConverter(typeof(LocalizedPropertyGridConverter))]
    public class MtMaterialAnimation
    {
        public MaterialAnimation animation;

        private material_anim_infoType info { get { return animation.Data.material_anim_info; } }

        [LocalizedPropertyName("MaterialTemplate_MaterialAnimation_FrameCount")]
        public int frame_count { get { return info.frame_count; } }

        [LocalizedPropertyName("MaterialTemplate_MaterialAnimation_Loop")]
        public string loop { get { return UIText.FlagYesNo(info.loop); } }

        [LocalizedPropertyName("MaterialTemplate_MaterialAnimation_FrameResolution")]
        public int frame_resolution { get { return info.frame_resolution; } }

        [LocalizedPropertyName("MaterialTemplate_MaterialAnimation_DccPreset")]
        public string dcc_preset { get { return info.dcc_preset; } }

        [LocalizedPropertyName("MaterialTemplate_MaterialAnimation_DccStartFrame")]
        public int dcc_start_frame { get { return info.dcc_start_frame; } }

        [LocalizedPropertyName("MaterialTemplate_MaterialAnimation_DccEndFrame")]
        public int dcc_end_frame { get { return info.dcc_end_frame; } }

        [LocalizedPropertyName("MaterialTemplate_MaterialAnimation_DccFps")]
        public float dcc_fps { get { return info.dcc_fps; } }

        [LocalizedPropertyName("MaterialTemplate_MaterialAnimation_BakeAll")]
        public string bake_all { get { return UIText.FlagYesNo(info.bake_all); } }

        [LocalizedPropertyName("MaterialTemplate_MaterialAnimation_BakeToleranceColor")]
        public float bake_tolerance_color { get { return info.bake_tolerance_color; } }

        [LocalizedPropertyName("MaterialTemplate_MaterialAnimation_BakeToleranceTexScale")]
        public float bake_tolerance_tex_scale { get { return info.bake_tolerance_tex_scale; } }

        [LocalizedPropertyName("MaterialTemplate_MaterialAnimation_BakeToleranceTexRotate")]
        public float bake_tolerance_tex_rotate { get { return info.bake_tolerance_tex_rotate; } }

        [LocalizedPropertyName("MaterialTemplate_MaterialAnimation_BakeToleranceTexTranslate")]
        public float bake_tolerance_tex_translate { get { return info.bake_tolerance_tex_translate; } }

        [LocalizedPropertyName("MaterialTemplate_MaterialAnimation_QuantizeToleranceTexScale")]
        public float quantize_tolerance_tex_scale { get { return info.bake_tolerance_tex_scale; } }

        [LocalizedPropertyName("MaterialTemplate_MaterialAnimation_QuantizeToleranceTexRotate")]
        public float quantize_tolerance_tex_rotate { get { return info.bake_tolerance_tex_rotate; } }

        [LocalizedPropertyName("MaterialTemplate_MaterialAnimation_QuantizeToleranceTexTranslate")]
        public float quantize_tolerance_tex_translate { get { return info.bake_tolerance_tex_translate; } }

        [ReadOnly(true)]
        [LocalizedPropertyName("Enum_GuiObjectID_ShaderParameterAnimation")]
        public MtParamAnims paramAnims { get; set; }

        [ReadOnly(true)]
        [LocalizedPropertyName("Enum_GuiObjectID_TexturePatternAnimation")]
        public MtPatternAnims patternAnims { get; set; }

        [ReadOnly(true)]
        [LocalizedPropertyName("Enum_GuiObjectID_MaterialVisibilityAnimation")]
        public MtMaterialVisibilityCurveItem visibilityCurveItem { get; set; }

        [ReadOnly(true)]
        [LocalizedPropertyName("MaterialTemplate_WorkData")]
        public MtWorkData workData { get; set; }
    }

    [TypeConverter(typeof(LocalizedPropertyGridConverter))]
    public class MtShaderParameterAnimation
    {
        public ShaderParameterAnimation animation;
        private shader_param_anim_infoType info { get { return animation.Data.shader_param_anim_info; } }

        [LocalizedPropertyName("MaterialTemplate_ShaderParameterAnimation_FrameCount")]
        public int frame_count { get { return info.frame_count; } }

        [LocalizedPropertyName("MaterialTemplate_ShaderParameterAnimation_Loop")]
        public string loop { get { return UIText.FlagYesNo(info.loop); } }

        [LocalizedPropertyName("MaterialTemplate_ShaderParameterAnimation_FrameResolution")]
        public int frame_resolution { get { return info.frame_resolution; } }

        [LocalizedPropertyName("MaterialTemplate_ShaderParameterAnimation_BakeAll")]
        public string bake_all { get { return UIText.FlagYesNo(info.bake_all); } }

        [LocalizedPropertyName("MaterialTemplate_ShaderParameterAnimation_BakeToleranceColor")]
        public float bake_tolerance_color { get { return info.bake_tolerance_color; } }

        [LocalizedPropertyName("MaterialTemplate_ShaderParameterAnimation_BakeToleranceTexScale")]
        public float bake_tolerance_tex_scale { get { return info.bake_tolerance_tex_scale; } }

        [LocalizedPropertyName("MaterialTemplate_ShaderParameterAnimation_BakeToleranceTexRotate")]
        public float bake_tolerance_tex_rotate { get { return info.bake_tolerance_tex_rotate; } }

        [LocalizedPropertyName("MaterialTemplate_ShaderParameterAnimation_BakeToleranceTexTranslate")]
        public float bake_tolerance_tex_trancelate { get { return info.bake_tolerance_tex_translate; } }

        [LocalizedPropertyName("MaterialTemplate_ShaderParameterAnimation_QuantizeToleranceTexScale")]
        public float quantize_tolerance_tex_scale { get { return info.quantize_tolerance_tex_scale; } }

        [LocalizedPropertyName("MaterialTemplate_ShaderParameterAnimation_QuantizeToleranceTexRotate")]
        public float quantize_tolerance_tex_rotate { get { return info.quantize_tolerance_tex_rotate; } }

        [LocalizedPropertyName("MaterialTemplate_ShaderParameterAnimation_QuantizeToleranceTexTranslate")]
        public float quantize_tolerance_tex_trancelate { get { return info.quantize_tolerance_tex_translate; } }

        [ReadOnly(true)]
        [LocalizedPropertyName("MaterialTemplate_Animation")]
        public MtParamAnims anims { get; set; }

        [ReadOnly(true)]
        [LocalizedPropertyName("MaterialTemplate_WorkData")]
        public MtWorkData workData { get; set; }
    }

    [TypeConverter(typeof(LocalizedPropertyGridConverter))]
    public class MtTexturePatternAnimation
    {
        public MtTexturePatternAnimation(TexturePatternAnimation animation)
        {
            this.animation = animation;
            patterns = new MtTexturePatterns()
            {
                Items = animation.TexPatterns.Select(x => new MtTexturePatternItem() { name = x.tex_name }).ToArray()
            };
        }

        public TexturePatternAnimation animation;
        private tex_pattern_anim_infoType info { get { return animation.Data.tex_pattern_anim_info; } }

        [LocalizedPropertyName("MaterialTemplate_ShaderParameterAnimation_FrameCount")]
        public int frame_count { get { return info.frame_count; } }


        [LocalizedPropertyName("MaterialTemplate_ShaderParameterAnimation_Loop")]
        public string loop { get { return UIText.FlagYesNo(info.loop); } }

        [LocalizedPropertyName("MaterialTemplate_ShaderParameterAnimation_FrameResolution")]
        public int frame_resolution { get { return info.frame_resolution; } }

        [LocalizedPropertyName("MaterialTemplate_ShaderParameterAnimation_BakeAll")]
        public string bake_all { get { return UIText.FlagYesNo(info.bake_all); } }

        [ReadOnly(true)]
        [LocalizedPropertyName("MaterialTemplate_TexturePatternAnimation_TexturePattern")]
        public MtTexturePatterns patterns { get; set; }

        [ReadOnly(true)]
        [LocalizedPropertyName("MaterialTemplate_Curve")]
        public MtPatternAnims anims { get; set; }

        [ReadOnly(true)]
        [LocalizedPropertyName("MaterialTemplate_WorkData")]
        public MtWorkData workData { get; set; }
    }

    [TypeConverter(typeof(LocalizedPropertyGridConverter))]
    public class MtTexturePatterns : ArrayTypeDescriptor<MtTexturePatternItem, MtTexturePatterns>
    {
        public MtTexturePatternItem[] Items;
        public override IEnumerable<MtTexturePatternItem> items
        {
            get { return Items; }
        }
    }

    [TypeConverter(typeof(LocalizedPropertyGridConverter))]
    public class MtTexturePatternItem
    {
        public override string ToString()
        {
            return name;
        }
        [ReadOnly(true)]
        public string name { get; set; }
    }

    [TypeConverter(typeof(LocalizedPropertyGridConverter))]
    public class MtMaterialVisibilityAnimation
    {
        public MaterialVisibilityAnimation animation;
        private mat_visibility_anim_infoType info { get { return animation.Data.mat_visibility_anim_info; } }

        [LocalizedPropertyName("MaterialTemplate_ShaderParameterAnimation_FrameCount")]
        public int frame_count { get { return info.frame_count; } }

        [LocalizedPropertyName("MaterialTemplate_ShaderParameterAnimation_Loop")]
        public string loop { get { return UIText.FlagYesNo(info.loop); } }

        [LocalizedPropertyName("MaterialTemplate_ShaderParameterAnimation_FrameResolution")]
        public int frame_resolution { get { return info.frame_resolution; } }

        [LocalizedPropertyName("MaterialTemplate_ShaderParameterAnimation_BakeAll")]
        public string bake_all { get { return UIText.FlagYesNo(info.bake_all); } }

        [ReadOnly(true)]
        [LocalizedPropertyName("MaterialTemplate_Curve")]
        public MtMaterialVisibilityCurveItem curve { get; set; }

        [ReadOnly(true)]
        [LocalizedPropertyName("MaterialTemplate_WorkData")]
        public MtWorkData workData { get; set; }
    }

    [TypeConverter(typeof(LocalizedPropertyGridConverter))]
    public class MtTexture
    {
        public Texture texture;
        private texture_infoType info { get { return texture.Data.texture_info; } }
        private Bitmap _colorThumbnail = null;
        private Bitmap _alphaThumbnail = null;
        public override string ToString()
        {
            return texture.FileName;
        }

        [LocalizedPropertyName("ListView_Texture_Column_Name")]
        public string name { get { return texture.Name; } }

        [LocalizedPropertyName("MaterialTemplate_Texture_Color")]
        public Image color {
            get { return _colorThumbnail ?? (_colorThumbnail = texture.GetColorThumbnailCopy()); }
        }

        [LocalizedPropertyName("MaterialTemplate_Texture_Alpha")]
        public Image alpha
        {
            get { return _alphaThumbnail ?? (_alphaThumbnail = texture.GetAlphaThumbnailCopy()); }
        }

        [LocalizedPropertyName("ListView_Texture_Column_Dimension")]
        public string dimension { get { return UIText.EnumValue(info.dimension); } }

        [LocalizedPropertyName("ListView_Texture_Column_Format")]
        public string format { get { return UIText.EnumValue(info.quantize_type); } }

        [LocalizedPropertyName("ListView_Texture_Column_Width")]
        public int width { get { return info.width; } }

        [LocalizedPropertyName("ListView_Texture_Column_Height")]
        public int height { get { return info.height; } }

        [LocalizedPropertyName("ListView_Texture_Column_Depth")]
        public int depth { get { return info.depth; } }

        [LocalizedPropertyName("ListView_Texture_Column_VRAMSize")]
        public string size { get { return DataSize.XBytesText(info.size); } }

        [LocalizedPropertyName("ListView_Texture_Column_MipLevel")]
        public int mip_level { get { return info.mip_level; } }

        [LocalizedPropertyName("ListView_Texture_Column_ComponentSelector")]
        public string comp_sel { get { return string.Concat(info.comp_sel.Select(x => x.ToString())); } }

        [LocalizedPropertyName("ListView_Texture_Column_WeightedCompress")]
        public string weighted_compress { get { return UIText.FlagYesNo(info.weighted_compress); } }

        [ReadOnly(true)]
        [LocalizedPropertyName("MaterialTemplate_WorkData")]
        public MtWorkData workData { get; set; }
    }

    [TypeConverter(typeof(LocalizedPropertyGridConverter))]
    public class MtParamAnims : ArrayTypeDescriptor<MtParamAnimItem, MtParamAnims>
    {
        public MtParamAnimItem[] Items;
        public override IEnumerable<MtParamAnimItem> items
        {
            get { return Items; }
        }
    }

    [TypeConverter(typeof(LocalizedPropertyGridConverter))]
    public class MtParamAnimItem
    {
        public override string ToString()
        {
            return id;
        }

        public MtParamAnimItem(ShaderParameterAnimation.ParamAnim anim)
        {
            this.anim = anim;
            curves = new MtCurves() {
                Items = anim.ParamAnimTargets.Select(x => new MtShaderParameterCurveItem() { curve = x }).ToArray()
            };
        }

        public ShaderParameterAnimation.ParamAnim anim;

        public string id { get { return anim.id; } }

        [LocalizedPropertyName("MaterialTemplate_OriginalHint")]
        public string original_hint { get { return anim.originalHint; } }

        [LocalizedPropertyName("MaterialTemplate_ShaderParamType")]
        public string type { get { return anim.type.ToString(); } }

        [ReadOnly(true)]
        [LocalizedPropertyName("MaterialTemplate_Curve")]
        public MtCurves curves { get; set; }
    }

    [TypeConverter(typeof(LocalizedPropertyGridConverter))]
    public class MtPatternAnimItem
    {
        public override string ToString()
        {
            return id;
        }

        public MtPatternAnimItem(ShaderParameterAnimation.ParamAnim anim)
        {
            this.anim = anim;
            curves = new MtCurves()
            {
                Items = anim.ParamAnimTargets.Select(x => new MtShaderParameterCurveItem() { curve = x }).ToArray()
            };
        }

        public ShaderParameterAnimation.ParamAnim anim;

        public string id { get { return anim.id; } }

        [LocalizedPropertyName("MaterialTemplate_OriginalHint")]
        public string original_hint { get { return anim.originalHint; } }

        [LocalizedPropertyName("MaterialTemplate_ShaderParamType")]
        public string type { get { return anim.type.ToString(); } }

        [ReadOnly(true)]
        [LocalizedPropertyName("MaterialTemplate_Curve")]
        public MtCurves curves { get; set; }
    }

    [TypeConverter(typeof(LocalizedPropertyGridConverter))]
    public class MtPatternAnims : ArrayTypeDescriptor<MtTexturePatternCurveItem, MtPatternAnims>
    {
        public MtTexturePatternCurveItem[] Items;
        public override IEnumerable<MtTexturePatternCurveItem> items
        {
            get { return Items; }
        }
    }

    [TypeConverter(typeof(LocalizedPropertyGridConverter))]
    public class MtCurves : ArrayTypeDescriptor<MtCurveItem, MtCurves>
    {
        public MtCurveItem[] Items;
        public override IEnumerable<MtCurveItem> items
        {
            get { return Items; }
        }
    }

    [TypeConverter(typeof(LocalizedPropertyGridConverter))]
    public abstract class MtCurveItem
    {
    }

    [TypeConverter(typeof(LocalizedPropertyGridConverter))]
    public class MtShaderParameterCurveItem : MtCurveItem
    {
        public override string ToString()
        {
            return "";
        }

        public ShaderParameterAnimation.ParamAnimTarget curve;

        [LocalizedPropertyName("MaterialTemplate_FrameCount")]
        public int count { get { return curve.KeyFrames.Count; } }

        [LocalizedPropertyName("MaterialTemplate_FrameType")]
        public string frame_type { get { return curve.QuantizationInfo.frame_type.ToString(); } }

        [LocalizedPropertyName("MaterialTemplate_FrameType")]
        public string key_type { get { return curve.QuantizationInfo.key_type.ToString(); } }

        [LocalizedPropertyName("MaterialTemplate_CurveScale")]
        public float scale { get { return curve.QuantizationInfo.scale; } }

        [LocalizedPropertyName("MaterialTemplate_CurveOffset")]
        public float offset { get { return curve.QuantizationInfo.offset; } }

        [LocalizedPropertyName("MaterialTemplate_CurvePreWrap")]
        public string pre_wrap { get { return curve.pre_wrap.ToString(); } }

        [LocalizedPropertyName("MaterialTemplate_CurvePostWrap")]
        public string post_wrap { get { return curve.post_wrap.ToString(); } }

        [LocalizedPropertyName("MaterialTemplate_CurveBaked")]
        public string baked { get { return UIText.FlagYesNo(curve.baked); } }
    }

    [TypeConverter(typeof(LocalizedPropertyGridConverter))]
    public class MtTexturePatternCurveItem : MtCurveItem
    {
        public override string ToString()
        {
            return "";
        }

        public TexturePatternAnimation.PatternAnimTarget curve;

        [LocalizedPropertyName("MaterialTemplate_SamplerName")]
        public string sampler_name { get { return curve.sampler_name; } }

        [LocalizedPropertyName("MaterialTemplate_Hint")]
        public string hint { get { return curve.hint; } }

        [LocalizedPropertyName("MaterialTemplate_FrameCount")]
        public int count { get { return curve.KeyFrames.Count; } }

        [LocalizedPropertyName("MaterialTemplate_FrameType")]
        public string frame_type { get { return curve.QuantizationInfo.frame_type.ToString(); } }

        [LocalizedPropertyName("MaterialTemplate_FrameType")]
        public string key_type { get { return curve.QuantizationInfo.key_type.ToString(); } }

        [LocalizedPropertyName("MaterialTemplate_CurveScale")]
        public float scale { get { return curve.QuantizationInfo.scale; } }

        [LocalizedPropertyName("MaterialTemplate_CurveOffset")]
        public float offset { get { return curve.QuantizationInfo.offset; } }

        [LocalizedPropertyName("MaterialTemplate_CurvePreWrap")]
        public string pre_wrap { get { return curve.pre_wrap.ToString(); } }

        [LocalizedPropertyName("MaterialTemplate_CurvePostWrap")]
        public string post_wrap { get { return curve.post_wrap.ToString(); } }

        [LocalizedPropertyName("MaterialTemplate_CurveBaked")]
        public string baked { get { return UIText.FlagYesNo(curve.baked); } }
    }


    [TypeConverter(typeof(LocalizedPropertyGridConverter))]
    public class MtMaterialVisibilityCurveItem : MtCurveItem
    {
        public override string ToString()
        {
            return curve.mat_name;
        }

        public MaterialVisibilityAnimation.MaterialVisibilityMatAnim curve;

        [LocalizedPropertyName("MaterialTemplate_FrameCount")]
        public int count { get { return curve.KeyFrames.Count; } }

        [LocalizedPropertyName("MaterialTemplate_FrameType")]
        public string frame_type { get { return curve.QuantizationInfo.frame_type.ToString(); } }

        [LocalizedPropertyName("MaterialTemplate_FrameType")]
        public string key_type { get { return curve.QuantizationInfo.key_type.ToString(); } }

        [LocalizedPropertyName("MaterialTemplate_CurveScale")]
        public float scale { get { return curve.QuantizationInfo.scale; } }

        [LocalizedPropertyName("MaterialTemplate_CurveOffset")]
        public float offset { get { return curve.QuantizationInfo.offset; } }

        [LocalizedPropertyName("MaterialTemplate_CurvePreWrap")]
        public string pre_wrap { get { return curve.pre_wrap.ToString(); } }

        [LocalizedPropertyName("MaterialTemplate_CurvePostWrap")]
        public string post_wrap { get { return curve.post_wrap.ToString(); } }

        [LocalizedPropertyName("MaterialTemplate_CurveBaked")]
        public string baked { get { return UIText.FlagYesNo(curve.baked); } }
    }

    [TypeConverter(typeof(LocalizedPropertyGridConverter))]
    public class MtHermitCurveItem : MtCurveItem
    {
        public hermite_curveType curve;

        public int count { get { return curve.count; } }

        public string frame_type { get { return curve.frame_type.ToString(); } }

        public string key_type { get { return curve.key_type.ToString(); } }

        public float scale { get { return curve.scale; } }

        public float offset { get { return curve.offset; } }

        public string pre_wrap { get { return curve.pre_wrap.ToString(); } }

        public string post_wrap { get { return curve.post_wrap.ToString(); } }

        public string baked { get { return UIText.FlagYesNo(curve.baked); } }
    }

    public interface IMtColor
    {
        Color ToColor();
        bool HasAlpha();
    }

    [Editor(typeof(MtColorEditor), typeof(System.Drawing.Design.UITypeEditor))]
    [TypeConverter(typeof(LocalizedPropertyGridConverter))]
    public class MtColor : IMtColor
    {
        public float[] color;

        public override string ToString()
        {
            StringBuilder builder = new StringBuilder();
            foreach (var item in color)
            {
                if (builder.Length > 0)
                {
                    builder.Append(", ");
                }
                builder.Append(item);
            }
            return builder.ToString();
        }

        public bool WorkColor = false;

        public Color ToColor()
        {
            if (color.Length == 3)
            {
                return (new RgbaColor(color[0], color[1], color[2], 1)).ToColor();
            }
            else if (color.Length == 4)
            {
                return (new RgbaColor(color[0], color[1], color[2], color[3])).ToColor();
            }

            return new Color();
        }

        public bool HasAlpha()
        {
            return color.Length == 4;
        }
    }

    public class MtColorEditor : System.Drawing.Design.UITypeEditor
    {
        public override bool GetPaintValueSupported(
      ITypeDescriptorContext context)
        {
            // プロパティ ブラウザに対し、
            // カスタム描画を行うことを通知する
            return true;
        }

        public override void PaintValue(System.Drawing.Design.PaintValueEventArgs e)
        {
            // ビットマップを、用意されたサーフェス上に描画する
            IMtColor value = (IMtColor)e.Value;

            if (value == null)
            {

                using (Brush brush = new HatchBrush(HatchStyle.SmallCheckerBoard, Color.LightGray, Color.White))
                {
                    e.Graphics.FillRectangle(brush, e.Bounds);
                }
                return;
            }

            Color color = value.ToColor();
            if (value.HasAlpha())
            {
                Rectangle rcColor = new Rectangle(e.Bounds.X, e.Bounds.Y, e.Bounds.Width / 2, e.Bounds.Height);
                Rectangle rcAlpha = new Rectangle(e.Bounds.X + e.Bounds.Width / 2, e.Bounds.Y, e.Bounds.Width / 2, e.Bounds.Height);
                e.Graphics.FillRectangle(new SolidBrush(color), rcColor);

                using (Brush brush = new HatchBrush(HatchStyle.SmallCheckerBoard, Color.LightGray, Color.Black))
                {
                    e.Graphics.FillRectangle(brush, rcAlpha);
                }
                using (Brush brush = new SolidBrush(Color.FromArgb(color.A, Color.White)))
                {
                    e.Graphics.FillRectangle(brush, rcAlpha);
                }
            }
            else
            {
                e.Graphics.FillRectangle(new SolidBrush(color), e.Bounds.X, e.Bounds.Y, e.Bounds.Width, e.Bounds.Height);
            }
        }
    }

    #region シェーダーアサイングループ

    [TypeConverter(typeof(LocalizedPropertyGridConverter))]
    public class MtShaderAssignGroup
    {
        public override string ToString()
        {
            return string.Empty;
        }

        [ReadOnly(true)]
        [LocalizedPropertyName("MaterialTemplate_AttributeAssign")]
        public MtShaderAssignGroupAttributeAssign AttributeAssigns { get; set; }

        [ReadOnly(true)]
        [LocalizedPropertyName("MaterialTemplate_RenderInfo")]
        public MtShaderAssignGroupRenderInfo RenderInfos { get; set; }

        [ReadOnly(true)]
        [LocalizedPropertyName("MaterialTemplate_ShaderOption")]
        public MtShaderAssignGroupShaderOption Options { get; set; }

        [ReadOnly(true)]
        [LocalizedPropertyName("MaterialTemplate_ShaderParameter")]
        public MtShaderAssignGroupShaderParameter ShaderParameter { get; set; }

        [ReadOnly(true)]
        [LocalizedPropertyName("MaterialTemplate_ShaderSamplerAssign")]
        public MtShaderAssignGroupSamplerAssign SamplerAssigns { get; set; }
    }

    [TypeConverter(typeof(LocalizedPropertyGridConverter))]
    public class MtShaderAssignGroupRenderInfo : ArrayTypeDescriptor<MtShaderAssignGroupRenderInfoItem, MtShaderAssignGroupRenderInfo>
    {
        public MtShaderAssignGroupRenderInfoItem[] Items;

        public override IEnumerable<MtShaderAssignGroupRenderInfoItem> items
        {
            get { return Items; }
        }
    }

    [TypeConverter(typeof(LocalizedPropertyGridConverter))]
    public class MtShaderAssignGroupRenderInfoItem
    {
        public int index;
        public List<Tuple<render_info_slotType, RenderInfo>> RenderInfos;

        public override string ToString()
        {
            return Name;
        }

        private RenderInfo render_info { get { return RenderInfos[index].Item2; } }

        [LocalizedPropertyName("MaterialTemplate_RenderInfoName")]
        public string Name { get { return render_info.name; } }

        [LocalizedPropertyName("MaterialTemplate_RenderInfoValue")]
        public string Value
        {
            get
            {
                return string.Concat(render_info.values.Select(x => x + " "));
            }
        }
    }

    [TypeConverter(typeof(LocalizedPropertyGridConverter))]
    public class MtShaderAssignGroupShaderOption : ArrayTypeDescriptor<MtShaderAssignGroupShaderOptionItem, MtShaderAssignGroupShaderOption>
    {
        public MtShaderAssignGroupShaderOptionItem[] Items;

        public override IEnumerable<MtShaderAssignGroupShaderOptionItem> items
        {
            get { return Items; }
        }
    }

    [TypeConverter(typeof(LocalizedPropertyGridConverter))]
    public class MtShaderAssignGroupShaderOptionItem
    {
        public override string ToString()
        {
            return id;
        }

        public int index;
        public List<Tuple<option_varType, shader_optionType>> ShaderOptions;
        private shader_optionType shaderOption { get { return ShaderOptions[index].Item2; } }

        [LocalizedPropertyName("MaterialTemplate_ShaderOptionId")]
        public string id { get { return shaderOption.id; } }

        [LocalizedPropertyName("MaterialTemplate_ShaderOptionValue")]
        public string value { get { return shaderOption.value; } }
    }

    [TypeConverter(typeof(LocalizedPropertyGridConverter))]
    public class MtShaderAssignGroupSamplerAssign : ArrayTypeDescriptor<MtShaderAssignGroupSamplerAssignItem, MtShaderAssignGroupSamplerAssign>
    {
        public MtShaderAssignGroupSamplerAssignItem[] Items;

        public override IEnumerable<MtShaderAssignGroupSamplerAssignItem> items
        {
            get { return Items; }
        }
    }

    [TypeConverter(typeof(LocalizedPropertyGridConverter))]
    public class MtShaderAssignGroupSamplerAssignItem
    {
        public override string ToString()
        {
            return id;
        }

        public int index;
        public List<Tuple<sampler_varType, sampler_assignType>> SamplerAssigns;
        private sampler_assignType samplerAssign { get { return SamplerAssigns[index].Item2; } }

        [LocalizedPropertyName("MaterialTemplate_SamplerAssignId")]
        public string id { get { return samplerAssign.id; } }

        [LocalizedPropertyName("MaterialTemplate_SamplerAssignSamplerName")]
        public string sampler_name { get { return samplerAssign.sampler_name; } }
    }

    [TypeConverter(typeof(LocalizedPropertyGridConverter))]
    public class MtShaderAssignGroupShaderParameter : ArrayTypeDescriptor<MtShaderAssignGroupShaderParameterItem, MtShaderAssignGroupShaderParameter>
    {
        public MtShaderAssignGroupShaderParameterItem[] Items;

        public override IEnumerable<MtShaderAssignGroupShaderParameterItem> items
        {
            get { return Items; }
        }
    }

    [TypeConverter(typeof(LocalizedPropertyGridConverter))]
    public class MtShaderAssignGroupShaderParameterItem
    {
        public override string ToString()
        {
            return id;
        }

        public int index;
        public List<Tuple<uniform_varType, shader_paramType>> ShaderParams;
        private shader_paramType shaderParam { get { return ShaderParams[index].Item2; } }

        [LocalizedPropertyName("MaterialTemplate_ShaderParamId")]
        public string id { get { return shaderParam.id; } }

        [LocalizedPropertyName("MaterialTemplate_ShaderParamType")]
        public shader_param_typeType type { get { return shaderParam.type; } }

        [LocalizedPropertyName("MaterialTemplate_ShaderParamValue")]
        public string value { get { return MaterialListView.valueString(shaderParam.type, shaderParam.Value); } }
    }

    [TypeConverter(typeof(LocalizedPropertyGridConverter))]
    public class MtShaderAssignGroupAttributeAssign : ArrayTypeDescriptor<MtShaderAssignGroupAttributeAssignItem, MtShaderAssignGroupAttributeAssign>
    {
        public MtShaderAssignGroupAttributeAssignItem[] Items;

        public override IEnumerable<MtShaderAssignGroupAttributeAssignItem> items
        {
            get { return Items; }
        }
    }

    [TypeConverter(typeof(LocalizedPropertyGridConverter))]
    public class MtShaderAssignGroupAttributeAssignItem
    {
        public override string ToString()
        {
            return id;
        }

        public int index;
        public List<Tuple<attrib_varType, attrib_assignType>> AttribAssigns;
        private attrib_assignType attribAssign { get { return AttribAssigns[index].Item2; } }

        [LocalizedPropertyName("MaterialTemplate_AttributeAssignId")]
        public string id { get { return attribAssign.id; } }

        [LocalizedPropertyName("MaterialTemplate_AttributeAssignAttributeName")]
        public string sampler_name { get { return attribAssign.attrib_name; } }
    }

    #endregion

    public abstract class ArrayTypeDescriptor<T, S> : ICustomTypeDescriptor
    {
        public override string ToString()
        {
            return "(" + items.Count().ToString() + ")";
        }

        public PropertyDescriptorCollection GetProperties()
        {
            // コレクションを準備
            PropertyDescriptorCollection collection = new PropertyDescriptorCollection(null);

            foreach (var item in items)
            {
                collection.Add(new CustomPropertyDescriptor<T, S>(item, "[" + collection.Count.ToString() + "]"));
            }

            return collection;
        }

        // 表示対象項目
        public abstract IEnumerable<T> items
        {
            get;
        }

        public String GetClassName()
        {
            return TypeDescriptor.GetClassName(this, true);
        }

        public AttributeCollection GetAttributes()
        {
            return TypeDescriptor.GetAttributes(this, true);
        }

        public String GetComponentName()
        {
            return TypeDescriptor.GetComponentName(this, true);
        }

        public TypeConverter GetConverter()
        {
            return TypeDescriptor.GetConverter(this, true);
        }

        public EventDescriptor GetDefaultEvent()
        {
            return TypeDescriptor.GetDefaultEvent(this, true);
        }

        public PropertyDescriptor GetDefaultProperty()
        {
            return TypeDescriptor.GetDefaultProperty(this, true);
        }

        public object GetEditor(Type editorBaseType)
        {
            return TypeDescriptor.GetEditor(this, editorBaseType, true);
        }

        public EventDescriptorCollection GetEvents(Attribute[] attributes)
        {
            return TypeDescriptor.GetEvents(this, attributes, true);
        }

        public EventDescriptorCollection GetEvents()
        {
            return TypeDescriptor.GetEvents(this, true);
        }

        public object GetPropertyOwner(PropertyDescriptor pd)
        {
            return this;
        }

        public PropertyDescriptorCollection GetProperties(Attribute[] attributes)
        {
            return GetProperties();
        }
    }

    /// <summary>
    /// ArrayTypeDescriptor の表示要素
    /// </summary>
    public class CustomPropertyDescriptor<T, S> : PropertyDescriptor
    {
        public T data;
        public string name;

        public CustomPropertyDescriptor(T data, string name) :
            base(name, new Attribute[] { new ReadOnlyAttribute(true), new DefaultValueAttribute(data.ToString()) })
        {
            this.data = data;
            this.name = name;
        }

        public override bool CanResetValue(object component)
        {
            return false;
        }

        public override Type ComponentType
        {
            get
            {
                return typeof(S);
            }
        }

        public override string DisplayName
        {
            get
            {
                return name;
            }
        }

        public override object GetValue(object component)
        {
            return data;
        }

        public override bool IsReadOnly
        {
            get { return true; }
        }

        public override string Name
        {
            get { return name; }
        }

        public override Type PropertyType
        {
            get { return data.GetType(); }
        }

        public override void ResetValue(object component)
        {
        }

        public override bool ShouldSerializeValue(object component)
        {
            return false;
        }

        public override void SetValue(object component, object value)
        {
        }
    }
}
