﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Drawing;
using System.Windows.Forms;
using App.Data;
using App.Properties;
using App.Utility;
using nw.g3d.nw4f_3dif;

namespace App.Controls
{
    /// <summary>
    /// ドキュメントチェックリストビュークラス。
    /// 全ての項目は Tag プロパティに Document 型を保持します。
    /// また、列情報は以下の条件を持ちます。
    /// 列０（必須）　　：ファイル名。
    /// 列１（任意）　　：種類（拡張子）。
    /// 列２以降（任意）：任意。
    /// </summary>
    [ToolboxItem(false)]
    public sealed class DocumentCheckedListView : UIListView
    {
        // 拡張子テーブル
        public static readonly List<string> _extTable = new List<string>();

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public DocumentCheckedListView()
        {
            CheckBoxes = true;
            GridLines = true;
            SmallImageList = new ImageList()
            {
                ImageSize = new Size(16, 16),
                ColorDepth = ColorDepth.Depth8Bit,
                TransparentColor = Color.Transparent
            };

            SmallImageList.Images.Add(ProjectDocument.DefaultDotExt.Substring(1),			Resources.TreeView_Doc_Project);
            SmallImageList.Images.Add(G3dPath.ModelExtension.Substring(1),					Resources.TreeView_Doc_Model);
            SmallImageList.Images.Add(G3dPath.TextureExtension.Substring(1),				Resources.TreeView_Doc_Texture);
            SmallImageList.Images.Add(G3dPath.SkeletalAnimExtension.Substring(1),			Resources.TreeView_Doc_SkeletalAnimation);
            SmallImageList.Images.Add(G3dPath.MaterialAnimExtension.Substring(1), Resources.TreeView_Doc_MaterialAnimation);
            SmallImageList.Images.Add(G3dPath.ShaderParamAnimExtension.Substring(1),		Resources.TreeView_Doc_ShaderParameterAnimation);
            SmallImageList.Images.Add(G3dPath.ColorAnimExtension.Substring(1),				Resources.TreeView_Doc_ColorAnimation);
            SmallImageList.Images.Add(G3dPath.TexSrtAnimExtension.Substring(1),				Resources.TreeView_Doc_TextureSrtAnimation);
            SmallImageList.Images.Add(G3dPath.TexPatternAnimExtension.Substring(1),			Resources.TreeView_Doc_TexturePatternAnimation);
            SmallImageList.Images.Add(G3dPath.BoneVisibilityAnimExtension.Substring(1),		Resources.TreeView_Doc_BoneVisibilityAnimation);
            SmallImageList.Images.Add(G3dPath.MatVisibilityAnimExtension.Substring(1),		Resources.TreeView_Doc_MaterialVisibilityAnimation);
            SmallImageList.Images.Add(G3dPath.ShapeAnimExtension.Substring(1), Resources.TreeView_Doc_ShapeAnimation);
            SmallImageList.Images.Add(G3dPath.SceneAnimExtension.Substring(1), Resources.TreeView_Doc_SceneAnimation);
            SmallImageList.Images.Add(G3dPath.ShaderDefinitionExtension.Substring(1), Resources.TreeView_Doc_ShaderDefinition);
            SmallImageList.Images.Add(G3dPath.MaterialExtension.Substring(1), Resources.TreeView_Doc_SeparateMaterial);
            SmallImageList.Images.Add("tga", Resources.TreeView_Doc_Texture);
        }

        /// <summary>
        /// タイプコンストラクタ。
        /// </summary>
        static DocumentCheckedListView()
        {
            // 拡張子テーブル
            // Substring(1) は . を削るため
            _extTable.Add(ProjectDocument.DefaultDotExt.Substring(1));
            _extTable.Add(G3dPath.ModelExtension.Substring(1));
            _extTable.Add(G3dPath.TextureExtension.Substring(1));
            _extTable.Add(G3dPath.SkeletalAnimExtension.Substring(1));
            _extTable.Add(G3dPath.MaterialAnimExtension.Substring(1));
            _extTable.Add(G3dPath.ShaderParamAnimExtension.Substring(1));
            _extTable.Add(G3dPath.ColorAnimExtension.Substring(1));
            _extTable.Add(G3dPath.TexSrtAnimExtension.Substring(1));
            _extTable.Add(G3dPath.TexPatternAnimExtension.Substring(1));
            _extTable.Add(G3dPath.BoneVisibilityAnimExtension.Substring(1));
            _extTable.Add(G3dPath.MatVisibilityAnimExtension.Substring(1));
            _extTable.Add(G3dPath.ShapeAnimExtension.Substring(1));
            _extTable.Add(G3dPath.SceneAnimExtension.Substring(1));
            _extTable.Add(G3dPath.ShaderDefinitionExtension.Substring(1));
            _extTable.Add(G3dPath.MaterialExtension.Substring(1));
            _extTable.Add("tga");
        }

        /// <summary>
        /// 全ての項目をチェック。
        /// </summary>
        public void CheckAllItems()
        {
            using (UIControlEventSuppressBlock block = new UIControlEventSuppressBlock())
            {
                BeginUpdate();
                {
                    foreach (ListViewItem item in Items)
                    {
                        item.Checked = true;
                    }
                }
                EndUpdate();
            }
        }

        /// <summary>
        /// 指定された拡張子の全ての項目をチェック。
        /// </summary>
        public void CheckAllItems(string fileExt)
        {
            using (UIControlEventSuppressBlock block = new UIControlEventSuppressBlock())
            {
                BeginUpdate();
                {
                    foreach (ListViewItem item in Items)
                    {
                        Document document = (Document)item.Tag;
                        if (document.FileExt.ToLower() == fileExt)
                        {
                            item.Checked = true;
                        }
                    }
                }
                EndUpdate();
            }
        }

        /// <summary>
        /// 全ての項目のチェックをクリア。
        /// </summary>
        public void ClearAllItems()
        {
            using (UIControlEventSuppressBlock block = new UIControlEventSuppressBlock())
            {
                BeginUpdate();
                {
                    foreach (ListViewItem item in Items)
                    {
                        item.Checked = false;
                    }
                }
                EndUpdate();
            }
        }

        /// <summary>
        /// 指定された拡張子の全ての項目のチェックをクリア。
        /// </summary>
        public void ClearAllItems(string fileExt)
        {
            using (UIControlEventSuppressBlock block = new UIControlEventSuppressBlock())
            {
                BeginUpdate();
                {
                    foreach (ListViewItem item in Items)
                    {
                        Document document = (Document)item.Tag;
                        if (document.FileExt.ToLower() == fileExt)
                        {
                            item.Checked = false;
                        }
                    }
                }
                EndUpdate();
            }
        }

        /// <summary>
        /// 指定された拡張子の項目を含むかどうか。
        /// </summary>
        public bool ContainsItem(string fileExt)
        {
            foreach (ListViewItem item in Items)
            {
                Document document = (Document)item.Tag;
                if (document.FileExt.ToLower() == fileExt)
                {
                    return true;
                }
            }
            return false;
        }

        /// <summary>
        /// 再定義。
        /// </summary>
        [ReadOnly(true)]
        [DefaultValue(true)]
        public new bool CheckBoxes
        {
            get { return base.CheckBoxes; }
            set { base.CheckBoxes = value; }
        }

        /// <summary>
        /// 再定義。
        /// </summary>
        [ReadOnly(true)]
        [DefaultValue(true)]
        public new bool GridLines
        {
            get { return base.GridLines; }
            set { base.GridLines = value; }
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override int OnCompareItem(CompareListViewItemEventArgs e)
        {
            Document document1 = (Document)e.Item1.Tag;
            Document document2 = (Document)e.Item2.Tag;

            // ファイル名
            if (e.ColumnIndex == 0)
            {
                // 名前で比較
                int compareTitle = string.Compare(document1.Name, document2.Name);
                if (compareTitle != 0) { return compareTitle; }

                // 種類で比較
                return e.CompareInt(
                    _extTable.IndexOf(document1.FileExt),
                    _extTable.IndexOf(document2.FileExt)
                );
            }
            // 種類
            else if (e.ColumnIndex == 1)
            {
                var v1 = _extTable.IndexOf(document1.FileExt.Substring(0, 3)); // xxxa,xxxbのabを除く
                var v2 = _extTable.IndexOf(document2.FileExt.Substring(0, 3));
                return (v1 == v2)
                    ? e.CompareString(document1.FileExt, document2.FileExt)
                    : e.CompareInt(v1, v2);
            }
            return base.OnCompareItem(e);
        }
    }

    //=========================================================================
    /// <summary>
    /// 保存ファイル数プロパティデータクラス。
    /// </summary>
    [Serializable]
    [TypeConverter(typeof(LocalizedPropertyGridConverter))]
    [DefaultProperty("Total")]
    public class IntermediateFileCountPropertyData
    {
        // 合計データ用テーブルキー
        private const string _totalDataKey = "TOTALDATA";
        // カウントデータテーブル
        protected readonly Dictionary<string, CountData> _countDatas = new Dictionary<string, CountData>();

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public IntermediateFileCountPropertyData(DocumentCheckedListView fileList)
        {
            // テーブル作成
            foreach (var ext in DocumentCheckedListView._extTable)
            {
                _countDatas.Add(ext, new CountData());
            }
            _countDatas.Add(_totalDataKey,		   new CountData());

            // カウントデータ設定
            foreach (ListViewItem item in fileList.Items)
            {
                string fileExt = ((Document)item.Tag).FileExt.ToLower();

                // アスキー・バイナリの部分を削る
                fileExt =
                    G3dPath.IsPath("." + fileExt) ?
                        PathUtility.MakeNoFormatIfName(fileExt) :
                        fileExt;

                CountData filesData;
                _countDatas.TryGetValue(fileExt, out filesData);
                CountData totalData = _countDatas[_totalDataKey];

                if (item.Checked)
                {
                    if (filesData != null)
                    {
                        filesData.SaveCount++;
                    }
                    totalData.SaveCount++;
                }

                if (filesData != null)
                {
                    filesData.TotalCount++;
                }
                totalData.TotalCount++;
            }
        }

        [PropertyOrder(0)]
        [LocalizedCategory("IO_SaveFileCountPropertyCategory")]
        public string fmd
        {
            get { return _countDatas[G3dPath.ModelExtension.Substring(1)].ToString(); }
        }

        [PropertyOrder(0)]
        [LocalizedCategory("IO_SaveFileCountPropertyCategory")]
        public string ftx
        {
            get { return _countDatas[G3dPath.TextureExtension.Substring(1)].ToString(); }
        }

        [PropertyOrder(0)]
        [LocalizedCategory("IO_SaveFileCountPropertyCategory")]
        public string fsk
        {
            get { return _countDatas[G3dPath.SkeletalAnimExtension.Substring(1)].ToString(); }
        }

        [PropertyOrder(0)]
        [LocalizedCategory("IO_SaveFileCountPropertyCategory")]
        public string fma
        {
            get { return _countDatas[G3dPath.MaterialAnimExtension.Substring(1)].ToString(); }
        }

        [PropertyOrder(0)]
        [LocalizedCategory("IO_SaveFileCountPropertyCategory")]
        public string fsp
        {
            get { return _countDatas[G3dPath.ShaderParamAnimExtension.Substring(1)].ToString(); }
        }

        [PropertyOrder(0)]
        [LocalizedCategory("IO_SaveFileCountPropertyCategory")]
        public string fcl
        {
            get { return _countDatas[G3dPath.ColorAnimExtension.Substring(1)].ToString(); }
        }

        [PropertyOrder(0)]
        [LocalizedCategory("IO_SaveFileCountPropertyCategory")]
        public string fts
        {
            get { return _countDatas[G3dPath.TexSrtAnimExtension.Substring(1)].ToString(); }
        }

        [PropertyOrder(0)]
        [LocalizedCategory("IO_SaveFileCountPropertyCategory")]
        public string ftp
        {
            get { return _countDatas[G3dPath.TexPatternAnimExtension.Substring(1)].ToString(); }
        }

        [PropertyOrder(0)]
        [LocalizedCategory("IO_SaveFileCountPropertyCategory")]
        public string fvb
        {
            get { return _countDatas[G3dPath.BoneVisibilityAnimExtension.Substring(1)].ToString(); }
        }

        [PropertyOrder(0)]
        [LocalizedCategory("IO_SaveFileCountPropertyCategory")]
        public string fvm
        {
            get { return _countDatas[G3dPath.MatVisibilityAnimExtension.Substring(1)].ToString(); }
        }

        [PropertyOrder(0)]
        [LocalizedCategory("IO_SaveFileCountPropertyCategory")]
        public string fsh
        {
            get { return _countDatas[G3dPath.ShapeAnimExtension.Substring(1)].ToString(); }
        }

        [PropertyOrder(0)]
        [LocalizedCategory("IO_SaveFileCountPropertyCategory")]
        public string fsn
        {
            get { return _countDatas[G3dPath.SceneAnimExtension.Substring(1)].ToString(); }
        }

        [PropertyOrder(0)]
        [LocalizedCategory("IO_SaveFileCountPropertyCategory")]
        public string fsd
        {
            get { return _countDatas[G3dPath.ShaderDefinitionExtension.Substring(1)].ToString(); }
        }

        [PropertyOrder(0)]
        [LocalizedCategory("IO_SaveFileCountPropertyCategory")]
        public string fmt
        {
            get { return _countDatas[G3dPath.MaterialExtension.Substring(1)].ToString(); }
        }

        [PropertyOrder(9999)]
        [LocalizedCategory("IO_SaveFileCountPropertyCategory")]
        [LocalizedPropertyName("IO_SaveFileCountPropertyTotal")]
        public string Total
        {
            get { return _countDatas[_totalDataKey].ToString(); }
        }

        /// <summary>
        /// カウントデータクラス。
        /// </summary>
        protected sealed class CountData
        {
            /// <summary>保存数。</summary>
            public int SaveCount = 0;
            /// <summary>合計数。</summary>
            public int TotalCount = 0;

            /// <summary>
            /// オーバーライド。
            /// </summary>
            public override string ToString()
            {
                if (TotalCount > 0)
                {
                    return string.Format("{0} / {1}", SaveCount, TotalCount);
                }
                else
                {
                    return string.Empty;
                }
            }
        }
    }

    //=========================================================================
    /// <summary>
    /// 閉じるファイル数プロパティデータクラス。
    /// </summary>
    [Serializable]
    [TypeConverter(typeof(LocalizedPropertyGridConverter))]
    [DefaultProperty("Total")]
    public class SaveFileCountPropertyData : IntermediateFileCountPropertyData
    {
        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public SaveFileCountPropertyData(DocumentCheckedListView fileList):
        base(fileList)
        {
        }

        [PropertyOrder(0)]
        [LocalizedCategory("IO_SaveFileCountPropertyCategory")]
        public string f3pj
        {
            get { return _countDatas[ProjectDocument.DefaultDotExt.Substring(1)].ToString(); }
        }
    }

    //=========================================================================
    /// <summary>
    /// 閉じるファイル数プロパティデータクラス。
    /// </summary>
    [Serializable]
    [TypeConverter(typeof(LocalizedPropertyGridConverter))]
    [DefaultProperty("Total")]
    public class CloseFileCountPropertyData : SaveFileCountPropertyData
    {
        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public CloseFileCountPropertyData(DocumentCheckedListView fileList):
        base(fileList)
        {
        }

        [PropertyOrder(1)]
        [LocalizedCategory("IO_SaveFileCountPropertyCategory")]
        public string tga
        {
            get { return _countDatas["tga"].ToString(); }
        }
    }
}
