﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Drawing;
using System.Windows.Forms;
using System.Linq;
using ConfigCommon;
using App.Data;
using App.Utility;
using nw.g3d.nw4f_3dif;

namespace App.Controls
{
    /// <summary>
    /// ドキュメントを選択して閉じるダイアログクラス。
    /// </summary>
    public sealed partial class UserCommandDialog : OkCancelDialog
    {
        // リスト初期化フラグ
        private bool loaded = false;

        // 閉じるドキュメントリスト
        private readonly List<Document> _targetDocuments = new List<Document>();
        // フォームサイズ
        private static Size _formSize = Size.Empty;

        private HashSet<GuiObjectID> validIDs;
        private HashSet<GuiObjectID> documentIDs;
        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public UserCommandDialog(Document[] documents, string[] exts)
        {
            InitializeComponent();
            validIDs = new HashSet<GuiObjectID>(exts.Select(x => ObjectIDUtility.ExtToId(x)).Where(x => x.HasValue).Select(x => x.Value));
            documentIDs = new HashSet<GuiObjectID>(documents.Select(x => x.ObjectID));


            //DialogUtility.SetHelp(this, "html/Graphics/3DEditor/html/ui/menu/file_menu.html#close_file");

            foreach (Document document in documents)
            {
                var name = document.FileName + DocumentManager.GetSameNameIndexText(document, true);

                ListViewItem item = new ListViewItem(name);
                item.Tag = document;
                item.Checked = false;
                item.ImageKey = PathUtility.MakeNoFormatIfName(document.FileExt);
                item.SubItems.Add(document.FileDotExt.ToLower());
                lvwFileList.Items.Add(item);
            }
            lvwFileList.Sort(false);

            // チェックコンテキストメニュー
            //InitializeContextMenuItem(cmiCheckF3pj, ProjectDocument.DefaultDotExt.Substring(1));
            InitializeContextMenuItem(cmiCheckFmd, G3dPath.ModelExtension.Substring(1), GuiObjectID.Model);
            InitializeContextMenuItem(cmiCheckFtx, G3dPath.TextureExtension.Substring(1), GuiObjectID.Texture);
            InitializeContextMenuItem(cmiCheckFsk, G3dPath.SkeletalAnimExtension.Substring(1), GuiObjectID.SkeletalAnimation);
            InitializeContextMenuItem(cmiCheckFsp, G3dPath.ShaderParamAnimExtension.Substring(1), GuiObjectID.ShaderParameterAnimation);
            InitializeContextMenuItem(cmiCheckFcl, G3dPath.ColorAnimExtension.Substring(1), GuiObjectID.ColorAnimation);
            InitializeContextMenuItem(cmiCheckFts, G3dPath.TexSrtAnimExtension.Substring(1), GuiObjectID.TextureSrtAnimation);
            InitializeContextMenuItem(cmiCheckFtp, G3dPath.TexPatternAnimExtension.Substring(1), GuiObjectID.TexturePatternAnimation);
            InitializeContextMenuItem(cmiCheckFvb, G3dPath.BoneVisibilityAnimExtension.Substring(1), GuiObjectID.BoneVisibilityAnimation);
            InitializeContextMenuItem(cmiCheckFvm, G3dPath.MatVisibilityAnimExtension.Substring(1), GuiObjectID.MaterialVisibilityAnimation);
            InitializeContextMenuItem(cmiCheckFsh, G3dPath.ShapeAnimExtension.Substring(1), GuiObjectID.ShapeAnimation);
            InitializeContextMenuItem(cmiCheckFsn, G3dPath.SceneAnimExtension.Substring(1), GuiObjectID.SceneAnimation);
            InitializeContextMenuItem(cmiCheckFsd, G3dPath.ShaderDefinitionExtension.Substring(1), GuiObjectID.ShaderDefinition);
            //InitializeContextMenuItem(cmiCheckTga, "tga");

            // クリアコンテキストメニュー
            //InitializeContextMenuItem(cmiClearF3pj, ProjectDocument.DefaultDotExt.Substring(1));
            InitializeContextMenuItem(cmiClearFmd, G3dPath.ModelExtension.Substring(1), GuiObjectID.Model);
            InitializeContextMenuItem(cmiClearFtx, G3dPath.TextureExtension.Substring(1), GuiObjectID.Texture);
            InitializeContextMenuItem(cmiClearFsk, G3dPath.SkeletalAnimExtension.Substring(1), GuiObjectID.SkeletalAnimation);
            InitializeContextMenuItem(cmiClearFsp, G3dPath.ShaderParamAnimExtension.Substring(1), GuiObjectID.ShaderParameterAnimation);
            InitializeContextMenuItem(cmiClearFcl, G3dPath.ColorAnimExtension.Substring(1), GuiObjectID.ColorAnimation);
            InitializeContextMenuItem(cmiClearFts, G3dPath.TexSrtAnimExtension.Substring(1), GuiObjectID.TextureSrtAnimation);
            InitializeContextMenuItem(cmiClearFtp, G3dPath.TexPatternAnimExtension.Substring(1), GuiObjectID.TexturePatternAnimation);
            InitializeContextMenuItem(cmiClearFvb, G3dPath.BoneVisibilityAnimExtension.Substring(1), GuiObjectID.BoneVisibilityAnimation);
            InitializeContextMenuItem(cmiClearFvm, G3dPath.MatVisibilityAnimExtension.Substring(1), GuiObjectID.MaterialVisibilityAnimation);
            InitializeContextMenuItem(cmiClearFsh, G3dPath.ShapeAnimExtension.Substring(1), GuiObjectID.ShapeAnimation);
            InitializeContextMenuItem(cmiClearFsn, G3dPath.SceneAnimExtension.Substring(1), GuiObjectID.SceneAnimation);
            InitializeContextMenuItem(cmiClearFsd, G3dPath.ShaderDefinitionExtension.Substring(1), GuiObjectID.ShaderDefinition);
            //InitializeContextMenuItem(cmiClearTga, "tga");

            lvwFileList_VisibleChanged(lvwFileList, EventArgs.Empty);

            // サイズ設定
            if (_formSize != Size.Empty) { Size = _formSize; }
            FormClosed += delegate { _formSize = Size; };

            UpdateState();
        }

        /// <summary>
        /// コンテキストメニュー項目初期化。
        /// </summary>
        private void InitializeContextMenuItem(ToolStripMenuItem menuItem, string fileExt, GuiObjectID id)
        {
            menuItem.Visible = validIDs.Count == 0 || validIDs.Contains(id);
            menuItem.Tag = fileExt;
            menuItem.Enabled = lvwFileList.ContainsItem(fileExt) || lvwFileList.ContainsItem(fileExt + G3dPath.Text) || lvwFileList.ContainsItem(fileExt + G3dPath.Binary);
            menuItem.Image = Const.IconImages[fileExt];
        }

        /// <summary>
        /// 閉じるドキュメントリスト。
        /// </summary>
        public List<Document> TargetDocuments
        {
            get { return _targetDocuments; }
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override bool OnResultOk()
        {
            _targetDocuments.Clear();
            foreach (ListViewItem item in lvwFileList.CheckedItems)
            {
                _targetDocuments.Add((Document)item.Tag);
            }
            return true;
        }

        private void lvwFileList_ItemChecked(object sender, ItemCheckedEventArgs e)
        {
            if (loaded)
            {
                UpdateState();
            }
        }

        /// <summary>
        /// カウンタ更新。
        /// </summary>
        private void UpdateState()
        {
            // プロパティグリッドを更新するだけ
            pgdFileCount.SelectedObject = new IntermediateFileCountPropertyData(lvwFileList);

            // チェックされていなければ OK ボタンは有効にならない
            btnOK.Enabled = lvwFileList.CheckedItems.Count > 0;
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void OnLoad(EventArgs e)
        {
            // 初期化終了
            // ListView.ItemChecked イベントが初回表示時に必ず呼ばれてしまうのを回避...
            loaded = true;
            base.OnLoad(e);
        }

        //---------------------------------------------------------------------
        // チェック状態変更
        private void cmiCheck_Click(object sender, EventArgs e)
        {
            string fileExt = (string)((ToolStripMenuItem)sender).Tag;

            // 全て
            if (fileExt == null)
            {
                lvwFileList.CheckAllItems();
            }
            // 種類別
            else
            {
                lvwFileList.CheckAllItems(fileExt);
                lvwFileList.CheckAllItems(fileExt + G3dPath.Text);
                lvwFileList.CheckAllItems(fileExt + G3dPath.Binary);
            }
            UpdateState();
        }

        private void cmiClear_Click(object sender, EventArgs e)
        {
            string fileExt = (string)((ToolStripMenuItem)sender).Tag;

            // 全て
            if (fileExt == null)
            {
                lvwFileList.ClearAllItems();
            }
            // 種類別
            else
            {
                lvwFileList.ClearAllItems(fileExt);
                lvwFileList.ClearAllItems(fileExt + G3dPath.Text);
                lvwFileList.ClearAllItems(fileExt + G3dPath.Binary);
            }
            UpdateState();
        }

        private void lvwFileList_VisibleChanged(object sender, EventArgs e)
        {
            lvwFileList.BeginInvokeOrExecute(new Action(() =>
                {
                    if (lvwFileList.ClientSize.Width > 0)
                    {
                        clhFileName.Width = lvwFileList.ClientSize.Width - clhFileType.Width;
                    }
                }));
        }
    }
}
