﻿using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Drawing;
using System.Data;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Windows.Forms;
using Microsoft.WindowsAPICodePack.Dialogs;

namespace App.Controls
{
    public partial class FileIoUserCommandDialog : OkCancelDialog
    {
        /// <summary>
        /// Tag に設定するデータ
        /// </summary>
        public class TagData
        {
            public TeamConfig.FileIo.UserCommand UserCommand;

            public TagData(TeamConfig.FileIo.UserCommand userCommand)
            {
                UserCommand = userCommand;
            }
        }

        const int EM_SETCUEBANNER = 0x1500 + 1;

        private List<Func<bool>> acceptable = new List<Func<bool>>();

        public FileIoUserCommandDialog()
        {
            InitializeComponent();
        }

        protected override void OnLoad(EventArgs e)
        {
            base.OnLoad(e);

            var data = (TagData)Tag;
            tbxUserCommandName.Text = data.UserCommand.Name;
            tbxUserCommandFilter.Text = data.UserCommand.Filter;
            tbxUserCommandArgs.Text = data.UserCommand.Args;
            cbxUserCommandReload.Checked = data.UserCommand.Reload;
            cbxUserCommandForceUnloadAndLoad.Checked = data.UserCommand.ForceUnloadAndLoad;
            cbxUserCommandTemporary.Checked = data.UserCommand.Temporary;
            tbxUserCommandShortCut.Text = data.UserCommand.ShortCut;
            cbxUserCommandFileTreeShortCut.Checked = data.UserCommand.FileTreeShortCut;
            tbxUserCommandPath.Text = data.UserCommand.PathXml;

            if (tbxUserCommandFilter.IsHandleCreated)
            {
                var ptr = System.Runtime.InteropServices.Marshal.StringToBSTR(App.res.Strings.FileIoUserCommand_FilterCueBanner);
                Win32.NativeMethods.SendMessage(tbxUserCommandFilter.Handle, EM_SETCUEBANNER, IntPtr.Zero, ptr);
            }

            // 入力必須項目に設定。
            if (tbxUserCommandPath.IsHandleCreated)
            {
                var ptr = System.Runtime.InteropServices.Marshal.StringToBSTR(App.res.Strings.MandatoryField);
                Win32.NativeMethods.SendMessage(tbxUserCommandPath.Handle, EM_SETCUEBANNER, IntPtr.Zero, ptr);
                acceptable.Add(() => { return !string.IsNullOrEmpty(tbxUserCommandPath.Text); });
            }
        }

        protected override void OnClosing(CancelEventArgs e)
        {
            base.OnClosing(e);

            // e.Cancel 操作は base.OnClosing() 後に行う必要がある。
            if ((DialogResult == DialogResult.OK) && !acceptable.All(x => x()))
            {
                e.Cancel = true;
            }
        }

        protected override void OnClosed(EventArgs e)
        {
            base.OnClosed(e);

            // 変更を適用。
            if (DialogResult == DialogResult.OK)
            {
                // null 文字から空文字への変更を回避。
                Func<string, string, string> CorrectString = (oldValue, newValue) =>
                {
                    return ((oldValue == null) && string.IsNullOrEmpty(newValue)) ? null : newValue;
                };

                var data = (TagData)Tag;
                data.UserCommand.Name = CorrectString(data.UserCommand.Name, tbxUserCommandName.Text);
                data.UserCommand.Filter = CorrectString(data.UserCommand.Filter, tbxUserCommandFilter.Text);
                data.UserCommand.Args = CorrectString(data.UserCommand.Args, tbxUserCommandArgs.Text);
                data.UserCommand.Reload = cbxUserCommandReload.Checked;
                data.UserCommand.ForceUnloadAndLoad = cbxUserCommandForceUnloadAndLoad.Checked;
                data.UserCommand.Temporary = cbxUserCommandTemporary.Checked;
                data.UserCommand.ShortCut = CorrectString(data.UserCommand.ShortCut, tbxUserCommandShortCut.Text);
                data.UserCommand.FileTreeShortCut = cbxUserCommandFileTreeShortCut.Checked;
                data.UserCommand.PathXml = CorrectString(data.UserCommand.PathXml, tbxUserCommandPath.Text);
            }
        }

        private void btnUserCommandPath_Click(object sender, EventArgs e)
        {
            using (var dialog = new CommonOpenFileDialog()
            {
                Title = res.Strings.IO_SpecifyTargetFile,
                IsFolderPicker = false,
                EnsureFileExists = true
            })
            {
                var initPath = System.Environment.ExpandEnvironmentVariables(tbxUserCommandPath.Text);

                if (!string.IsNullOrEmpty(initPath))
                {
                    var initDir = System.IO.Path.GetDirectoryName(initPath);
                    if (!string.IsNullOrEmpty(initDir))
                    {
                        // 期待通りに動作させるにはディレクトリセパレータを正しておく必要がある。
                        dialog.InitialDirectory = initDir.Replace(System.IO.Path.AltDirectorySeparatorChar, System.IO.Path.DirectorySeparatorChar);
                    }

                    var initFile = System.IO.Path.GetFileName(initPath);
                    if (!string.IsNullOrEmpty(initFile))
                    {
                        dialog.DefaultFileName = initFile;
                    }
                }

                if (dialog.ShowDialog(Handle) == CommonFileDialogResult.Ok)
                {
                    tbxUserCommandPath.Text = dialog.FileName;
                }
            }
        }
    }
}
