﻿using System;
using System.ComponentModel;
using System.Drawing;
using System.Windows.Forms;

namespace ColorPicker
{
    /// <summary>
    /// RGBA、HSVA直接編集パネルコントロールクラス。
    /// </summary>
    public sealed class ComponentPanel : UserControl
    {
        // 現在のカラー
        private RGB _srgbColor = new RGB(0,0,0);
        private RGB _linearColor = new RGB(0, 0, 0);
        // ＨＳＶカラー
        private HSV _hsv;
        // 値モード
        private ConfigCommon.ValueMode _valueMode = ConfigCommon.ValueMode.Float;
        // カラーモード
        private ConfigCommon.ColorMode _colorMode = ConfigCommon.ColorMode.RGBA;

        private ColorPicker.TextSliderControl m_rh;
        private ColorPicker.TextSliderControl m_gs;
        private ColorPicker.TextSliderControl m_bv;
        private ColorPicker.TextSliderControl m_alpha;

        /// <summary>
        /// 必要なデザイナ変数です。
        /// </summary>
        private readonly System.ComponentModel.Container components = null;

        /*
        public float Gamma
        {
            set
            {
                m_rh.Gamma = value;
                m_gs.Gamma = value;
                m_bv.Gamma = value;
                m_alpha.Gamma = value; // alpha は補正されないが、ブレンド結果は補正されるのでガンマを施す必要あり。
            }
        }*/

        public bool IsDefaultLinear
        {
            get
            {
                return m_alpha.IsDefaultLinear;
            }
            set
            {
                m_alpha.IsDefaultLinear = value;
            }
        }

        /// <summary>
        /// HDR の上限値を設定する。
        /// </summary>
        public float HDRUpperBound
        {
            set
            {
                m_rh.HDRUpperBound = value;
                m_gs.HDRUpperBound = value;
                m_bv.HDRUpperBound = value;
                m_alpha.HDRUpperBound = value;
            }
        }

        /// <summary>
        /// カラースケールを設定する。
        /// </summary>
        public float ColorScale
        {
            set
            {
                m_rh.ColorScale = value;
                m_gs.ColorScale = value;
                m_bv.ColorScale = value;
                m_alpha.ColorScale = value;
            }
        }

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public ComponentPanel()
        {
            // この呼び出しは、Windows.Forms フォーム デザイナで必要です。
            InitializeComponent();

            // イベント、プロパティ設定
            m_rh.ColorSelected += new TextSliderControl.ColorSelectedHandler(ColorSlider_ColorSelected);
            m_gs.ColorSelected += new TextSliderControl.ColorSelectedHandler(ColorSlider_ColorSelected);
            m_bv.ColorSelected += new TextSliderControl.ColorSelectedHandler(ColorSlider_ColorSelected);
            m_alpha.ColorSelected += new TextSliderControl.ColorSelectedHandler(AlphaSlider_ColorSelected);

            m_rh.SliderMaxValueChanged += new SliderMaxValueChangedEventHandler(rh_MaxColorValueChanged);
            m_gs.SliderMaxValueChanged += new SliderMaxValueChangedEventHandler(gs_MaxColorValueChanged);
            m_bv.SliderMaxValueChanged += new SliderMaxValueChangedEventHandler(bv_MaxColorValueChanged);

            _hsv = HSV.FromRGB(_srgbColor);

            DoubleBuffered = true;
        }

        /// <summary>
        /// 使用されているリソースに後処理を実行します。
        /// </summary>
        protected override void Dispose(bool disposing)
        {
            if (disposing)
            {
                if (components != null)
                {
                    components.Dispose();
                }
            }
            base.Dispose(disposing);
        }

        #region コンポーネント デザイナで生成されたコード
        /// <summary>
        /// デザイナ サポートに必要なメソッドです。このメソッドの内容を
        /// コード エディタで変更しないでください。
        /// </summary>
        private void InitializeComponent()
        {
            m_rh = new ColorPicker.TextSliderControl();
            m_gs = new ColorPicker.TextSliderControl();
            m_bv = new ColorPicker.TextSliderControl();
            m_alpha = new ColorPicker.TextSliderControl();
            SuspendLayout();
            //
            // m_rh
            //
            m_rh.Anchor = ((System.Windows.Forms.AnchorStyles)(((System.Windows.Forms.AnchorStyles.Top | System.Windows.Forms.AnchorStyles.Left)
                | System.Windows.Forms.AnchorStyles.Right)));
            m_rh.Location = new System.Drawing.Point(0, 0);
            m_rh.Name = "m_rh";
            m_rh.Size = new System.Drawing.Size(336, 32);
            m_rh.TabIndex = 0;
            //
            // m_gs
            //
            m_gs.Anchor = ((System.Windows.Forms.AnchorStyles)(((System.Windows.Forms.AnchorStyles.Top | System.Windows.Forms.AnchorStyles.Left)
                | System.Windows.Forms.AnchorStyles.Right)));
            m_gs.ComponentType = ColorPicker.ColorType.G;
            m_gs.Location = new System.Drawing.Point(0, 24);
            m_gs.Name = "m_gs";
            m_gs.Size = new System.Drawing.Size(336, 32);
            m_gs.TabIndex = 1;
            //
            // m_bv
            //
            m_bv.Anchor = ((System.Windows.Forms.AnchorStyles)(((System.Windows.Forms.AnchorStyles.Top | System.Windows.Forms.AnchorStyles.Left)
                | System.Windows.Forms.AnchorStyles.Right)));
            m_bv.ComponentType = ColorPicker.ColorType.B;
            m_bv.Location = new System.Drawing.Point(0, 48);
            m_bv.Name = "m_bv";
            m_bv.Size = new System.Drawing.Size(336, 32);
            m_bv.TabIndex = 2;
            //
            // m_alpha
            //
            m_alpha.Anchor = ((System.Windows.Forms.AnchorStyles)(((System.Windows.Forms.AnchorStyles.Top | System.Windows.Forms.AnchorStyles.Left)
                | System.Windows.Forms.AnchorStyles.Right)));
            m_alpha.ComponentType = ColorPicker.ColorType.A;
            m_alpha.Location = new System.Drawing.Point(0, 72);
            m_alpha.Name = "m_alpha";
            m_alpha.Size = new System.Drawing.Size(336, 24);
            m_alpha.TabIndex = 3;
            //
            // ComponentPanel
            //
            Controls.Add(m_alpha);
            Controls.Add(m_bv);
            Controls.Add(m_gs);
            Controls.Add(m_rh);
            Name = "ComponentPanel";
            Size = new System.Drawing.Size(336, 104);
            ResumeLayout(false);

        }
        #endregion

        #region イベント

        /// <summary>
        /// カラーが変更されたときに発生します。
        /// </summary>
        [Category("カスタム")]
        [Description("カラーが変更されたときに発生します。")]
        public event ColorChangedEventHandler ColorChanged;

        /// <summary>
        /// デフォルトリニアが変更されたときに発生します。
        /// </summary>
        [Category("カスタム")]
        [Description("デフォルトリニアが変更されたときに発生します。")]
        public event EventHandler IsDefaultLinearChanaged
        {
            add
            {
                m_alpha.IsDefaultLinearChanged += value;
            }
            remove
            {
                m_alpha.IsDefaultLinearChanged -= value;
            }
        }
        #endregion

        #region プロパティ

        /// <summary>
        /// 選択カラー。
        /// </summary>
        [Description("選択カラー。")]
        //[DefaultValue(typeof(Color), "Black")]
        public RGB SrgbColor
        {
            get { return _srgbColor; }
            set
            {
                _srgbColor = value;
                _linearColor = value.ToLinearFromSrgb();
                _hsv = HSV.FromRGB(_srgbColor);
                if (Created)
                {
                    m_rh.UpdateColor(GetColorOfColorMode(), _hsv, true, true);
                    m_gs.UpdateColor(GetColorOfColorMode(), _hsv, true, true);
                    m_bv.UpdateColor(GetColorOfColorMode(), _hsv, true, true);
                    m_alpha.UpdateColor(GetColorOfColorMode(), _hsv, true, true);
                    Invalidate();
                }
            }
        }

        public RGB LinearColor
        {
            get { return _linearColor; }
            set
            {
                _linearColor = value;
                _srgbColor = value.ToSrgbFromLinear();
                _hsv = HSV.FromRGB(_srgbColor);
                if (Created)
                {
                    m_rh.UpdateColor(GetColorOfColorMode(), _hsv, true, true);
                    m_gs.UpdateColor(GetColorOfColorMode(), _hsv, true, true);
                    m_bv.UpdateColor(GetColorOfColorMode(), _hsv, true, true);
                    m_alpha.UpdateColor(GetColorOfColorMode(), _hsv, true, true);
                    Invalidate();
                }
            }
        }

        /// <summary>
        /// 選択カラー。
        /// </summary>
        [Description("選択カラー。")]
        [DefaultValue(typeof(Color), "Black")]
        public HSV HSV
        {
            get
            {
                return new HSV(m_rh.ColorValue, m_gs.ColorValue, m_bv.ColorValue);
            }
        }

        /// <summary>
        /// カラー値の表示方式。
        /// </summary>
        [Description("カラー値の表示方式。")]
        [DefaultValue(ConfigCommon.ValueMode.Float)]
        public ConfigCommon.ValueMode ValueMode
        {
            get { return _valueMode; }
            set
            {
                _valueMode = value;
                m_rh.ValueMode = value;
                m_gs.ValueMode = value;
                m_bv.ValueMode = value;
                m_alpha.ValueMode = value;
                if (Created)
                {
                    m_rh.UpdateColor(GetColorOfColorMode(), _hsv, true, true);
                    m_gs.UpdateColor(GetColorOfColorMode(), _hsv, true, true);
                    m_bv.UpdateColor(GetColorOfColorMode(), _hsv, true, true);
                    m_alpha.UpdateColor(GetColorOfColorMode(), _hsv, true, true);
                    Invalidate();
                }
            }
        }

        /// <summary>
        /// カラー成分のモード。
        /// </summary>
        [Description("カラー成分のモード。")]
        [DefaultValue(ConfigCommon.ColorMode.RGBA)]
        public ConfigCommon.ColorMode ColorMode
        {
            get { return _colorMode; }
            set
            {
                _colorMode = value;
                SetControlVisibility();
                m_rh.ColorMode = value;
                m_gs.ColorMode = value;
                m_bv.ColorMode = value;
                m_alpha.ColorMode = value;
                if (Created)
                {
                    InitializePosition(Size);
                    m_rh.UpdateColor(GetColorOfColorMode(), _hsv, true, true);
                    m_gs.UpdateColor(GetColorOfColorMode(), _hsv, true, true);
                    m_bv.UpdateColor(GetColorOfColorMode(), _hsv, true, true);
                    m_alpha.UpdateColor(GetColorOfColorMode(), _hsv, true, true);
                    Invalidate();
                }
            }
        }

        public bool IsLinear()
        {
            return ColorMode == ConfigCommon.ColorMode.LinearRgba || ColorMode == ConfigCommon.ColorMode.LinearRgb;
        }

        public RGB GetColorOfColorMode()
        {
            if (ColorMode == ConfigCommon.ColorMode.LinearRgba || ColorMode == ConfigCommon.ColorMode.LinearRgb)
            {
                return LinearColor;
            }
            else
            {
                return SrgbColor;
            }
        }

        /// <summary>
        /// テキストボックスを右側に表示するかどうかのフラグ。
        /// </summary>
        [Description("テキストボックスを右側に表示するかどうかを示します。")]
        [DefaultValue(false)]
        public bool TextBoxIsRight
        {
            get { return m_rh.TextBoxIsRight; }
            set
            {
                m_rh.TextBoxIsRight = value;
                m_gs.TextBoxIsRight = value;
                m_bv.TextBoxIsRight = value;
                m_alpha.TextBoxIsRight = value;
            }
        }

        #endregion

        #region オーバーライド
        /// <summary>
        /// OnHandleCreatedのオーバーライド（コントロールの初期化）。
        /// </summary>
        /// <param name="e"></param>
        protected override void OnHandleCreated(EventArgs e)
        {
            base.OnHandleCreated (e);

            InitializeControl();
        }

        /// <summary>
        /// デザインモード時は親を再描画
        /// </summary>
        /// <param name="e"></param>
        protected override void OnSizeChanged(EventArgs e)
        {
            base.OnSizeChanged (e);
            InitializePosition( Size );
            if( DesignMode )
            {
                m_rh.UpdateColor(GetColorOfColorMode(), _hsv, true, false);
                m_gs.UpdateColor(GetColorOfColorMode(), _hsv, true, false);
                m_bv.UpdateColor(GetColorOfColorMode(), _hsv, true, false);
                Invalidate(true);
            }
        }

        #endregion

        /// <summary>
        /// イベント発行せずに、現在のカラーを変更する。
        /// </summary>
        public void SetSrgbColor(RGB color)
        {
            _srgbColor = color;
            _linearColor = color.ToLinearFromSrgb();
            _hsv = HSV.FromRGB(color);
            UpdateSrgbColor(_srgbColor, true);
            Invalidate();
        }

        public void SetLinearColor(RGB color)
        {
            _linearColor = color;
            _srgbColor = color.ToSrgbFromLinear();
            _hsv = HSV.FromRGB(_srgbColor);
            UpdateLinearColor(_linearColor, true);
            Invalidate();
        }

        /// <summary>
        /// RGBスライダーでカラー選択されたときに実行される
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void ColorSlider_ColorSelected(object sender, ColorChangedEventArgs e)
        {
            var select = new RGB(1f, e.Color);
            /*
            if (!m_rh.Color.Equals(select) ||
                !m_gs.Color.Equals(select) ||
                !m_bv.Color.Equals(select))
            {
                UpdateColor(sender);
            }
            */
            Invalidate(true);
            // 更新
            if (sender != this)
            {
                bool redEdited		= true;
                bool greenEdited	= true;
                bool blueEdited		= true;
                bool alphaEdited	= false;

                if ((_colorMode == ConfigCommon.ColorMode.RGBA) || (_colorMode == ConfigCommon.ColorMode.RGB) ||
                    (_colorMode == ConfigCommon.ColorMode.LinearRgb) || (_colorMode == ConfigCommon.ColorMode.LinearRgb))
                {
                    redEdited	= sender.Equals(m_rh);
                    greenEdited	= sender.Equals(m_gs);
                    blueEdited	= sender.Equals(m_bv);
                }

                ChangeColor(sender, new ColorChangedEventArgs(new RGB(_srgbColor.A, e.Color), e.EditFixed, redEdited, greenEdited, blueEdited, alphaEdited, IsLinear()));
            }
        }

        /// <summary>
        /// Alphaスライダーでカラー選択されたときに実行される
        /// </summary>
        private void AlphaSlider_ColorSelected(object sender, ColorChangedEventArgs e)
        {
            if (_srgbColor.A != e.Color.A)
            {
                m_alpha.Invalidate();
            }
            if (sender != this)
            {
                bool redEdited		= false;
                bool greenEdited	= false;
                bool blueEdited		= false;
                bool alphaEdited	= true;

                if ((_colorMode == ConfigCommon.ColorMode.RGBA) || (_colorMode == ConfigCommon.ColorMode.A) || (_colorMode == ConfigCommon.ColorMode.HSVA) || (_colorMode == ConfigCommon.ColorMode.LinearRgba))
                {
                    alphaEdited	= sender.Equals(m_alpha);
                }

                ChangeColor(new ColorChangedEventArgs(new RGB(e.Color.A, GetColorOfColorMode()), e.EditFixed, redEdited, greenEdited, blueEdited, alphaEdited, IsLinear()));
            }
        }

        private void rh_MaxColorValueChanged(object sender, SliderMaxValueChangedEventArgs e)
        {
            // RGB スライダーの最大値をそろえる。
            if (e.ComponentType == ColorType.R)
            {
                if (m_gs.ComponentType == ColorType.G)
                {
                    m_gs.SliderMaxValue = e.MaxValue;
                }
                if (m_bv.ComponentType == ColorType.B)
                {
                    m_bv.SliderMaxValue = e.MaxValue;
                }
                UpdateColor(sender);
            }
        }

        private void gs_MaxColorValueChanged(object sender, SliderMaxValueChangedEventArgs e)
        {
            // RGB スライダーの最大値をそろえる。
            if (e.ComponentType == ColorType.G)
            {
                if (m_rh.ComponentType == ColorType.R)
                {
                    m_rh.SliderMaxValue = e.MaxValue;
                }
                if (m_bv.ComponentType == ColorType.B)
                {
                    m_bv.SliderMaxValue = e.MaxValue;
                }
                UpdateColor(sender);
            }
        }

        private void bv_MaxColorValueChanged(object sender, SliderMaxValueChangedEventArgs e)
        {
            // RGB スライダーの最大値をそろえる。
            if (e.ComponentType == ColorType.B)
            {
                if (m_rh.ComponentType == ColorType.R)
                {
                    m_rh.SliderMaxValue = e.MaxValue;
                }
                if (m_gs.ComponentType == ColorType.G)
                {
                    m_gs.SliderMaxValue = e.MaxValue;
                }
                UpdateColor(sender);
            }
        }

        /// <summary>
        /// RGBカラーが変更されたときに呼ばれる
        /// </summary>
        private void ChangeColor(object sender, ColorChangedEventArgs e)
        {
            if (e.IsLinear)
            {
                _linearColor = e.Color;
                _srgbColor = _linearColor.ToSrgbFromLinear();
            }
            else
            {
                _srgbColor = e.Color;
                _linearColor = _srgbColor.ToLinearFromSrgb();
            }
            _hsv = HSV.FromRGB(_srgbColor);
            UpdateColor(sender);
            if (ColorChanged != null)
            {
                ColorChanged(this, e);
            }
        }

        /// <summary>
        /// Alphaカラーが変更されたときに呼ばれる
        /// </summary>
        private void ChangeColor(ColorChangedEventArgs e)
        {
            if (e.IsLinear)
            {
                _linearColor = e.Color;
                _srgbColor = _linearColor.ToSrgbFromLinear();
            }
            else
            {
                _srgbColor = e.Color;
                _linearColor = _srgbColor.ToLinearFromSrgb();
            }
            _hsv = HSV.FromRGB(_srgbColor);
            UpdateColor(m_alpha);
            if (ColorChanged != null)
            {
                ColorChanged(this, e);
            }
        }

        /// <summary>
        /// プロパティ値からコントロールの設定を初期化
        /// </summary>
        private void InitializeControl()
        {
            InitializePosition(Size);
            m_rh.UpdateColor(GetColorOfColorMode(), _hsv, true, true);
            m_gs.UpdateColor(GetColorOfColorMode(), _hsv, true, true);
            m_bv.UpdateColor(GetColorOfColorMode(), _hsv, true, true);
            m_alpha.UpdateColor(GetColorOfColorMode(), _hsv, true, true);

            SetControlVisibility();
            Invalidate();
        }

        /// <summary>
        /// ラインの再設定
        /// </summary>
        private void InitializePosition(Size size)
        {
            //	小さくしてるので無視
            if (size.Height < 104)
                return;
            int add = (size.Height - 104) / 3;
            if (add < 0)
                return;
            add += 24;

            int y = 0;
            if (_colorMode == ConfigCommon.ColorMode.A)
            {
                y = add * 3;
                add = -add;
            }
            m_rh.Location = new Point(m_rh.Location.X, y);
            y += add;
            m_gs.Location = new Point(m_gs.Location.X, y);
            y += add;
            m_bv.Location = new Point(m_bv.Location.X, y);
            y += add;
            m_alpha.Location = new Point(m_bv.Location.X, y);
        }

        /// <summary>
        /// コントロールに設定を反映。
        /// </summary>
        private void SetControlVisibility()
        {
            foreach (Control ctrl in Controls)
            {
                ctrl.Enabled = true;
            }
            if (_colorMode == ConfigCommon.ColorMode.HSVA || _colorMode == ConfigCommon.ColorMode.HSV)
            {
                m_rh.ComponentType = ColorType.H;
                m_gs.ComponentType = ColorType.S;
                m_bv.ComponentType = ColorType.V;
            }
            else
            {
                m_rh.ComponentType = ColorType.R;
                m_gs.ComponentType = ColorType.G;
                m_bv.ComponentType = ColorType.B;
            }
            switch (_colorMode)
            {
                case ConfigCommon.ColorMode.HSVA:
                case ConfigCommon.ColorMode.RGBA:
                case ConfigCommon.ColorMode.LinearRgba:
                    m_rh.Visible = true;
                    m_gs.Visible = true;
                    m_bv.Visible = true;
                    m_alpha.Visible = true;
                    break;
                case ConfigCommon.ColorMode.HSV:
                case ConfigCommon.ColorMode.RGB:
                case ConfigCommon.ColorMode.LinearRgb:
                    m_rh.Visible = true;
                    m_gs.Visible = true;
                    m_bv.Visible = true;
                    m_alpha.Visible = false;
                    m_alpha.Enabled = false;
                    break;
                case ConfigCommon.ColorMode.A:
                    m_rh.Visible = false;
                    m_gs.Visible = false;
                    m_bv.Visible = false;
                    m_rh.Enabled = false;
                    m_gs.Enabled = false;
                    m_bv.Enabled = false;
                    m_alpha.Visible = true;
                    break;
            }
        }

        /// <summary>
        /// 指定されたカラーで描画するビットマップとカラーを更新する
        /// </summary>
        private void UpdateColor(object sender)
        {
            HSV hsv = new HSV(m_rh.ColorValue, m_gs.ColorValue, m_bv.ColorValue);
            if (!sender.Equals(m_rh))
                m_rh.UpdateColor(GetColorOfColorMode(), hsv, true, false);
            if (!sender.Equals(m_gs))
                m_gs.UpdateColor(GetColorOfColorMode(), hsv, true, false);
            if (!sender.Equals(m_bv))
                m_bv.UpdateColor(GetColorOfColorMode(), hsv, true, false);
            if (!sender.Equals(m_alpha))
                m_alpha.UpdateColor(GetColorOfColorMode(), hsv, true, false);
            Invalidate();
        }

        /// <summary>
        /// 指定されたカラーで描画するビットマップとカラーを更新する
        /// </summary>
        public void UpdateSrgbColor(RGB color, bool resetPoint)
        {
            _srgbColor = color;
            _linearColor = color.ToLinearFromSrgb();
            _hsv = HSV.FromRGB(_srgbColor);

            if (resetPoint)
            {
                var c = GetColorOfColorMode();
                var m = Math.Max(Math.Max(Math.Max(c.R, c.G), c.B), 1.0f);
                if (m_rh.ComponentType == ColorType.R)
                {
                    m_rh.SliderMaxValue = m;
                }
                if (m_gs.ComponentType == ColorType.G)
                {
                    m_gs.SliderMaxValue = m;
                }
                if (m_bv.ComponentType == ColorType.B)
                {
                    m_bv.SliderMaxValue = m;
                }
                else if (m_bv.ComponentType == ColorType.V)
                {
                    m_bv.SliderMaxValue = Math.Max(_hsv.V, 100.0f);
                }
            }

            m_rh.UpdateColor(GetColorOfColorMode(), _hsv, resetPoint, false);
            m_gs.UpdateColor(GetColorOfColorMode(), _hsv, resetPoint, false);
            m_bv.UpdateColor(GetColorOfColorMode(), _hsv, resetPoint, false);
            m_alpha.UpdateColor(GetColorOfColorMode(), _hsv, resetPoint, false);
            Invalidate();
        }

        public void UpdateLinearColor(RGB color, bool resetPoint)
        {
            _linearColor = color;
            _srgbColor = color.ToSrgbFromLinear();
            _hsv = HSV.FromRGB(_srgbColor);

            if (resetPoint)
            {
                var c = GetColorOfColorMode();
                var m = Math.Max(Math.Max(Math.Max(c.R, c.G), c.B), 1.0f);
                if (m_rh.ComponentType == ColorType.R)
                {
                    m_rh.SliderMaxValue = m;
                }
                if (m_gs.ComponentType == ColorType.G)
                {
                    m_gs.SliderMaxValue = m;
                }
                if (m_bv.ComponentType == ColorType.B)
                {
                    m_bv.SliderMaxValue = m;
                }
                else if (m_bv.ComponentType == ColorType.V)
                {
                    m_bv.SliderMaxValue = Math.Max(_hsv.V, 100.0f);
                }
            }

            m_rh.UpdateColor(GetColorOfColorMode(), _hsv, resetPoint, false);
            m_gs.UpdateColor(GetColorOfColorMode(), _hsv, resetPoint, false);
            m_bv.UpdateColor(GetColorOfColorMode(), _hsv, resetPoint, false);
            m_alpha.UpdateColor(GetColorOfColorMode(), _hsv, resetPoint, false);
            Invalidate();

        }

        /// <summary>
        /// 指定したチャンネルのテキストボックス前景色を設定する
        /// </summary>
        /// <param name="ch">設定するチャンネル</param>
        /// <param name="color">前景色</param>
        public void SetTextBoxForeColor(ColorChannel ch, Color color)
        {
            if (ch == ColorChannel.R)
                m_rh.TextBoxForeColor = color;
            else if (ch == ColorChannel.G)
                m_gs.TextBoxForeColor = color;
            else if (ch == ColorChannel.B)
                m_bv.TextBoxForeColor = color;
            else
                m_alpha.TextBoxForeColor = color;
        }

        /// <summary>
        /// 指定したチャンネルのテキストボックス前景色を取得する
        /// </summary>
        /// <param name="ch">取得するチャンネル</param>
        /// <returns>前景色</returns>
        public Color GetTextBoxForeColor(ColorChannel ch)
        {
            if (ch == ColorChannel.R)
                return m_rh.TextBoxForeColor;
            else if (ch == ColorChannel.G)
                return m_gs.TextBoxForeColor;
            else if (ch == ColorChannel.B)
                return m_bv.TextBoxForeColor;
            else
                return m_alpha.TextBoxForeColor;
        }

        /// <summary>
        /// 指定したチャンネルのテキストボックス背景色を設定する
        /// </summary>
        /// <param name="ch">設定するチャンネル</param>
        /// <param name="color">背景色</param>
        public void SetTextBoxBackColor(ColorChannel ch, Color color)
        {
            if (ch == ColorChannel.R)
                m_rh.TextBoxBackColor = color;
            else if (ch == ColorChannel.G)
                m_gs.TextBoxBackColor = color;
            else if (ch == ColorChannel.B)
                m_bv.TextBoxBackColor = color;
            else
                m_alpha.TextBoxBackColor = color;
        }

        /// <summary>
        /// 指定したチャンネルのテキストボックス背景色を取得する
        /// </summary>
        /// <param name="ch">取得するチャンネル</param>
        /// <returns>背景色</returns>
        public Color GetTextBoxBackColor(ColorChannel ch)
        {
            if (ch == ColorChannel.R)
                return m_rh.TextBoxBackColor;
            else if (ch == ColorChannel.G)
                return m_gs.TextBoxBackColor;
            else if (ch == ColorChannel.B)
                return m_bv.TextBoxBackColor;
            else
                return m_alpha.TextBoxBackColor;
        }

        /// <summary>
        /// 指定したチャンネルのテキストボックスフォントを設定する
        /// </summary>
        /// <param name="ch">設定するチャンネル</param>
        /// <param name="font">フォント</param>
        public void SetTextBoxFont(ColorChannel ch, Font font)
        {
            if (ch == ColorChannel.R)
                m_rh.TextBoxFont = font;
            else if (ch == ColorChannel.G)
                m_gs.TextBoxFont = font;
            else if (ch == ColorChannel.B)
                m_bv.TextBoxFont = font;
            else
                m_alpha.TextBoxFont = font;
        }

        /// <summary>
        /// 指定したチャンネルのテキストボックスフォントを取得する
        /// </summary>
        /// <param name="ch">取得するチャンネル</param>
        /// <returns>フォント</returns>
        public Font GetTextBoxFont(ColorChannel ch)
        {
            if (ch == ColorChannel.R)
                return m_rh.TextBoxFont;
            else if (ch == ColorChannel.G)
                return m_gs.TextBoxFont;
            else if (ch == ColorChannel.B)
                return m_bv.TextBoxFont;
            else
                return m_alpha.TextBoxFont;
        }
    }
}
