﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.ComponentModel;
using System.Drawing;
using System.Drawing.Drawing2D;
using System.Windows.Forms;
using App.Data;
using App.Utility;

namespace App.Controls
{
    /// <summary>
    /// カラー編集用サマリマーククラス。
    /// </summary>
    public sealed class ColorEditSummary : UnselectableControl, IColorPickerTrigger
    {
        // カラー
        private RgbaColor _color = RgbaColor.FromColor(System.Drawing.Color.White);
        // 編集モード
        private ColorEditMode _editMode = ColorEditMode.RGBA;

        /// <summary>
        /// ColorChange
        /// </summary>
        public event EventHandler ColorChange = null;

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public ColorEditSummary()
        {
        }

        public bool IsDefaultLinear
        {
            get;
            set;
        }

        /// <summary>
        /// カラー。
        /// </summary>
        [Category(UIControlHelper.OriginalPropertyCategoryName)]
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        [Description("カラー。")]
        [Browsable(false)]
        public RgbaColor Color
        {
            get { return _color; }
            set
            {
                if(_color != value)
                {
                    _color = value;
                    Invalidate();

                    if(ColorChange != null){
                        ColorChange(this, EventArgs.Empty);
                    }
                }
            }
        }

        /// <summary>
        /// 編集用カラーかどうか。
        /// </summary>
        public bool IsMarkColor
        {
            get { return false; }
        }

        /// <summary>
        /// 編集モード。
        /// </summary>
        [DefaultValue(ColorEditMode.RGBA)]
        [Category(UIControlHelper.OriginalPropertyCategoryName)]
        [Description("編集モード。")]
        public ColorEditMode EditMode
        {
            get { return _editMode; }
            set
            {
                _editMode = value;
                Invalidate();

                // カーソル変更
                Cursor = Cursors.Hand;
            }
        }

        /// <summary>
        /// アルファ値有効フラグ。
        /// </summary>
        [Browsable(false)]
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public bool EnableAlpha
        {
            // TODO: ColorPickerAdapterに対応させるための暫定処理
            get { return _editMode != ColorEditMode.RGB; }
        }

        /// <summary>
        /// HDR の上限。
        /// </summary>
        [DefaultValue(1.0)]
        [Description("HDR の上限。")]
        public float HDRUpperBound
        {
            get;
            set;
        }

        public bool ReadOnly
        {
            get
            {
                return readOnly_;
            }

            set
            {
                readOnly_ = value;
                Invalidate();
            }
        }
        private bool readOnly_ = false;

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override Size DefaultSize
        {
            get { return new Size(64, 64); }
        }



        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void OnPaint(PaintEventArgs e)
        {
            Graphics g = e.Graphics;

            // 背景
            Rectangle rcClient = ClientRectangle;
            g.FillRectangle(SystemBrushes.Control, rcClient);

            // カラー
            Rectangle rcColor = rcClient;
            if (Enabled || ReadOnly)
            {
                // ＲＧＢカラー
                if (_editMode != ColorEditMode.A)
                {
                    Color color = _color.ToSrgbColor(IsDefaultLinear);
                    using (Brush brush = new SolidBrush(color))
                    {
                        g.FillRectangle(brush, rcColor);
                    }
                }
                // Ａ用透明市松模様
                else
                {
                    var light = Utility.ColorUtility.Pow(Utility.ColorUtility.Blend(System.Drawing.Color.LightGray, System.Drawing.Color.White, _color.A), IsDefaultLinear ? 1/2.2:1);
                    var dark = Utility.ColorUtility.Pow(Utility.ColorUtility.Blend(System.Drawing.Color.Black, System.Drawing.Color.White, _color.A), IsDefaultLinear ? 1/2.2:1);
                    using (Brush brush = new HatchBrush(HatchStyle.LargeCheckerBoard, light, dark))
                    {
                        g.FillRectangle(brush, rcColor);
                    }
                }

                GraphicsUtility.DrawRectangle(
                    g,
                    ReadOnly ?
                        SystemPens.ControlDark :
                        Pens.Black,
                    rcColor
                );
            }
            else
            {
                g.FillRectangle(SystemBrushes.Control, rcColor);
                GraphicsUtility.DrawRectangle(g, SystemPens.GrayText, rcColor);
            }
            base.OnPaint(e);
        }

    }
}
