﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.ComponentModel;
using App.Data;

namespace App.Controls
{
    /// <summary>
    /// カラー編集ボタンクラス。
    /// </summary>
    [ToolboxItem(true)]
    [DefaultEvent("SequentialValueChanged")]
    public sealed partial class ColorEditButton : UIUserControl
    {
        // カラー
        private RgbaColor _color = new RgbaColor();//Color.White;

        // カラーピッカーアダプタ
        private readonly ColorPickerAdapter _adapter;

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public ColorEditButton()
        {
            Color = new RgbaColor();
            InitializeComponent();

            // アダプタ作成
            _adapter = new ColorPickerAdapter(cbnButton);
            _adapter.ColorEdit += new ColorEditEventHandler(adapter_ColorEdit);
        }

        public bool IsDefaultLinear
        {
            set
            {
                cbnButton.IsDefaultLinear = value;
            }
        }

        #region プロパティ
        /// <summary>
        /// カラー。
        /// </summary>
        [Category(UIControlHelper.OriginalPropertyCategoryName)]
        [Description("カラー。")]
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public RgbaColor Color
        {
            get { return _color; }
            set
            {
                if (_color != value)
                {
                    _color = value;

                    // フォーム更新
                    UpdateForm();

                    // 設定された値をアダプタに通知
                    _adapter.NotifyUpdate();
                }
            }
        }

        /// <summary>
        /// アルファ値有効フラグ。
        /// </summary>
        public bool EnableAlpha
        {
            get
            {
                return cbnButton.EnableAlpha;
            }

            set
            {
                cbnButton.EnableAlpha = value;
            }
        }

        /// <summary>
        /// カラーピッカーダイアログのタイトル
        /// </summary>
        public string ColorPickerText
        {
            get
            {
                return _adapter.ColorPickerText;
            }
            set
            {
                _adapter.ColorPickerText = value;
            }
        }
        #endregion

        #region イベント
        //---------------------------------------------------------------------
        private static readonly object EVENT_SEQUENTIALVALUECHANGED = new object();

        /// <summary>
        /// 値変更イベント。
        /// </summary>
        [Category(UIControlHelper.OriginalEventCategoryName)]
        [Description("値が変更された時に発生します。")]
        public event SequentialValueChangedEventHandler SequentialValueChanged
        {
            add { base.Events.AddHandler(EVENT_SEQUENTIALVALUECHANGED, value); }
            remove { base.Events.RemoveHandler(EVENT_SEQUENTIALVALUECHANGED, value); }
        }

        /// <summary>
        /// 値変更ハンドラ。
        /// </summary>
        private void OnSequentialValueChanged(SequentialValueChangedEventArgs e)
        {
            SequentialValueChangedEventHandler handler = (SequentialValueChangedEventHandler)base.Events[EVENT_SEQUENTIALVALUECHANGED];
            if (handler != null) { handler(this, e); }
        }
        #endregion

        /// <summary>
        /// フォーム更新。
        /// </summary>
        private void UpdateForm()
        {
            cbnButton.Color = _color;
        }

        #region オーバーライド
        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void OnLoad(EventArgs e)
        {
            // カラー初期値とコントロールの同期を取る
            if (!DesignMode)
            {
                UpdateForm();
            }
            base.OnLoad(e);
        }
        #endregion

        //---------------------------------------------------------------------
        // アダプタ
        private void adapter_ColorEdit(object sender, ColorEditEventArgs e)
        {
            // 内部値は常に更新
            _color = e.Color;

            // 変更後のみフォーム更新
            if (e.EditFixed)
            {
                UpdateForm();
            }
            OnSequentialValueChanged(new SequentialValueChangedEventArgs(!e.EditFixed));
        }
    }
}
