﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.IO;
using System.Linq;
using System.Reflection;
using System.Text;
using System.Windows.Forms;
using System.Xml;
using System.Xml.Serialization;
using App.Controls;
using App.Utility;
using App.res;
using ConfigCommon;
using TeamConfig;

namespace App.ConfigData
{
    public static class ApplicationConfig
    {
        private static string AppDataFolderpath
        {
            get
            {
                var folder = System.Environment.GetFolderPath(Environment.SpecialFolder.ApplicationData);
                return Path.Combine(folder, "Nintendo", "3dEditor");
            }
        }

        /// <summary>
        /// 環境変数展開済みの SettingName
        /// </summary>
        private static string _settingName = null;

        /// <summary>
        /// 環境変数展開済みの SettingName
        /// </summary>
        private static string SettingName
        {
            get
            {
                if (_settingName != null)
                {
                    return _settingName;
                }
                var invalidChars = Path.GetInvalidFileNameChars();
                var name = "";
                if (Preset != null && Preset.FullSettingName != null)
                {
                    name = new string(Preset.FullSettingName.Select(x => invalidChars.Contains(x) ? '_' : x).ToArray());
                }
                _settingName = string.IsNullOrEmpty(name) ? DefaultSettingFilePrefix : name;
                return _settingName;
            }
        }

        private static string GetSettingFilePath(string prefix)
        {
            return Path.Combine(AppDataFolderpath, Application.ProductVersion, prefix);
        }

        private static string DefaultAppSettingFilepath
        {
            get { return GetSettingFilePath(DefaultSettingFilePrefix + SettingFilePostfix); }
        }


        private static string AppSettingFilepath
        {
            get { return GetSettingFilePath(SettingName + SettingFilePostfix); }
        }

        public static string DefaultUserSettingFilePath
        {
            get { return GetSettingFilePath(DefaultSettingFilePrefix + UserSettingFilePostfix); }
        }

        public static string UserSettingFilePath
        {
            get { return GetSettingFilePath(SettingName + UserSettingFilePostfix); }
        }

        public static string TeamConfigDefaultPath
        {
            get
            {
                string path = Environment.GetEnvironmentVariable("NW4F_3DE_TEAM_CONFIG");
                if (path != null && Directory.Exists(path))
                {
                    return path;
                }
                try
                {
                    return Path.Combine(Environment.GetEnvironmentVariable("NW4F_3DEDITOR_ROOT"), @"TeamConfig");
                }
                catch (Exception)
                {
                    return Path.Combine(Path.GetDirectoryName(Assembly.GetEntryAssembly().Location), "TeamConfig");
                }
            }
        }

        public static string PlatformPresetDirectoryPath
        {
            get
            {
                return Path.Combine(Environment.GetEnvironmentVariable("NW4F_3DEDITOR_ROOT"), @"Platforms");
            }
        }

        public static string TeamSettingsFilePath = null;
        private static readonly string SettingFilePostfix = ".application.config";
        private static readonly string UserSettingFilePostfix = ".user.config";
        private static readonly string DefaultSettingFilePrefix = "3dEditor";

        public static AppConfig.Setting					Setting { get; set; }
        public static UserConfig.UserSetting			UserSetting { get; set; }
        public static TeamConfig.FileIo					FileIo { get; set; }

        public static TeamConfig.Preset					Preset { get; set; }
        public static TeamConfig.TeamConfigPreview		Preview { get; set; }
        public static TeamConfig.Error					Error { get; set; }
        public static TeamConfig.ConfigColor			Color { get; set; }
        public static TeamConfig.DefaultValue			DefaultValue { get; set; }
        public static List<TeamConfig.PlatformPreset>	PlatformPresets { get; set; }

        public static IEnumerable<TeamConfig.FileIo.SearchPath> GetSearchPaths()
        {
            return FileIo.SearchPaths.Concat(UserSetting.SearchPaths);
        }

        static ApplicationConfig()
        {
            Setting			= new AppConfig.Setting();
            UserSetting		= new UserConfig.UserSetting();
            FileIo			= new TeamConfig.FileIo();

            Preset			= new TeamConfig.Preset();
            Preview			= new TeamConfig.TeamConfigPreview();
            Error			= new TeamConfig.Error();
            Color			= new TeamConfig.ConfigColor();
            DefaultValue	= new TeamConfig.DefaultValue();
        }

        /// <summary>
        /// 0.0.0... 形式のバージョンを比較する
        /// </summary>
        internal class VersionComparer : IComparer<string>
        {
            public int Compare(string x, string y)
            {
                var ix = StringUtility.ParseVersion(Path.GetFileName(x));
                var iy = StringUtility.ParseVersion(Path.GetFileName(y));
                var ixcount = ix.Count();
                var iycount = iy.Count();
                var mincount = Math.Min(ixcount, iycount);

                // ドットで区切られた各数を比較して大小が有れば比較終了
                for (var i = 0; i < mincount; i++)
                {
                    if (ix[i] < iy[i])
                    {
                        return -1;
                    }
                    else if (ix[i] > iy[i])
                    {
                        return 1;
                    }
                }

                //ドットで区切られた数の多さで比較
                // ここに来るのは 1.1.1 と 1.1.1.1の比較の場合
                if (ixcount < iycount)
                {
                    return -1;
                }
                else if (ixcount > iycount)
                {
                    return 1;
                }
                return 0; // 1.1.1 と 1.1.01 のときにここに来る
            }
        }

        private static string DefaultTeamSettingsFilePath
        {
            get
            {
                return Path.Combine(Environment.GetEnvironmentVariable("NW4F_3DEDITOR_ROOT"), "3dEditor.teamsettings");
            }
        }

        public static void Load(string teamSettingsFile)
        {
            TeamSettingsFilePath = teamSettingsFile;

            // "NW4F_3DEDITOR_ROOT" を仮設定
            SetEnvironmentVariables();

            string nw4f3deTeamConfig = Environment.GetEnvironmentVariable("NW4F_3DE_TEAM_CONFIG");
            DebugConsole.WriteLine("NW4F_3DE_TEAM_CONFIG "+ (nw4f3deTeamConfig ?? ""));

            // 引数指定がなく、環境変数による旧チーム設定もなく、デフォルトの場所にファイルがあればそれを使う
            if (string.IsNullOrEmpty(TeamSettingsFilePath) &&
                (nw4f3deTeamConfig == null || !Directory.Exists(nw4f3deTeamConfig + @"\Config")) &&
                 File.Exists(DefaultTeamSettingsFilePath))
            {
                TeamSettingsFilePath = DefaultTeamSettingsFilePath;
            }

            // チーム設定の読み込み
            using (var watch = new DebugStopWatch("ApplicationConfig.Load() : TeamConfig"))
            {
                if (!string.IsNullOrEmpty(TeamSettingsFilePath))
                {
                    TeamSettings teamSettings = null;
                    if (!File.Exists(TeamSettingsFilePath))
                    {
                        UIMessageBox.Error(string.Format(res.Strings.TeamSettingsFileNotExists, TeamSettingsFilePath));
                    }
                    else
                    {
                        teamSettings = Deserialize<TeamConfig.TeamSettings>(TeamSettingsFilePath);
                    }

                    if (teamSettings != null)
                    {
                        FileIo = teamSettings.FileIo;
                        ValidateUserCommand();
                        ValidateRuntimeUserScript();
                        ValidateCreateFsv();

                        Preset = teamSettings.Preset;
                        ValidatePreset(Preset);

                        Preview = teamSettings.Preview;
                        Error = teamSettings.Error;
                        Color = teamSettings.Color;
                        DefaultValue = teamSettings.DefaultValue;
                        if (teamSettings.Preset != null)
                        {
                            PlatformPresets = teamSettings.Preset.PlatformPresets;
                            foreach (var platformPreset in PlatformPresets)
                            {
                                platformPreset.FilePath = TeamSettingsFilePath;
                            }
                        }
                    }
                }
                else
                {
                    string teamConfigPath = TeamConfigDefaultPath;
                    string fileIoPath = Path.Combine(teamConfigPath, @"Config\FileIo.config");
                    var fileIoFile = Deserialize<TeamConfig.FileIo>(fileIoPath);
                    if (fileIoFile != null)
                    {
                        FileIo = fileIoFile;
                        ValidateUserCommand();
                        ValidateRuntimeUserScript();
                        ValidateCreateFsv();
                    }

                    string presetFilePath = Path.Combine(teamConfigPath, @"Config\Preset.config");
                    var presetFile = Deserialize<TeamConfig.Preset>(presetFilePath);
                    if (presetFile != null)
                    {
                        Preset = presetFile;
                        ValidatePreset(Preset);
                    }

                    string previewFilePath = Path.Combine(teamConfigPath, @"Config\Preview.config");
                    var previewFile = Deserialize<TeamConfig.TeamConfigPreview>(previewFilePath);
                    if (previewFile != null)
                    {
                        Preview = previewFile;
                    }

                    string errorFilePath = Path.Combine(teamConfigPath, @"Config\Error.config");
                    var errorFile = Deserialize<TeamConfig.Error>(errorFilePath);
                    if (errorFile != null)
                    {
                        Error = errorFile;
                    }

                    string colorFilePath = Path.Combine(teamConfigPath, @"Config\Color.config");
                    var colorConfig = Deserialize<TeamConfig.ConfigColor>(colorFilePath);
                    if (colorConfig != null)
                    {
                        Color = colorConfig;
                    }

                    string defaultValueFilePath = Path.Combine(teamConfigPath, @"Config\DefaultValue.config");
                    var defaultValueConfig = Deserialize<TeamConfig.DefaultValue>(defaultValueFilePath);
                    if (defaultValueConfig != null)
                    {
                        DefaultValue = defaultValueConfig;
                    }

                    PlatformPresets = new List<TeamConfig.PlatformPreset>();
                }
            }

            // アプリケーションコンフィグの読み込み
            using (var watch = new DebugStopWatch("ApplicationConfig.Load() : AppConfig"))
            {
                var configPath = AppSettingFilepath;
                try
                {
                    configPath = GetConfigPath(AppSettingFilepath, DefaultAppSettingFilepath);

                    if (File.Exists(configPath))
                    {
                        using (var fileStream = new FileStream(configPath, FileMode.Open, FileAccess.Read))
                        {
                            Setting = (AppConfig.Setting)(new XmlSerializer(typeof(AppConfig.Setting))).Deserialize(fileStream);
                        }
                        // 現在のバージョンフォルダから読み込んでいない場合は保存する
                        if (configPath != AppSettingFilepath)
                        {
                            ApplicationConfig.Save();
                        }
                    }
                    else
                    {
                        ApplicationConfig.Save();
                    }
                }
                catch(Exception exception)
                {
                    UIMessageBox.Error(Strings.IO_Load_Failed + "\n{0}\n{1}", configPath, exception.Message);
                }
            }

            // ユーザー設定の読み込み
            using (var watch = new DebugStopWatch("ApplicationConfig.Load() : UsrConfig"))
            {
                var configPath = UserSettingFilePath;
                try
                {
                    configPath = GetConfigPath(UserSettingFilePath, DefaultUserSettingFilePath);

                    if (File.Exists(configPath))
                    {
                        using (var fileStream = new FileStream(configPath, FileMode.Open, FileAccess.Read))
                        {
                            UserSetting = (UserConfig.UserSetting)(new XmlSerializer(typeof(UserConfig.UserSetting))).Deserialize(fileStream);
                        }
                        // 現在のバージョンフォルダから読み込んでいない場合は保存する
                        if (configPath != UserSettingFilePath)
                        {
                            ApplicationConfig.SaveUserSetting();
                        }
                    }
                    else
                    {
                        ApplicationConfig.SaveUserSetting();
                    }
                }
                catch (Exception exception)
                {
                    UIMessageBox.Error(Strings.IO_Load_Failed + "\n{0}\n{1}", configPath, exception.Message);
                }
            }

            // プラットフォーム設定ファイルの読み込み
            using (var watch = new DebugStopWatch("ApplicationConfig.Load() : PlatformPreset"))
            {
                if ((PlatformPresets == null) || (PlatformPresets.Count == 0))
                {
                    // PlatformPresets が Preset.PlatformPresets を指しているので、ここで別インスタンスとして切り離す。
                    // PlatformPresets != Preset.PlatformPresets の場合でプラットフォーム設定ファイルを使っていることを意味する。
                    PlatformPresets = new List<TeamConfig.PlatformPreset>();
                }

                // チーム設定にプラットフォーム設定が記述されていなければプラットフォーム設定ファイルを読み込む
                if (PlatformPresets.Count == 0)
                {
                    try
                    {
                        if (Directory.Exists(PlatformPresetDirectoryPath))
                        {
                            foreach (var file in Directory.EnumerateFiles(PlatformPresetDirectoryPath, "*.preset").OrderBy(x => x))
                            {
                                var spec = Deserialize<TeamConfig.PlatformPreset>(file);
                                if (spec != null)
                                {
                                    spec.FilePath = file;
                                    PlatformPresets.Add(spec);
                                }
                            }
                        }
                    }
                    catch
                    {
                        UIMessageBox.Error(res.Strings.PlatformPresetDeserializeError);
                    }
                }

                // 空だと面倒なので
                if (PlatformPresets.Count == 0)
                {
                    PlatformPresets.Add(new PlatformPreset()
                    {
                        FilePath = string.Empty,
                        Name = "Win",
                        DeviceOptions = new List<PlatformPreset.DeviceOption>()
                        {
                            new PlatformPreset.DeviceOption
                            {
                                DeviceType = "Pc",
                                ShaderConverterAdditionalArgs = "--code-type=Source --api-type=Gl",
                                TextureConverterAdditionalArgs = "--tile-mode=linear",
                                PeerType = "Generic-Windows",
                            }
                        }
                    });
                }

                App.AppContext.SelectedPlatformPreset = PlatformPresets.FirstOrDefault(x => x.Name == ApplicationConfig.Setting.MainFrame.SelectedSpec) ?? PlatformPresets.First();
            }
        }

        private static string GetConfigPath(string customFilePath, string defaultFilePath)
        {
            var configPath = customFilePath;

            // 現在のバージョンのコンフィグがない場合は古いバージョンを探す
            if (!File.Exists(customFilePath) && Directory.Exists(AppDataFolderpath))
            {
                if (File.Exists(defaultFilePath)) //まずデフォルトファイルネームで探す
                {
                    configPath = defaultFilePath;
                }
                else
                {
                    // バージョン番号が新しい順にフォルダを列挙
                    var fileName = Path.GetFileName(customFilePath) ?? string.Empty;
                    var defaultFileName = Path.GetFileName(defaultFilePath) ?? string.Empty;
                    var dirs =
                        Directory.GetDirectories(AppDataFolderpath, "*.*.*")
                            .OrderByDescending(x => x, new VersionComparer())
                            .ToList();
                    // バージョン番号のないパスは最後に探す
                    dirs.Add(AppDataFolderpath);
                    // 各フォルダで、カスタムファイルネーム、デフォルトファイルネームの順で探す
                    configPath =
                        dirs.SelectMany(x => new[] {Path.Combine(x, fileName), Path.Combine(x, defaultFileName)})
                            .FirstOrDefault(File.Exists);
                }
            }
            return configPath;
        }

        private static void ValidateCreateFsv()
        {
            if (string.IsNullOrEmpty(FileIo.BuildFsv.ScriptFilePath) != string.IsNullOrEmpty(FileIo.BuildFsv.TemplateFilePath))
            {
                TheApp.OnMainFrameShown += (s, e) =>
                {
                    UIMessageBox.Error(res.Strings.ApplicationConfig_ValidateBuildFsvScriptPath);
                };
            }
            else if (!string.IsNullOrEmpty(FileIo.BuildFsv.ScriptFilePath) && !string.IsNullOrEmpty(FileIo.CreateShaderVariationCommand.PathXml))
            {
                TheApp.OnMainFrameShown += (s, e) =>
                {
                    UIMessageBox.Error(res.Strings.ApplicationConfig_ValidateCreateAndBuildFsv);
                };
            }
        }

        private static void ValidateUserCommand()
        {
            foreach (var command in FileIo.UserCommands)
            {
                command.Enable = false;
                var name = command.Name;
                var path = command.FullPath;
                var xpath = command.PathXml;

                // null なら空白にしておく
                if (name == null)
                {
                    name = command.Name = string.Empty;
                }

                // ショートカットの確認
                if (!string.IsNullOrEmpty(command.ShortCut))
                {
                    var sc = command.ShortCut;
                    var keys = sc.ToUpper().Split(new[]{'+', ' '}, StringSplitOptions.RemoveEmptyEntries);
                    var keybind = Keys.None;

                    var hasKey = false;
                    foreach (var key in keys)
                    {
                        var err = false;
                        switch (key)
                        {
                            case "CTRL":
                            case "CONTROL":
                                keybind |= Keys.Control;
                                break;
                            case "SHIFT":
                                keybind |= Keys.Shift;
                                break;
                            case "ALT":
                                keybind |= Keys.Alt;
                                break;
                            case "ESC":
                            case "ESCAPE":
                                keybind |= Keys.Escape;
                                hasKey = true;
                                break;
                            default:
                                {
                                    // 複数のキーは登録できない
                                    if (hasKey)
                                    {
                                        TheApp.OnMainFrameShown += (sender, args) => UIMessageBox.Error(Strings.ApplicationConfig_ValidateUserCommand_MultiKeyShortcutNotSupport, name, sc);
                                        err = true;
                                        break;
                                    }

                                    var mainkey = Keys.None;
                                    try
                                    {
                                        var keyconv = new KeysConverter();
                                        var k = keyconv.ConvertFromString(key);
                                        if (k == null)
                                        {
                                            Keys result;
                                            if (Enum.TryParse<Keys>(key, true, out result))
                                            {
                                                k = result;
                                            }
                                        }
                                        mainkey = k != null? (Keys)k : Keys.None;
                                    }
                                    catch (Exception)
                                    {
                                        mainkey = Keys.None;
                                    }

                                    if (mainkey == Keys.None)
                                    {
                                        var k = key;
                                        TheApp.OnMainFrameShown += (sender, args) => UIMessageBox.Error(Strings.ApplicationConfig_ValidateUserCommand_UnusableKeyInShortcut, k, name, sc);
                                        err = true;
                                        break;
                                    }

                                    keybind |= mainkey;
                                    hasKey = true;
                                }
                                break;
                        }

                        if (err)
                        {
                            keybind = Keys.None;
                            break;
                        }
                    }
                    Debug.Write(string.Format("Shortcut:{0}, ({1})\n", sc, keybind.ToString()));

                    if (keybind != Keys.None)
                    {
                        // ショートカットの重複をチェック
                        if (FileIo.UserCommands.Any(x => x.ShortCutKeyBind == keybind))
                        {
                            TheApp.OnMainFrameShown += (sender, args) => UIMessageBox.Error(Strings.ApplicationConfig_ValidateUserCommand_SameShortcutKeyUsed, name, sc);
                        }
                        else
                        {
                            command.ShortCutKeyBind = keybind;
                        }
                    }
                }

                // ユーザープログラムのパス解決の確認
                if (path.Contains('%'))
                {
                    TheApp.OnMainFrameShown +=
                        (sender, args) =>
                        UIMessageBox.Error(
                            Strings.ApplicationConfig_ValidateUserCommand_InvalidEnv,
                            Strings.ApplicationConfig_UserCommand,
                            name,
                            xpath);
                    continue;
                }

                try
                {
                    var progs = command.Commands.ToList();
                    if (!progs.Any())
                    {
                        TheApp.OnMainFrameShown +=
                            (sender, args) =>
                            UIMessageBox.Error(
                                Strings.ApplicationConfig_ValidateUserCommand_FileNotFound,
                                Strings.ApplicationConfig_UserCommand,
                                name,
                                xpath);
                        continue;
                    }

                    var wrongPath = false;
                    if (progs.Any(prog => !File.Exists(prog)))
                    {
                        TheApp.OnMainFrameShown +=
                            (sender, args) =>
                            UIMessageBox.Error(
                                Strings.ApplicationConfig_ValidateUserCommand_FileNotFound,
                                Strings.ApplicationConfig_UserCommand,
                                name,
                                xpath);
                        wrongPath = true;
                    }

                    if (wrongPath)
                    {
                        continue;
                    }

                }
                catch (Exception)
                {
                    TheApp.OnMainFrameShown +=
                        (sender, args) =>
                        UIMessageBox.Error(
                            Strings.ApplicationConfig_ValidateUserCommand_FileNotFound,
                            Strings.ApplicationConfig_UserCommand,
                            name,
                            xpath);
                    continue;
                }

                // 対象ファイルの確認
                var wrongExt = false;
                foreach (var ext in command.FilterExts)
                {
                    var id = ObjectIDUtility.ExtToId(ext);
                    if (id == null)
                    {
                        var e = ext; // for closure
                        TheApp.OnMainFrameShown += (sender, args) => UIMessageBox.Error(Strings.ApplicationConfig_ValidateUserCommand_InvalidExtension, name, e);
                        wrongExt = true;
                        break;
                    }
                    command.FileIDs.Add(id.GetValueOrDefault());
                }

                if (wrongExt)
                {
                    continue;
                }

                // ショートカットがオブジェクトビュー対象で、fmd,ftxが含まれていない場合はショートカットを無効に
                if (!command.FileTreeShortCut && command.ShortCutKeyBind != Keys.None && command.FileIDs.Any())
                {
                    if (command.FileIDs.All(x => x != GuiObjectID.Model && x != GuiObjectID.Texture))
                    {
                        var sc = command.ShortCut;
                        TheApp.OnMainFrameShown += (sender, args) => UIMessageBox.Error(Strings.ApplicationConfig_ValidateUserCommand_ObjectViewShortCutMustAssignForModelOrTexture, name, sc);
                        command.ShortCutKeyBind = Keys.None;
                    }
                }

                // コマンドを有効化
                command.Enable = true;

            }
        }

        private static void ValidateRuntimeUserScript()
        {
            foreach (var script in FileIo.RuntimeUserScripts)
            {
                script.Enable = false;
                var name = script.Name;
                var path = script.FullPath;
                var xpath = script.PathXml;

                // null なら空白にしておく
                if (name == null)
                {
                    name = script.Name = string.Empty;
                }

#if false
                // ショートカットの確認
                if (!string.IsNullOrEmpty(script.ShortCut))
                {
                    var sc = script.ShortCut;
                    var keys = sc.ToUpper().Split(new[] { '+', ' ' }, StringSplitOptions.RemoveEmptyEntries);
                    var keybind = Keys.None;

                    var hasKey = false;
                    foreach (var key in keys)
                    {
                        var err = false;
                        switch (key)
                        {
                            case "CTRL":
                            case "CONTROL":
                                keybind |= Keys.Control;
                                break;
                            case "SHIFT":
                                keybind |= Keys.Shift;
                                break;
                            case "ALT":
                                keybind |= Keys.Alt;
                                break;
                            case "ESC":
                            case "ESCAPE":
                                keybind |= Keys.Escape;
                                hasKey = true;
                                break;
                            default:
                                {
                                    // 複数のキーは登録できない
                                    if (hasKey)
                                    {
                                        TheApp.OnMainFrameShown += (sender, args) => UIMessageBox.Error(Strings.ApplicationConfig_ValidateUserCommand_MultiKeyShortcutNotSupport, name, sc);
                                        err = true;
                                        break;
                                    }

                                    var mainkey = Keys.None;
                                    try
                                    {
                                        var keyconv = new KeysConverter();
                                        var k = keyconv.ConvertFromString(key);
                                        if (k == null)
                                        {
                                            Keys result;
                                            if (Enum.TryParse<Keys>(key, true, out result))
                                            {
                                                k = result;
                                            }
                                        }
                                        mainkey = k != null ? (Keys)k : Keys.None;
                                    }
                                    catch (Exception)
                                    {
                                        mainkey = Keys.None;
                                    }

                                    if (mainkey == Keys.None)
                                    {
                                        var k = key;
                                        TheApp.OnMainFrameShown += (sender, args) => UIMessageBox.Error(Strings.ApplicationConfig_ValidateUserCommand_UnusableKeyInShortcut, k, name, sc);
                                        err = true;
                                        break;
                                    }

                                    keybind |= mainkey;
                                    hasKey = true;
                                }
                                break;
                        }

                        if (err)
                        {
                            keybind = Keys.None;
                            break;
                        }
                    }
                    Debug.Write(string.Format("Shortcut:{0}, ({1})\n", sc, keybind.ToString()));

                    if (keybind != Keys.None)
                    {
                        // ショートカットの重複をチェック
                        if (FileIo.UserCommands.Any(x => x.ShortCutKeyBind == keybind))
                        {
                            TheApp.OnMainFrameShown += (sender, args) => UIMessageBox.Error(Strings.ApplicationConfig_ValidateUserCommand_SameShortcutKeyUsed, name, sc);
                        }
                        else
                        {
                            script.ShortCutKeyBind = keybind;
                        }
                    }
                }
#endif

                // ランタイムユーザースクリプトのパス解決の確認
                if (path.Contains('%'))
                {
                    TheApp.OnMainFrameShown +=
                        (sender, args) =>
                        UIMessageBox.Error(
                            Strings.ApplicationConfig_ValidateUserCommand_InvalidEnv,
                            Strings.ApplicationConfig_UserScript,
                            name,
                            xpath);
                    continue;
                }

                try
                {
                    var progs = script.Scripts.ToList();
                    if (!progs.Any())
                    {
                        TheApp.OnMainFrameShown +=
                            (sender, args) =>
                            UIMessageBox.Error(
                                Strings.ApplicationConfig_ValidateUserCommand_FileNotFound,
                                Strings.ApplicationConfig_UserScript,
                                name,
                                xpath);
                        continue;
                    }

                    var wrongPath = false;
                    if (progs.Any(prog => !File.Exists(prog)))
                    {
                        TheApp.OnMainFrameShown +=
                            (sender, args) =>
                            UIMessageBox.Error(
                                Strings.ApplicationConfig_ValidateUserCommand_FileNotFound,
                                Strings.ApplicationConfig_UserScript,
                                name,
                                xpath);
                        wrongPath = true;
                    }

                    if (wrongPath)
                    {
                        continue;
                    }

                }
                catch (Exception)
                {
                    TheApp.OnMainFrameShown +=
                        (sender, args) =>
                        UIMessageBox.Error(
                            Strings.ApplicationConfig_ValidateUserCommand_FileNotFound,
                            Strings.ApplicationConfig_UserScript,
                            name,
                            xpath);
                    continue;
                }

                // 対象ファイルの確認
#if false
                var wrongExt = false;
                foreach (var ext in script.FilterExts)
                {
                    var id = ObjectIDUtility.ExtToId(ext);
                    if (id == null)
                    {
                        var e = ext; // for closure
                        TheApp.OnMainFrameShown += (sender, args) => UIMessageBox.Error(Strings.ApplicationConfig_ValidateUserCommand_InvalidExtension, name, e);
                        wrongExt = true;
                        break;
                    }
                    script.FileIDs.Add(id.GetValueOrDefault());
                }

                if (wrongExt)
                {
                    continue;
                }

                // ショートカットがオブジェクトビュー対象で、fmd,ftxが含まれていない場合はショートカットを無効に
                if (!script.FileTreeShortCut && script.ShortCutKeyBind != Keys.None && script.FileIDs.Any())
                {
                    if (script.FileIDs.All(x => x != GuiObjectID.Model && x != GuiObjectID.Texture))
                    {
                        var sc = script.ShortCut;
                        TheApp.OnMainFrameShown += (sender, args) => UIMessageBox.Error(Strings.ApplicationConfig_ValidateUserCommand_ObjectViewShortCutMustAssignForModelOrTexture, name, sc);
                        script.ShortCutKeyBind = Keys.None;
                    }
                }
#endif
                // コマンドを有効化
                script.Enable = true;

            }
        }

        private static void ValidatePreset(Preset presetFile)
        {
            if (presetFile != null)
            {
                // 訂正が必要な検証 (と、その訂正) を行う。
                // 後段では検証失敗時に検証を中断する箇所が多いため、
                // 訂正が必要なものはここで先に処理しておかなければならない。
                {
                    // ライトアニメーションプリセットはタイプをキーにして辞書登録される。
                    // 登録時の例外を避けるために null と重複を取り除く必要がある。
                    var lpres = presetFile.LightAnimPresets;

                    // ライトアニメーションプリセットでは null タイプは許可しない。
                    // ライトアニメーションプリセットはタイプをキーにして辞書登録されるので、
                    // 辞書登録時の例外を避けるために null タイプを取り除いて警告する。
                    foreach (var item in lpres.Where(x => x.Type == null).ToArray())
                    {
                        // タイプが null の要素を削除する。
                        lpres.Remove(item);
                        var label = !string.IsNullOrEmpty(item.Label) ? item.Label : string.Empty;
                        TheApp.OnMainFrameShown += (sender, args) => UIMessageBox.Error(Strings.Preset_Load_LightAnimPreset_NoType, label);
                    }

                    // ライトアニメーションプリセットでは同名のタイプは許可しない。
                    // ライトアニメーションプリセットはタイプをキーにして辞書登録されるので、
                    // 辞書登録時の例外を避けるために重複を取り除いて警告する。
                    foreach (var group in lpres.ToLookup(x => x.Type).Where(y => y.Count() > 1))
                    {
                        var deleteList = group.OrderBy(x => lpres.IndexOf(x)).ToList();

                        // 先頭要素を採用するので、先頭は削除対象外。
                        deleteList.RemoveAt(0);

                        // 後続の要素を削除する。
                        foreach (var item in deleteList)
                        {
                            // 重複要素を削除する。
                            lpres.Remove(item);
                            var label = !string.IsNullOrEmpty(item.Label) ? item.Label : string.Empty;
                            var type = !string.IsNullOrEmpty(item.Type) ? item.Type : string.Empty;
                            TheApp.OnMainFrameShown += (sender, args) => UIMessageBox.Error(Strings.Preset_Load_LightAnimPreset_DuplicatedType, label, type);
                        }
                    }
                }

                var functions = new Dictionary<List<Preset.AttenuationFunction>, string>
                    {
                        { presetFile.LightDistAttnPresets,  Strings.TeamConfigSettingDialog_pnlPreset_LightDistAttnFunc},
                        { presetFile.LightAngAttnPresets,   Strings.TeamConfigSettingDialog_pnlPreset_LightAngleAttnFunc},
                        { presetFile.FogDistAttnPresets,    Strings.TeamConfigSettingDialog_pnlPreset_FogDistAttnFunc}
                    };

                // プリセットでタイプの未指定はエラーとする
                if (presetFile.LightAnimPresets.Any(x => x.Type == null))
                {
                    TheApp.OnMainFrameShown += (sender, args) => UIMessageBox.Error(Strings.Preset_Load_LightTarget_NoType);
                    return;
                }
                foreach (var elem in functions)
                {
                    var func = elem.Key;
                    var kind = elem.Value;
                    if (func.Any(x => x.Function == null))
                    {
                        TheApp.OnMainFrameShown += (sender, args) => UIMessageBox.Error(Strings.Preset_Load_DistFunc_NoType, kind);
                        return;
                    }
                }

                // プリセットでラベルが指定されていない場合は、タイプや関数と同じにする
                {
                    var noLabels = presetFile.LightAnimPresets.Where(x => x.Label == null);
                    foreach (var l in noLabels) { l.Label = l.Type; }
                }

                foreach (var elem in functions)
                {
                    var noLabels = elem.Key.Where(x => x.Label == null);
                    foreach (var l in noLabels) { l.Label = l.Function; }
                }

                // ライトアニメーションプリセットで同名のラベルやタイプは許可しない
                {
                    var lpres = presetFile.LightAnimPresets;
                    var conflict = lpres.GroupBy(x => x.Label).Where(x => x.Count() > 1);
                    conflict = conflict.Concat(lpres.GroupBy(x => x.Type).Where(x => x.Count() > 1));
                    if (conflict.Any())
                    {
                        var confstr = string.Join(", ", conflict.Select(x => x.Key));
                        TheApp.OnMainFrameShown += (sender, args) => UIMessageBox.Error(Strings.Preset_Load_Duplicated_Type_Or_Label, confstr);
                        return;
                    }
                }

                // ライトアニメーションのプリセット内に重複するターゲットは許可しない
                foreach (var item in presetFile.LightAnimPresets)
                {
                    var confilict = item.LightAnimTargets.GroupBy(x => x.Target).Where(x => x.Count() > 1);
                    if (confilict.Any())
                    {
                        var confstr = string.Join(", ", confilict.Select(x => x.Key));
                        TheApp.OnMainFrameShown += (s, a) => UIMessageBox.Error(Strings.Preset_Load_Duplicated_Target, item.Type, confstr);
                        return;
                    }
                }

                // それぞれの減衰関数で同名のラベルや関数は許可しない

                foreach (var elem in functions)
                {
                    var func = elem.Key;
                    var kind = elem.Value;
                    var conflict = func.GroupBy(x => x.Label).Where(x => x.Count() > 1);
                    conflict = conflict.Concat(func.GroupBy(x => x.Function).Where(x => x.Count() > 1));
                    if (conflict.Any())
                    {
                        var confstr = string.Join(", ", conflict.Select(x => x.Key));
                        TheApp.OnMainFrameShown += (sender, args) => UIMessageBox.Error(Strings.Preset_Load_Duplicated_Func_Or_Label, kind, confstr);
                        return;
                    }
                }

                // ライトアニメーションターゲットのデフォルト値が正しく設定されているかチェック
                foreach (var pre in presetFile.LightAnimPresets)
                {
                    foreach (var anim in pre.LightAnimTargets)
                    {
                        if (!anim.HasDefault()) continue;

                        switch (anim.Target)
                        {
                            case Preset.AnimTargetEnum.enable:
                                {
                                    var val = anim.GetDefaultBool();
                                    if (val == null)
                                    {
                                        TheApp.OnMainFrameShown += (sender, args) => UIMessageBox.Error(Strings.Preset_Load_BadDefaultValue
                                                           , pre.Label, anim.Target.ToString(), anim.DefaultValue, Strings.Preset_Load_BadDefaultValue_BoolOr01);
                                        anim.DefaultValue = null;
                                    }
                                }
                                break;
                            case Preset.AnimTargetEnum.angle_attn:
                            case Preset.AnimTargetEnum.dist_attn:
                                {
                                    var val = anim.GetDefaultFloat2();
                                    if (val == null)
                                    {
                                        TheApp.OnMainFrameShown += (sender, args) => UIMessageBox.Error(Strings.Preset_Load_BadDefaultValue
                                                           , pre.Label, anim.Target.ToString(), anim.DefaultValue, "float2");
                                        anim.DefaultValue = null;
                                    }
                                }
                                break;
                            default:
                                {
                                    var val = anim.GetDefaultFloat3();
                                    if (val == null)
                                    {
                                        TheApp.OnMainFrameShown += (sender, args) => UIMessageBox.Error(Strings.Preset_Load_BadDefaultValue
                                                           , pre.Label, anim.Target.ToString(), anim.DefaultValue, "float3");
                                        anim.DefaultValue = null;
                                    }
                                }
                                break;
                        }
                    }
                }

                // ライトアニメーションターゲットでDirectionとAnimが同時に定義されている場合は
                // Directionを削除して警告する
                var badPresets = presetFile.LightAnimPresets
                                        .Where(pre => pre.LightAnimTargets
                                                        .Any(anim => anim.Target == Preset.AnimTargetEnum.direction))
                                        .Where(pre => pre.LightAnimTargets
                                                        .Any(anim => anim.Target == Preset.AnimTargetEnum.aim));

                if (badPresets.Any())
                {
                    foreach (var anim in badPresets)
                    {
                        var index =
                            anim.LightAnimTargets.FindIndex(x => x.Target == Preset.AnimTargetEnum.direction);
                        if (index >= 0) anim.LightAnimTargets.RemoveAt(index);
                        TheApp.OnMainFrameShown += (sender, args) => UIMessageBox.Error(Strings.Preset_Load_BadLightTarget, anim.Label, anim.Type);
                    }
                }
            }
        }

        private static T Deserialize<T>(string filePath) where T: class
        {
            try
            {
                if (File.Exists(filePath))
                {
                    using (FileStream fileStream = new FileStream(filePath, FileMode.Open, FileAccess.Read))
                    {
                        return (T)(new XmlSerializer(typeof(T))).Deserialize(fileStream);
                    }
                }
            }
            catch (Exception exception)
            {
                var builder = new StringBuilder();
                builder.Append(res.Strings.IO_Load_Failed);

                // 内部例外をさかのぼる
                Exception ex = exception;
                while (ex != null)
                {
                    if (!string.IsNullOrEmpty(ex.Message))
                    {
                        builder.Append(ex.Message);
                    }
                    ex = ex.InnerException;
                }

                UIMessageBox.Error(App.res.Strings.IO_Load_Failed + "\n{0}\n{1}", filePath, builder.ToString());
            }
            return null;
        }

        public static void SetEnvironmentVariables()
        {
            // 上書き設定
            string root = "NW4F_3DEDITOR_ROOT";
            if (Environment.GetEnvironmentVariable(root) == null)
            {
                string relativePath = @".";
                string executableDirectory = App.TheApp.AssemblyDirectory;
                if (string.Compare(Path.GetFileName(executableDirectory), "Debug", true) == 0 ||
                    string.Compare(Path.GetFileName(executableDirectory), "Release", true) == 0)
                {
                    relativePath = @"..\..\..\..\..\..\..\..\..\Tools\Graphics\3dEditor";
                }

                string path = Path.Combine(executableDirectory, relativePath);
                try
                {
                    path = Path.GetFullPath(path);
                }
                catch (Exception)
                {
                }
                Environment.SetEnvironmentVariable(root, path);
            }
        }

        public static void Save()
        {
            try
            {
                var xmlWriterSettings = new XmlWriterSettings()
                {
                    Encoding = new UTF8Encoding(true),			// BOM付き
                    Indent = true,
                    IndentChars = "\t",
                    CloseOutput = false,
                };

                Directory.CreateDirectory(AppDataFolderpath);

                Directory.CreateDirectory(Path.GetDirectoryName(AppSettingFilepath));

                using (var stream = new FileStream(AppSettingFilepath, FileMode.Create))
                using (var xmlWriter = XmlTextWriter.Create(stream, xmlWriterSettings))
                {
                    var ns = new XmlSerializerNamespaces();
                    ns.Add(string.Empty, string.Empty);

                    (new XmlSerializer(typeof(AppConfig.Setting), "")).Serialize(xmlWriter, Setting, ns);
                }
            }
            catch(Exception exception)
            {
                UIMessageBox.Error(App.res.Strings.Config_SaveFailed, exception.Message);
            }
        }

        public static bool SavePlatformPresets(System.Collections.Generic.IEnumerable<TeamConfig.PlatformPreset> platformPresets)
        {
            bool ret = false;
            try
            {
                var xmlWriterSettings = new XmlWriterSettings()
                {
                    Encoding = new UTF8Encoding(true),			// BOM付き
                    Indent = true,
                    IndentChars = "\t",
                    CloseOutput = false,
                };

                foreach (var platformPreset in platformPresets)
                {
                    var filePath = platformPreset.FilePath;
                    var dirPath = Path.GetDirectoryName(filePath);
                    if (!string.IsNullOrEmpty(dirPath))
                    {
                        Directory.CreateDirectory(dirPath);
                    }

                    using (var stream = new FileStream(filePath, FileMode.Create))
                    using (var xmlWriter = XmlTextWriter.Create(stream, xmlWriterSettings))
                    {
                        var ns = new XmlSerializerNamespaces();
                        ns.Add(string.Empty, string.Empty);

                        (new XmlSerializer(typeof(TeamConfig.PlatformPreset), "")).Serialize(xmlWriter, platformPreset, ns);
                    }
                }

                ret = true;
            }
            catch (Exception exception)
            {
                UIMessageBox.Error(App.res.Strings.Config_SaveFailed, exception.Message);
            }
            return ret;
        }

        public static bool SaveTeamSettings(TeamSettings teamSettings)
        {
            bool ret = false;
            try
            {
                var xmlWriterSettings = new XmlWriterSettings()
                {
                    Encoding = new UTF8Encoding(true),			// BOM付き
                    Indent = true,
                    IndentChars = "\t",
                    CloseOutput = false,
                };

                // チーム設定ファイルの保存先。
                // 全部入りの設定ファイルが読み込まれていない場合は DefaultTeamSettingsFilePath に保存する。
                var teamSettingsFilePath = !string.IsNullOrEmpty(TeamSettingsFilePath) ? TeamSettingsFilePath : DefaultTeamSettingsFilePath;
                if (!string.IsNullOrEmpty(teamSettingsFilePath))
                {
                    var dirPath = Path.GetDirectoryName(teamSettingsFilePath);
                    if (!string.IsNullOrEmpty(dirPath))
                    {
                        Directory.CreateDirectory(dirPath);
                    }

                    using (var stream = new FileStream(teamSettingsFilePath, FileMode.Create))
                    using (var xmlWriter = XmlTextWriter.Create(stream, xmlWriterSettings))
                    {
                        var ns = new XmlSerializerNamespaces();
                        ns.Add(string.Empty, string.Empty);

                        (new XmlSerializer(typeof(TeamSettings), "")).Serialize(xmlWriter, teamSettings, ns);
                    }

                    // 保存先の変更を適用。
                    if (teamSettingsFilePath != TeamSettingsFilePath)
                    {
                        TeamSettingsFilePath = teamSettingsFilePath;
                    }
                }
                else
                {
                    // ※全部入りの設定ファイルが読み込まれていない場合は
                    // DefaultTeamSettingsFilePath に保存するようにした (teamSettingsFilePath を参照) ので、ここには到達しない。
                    throw new NullReferenceException();

                    #if false
                    // 全部入りの設定ファイルが読み込まれていない場合は旧型式 (設定毎に別ファイル) で保存する。
                    var teamConfigPath = TeamConfigDefaultPath;

                    var fileIoPath = Path.Combine(teamConfigPath, @"Config\FileIo.config");
                    Directory.CreateDirectory(Path.GetDirectoryName(fileIoPath));
                    using (var stream = new FileStream(fileIoPath, FileMode.Create))
                    using (var xmlWriter = XmlTextWriter.Create(stream, xmlWriterSettings))
                    {
                        var ns = new XmlSerializerNamespaces();
                        ns.Add(string.Empty, string.Empty);

                        (new XmlSerializer(typeof(TeamConfig.FileIo), "")).Serialize(xmlWriter, teamSettings.FileIo, ns);
                    }

                    var presetFilePath = Path.Combine(teamConfigPath, @"Config\Preset.config");
                    Directory.CreateDirectory(Path.GetDirectoryName(presetFilePath));
                    using (var stream = new FileStream(presetFilePath, FileMode.Create))
                    using (var xmlWriter = XmlTextWriter.Create(stream, xmlWriterSettings))
                    {
                        var ns = new XmlSerializerNamespaces();
                        ns.Add(string.Empty, string.Empty);

                        (new XmlSerializer(typeof(TeamConfig.Preset), "")).Serialize(xmlWriter, teamSettings.Preset, ns);
                    }

                    var previewFilePath = Path.Combine(teamConfigPath, @"Config\Preview.config");
                    Directory.CreateDirectory(Path.GetDirectoryName(previewFilePath));
                    using (var stream = new FileStream(previewFilePath, FileMode.Create))
                    using (var xmlWriter = XmlTextWriter.Create(stream, xmlWriterSettings))
                    {
                        var ns = new XmlSerializerNamespaces();
                        ns.Add(string.Empty, string.Empty);

                        (new XmlSerializer(typeof(TeamConfig.TeamConfigPreview), "")).Serialize(xmlWriter, teamSettings.Preview, ns);
                    }

                    var errorFilePath = Path.Combine(teamConfigPath, @"Config\Error.config");
                    Directory.CreateDirectory(Path.GetDirectoryName(errorFilePath));
                    using (var stream = new FileStream(errorFilePath, FileMode.Create))
                    using (var xmlWriter = XmlTextWriter.Create(stream, xmlWriterSettings))
                    {
                        var ns = new XmlSerializerNamespaces();
                        ns.Add(string.Empty, string.Empty);

                        (new XmlSerializer(typeof(TeamConfig.Error), "")).Serialize(xmlWriter, teamSettings.Error, ns);
                    }

                    var colorFilePath = Path.Combine(teamConfigPath, @"Config\Color.config");
                    Directory.CreateDirectory(Path.GetDirectoryName(colorFilePath));
                    using (var stream = new FileStream(colorFilePath, FileMode.Create))
                    using (var xmlWriter = XmlTextWriter.Create(stream, xmlWriterSettings))
                    {
                        var ns = new XmlSerializerNamespaces();
                        ns.Add(string.Empty, string.Empty);

                        (new XmlSerializer(typeof(TeamConfig.ConfigColor), "")).Serialize(xmlWriter, teamSettings.Color, ns);
                    }

                    var defaultValueFilePath = Path.Combine(teamConfigPath, @"Config\DefaultValue.config");
                    Directory.CreateDirectory(Path.GetDirectoryName(defaultValueFilePath));
                    using (var stream = new FileStream(defaultValueFilePath, FileMode.Create))
                    using (var xmlWriter = XmlTextWriter.Create(stream, xmlWriterSettings))
                    {
                        var ns = new XmlSerializerNamespaces();
                        ns.Add(string.Empty, string.Empty);

                        (new XmlSerializer(typeof(TeamConfig.DefaultValue), "")).Serialize(xmlWriter, teamSettings.DefaultValue, ns);
                    }
                    #endif
                }

                ret = true;
            }
            catch (Exception exception)
            {
                UIMessageBox.Error(App.res.Strings.Config_SaveFailed, exception.Message);
            }
            return ret;
        }

        public static void SaveUserSetting()
        {
            try
            {
                var xmlWriterSettings = new XmlWriterSettings()
                {
                    Encoding = new UTF8Encoding(true),			// BOM付き
                    Indent = true,
                    IndentChars = "\t",
                    CloseOutput = false,
                };

                Directory.CreateDirectory(Path.GetDirectoryName(UserSettingFilePath));

                using (var stream = new FileStream(UserSettingFilePath, FileMode.Create))
                using (var xmlWriter = XmlTextWriter.Create(stream, xmlWriterSettings))
                {
                    var ns = new XmlSerializerNamespaces();
                    ns.Add(string.Empty, string.Empty);

                    (new XmlSerializer(typeof(UserConfig.UserSetting), "")).Serialize(xmlWriter, UserSetting, ns);
                }
            }
            catch(Exception exception)
            {
                UIMessageBox.Error(App.res.Strings.Config_SaveFailed, exception.Message);
            }
        }
    }

    // 拡張メソッド
    public static class ConfigExtensions
    {

        /// <summary>
        /// ApplicationConfig.GetSearchPaths() を使ってフルパスを生成する。
        /// </summary>
        /// <param name="startUp"></param>
        /// <returns></returns>
        public static string FullPath(this ConfigCommon.StartUp startUp)
        {
            return FullPath(startUp, ApplicationConfig.GetSearchPaths().ToList());
        }

        /// <summary>
        /// フルパスを生成する。
        /// </summary>
        /// <param name="startUp"></param>
        /// <param name="searchPaths">フルパス生成に参照する検索パス</param>
        /// <returns></returns>
        public static string FullPath(this ConfigCommon.StartUp startUp, IReadOnlyList<FileIo.SearchPath> searchPaths)
        {
            // TODO: エラーハンドリング
            try
            {
                string path = startUp.pathXml == null ?
                    "" :
                    System.Environment.ExpandEnvironmentVariables(startUp.pathXml);
                if (startUp.Base != null)
                {
                    var searchPath = searchPaths.FirstOrDefault(x => x.Name == startUp.Base);
                    if (searchPath == null)
                    {
                        return null;
                    }
                    path = Path.Combine(searchPath.path, path);
                }

                if (!Path.IsPathRooted(path))
                {
                    return null;
                }

                path = Path.GetFullPath(path);
                return path;
            }
            catch (Exception)
            {
                return null;
            }
        }

        public static UserConfig.ReloadOptions.Option GetOption(this UserConfig.ReloadOptions reloadOptions, GuiObjectID id)
        {
            switch (id)
            {
                case GuiObjectID.Model:
                    return reloadOptions.Model;
                case GuiObjectID.Texture:
                    return reloadOptions.Texture;
                case GuiObjectID.SkeletalAnimation:
                    return reloadOptions.SkeletalAnimation;
                case GuiObjectID.ShapeAnimation:
                    return reloadOptions.ShapeAnimation;
                case GuiObjectID.ShaderParameterAnimation:
                    return reloadOptions.ShaderParameterAnimation;
                case GuiObjectID.ColorAnimation:
                    return reloadOptions.ColorAnimation;
                case GuiObjectID.TextureSrtAnimation:
                    return reloadOptions.TextureSrtAnimation;
                case GuiObjectID.TexturePatternAnimation:
                    return reloadOptions.TexturePatternAnimation;
                case GuiObjectID.BoneVisibilityAnimation:
                    return reloadOptions.BoneVisibilityAnimation;
                case GuiObjectID.MaterialVisibilityAnimation:
                    return reloadOptions.MaterialVisibilityAnimation;
                case GuiObjectID.SceneAnimation:
                    return reloadOptions.SceneAnimation;
                case GuiObjectID.ShaderDefinition:
                    return reloadOptions.ShaderDefinition;
                case GuiObjectID.MaterialAnimation:
                    return reloadOptions.MaterialAnimation;
                case GuiObjectID.SeparateMaterial:
                    return reloadOptions.SeparateMaterial;
            }
            Debug.Assert(false);
            return new UserConfig.ReloadOptions.Option();
        }

        public static AppConfig.ListView GetListView(this AppConfig.ObjectView objectView, App.ObjectView.List.ViewID id)
        {
            switch (id)
            {
                case App.ObjectView.List.ViewID.Model:
                    return objectView.ModelListView;
                case App.ObjectView.List.ViewID.Shape:
                    return objectView.ShapeListView;
                case App.ObjectView.List.ViewID.Bone:
                    return objectView.BoneListView;
                case App.ObjectView.List.ViewID.Texture:
                    return objectView.TextureListView;
                case App.ObjectView.List.ViewID.Material:
                    return objectView.MaterialListView;
            }
            return null;
        }

        public static App.ObjectView.List.ColumnItem GetColumnItem(this AppConfig.Column column)
        {
            return (App.ObjectView.List.ColumnItem)column.columnItem;
        }

        public static void SetColumnItem(this AppConfig.Column column, App.ObjectView.List.ColumnItem value)
        {
            if (value == null)
            {
                if (column.columnItem != null)
                {
                    column.width = column.columnItem.Width;
                }
            }
            else
            {
                column.width = value.Width;
            }
            column.columnItem = value;
        }
    }
}
