﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Drawing;
using System.Globalization;
using System.IO;
using System.Linq;
using System.Reflection;
using System.Threading;
using System.Web;
using System.Windows.Forms;
using System.Runtime.InteropServices;
using App.Command;
using App.ConfigData;
using App.Controls;
using App.Data;
using App.res;
using App.Utility;

namespace App
{
    #region LanguageMode
    /// <summary>
    /// 言語モード。
    /// </summary>
    public enum LanguageMode
    {
        /// <summary>日本語</summary>
        Japanese = 0,
        /// <summary>英語</summary>
        English,
    }
    #endregion

    public class Program : Microsoft.VisualBasic.ApplicationServices.WindowsFormsApplicationBase
    {
        [STAThread]
        static void Main(string[] args)
        {
            // マルチコアJITを有効にする。
            System.Runtime.ProfileOptimization.SetProfileRoot(TheApp.LocalAppDataDirectory);
            System.Runtime.ProfileOptimization.StartProfile("Startup.Profile");

            if (Environment.Is64BitProcess == false)
            {
                UIMessageBox.Error(App.res.Strings.Environment_Not64Bit);
                return;
            }

            // コマンドライン引数情報を初期化する
            CommandLineOptionManager.Initialize(args);
            if (CommandLineOptionManager.IsConsoleOptionSpecified)
            {
                // 標準出力がコンソールにアタッチされた場合は3DEditorを起動しない
                return;
            }

            Application.SetCompatibleTextRenderingDefault(false);

            if (!string.IsNullOrEmpty(CommandLineOptionManager.ErrorMessage))
            {
                UIMessageBox.Error(CommandLineOptionManager.ErrorMessage);
            }

            TheProgram = new Program();
#if DEBUG
            TheProgram.Run(args);
#else
            try
            {
                TheProgram.Run(args);
            }
            catch (Exception e)
            {
                TheApp.OnFatalErrorOccured(e, false);
                Environment.Exit(0);
            }
#endif
        }

        /// <summary>
        /// Shutdown 完了後のイベント
        /// </summary>
        public event EventHandler PostShutdown;

        public Program() :
            base(Microsoft.VisualBasic.ApplicationServices.AuthenticationMode.Windows){}

        protected override void OnCreateMainForm()
        {
            TheApp.InitializeBeforeMainFrameCreate(CommandLineArgs.ToArray());

            // メインフレーム作成
            using (var watch = new DebugStopWatch("TheApp.MainFrame = new MainFrame()"))
            {
                TheApp.MainFrame = new MainFrame();
            }

            using (var watch = new DebugStopWatch("TheApp.MainFrame.Handle"))
            {
                // BeginInvoke や Invoke を呼べるようにする。
                // ハンドルが作成されていない状態でハンドルのプロパティを取得するとハンドルが作成される。
                var handle = TheApp.MainFrame.Handle;
            }

            TheApp.InitializeAfterMainFrameCreate();

            MainForm = TheApp.MainFrame;
        }

        protected override void OnShutdown()
        {
            TheApp.Destroy();

            TemporaryFileUtility.Destroy();
            App.Bridge.BridgeManager.Destroy();

            if (PostShutdown != null)
            {
                PostShutdown(this, EventArgs.Empty);
            }
        }

        protected override bool OnUnhandledException(Microsoft.VisualBasic.ApplicationServices.UnhandledExceptionEventArgs e)
        {
            TheApp.OnFatalErrorOccured(e.Exception, false);

            return base.OnUnhandledException(e);
        }

        protected override void OnCreateSplashScreen()
        {
            SplashScreen = new SplashScreenForm();
            SplashScreen.Icon = Icon.ExtractAssociatedIcon(Assembly.GetEntryAssembly().Location);
        }

        static public Program TheProgram;
    }

    static class TheApp
    {
#region プロパティ

        // コマンドマネージャ
        public static CommandManager CommandManager { get; private set; }

        // 英語ＵＩモード
        public static bool IsEnglishUI { get; private set; }

        // ＧＵＩフォント
        public static readonly Font GuiFont				= new Font("Tahoma", 8.25F, FontStyle.Regular, GraphicsUnit.Point, 0);
        public static readonly Font GuiFontSmall		= new Font("Tahoma", 6.25F, FontStyle.Regular, GraphicsUnit.Point, 0);
        public static readonly Font GuiFontBold			= new Font(GuiFont, FontStyle.Bold);
        public static readonly Font GuiFixedFont		= new Font("ＭＳ ゴシック", 9.0f, FontStyle.Regular, GraphicsUnit.Point, 0);
        public static readonly Font GuiFixedFontBold	= new Font(GuiFixedFont, FontStyle.Bold);

        // メインフレーム。
        public static MainFrame MainFrame { get; set; }
        public static bool IsInitializedMainFrame{ get{ return MainFrame != null; } }
        public static IntPtr MainFrameHandle
        {
            get
            {
                // 別スレッドからの呼び出し用
                if (Thread.CurrentThread != TheApp.MainThread && TheApp.MainFrame != null && TheApp.MainFrame.IsHandleCreated)
                {
                    IntPtr handle = IntPtr.Zero;
                    {
                        var resetEvent = new ManualResetEventSlim();
                        {
                            MainFrame.BeginInvoke(
                                new MethodInvoker(
                                    () =>
                                    {
                                        handle = MainFrame.Handle;
                                        resetEvent.Set();
                                    }
                                )
                            );
                        }
                        resetEvent.Wait();
                    }
                    return handle;
                }
                else
                {
                    return MainFrame.Handle;
                }
            }
        }

        // メインスレッド。
        public static Thread MainThread { get; private set; }

        // 例外処理済みフラグ
        private static bool fatalErrorHandled_ = false;

        // ウィンドウをずらしながら重ねる際のマージン。
        public static int WindowCascadingMargin { get { return SystemInformation.CaptionHeight; } }

        // ヘルプファイルパス。
        public static string HelpNamespace { get; private set; }

        public static readonly string AssemblyDirectory = Path.GetDirectoryName(Assembly.GetExecutingAssembly().Location);
        public static readonly string AssemblyPath = Assembly.GetExecutingAssembly().Location;
        public static readonly string AssemblyName = Path.GetFileNameWithoutExtension(Assembly.GetExecutingAssembly().GetName().Name);

        public static string LocalAppDataDirectory
        {
            get
            {
                var folder = Path.Combine(Environment.GetFolderPath(Environment.SpecialFolder.LocalApplicationData), "Nintendo", "3dEditor");
                if(!Directory.Exists(folder))
                {
                    Directory.CreateDirectory(folder);
                }
                return folder;
            }
        }

        public static EventHandler OnMainFrameShown;

#endregion

        public static void InitializeBeforeMainFrameCreate(string[] args)
        {
            // メインスレッドの取得
            MainThread = Thread.CurrentThread;

            // VisualStyleを有効にする（.manifestは必要なし）
            Application.EnableVisualStyles();
//			Application.SetCompatibleTextRenderingDefault(false);
            Application.DoEvents();

            // ＵＩ言語初期化
            if (File.Exists(AssemblyDirectory + "\\en\\3dEditor.resources.dll"))
            {
                Thread.CurrentThread.CurrentUICulture = new CultureInfo("en", false);
                IsEnglishUI = true;
            }

#if DEBUG
            // デバッグコンソール作成
            using (var watch = new DebugStopWatch("DebugConsole.Initialize()"))
            {
                DebugConsole.Initialize();
            }
#endif

            // コンフィグの設定
            using (var watch = new DebugStopWatch("ApplicationConfig.Load()"))
            {
                ApplicationConfig.Load(CommandLineOptionManager.TeamSettingsFile);
            }

            // テンポラリファイルの初期化
            using (var watch = new DebugStopWatch("TemporaryFileUtility.Initialize()"))
            {
                TemporaryFileUtility.Initialize();
            }

            // スタートアップパスがなければ作成
            DocumentManager.CreateStartupFolderIfNotExists();

            // G3DHIOLIBプロクシーの初期化
            using (var watch = new DebugStopWatch("G3dHioLibProxy.Initialize()"))
            {
                G3dHioLibProxy.Initialize();
            }

            // テクスチャユーティリティプロクシーの初期化
            using (var watch = new DebugStopWatch("Data.TexUtilsProxy.Initialize()"))
            {
                Data.TexUtilsProxy.Initialize();
            }

            // バイナリコンバーターマネージャーの初期化
            using (var watch = new DebugStopWatch("BinaryConverterManager.Initialize()"))
            {
                BinaryConverterManager.Initialize();
            }

            // シェーダーコンバーターマネージャーの初期化
            using (var watch = new DebugStopWatch("ShdrcvtrManager.Initialize()"))
            {
                ShdrcvtrManager.Initialize();
            }

            // バッググラウンドタスクマネージャーの初期化
            using (var watch = new DebugStopWatch("BackgroundTaskManager.Initialize()"))
            {
                BackgroundTaskManager.Initialize();
            }

            // テクスチャコンバーターマネージャーの初期化
            using (var watch = new DebugStopWatch("TexcvtrManager.Initialize()"))
            {
                TexcvtrManager.Initialize();
            }

            // HIOユーティリティの初期化
            using (var watch = new DebugStopWatch("Viewer.HioUtility.Initialize()"))
            {
                Viewer.HioUtility.Initialize();
            }

            // ErrorLogの初期化
            using (var watch = new DebugStopWatch("ErrorLog.Initialize()"))
            {
                ErrorLog.Initialize();
            }

            // BridgeManagerの初期化
            using (var watch = new DebugStopWatch("BridgeManager.Initialize()"))
            {
                App.Bridge.BridgeManager.Initialize();
            }

            // LightAnimationのデフォルト値を初期化する
            LightAnimation.InitializeTargetBases();

            // スレッド例外を補足する
            Application.ThreadException += (sender, e) =>
            {
                OnFatalErrorOccured(e.Exception, true);

                // メッセージループを抜けないので強制終了させる
                Application.Exit();
            };

            // コマンドマネージャ作成
            using (var watch = new DebugStopWatch("CommandManager = new CommandManager()"))
            {
                CommandManager = new CommandManager();
            }
        }

        public static void InitializeAfterMainFrameCreate()
        {
            OneShotIdleProcess.Execute(() =>
            {
                // シェーダーコンバーターマネージャーのGL関連の初期化
                using (var watch = new DebugStopWatch("ShdrcvtrManager.InitializeGL()"))
                {
                    ShdrcvtrManager.InitializeGL();
                }

                // メモリに乗せる為、DLL関数を一度呼んでみる。
                using (var watch = new DebugStopWatch("ShdrcvtrManager.Initialize2()"))
                {
                    ShdrcvtrManager.Initialize2();
                }
            });

            // ViewerManager初期化
            using (var watch = new DebugStopWatch("Viewer.Manager.Instance.Initialize()"))
            {
                Viewer.Manager.Instance.Initialize();
            }

            // ログ作成
            using (var watch = new DebugStopWatch("MessageLog.Initialize()"))
            {
                MessageLog.Initialize();
            }

            // スタートアップフォルダからの読み込み
            using (var watch = new DebugStopWatch("DocumentManager.LoadFromStartup()"))
            {
                DocumentManager.LoadFromStartup();
            }
            // コマンドラインオプションからの読み込み
            using (var watch = new DebugStopWatch("DocumentManager.LoadFromCommandLine()"))
            {
                DocumentManager.LoadFromCommandLine();
            }
        }

        public static void Destroy()
        {
            // ログ終了
            MessageLog.Terminate();

            // ViewerManager後処理
            Viewer.Manager.Instance.Destroy();

            // HIOユーティリティの開放
            Viewer.HioUtility.Destroy();

            // テクスチャコンバーターマネージャーの開放
            TexcvtrManager.Destroy();

            // バッググラウンドタスクマネージャーの開放
            BackgroundTaskManager.Destroy();

            // シェーダーコンバーターマネージャーの関数
            ShdrcvtrManager.Destroy();

            // バイナリコンバートマネージャーの開放
            BinaryConverterManager.Destroy();

            // テクスチャユーティリティプロクシーの開放
            Data.TexUtilsProxy.Destroy();

            // G3DHIOLIBプロクシーの開放
            G3dHioLibProxy.Destroy();

            // シェーダーのウォッチを終了
            foreach (var sdef in DocumentManager.ShaderDefinitions)
            {
                sdef.DisableSourceWatcher();
            }

            // デバッグコンソール終了
            DebugConsole.Terminate();

            ApplicationConfig.Save();
        }

#region 例外発生時処理
        /// <summary>
        /// 致命的なエラー発生時の処理。
        /// </summary>
        public static void OnFatalErrorOccured(Exception error, bool fromThreadException)
        {
            // 一度処理したら処理させない
            if (fatalErrorHandled_) { return; }

            string errorMessage = App.res.Strings.FatalError;
            // ログファイルを出力
            try
            {
                // ログが上書きされないようにファイル名に作成日時を入れる
                DateTime dtNow = DateTime.Now;
                var logFileName = "error_";
                logFileName += dtNow.ToString();
                logFileName += ".log";
                logFileName = logFileName.Replace("/", "");
                logFileName = logFileName.Replace(":", "");
                var logFilePath = Path.Combine(TheApp.LocalAppDataDirectory, logFileName);

                var writer = new StreamWriter(logFilePath, false);
                writer.WriteLine(GetErrorMessage(error));
                writer.WriteLine(error.StackTrace);
                writer.Close();

                errorMessage += "\n\n\"" + logFilePath + "\"\n" + App.res.Strings.FatalError_Log;
            }
            // ログファイル出力失敗時はもうどうしようもない
            catch (Exception) {}

            if (ApplicationConfig.Error.Mail != null)
            {
                if (UIMessageBox.MailCancel(errorMessage))
                {
                    string debugMessage = "";
#if DEBUG
                    string buildDate = "DEBUG";
#else
                    string buildDate = VersionInformation.BuildDate.ToString();
#endif
                    // デバッグ用付加情報
                    debugMessage += "[Version]: " + string.Format("{0} {1} Rvn. {2} ({3})",
                        VersionInformation.Name,
                        VersionInformation.Version,
                        VersionInformation.SvnRevision,
                        buildDate);

                    debugMessage += "\n[From]: " + (fromThreadException ? "ThreadException" : "WinMain");
                    debugMessage += "\n[Message]: " + GetErrorMessage(error);
                    debugMessage += "[StackTrace]: " + error.StackTrace;

                    System.Text.StringBuilder builder = new System.Text.StringBuilder();
                    builder.Append("mailto:");
                    if (!string.IsNullOrEmpty(ApplicationConfig.Error.Mail.to))
                    {
                        builder.Append(ApplicationConfig.Error.Mail.to);
                    }

                    string separator = "?";
                    if (!string.IsNullOrEmpty(ApplicationConfig.Error.Mail.cc))
                    {
                        builder.Append(separator);
                        separator = "&";
                        builder.Append("cc=");
                        builder.Append(ApplicationConfig.Error.Mail.cc);
                    }

                    if (!string.IsNullOrEmpty(ApplicationConfig.Error.Mail.bcc))
                    {
                        builder.Append(separator);
                        separator = "&";
                        builder.Append("bcc=");
                        builder.Append(ApplicationConfig.Error.Mail.bcc);
                    }

                    if (!string.IsNullOrEmpty(ApplicationConfig.Error.Mail.subject))
                    {
                        builder.Append(separator);
                        separator = "&";
                        builder.Append("subject=");
                        builder.Append(PartialURLEncode(ApplicationConfig.Error.Mail.subject));
                    }

                    {
                        builder.Append(separator);
                        separator = "&";
                        builder.Append("body=");
                        builder.Append(PartialURLEncode(debugMessage));
                    }

                    Process.Start(builder.ToString());
                }
            }
            else
            {
#if DEBUG
                string buildDate = "DEBUG";
                string debugMessage = "";
                // デバッグ用付加情報
                debugMessage += "[Version]: " + string.Format("{0} {1} Rvn. {2} ({3})",
                    VersionInformation.Name,
                    VersionInformation.Version,
                    VersionInformation.SvnRevision,
                    buildDate);

                debugMessage += "\n[From]: " + (fromThreadException ? "ThreadException" : "WinMain");
                debugMessage += "\n[Message]: " + GetErrorMessage(error);
                debugMessage += "[StackTrace]: " + error.StackTrace;
                errorMessage += "\n" + debugMessage;

#endif
                try
                {
                    // エラーメッセージ表示
                    UIMessageBox.Error(errorMessage);
                }
                catch (Exception)
                {
                    // クリップボードの操作で ExternalException が発生した場合はここで無効にする
                }
            }

            var tgaFiles = new List<string>();
            var saveDocuments = new List<Document>();
            MainFrame.GetSaveDocuments(saveDocuments, tgaFiles);
            if (saveDocuments.Any())
            {
                if (UIMessageBox.OkCancel(Strings.TheApp_OnFatalErrorOccured_FatalError_FinalChance_SaveAllTo))
                {
                    try
                    {
                        using (var propertyChangedSuppressBlock = new App.AppContext.PropertyChangedSuppressBlock())
                        {
                            propertyChangedSuppressBlock.IsIgnore = true;
                            var saved = (new DocumentSaver()).SaveWithChildren(saveDocuments);
                            if (saved)
                            {
                                UIMessageBox.Information(Strings.TheApp_OnFatalErrorOccured_FinalSave_Success);
                            }
                        }
                    }
                    catch (Exception)
                    {
                        try
                        {
                            UIMessageBox.Error(Strings.TheApp_OnFatalErrorOccured_FainalSave_Fail);
                        }
                        catch (Exception)
                        {
                            // クリップボードの操作で ExternalException が発生した場合はここで無効にする
                        }
                    }
                }
            }

            fatalErrorHandled_ = true;
        }

        public static string PartialURLEncode(string text)
        {
            // http://dobon.net/vb/dotnet/process/openbrowser.html
            string[] sources = new string[] {"%", "\n", "&", "=", "?", ","};
            foreach (var source in sources)
            {
                string replaced = HttpUtility.UrlEncode(source);
                text = text.Replace(source, replaced);
            }
            return text;
        }

        /// <summary>
        /// デバッグ用にpublic に変更
        /// </summary>
        public static string GetErrorMessage(Exception error)
        {
            string result = error.GetType().ToString() + ": " + error.Message + "\n";
            if (error.InnerException != null)
            {
                result += "\n	" + GetErrorMessage(error.InnerException);
            }
            return result;
        }

        /// <summary>
        /// スタックトレース
        /// </summary>
        public static string GetStackTrace(Exception error)
        {
            string result = error.StackTrace + "\n";
            if (error.InnerException != null)
            {
                return GetStackTrace(error.InnerException) + "	" + result;
            }
            return result;
        }
#endregion
    }
}
