﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Linq;
using App.Data;
using App.FileView;
using ConfigCommon;
using App;
using App.Utility;
using App.ConfigData;

namespace App
{
    public static class AppContext
    {
        public static event EventHandler SelectedTargetChanged
        {
            add
            {
                SelectedTarget.Changed += value;
            }
            remove
            {
                SelectedTarget.Changed -= value;
            }
        }

        public static event DocumentPropertyChangedEventHandler PropertyChanged = null;
        public static event DocumentAddOrRemovedEventHandler DocumentAddedOrRemoved = null;
        public static event Action<Document> DocumentSaved = null;
        public static void NotifyDocumentSaved(Document document)
        {
            if (DocumentSaved != null)
            {
                DocumentSaved(document);
            }
        }

        public static event EventHandler PreviewCurrentFrameChanged = null;
        public static void NotifyPreviewCurrentFrameChanged()
        {
            if (PreviewCurrentFrameChanged != null)
            {
                PreviewCurrentFrameChanged(null, EventArgs.Empty);
            }
        }
        public static double PreviewCurrentFrame
        {
            get { return TheApp.MainFrame.PreviewCurrentFrame;  }
            set { TheApp.MainFrame.PreviewCurrentFrame = value; }
        }
        public static bool IsEdittingCurrentFrame
        {
            get { return TheApp.MainFrame.IsEdittingCurrentFrame;  }
            set { TheApp.MainFrame.IsEdittingCurrentFrame = value; }
        }

        public static event Action<bool> PauseChangedEvent;
        public static void OnPauseChanged(bool stateChanged)
        {
            PauseChangedEvent?.Invoke(stateChanged);
        }

        public static event Action SpecChanged;
        public static void OnSpecChanged()
        {
            SpecChanged?.Invoke();
        }

        public static event Action MaterialReferenceBehaviorSettingChanged;
        public static void OnMaterialReferenceBehaviorSettingChanged()
        {
            MaterialReferenceBehaviorSettingChanged?.Invoke();
        }

        public static GuiObjectGroup selectedTarget_ = new GuiObjectGroup();

        public static GuiObjectGroup SelectedTarget
        {
            get
            {
                return selectedTarget_;
            }
        }


        public static FileTreeView FileTreeView { get; set; }
        public static GuiObject SelectedFileViewObject => FileTreeView?.SelectedFileViewObject;
        public static GuiObject SelectedFileViewObjectOwner => FileTreeView?.SelectedFileViewObjectOwner;
        public static IEnumerable<GuiObject> SelectedFileViewObjects => FileTreeView?.SelectedFileViewObjects?.Distinct();

        public static GuiObject CurrentSelectedObject(MenuCommandArgs args)
        {
            return CurrentSelectedObject(args.CommandUI.IsObjectViewContextMenu);
        }

        public static GuiObject CurrentSelectedObject(bool isObjectViewContextMenu)
        {
            return isObjectViewContextMenu ? SelectedTarget.Active : SelectedFileViewObject;
        }

        public static event MenuCommandHandler BeginRenameEvent;
        public static void BeginRename(MenuCommandArgs args)
        {
            if (args.RequireUpdate)
            {
                args.CommandUI.Enabled = false;
            }

            if (BeginRenameEvent != null)
            {
                BeginRenameEvent(args);
            }
        }
        //---------------------------------------------------------------------
        #region プロパティ変更抑制ブロック
        /// <summary>
        /// プロパティ変更抑制ブロック
        /// </summary>
        public sealed class PropertyChangedSuppressBlock : IDisposable
        {
            private static readonly List<PropertyChangedSuppressBlock> _instances = new List<PropertyChangedSuppressBlock>();
            private static readonly DocumentPropertyChangedEventArgs _args = new DocumentPropertyChangedEventArgs();

            public bool IsIgnore { get; set; }

            /// <summary>
            /// コンストラクタ。
            /// </summary>
            public PropertyChangedSuppressBlock()
            {
                _instances.Add(this);

                IsIgnore = false;
            }

            /// <summary>
            /// 開放処理。
            /// </summary>
            public void Dispose()
            {
                Debug.Assert(_instances.Count > 0);
                _instances.Remove(this);

                if (_instances.Count == 0)
                {
                    if (IsIgnore == false)
                    {
                        // イベントの発行
                        App.AppContext.NotifyPropertyChanged(null, _args);
                    }
                    else
                    {
                        IsIgnore = false;
                    }

                    _args.Clear();
                }
            }

            /// <summary>
            /// 通知
            /// </summary>
            public static bool Notify(DocumentPropertyChangedEventArgs args)
            {
                if (_instances.Count > 0)
                {
                    _args.Merge(args);
                    return true;
                }
                return false;
            }

            /// <summary>
            /// 一旦通知
            /// </summary>
            public bool Release()
            {
                if (_instances.Count == 1 && _instances[0] == this)
                {
                    _instances.Remove(this);
                    // イベントの発行
                    App.AppContext.NotifyPropertyChanged(null, _args);
                    _args.Clear();
                    _instances.Add(this);
                    return true;
                }
                return false;
            }
        }
        #endregion

        /// <summary>
        /// ドキュメントプロパティ変更通知。
        /// </summary>
        public static void NotifyPropertyChanged(object sender, DocumentPropertyChangedArgs args)
        {
            NotifyPropertyChanged(sender, new DocumentPropertyChangedEventArgs(args));
        }

        /// <summary>
        /// ドキュメントプロパティ変更通知。
        /// </summary>
        public static void NotifyPropertyChanged(object sender, DocumentPropertyChangedEventArgs args)
        {
            if (!args.HasArgs()) { return; }

            // 抑制処理
            if (PropertyChangedSuppressBlock.Notify(args)) { return; }

            DebugConsole.WriteLine("event:DocumentPropertyChanged args={0} objects={1}",
                args.GetArgs().Count, args.GetObjects().Count);

            foreach (var arg in args.GetArgs())
            {
                if (arg.Command != null)
                {
                    if (arg.Target.ObjectID == GuiObjectID.Project)
                    {
                        var project = (ProjectDocument)arg.Target;
                        if (project.DefaultProjectModifiedCommand == arg.Command)
                        {
                            project.DefaultProjectModifiedCommand = null;
                            project.DefaultProject = true;
                        }
                        else if (project.DefaultProject)
                        {
                            bool modify = true;
                            if (arg is DocumentAddedOrRemovedArg)
                            {
                                modify = false;
                            }
                            else if (arg is DocumentAnimationBindArg)
                            {
                                if (((DocumentAnimationBindArg)arg).AutoBound)
                                {
                                    modify = false;
                                }
                            }
                            else if (arg is ObjectRenamedArg)
                            {
                                if (((ObjectRenamedArg)arg).RenamedObject != project)
                                {
                                    modify = false;
                                }
                            }

                            if (modify)
                            {
                                project.DefaultProjectModifiedCommand = arg.Command;
                                project.DefaultProject = false;
                            }
                        }
                    }
                }
            }

            // 同名ファイル情報の更新
            DocumentManager.UpdateIndexInSameNames();

            Document[] addedDocuments = args.GetArgs().Where(x => x is DocumentAddedOrRemovedArg).SelectMany(x => ((DocumentAddedOrRemovedArg)x).AddedDocuments).Distinct().ToArray();
            Document[] removedDocuments = (from arg in args.GetArgs().OfType<DocumentAddedOrRemovedArg>()
                                          from document in arg.RemovedDocuments
                                          select document).Distinct().ToArray();

            var addedGroups = (from added in addedDocuments
                               where added.ObjectID != GuiObjectID.Project
                               from content in added.ContentObjects
                              group content by (content.Name + "/" + content.ObjectID.ToString())).ToArray();
            var removedGroups = (from removed in removedDocuments
                                 where removed.ObjectID != GuiObjectID.Project
                                 from content in removed.ContentObjects
                                group content by (content.Name + "/" + content.ObjectID.ToString())).ToArray();

            var swapedContents = new Dictionary<GuiObject, GuiObject>();
            foreach (var addedGroup in addedGroups)
            {
                foreach (var removedGroup in removedGroups)
                {
                    if (addedGroup.Key == removedGroup.Key)
                    {
                        var removedList = removedGroup.ToList();
                        var addedList = addedGroup.ToList();
                        if (addedList.Count > removedList.Count)
                        {
                            // 開かれた回数が多い場合は残りが入れ替え
                            // 閉じられた回数が多い場合は最後以外が入れ替え
                            addedList.RemoveAt(0);
                        }

                        foreach (var p in removedList.Zip(addedList, (removed, added) => new { removed, added }))
                        {
                            swapedContents.Add(p.removed, p.added);
                        }
                    }
                }
            }

            var removedContents = (from document in removedDocuments
                                   from content in document.ContentObjects
                                   select content).Except(swapedContents.Keys);

            var reloadedDocuments = (from arg in args.GetArgs().OfType<Bridge.BridgeManager.DocumentResetArgs>()
                                     select (Document)arg.Target).Distinct().ToArray();

            if (DocumentAddedOrRemoved != null && (addedDocuments.Any() || removedDocuments.Any() || reloadedDocuments.Any()))
            {
                // アニメーションをプレビュー対象としてバインドしているモデルを更新
                DocumentManager.UpdatePreviewAnimations();

                // モデルとテクスチャの順番を更新
                DocumentManager.UpdateModelsAndTexturesOrder(swapedContents);

                DocumentAddedOrRemoved(sender, addedDocuments, removedDocuments, swapedContents, reloadedDocuments);
            }

            using (var block = new Utility.SuppressBlock(SelectedTarget.InvokeBlockCounter))
            {
                foreach (var swaped in swapedContents)
                {
                    SelectedTarget.Swap(swaped.Key, swaped.Value);
                }

                foreach (var reloaded in reloadedDocuments)
                {
                    foreach (var selected in SelectedTarget.Objects.ToArray())
                    {
                        if (selected.OwnerDocument == reloaded)
                        {
                            var newContent = reloaded.ContentObjects.FirstOrDefault(x => x.ObjectID == selected.ObjectID && x.Name == selected.Name);
                            if (newContent != null)
                            {
                                SelectedTarget.Swap(selected, newContent);
                            }
                            else
                            {
                                SelectedTarget.Remove(selected);
                            }
                        }
                    }
                }

                foreach (var removed in SelectedTarget.Objects.Where(x => removedDocuments.Contains(x.OwnerDocument)).ToArray())
                {
                    SelectedTarget.Remove(removed);
                }
            }

            if (PropertyChanged != null)
            {
                PropertyChanged(sender, args.GetArgs());
            }
        }

        public static void ExecutePropertyChangedEvent(object sender, IEnumerable<DocumentPropertyChangedArgs> args)
        {
            if (PropertyChanged != null)
            {
                PropertyChanged(sender, args);
            }
        }

        private static  INotificationHandler _notificationHandler = new NotificationHandler();
        private static  INotificationHandler _customNotificationHandler = null;
        public static INotificationHandler NotificationHandler
        {
            get { return _customNotificationHandler ?? _notificationHandler; }
            set { _customNotificationHandler = value; }
        }



        // UI スレッド用
        public static TeamConfig.PlatformPreset SelectedPlatformPreset
        {
            get;
            set;
        }

        #region 外部からの変更抑制
        public static SuppressBlockCounter UpdateFromOutSideBlockCounter = new SuppressBlockCounter();
        #endregion
    }
}
