﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "stdafx.h"

#include "msclr/marshal.h"
#include "msclr/marshal_windows.h"
#include "msclr/marshal_cppstd.h"
#include "msclr/marshal_atl.h"
#include "HIO.h"

#include "CommandMaker.h"
#include "CommandUtility.h"
#include "Utility.h"
#include <nw/g3d/res/g3d_ResUtility.h>
#include <nw/g3d/ut/g3d_Inlines.h>
#include <nn/g3d/viewer/detail/g3d_ViewerPacketDefine.h>
#include <nn/g3d/detail/g3d_Inlines.h>

using namespace System;
using namespace msclr::interop;
using namespace nw::g3d::edit::detail;

namespace
{
String^ ConvertHostFileIOPath(String^ fileName)
{
    if (fileName == nullptr)
    {
        return nullptr;
    }

    // HostFileIO 用のパスに変換
    String^ rootPath = System::IO::Path::GetPathRoot(fileName);
    String^ hfioRootPath = rootPath->Replace(":\\", "/")->ToLower();
    String^ hfioFileName = fileName->Replace(rootPath, hfioRootPath)->Replace("\\", "/");

    String^ result = String::Concat("/", hfioFileName);

    return result;
}

}

namespace NintendoWare { namespace G3d { namespace Edit {

int CommandMaker::CalcSystemPacketSize()
{
    return static_cast<int>(nw::g3d::tool::Align(CommandUtility::GetPacketHeaderSize(), NW_G3D_EDIT_ALIGNMENT));
}

int CommandMaker::CalcFileDataPacketSize(String^ fileName)
{
    if (fileName == nullptr)
    {
        return -1;
    }

    String^ hfioFileName = String::Empty;
    std::string hfioFullPath("");
    if (fileName != String::Empty)
    {
        hfioFileName = ConvertHostFileIOPath(fileName);
        hfioFullPath = marshal_as<std::string>(hfioFileName);
    }

    size_t pathSize = hfioFullPath.length();
    if (pathSize >= NW_G3D_EDIT_FILENAME_MAX)
    {
        return -1;
    }

    size_t totalSize =
        nw::g3d::tool::Align(CommandUtility::GetPacketHeaderSize() + CommandUtility::GetFileDataBlockSize(), NW_G3D_EDIT_ALIGNMENT);

    return static_cast<int>(totalSize);
}

int CommandMaker::CalcEditValuePacketSize(array<INT32>^ indices)
{
    if (indices == nullptr)
    {
        return -1;
    }

    if (indices->Length <= 0)
    {
        return -1;
    }

    size_t indexByteSize = indices->Length * sizeof(s32);
    size_t valueBlockSize = sizeof(EditSimpleValue) + indexByteSize;

    size_t totalSize =
        nw::g3d::tool::Align(CommandUtility::GetPacketHeaderSize() + CommandUtility::GetEditValueInfoBlockSize() + valueBlockSize, NW_G3D_EDIT_ALIGNMENT);
    return static_cast<int>(totalSize);
}

int CommandMaker::CalcModelShaderArchivePacketSize(
    String^ bfresFile, array<String^>^ fileNames, array<MaterialShaderInfo^>^ materialShaderInfos)
{
    if (bfresFile == nullptr)
    {
        return -1;
    }

    int fileCount = fileNames->Length;
    if (materialShaderInfos == nullptr)
    {
        return -1;
    }

    int materialCount = materialShaderInfos->Length;
    if (materialCount <= 0)
    {
        return -1;
    }

    std::vector<std::string> convertFileNames;
    for (int i = 0; i < fileCount; ++i)
    {
        String^ hfioFileName = String::Empty;
        std::string hfioFullPath("");
        if (!String::IsNullOrEmpty(fileNames[i]))
        {
            hfioFileName = ConvertHostFileIOPath(fileNames[i]);
            hfioFullPath = marshal_as<std::string>(hfioFileName);
        }
        u32 pathSize = static_cast<u32>(hfioFullPath.length());
        if (pathSize >= NW_G3D_EDIT_FILENAME_MAX)
        {
            return -1;
        }
        convertFileNames.push_back(hfioFullPath);
    }

    // bfres 検査
    {
        String^ hfioFileName = String::Empty;
        std::string hfioFullPath("");
        if (bfresFile != String::Empty)
        {
            hfioFileName = ConvertHostFileIOPath(bfresFile);
            hfioFullPath = marshal_as<std::string>(hfioFileName);
        }
        u32 pathSize = static_cast<u32>(hfioFullPath.length());
        if (pathSize >= NW_G3D_EDIT_FILENAME_MAX)
        {
            return -1;
        }
        convertFileNames.push_back(hfioFullPath);
    }

    // bfres 分追加
    ++fileCount;

    size_t blockSize = nw::g3d::tool::Align(CommandUtility::GetModelOptimizedShaderBlockSize(fileCount, materialCount), NW_G3D_EDIT_ALIGNMENT);
    size_t totalFileNamesSize = 0;
    for (int i = 0; i < fileCount; ++i)
    {
        const std::string& fileName = convertFileNames[i];
        totalFileNamesSize += nw::g3d::tool::Align(fileName.length() + 1, NW_G3D_EDIT_ALIGNMENT);
    }

    size_t alignedSize =  nw::g3d::tool::Align(blockSize + totalFileNamesSize, NW_G3D_EDIT_ALIGNMENT);
    return static_cast<int>(alignedSize);
}

bool CommandMaker::MakeSystemPacket(
    array<Byte>^ packetBuffer,
    CommandFlag command,
    TargetEndianKind endianKind)
{
    if (packetBuffer == nullptr)
    {
        return false;
    }
    if (packetBuffer->Length <= 0)
    {
        return false;
    }

    size_t totalSize =
        nw::g3d::tool::Align(CommandUtility::GetPacketHeaderSize(), NW_G3D_EDIT_ALIGNMENT);

    if (packetBuffer->Length != totalSize)
    {
        return false;
    }

    pin_ptr<u8> bufferPtr = &packetBuffer[0];

    PacketHeader* packet = reinterpret_cast<PacketHeader*>(bufferPtr);
    *packet = CommandUtility::GetPacketHeader();
    packet->command = command;
    packet->dataSize = static_cast<s32>(totalSize - CommandUtility::GetPacketHeaderSize());

    if (endianKind == TargetEndianKind::BigEndian)
    {
        nw::g3d::tool::Endian::Swap(packet);
    }

    // pin_ptr 解除
    bufferPtr = nullptr;
    return true;
}

bool CommandMaker::MakeFileDataPacket(
    array<Byte>^ packetBuffer,
    CommandFlag command,
    FileDataKind kind,
    UINT32 key,
    UINT32 resFileKey,
    FileData^ fileData,
    TargetEndianKind endianKind)
{
    if (packetBuffer == nullptr)
    {
        return false;
    }
    if (packetBuffer->Length <= 0)
    {
        return false;
    }

    if (fileData->FileName == nullptr)
    {
        return false;
    }

    u32 pathSize = 0;
    u32 fileSize = 0;
    std::string hfioFullPath("");
    if (fileData->FileName != String::Empty)
    {
        if (!System::IO::File::Exists(fileData->FileName))
        {
            return false;
        }

        System::IO::FileInfo^ fileInfo = gcnew System::IO::FileInfo(fileData->FileName);
        fileSize = static_cast<u32>(fileInfo->Length);

        String^ hfioFileName = ConvertHostFileIOPath(fileData->FileName);
        hfioFullPath = marshal_as<std::string>(hfioFileName);

        pathSize = static_cast<u32>(hfioFullPath.length());
        if (pathSize >= NW_G3D_EDIT_FILENAME_MAX)
        {
            return false;
        }
    }

    size_t totalSize =
        nw::g3d::tool::Align(CommandUtility::GetPacketHeaderSize() + CommandUtility::GetFileDataBlockSize(), NW_G3D_EDIT_ALIGNMENT);

    if (packetBuffer->Length != totalSize)
    {
        return false;
    }

    pin_ptr<u8> bufferPtr = &packetBuffer[0];

    FileDataPacket* packet = reinterpret_cast<FileDataPacket*>(bufferPtr);
    packet->header = CommandUtility::GetPacketHeader();
    packet->header.command = command;
    packet->header.dataSize = static_cast<s32>(totalSize - CommandUtility::GetPacketHeaderSize());
    packet->block.key = key;
    packet->block.resFileKey = resFileKey;
    packet->block.kind = kind;
    packet->block.fileSize = fileSize;
    packet->block.pathSize = pathSize;
    packet->block.fileAlignment = fileData->Alignment;

    memset(&packet->block.fileName, 0, NW_G3D_EDIT_FILENAME_MAX);
    std::memcpy(&packet->block.fileName, hfioFullPath.c_str(), hfioFullPath.length());

    if (endianKind == TargetEndianKind::BigEndian)
    {
        nw::g3d::tool::Endian::Swap(packet);
    }

    // pin_ptr 解除
    bufferPtr = nullptr;
    return true;
}

bool CommandMaker::OverwriteModifiedShaderProgramPart(
    array<Byte>^ packetBuffer,
    INT32 shadingModelIndex,
    INT32 shaderProgramIndex,
    TargetEndianKind endianKind)
{
    if (packetBuffer == nullptr)
    {
        return false;
    }
    if (packetBuffer->Length <= 0)
    {
        return false;
    }

    pin_ptr<u8> bufferPtr = &packetBuffer[0];

    FileDataPacket* packet = reinterpret_cast<FileDataPacket*>(bufferPtr);

    packet->block.shadingModelIndex = shadingModelIndex;
    packet->block.shaderProgramIndex = shaderProgramIndex;

    if (endianKind == TargetEndianKind::BigEndian)
    {
        nw::g3d::tool::Endian::Swap(&packet->block.shadingModelIndex);
        nw::g3d::tool::Endian::Swap(&packet->block.shaderProgramIndex);
    }

    // pin_ptr 解除
    bufferPtr = nullptr;
    return true;
}

generic<typename ValueType>
bool CommandMaker::MakeEditValuePacket(
    array<Byte>^ packetBuffer,
    CommandFlag command,
    EditTargetKind editTargetKind,
    UINT32 key,
    array<INT32>^ indices,
    ValueType value,
    TargetEndianKind endianKind)
{
    if (packetBuffer == nullptr)
    {
        return false;
    }
    if (packetBuffer->Length <= 0)
    {
        return false;
    }

    if (indices == nullptr)
    {
        return false;
    }

    if (indices->Length <= 0)
    {
        return false;
    }

    size_t indexByteSize = indices->Length * sizeof(s32);
    size_t valueBlockSize = sizeof(EditSimpleValue) + indexByteSize;

    size_t totalSize =
        nw::g3d::tool::Align(CommandUtility::GetPacketHeaderSize() + CommandUtility::GetEditValueInfoBlockSize() + valueBlockSize, NW_G3D_EDIT_ALIGNMENT);

    if (packetBuffer->Length != totalSize)
    {
        return false;
    }

    pin_ptr<u8> bufferPtr = &packetBuffer[0];

    EditValuePacket* packet = reinterpret_cast<EditValuePacket*>(bufferPtr);
    packet->header = CommandUtility::GetPacketHeader();
    packet->header.command = command;
    packet->header.dataSize = static_cast<s32>(totalSize - CommandUtility::GetPacketHeaderSize());

    packet->block.key = key;
    packet->block.editTargetKind = editTargetKind;
    packet->block.length = static_cast<u32>(packet->header.dataSize - CommandUtility::GetEditValueInfoBlockSize());
    packet->block.indexSize = indices->Length;

    EditValueBlock* valueBlock = reinterpret_cast<EditValueBlock*>(&packet[1]);

    for (u32 i = 0, end = packet->block.indexSize; i < end; ++i)
    {
        valueBlock->index[i] = indices[i];
    }

    if (ValueType::typeid == bool::typeid)
    {
        packet->block.valueKind = BOOL_VALUE;
        valueBlock->value.bValue = Convert::ToBoolean(value);
        if (valueBlock->value.bValue)
        {
            valueBlock->value.uValue = 1;
        }
        else
        {
            valueBlock->value.uValue = 0;
        }
    }
    else if (ValueType::typeid == int::typeid)
    {
        packet->block.valueKind = S32_VALUE;
        valueBlock->value.sValue = Convert::ToInt32(value);
    }
    else if (ValueType::typeid == unsigned int::typeid)
    {
        packet->block.valueKind = U32_VALUE;
        valueBlock->value.uValue = Convert::ToUInt32(value);
    }
    else if (ValueType::typeid == float::typeid)
    {
        packet->block.valueKind = F32_VALUE;
        valueBlock->value.fValue = Convert::ToSingle(value);
    }
    else
    {
        // pin_ptr 解除
        bufferPtr = nullptr;
        // 対象外の型は失敗にする
        return false;
    }

    if (endianKind == TargetEndianKind::BigEndian)
    {
        u32 indexSize = packet->block.indexSize;
        nw::g3d::tool::Endian::Swap(packet);

        if (ValueType::typeid == bool::typeid)
        {
            nw::g3d::tool::Endian::SwapValue(valueBlock, true);
        }
        else
        {
            nw::g3d::tool::Endian::SwapValue(valueBlock, false);
        }

        nw::g3d::tool::Endian::SwapIndices(valueBlock, indexSize);
    }
    // pin_ptr 解除
    bufferPtr = nullptr;
    return true;
}

bool CommandMaker::MakeModelShaderArchivePacket(
    array<Byte>^ packetBuffer,
    UINT32 modelObjKey,
    FileData^ bfresFileData,
    array<String^>^ fileNames,
    array<MaterialShaderInfo^>^ materialOptimizeInfos,
    TargetEndianKind endianKind)
{
    if (packetBuffer == nullptr)
    {
        return false;
    }
    if (packetBuffer->Length <= 0)
    {
        return false;
    }

    if (bfresFileData->FileName == nullptr)
    {
        return false;
    }

    if (fileNames == nullptr)
    {
        return false;
    }

    if (materialOptimizeInfos == nullptr)
    {
        return false;
    }

    int materialCount = materialOptimizeInfos->Length;
    if (materialCount <= 0)
    {
        return false;
    }

    std::vector<std::string> convertFileNames;
    std::vector<u32> fileSizeInfos;
    int fileCount = fileNames->Length;
    for (int i = 0; i < fileCount; ++i)
    {
        u32 fileSize = 0;

        String^ hfioFileName = String::Empty;
        std::string hfioFullPath("");
        if (!String::IsNullOrEmpty(fileNames[i]))
        {
            hfioFileName = ConvertHostFileIOPath(fileNames[i]);
            hfioFullPath = marshal_as<std::string>(hfioFileName);

            System::IO::FileInfo^ fileInfo = gcnew System::IO::FileInfo(fileNames[i]);
            fileSize = static_cast<u32>(fileInfo->Length);
        }
        u32 pathSize = static_cast<u32>(hfioFullPath.length());
        if (pathSize >= NW_G3D_EDIT_FILENAME_MAX)
        {
            return false;
        }
        convertFileNames.push_back(hfioFullPath);
        fileSizeInfos.push_back(fileSize);
    }

    // bfres 分
    {
        System::IO::FileInfo^ fileInfo = gcnew System::IO::FileInfo(bfresFileData->FileName);
        u32 fileSize = static_cast<u32>(fileInfo->Length);

        String^ hfioFileName = String::Empty;
        std::string hfioFullPath("");
        if (bfresFileData->FileName != String::Empty)
        {
            hfioFileName = ConvertHostFileIOPath(bfresFileData->FileName);
            hfioFullPath = marshal_as<std::string>(hfioFileName);
        }
        u32 pathSize = static_cast<u32>(hfioFullPath.length());
        if (pathSize >= NW_G3D_EDIT_FILENAME_MAX)
        {
            return false;
        }
        convertFileNames.push_back(hfioFullPath);
        fileSizeInfos.push_back(fileSize);
    }

    // bfres 分追加
    ++fileCount;

    size_t totalSize = CommandMaker::CalcModelShaderArchivePacketSize(bfresFileData->FileName, fileNames, materialOptimizeInfos);
    if (packetBuffer->Length != totalSize)
    {
        return false;
    }

    MakeModelOptimizedShaderPacketImpl(
        packetBuffer,
        totalSize,
        modelObjKey,
        convertFileNames,
        materialCount,
        materialOptimizeInfos,
        fileCount,
        fileSizeInfos,
        bfresFileData,
        endianKind);

    return true;
}

void CommandMaker::MakeModelOptimizedShaderPacketImpl(
    array<Byte>^ packetBuffer,
    size_t totalSize,
    u32 modelObjKey,
    const std::vector<std::string>& convertFileNames,
    int materialCount,
    array<MaterialShaderInfo^>^ materialShaderInfos,
    int fileCount,
    const std::vector<u32>& fileSizeInfos,
    FileData^ bfresFileData,
    TargetEndianKind endianKind)
{
    pin_ptr<u8> bufferPtr = &packetBuffer[0];
    switch (HIO::GetInstance()->TargetDeviceType)
    {
    case HIOBase::TargetType::Cafe:
    case HIOBase::TargetType::OldPc:
        {
            ModelOptimizedShaderPacket* packet = reinterpret_cast<ModelOptimizedShaderPacket*>(bufferPtr);
            memset(packet, 0, packetBuffer->Length);

            packet->header = CommandUtility::GetPacketHeader();
            packet->header.command = EDIT_LOAD_SHADER_ARCHIVE_COMMAND_FLAG;
            packet->header.dataSize = static_cast<s32>(totalSize - CommandUtility::GetPacketHeaderSize());
            packet->block.modelObjKey = modelObjKey;
            packet->block.multiFile.numFile = static_cast<u16>(convertFileNames.size());
            packet->block.multiFile.loopCount = 0;

            // マテリアルがどのシェーダーアーカイブを使用しているかのインデックス配列
            MaterialShaderIndexData* indexPtrArray = nw::g3d::ut::AddOffset<MaterialShaderIndexData>(packet, sizeof(ModelOptimizedShaderPacket));
            size_t indexPtrArraySize = sizeof(MaterialShaderIndexData) + sizeof(s32) * (materialCount - 1);
            packet->block.ofsMaterialShaderIndices.set_ptr(indexPtrArray);
            indexPtrArray->numMaterial = materialCount;
            s32* ofsIndexPtr = &(indexPtrArray->index[0]);
            for (int i = 0; i < materialCount; ++i)
            {
                ofsIndexPtr[i] = materialShaderInfos[i]->ShaderArchiveIndex;
            }

            // シェーダーアーカイブファイル情報
            FileInfoData* fileInfoDataPtrArray = nw::g3d::ut::AddOffset<FileInfoData>(indexPtrArray, indexPtrArraySize);
            size_t fileInfoDataPtrArraySize = sizeof(FileInfoData) + sizeof(FileInfoData::OffsetFileInfo) * (fileCount - 1);
            packet->block.multiFile.ofsFileInfo.set_ptr(fileInfoDataPtrArray);

            ModelOptimizedShaderBlock* block = &packet->block;
            size_t fileNameOffset = nw::g3d::ut::Align(CommandUtility::GetModelOptimizedShaderBlockSize(fileCount, materialCount), NW_G3D_EDIT_ALIGNMENT) - CommandUtility::GetPacketHeaderSize();
            char* fileNamePtr = nw::g3d::ut::AddOffset<char>(block, fileNameOffset);
            FileInfoData::OffsetFileInfo* ofsFileInfoPtr = &(fileInfoDataPtrArray->fileInfo[0]);

            size_t offset = static_cast<u32>(fileNameOffset);
            size_t convertFileCount = convertFileNames.size();
            for (size_t i = 0; i < convertFileCount - 1; ++i)
            {
                const std::string& convertFile = convertFileNames[i];
                std::memcpy(fileNamePtr, convertFile.c_str(), convertFile.length());
                ofsFileInfoPtr[i].ofsFileName = static_cast<u32>(offset);
                ofsFileInfoPtr[i].fileSize = fileSizeInfos[i];
                ofsFileInfoPtr[i].fileDataKind = FILEDATA_SHADER_ARCHIVE;
                ofsFileInfoPtr[i].fileAlignment = nw::g3d::ut::DEFAULT_ALIGNMENT; // シェーダーのアライメントは固定
                offset += nw::g3d::tool::Align(convertFile.length() + 1, NW_G3D_EDIT_ALIGNMENT);
                fileNamePtr = nw::g3d::ut::AddOffset<char>(block, offset);
            }

            // bfres 分
            {
                size_t lastPos = convertFileCount - 1;
                const std::string& convertFile = convertFileNames[lastPos];
                std::memcpy(fileNamePtr, convertFile.c_str(), convertFile.length());
                ofsFileInfoPtr[lastPos].ofsFileName = static_cast<u32>(offset);
                ofsFileInfoPtr[lastPos].fileSize = fileSizeInfos[lastPos];
                ofsFileInfoPtr[lastPos].fileDataKind = FILEDATA_MODEL;
                ofsFileInfoPtr[lastPos].fileAlignment = bfresFileData->Alignment;
                offset += nw::g3d::tool::Align(static_cast<u32>(convertFile.length()) + 1, NW_G3D_EDIT_ALIGNMENT);
                fileNamePtr = nw::g3d::ut::AddOffset<char>(block, offset);
            }

            if (endianKind == TargetEndianKind::BigEndian)
            {
                nw::g3d::tool::Endian::Swap(packet);
            }
        }
        break;
    case HIOBase::TargetType::Htc:
        {
            nn::g3d::viewer::detail::ModelOptimizedShaderPacket* packet = reinterpret_cast<nn::g3d::viewer::detail::ModelOptimizedShaderPacket*>(bufferPtr);
            memset(packet, 0, packetBuffer->Length);

            packet->header = CommandUtility::GetNnG3dViewerPacketHeader();
            packet->header.command = EDIT_LOAD_SHADER_ARCHIVE_COMMAND_FLAG;
            packet->header.dataSize = static_cast<s32>(totalSize - CommandUtility::GetPacketHeaderSize());
            packet->block.modelObjKey = modelObjKey;
            packet->block.multiFile.numFile = static_cast<u16>(convertFileNames.size());
            packet->block.multiFile.loopCount = 0;

            // マテリアルがどのシェーダーアーカイブを使用しているかのインデックス配列
            nn::g3d::viewer::detail::MaterialShaderInfoArrayData* infoPtrArray =
                nw::g3d::ut::AddOffset<nn::g3d::viewer::detail::MaterialShaderInfoArrayData>(packet, sizeof(nn::g3d::viewer::detail::ModelOptimizedShaderPacket));
            {
                SetOffset((nn::util::BinPtr*)&packet->block.ofsMaterialShaderIndices, infoPtrArray);
                infoPtrArray->numMaterial = materialCount;
                nn::g3d::viewer::detail::MaterialShaderInfoData* ofsInfoPtr = &(infoPtrArray->arrayData[0]);

                for (int i = 0; i < materialCount; ++i)
                {
                    ofsInfoPtr[i].shaderArchiveIndex = materialShaderInfos[i]->ShaderArchiveIndex;
                    ofsInfoPtr[i].isOptimized = materialShaderInfos[i]->IsOptimized;
                    ofsInfoPtr[i].isOptimizationSkipped = materialShaderInfos[i]->IsOptimizationSkipped;
                }
            }

            // シェーダーアーカイブファイル情報
            size_t indexPtrArraySize = sizeof(nn::g3d::viewer::detail::MaterialShaderInfoArrayData) + sizeof(nn::g3d::viewer::detail::MaterialShaderInfoData) * (materialCount - 1);
            nn::g3d::viewer::detail::FileInfoData* fileInfoDataPtrArray = nw::g3d::ut::AddOffset<nn::g3d::viewer::detail::FileInfoData>(infoPtrArray, indexPtrArraySize);
            size_t fileInfoDataPtrArraySize = sizeof(nn::g3d::viewer::detail::FileInfoData) + sizeof(nn::g3d::viewer::detail::FileInfoData::OffsetFileInfo) * (fileCount - 1);
            SetOffset((nn::util::BinPtr*)&packet->block.multiFile.ofsFileInfo, fileInfoDataPtrArray);

            nn::g3d::viewer::detail::ModelOptimizedShaderBlock* block = &packet->block;
            size_t fileNameOffset = nw::g3d::ut::Align(CommandUtility::GetModelOptimizedShaderBlockSize(fileCount, materialCount), NW_G3D_EDIT_ALIGNMENT) - CommandUtility::GetPacketHeaderSize();
            char* fileNamePtr = nw::g3d::ut::AddOffset<char>(block, fileNameOffset);
            nn::g3d::viewer::detail::FileInfoData::OffsetFileInfo* ofsFileInfoPtr = &(fileInfoDataPtrArray->fileInfo[0]);
            size_t offset = fileNameOffset;
            size_t convertFileCount = convertFileNames.size();
            for (size_t i = 0; i < convertFileCount - 1; ++i)
            {
                const std::string& convertFile = convertFileNames[i];
                std::memcpy(fileNamePtr, convertFile.c_str(), convertFile.length());
                ofsFileInfoPtr[i].ofsFileName = static_cast<u32>(offset);
                ofsFileInfoPtr[i].fileSize = fileSizeInfos[i];
                ofsFileInfoPtr[i].fileDataKind = FILEDATA_SHADER_ARCHIVE;

                size_t alignment;
                NintendoWare::G3d::Edit::ReadResShaderFileAlignment(&alignment, marshal_as<System::String^>(convertFile));
                ofsFileInfoPtr[i].fileAlignment = static_cast<uint32_t>(alignment);

                offset += nw::g3d::tool::Align(convertFile.length() + 1, NW_G3D_EDIT_ALIGNMENT);
                fileNamePtr = nw::g3d::ut::AddOffset<char>(block, offset);
            }

            // bfres 分
            {
                size_t lastPos = convertFileCount - 1;
                const std::string& convertFile = convertFileNames[lastPos];
                std::memcpy(fileNamePtr, convertFile.c_str(), convertFile.length());
                ofsFileInfoPtr[lastPos].ofsFileName = static_cast<u32>(offset);
                ofsFileInfoPtr[lastPos].fileSize = fileSizeInfos[lastPos];
                ofsFileInfoPtr[lastPos].fileDataKind = FILEDATA_MODEL;
                ofsFileInfoPtr[lastPos].fileAlignment = bfresFileData->Alignment;
                offset += nw::g3d::tool::Align(static_cast<u32>(convertFile.length()) + 1, NW_G3D_EDIT_ALIGNMENT);
                fileNamePtr = nw::g3d::ut::AddOffset<char>(block, offset);
            }

            if (endianKind == TargetEndianKind::BigEndian)
            {
                // とりあえずビッグエンディアンは非サポート
                //nw::g3d::tool::Endian::Swap(packet);
                unexpected();
            }
        }
        break;
    default:
        unexpected();
    }

    // pin_ptr 解除
    bufferPtr = nullptr;
}

}}} // namespace NintendoWare.G3d.Edit
