﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;

namespace NintendoWare.G3d.Edit
{
    public abstract class RenderInfoBase : IRenderInfo
    {
        private readonly string name;
        private readonly RenderInfoKind kind;

        protected RenderInfoBase(string name, RenderInfoKind kind)
        {
            this.name = name;
            this.kind = kind;
        }

        protected RenderInfoBase(RenderInfoBase src)
        {
            this.name = src.name.CopyClone();
            this.kind = src.kind.CopyClone();
        }

        public string Name
        {
            get { return this.name; }
        }

        public RenderInfoKind Kind
        {
            get { return this.kind; }
        }

        public virtual int ItemCount
        {
            get { return 0; }
        }

        public virtual int ValueCount
        {
            get { return 0; }
        }

        public abstract object Clone();
    }

    public abstract class RenderInfoBase<T> : RenderInfoBase, IRenderInfo<T>
    {
        private readonly List<IRenderInfoItem<T>> items = new List<IRenderInfoItem<T>>();
        private readonly List<IRenderInfoValue<T>> values = new List<IRenderInfoValue<T>>();

        public class Item : IRenderInfoItem<T>
        {
            private T choice;
            private List<byte> alias = new List<byte>();

            public Item(T choice)
            {
                this.choice = choice.CopyClone();
            }

            public Item(T choice, byte[] alias)
            {
                this.choice = choice.CopyClone();
                this.alias.AddRange(alias);
            }

            protected Item(Item src)
            {
                this.choice = src.choice.CopyClone();
                this.alias = new List<byte>(src.alias.ToArray());
            }

            public T Choice
            {
                get { return this.choice; }
            }

            public byte[] Alias
            {
                get { return this.alias.ToArray(); }
            }

            public object Clone()
            {
                return new Item(this);
            }
        }

        public class Value : IRenderInfoValue<T>
        {
            private T defaultValue;
            private T editedValue;

            public Value(T defaultValue)
            {
                this.defaultValue = defaultValue.CopyClone();
                this.editedValue = defaultValue.CopyClone();
            }

            public Value(T defaultValue, T editedValue)
            {
                this.defaultValue = defaultValue.CopyClone();
                this.editedValue = editedValue.CopyClone();
            }

            protected Value(Value src)
            {
                this.defaultValue = src.defaultValue.CopyClone();
                this.editedValue = src.editedValue.CopyClone();
            }

            public T DefaultValue
            {
                get { return this.defaultValue; }
            }

            public T EditedValue
            {
                get { return this.editedValue; }
            }

            public object Clone()
            {
                return new Value(this);
            }
        }

        protected RenderInfoBase(string name, RenderInfoKind kind)
            : base(name, kind)
        {
        }

        protected RenderInfoBase(RenderInfoBase<T> src)
            : base(src)
        {
            this.items = new List<IRenderInfoItem<T>>();
            this.values = new List<IRenderInfoValue<T>>();

            foreach (var item in src.Items)
            {
                this.items.Add((IRenderInfoItem<T>)item.Clone());
            }

            foreach (var value in src.Values)
            {
                this.values.Add((IRenderInfoValue<T>)value.Clone());
            }
        }

        public IList<IRenderInfoItem<T>> Items
        {
            get { return this.items; }
        }

        public Item GetItem(int index)
        {
            return this.items[index] as Item;
        }

        public void AddItem(Item item)
        {
            this.items.Add(item);
        }

        public IList<IRenderInfoValue<T>> Values
        {
            get { return this.values; }
        }

        public Value GetValue(int index)
        {
            return this.values[index] as Value;
        }

        public void AddValue(Value value)
        {
            this.values.Add(value);
        }

        public void AddValue(T defaultValue, T editedValue)
        {
            Value value = new Value(defaultValue, editedValue);
            AddValue(value);
        }

        public override int ItemCount
        {
            get { return this.items.Count; }
        }

        public override int ValueCount
        {
            get { return this.values.Count; }
        }
    }

    public sealed class StringRenderInfo : RenderInfoBase<string>
    {
        public StringRenderInfo(string name)
            : base(name, RenderInfoKind.String)
        { }

        private StringRenderInfo(StringRenderInfo src)
            : base(src)
        { }

        public override object Clone()
        {
            return new StringRenderInfo(this);
        }
    }

    public sealed class IntRenderInfo : RenderInfoBase<int>
    {
        private readonly int min;
        private readonly int max;
        private readonly bool hasMinMax;

        public IntRenderInfo(string name)
            : base(name, RenderInfoKind.Int)
        {
            this.hasMinMax = false;
        }

        public IntRenderInfo(string name, int min, int max)
            : base(name, RenderInfoKind.Int)
        {
            this.min = min;
            this.max = max;
            this.hasMinMax = true;
        }

        public int Min
        {
            get { return this.min; }
        }

        public int Max
        {
            get { return this.max; }
        }

        public bool HasMinMax
        {
            get { return this.hasMinMax; }
        }

        private IntRenderInfo(IntRenderInfo src)
            : base(src)
        { }

        public override object Clone()
        {
            return new IntRenderInfo(this);
        }
    }

    public sealed class FloatRenderInfo : RenderInfoBase<float>
    {
        private readonly float min;
        private readonly float max;
        private readonly bool hasMinMax;

        public FloatRenderInfo(string name)
            : base(name, RenderInfoKind.Float)
        {
            this.hasMinMax = false;
        }

        public FloatRenderInfo(string name, float min, float max)
            : base(name, RenderInfoKind.Float)
        {
            this.min = min;
            this.max = max;
            this.hasMinMax = true;
        }

        public float Min
        {
            get { return this.min; }
        }

        public float Max
        {
            get { return this.max; }
        }

        public bool HasMinMax
        {
            get { return this.hasMinMax; }
        }

        private FloatRenderInfo(FloatRenderInfo src)
            : base(src)
        { }

        public override object Clone()
        {
            return new FloatRenderInfo(this);
        }
    }
}
