﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Linq;
using System.Text;
using NintendoWare.G3d.Edit.Math;

namespace NintendoWare.G3d.Edit
{
    /// <summary>
    /// ランタイムエラー発生時イベントの引数クラスです。
    /// </summary>
    public sealed class RuntimeErrorOccuredEventArgs : EventArgs
    {
        private RuntimeError runtimeError;

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="runtimeError">発生したランタイムエラーです。</param>
        internal RuntimeErrorOccuredEventArgs(RuntimeError runtimeError)
        {
            this.runtimeError = runtimeError;
        }

        /// <summary>
        /// ランタイムエラーを取得します。
        /// </summary>
        public RuntimeError RuntimeError
        {
            get
            {
                return this.runtimeError;
            }
        }
    }

    public sealed class ModelFileLoadedEventArgs : EventArgs
    {
        internal ModelFileLoadedEventArgs(
            uint toolKey, uint modelObjKey, uint resFileKey, uint resModelKey)
        {
            this.ToolKey = toolKey;
            this.ModelObjKey = modelObjKey;
            this.ResFileKey = resFileKey;
            this.ResModelKey = resModelKey;
        }

        public uint ToolKey
        {
            get;
            internal set;
        }

        public uint ModelObjKey
        {
            get;
            internal set;
        }

        public uint ResFileKey
        {
            get;
            internal set;
        }

        public uint ResModelKey
        {
            get;
            internal set;
        }
    }

    public sealed class FileLoadedEventArgs : EventArgs
    {
        internal FileLoadedEventArgs(
            uint toolKey, uint resFileKey)
        {
            this.ToolKey = toolKey;
            this.ResFileKey = resFileKey;
        }

        public uint ToolKey
        {
            get;
            internal set;
        }

        public uint ResFileKey
        {
            get;
            internal set;
        }
    }

    public sealed class FileReloadedEventArgs : EventArgs
    {
        internal FileReloadedEventArgs(
            uint newResFileKey, uint resFileKey)
        {
            this.NewResFileKey = newResFileKey;
            this.ResFileKey = resFileKey;
        }

        public uint NewResFileKey
        {
            get;
            internal set;
        }

        public uint ResFileKey
        {
            get;
            internal set;
        }
    }

    public sealed class ModelFileReloadedEventArgs : EventArgs
    {
        public uint ToolKey
        {
            get;
            internal set;
        }

        public uint ResFileKey
        {
            get;
            internal set;
        }

        public uint ModelObjKey
        {
            get;
            internal set;
        }

        public uint ResModelKey
        {
            get;
            internal set;
        }
    }

    public sealed class AttachModelReceivedArgs : EventArgs
    {
        internal AttachModelReceivedArgs(
            uint toolKey, uint modelObjKey, string attachName, string attachPath)
        {
            this.ToolKey = toolKey;
            this.ModelObjKey = modelObjKey;
            this.AttachName = attachName;
            this.AttachPath = attachPath;
        }

        public uint ToolKey
        {
            get;
            internal set;
        }

        public uint ModelObjKey
        {
            get;
            internal set;
        }

        /// <summary>
        /// アタッチされている名前を取得します。
        /// </summary>
        public string AttachName
        {
            get;
            internal set;
        }

        /// <summary>
        /// アタッチ対象のファイルパスを取得します。
        /// </summary>
        public string AttachPath
        {
            get;
            internal set;
        }
    }

    public sealed class DetachModelReceivedArgs : EventArgs
    {
        internal DetachModelReceivedArgs(uint toolKey, uint modelObjKey)
        {
            this.ToolKey = toolKey;
            this.ModelObjKey = modelObjKey;
        }

        public uint ToolKey
        {
            get;
            internal set;
        }

        public uint ModelObjKey
        {
            get;
            internal set;
        }
    }

    public sealed class AttachShaderArchiveReceivedArgs : EventArgs
    {
        internal AttachShaderArchiveReceivedArgs(
            uint toolKey, uint shaderArchiveKey, string attachName, string attachPath, bool isAttachShaderArchiveBinary)
        {
            this.ToolKey = toolKey;
            this.ShaderArchiveKey = shaderArchiveKey;
            this.AttachName = attachName;
            this.AttachPath = attachPath;
            this.IsAttachShaderArchiveBinary = isAttachShaderArchiveBinary;
        }

        public uint ToolKey
        {
            get;
            internal set;
        }

        public uint ShaderArchiveKey
        {
            get;
            internal set;
        }

        /// <summary>
        /// アタッチされている名前を取得します。
        /// </summary>
        public string AttachName
        {
            get;
            internal set;
        }

        /// <summary>
        /// アタッチ対象のファイルパスを取得します。
        /// </summary>
        public string AttachPath
        {
            get;
            internal set;
        }

        /// <summary>
        /// アタッチされる PC 版シェーダーアーカイブがバイナリ版かどうかを判定します。
        /// </summary>
        public bool IsAttachShaderArchiveBinary
        {
            get;
            internal set;
        }
    }

    public sealed class DetachShaderArchiveReceivedArgs : EventArgs
    {
        internal DetachShaderArchiveReceivedArgs(uint toolKey, uint shaderArchiveKey)
        {
            this.ToolKey = toolKey;
            this.ShaderArchiveKey = shaderArchiveKey;
        }

        public uint ToolKey
        {
            get;
            internal set;
        }

        public uint ShaderArchiveKey
        {
            get;
            internal set;
        }
    }

    public sealed class UpdatedShaderProgramReceivedArgs : EventArgs
    {
        private readonly List<IShaderCompileInfo> shaderCompileInfoArray = new List<IShaderCompileInfo>();

        public IList<IShaderCompileInfo> ShaderCompileInfoArray
        {
            get
            {
                return shaderCompileInfoArray;
            }
        }

        public uint ShaderArchiveKey
        {
            get;
            internal set;
        }

        public int ShadingModelIndex
        {
            get;
            internal set;
        }

        public int ShaderProgramIndex
        {
            get;
            internal set;
        }

        internal void AddShaderCompileInfos(IList<IShaderCompileInfo> infos)
        {
            this.shaderCompileInfoArray.AddRange(infos);
        }
    }

    public sealed class RenderInfoReceivedArgs : EventArgs
    {
        private readonly RenderInfoPackQueue renderInfoPackQueue = new RenderInfoPackQueue();
        private RenderInfoPack sentRenderInfoPack = null;

        /// <summary>
        /// 送信された描画情報パックを取得します。
        /// </summary>
        public RenderInfoPack SentRenderInfoPack
        {
            get { return this.sentRenderInfoPack; }
            internal set
            {
                this.sentRenderInfoPack = value;
            }
        }

        internal RenderInfoPackQueue RenderInfoPackQueue
        {
            get
            {
                return this.renderInfoPackQueue;
            }
        }
    }

    public sealed class ModelLayoutReceivedArgs : EventArgs
    {
        /// <summary>
        /// ModelObjのキーを取得します。
        /// </summary>
        public uint ModelObjKey
        {
            get;
            internal set;
        }

        public Vector3 Scale
        {
            get;
            internal set;
        }

        public Vector3 Rotate
        {
            get;
            internal set;
        }

        public Vector3 Translate
        {
            get;
            internal set;
        }
    }

    public sealed class AbnormalPacketReceivedArgs : EventArgs
    {
    }

    public sealed class IncorrectVersionReceivedArgs : EventArgs
    {
        public IncorrectVersionReceivedArgs(byte major, byte minor, byte micro, byte bugfix)
        {
            this.MajorVersion = major;
            this.MinorVersion = minor;
            this.MicroVersion = micro;
            this.BugFixVersion = bugfix;
        }

        public byte MajorVersion
        {
            get;
            internal set;
        }

        public byte MinorVersion
        {
            get;
            internal set;
        }

        public byte MicroVersion
        {
            get;
            internal set;
        }

        public byte BugFixVersion
        {
            get;
            internal set;
        }
    }

    public sealed class SelectionTargetReceivedArgs : EventArgs
    {
        private readonly List<ISelectionTargetInfo> selectionTargetInfoArray = new List<ISelectionTargetInfo>();

        public IList<ISelectionTargetInfo> SelectionTargetInfoArray
        {
            get { return this.selectionTargetInfoArray; }
        }
    }

    public sealed class CodePageUpdateReceivedArgs : EventArgs
    {
        public ushort CodePage
        {
            get;
            internal set;
        }
    }

    public sealed class BeginFreezeReceivedArgs : EventArgs
    {
    }

    public sealed class EndFreezeReceivedArgs : EventArgs
    {
    }

    public sealed class PlayFrameCtrlReceivedArgs : EventArgs
    {
        public float Frame
        {
            get;
            internal set;
        }
    }

    public sealed class StopFrameCtrlReceivedArgs : EventArgs
    {
        public float Frame
        {
            get;
            internal set;
        }
    }

    public sealed class SendFrameReceivedArgs : EventArgs
    {
        public float Frame
        {
            get;
            internal set;
        }
    }

    public sealed class SendFrameStepReceivedArgs : EventArgs
    {
        public float FrameStep
        {
            get;
            internal set;
        }
    }

    public sealed class SendModelNextAnimReceivedArgs : EventArgs
    {
        public uint ModelObjKey
        {
            get;
            internal set;
        }
    }

    public sealed class SendModelPrevAnimReceivedArgs : EventArgs
    {
        public uint ModelObjKey
        {
            get;
            internal set;
        }
    }

    public sealed class SceneAnimFileLoadedArgs : EventArgs
    {
    }

    public sealed class SceneAnimFileReloadedArgs : EventArgs
    {
    }

    /// <summary>
    /// メッセージの種類です。
    /// nn::g3d::viewer::detail::MessageType と値が一致します。
    /// </summary>
    public enum MessageType
    {
        /// <summary>
        /// ユーザーから送られた任意の情報メッセージです。
        /// </summary>
        UserInfo = 0,

        /// <summary>
        /// ユーザーから送られた任意の警告メッセージです。
        /// </summary>
        UserWarning = 1,

        /// <summary>
        /// ユーザーから送られた任意のエラーメッセージです。
        /// </summary>
        UserError = 2,

        /// <summary>
        /// ランタイムライブラリから送られた情報メッセージです。
        /// </summary>
        RuntimeInfo = 3,

        /// <summary>
        /// ランタイムライブラリから送られた警告メッセージです。
        /// </summary>
        RuntimeWarning = 4,

        /// <summary>
        /// ランタイムライブラリから送られたエラーメッセージです。
        /// </summary>
        RuntimeError = 5,
    };

    /// <summary>
    /// メッセージの出力先です。
    /// nn::g3d::viewer::ViewerServer::UserMessageArg::MessageDestination と値が一致します。
    /// </summary>
    public enum MessageDestination
    {
        /// <summary>
        /// ダイアログボックスに表示します。
        /// </summary>
        Dialog = 0,

        /// <summary>
        /// ログバーに表示します。
        /// </summary>
        Log = 1,
    };

    public sealed class ShowMessageRequestedArgs : EventArgs
    {
        public string Message
        {
            get;
            internal set;
        }

        public MessageType MessageType
        {
            get;
            internal set;
        }

        public MessageDestination MessageDestination
        {
            get;
            internal set;
        }
    }

    public interface IRecvInfo : INotifyPropertyChanged
    {
        /// <summary>
        /// バージョンが一致しないパケットを受信したときにコールされます。
        /// </summary>
        event EventHandler<IncorrectVersionReceivedArgs> IncorrectVersionReceived;

        /// <summary>
        /// 選択対象の返答があったときにコールされます。
        /// </summary>
        event EventHandler<SelectionTargetReceivedArgs> SelectionTargetReceived;

        /// <summary>
        /// コードページ更新の受信があったときにコールされます。
        /// </summary>
        event EventHandler<CodePageUpdateReceivedArgs> CodePageUpdateReceived;

        /// <summary>
        /// ランタイムでエラーが発生したときにコールされます。
        /// </summary>
        event EventHandler<RuntimeErrorOccuredEventArgs> RuntimeErrorOccured;

        /// <summary>
        /// ファイルロードが終了したときにコールされます。
        /// </summary>
        event EventHandler<FileLoadedEventArgs> FileLoaded;

        /// <summary>
        /// モデルファイルのリロードが終了したときにコールされます。
        /// </summary>
        event EventHandler<ModelFileReloadedEventArgs> ModelFileReloaded;

        /// <summary>
        /// ファイルリロードが終了したときにコールされます。
        /// </summary>
        event EventHandler<FileReloadedEventArgs> FileReloaded;

        /// <summary>
        /// モデルファイルロードが終了したときにコールされます。
        /// </summary>
        event EventHandler<ModelFileLoadedEventArgs> ModelFileLoaded;

        /// <summary>
        /// モデルアタッチの返答があったときにコールされます。
        /// </summary>
        event EventHandler<AttachModelReceivedArgs> AttachModelReceived;

        /// <summary>
        /// モデルデタッチの返答があったときにコールされます。
        /// </summary>
        event EventHandler<DetachModelReceivedArgs> DetachModelReceived;

        /// <summary>
        /// シェーダーアーカイブアタッチの返答があったときにコールされます。
        /// </summary>
        event EventHandler<AttachShaderArchiveReceivedArgs> AttachShaderArchiveReceived;

        /// <summary>
        /// シェーダーアーカイブデタッチの返答があったときにコールされます。
        /// </summary>
        event EventHandler<DetachShaderArchiveReceivedArgs> DetachShaderArchiveReceived;

        /// <summary>
        /// シェーダープログラム更新の返答があったときにコールされます。
        /// </summary>
        event EventHandler<UpdatedShaderProgramReceivedArgs> UpdatedShaderProgramReceived;

        /// <summary>
        /// 描画情報要求後の返答があったときにコールされます。
        /// </summary>
        event EventHandler<RenderInfoReceivedArgs> RenderInfoReceived;

        /// <summary>
        /// モデル配置情報要求後の返答があったときにコールされます。
        /// </summary>
        event EventHandler<ModelLayoutReceivedArgs> ModelLayoutReceived;

        /// <summary>
        /// 解析できないパケットを受信したときにコールされます。
        /// </summary>
        event EventHandler<AbnormalPacketReceivedArgs> AbnormalPacketReceived;

        /// <summary>
        /// 受信処理を区間設定開始ときにコールされます。
        /// </summary>
        event EventHandler<BeginFreezeReceivedArgs> BeginFreezeReceived;

        /// <summary>
        /// 受信処理を区間設定終了ときにコールされます。
        /// </summary>
        event EventHandler<EndFreezeReceivedArgs> EndFreezeReceived;

        /// <summary>
        /// フレームコントロール再生の受信があったときにコールされます。
        /// </summary>
        event EventHandler<PlayFrameCtrlReceivedArgs> PlayFrameCtrlReceived;

        /// <summary>
        /// フレームコントロール停止の受信があったときにコールされます。
        /// </summary>
        event EventHandler<StopFrameCtrlReceivedArgs> StopFrameCtrlReceived;

        /// <summary>
        /// フレーム設定の受信があったときにコールされます。
        /// </summary>
        event EventHandler<SendFrameReceivedArgs> SendFrameReceived;

        /// <summary>
        /// フレームステップ設定の受信があったときにコールされます。
        /// </summary>
        event EventHandler<SendFrameStepReceivedArgs> SendFrameStepReceived;

        /// <summary>
        /// ひとつ先のアニメーションセットを選択の受信があったときにコールされます。
        /// </summary>
        event EventHandler<SendModelNextAnimReceivedArgs> SendModelNextAnimReceived;

        /// <summary>
        /// ひとつ前のアニメーションセットを選択の受信があったときにコールされます。
        /// </summary>
        event EventHandler<SendModelPrevAnimReceivedArgs> SendModelPrevAnimReceived;

        /// <summary>
        /// シーンアニメーションファイルロードが終了したときにコールされます。
        /// </summary>
        event EventHandler<SceneAnimFileLoadedArgs> SceneAnimFileLoaded;

        /// <summary>
        /// シーンアニメーションファイルリロードが終了したときにコールされます。
        /// </summary>
        event EventHandler<SceneAnimFileReloadedArgs> SceneAnimFileReloaded;

        /// <summary>
        /// メッセージの表示が要求されたときにコールされます。
        /// </summary>
        event EventHandler<ShowMessageRequestedArgs> ShowMessageRequested;
    }
}
