﻿using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.IO;
using Nintendo.InGameEditing.Utilities;

namespace Nintendo.InGameEditing.UI
{
    /// <summary>
    /// GUI 向けコントロールの抽象モデルクラスです。
    /// </summary>
    /// <remarks>
    /// 各コントロールモデルクラスは次の条件を満たす必要があります。
    /// 1. ControlModel を継承する
    /// 2. static readonly string UiType フィールドを定義し、ランタイム側で指定される型名を指定する
    /// 3. 対応する Node インスタンスと属性辞書を引数に取るコンストラクタを持つ
    /// </remarks>
    public abstract class ControlModel : INotifyPropertyChanged
    {
        private static readonly PropertyChangedEventArgs IsEnabledEventArgs = new PropertyChangedEventArgs(nameof(IsEnabled));
        private static readonly PropertyChangedEventArgs HeaderEventArgs = new PropertyChangedEventArgs(nameof(Header));
        private static readonly PropertyChangedEventArgs ToolTipEventArgs = new PropertyChangedEventArgs(nameof(ToolTip));

        private string header = null;
        private string tooltip = null;
        private bool isEnabled = true;

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="node">対応するノードインスタンス</param>
        /// <param name="attrs">メタデータに指定された属性の辞書</param>
        protected ControlModel(Node node, Dictionary<string, string> attrs)
        {
            if (node == null) { throw new ArgumentNullException(nameof(node)); }
            if (attrs == null) { throw new ArgumentNullException(nameof(attrs)); }

            Node = node;
            UpdateAttributes(attrs);

            node.MetadataReceived += (s, e) => UpdateAttributes(ParseAttributes(e.Data));
            node.Removed += (s, e) => OnPropertyChanged(IsEnabledEventArgs);
        }

        /// <summary>
        /// プロパティ値の変更を通知します。
        /// </summary>
        public event PropertyChangedEventHandler PropertyChanged;

        /// <summary>
        /// ヘッダを取得します。
        /// </summary>
        public string Header
        {
            get { return header; }
            private set { SetPropertyValue(ref header, value, HeaderEventArgs); }
        }

        /// <summary>
        /// ツールチップを取得します。
        /// </summary>
        public string ToolTip
        {
            get { return tooltip; }
            private set { SetPropertyValue(ref tooltip, value, ToolTipEventArgs); }
        }

        internal Node Node { get; }

        /// <summary>
        /// サーバー側で削除された状態かどうかを取得します。
        /// </summary>
        internal bool IsRemoved => Node.IsRemoved;

        /// <summary>
        /// コントロールが有効かどうかを取得します。
        /// </summary>
        public bool IsEnabled
        {
            get { return isEnabled && !Node.IsRemoved; }
            private set
            {
                var oldValue = IsEnabled;
                isEnabled = value;
                var newValue = IsEnabled;

                if (oldValue != newValue)
                {
                    OnPropertyChanged(IsEnabledEventArgs);
                }
            }
        }

        /// <summary>
        /// プロパティ変更イベントを発行します。
        /// </summary>
        /// <param name="args">イベント引数</param>
        protected virtual void OnPropertyChanged(PropertyChangedEventArgs args)
        {
            if (args == null) { throw new ArgumentNullException(nameof(args)); }
            PropertyChanged?.Invoke(this, args);
        }

        /// <summary>
        /// フィールドと値を比較し、異なる場合に値のセットとプロパティ変更イベントの発行を行います。
        /// </summary>
        /// <typeparam name="T">値の型</typeparam>
        /// <param name="field">フィールドの参照</param>
        /// <param name="value">設定する値</param>
        /// <param name="args">イベント引数</param>
        /// <returns>値のセットが行われたかどうか</returns>
        protected bool SetPropertyValue<T>(ref T field, T value, PropertyChangedEventArgs args)
        {
            if (args == null) { throw new ArgumentNullException(nameof(args)); }
            if (EqualityComparer<T>.Default.Equals(field, value)) { return false; }
            field = value;
            OnPropertyChanged(args);
            return true;
        }

        // key:value 形式の null 終端文字列 * n を辞書に詰める
        internal static Dictionary<string, string> ParseAttributes(byte[] metadata)
        {
            var result = new Dictionary<string, string>();
            var delimiter = new[] { ':' };

            using (var stream = new MemoryStream(metadata, 0, metadata.Length, false, true))
            {
                var line = stream.ReadNullTerminatedUTF8String();

                while (!string.IsNullOrEmpty(line))
                {
                    var pair = line.Split(delimiter, 2);

                    if (pair.Length == 2 && !string.IsNullOrEmpty(pair[0]))
                    {
                        if (!result.ContainsKey(pair[0]))
                        {
                            result.Add(pair[0], pair[1]);
                        }
                    }

                    line = stream.ReadNullTerminatedUTF8String();
                }

                return result;
            }
        }

        /// <summary>
        /// コントロール属性の更新を行います。メタデータの受信時に呼び出されます。
        /// </summary>
        /// <param name="attrs">受信した属性の辞書</param>
        protected virtual void UpdateAttributes(Dictionary<string, string> attrs)
        {
            if (attrs == null) { throw new ArgumentNullException(nameof(attrs)); }

            var header = default(string);
            if (attrs.TryGetValue(nameof(header), out header)) { Header = header; }

            var tooltip = default(string);
            if (attrs.TryGetValue(nameof(tooltip), out tooltip)) { ToolTip = tooltip; }

            var enabled = default(string);
            if (attrs.TryGetValue(nameof(enabled), out enabled))
            {
                bool value;
                IsEnabled = !bool.TryParse(enabled, out value) || value;
            }
            else
            {
                IsEnabled = true;
            }
        }
    }
}
