﻿using System;
using System.Linq;

namespace Nintendo.InGameEditing
{
    internal class ByteTypeSerializer : ITypeSerializer<byte>, ITypeSerializer<byte[]>
    {
        byte[] ITypeSerializer<byte>.Serialize(byte value) => new[] { value };

        byte ITypeSerializer<byte>.Deserialize(byte[] data)
        {
            if (data == null) { throw new ArgumentNullException(nameof(data)); }
            if (data.Length == 0) { throw new ArgumentException(nameof(data)); }
            return data[0];
        }

        byte[] ITypeSerializer<byte[]>.Serialize(byte[] values) => values ?? Const.EmptyBytes;

        byte[] ITypeSerializer<byte[]>.Deserialize(byte[] data) => data ?? Const.EmptyBytes;

        internal static void RegisterTo(DefaultTypeSerializerProvider provider)
        {
            var serializer = new ByteTypeSerializer();
            provider?.Register(new[] { "byte", "uint8_t" }, typeof(byte), serializer);
            provider?.Register(new[] { "byte[]", "uint8_t[]" }, typeof(byte[]), serializer);
        }
    }

    internal class SByteTypeSerializer : ITypeSerializer<sbyte>, ITypeSerializer<sbyte[]>
    {
        private static readonly sbyte[] EmptyItems = new sbyte[0];

        byte[] ITypeSerializer<sbyte>.Serialize(sbyte value) => new[] { (byte)value };

        sbyte ITypeSerializer<sbyte>.Deserialize(byte[] data)
        {
            if (data == null) { throw new ArgumentNullException(nameof(data)); }
            if (data.Length == 0) { throw new ArgumentException(nameof(data)); }
            return (sbyte)data[0];
        }

        byte[] ITypeSerializer<sbyte[]>.Serialize(sbyte[] values) => values?.Select(v => (byte)v).ToArray() ?? Const.EmptyBytes;

        sbyte[] ITypeSerializer<sbyte[]>.Deserialize(byte[] data) => data?.Select(v => (sbyte)v).ToArray() ?? EmptyItems;

        internal static void RegisterTo(DefaultTypeSerializerProvider provider)
        {
            var serializer = new SByteTypeSerializer();
            provider?.Register(new[] { "sbyte", "int8_t" }, typeof(sbyte), serializer);
            provider?.Register(new[] { "sbyte[]", "int8_t[]" }, typeof(sbyte[]), serializer);
        }
    }
}
