﻿using System;
using System.Collections.Generic;
using System.Linq;

namespace Nintendo.InGameEditing
{
    /// <summary>
    /// ITypeSerializerProvider のデフォルト実装クラスです。
    /// </summary>
    internal class DefaultTypeSerializerProvider : ITypeSerializerProvider
    {
        private readonly Dictionary<string, Type> typeDictionary = new Dictionary<string, Type>();
        private readonly Dictionary<string, object> serializerDictionary = new Dictionary<string, object>();

        internal static ITypeSerializerProvider Instance { get; }

        /// <summary>
        /// 基本型のシリアライズを登録します。
        /// </summary>
        static DefaultTypeSerializerProvider()
        {
            var provider = new DefaultTypeSerializerProvider();

            provider.RegisterPrimitive<short>("short", "int16_t");
            provider.RegisterPrimitive<ushort>("ushort", "uint16_t");
            provider.RegisterPrimitive<int>("int", "int32_t");
            provider.RegisterPrimitive<uint>("uint", "uint32_t");
            provider.RegisterPrimitive<long>("long", "int64_t");
            provider.RegisterPrimitive<ulong>("ulong", "uint64_t");
            provider.RegisterPrimitive<float>("float", "Single");
            provider.RegisterPrimitive<double>("double", "Double");

            BoolTypeSerializer.RegisterTo(provider);
            ByteTypeSerializer.RegisterTo(provider);
            SByteTypeSerializer.RegisterTo(provider);
            StringTypeSerializer.RegisterTo(provider);

            Instance = provider;
        }

        private DefaultTypeSerializerProvider() { }

        public bool TryResolveTypeName(string typeName, out Type supportedType)
            => typeDictionary.TryGetValue(typeName, out supportedType);

        public bool TryGetTypeSerializer<T>(string typeName, out ITypeSerializer<T> serializer)
        {
            object instance;
            serializerDictionary.TryGetValue(typeName, out instance);
            serializer = instance as ITypeSerializer<T>;
            return serializer != null;
        }

        public bool TryGetArrayTypeSerializer<T>(string typeName, out ITypeSerializer<T[]> serializer)
        {
            object instance;
            serializerDictionary.TryGetValue(typeName, out instance);
            serializer = instance as ITypeSerializer<T[]>;
            return serializer != null;
        }

        internal void Register(IEnumerable<string> typeNames, Type type, object serializer)
        {
            if (typeNames == null) { throw new ArgumentNullException(nameof(typeNames)); }
            if (type == null) { throw new ArgumentNullException(nameof(type)); }
            if (serializer == null) { throw new ArgumentNullException(nameof(serializer)); }

            foreach (var typeName in typeNames)
            {
                typeDictionary.Add(typeName, type);
                serializerDictionary.Add(typeName, serializer);
            }
        }

        private void RegisterPrimitive<T>(params string[] typeNames) where T : struct
        {
            // 配列も一緒に登録する
            var serializer = new PrimitiveTypeSerializer<T>();
            Register(typeNames, typeof(T), serializer);
            Register(typeNames.Select(typename => $"{typename}[]"), typeof(T[]), serializer);
        }
    }
}
