﻿using System;
using System.Collections.Concurrent;
using System.Collections.Generic;
using System.Linq;
using System.Net;
using System.Reactive.Linq;
using System.Threading.Tasks;
using System.Reactive.Disposables;
using System.Threading;
using System.Reactive.Subjects;
using Nintendo.InGameEditing.Communication;
using Nintendo.InGameEditing.Messages;
using Nintendo.InGameEditing.Utilities;
using System.Reflection;
using System.Reactive.Concurrency;
using System.Diagnostics;
using System.IO;

namespace Nintendo.InGameEditing
{
    /// <summary>
    /// C++ ランタイムサーバーに接続し、ターゲット上の値を編集できるようにする機能を提供します。
    /// C++ ランタイム上では &lt;nn/ige/ige_EditService.h&gt; を使用してください。
    /// </summary>
    public sealed class EditService : IDisposable
    {
        private const int RootId = 0;

        private readonly AsyncLockObject lockObj = new AsyncLockObject();
        private readonly bool isPeerTypeSpecified;
        private readonly ConcurrentDictionary<uint, Node> nodeDictionary = new ConcurrentDictionary<uint, Node>();
        private readonly HtcsConnectionManager htcsManager = HtcsConnectionManager.Instance;
        private readonly List<ITypeSerializerProvider> providers = new List<ITypeSerializerProvider> { DefaultTypeSerializerProvider.Instance };

        private static class InternalState
        {
            internal const int Disconnected = 0;
            internal const int Connecting = 1;
            internal const int Connected = 2;
            internal const int Disconnecting = 3;
            internal const int Disposed = 4;
        }

        private int state = InternalState.Disconnected;
        private IDisposable subscriber;
        private Channel channel;
        private bool isAutoConnectionEnabled;

        /// <summary>
        /// ポート名を指定してインスタンスを初期化します。
        /// </summary>
        /// <param name="portName">ポート名</param>
        public EditService(string portName) : this(portName, (string)null)
        {
        }

        /// <summary>
        /// ポート名とピアタイプを指定してインスタンスを初期化します。
        /// </summary>
        /// <param name="portName">ポート名</param>
        /// <param name="peerType">ピアタイプ</param>
        public EditService(string portName, string peerType) : this(portName, peerType, null)
        {
        }

        /// <summary>
        /// ポート名と <see cref="ITypeSerializerProvider"/> を指定してインスタンスを初期化します。
        /// </summary>
        /// <param name="portName">ポート名</param>
        /// <param name="providers">型の処理を拡張するプロバイダ。型名の解決はプロバイダの指定順に行われます。</param>
        public EditService(string portName, params ITypeSerializerProvider[] providers) : this(portName, null, providers)
        {
        }

        /// <summary>
        /// ポート名、ピアタイプと <see cref="ITypeSerializerProvider"/> を指定してインスタンスを初期化します。
        /// </summary>
        /// <param name="portName">ポート名</param>
        /// <param name="peerType">ピアタイプ</param>
        /// <param name="providers">型の処理を拡張するプロバイダ。型名の解決はプロバイダの指定順に行われます。</param>
        public EditService(string portName, string peerType, params ITypeSerializerProvider[] providers)
        {
            if (string.IsNullOrEmpty(portName)) { throw new ArgumentException(nameof(portName)); }

            PortName = portName;
            PeerType = peerType;
            isPeerTypeSpecified = !string.IsNullOrEmpty(peerType);

            if (providers?.Any() == true)
            {
                // Default を最後にもってくるため
                this.providers.InsertRange(0, providers);
            }

            ((IObservable<Message>)Sender)
                .Where(_ => IsConnected && channel?.IsDisposed == false)
                .Select(message => message.ToPacket())
                .Subscribe(packet => { try { channel?.Write(packet); } catch { /* ignored */ } });

            htcsManager.ConnectionChanged += Client_ConnectionChanged;

            if (!htcsManager.IsStarted)
            {
                htcsManager.Start();
            }

            ErrorReceiver.Subscribe(e =>
            {
                Debug.WriteLine($"{e.Message}");
                IsAutoConnectionEnabled = false;
                Disconnect();
            });
        }

        /// <summary>
        /// 接続状態が変化したときに発行されるイベントです。
        /// </summary>
        public event EventHandler<ConnectionChangedEventArgs> ConnectionChanged;

        /// <summary>
        /// ポート名を取得します。
        /// </summary>
        public string PortName { get; }

        /// <summary>
        /// ピアタイプを取得します。指定されていない場合、null を返します。
        /// </summary>
        public string PeerType { get; }

        /// <summary>
        /// サーバーと接続済みかどうかを取得します。
        /// </summary>
        public bool IsConnected => state == InternalState.Connected;

        /// <summary>
        /// サーバーに接続可能かどうかを取得します。
        /// </summary>
        public bool Connectable { get; private set; }

        /// <summary>
        /// サーバーに接続可能になったタイミングで自動的に接続を行うかどうかを取得します。
        /// </summary>
        public bool IsAutoConnectionEnabled
        {
            get { return isAutoConnectionEnabled; }
            set
            {
                if (isAutoConnectionEnabled == value) { return; }
                isAutoConnectionEnabled = value;

                if (isAutoConnectionEnabled && !IsConnected && Connectable)
                {
                    ConnectAsyncFireAndForget();
                }
            }
        }

        private void Client_ConnectionChanged(object sender, ConnectionInfoUpdatedEventArgs e)
        {
            IPEndPoint _;
            if (Connectable != TryGetRemoteEndPoint(out _))
            {
                Connectable = !Connectable;

                if (!Connectable && IsConnected)
                {
                    DisconnectInternal();
                }

                OnConnectionChanged();
            }

            if (IsAutoConnectionEnabled && !IsConnected && Connectable)
            {
                try { Connect(); }
                catch { /* ignored */ }
            }
        }

        /// <summary>
        /// 子ノードのリストを取得します。
        /// </summary>
        public NodeList Children { get; } = new NodeList();

        // 集約エラーハンドラ
        internal ISubject<Exception> ErrorReceiver { get; } = new Subject<Exception>();

        /// <summary>
        /// サーバーに接続します。
        /// </summary>
        public void Connect()
        {
            var current = Interlocked.CompareExchange(ref state, InternalState.Connecting, InternalState.Disconnected);
            if (current != InternalState.Disconnected) { return; }

            using (lockObj.Lock())
            {
                if (state != InternalState.Connecting) { return; }

                CleanUp();

                IPEndPoint ep;
                if (!TryGetRemoteEndPoint(out ep))
                {
                    return; // とりあえず例外にしない
                }

                Channel newChannel = null;
                IDisposable newDisposable;

                try
                {
                    newChannel = Channel.Open(ep, Const.Signature, NewThreadScheduler.Default);
                    newDisposable = SubscribeChannel(newChannel);
                }
                catch
                {
                    newChannel?.Dispose();
                    state = InternalState.Disconnected;
                    throw;
                }

                subscriber = newDisposable;
                channel = newChannel;
                state = InternalState.Connected;
            }

            // ここへ来る == 未接続 -> 接続
            OnConnectionChanged();
        }

        /// <summary>
        /// サーバーに非同期で接続します。
        /// </summary>
        public async Task ConnectAsync()
        {
            var current = Interlocked.CompareExchange(ref state, InternalState.Connecting, InternalState.Disconnected);
            if (current != InternalState.Disconnected) { return; }

            using (await lockObj.LockAsync())
            {
                if (state != InternalState.Connecting) { return; }

                CleanUp();

                IPEndPoint ep;
                if (!TryGetRemoteEndPoint(out ep))
                {
                    return; // とりあえず例外にしない
                }

                Channel newChannel = null;
                IDisposable newDisposable;

                try
                {
                    newChannel = await Channel.OpenAsync(ep, Const.Signature, NewThreadScheduler.Default);
                    newDisposable = SubscribeChannel(newChannel);
                }
                catch
                {
                    newChannel?.Dispose();
                    state = InternalState.Disconnected;
                    throw;
                }

                subscriber = newDisposable;
                channel = newChannel;
                state = InternalState.Connected;
            }

            // ここへ来る == 未接続 -> 接続
            OnConnectionChanged();
        }

        private async void ConnectAsyncFireAndForget()
        {
            try { await ConnectAsync(); }
            catch { /* ignored */ }
        }

        private IDisposable SubscribeChannel(Channel newChannel)
        {
            if (newChannel?.IsDisposed != false) { throw new ArgumentException(nameof(newChannel)); }

            var newDisposable = new CompositeDisposable();
            newChannel.AddTo(newDisposable);

            // 先に Subject を構築し、購読をまとめておく
            var messageReceiver = new Subject<Message>();

            messageReceiver.AddTo(newDisposable);

            // newChannel 内でエラーが流れた場合のハンドラ
            messageReceiver
                .Subscribe(
                    _ => { },
                    e => ErrorReceiver.OnNext(e));

            messageReceiver
                .OfType<VersionMessage>()
                .Subscribe(message =>
                {
                    if (Const.CurrentVersion.IsCompatible(message.Version)) { return; }

                    // OnError で受けると例外が ↑ と重複する可能性がある
                    try
                    {
                        throw new InvalidDataException($"[EditService] version check failed. expected '{Const.CurrentVersion}', received '{message.Version}'");
                    }
                    catch (Exception e)
                    {
                        ErrorReceiver.OnNext(e);
                    }
                });

            messageReceiver
                .OfType<NodeMessage>()
                .Where(message => !nodeDictionary.ContainsKey(message.Id))
                .Subscribe(CreateNode);

            messageReceiver
                .OfType<RemoveNodeMessage>()
                .Subscribe(message =>
                {
                    Node node;
                    if (!nodeDictionary.TryRemove(message.Id, out node)) { return; }
                    node.Disposer.Dispose();
                });

            messageReceiver
                .OfType<NodeTargetMessage>()
                .Where(message => message.Id != 0) // Id == 0 は自分用に予約
                .Subscribe(message =>
                {
                    Node target;
                    if (!nodeDictionary.TryGetValue(message.Id, out target)) { return; }
                    target?.Receiver.OnNext(message);
                });

            newChannel
                .Select(MessageHelper.Parse)
                .Subscribe(messageReceiver)
                .AddTo(newDisposable);

            return newDisposable;
        }

        /// <summary>
        /// サーバーから切断します。
        /// </summary>
        public void Disconnect()
        {
            if (DisconnectInternal())
            {
                OnConnectionChanged();
            }
        }

        private bool DisconnectInternal()
        {
            if (state == InternalState.Disconnected ||
                state == InternalState.Disconnecting ||
                state == InternalState.Disposed)
            {
                return false;
            }

            using (lockObj.Lock())
            {
                if (state == InternalState.Disconnected ||
                    state == InternalState.Disconnecting ||
                    state == InternalState.Disposed)
                {
                    return false;
                }

                state = InternalState.Disconnecting;
                CleanUp();
                state = InternalState.Disconnected;
            }

            return true;
        }

        /// <summary>
        /// サーバーにルートノードの情報を要求し、ルートノードを更新します。
        /// </summary>
        internal void RequestChildren()
        {
            Sender.OnNext(new ChildNodeRequestMessage(RootId));
        }

        private void CleanUp()
        {
            channel = null;

            subscriber?.Dispose();
            subscriber = null;
            nodeDictionary.Clear();
            Children.RemoveItems();
        }

        internal IObserver<Message> Sender { get; } = new Subject<Message>();

        private void CreateNode(NodeMessage message)
        {
            NodeList targetList;
            if (message.ParentId == RootId)
            {
                targetList = Children;
            }
            else
            {
                Node parentNode;
                if (!nodeDictionary.TryGetValue(message.ParentId, out parentNode)) { return; }
                targetList = parentNode.Children;
            }

            Node newNode;
            switch (message.NodeType)
            {
                case NodeType.Normal:
                    newNode = new Node(this, message);
                    break;
                case NodeType.Command:
                    newNode = new CommandNode(this, message);
                    break;
                case NodeType.Value:
                    {
                        Type type;
                        if (!TryResolveTypeName(message.TypeName, out type)) { return; }

                        var genericType = typeof(ValueNode<>).MakeGenericType(type);

                        newNode = (Node)Activator.CreateInstance(
                            genericType,
                            BindingFlags.Instance | BindingFlags.NonPublic,
                            null,
                            new object[] { this, message },
                            null);
                    }
                    break;
                default:
                    return;
            }

            if (!nodeDictionary.TryAdd(message.Id, newNode)) { return; }

            try { targetList.Add(newNode); }
            catch { /* ignored */ }
        }

        private bool TryGetRemoteEndPoint(out IPEndPoint endPoint)
        {
            endPoint = null;

            if (!htcsManager.IsConnected) { return false; }

            var targetInfos = htcsManager.TargetInfos; // copy reference

            if (!targetInfos.Any()) { return false; }

            IEnumerable<PortInfo> portInfos;

            if (isPeerTypeSpecified)
            {
                portInfos = targetInfos.Where(info => info.PeerType == PeerType).SelectMany(info => info.PortInfos);
            }
            else
            {
                portInfos = targetInfos.SelectMany(info => info.PortInfos);
            }

            endPoint = portInfos.FirstOrDefault(info => info.PortName == PortName)?.EndPoint;

            return endPoint != null;
        }

        private bool TryResolveTypeName(string typeName, out Type targetType)
        {
            foreach (var provider in providers)
            {
                if (provider.TryResolveTypeName(typeName, out targetType) && targetType != null) { return true; }
            }

            targetType = null;
            return false;
        }

        internal bool TryGetTypeSerializer<T>(string typeName, out ITypeSerializer<T> genericSerializer)
        {
            foreach (var provider in providers)
            {
                if (provider.TryGetTypeSerializer(typeName, out genericSerializer) && genericSerializer != null) { return true; }
            }

            genericSerializer = null;
            return false;
        }

        /// <summary>
        /// インスタンスを破棄します。全ての操作は無効化され、再接続できない状態になります。
        /// </summary>
        public void Dispose()
        {
            ConnectionChanged = null;
            htcsManager.ConnectionChanged -= Client_ConnectionChanged;
            isAutoConnectionEnabled = false;

            using (lockObj.Lock())
            {
                CleanUp();
                state = InternalState.Disposed;
            }
        }

        private void OnConnectionChanged()
            => ConnectionChanged?.Invoke(this, new ConnectionChangedEventArgs(IsConnected, Connectable));
    }
}
