﻿using System;
using System.IO;
using System.Net;
using System.Net.Sockets;
using System.Reactive.Concurrency;
using System.Reactive.Disposables;
using System.Reactive.Linq;
using System.Reactive.Subjects;
using System.Threading.Tasks;
using Nintendo.InGameEditing.Utilities;

namespace Nintendo.InGameEditing.Communication
{
    internal sealed class Channel : IObservable<Packet>, IDisposable
    {
        private readonly AsyncLockObject lockObj = new AsyncLockObject();
        private readonly CompositeDisposable disposable = new CompositeDisposable();
        private readonly TcpClient client = new TcpClient() { ReceiveBufferSize = 65536 };
        private IConnectableObservable<Packet> observable;
        private Stream stream;

        private bool isFirstSubscription = true;

        public static async Task<Channel> OpenAsync(IPEndPoint remoteEp, int? signature = null, IScheduler receiveScheduler = null)
        {
            if (remoteEp == null) { throw new ArgumentNullException(nameof(remoteEp)); }
            var scheduler = receiveScheduler ?? Scheduler.Default;

            var channel = new Channel(remoteEp, signature, scheduler);
            await channel.client.ConnectAsync(remoteEp.Address, remoteEp.Port).ConfigureAwait(false);

            channel.stream = new CachedStream(channel.client.GetStream());
            channel.observable = channel.client.GetStream().ToPacketObservable(signature, scheduler).Where(p => p.Payload.Length > 0).Publish();

            return channel;
        }

        public static Channel Open(IPEndPoint remoteEp, int? signature = null, IScheduler receiveScheduler = null)
        {
            if (remoteEp == null) { throw new ArgumentNullException(nameof(remoteEp)); }
            var scheduler = receiveScheduler ?? Scheduler.Default;

            var channel = new Channel(remoteEp, signature, scheduler);
            channel.client.Connect(remoteEp);

            channel.stream = new CachedStream(channel.client.GetStream());
            channel.observable = channel.client.GetStream().ToPacketObservable(signature, scheduler).Where(p => p.Payload.Length > 0).Publish();

            return channel;
        }

        private Channel(IPEndPoint remoteEp, int? signature, IScheduler scheduler)
        {
            EndPoint = remoteEp;
            Signature = signature;
            ReceiveScheduler = scheduler;

            disposable.Add(client);
            disposable.Add(Disposable.Create(() => { stream?.Dispose(); stream = null; }));
            disposable.Add(Disposable.Create(() => observable = null));
        }

        public bool IsConnected => client.Connected;

        public IPEndPoint EndPoint { get; }

        public int? Signature { get; }

        public IScheduler ReceiveScheduler { get; }

        public bool IsDisposed => disposable.IsDisposed;

        public async Task WriteAsync(Packet packet)
        {
            if (packet == null) { throw new ArgumentNullException(nameof(packet)); }
            if (disposable.IsDisposed) { throw new ObjectDisposedException(typeof(Channel).FullName); }

            using (await lockObj.LockAsync())
            {
                if (disposable.IsDisposed) { return; }

                await stream.WriteAndFlushAsync(packet).ConfigureAwait(false);
            }
        }

        public void Write(Packet packet)
        {
            if (packet == null) { throw new ArgumentNullException(nameof(packet)); }
            if (disposable.IsDisposed) { throw new ObjectDisposedException(typeof(Channel).FullName); }

            using (lockObj.Lock())
            {
                if (disposable.IsDisposed) { return; }

                stream.WriteAndFlush(packet);
            }
        }

        public IDisposable Subscribe(IObserver<Packet> observer)
        {
            if (observer == null) { throw new ArgumentNullException(nameof(observer)); }
            if (disposable.IsDisposed) { throw new ObjectDisposedException(typeof(Channel).FullName); }

            using (lockObj.Lock())
            {
                if (disposable.IsDisposed) { return Disposable.Empty; }

                var subscribed = observable.Subscribe(observer);

                if (isFirstSubscription)
                {
                    disposable.Add(observable.Connect());
                    isFirstSubscription = false;
                }

                return subscribed;
            }
        }

        public int SendTimeout
        {
            get { return client.SendTimeout; }
            set { client.SendTimeout = value; }
        }

        public int SendBufferSize
        {
            get { return client.SendBufferSize; }
            set { client.SendBufferSize = value; }
        }

        public int ReceiveTimeout
        {
            get { return client.ReceiveTimeout; }
            set { client.ReceiveTimeout = value; }
        }

        public int ReceiveBufferSize
        {
            get { return client.ReceiveBufferSize; }
            set { client.ReceiveBufferSize = value; }
        }

        public void Dispose()
        {
            if (disposable.IsDisposed) { return; }

            using (lockObj.Lock())
            {
                if (disposable.IsDisposed) { return; }

                disposable.Dispose();
            }
        }
    }
}
