﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Net;
using System.Net.Sockets;
using System.Threading;
using System.Threading.Tasks;
using Nintendo.InGameEditing.Communication.Htcs;

namespace Nintendo.InGameEditing.Communication
{
    /// <summary>
    /// ターゲットコネクタの共通実装クラスです。
    /// </summary>
    public abstract partial class TargetConnector : IDisposable
    {
        /// <summary>
        /// リソース破棄フラグです。
        /// </summary>
        private bool _disposed = false;

        /// <summary>
        /// 通信スレッドです。
        /// </summary>
        private Task _receiveTask;

        /// <summary>
        /// TcpClient のロックオブジェクトです。
        /// </summary>
        private readonly object clientLock = new object();

        /// <summary>
        /// コネクタが停止されるとfalseになり、内部スレッドを終了します。
        /// </summary>
        protected bool IsValid = true;

        /// <summary>
        /// デーモンやホストブリッジからの受信処理を実装します。
        /// </summary>
        protected abstract bool ReceiveInfo();

        /// <summary>
        /// アドオンコネクタで利用するTCPクライアントを取得します。
        /// </summary>
        protected TcpClient Client { get; private set; }

        /// <summary>
        /// ターゲットリストを取得します。
        /// </summary>
        internal TargetList TargetList { get; set; } = new TargetList();

        /// <summary>
        /// ポートマッピング情報を取得します。
        /// </summary>
        internal PortMap PortMap { get; set; } = new PortMap();

        /// <summary>
        /// 接続情報が更新された時に発生するイベントです。
        /// </summary>
        public event EventHandler<ConnectionInfoUpdatedEventArgs> ConnectionInfoUpdated;

        /// <summary>
        /// デーモンやホストブリッジに接続できているかを取得します。
        /// </summary>
        public bool IsConnected { get; private set; }

        /// <summary>
        /// 接続情報の受信待ちを開始します。
        /// </summary>
        /// <returns>開始できたらtrue,開始できなかったらfalse.</returns>
        public virtual bool Start()
        {
            if (this._receiveTask?.Status == TaskStatus.Running || this._disposed || !this.IsValid)
            {
                return false;
            }

            this._receiveTask = Task.Factory.StartNew(ThreadToolReceiver, TaskCreationOptions.LongRunning);
            return true;
        }

        /// <summary>
        /// 接続情報の受信待ちを停止します。
        /// </summary>
        public virtual void Stop()
        {
            lock (this)
            {
                // Clientはここで破棄しないとRecieveを抜けない
                if (this.Client != null)
                {
                    this.Client.Close();
                    this.Client = null;
                }

                // 内部スレッドの終了を指示し、
                // Client以外のDisposeは内部スレッドで行う
                this.IsValid = false;
            }
        }

        /// <summary>
        /// リソースを破棄します。
        /// </summary>
        public virtual void Dispose()
        {
            this.Dispose(true);
            GC.SuppressFinalize(this);
        }

        /// <summary>
        /// リソースを破棄します。マネージド・アンマネージドによって処理を分岐します。
        /// </summary>
        protected virtual void Dispose(bool disposing)
        {
            if (this._disposed)
            {
                return;
            }

            if (disposing)
            {
                // ここでマネージドリソースをDisposeする
                if (this.Client != null)
                {
                    this.Client.Close();
                    this.Client = null;
                }

                if (this.IsConnected)
                {
                    this.IsConnected = false;
                    this.ClearConnections();
                    this.RaiseConnectionInfoUpdated();
                }
            }

            // ここでアンマネージドリソースを解放する
            this._disposed = true;
        }

        protected abstract int Port { get; }

        /// <summary>
        /// 指定されたポート番号に接続を試行し、確立したら受信待ちに移行します。
        /// </summary>
        protected virtual void ThreadToolReceiver()
        {
            while (this.IsValid)
            {
                this.IsConnected = false;
                while (!this.IsConnected)
                {
                    try
                    {
                        lock (this.clientLock)
                        {
                            this.Client = new TcpClient("127.0.0.1", Port);
                            Logger.WriteLine("Connected by {0} with {1}.", this.GetType().Name, this.Client.Client.LocalEndPoint);
                            this.IsConnected = true;
                        }
                    }
                    catch
                    {
                        // 接続できるまで待つ
                        Thread.Sleep(100);

                        if (!this.IsValid)
                        {
                            // Disposeを内部スレッド終了時の責務とする
                            this.Dispose();
                            return;
                        }
                    }
                }

                while (this.IsConnected && this.IsValid)
                {
                    if (!this.ReceiveInfo())
                    {
                        break;
                    }
                }

                lock(this.clientLock)
                {
                    if (this.Client != null)
                    {
                        this.Client.Close();
                        this.Client = null;
                    }
                }

                // 切断後サーバ登録を全て解除する
                this.IsConnected = false;
                this.ClearConnections();
                this.RaiseConnectionInfoUpdated();
            }

            // Disposeを内部スレッド終了時の責務とする
            this.Dispose();
        }

        /// <summary>
        /// 接続情報をクリアします。
        /// </summary>
        protected virtual void ClearConnections()
        {
            this.TargetList = new TargetList();
            this.PortMap = new PortMap();
        }

        /// <summary>
        /// 外部プラグインからターゲット情報を追加します。
        /// </summary>
        protected virtual void AddTargetInfo(string peerType, string htcsPeerName)
        {
            this.TargetList.Add(new Target(peerType, htcsPeerName));
        }

        /// <summary>
        /// 外部プラグインからポート情報を追加します。
        /// </summary>
        protected virtual void AddPortInfo(HtcsPortDescriptor desc, IPEndPoint endPoint)
        {
            this.PortMap.Add(new PortMapItem(desc, endPoint));
        }

        /// <summary>
        /// 接続情報更新イベントを発生させます。
        /// </summary>
        protected virtual void RaiseConnectionInfoUpdated()
        {
            this.ConnectionInfoUpdated?.Invoke(this, new ConnectionInfoUpdatedEventArgs(this.TargetList, this.PortMap));
        }
    }
}
