﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "stdafx.h"
#include "atkTool_BfstmReader.h"
#include "atkTool_WaveDecoder.h"
#include <nn/nn_Assert.h>
#include <nn/atk.h>
#include <nn/atk/atk_StreamSoundFile.h>
#include <nn/atkTool/atkTool_WaveBinaryInfo.h>
#include <nn/codec/codec_AdpcmDecoder.h>
#include <nn/util/util_BytePtr.h>
#include <nn/util/util_StringUtil.h>

#ifndef NDEBUG
#define ENABLE_BUFFER_OVERFLOW_CHECK
#endif

#define REQUIRES(exp, returnValue) if(!(exp)) { return returnValue; }

namespace nn { namespace atkTool { namespace detail {

BfstmReader::BfstmReader(const void* pBfstmData)
    : m_Reader()
    , m_pBfstmData(pBfstmData)
    , m_IsAvailable(false)
{
    m_Reader.Initialize(pBfstmData);

    if (m_Reader.IsAvailable())
    {
        m_IsAvailable = m_Reader.ReadStreamSoundInfo(&m_StreamSoundInfo);
    }
}

void BfstmReader::GetWaveBinaryInfo(nnatktoolWaveBinaryInfo& info) const
{
    switch(m_StreamSoundInfo.encodeMethod)
    {
    case nn::atk::detail::WaveFile::EncodeMethod_Pcm16:
        nn::util::Strlcpy(info.format.encoding, nnatktoolWaveEncodingPcm16, sizeof(info.format.encoding));
        break;

    case nn::atk::detail::WaveFile::EncodeMethod_DspAdpcm:
        nn::util::Strlcpy(info.format.encoding, nnatktoolWaveEncodingAdpcm, sizeof(info.format.encoding));
        break;

    default:
        break;
    }

    info.format.channelCount = m_StreamSoundInfo.channelCount;
    info.format.samplerate = m_StreamSoundInfo.sampleRate;
    info.sampleCount = GetSampleCount();

    info.hasLoop = m_StreamSoundInfo.isLoop;

    if(m_StreamSoundInfo.isLoop)
    {
        info.loopStartFrame = m_StreamSoundInfo.loopStart;
        info.loopEndFrame = m_StreamSoundInfo.frameCount - 1;
        info.originalLoopStartFrame = m_StreamSoundInfo.originalLoopStart;
        info.originalLoopEndFrame = m_StreamSoundInfo.originalLoopEnd - 1;
    }
    else
    {
        info.loopStartFrame = 0;
        info.loopEndFrame = 0;
        info.originalLoopStartFrame = 0;
        info.originalLoopEndFrame = 0;
    }
}

nnatktoolResult BfstmReader::DecodeAll(int channel, void* outputBuffer, size_t outputBufferSize) const
{
    REQUIRES(channel >= 0, ResultInvalidArgument);
    REQUIRES(channel < m_StreamSoundInfo.channelCount, ResultInvalidArgument);
    REQUIRES(outputBuffer != nullptr, ResultInvalidArgument);
    REQUIRES(outputBufferSize >= GetRequiredBufferSizeForDecodeAll(), ResultBufferTooShort);

    nn::codec::AdpcmContext adpcmContext = {0};
    nn::codec::AdpcmParameter adpcmParameter;

    if(m_StreamSoundInfo.encodeMethod == nn::atk::SampleFormat_DspAdpcm)
    {
        // この関数ではループ展開しないので、ループ情報は読み捨てます。
        nn::atk::detail::DspAdpcmLoopParam loopParam;
        nn::atk::DspAdpcmParam atkAdpcmParam;

        if(!m_Reader.ReadDspAdpcmChannelInfo(&atkAdpcmParam, &loopParam, channel))
        {
            return ::ResultInvalidData;
        }

        adpcmContext._predScale = atkAdpcmParam.predScale;

        for(int i = 0; i < 16; i++)
        {
            adpcmParameter._coefficients[i] = atkAdpcmParam.coef[i / 2][i % 2];
        }
    }

    size_t currentOutputBufferPosition = 0;

    // ブロックごとにデコード
    for (int block = 0; block < static_cast<int>(m_StreamSoundInfo.blockCount); block++)
    {
        const void* srcData = GetData(block, channel);
        int16_t* destData = util::BytePtr(outputBuffer, currentOutputBufferPosition).Get<int16_t>();

        int sampleCount = GetSampleCountForBlockChannel(block);
        size_t decodedSize = sampleCount * sizeof(int16_t);

        currentOutputBufferPosition += decodedSize;

        if (currentOutputBufferPosition > outputBufferSize)
        {
            return ::ResultBufferTooShort;
        }

#ifdef ENABLE_BUFFER_OVERFLOW_CHECK
        if(IsLastBlock(block))
        {
            auto fileHeader = reinterpret_cast<const nn::atk::detail::StreamSoundFile::FileHeader*>(m_pBfstmData);
            ptrdiff_t srcDataOffset = util::ConstBytePtr(m_pBfstmData).Distance(srcData);

            switch (m_StreamSoundInfo.encodeMethod)
            {
            case nn::atk::detail::WaveFile::EncodeMethod_DspAdpcm:
                srcDataOffset += sampleCount * nn::codec::AdpcmFrameSize / nn::codec::AdpcmFrameSampleCount;
                break;

            case nn::atk::detail::WaveFile::EncodeMethod_Pcm16:
                srcDataOffset += decodedSize;
                break;

            default:
                return ::ResultInvalidWaveEncoding;
            }

            NN_ASSERT_LESS_EQUAL(static_cast<uint32_t>(srcDataOffset), fileHeader->fileSize);
        }
#endif

        switch (m_StreamSoundInfo.encodeMethod)
        {
        case nn::atk::SampleFormat_DspAdpcm:
            DecodeAdpcm(destData, adpcmContext, adpcmParameter, srcData, sampleCount);
            break;

        case nn::atk::SampleFormat_PcmS16:
            DecodePcm16LE(srcData, destData, sampleCount, decodedSize);
            break;

        default:
            return ::ResultInvalidWaveEncoding;
        }
    }

    return ::ResultSuccess;
}

bool BfstmReader::IsLastBlock(int block) const
{
    NN_ASSERT_GREATER_EQUAL(block, 0);
    NN_ASSERT_LESS(block, static_cast<int>(m_StreamSoundInfo.blockCount));

    if(m_StreamSoundInfo.blockCount == 0)
    {
        return false;
    }
    else
    {
        return block == (static_cast<int>(m_StreamSoundInfo.blockCount) - 1);
    }
}

const void* BfstmReader::GetData(int block) const
{
    NN_ASSERT_GREATER_EQUAL(block, 0);
    NN_ASSERT_LESS(block, static_cast<int>(m_StreamSoundInfo.blockCount));

    return util::ConstBytePtr(
        m_pBfstmData,
        m_Reader.GetSampleDataOffset() + (GetBlockSize() * block)
        ).Get();
}

const void* BfstmReader::GetData(int block, int channel) const
{
    return util::ConstBytePtr(
        GetData(block),
        GetDataSizeForBlockChannel(block) * channel
        ).Get();
}

} } }
