﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/util/util_Vector.h>
#include "gfxutil/GfxContext.h"
#include "EffectPanel.h"
#include "HidPad.h"

namespace
{
    //  パネルのタイトル
    const char* PanelTitle = "Effect";

#if  defined( ATKPLAYER_BUILD_CONFIG_ENABLE_GFX )
    //  AuxBus の矢印の開始位置, マージン
    const float AuxBusArrowStartPositionX = 60.0f;
    const float AuxBusMarginY = 4.0f;
    //  矢印の傘の部分
    const float HeadOfArrowX = 10.0f;
    const float HeadOfArrowY = 0.5f * HeadOfArrowX;
#endif

    //  小さめに表示するテキストのスケール
    const float SmallTextScaleX = 0.8f;
    const float SmallTextScaleY = 0.85f;
    //  パラメータが表示される位置のマージン
    const float ParameterMarginX = 12.0f;
    //  EffectItem のラベルの幅
    const float EffectItemLabelWidth = 70.0f;

    //  描かれる線分の色
    const nn::util::Uint8x4 LineColor = GetUint8x4( 64, 64, 64, 255 );
}

NN_DEFINE_STATIC_CONSTANT( const int EffectPanel::EffectItemCount );

//  初期化します
void EffectPanel::Initialize(float positionX, float positionY, float sizeX, float sizeY) NN_NOEXCEPT
{
    m_Panel.SetTitle( PanelTitle );
    m_Panel.SetPosition( positionX, positionY );
    m_Panel.SetSize( sizeX, sizeY );

    m_CurrentBusIndex = 0;
    m_CurrentEffectItemIndex = 0;
    m_IsUpdateParameters = false;
    m_AuxBusLabel[nn::atk::AuxBus_A].SetText( "AuxA" );
    m_AuxBusLabel[nn::atk::AuxBus_B].SetText( "AuxB" );
    m_AuxBusLabel[nn::atk::AuxBus_C].SetText( "AuxC" );

    m_UsingMemorySizeLabel.SetDrawAlign( gfxutil::Label::DrawAlign_Left );
    m_UsingMemorySizeLabel.SetScale( SmallTextScaleX, SmallTextScaleY );

    m_CurrentEffectLabel.SetScale( SmallTextScaleX, SmallTextScaleY );

    const nn::util::Float2 clientLeftTopPos = m_Panel.GetClientPositionLeftTop();
    const nn::util::Float2 clientRightBottomPos = m_Panel.GetClientPositionRightBottom();
    const float parameterClientAreaSizeX = clientRightBottomPos.x - clientLeftTopPos.x - ParameterMarginX;
    for( int bus = 0; bus < nn::atk::AuxBus_Count; bus++ )
    {
        for( int item = 0; item < EffectItemCount; item++ )
        {
            m_EffectItem[bus][item].Initialize( parameterClientAreaSizeX, EffectItemLabelWidth );
        }
    }
}
//  終了します
void EffectPanel::Finalize(nn::mem::StandardAllocator* pAllocator) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL( pAllocator );
    for( int bus = 0; bus < nn::atk::AuxBus_Count; bus++ )
    {
        for( int item = 0; item < EffectItemCount; item++ )
        {
            m_EffectItem[bus][item].Finalize( pAllocator );
        }
    }
}
//  更新します
void EffectPanel::Update(nn::mem::StandardAllocator* pAllocator) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL( pAllocator );
    for( int bus = 0; bus < nn::atk::AuxBus_Count; bus++ )
    {
        bool isNeedToAppend  = false;
        EffectItem* pEffectItem = m_EffectItem[bus];

        for( int item = 0; item < EffectItemCount; item++ )
        {
            pEffectItem[item].Update();
            isNeedToAppend = pEffectItem[item].IsNeedToAppend() || isNeedToAppend;
        }

        if( isNeedToAppend )
        {
            //  AuxBus の Effect を Clear し、Append します
            bool isClearable = true;
            for( int item = 0; item < EffectItemCount; item++ )
            {
                isClearable = pEffectItem[item].PrepareToClear() && isClearable;
            }

            if( isClearable )
            {
                const nn::atk::AuxBus auxBus = static_cast<nn::atk::AuxBus>( bus );
                nn::atk::SoundSystem::ClearEffect( auxBus );

                for( int item = 0; item < EffectItemCount; item++ )
                {
                    pEffectItem[item].Append( auxBus, pAllocator );
                }
            }
        }
    }

    NN_ABORT_UNLESS_RANGE( m_CurrentBusIndex, 0, nn::atk::AuxBus_Count );
    m_CurrentEffectLabel.SetText( "%s-%d:None", m_AuxBusLabel[m_CurrentBusIndex].GetText(), m_CurrentEffectItemIndex );

    size_t usingMemorySize = 0;
    for( int bus = 0; bus < nn::atk::AuxBus_Count; bus++ )
    {
        for( int item = 0; item < EffectItemCount; item++ )
        {
            usingMemorySize += m_EffectItem[bus][item].GetUsingMemorySize();
        }
    }
    m_UsingMemorySizeLabel.SetText( "Memory:%dKB", usingMemorySize / 1000 );
}
//  入力による更新を行います
void EffectPanel::UpdateByHid(const HidPad& hidPad) NN_NOEXCEPT
{
    if( m_Panel.IsFocused() )
    {
        NN_ABORT_UNLESS_RANGE( m_CurrentBusIndex, 0, nn::atk::AuxBus_Count );
        NN_ABORT_UNLESS_RANGE( m_CurrentEffectItemIndex, 0, EffectItemCount );
        EffectItem& effectItem = m_EffectItem[m_CurrentBusIndex][m_CurrentEffectItemIndex];

        //  エフェクトの選択, パラメータの調整の切り替え
        if( hidPad.IsTrigger( HidPad::Button_Y ) )
        {
            if( effectItem.IsHasParameters() )
            {
                m_IsUpdateParameters = !m_IsUpdateParameters;
            }
        }

        if( m_IsUpdateParameters )
        {
            //  パラメータの調整
            effectItem.UpdateParameters( hidPad );
        }
        else
        {
            //  エフェクトの選択
            if( hidPad.IsContinue( HidPad::Button_Left ) )
            {
                m_CurrentEffectItemIndex = GetNextValueOnLoopSelection( m_CurrentEffectItemIndex - 1, EffectItemCount );
            }
            else if( hidPad.IsContinue( HidPad::Button_Right ) )
            {
                m_CurrentEffectItemIndex = GetNextValueOnLoopSelection( m_CurrentEffectItemIndex + 1, EffectItemCount );
            }
            if( hidPad.IsContinue( HidPad::Button_Up ) )
            {
                m_CurrentBusIndex = GetNextValueOnLoopSelection( m_CurrentBusIndex - 1, nn::atk::AuxBus_Count );
            }
            else if( hidPad.IsContinue( HidPad::Button_Down ) )
            {
                m_CurrentBusIndex = GetNextValueOnLoopSelection( m_CurrentBusIndex + 1, nn::atk::AuxBus_Count );
            }

            if( hidPad.IsTrigger( HidPad::Button_X ) )
            {
                effectItem.ChangeEffect();
            }
        }
    }
}
//  描画します
void EffectPanel::Draw(gfxutil::GfxContext& gfxContext) NN_NOEXCEPT
{
    m_Panel.Draw( gfxContext );

#if  defined( ATKPLAYER_BUILD_CONFIG_ENABLE_GFX )
    const nn::util::Float2 clientLeftTopPos = m_Panel.GetClientPositionLeftTop();
    const nn::util::Float2 clientRightBottomPos = m_Panel.GetClientPositionRightBottom();
    const bool isFocused = m_Panel.IsFocused();

    float drawPosY = clientLeftTopPos.y + AuxBusMarginY;
    gfxutil::FontRenderer& fontRenderer = gfxContext.GetFontRenderer();
    for( int bus = 0; bus < nn::atk::AuxBus_Count; bus++ )
    {
        //  Bus
        m_AuxBusLabel[bus].SetPosition( clientLeftTopPos.x, drawPosY );
        m_AuxBusLabel[bus].Draw( gfxContext );

        //  矢印
        const float height = m_AuxBusLabel[bus].CalculateDrawSize( fontRenderer ).y;
        const nn::util::Vector3f begin( clientLeftTopPos.x + AuxBusArrowStartPositionX, drawPosY + 0.5f * height, 0.0f );
        const nn::util::Vector3f end( clientRightBottomPos.x, begin.GetY(), begin.GetZ() );
        gfxContext.DrawLine( begin, end, LineColor );
        gfxContext.DrawLine( end, nn::util::Vector3f( end.GetX() - HeadOfArrowX, end.GetY() + HeadOfArrowY, 0.0f ), LineColor );
        gfxContext.DrawLine( end, nn::util::Vector3f( end.GetX() - HeadOfArrowX, end.GetY() - HeadOfArrowY, 0.0f ), LineColor );

        //  エフェクトの項目
        const float arrowLength = end.GetX() - begin.GetX() - HeadOfArrowX;
        const float clearance = ( arrowLength - EffectItemCount * EffectItemLabelWidth ) / ( EffectItemCount + 1 );
        float itemPosX = begin.GetX() + clearance;
        for( int item = 0; item < EffectItemCount; item++ )
        {
            m_EffectItem[bus][item].DrawItem( gfxContext, itemPosX, drawPosY, isFocused && !m_IsUpdateParameters, bus == m_CurrentBusIndex && item == m_CurrentEffectItemIndex );
            itemPosX += clearance + EffectItemLabelWidth;
        }

        drawPosY += height + AuxBusMarginY;
    }

    //  使用メモリ量
    m_UsingMemorySizeLabel.SetPosition( clientRightBottomPos.x, drawPosY );
    m_UsingMemorySizeLabel.Draw( gfxContext );
    drawPosY += m_UsingMemorySizeLabel.CalculateDrawSize( fontRenderer ).y;

    //  エフェクトリストとパラメータリストの区切り線
    {
        const nn::util::Vector3f from( clientLeftTopPos.x, drawPosY, 0.0f );
        const nn::util::Vector3f to( clientRightBottomPos.x, from.GetY(), from.GetZ() );
        gfxContext.DrawLine( from, to, LineColor );
    }

    //  選択中の EffectItem
    m_CurrentEffectLabel.SetPosition( clientLeftTopPos.x, drawPosY );
    m_CurrentEffectLabel.Draw( gfxContext );
    drawPosY += m_CurrentEffectLabel.CalculateDrawSize( fontRenderer ).y;

    //  パラメータの表示
    m_EffectItem[m_CurrentBusIndex][m_CurrentEffectItemIndex].DrawParameters( gfxContext, clientLeftTopPos.x + ParameterMarginX, drawPosY, isFocused, m_IsUpdateParameters );
#endif
}

//  フォーカスを設定します
void EffectPanel::SetFocused(bool isFocused) NN_NOEXCEPT
{
    m_Panel.SetFocused( isFocused );
}
