﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

// TODO : なるべく : スレッド周りを nn 利用に切り替え。


#include <nn/vfx/vfx_TargetDef.h>
#include <nn/vfx/vfx_Misc.h>
#include <nn/vfx/viewer/vfx_ToolConnector.h>
#include <nn/vfx/viewer/vfx_Message.h>
#include <nn/ige/ige_HtcsHelper.h>

#include <nn/os.h>

#include <nn/vfx/viewer/vfx_CmdReceiver.h>

namespace nn {
namespace vfx {
namespace viewer {
namespace detail {

static const size_t SendBufferSize = ( 1024 * 1024 );             //!< TBD
static const size_t ReadBufferSize = ( 1024 * 1024 * 5 );         //!< TBD

//------------------------------------------------------------------------------
//  初期化
//------------------------------------------------------------------------------
bool ToolConnectorActual::Initialize( nn::vfx::Heap*  pHeap ) NN_NOEXCEPT
{
    m_pVfxHeap       = pHeap;
    m_ReadBufferSize = ReadBufferSize;
    m_pReadBuffer    = NULL;
    m_IsDetermined   = true;
    m_IsConnected    = false;
    m_pReceive       = detail::CreateCommandReceiver( m_pVfxHeap );
    m_pSender        = detail::CreateCommandSender( m_pVfxHeap, detail::SendBufferSize );

    m_pSender->SetToolConnector( this );
    // 読み込みバッファ確保
    m_pReadBuffer = reinterpret_cast< uint8_t* >( m_pVfxHeap->Alloc( m_ReadBufferSize ) );
    memset( m_pReadBuffer, 0, m_ReadBufferSize );

    // 通信スレッドの起動

    m_IsOK = true;

    return m_IsOK;
}

void ToolConnectorActual::StopThread() NN_NOEXCEPT
{
    if( m_IsOK )
    {
        m_IsConnected = false;
        m_IsDetermined = false;
        m_IsOK = false;
    }
}

//------------------------------------------------------------------------------
//  終了処理
//------------------------------------------------------------------------------
void ToolConnectorActual::Finalize() NN_NOEXCEPT
{
    if( m_IsOK )
    {
        m_IsConnected = false;
        m_IsDetermined = false;
        m_IsOK = false;
    }

    if( m_Handle != InvalidSocket )
    {
        nn::ige::HtcsHelperClose( m_Handle );
    }

    if( m_pReadBuffer )
    {
        m_pVfxHeap->Free( m_pReadBuffer );
        m_pReadBuffer = NULL;
    }
        // コマンド送信クラスの終了処理
    detail::DestroyCommandSender( m_pVfxHeap );

    // コマンド受信クラスの終了処理
    detail::DestroyCommandReceiver( m_pVfxHeap );

}

//------------------------------------------------------------------------------
//  パケット送信
//------------------------------------------------------------------------------
void ToolConnectorActual::SendPacket() NN_NOEXCEPT
{
    if( m_Handle != InvalidSocket )
    {
        SendPacket( m_Handle );
    }
}

void ToolConnectorActual::SendPacket( int param ) NN_NOEXCEPT
{
    const int sock = param;

    while( m_pSender->GetCommand() != NULL )
    {
        Command* cmd = m_pSender->GetCommand();

        void* buffer = cmd->GetCommandBuffer();

        int len = static_cast< int >( nn::ige::HtcsHelperSend(sock, buffer, cmd->GetCommandBufferSize(), 0) );
        if (len == InvalidSocket)
        {
            nn::vfx::detail::OutputLog( "ERROR: Send Packet error %d\n", nn::ige::HtcsHelperGetLastError() );
        }

        m_pSender->PopCommand();
    }
}

//------------------------------------------------------------------------------
//  パケットの確認
//------------------------------------------------------------------------------
bool ToolConnectorActual::IsHandlePacket( uint8_t* pBuffer ) NN_NOEXCEPT
{
    if( pBuffer == NULL )
    {
        return false;
    }

    nn::vfx::viewer::detail::Message* pHeader = reinterpret_cast< nn::vfx::viewer::detail::Message* >( pBuffer );
    nn::vfx::viewer::detail::Message  msg = ( *pHeader );

    // PINGパケットはそれはそれでよし
    if( msg.type == 0 && msg.bufferSize == 0 )
    {
        return true;
    }

    if( ( msg.type >= nn::vfx::viewer::detail::MessageType_Control )
        && ( msg.type <= nn::vfx::viewer::detail::MessageType_EmitterSetRequest ) )
    {
        if( msg.bufferSize != 0 )
        {
            return true;
        }
    }

    return false;
}

//------------------------------------------------------------------------------
//  パケットのスキップ
//------------------------------------------------------------------------------
uint8_t* ToolConnectorActual::SkipPacket( size_t* pOutSize, uint8_t* pBuffer ) NN_NOEXCEPT
{
    *pOutSize = 0;
    if( pBuffer != NULL )
    {
        *pOutSize = sizeof( nn::vfx::viewer::detail::Message );
        pBuffer += *pOutSize;
    }

    return pBuffer;
}

//------------------------------------------------------------------------------
//  パケットの受信
//------------------------------------------------------------------------------
uint32_t ToolConnectorActual::ReceivePacket( int handle, void* pReceiveBuffer, size_t receiveSize ) NN_NOEXCEPT
{
    const int sock = handle;
    int status = static_cast< int >( nn::ige::HtcsHelperRecv(sock, pReceiveBuffer, receiveSize, 0) );

    if ( status == InvalidSocket )
    {
        nn::vfx::detail::OutputLog( "Error in recv socket %d\n", nn::ige::HtcsHelperGetLastError() );
        return 0;
    }
    else if( status == 0 )
    {
        nn::vfx::detail::OutputLog( "Connenction closed %d\n", nn::ige::HtcsHelperGetLastError() );
        return 0;
    }
    else
    {
        return static_cast< uint32_t >( status );
    }
}


//------------------------------------------------------------------------------
//  パケットデータの取得
//------------------------------------------------------------------------------
bool ToolConnectorActual::ReadPacket( int param ) NN_NOEXCEPT
{
    if( m_pReceive == NULL )
    {
        return true;
    }

    if( m_pReceive->g_pCommandReceiver == NULL )
    {
        return true;
    }

    // ヘッダ分読み込み
    size_t headerSize = ReceivePacket( param, m_pReadBuffer, sizeof( nn::vfx::viewer::detail::Message ) );

    // 受信データがない場合はfalseとして終了
    if( headerSize == 0 )
    {
        return false;
    }

    // ヘッダ分読めない場合はパケットぶっ壊れてるから何やってもダメ
    if( headerSize != sizeof( nn::vfx::viewer::detail::Message ) )
    {
        nn::vfx::detail::OutputLog( "[RecvErr]Imcomplete header!\n" );
        return false;
    }

    // 無効なヘッダを読んじゃった場合もダメ
    if( !IsHandlePacket( m_pReadBuffer ) )
    {
        nn::vfx::detail::OutputLog( "[RecvErr]Invalid message!\n" );
        return false;
    }

    nn::vfx::viewer::detail::Message* pHeader = reinterpret_cast< nn::vfx::viewer::detail::Message* >( m_pReadBuffer );
    nn::vfx::viewer::detail::Message msg = ( *pHeader );

    if( ( msg.type >  nn::vfx::viewer::detail::MessageType_Control ) &&
        ( msg.type <= nn::vfx::viewer::detail::MessageType_EmitterSetRequest ) )
    {
        // 期待サイズを算出.
        size_t size = msg.bufferSize;

        // バッファサイズ分読み込み
        size_t readSize = ReceivePacket( param, m_pReadBuffer + sizeof( nn::vfx::viewer::detail::Message ), size );

        if( size == readSize )
        {
            // バッファサイズ分読めてるならコマンドとして積む
            m_pReceive->ReceiveProc( m_pReadBuffer, size + sizeof( nn::vfx::viewer::detail::Message ) );
        }
        else
        {
            // バッファサイズ分読めてないなら何やってもダメ
            nn::vfx::detail::OutputLog( "[RecvErr]Imcomplete message!\n" );
            return false;
        }
    }

    return true;
}

//------------------------------------------------------------------------------
//  ソケットをオープンする
//------------------------------------------------------------------------------
bool ToolConnectorActual::OpenSocket( int *listenSocket, int *sock ) NN_NOEXCEPT
{
    // ソケットの作成
    *listenSocket = nn::ige::HtcsHelperSocket();
    if ( *listenSocket == InvalidSocket )
    {
        nn::vfx::detail::OutputLog( "+----------------------------------+ \n" );
        nn::vfx::detail::OutputLog( "| Please launch the Target Manager | \n" );
        nn::vfx::detail::OutputLog( "+----------------------------------+ \n" );
        while ( m_IsDetermined )
        {
            *listenSocket = nn::ige::HtcsHelperSocket();
            if (*listenSocket != InvalidSocket)
            {
                break;
            }
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(100));
        }
    }

    if ( *listenSocket == InvalidSocket )
    {
        return false;
    }

    // ソケットの設定
    nn::htcs::SockAddrHtcs addr;
    addr.family = nn::htcs::HTCS_AF_HTCS;
    addr.peerName = nn::ige::HtcsHelperGetPeerNameAny();

    for (int i = 0; i < sizeof(STR_HEADER) / sizeof(char); ++i)
    {
        addr.portName.name[i] = STR_HEADER[i];
    }

    // Bindしつつ、Accept待ち時にスレッドが死ねるようにノンブロッキングにする
    nn::ige::HtcsHelperBind( *listenSocket, &addr );
    nn::ige::HtcsHelperFcntl( *listenSocket, nn::htcs::HTCS_F_SETFL, nn::htcs::HTCS_O_NONBLOCK );

    // TCPクライアントからの接続要求を待てる状態にする
    int ret = nn::ige::HtcsHelperListen( *listenSocket, 5 );
    if ( ret == InvalidSocket )
    {
        return false;
    }

    // TCPクライアントからの接続要求を受け付ける
    nn::vfx::detail::OutputLog( "Waiting for connect...\n" );
    while ( m_IsDetermined )
    {
        *sock = nn::ige::HtcsHelperAccept( *listenSocket, NULL );
        if (*sock != InvalidSocket)
        {
            nn::ige::HtcsHelperFcntl( *sock, nn::htcs::HTCS_F_SETFL, 0 );
            break;
        }
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(100));
    }

    if ( *sock == InvalidSocket )
    {
        nn::vfx::detail::OutputLog( "Socket open error %d\n", nn::ige::HtcsHelperGetLastError() );
        if ( *listenSocket != InvalidSocket )
        {
            nn::ige::HtcsHelperClose( *listenSocket );
            *listenSocket = InvalidSocket;
        }
        return false;
    }

    m_Handle = *sock;
    nn::vfx::detail::OutputLog( "Server start \n" );
    return true;
}

//------------------------------------------------------------------------------
//  パケットの受信スレッド
//------------------------------------------------------------------------------
int ToolConnectorActual::ThreadToolReceiver( int intArg, void * ptrArg ) NN_NOEXCEPT
{
    NN_UNUSED( intArg );
    NN_UNUSED( ptrArg );

    int listenSocket = InvalidSocket;
    int sock = InvalidSocket;
    m_IsConnected = false;

    while ( m_IsDetermined )
    {
        // 接続試行処理
        if ( m_IsConnected == false )
        {
            if (listenSocket != InvalidSocket)
            {
                // このクローズでDaemonからの登録を解除する
                nn::ige::HtcsHelperClose( listenSocket );
                listenSocket = InvalidSocket;
            }

            m_IsConnected = OpenSocket( &listenSocket, &sock );
            if ( m_IsConnected )
            {
                m_Handle = sock;
            }
            else
            {
                // スリープしながら試行を続ける
                nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(100));
                continue;
            }
        }

        // 通信処理
        if ( m_IsConnected == true )
        {
            if( !ReadPacket( sock ) )
            {
                if (nn::ige::HtcsHelperClose(sock) == InvalidSocket)
                {
                    nn::vfx::detail::OutputLog( "ERROR: Can't socket close %d\n", nn::ige::HtcsHelperGetLastError() );
                }

                m_IsConnected = false;
                m_Handle = InvalidSocket;
            }
        }
    }

    // TCPセッションの終了
    if ( sock != InvalidSocket )
    {
        nn::ige::HtcsHelperClose(sock);
    }

    if ( listenSocket != InvalidSocket )
    {
        nn::ige::HtcsHelperClose(listenSocket);
        nn::ige::HtcsHelperShutdown( sock, nn::htcs::HTCS_SHUT_RDWR );
    }

    nn::vfx::detail::OutputLog( "Server stop\n" );

    return 0;
}// NOLINT(readability/fn_size)

} // namespace detail
} // namespace viewer
} // namespace vfx
} // namespace nn

