﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/vfx/vfx_EmitterSet.h>
#include <nn/vfx/vfx_System.h>
#include <nn/vfx/vfx_EmitterCalc.h>
#include <nn/vfx/vfx_MemUtil.h>
#include <nn/vfx/vfx_Callback.h>
#include <nn/vfx/vfx_Stripe.h>
#include <nn/vfx/vfx_StripeConnection.h>
#include <nn/vfx/vfx_SuperStripe.h>

namespace nn {
namespace vfx {

//---------------------------------------------------------------------------
//  初期化処理を行います。
//---------------------------------------------------------------------------
bool EmitterSet::Initialize( int emitterSetId, int createId, int resourceId, int groupId, int maxParticleCount, Heap* pHeap ) NN_NOEXCEPT
{
    // リソースを取得
    Resource* pResource = m_pSystem->GetResource( resourceId );
    if( !pResource )
    {
        detail::OutputWarning( "The Resource is not found in VfxSystem. ResourceId : %d.\n", resourceId );
        return false;
    }

    // Gfxデバイスを取得
    m_pDevice = pResource->GetGfxDevice();

    // リソースセットを取得
    m_pEmitterSetResource = pResource->GetEmitterSetResource( emitterSetId );
    if( !m_pEmitterSetResource )
    {
        detail::OutputWarning( "The EmitterSet is not found in this resource. EmitterSet Id : %d. ResourceId : %d\n", emitterSetId, resourceId );
        return false;
    }

    // エミッタリソースが未セットアップ状態であれば、セットアップを行う
    EmitterResource* pEmitterResource = m_pEmitterSetResource->pEmitterResource;
    while ( pEmitterResource )
    {
        if ( !pEmitterResource->Setup( pResource ) )
        {
            detail::OutputWarning( "EmitterResource Setup Failed.\n" );
            return false;
        }

        for ( int i = 0; i < pEmitterResource->m_ChildEmitterResCount; i++ )
        {
            EmitterResource* pChildEmitterResource = pEmitterResource->m_ChildEmitterResSet[ i ];
            if ( !pChildEmitterResource->Setup( pResource ) )
            {
                detail::OutputWarning( "EmitterResource Setup Failed.\n" );
                return false;
            }
        }

        pEmitterResource = pEmitterResource->m_NextEmitterResource;
    }

    m_EmitterSetId = emitterSetId;
    m_EmitterSetCreateId = createId;
    m_ResourceId = resourceId;
    m_GroupId = groupId;
    m_DrawPriority = detail::DrawPriority_Default;
    m_pNextEmitterSet = NULL;
    m_pPrevEmitterSet = NULL;
    m_pEmitterHead = NULL;
    m_pEmitterTail = NULL;
    m_pResetEmitterHead = NULL;
    m_ResetEmitterCount = 0;
    m_IsLoopEffect = false;
    m_IsFade = false;
    m_IsAnyEmitterFade = false;
    m_EnableDistanceBasedEmission = true;
    m_IsDraw = true;
    m_IsCalculationEnabled = true;
    m_IsSetDirectional = false;
    m_IsDelayCreate = false;
    m_pHeap = pHeap;
    m_DrawPathFlag = 0;
    m_StartFrame = 0;
    m_RuntimeUserPtr = 0;
    m_EmitterFirstCount = 0;
    m_EmitterCreateId = 0;
    nn::util::VectorSet( &m_EmitterVolumeScale, 1.0f, 1.0f, 1.0f );
    nn::util::VectorSet( &m_AutoCalculatedScale, 1.0f, 1.0f, 1.0f );
    nn::util::VectorSet( &m_ParticleScaleForCalculation, 1.0f, 1.0f, 1.0f );
    nn::util::VectorSet( &m_InitialRoate, 0.0f, 0.0f, 0.0f );
    m_DirectionalVel = 1.0f;
    nn::util::VectorSet( &m_Directional, 0.0f, 0.0f, 0.0f );
    m_DrawViewFlag = DrawViewFlag_All;
    m_ProcessingInfo.Clear();
    m_EmissionRatioScale = 1.0f;
    m_EmissionIntervalScale = 1.0f;
    m_ParticleLifeScale = 1.0f;
    m_ReqFrameBufferTexturePath = 0;
    m_ReqDepthBufferTexturePath = 0;
    m_IsManualEmission = false;
    m_ManualEmissionCount = 0;
    m_FrameRate = 1.0f;
    m_InitializeCallback = GetSystem()->GetEmitterSetInitializeCallback();
    m_FinalizeCallback = GetSystem()->GetEmitterSetFinalizeCallback();
    m_IsEmitterSrtDirty = 0;
    m_ManualEmissionAssignCount = maxParticleCount;
    m_pEmitReservationListHead = NULL;
    nn::util::MatrixIdentity( &m_MatrixSrt );
    nn::util::MatrixIdentity( &m_MatrixRt );

    // 共通のリセット処理
    ResetCommon();

    nn::util::VectorSet( &m_Color, 1.0f, 1.0f, 1.0f, 1.0f );
    nn::util::VectorSet( &m_ParticleScale, 1.0f, 1.0f, 1.0f );
    nn::util::VectorSet( &m_ParticleEmissionScale, 1.0f, 1.0f, 1.0f );

    m_FigureVel = 1.0f;
    m_DirectionalVel = 1.0f;
    m_RandomVel = 1.0f;
    nn::util::VectorSet( &m_AdditionalVel, 0.0f, 0.0f, 0.0f );

    m_IsEmitterSrtDirty = 1;

    // エミッタセット初期化コールバックの呼び出し
    if ( m_InitializeCallback )
    {
        EmitterSetInitializeArg arg;
        arg.pSystem = GetSystem();
        arg.pEmitterSet = this;

        m_InitializeCallback( arg );
    }

    return true;
}// NOLINT(readability/fn_size)

//---------------------------------------------------------------------------
//  EmitterSet::Reset() と EmitterSet::Initialize() の共通のリセット処理です
//---------------------------------------------------------------------------
void EmitterSet::ResetCommon() NN_NOEXCEPT
{
    // 親エミッタ数
    m_EmitterCount = m_pEmitterSetResource->emitterCount;

    // エミッタ間共通の乱数の種を生成
    // セット間共通の乱数
    detail::RandomGenerator* pRandomGenerator = detail::Random::GetGlobalRandom();
    m_RandomSeed = pRandomGenerator->GetUnsignedInteger();

    EmitterResource* pEmitterResource = m_pEmitterSetResource->pEmitterResource;
    while( pEmitterResource )
    {
        // エミッタ生成処理
        Emitter* pEmitter = CreateEmitter( pEmitterResource, m_ManualEmissionAssignCount );
        if( !pEmitter )
        {
            pEmitterResource = pEmitterResource->m_NextEmitterResource;
            continue;
        }
        m_EmitterFirstCount++;
        m_ProcessingInfo.emitterCount++;

        // ループエフェクトを含むかどうか。
        if( !pEmitterResource->m_pResEmitter->emission.isOneTime )
        {
            m_IsLoopEffect = true;
        }

        // 「軽量版チャイルド」の場合の共通エミッタ作成
        for( int i = 0; i < pEmitter->m_pEmitterRes->m_ChildEmitterResCount; i++ )
        {
            const EmitterResource* pChildRes = pEmitter->m_pChildEmitterResSet[ i ];
            if( pChildRes && !pChildRes->m_pResEmitter->inherit.enableEmitterParticle )
            {
                // 「軽量版チャイルド」の場合、ここでチャイルド用の共通エミッタを作成
                Emitter* pChildEmitter = CreateEmitter( pEmitter->m_pChildEmitterResSet[ i ], 0, pEmitter, i );
                if ( !pChildEmitter ) continue;

                // 親エミッタの情報を登録
                pChildEmitter->m_ParentEmitterCreateId = pEmitter->m_EmitterCreateId;           // 親エミッタの生成ID
                nn::util::MatrixIdentity( &pChildEmitter->m_MatrixSrt );
                nn::util::MatrixIdentity( &pChildEmitter->m_MatrixRt );

                // ※軽量版で使えなくなる情報（子エミッタ内に、親パーティクルの情報は保持できなくなる）
                //   適切にクリア（害が無い値に）する必要がある
                {
                    const nn::util::Float4 zero = NN_UTIL_FLOAT_4_INITIALIZER( 0, 0, 0, 0 );
                    const nn::util::Float4 e = NN_UTIL_FLOAT_4_INITIALIZER( 1, 1, 1, 1 );
                    pChildEmitter->m_ParentParticleCreateId = -1;                                                       // 親パーティクルの生成ID
                    pChildEmitter->m_ParentParticleLife = static_cast< float >( pEmitter->m_pEmitterData->ptcl.life );  // 親パーティクルの寿命（※ランダム無視）
                    pChildEmitter->m_ParentParticleBirthTime = 0;                                                       // 親パーティクルの生成時間
                    pChildEmitter->m_ParentParticleAttrIndex = -1;                                                      // 親パーティクルのインデックス
                    nn::util::VectorLoad( &pChildEmitter->m_ParentParticleLocalPos, zero.v );                           // 親パーティクルの LocalPos
                    nn::util::VectorLoad( &pChildEmitter->m_ParentParticleLocalVec, zero.v );                           // 親パーティクルの LocalVec
                    nn::util::VectorLoad( &pChildEmitter->m_ParentParticleScale, e.v );                                 // 親パーティクルの Scale
                    nn::util::VectorLoad( &pChildEmitter->m_ParentParticleRotate, zero.v );                             // 親パーティクルの Rotate
                    nn::util::VectorLoad( &pChildEmitter->m_ParentParticleRandom, zero.v );                             // 親パーティクルの 粒の乱数値
                }
            }
        }

        pEmitterResource = pEmitterResource->m_NextEmitterResource;
    }

    m_IsUsage = true;
}

//---------------------------------------------------------------------------
//  リセット処理を行います。
//---------------------------------------------------------------------------
void EmitterSet::Reset() NN_NOEXCEPT
{
    if ( m_pResetEmitterHead )
    {
        FinalizeEmitterList( m_pResetEmitterHead );
        m_pResetEmitterHead = NULL;
        m_ResetEmitterCount = 0;
    }

    // 再生中のエフェクトを再生停止状態する
    {
        Emitter* pEmitter = m_pEmitterHead;
        while (pEmitter)
        {
            if (pEmitter->m_EmitterSetCreateId == m_EmitterSetCreateId &&
                pEmitter->IsAlive())
            {
                pEmitter->m_IsVisible = false;
                pEmitter->m_SkipCalculationFlag = true;
            }

            pEmitter = pEmitter->m_pNextEmitter;
        }
    }

    m_pResetEmitterHead = m_pEmitterHead;
    m_ResetEmitterCount = 3;    // 3フレームのちにm_pResetEmitterHeadを解放

    m_pEmitterHead = NULL;
    m_pEmitterTail = NULL;

    // Dirty立て直し
    m_IsEmitterSrtDirty = 1;

    // リソースを取得しなおす
    Resource* pResource = m_pSystem->GetResource( m_ResourceId );
    NN_SDK_ASSERT_NOT_NULL( pResource );
    m_pEmitterSetResource = pResource->GetEmitterSetResource( m_EmitterSetId );
    NN_SDK_ASSERT_NOT_NULL( m_pEmitterSetResource );

    // 共通のリセット処理
    ResetCommon();
}

//---------------------------------------------------------------------------
//  終了処理を行います。
//---------------------------------------------------------------------------
bool EmitterSet::Finalize() NN_NOEXCEPT
{
    if ( m_pResetEmitterHead )
    {
        FinalizeEmitterList( m_pResetEmitterHead );
        m_pResetEmitterHead = NULL;
        m_ResetEmitterCount = 0;
    }

    bool ret = FinalizeEmitterList( m_pEmitterHead );

    // エミッタセット破棄コールバックの呼び出し
    if( m_FinalizeCallback )
    {
        EmitterSetFinalizeArg arg;
        arg.pSystem = GetSystem();
        arg.pEmitterSet = this;

        m_FinalizeCallback( arg );
        m_FinalizeCallback = NULL;
    }

    m_IsUsage = false;
    m_EmitterSetCreateId = 0xFFFFFFFF;

    return ret;
}

//---------------------------------------------------------------------------
//  終了処理を行います。
//---------------------------------------------------------------------------
bool EmitterSet::FinalizeEmitterList( Emitter* pEmitterHead ) NN_NOEXCEPT
{
    // MEMO: このメソッド内では、エミッタリソースへのアクセスは行わない。

    bool ret = true;

    Emitter* pEmitter = pEmitterHead;
    while( pEmitter )
    {
        Emitter* pNext = pEmitter->m_pNextEmitter;

        if( pEmitter->m_EmitterSetCreateId == m_EmitterSetCreateId &&
            pEmitter->m_pEmitterCalculator &&
            pEmitter->IsAlive() )
        {
            // 子エミッタの終了処理
            for( int i = 0; i < SystemParameters_MaxEmitterInclusionCount; i++ )
            {
                Emitter* pChildEmitter = pEmitter->m_pChildHead[ i ];
                while( pChildEmitter )
                {
                    Emitter* pNextChild = pChildEmitter->m_pNextEmitter;

                    if( pChildEmitter->m_EmitterSetCreateId == m_EmitterSetCreateId &&
                        pChildEmitter->m_pEmitterCalculator &&
                        pChildEmitter->IsAlive() )
                    {
                        // エミッタの終了処理
                        pChildEmitter->Finalize( m_pDevice );

                        // システム依存のエミッタ終了処理
                        m_pSystem->FinalizeEmitter( pChildEmitter );
                    }

                    pChildEmitter = pNextChild;
                }
            }

            // エミッタの終了処理
            pEmitter->Finalize( m_pDevice );

            // システム依存のエミッタ終了処理
            m_pSystem->FinalizeEmitter( pEmitter );
        }

        pEmitter = pNext;
    }

    return ret;
}

//---------------------------------------------------------------------------
//  エミッタセットの計算を指定回数回す。
//---------------------------------------------------------------------------
void EmitterSet::ForceCalculate( int loopCount ) NN_NOEXCEPT
{
    if( loopCount == 0 )
    {
        return;
    }

    Calculate( 1.0f, BufferSwapMode_None, true, NULL );

    for( int i = 0; i < loopCount - 1; i++ )
    {
        Calculate( 1.0f, BufferSwapMode_None, true, NULL );
    }
}

//---------------------------------------------------------------------------
//  Calculate 内で必須な処理を行います。
//---------------------------------------------------------------------------
void EmitterSet::RequiredProcessOnCalculate() NN_NOEXCEPT
{
    // リアルタイム編集で削除されたエミッタを破棄する
    if ( m_pResetEmitterHead && m_ResetEmitterCount > 0 )
    {
        m_ResetEmitterCount--;
        if ( m_ResetEmitterCount == 0 )
        {
            FinalizeEmitterList( m_pResetEmitterHead );
            m_pResetEmitterHead = NULL;
            m_ResetEmitterCount = 0;
        }
    }
}

//---------------------------------------------------------------------------
//  エミッタセットの計算処理を行います。
//---------------------------------------------------------------------------
void EmitterSet::Calculate( float frameRate, BufferSwapMode bufferSwapMode, EmitterCalculateLodCallback emitterCalculateLodCallback ) NN_NOEXCEPT
{
    return Calculate( frameRate, bufferSwapMode, false, emitterCalculateLodCallback );
}

//---------------------------------------------------------------------------
//  エミッタセットの計算処理を行います。
//---------------------------------------------------------------------------
void EmitterSet::Calculate( float frameRate, BufferSwapMode bufferSwapMode, bool isForceCalc, EmitterCalculateLodCallback emitterCalculateLodCallback ) NN_NOEXCEPT
{
    // 必須な処理を行う
    RequiredProcessOnCalculate();

    if( !m_IsCalculationEnabled )
    {
        return;
    }

    m_ProcessingInfo.Clear();
    m_DrawPathFlag = 0;
    m_ReqFrameBufferTexturePath = 0;
    m_ReqDepthBufferTexturePath = 0;
    m_FrameRate = frameRate;

    Emitter* pEmitter = m_pEmitterHead;
    bool enableIsAnyEmitterFade = false;

    while( pEmitter )
    {
        Emitter* pNext = pEmitter->m_pNextEmitter;

        // 親エミッタの処理
        // マニュアル放出モード時は既に放出済みの場合のみ処理
        if( m_IsManualEmission && pEmitter->m_ParticleCount == 0 && m_ManualEmissionCount == 0 )
        {
            pEmitter->m_Time += frameRate;
            m_ProcessingInfo.emitterCount++;
        }
        else
        {
            EmitterCalculationResult ret = CalculateEmitter( pEmitter, bufferSwapMode, isForceCalc, emitterCalculateLodCallback );

            // この時点では Emitter はまだ Kill されていないので、「消える予定なのか」を知るためにメモしておく
            pEmitter->m_LastCalculateResult = static_cast< uint8_t >( ret );

            // 子エミッタの処理
            pEmitter->m_IsAliveChildren = false;

            for( int i = 0; i < pEmitter->m_pEmitterRes->m_ChildEmitterResCount; i++ )
            {
                Emitter* pChildEmitter = pEmitter->m_pChildHead[ i ];
                while( pChildEmitter )
                {
                    Emitter* pNextChild = pChildEmitter->m_pNextEmitter;
                    const EmitterCalculationResult childCalcResult = CalculateEmitter( pChildEmitter, bufferSwapMode, isForceCalc, emitterCalculateLodCallback );
                    pChildEmitter->m_LastCalculateResult = static_cast< uint8_t >( ret );

                    if( childCalcResult == EmitterCalculationResult_Kill )
                    {
                        KillEmitter( pChildEmitter );
                    }
                    else
                    {
                        pEmitter->m_IsAliveChildren = true;
                    }

                    if( pChildEmitter->m_IsSoloFade )
                    {
                        // m_IsSoloFade が立っているのを検知したら、通知フラグを立てる
                        enableIsAnyEmitterFade = true;
                    }

                    pChildEmitter = pNextChild;
                }
            }

            // 子エミッタも全死亡であれば親エミッタも削除
            if( ret == EmitterCalculationResult_Kill && !pEmitter->m_IsAliveChildren )
            {
                KillEmitter( pEmitter );
            }
        }

        if( pEmitter->m_IsSoloFade )
        {
            // m_IsSoloFade が立っているのを検知したら、通知フラグを立てる
            enableIsAnyEmitterFade = true;
        }
        pEmitter = pNext;
    }

    if( enableIsAnyEmitterFade )
    {
        // エミッタのどれかが単独で Fade されている
        m_IsAnyEmitterFade = true;
    }

    if( m_IsManualEmission && m_ManualEmissionCount > 0 && frameRate > 0.0f )
    {
        // マニュアル放出: 放出処理が済んだので、ユーザーデータは消しておく
        for( int i = 0; i < m_MaxEmitCountPerFrame; ++i )
        {
            m_pEmitReservationListHead[ i ].pUserData = NULL;
        }
        m_ManualEmissionCount = 0;
    }

    // Dirtyフラグ解除
    if ( frameRate > 0.0f )
    {
        m_IsEmitterSrtDirty = 0;
    }
}


//---------------------------------------------------------------------------
//  エミッタ計算処理
//---------------------------------------------------------------------------
EmitterCalculationResult EmitterSet::CalculateEmitter( Emitter* pEmitter, BufferSwapMode bufferSwapMode, bool forceCalc, EmitterCalculateLodCallback emitterCalculateLodCallback ) NN_NOEXCEPT
{
    bool ret = true;
    bool isBatchProcComputeShader = m_pSystem->IsBatchProcessComputeShaderEmitter();

    if( m_FrameRate > 0.0f )
    {
        // エミッタカリングコールバック
        // コールバック内で emitter->calcSkipFlag/emitter->drawViewFlag のフラグ操作が行われる
        if( emitterCalculateLodCallback )
        {
            EmitterCalculateLodArg arg;
            arg.pEmitter = pEmitter;
            const EmitterCalculationResult lodResult = emitterCalculateLodCallback( arg );
            if( lodResult == EmitterCalculationResult_Kill )
            {
                // Kill の場合はもう処理を回さない
                return EmitterCalculationResult_Kill;
            }
            else if( lodResult == EmitterCalculationResult_Fade )
            {
                // Fade の場合は個別フェードのスイッチを入れる
                if( !pEmitter->m_IsSoloFade )
                {
                    pEmitter->Fade();
                }
            }
        }

        if( !pEmitter->m_SkipCalculationFlag )
        {
            // 通常計算処理
            ret = pEmitter->m_pEmitterCalculator->Calculate( pEmitter, m_FrameRate, bufferSwapMode, ( m_IsFade != 0 ), !m_IsManualEmission, forceCalc );

            // エミッタ計算処理内での生死判定
            if( ret )
            {
                // ストリームアウトエミッタ一括処理モードの場合はリストに追加
                if( isBatchProcComputeShader && pEmitter->m_pEmitterData->emitter.calcType == detail::EmitterCalculationMode_GpuStreamOut && ( pEmitter->m_ParticleCount > 0 || pEmitter->m_pEmitterRes->m_pEmitterPluginData ) )
                {
                    m_pSystem->AddComputeShaderEmitterList( pEmitter );
                }

                UpdateProcessingInfo( pEmitter );
            }
        }
        else
        {
            // calcSkipFlagで処理をスキップ
            pEmitter->m_FrameRate = m_FrameRate;
            if ( pEmitter->SwapBuffer( bufferSwapMode ) )
            {
                pEmitter->m_pEmitterCalculator->UpdateEmitterMatrix( pEmitter );
                pEmitter->m_pEmitterCalculator->UpdateCurrentParticleGpuBufferForCpuEmitter( pEmitter );
                pEmitter->m_pEmitterCalculator->MakeDynamicConstantBuffer( pEmitter, m_FrameRate, m_FrameRate );
            }

            UpdateProcessingInfo( pEmitter );
            m_ProcessingInfo.emitterCountSkippedCalculation++;
        }

        m_GlobalCounter++;
    }
    else
    {
        // フレームを進めない
        // このときは pEmitter->m_FrameRate を更新しない（直前のフレームレートを残しておくため）
        if ( pEmitter->SwapBuffer( bufferSwapMode ) )
        {
            pEmitter->m_pEmitterCalculator->UpdateEmitterMatrix( pEmitter );
            pEmitter->m_pEmitterCalculator->UpdateCurrentParticleGpuBufferForCpuEmitter( pEmitter );
            pEmitter->m_pEmitterCalculator->MakeDynamicConstantBuffer( pEmitter, m_FrameRate, m_FrameRate );
        }

        UpdateProcessingInfo( pEmitter );
    }

    // 描画パスフラグを立てる
    m_DrawPathFlag |= ( 0x1 << pEmitter->m_DrawPathFlag );

    // フレームバッファ要求フラグを立てる
    if( pEmitter->IsRequestFrameBufferTexture() )
    {
        m_ReqFrameBufferTexturePath |= ( 0x1 << pEmitter->m_DrawPathFlag );
    }

    // デプスバッファ要求フラグを立てる
    if( pEmitter->IsRequestDepthTexture() )
    {
        m_ReqDepthBufferTexturePath |= ( 0x1 << pEmitter->m_DrawPathFlag );
    }

    // ここまで処理が来た場合は、 true なら生存状態、 false なら Kill して終了する
    return ret ? EmitterCalculationResult_Alive : EmitterCalculationResult_Kill;
}


//---------------------------------------------------------------------------
// エミッタ描画GPU処理コスト計測コールバック
//---------------------------------------------------------------------------
void DrawEmitterProfilerCallbackDefault( DrawEmitterProfileCallback profileCallback,
    nn::gfx::CommandBuffer* pCommandBuffer,
    System*                 pSystem,
    EmitterSet*             pEmitterSet,
    Emitter*                pEmitter,
    bool                    beforeRender,
    ShaderType              shaderType,
    bool                    doComputeShaderProcess,
    void*                   pUserParam,
    int                     resourceId,
    DrawParameterArg*       pDrawParameterArg ) NN_NOEXCEPT
{
    DrawEmitterProfilerArg arg;
    arg.pCommandBuffer = pCommandBuffer;
    arg.pSystem = pSystem;
    arg.pEmitterSet = pEmitterSet;
    arg.pEmitter = pEmitter;
    arg.beforeRenderEmitter = beforeRender;
    arg.shaderType = shaderType;
    arg.isComputeShaderMode = doComputeShaderProcess;
    arg.pUserParam = pUserParam;
    arg.resourceId = resourceId;
    arg.pDrawParameterArg = pDrawParameterArg;

    profileCallback( arg );
}

//---------------------------------------------------------------------------
//  チャイルドエミッタセットの描画処理を行います。
//---------------------------------------------------------------------------
void EmitterSet::DrawChildEmitter( nn::gfx::CommandBuffer*     pCommandBuffer,
                                   bool                        beforeThanParent,
                                   Emitter*                    pParentEmitter,
                                   uint32_t                    drawPathFlag,
                                   bool                        doComputeShaderProcess,
                                   void*                       pUserParam,
                                   DrawParameterArg*           pDrawParameterArg,
                                   EmitterDrawCullingCallback  emitterDrawCullingCallback,
                                   DrawEmitterProfileCallback  profileCallback ) NN_NOEXCEPT
{
    // 子エミッタの処理
    for( int i = 0; i < pParentEmitter->m_pEmitterRes->m_ChildEmitterResCount; i++ )
    {
        if( pParentEmitter->m_pEmitterRes->m_ChildEmitterResSet[ i ]->m_pResEmitter->inherit.preDraw == static_cast< uint8_t >( beforeThanParent ) )
        {
            Emitter* pChildEmitter = pParentEmitter->m_pChildHead[ i ];
            while( pChildEmitter )
            {
                Emitter* pNextChild = pChildEmitter->m_pNextEmitter;
                DrawEmitter( pCommandBuffer, pChildEmitter, drawPathFlag, doComputeShaderProcess, pUserParam, pDrawParameterArg, emitterDrawCullingCallback, profileCallback );
                pChildEmitter = pNextChild;
            }
        }
    }
}

//---------------------------------------------------------------------------
//  エミッタセットの描画コマンド発行処理を行います。
//---------------------------------------------------------------------------
void EmitterSet::Draw(
    nn::gfx::CommandBuffer*    pCommandBuffer,
    int                        processingIndex,
    uint32_t                   drawPathFlag,
    bool                       doComputeShaderProcess,
    void*                      pUserParam,
    EmitterDrawCullingCallback emitterDrawCullingCallback,
    DrawEmitterProfileCallback emitterDrawProfileCallback ) NN_NOEXCEPT
{
    DrawParameterArg* pDrawParameterArg = m_pSystem->GetDrawParameterArg( processingIndex );

    return Draw( pCommandBuffer, drawPathFlag, doComputeShaderProcess, pUserParam,
                pDrawParameterArg, emitterDrawCullingCallback, emitterDrawProfileCallback );
}

//---------------------------------------------------------------------------
//  エミッタセットの描画コマンド発行処理を行います。
//---------------------------------------------------------------------------
void EmitterSet::Draw( nn::gfx::CommandBuffer*    pCommandBuffer,
                       uint32_t                   drawPathFlag,
                       bool                       doComputeShaderProcess,
                       void*                      pUserParam,
                       DrawParameterArg*          pDrawParameterArg,
                       EmitterDrawCullingCallback emitterDrawCullingCallback,
                       DrawEmitterProfileCallback emitterDrawProfileCallback ) NN_NOEXCEPT
{
    // TODO : なるべく : EffectMakerによるVisibility操作
    if( !m_pEmitterSetResource->isVisible && m_GroupId == ( SystemParameters_MaxGroupCount - 1 ) )
    {
        return;
    }
#if !defined( NN_SDK_BUILD_RELEASE )
    if( emitterDrawProfileCallback )
    {
        DrawEmitterProfilerCallbackDefault( emitterDrawProfileCallback, pCommandBuffer, this->GetSystem(), this, NULL, true, pDrawParameterArg->m_ShaderType, doComputeShaderProcess, pUserParam, m_ResourceId, pDrawParameterArg );
    }
#endif
    Emitter* pEmitter = m_pEmitterHead;
    while( pEmitter )
    {
        // 子エミッタの描画処理(親より前に描画)
        if( pEmitter->m_IsAliveChildren )
        {
            DrawChildEmitter( pCommandBuffer, true, pEmitter, drawPathFlag, doComputeShaderProcess, pUserParam, pDrawParameterArg, emitterDrawCullingCallback, emitterDrawProfileCallback );
        }

        // 親エミッタの描画処理
        DrawEmitter( pCommandBuffer, pEmitter, drawPathFlag, doComputeShaderProcess, pUserParam, pDrawParameterArg, emitterDrawCullingCallback, emitterDrawProfileCallback );

        // 子エミッタの描画処理(親より後に描画)
        if( pEmitter->m_IsAliveChildren )
        {
            DrawChildEmitter( pCommandBuffer, false, pEmitter, drawPathFlag, doComputeShaderProcess, pUserParam, pDrawParameterArg, emitterDrawCullingCallback, emitterDrawProfileCallback );
        }

        pEmitter = pEmitter->m_pNextEmitter;
    }
#if !defined( NN_SDK_BUILD_RELEASE )
    if( emitterDrawProfileCallback )
    {
        DrawEmitterProfilerCallbackDefault( emitterDrawProfileCallback, pCommandBuffer, this->GetSystem(), this, NULL, false, pDrawParameterArg->m_ShaderType, doComputeShaderProcess, pUserParam, m_ResourceId, pDrawParameterArg );
    }
#endif
}

//---------------------------------------------------------------------------
//  エミッタ描画処理
//---------------------------------------------------------------------------
void EmitterSet::DrawEmitter( nn::gfx::CommandBuffer*     pCommandBuffer,
                              Emitter*                    pEmitter,
                              uint32_t                    drawPathFlag,
                              bool                        doComputeShaderProcess,
                              void*                       pUserParam,
                              DrawParameterArg*           pDrawParameterArg,
                              EmitterDrawCullingCallback  emitterDrawCullingCallback,
                              DrawEmitterProfileCallback  profileCallback ) NN_NOEXCEPT
{
    bool drawCullCheck = true;
    bool isAlive = pEmitter->IsAlive();
    bool isRenderEnabled = pEmitter->IsRenderAvailable();
    bool drawViewEnable = ( pEmitter->m_DrawViewFlag & pDrawParameterArg->m_DrawViewFlag ) != 0;
    bool drawPathEnable = ( drawPathFlag & ( 0x1 << pEmitter->m_DrawPathFlag ) ) != 0;

    // 描画パスコールバックを更新
    pEmitter->m_DrawPathCallback = GetSystem()->GetDrawPathRenderStateSetCallback( static_cast< DrawPathFlag >( 0x01 << pEmitter->m_DrawPathFlag ) );

    if( isAlive && isRenderEnabled && drawViewEnable && drawPathEnable )
    {
        if( emitterDrawCullingCallback )
        {
            EmitterDrawCullArg arg;
            arg.pEmitter = pEmitter;
            arg.pUserParam = pUserParam;
            drawCullCheck = emitterDrawCullingCallback( arg );
        }

        if( drawCullCheck )
        {
            if( profileCallback )
            {
                DrawEmitterProfilerCallbackDefault( profileCallback, pCommandBuffer, m_pSystem, NULL, pEmitter, false, pDrawParameterArg->m_ShaderType, doComputeShaderProcess, pUserParam, m_ResourceId, pDrawParameterArg );
            }
            else
            {
                m_pSystem->DrawEmitter( pCommandBuffer, pEmitter, doComputeShaderProcess, pUserParam, pDrawParameterArg );
            }
        }
    }
}

//---------------------------------------------------------------------------
//  エミッタを生成します。
//---------------------------------------------------------------------------
Emitter* EmitterSet::CreateEmitter( const EmitterResource* pEmitterResource, int maxParticleCount, Emitter* pParentEmitter, int childIndex ) NN_NOEXCEPT
{
    Emitter* pEmitter = m_pSystem->AllocEmitter();
    if( !pEmitter )
    {
        return NULL;
    }

    // エミッタプラグインコールバックの取得
    if( pEmitterResource->m_EmitterPluginIndex > 0 )
    {
        pEmitter->m_pCallbackSet[ detail::CallbackSetType_EmitterPlugin ] =
            m_pSystem->GetEmitterPluginCallbackSet( static_cast< detail::EmitterPluginCallbackIndex >( pEmitterResource->m_EmitterPluginIndex - 1 ) );
    }

    // カスタムシェーダコールバックの取得
    if( pEmitterResource->m_pResEmitter->shader.customShaderIndex > 0 )
    {
        const int callbackId = pEmitterResource->m_pResEmitter->shader.customShaderIndex + CallbackId_CustomShaderNone;
        if( m_pSystem->IsEnabelCallbackSet( static_cast< CallbackId >( callbackId ) ) )
        {
            CallbackSet* pCallbackSet = m_pSystem->GetCallbackSet( static_cast< CallbackId >( callbackId ) );
            pEmitter->m_pCallbackSet[ detail::CallbackSetType_CustomShader ] = pCallbackSet;
        }
        else
        {
            detail::OutputWarning( "CustomShader Callback is not established. CustomShaderID : %d\n", pEmitterResource->m_pResEmitter->shader.customShaderIndex );
        }
    }
    else
    {
        // カスタムシェーダ 設定ナシの場合
        if( m_pSystem->IsEnabelCallbackSet( static_cast< CallbackId >( CallbackId_CustomShaderNone ) ) )
        {
            CallbackSet* pCallbackSet = m_pSystem->GetCallbackSet( CallbackId_CustomShaderNone );
            pEmitter->m_pCallbackSet[ detail::CallbackSetType_CustomShader ] = pCallbackSet;
        }
    }

    // カスタムアクションコールバックの取得
    if( pEmitterResource->m_pResEmitter->action.customActionIndex > 0 )
    {
        if( pEmitterResource->m_pCustomActionParam )
        {
            const int callbackId = pEmitterResource->m_pResEmitter->action.customActionIndex - 1;
            if( m_pSystem->IsEnabelCallbackSet( static_cast< CallbackId >( callbackId ) ) )
            {
                CallbackSet* pCallbackSet = m_pSystem->GetCallbackSet( static_cast< CallbackId >( callbackId ) );
                pEmitter->m_pCallbackSet[ detail::CallbackSetType_CustomAction ] = pCallbackSet;
            }
            else
            {
                detail::OutputWarning( "CustomAction Callback is not established. CustomActionID : %d\n", pEmitterResource->m_pResEmitter->action.customActionIndex );
            }
        }
        else
        {
            // MEMO: 下の pEmitter->Initialize() が呼ばれるまで、pEmitter->m_pEmitterSet が設定されていないので一旦設定しておく
            pEmitter->m_pEmitterSet = this;
            detail::Warning( reinterpret_cast< void* >( pEmitter ), detail::RuntimeWarningId_NoCustomActionParameter );
        }
    }

    // エミッタの初期化
    bool ret = pEmitter->Initialize( m_pDevice, this, pEmitterResource, maxParticleCount, m_pHeap, pParentEmitter );
    if( !ret )
    {
        // vfxランタイム側でエミッタ初期化失敗時は、初期化コールバックを呼び出していないので、
        // pEmitter->Finalize では破棄コールバックを呼び出さない
        bool invokeCallback = pEmitter->m_IsEmitterInitializeFailed ? false : true;
        if( pEmitter->m_IsEmitterInitializeFailed )
        {
            detail::OutputWarning( "Emitter Initialize Failed.\n" );
        }
        pEmitter->Finalize( m_pDevice, invokeCallback );
        m_pSystem->FinalizeEmitter( pEmitter );
        return NULL;
    }

    // システム依存のエミッタ初期化
    m_pSystem->InitializeEmitter( pEmitter );

    // 描画パスコールバック
    pEmitter->SetDrawPath( pEmitterResource->m_pResEmitter->emitter.drawPath );

    // チャイルドエミッタであれば、インデックスを付加
    if( childIndex != -1 )
    {
        pEmitter->m_ChildResIndex = childIndex;
    }
    else
    {
        pEmitter->m_ChildResIndex = -1;
    }

    // emitterをセットツリーに追加
    AddEmitterToList( pEmitter, pParentEmitter );

    pEmitter->m_EmitterCreateId = m_EmitterCreateId;
    m_EmitterCreateId++;

    return pEmitter;
}

//---------------------------------------------------------------------------
//  エミッタを削除します。
//---------------------------------------------------------------------------
void EmitterSet::KillEmitter( Emitter* pEmitter ) NN_NOEXCEPT
{
    KillEmitterImpl( pEmitter );
}

//---------------------------------------------------------------------------
//  エミッタを削除します。
//---------------------------------------------------------------------------
void EmitterSet::KillEmitterImpl( Emitter* pEmitter ) NN_NOEXCEPT
{
    // エミッタをリストから削除
    RemoveEmitterFromList( pEmitter );

    // エミッタの終了処理
    pEmitter->Finalize( m_pDevice );

    // システム依存のエミッタ終了処理
    m_pSystem->FinalizeEmitter( pEmitter );
}

//---------------------------------------------------------------------------
//  エミッタを追加します。
//---------------------------------------------------------------------------
void EmitterSet::AddEmitterToList( Emitter* pEmitter, Emitter* pParentEmitter ) NN_NOEXCEPT
{
    // 親の場合
    if( !pEmitter->m_IsChild )
    {
        if( m_pEmitterTail == NULL )
        {
            m_pEmitterTail = pEmitter;
            pEmitter->m_pPrevEmitter = NULL;
            pEmitter->m_pNextEmitter = NULL;
        }
        else
        {
            m_pEmitterTail->m_pNextEmitter = pEmitter;
            pEmitter->m_pPrevEmitter = m_pEmitterTail;
            m_pEmitterTail = pEmitter;
            pEmitter->m_pNextEmitter = NULL;
        }

        if( m_pEmitterHead == NULL )
        {
            m_pEmitterHead = pEmitter;
        }
    }
    else
        // 子の場合
    {
        NN_SDK_ASSERT_NOT_NULL( pParentEmitter );
        pEmitter->m_pParentEmitter = pParentEmitter;
        int index = pEmitter->m_ChildResIndex;

        if( pParentEmitter->m_pChildTail[ index ] == NULL )
        {
            pParentEmitter->m_pChildTail[ index ] = pEmitter;
            pEmitter->m_pPrevEmitter = NULL;
            pEmitter->m_pNextEmitter = NULL;
        }
        else
        {
            pParentEmitter->m_pChildTail[ index ]->m_pNextEmitter = pEmitter;
            pEmitter->m_pPrevEmitter = pParentEmitter->m_pChildTail[ index ];
            pParentEmitter->m_pChildTail[ index ] = pEmitter;
            pEmitter->m_pNextEmitter = NULL;
        }

        if( pParentEmitter->m_pChildHead[ index ] == NULL )
        {
            pParentEmitter->m_pChildHead[ index ] = pEmitter;
        }
    }
}

//---------------------------------------------------------------------------
//  エミッタを削除します。
//---------------------------------------------------------------------------
void EmitterSet::RemoveEmitterFromList( Emitter* pEmitter ) NN_NOEXCEPT
{
    // 親の場合
    if( !pEmitter->m_IsChild )
    {
        if( pEmitter->m_pPrevEmitter && pEmitter->m_pNextEmitter )
        {
            pEmitter->m_pPrevEmitter->m_pNextEmitter = pEmitter->m_pNextEmitter;
            pEmitter->m_pNextEmitter->m_pPrevEmitter = pEmitter->m_pPrevEmitter;
        }
        else if( m_pEmitterHead == pEmitter )
        {
            m_pEmitterHead = pEmitter->m_pNextEmitter;
            if( pEmitter->m_pNextEmitter )
            {
                pEmitter->m_pNextEmitter->m_pPrevEmitter = pEmitter->m_pPrevEmitter;
            }
            else
            {
                m_pEmitterTail = NULL;
            }
        }
        else if( m_pEmitterTail == pEmitter )
        {
            m_pEmitterTail = pEmitter->m_pPrevEmitter;
            if( pEmitter->m_pPrevEmitter )
            {
                pEmitter->m_pPrevEmitter->m_pNextEmitter = pEmitter->m_pNextEmitter;
            }
            else
            {
                m_pEmitterHead = NULL;
            }
        }
    }
    else
        // 子の場合
    {
        NN_SDK_ASSERT_NOT_NULL( pEmitter->m_pParentEmitter );
        Emitter* pParentEmitter = pEmitter->m_pParentEmitter;
        const int childIndex = pEmitter->m_ChildResIndex;

        if( pEmitter->m_pPrevEmitter && pEmitter->m_pNextEmitter )
        {
            pEmitter->m_pPrevEmitter->m_pNextEmitter = pEmitter->m_pNextEmitter;
            pEmitter->m_pNextEmitter->m_pPrevEmitter = pEmitter->m_pPrevEmitter;
        }
        else if( pParentEmitter->m_pChildHead[ childIndex ] == pEmitter )
        {
            pParentEmitter->m_pChildHead[ childIndex ] = pEmitter->m_pNextEmitter;
            if( pEmitter->m_pNextEmitter )
            {
                pEmitter->m_pNextEmitter->m_pPrevEmitter = pEmitter->m_pPrevEmitter;
            }
            else
            {
                pParentEmitter->m_pChildTail[ childIndex ] = NULL;
            }
        }
        else if( pParentEmitter->m_pChildTail[ childIndex ] == pEmitter )
        {
            pParentEmitter->m_pChildTail[ childIndex ] = pEmitter->m_pPrevEmitter;
            if( pEmitter->m_pPrevEmitter )
            {
                pEmitter->m_pPrevEmitter->m_pNextEmitter = pEmitter->m_pNextEmitter;
            }
            else
            {
                pParentEmitter->m_pChildHead[ childIndex ] = NULL;
            }
        }
    }
}

//---------------------------------------------------------------------------
//  エミッタの放出を停止します。
//---------------------------------------------------------------------------
void EmitterSet::Fade() NN_NOEXCEPT
{
    m_IsFade = true;
}

//---------------------------------------------------------------------------
//  リソース更新に伴うアップデートを行います。
//---------------------------------------------------------------------------
bool EmitterSet::UpdateFromResource( EmitterResource* pEmitterResource ) NN_NOEXCEPT
{
    bool result = true;

    Emitter* pEmitter = m_pEmitterHead;

    while( pEmitter )
    {
        if( pEmitter->IsAlive() && pEmitter->m_pEmitterRes == pEmitterResource )
        {
            // エミッタオブジェクトの更新
            // MEMO: こちらのパスから呼ばれる場合は、 pEmitter->m_pParentEmitter が決まっているので直接与える。
            result = pEmitter->ResourceUpdate( m_pDevice, pEmitter->m_pParentEmitter );
            if( !result )
            {
                return false;
            }
        }
        else
        {
            // pEmitterResource でチャイルドのリソースが与えられることもあるので、チャイルドエミッタもトラバースする
            for( int i = 0; i < pEmitter->m_pEmitterRes->m_ChildEmitterResCount; i++ )
            {
                Emitter* pChildEmitter = pEmitter->m_pChildHead[ i ];
                while( pChildEmitter )
                {
                    if( pChildEmitter->IsAlive() && pChildEmitter->m_pEmitterRes == pEmitterResource )
                    {
                        result = pChildEmitter->ResourceUpdate( m_pDevice, pChildEmitter->m_pParentEmitter );
                        if( !result )
                        {
                            return false;
                        }
                    }

                    pChildEmitter = pChildEmitter->m_pNextEmitter;
                }
            }
        }

        pEmitter = pEmitter->m_pNextEmitter;
    }

    return true;
}

//---------------------------------------------------------------------------
//  放出レートのスケール値を設定します。
//---------------------------------------------------------------------------
void EmitterSet::SetEmissionRatioScale( float ratio ) NN_NOEXCEPT
{
    m_EmissionRatioScale = ratio;
    if( m_EmissionRatioScale > 1.0f )
    {
        m_EmissionRatioScale = 1.0f;
    }
}

//---------------------------------------------------------------------------
//  放出間隔のスケール値を設定します。
//---------------------------------------------------------------------------
void EmitterSet::SetEmissionIntervalScale( float ratio ) NN_NOEXCEPT
{
    m_EmissionIntervalScale = ratio;
    if( m_EmissionIntervalScale < 1.0f )
    {
        m_EmissionIntervalScale = 1.0f;
    }
}

//---------------------------------------------------------------------------
//  パーティクル寿命のスケール値を設定します。
//---------------------------------------------------------------------------
void EmitterSet::SetParticleLifeScale( float ratio ) NN_NOEXCEPT
{
    m_ParticleLifeScale = ratio;
    if( m_ParticleLifeScale > 1.0f )
    {
        m_ParticleLifeScale = 1.0f;
    }
}

//---------------------------------------------------------------------------
//  マトリクスを設定します。
//---------------------------------------------------------------------------
void EmitterSet::SetMatrix( const nn::util::Matrix4x3fType& matrixSrt ) NN_NOEXCEPT
{
    m_MatrixSrt = matrixSrt;

    // matrixRT
    nn::util::Vector3fType basisX;
    nn::util::Vector3fType basisY;
    nn::util::Vector3fType basisZ;
    nn::util::MatrixGetAxisX( &basisX, matrixSrt );
    nn::util::MatrixGetAxisY( &basisY, matrixSrt );
    nn::util::MatrixGetAxisZ( &basisZ, matrixSrt );

    nn::util::VectorSetX( &m_AutoCalculatedScale, nn::util::VectorLength( basisX ) );
    nn::util::VectorSetY( &m_AutoCalculatedScale, nn::util::VectorLength( basisY ) );
    nn::util::VectorSetZ( &m_AutoCalculatedScale, nn::util::VectorLength( basisZ ) );

    if( nn::util::VectorGetX( m_AutoCalculatedScale ) > 0.0f )
    {
        const float inv = 1.0f / nn::util::VectorGetX( m_AutoCalculatedScale );
        nn::util::Vector3fType v;
        nn::util::MatrixGetAxisX( &v, matrixSrt );
        nn::util::VectorMultiply( &v, v, inv );
        nn::util::MatrixSetAxisX( &m_MatrixRt, v );
    }
    else
    {
        const nn::util::Vector3fType zero = NN_UTIL_VECTOR_3F_INITIALIZER( 0, 0, 0 );
        nn::util::MatrixSetAxisX( &m_MatrixRt, zero );
    }

    if( nn::util::VectorGetY( m_AutoCalculatedScale ) > 0.0f )
    {
        const float inv = 1.0f / nn::util::VectorGetY( m_AutoCalculatedScale );
        nn::util::Vector3fType v;
        nn::util::MatrixGetAxisY( &v, matrixSrt );
        nn::util::VectorMultiply( &v, v, inv );
        nn::util::MatrixSetAxisY( &m_MatrixRt, v );
    }
    else
    {
        const nn::util::Vector3fType zero = NN_UTIL_VECTOR_3F_INITIALIZER( 0, 0, 0 );
        nn::util::MatrixSetAxisY( &m_MatrixRt, zero );
    }

    if( nn::util::VectorGetZ( m_AutoCalculatedScale ) > 0.0f )
    {
        float inv = 1.0f / nn::util::VectorGetZ( m_AutoCalculatedScale );
        nn::util::Vector3fType v;
        nn::util::MatrixGetAxisZ( &v, matrixSrt );
        nn::util::VectorMultiply( &v, v, inv );
        nn::util::MatrixSetAxisZ( &m_MatrixRt, v );
    }
    else
    {
        const nn::util::Vector3fType zero = NN_UTIL_VECTOR_3F_INITIALIZER( 0, 0, 0 );
        nn::util::MatrixSetAxisZ( &m_MatrixRt, zero );
    }

    nn::util::Vector3fType trans;
    nn::util::MatrixGetAxisW( &trans, matrixSrt );
    nn::util::MatrixSetAxisW( &m_MatrixRt, trans );

    UpdateParticleScale();
    m_IsEmitterSrtDirty = 1;
}

//---------------------------------------------------------------------------
//  マトリクスを設定します。
//---------------------------------------------------------------------------
void EmitterSet::SetPos( const nn::util::Vector3fType& pos ) NN_NOEXCEPT
{
    nn::util::MatrixSetAxisW( &m_MatrixSrt, pos );
    nn::util::MatrixSetAxisW( &m_MatrixRt, pos );
    m_IsEmitterSrtDirty = 1;
}

//---------------------------------------------------------------------------
//  エミッタセットを削除します。
//---------------------------------------------------------------------------
void EmitterSet::Kill( bool immediate ) NN_NOEXCEPT
{
    m_pSystem->KillEmitterSet( this, immediate );
}

//---------------------------------------------------------------------------
//  無限寿命のエミッタを削除します。
//---------------------------------------------------------------------------
void EmitterSet::KillInfinityEmitter() NN_NOEXCEPT
{
    Emitter* pEmitter = m_pEmitterHead;
    while( pEmitter )
    {
        Emitter* pNext = pEmitter->GetNextEmitter();
        const nn::vfx::detail::ResEmitter* pResEmitter = pEmitter->GetResEmitter();

        for( int i = 0; i < nn::vfx::SystemParameters_MaxEmitterInclusionCount; ++i )
        {
            nn::vfx::Emitter* pChildEmitter = pEmitter->GetChildEmitterHead( i );
            while( pChildEmitter )
            {
                Emitter* pNextChild = pChildEmitter->GetNextEmitter();
                const nn::vfx::detail::ResEmitter* pResChildEmitter = pChildEmitter->GetResEmitter();
                if( pResChildEmitter->ptcl.isLifeInfinity && !pResChildEmitter->emitter.isScaleFade && !pResChildEmitter->emitter.isFadeAlphaFade )
                {
                    // αフェードもスケールフェードも入ってない場合。（※放出停止の有無はこの時点で関係ない）
                    KillEmitter( pChildEmitter );
                }
                pChildEmitter = pNextChild;
            }
        }

        if( pResEmitter->ptcl.isLifeInfinity && !pResEmitter->emitter.isScaleFade && !pResEmitter->emitter.isFadeAlphaFade )
        {
            // αフェードもスケールフェードも入ってない場合。（※放出停止の有無はこの時点で関係ない）
            KillEmitter( pEmitter );
        }
        pEmitter = pNext;
    }
}

//---------------------------------------------------------------------------
//  無限寿命のエミッタ( パーティクル )を含むかどうかチェックします。
//---------------------------------------------------------------------------
bool EmitterSet::IsHaveInfinityEmitter() const NN_NOEXCEPT
{
    return m_pEmitterSetResource->isLifeInfinity;
}

//---------------------------------------------------------------------------
//  所属エミッタインスタンスを取得します。
//---------------------------------------------------------------------------
Emitter* EmitterSet::GetAliveEmitter( int index ) const NN_NOEXCEPT
{
    int count = 0;
    Emitter* pEmitter = m_pEmitterHead;
    while( pEmitter )
    {
        if( count == index )
        {
            return pEmitter;
        }
        count++;
        pEmitter = pEmitter->m_pNextEmitter;
    }

    return NULL;
}

//---------------------------------------------------------------------------
//  エミッタ描画処理の有効/無効を設定します。
//---------------------------------------------------------------------------
void EmitterSet::SetEmitterVisible( const char* emitterName, bool flag ) NN_NOEXCEPT
{
    Emitter* pEmitter = m_pEmitterHead;
    while( pEmitter )
    {
        if( strcmp( pEmitter->m_pEmitterData->name, emitterName ) == 0 )
        {
            pEmitter->m_IsVisible = flag;
        }
        pEmitter = pEmitter->m_pNextEmitter;
    }
}

//---------------------------------------------------------------------------
//  パーティクルを手動で放出します。
//---------------------------------------------------------------------------
void EmitterSet::EmitParticle( const nn::util::Vector3fType& pos, void* pUserData ) NN_NOEXCEPT
{
    if( !m_IsManualEmission )
    {
        detail::Warning( this, detail::RuntimeWarningId_IsNotManualEmitMode );
        return;
    }

    if( m_ManualEmissionCount >= m_MaxEmitCountPerFrame )
    {
        // 1フレームあたりの最大放出数を超えて放出しようとしている！！
        detail::Warning( this, detail::RuntimeWarningId_ManualEmitSizeOver );
        return;
    }

    // エミッタが既に溢れていたら警告を出す
    bool isAnyFullEmitter = false;
    Emitter* pEmitter = m_pEmitterHead;
    while( pEmitter )
    {
        const int worstEmitNum = static_cast< int >( GetEmissionRatioScale() * pEmitter->GetResEmitter()->emission.rate * pEmitter->m_EmitRatio );
        if( pEmitter->GetCalculationType() == detail::EmitterCalculationMode_Cpu &&
            pEmitter->m_ParticleProcessingCount + worstEmitNum > pEmitter->m_MaxParticleAssignmentCount )
        {
            // 最悪ケースの場合。放出レートランダムの設定によっては最大放出レート分損するが、
            // 途中で放出できなくなるよりはマシとする。
            detail::Warning( pEmitter, detail::RuntimeWarningId_ManualEmitterIsFull );
            isAnyFullEmitter = true;
        }
        pEmitter = pEmitter->m_pNextEmitter;
    }
    if( isAnyFullEmitter )
    {
        return;
    }

    // 予約リスト（EmitterSetが管理する）に一つ追加
    const int index = m_ManualEmissionCount;
    nn::util::MatrixSet( &m_pEmitReservationListHead[ index ].emitParam,
        0.0f, 0.0f, 0.0f,
        0.0f, 0.0f, 0.0f,
        0.0f, 0.0f, 0.0f,
        nn::util::VectorGetX( pos ), nn::util::VectorGetY( pos ), nn::util::VectorGetZ( pos ) );
    m_pEmitReservationListHead[ index ].pUserData     = pUserData;
    m_pEmitReservationListHead[ index ].isEmitMatrix  = false;
    m_pEmitReservationListHead[ index ].emissionRatio = GetEmissionRatioScale();
    m_ManualEmissionCount++;
}

//---------------------------------------------------------------------------
//  パーティクルを手動で放出します。
//---------------------------------------------------------------------------
void EmitterSet::EmitParticle( const nn::util::Matrix4x3fType& emitMatrix, void* pUserData ) NN_NOEXCEPT
{
    if ( !m_IsManualEmission )
    {
        detail::Warning( this, detail::RuntimeWarningId_IsNotManualEmitMode );
        return;
    }

    if ( m_ManualEmissionCount >= m_MaxEmitCountPerFrame )
    {
        // 1フレームあたりの最大放出数を超えて放出しようとしている！！
        detail::Warning( this, detail::RuntimeWarningId_ManualEmitSizeOver );
        return;
    }

    // エミッタが既に溢れていたら警告を出す
    bool isAnyFullEmitter = false;
    Emitter* pEmitter = m_pEmitterHead;
    while( pEmitter )
    {
        const int worstEmitNum = static_cast< int >( GetEmissionRatioScale() * pEmitter->GetResEmitter()->emission.rate * pEmitter->m_EmitRatio );
        if( pEmitter->GetCalculationType() == detail::EmitterCalculationMode_Cpu &&
            pEmitter->m_ParticleProcessingCount + worstEmitNum > pEmitter->m_MaxParticleAssignmentCount )
        {
            // 最悪ケースの場合。放出レートランダムの設定によっては最大放出レート分損するが、
            // 途中で放出できなくなるよりはマシとする。
            detail::Warning( pEmitter, detail::RuntimeWarningId_ManualEmitterIsFull );
            isAnyFullEmitter = true;
        }
        pEmitter = pEmitter->m_pNextEmitter;
    }
    if( isAnyFullEmitter )
    {
        return;
    }

    // 予約リスト（EmitterSetが管理する）に一つ追加
    const int index = m_ManualEmissionCount;
    m_pEmitReservationListHead[ index ].emitParam    = emitMatrix;
    m_pEmitReservationListHead[ index ].pUserData    = pUserData;
    m_pEmitReservationListHead[ index ].isEmitMatrix = true;
    m_pEmitReservationListHead[ index ].emissionRatio = GetEmissionRatioScale();
    m_ManualEmissionCount++;
}



//---------------------------------------------------------------------------
//  カラー RGBA値(乗算値)を設定します。
//---------------------------------------------------------------------------
void EmitterSet::SetEmitterColor( const nn::util::Vector4fType& color0, const nn::util::Vector4fType& color1 ) NN_NOEXCEPT
{
    Emitter* pEmitter = m_pEmitterHead;

    while( pEmitter )
    {
        pEmitter->m_Color0 = color0;
        pEmitter->m_Color1 = color1;
        pEmitter = pEmitter->m_pNextEmitter;
    }
}

//---------------------------------------------------------------------------
//  カラー RGBA値(乗算値)を設定します。
//---------------------------------------------------------------------------
void EmitterSet::SetEmitterColor0( const nn::util::Vector4fType& color0 ) NN_NOEXCEPT
{
    Emitter* pEmitter = m_pEmitterHead;

    while( pEmitter )
    {
        pEmitter->m_Color0 = color0;
        pEmitter = pEmitter->m_pNextEmitter;
    }
}

//---------------------------------------------------------------------------
//  カラー RGBA値(乗算値)を設定します。
//---------------------------------------------------------------------------
void EmitterSet::SetEmitterColor1( const nn::util::Vector4fType& color1 ) NN_NOEXCEPT
{
    Emitter* pEmitter = m_pEmitterHead;

    while( pEmitter )
    {
        pEmitter->m_Color1 = color1;
        pEmitter = pEmitter->m_pNextEmitter;
    }
}

//---------------------------------------------------------------------------
//  個数を指定してパーティクルを手動で放出するモードに設定します。
//---------------------------------------------------------------------------
void EmitterSet::SetManualParticleEmissionWithParticleCount( int particleCount ) NN_NOEXCEPT
{
    m_IsManualEmission = true;
    m_ManualEmissionAssignCount = particleCount;

    Emitter* pEmitter = m_pEmitterHead;
    while( pEmitter )
    {
        if( pEmitter->IsAlive() )
        {
            pEmitter->m_MaxParticleAssignmentCount = m_ManualEmissionAssignCount;
            // MEMO: こちらのパスから呼ばれる場合は、 pEmitter->m_pParentEmitter が決まっているので直接与える。
            pEmitter->ResourceUpdate( m_pDevice, pEmitter->m_pParentEmitter );
        }
        pEmitter = pEmitter->m_pNextEmitter;
    }
}

//---------------------------------------------------------------------------
//  配下のエミッタが持つカスタムアクション用 CallbackSet を上書きします。
//---------------------------------------------------------------------------
void EmitterSet::OverwriteCustomActionCallbackSet( CallbackSet* pCustomActionCallbackSet ) NN_NOEXCEPT
{
    Emitter* pEmitter = m_pEmitterHead;
    while( pEmitter )
    {
        // エミッタプラグインにもコールバックセットはあるが、これは上書きさせない
        pEmitter->m_pCallbackSet[ detail::CallbackSetType_CustomAction ] = pCustomActionCallbackSet;
        pEmitter = pEmitter->m_pNextEmitter;
    }
}

//---------------------------------------------------------------------------
//  エミッタセットへのポインタを取得します。
//---------------------------------------------------------------------------
EmitterSet* Handle::GetEmitterSet() NN_NOEXCEPT
{
#ifdef _VFX_NSDK01215_COMPATIBLE
    return m_pEmitterSet;
#else
    if ( IsValid() )
    {
        return m_pEmitterSet;
    }
    else
    {
        return nullptr;
    }
#endif
}

//---------------------------------------------------------------------------
// エミッタセットへのポインタを取得します。( const版 )
//---------------------------------------------------------------------------
const EmitterSet* Handle::GetEmitterSet() const NN_NOEXCEPT
{
#ifdef _VFX_NSDK01215_COMPATIBLE
    return m_pEmitterSet;
#else
    if ( IsValid() )
    {
        return m_pEmitterSet;
    }
    else
    {
        return nullptr;
    }
#endif
}


//---------------------------------------------------------------------------
//  パーティクル処理数／消費メモリ量などの情報更新
//---------------------------------------------------------------------------
void ProcessingInfo::UpdateProcessingInfo( const Emitter* pEmitter ) NN_NOEXCEPT
{
    if ( pEmitter->GetCalculationType() == detail::EmitterCalculationMode_Cpu )
    {
        cpuParticleCount += pEmitter->GetProcessingParticleCount();
        cpuEmitterCount++;
    }
    if ( pEmitter->GetCalculationType() == detail::EmitterCalculationMode_Gpu )
    {
        gpuParticleCount += pEmitter->GetParticleCount();
        gpuEmitterCount++;
    }
    if ( pEmitter->GetCalculationType() == detail::EmitterCalculationMode_GpuStreamOut )
    {
        gpusoParticleCount += pEmitter->GetParticleCount();
        computeEmitterCount++;
    }
    emitterCount++;

    switch ( pEmitter->GetEmitterResource()->m_EmitterPluginIndex )
    {
    case detail::ConnectionStripeSystem::PluginId:
        connectionStripeCount++;
        break;
    case detail::StripeSystem::PluginId:
        stripeCount += detail::StripeSystem::GetActualStripeCalclationCount( pEmitter );
        break;
    case detail::SuperStripeSystem::PluginId:
        superStripeCount += detail::SuperStripeSystem::GetActualSuperStripeCalclationCount( pEmitter );
        break;
    default:
        break;
    }

    emitterAnimCount += pEmitter->GetCalculatedEmitterAnimationCount();
    allocatedDynamicHeapSize += pEmitter->GetAllocatedFromDynamicHeapSize();
    allocatedGpuBufferSize += pEmitter->GetAllocatedFromGpuBufferSize();
    allocatedGpuBufferSize += pEmitter->GetGfxObjects()->m_ParticleEmitterPluginAttribute.GetAllocatedBufferSize();
    for ( int i = 0; i < CustomShaderConstantBufferIndex_MaxIndexCount; i++ )
    {
        allocatedGpuBufferSize += pEmitter->GetGfxObjects()->m_CustomConstantBuffer[ i ].GetAllocatedBufferSize();
    }
}


//---------------------------------------------------------------------------
//  エミッタセットが保持するランダムの種を設定します。
//---------------------------------------------------------------------------
void EmitterSet::SetEmitterSetRandomSeed( uint32_t  randomSeed ) NN_NOEXCEPT
{
    m_RandomSeed = randomSeed;

    Emitter* pEmitter = m_pEmitterHead;
    while ( pEmitter )
    {
        if ( pEmitter->GetEmitterResource()->m_pResEmitter->emitter.randomSeedType == detail::EmitterRandomSeedType_Eset )
        {
            pEmitter->m_Random.SetSeed( m_RandomSeed );
            pEmitter->CreateResMatrix();
        }
        pEmitter = pEmitter->m_pNextEmitter;
    }
}


} // namespace vfx
} // namespace nn
