﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/vfx/vfx_Resource.h>
#include <nn/vfx/vfx_Macro.h>
#include <nn/vfx/vfx_EmitterRes.h>
#include <nn/vfx/vfx_ShaderFlag.h>
#include <nn/vfx/vfx_Stripe.h>
#include <nn/vfx/vfx_StripeConnection.h>
#include <nn/vfx/vfx_SuperStripe.h>
#include <nn/vfx/vfx_AreaLoop.h>


namespace nn {
namespace vfx {


//---------------------------------------------------------------------------
//  EmitterResource のコンストラクタ
//---------------------------------------------------------------------------
EmitterResource::EmitterResource() NN_NOEXCEPT
{
    m_IsInitialized = false;
    m_IsUseEmitterAnim = false;
    m_IsEmitterSrtAnim = false;
    m_IsUseField = false;
    m_IsChildEmitter = false;
    m_EmitterIndex = 0;
    m_pResEmitter = NULL;
    m_ResEmitterStaticConstantBuffer = NULL;
    m_pResourceConstantBuffer = NULL;
    m_EmitterStaticBufferOffset = 0;
    m_EmitterPluginConstantBufferOffset = 0;
    m_CustomShaderConstantBufferOffset = 0;
    m_FieldConstantBufferOffset = 0;
    for ( int i = 0; i < TextureSlotId_MaxStandardTextureSlotId; i++ )
    {
        m_TextureSampler[ i ] = NULL;
    }
    m_pPrimitive = NULL;
    m_pVolumePrimitive = NULL;
    m_ChildEmitterResCount = 0;
    for ( int i = 0; i < SystemParameters_MaxEmitterInclusionCount; i++ )
    {
        m_ChildEmitterResSet[ i ] = NULL;
    }
    m_NextEmitterResource = NULL;
    m_IsVisible = true;
    m_pFieldGpuNoiseData = NULL;
    m_pFieldRandomSimpleData = NULL;
    m_pFieldMagnetData = NULL;
    m_pFieldSpinData = NULL;
    m_pFieldCollisionData = NULL;
    m_pFieldConvergenceData = NULL;
    m_pFieldPosAddData = NULL;
    m_pFieldCurlNoiseData = NULL;
    m_pFieldCustomData = NULL;
    for ( int i = 0; i < detail::EmitterAnimationType_MaxAnimationType; i++ )
    {
        m_EmitterAnimationArray[ i ] = NULL;
    }
    m_pCustomShaderParam = NULL;
    m_CustomShaderParamSize = 0;
    m_pCustomActionParam = NULL;
    m_pCustomDataParam = NULL;
    m_EmitterPluginIndex = 0;
    m_pEmitterPluginData = NULL;
    m_pShader[ ShaderType_Normal ] = NULL;
    m_pShader[ ShaderType_DefaultPath1 ] = NULL;
    m_pShader[ ShaderType_DefaultPath2 ] = NULL;
    m_pComputeShader = NULL;
    m_pVertexStateBuffer = NULL;
    nn::util::MatrixIdentity( &m_ResMatrixSrt );
    nn::util::MatrixIdentity( &m_ResMatrixRt );

    for ( int i = 0; i < detail::ParticleFigureAttributeBufferIndex_MaxCount; i++ )
    {
        m_ParticleFigureBufferSlot[ i ] = InvalidValueId_AttributeId;
    }

    for ( int i = 0; i < detail::ParticlePropertyAttributeBufferIndex_MaxIndexCount; i++ )
    {
        m_ParticlePropertyBufferSlot[ i ] = InvalidValueId_AttributeId;
    }
}

//---------------------------------------------------------------------------
//  Gfxリソースを初期化します。
//---------------------------------------------------------------------------
void EmitterResource::InitializeRenderState( nn::gfx::Device* pDevice ) NN_NOEXCEPT
{
    m_RenderState.Initialize( pDevice, &m_pResEmitter->renderState );
}


//---------------------------------------------------------------------------
//  Gfxリソースを破棄します。
//---------------------------------------------------------------------------
void EmitterResource::FinalizeRenderState( nn::gfx::Device* pDevice ) NN_NOEXCEPT
{
    m_RenderState.Finalize( pDevice );
}

//---------------------------------------------------------------------------
//  コンスタントバッファを初期化します。
//---------------------------------------------------------------------------
bool EmitterResource::InitializeConstantBuffer( detail::ConstantBuffer* pConstantBuffer ) NN_NOEXCEPT
{
    m_pResourceConstantBuffer = pConstantBuffer;

    // エミッタ静的定数バッファを初期化
    m_EmitterStaticBufferOffset = m_pResourceConstantBuffer->GetCurrentOffset();
    m_pResourceConstantBuffer->Cut( sizeof( detail::ResCommon ) + sizeof( detail::EmitterStaticUniformBlock ) );

    // エミッタプラグイン定数バッファを初期化
    if ( m_pEmitterPluginData )
    {
        m_EmitterPluginConstantBufferOffset = m_pResourceConstantBuffer->GetCurrentOffset();
        m_pResourceConstantBuffer->Cut( sizeof( nn::util::Float4 ) * 8 );
    }

    // カスタムシェーダバッファを初期化
    if ( m_pCustomShaderParam )
    {
        m_CustomShaderConstantBufferOffset = m_pResourceConstantBuffer->GetCurrentOffset();
        m_pResourceConstantBuffer->Cut( m_CustomShaderParamSize );
    }

    if ( m_IsUseField )
    {
        m_FieldConstantBufferOffset = m_pResourceConstantBuffer->GetCurrentOffset();
        m_pResourceConstantBuffer->Cut( sizeof( detail::EmitterFieldConstantBuffer ) );
    }

    NN_SDK_ASSERT( static_cast<size_t>( m_pResourceConstantBuffer->GetCurrentOffset() ) <= m_pResourceConstantBuffer->GetBufferSize() );

    return true;
}

//---------------------------------------------------------------------------
//  頂点ステートを初期化します。
//---------------------------------------------------------------------------
bool EmitterResource::InitializeVertexState( nn::gfx::Device* pDevice ) NN_NOEXCEPT
{
    if ( !m_pShader[ ShaderType_Normal ] )
    {
        return false;
    }

    int idxElement = 0;
    int figureElement = 0;
    int propertyElement = 0;
    int emitterPluginElement = 0;
    const nn::gfx::Shader* pGfxShader = m_pShader[ ShaderType_Normal ]->GetShader();
    nn::gfx::VertexAttributeStateInfo vertexAttribs[ 16 ];

    const char* figureAttribName[] = { "sysPosAttr", "sysNormalAttr", "sysTangentAttr", "sysVertexColor0Attr", "sysTexCoordAttr" };

    const int figureAttribCount = sizeof( figureAttribName ) / sizeof( char* );
    const char* propertyAttribName[] = { "sysLocalPosAttr", "sysLocalVecAttr", "sysLocalDiffAttr", "sysScaleAttr", "sysRandomAttr",
                                         "sysInitRotateAttr", "sysColor0Attr", "sysColor1Attr", "sysEmtMat0Attr", "sysEmtMat1Attr", "sysEmtMat2Attr" };
    const int propertyAttribCount = sizeof( propertyAttribName ) / sizeof( char* );
    const char* emitterPluginAttribName[] = { "sysEmitterPluginAttr0", "sysEmitterPluginAttr1", "sysEmitterPluginAttr2", "sysEmitterPluginAttr3", "sysEmitterPluginAttr4" };
    const int emitterPluginAttribCount = sizeof( emitterPluginAttribName ) / sizeof( char* );

    int slot = -1;
    int figureBufferCount = 0;

    // 入力プリミティブがある場合
    if ( m_pPrimitive )
    {
        for ( int i = 0; i < figureAttribCount; i++ )
        {
            slot = pGfxShader->GetInterfaceSlot( nn::gfx::ShaderStage_Vertex, nn::gfx::ShaderInterfaceType_Input, figureAttribName[ i ] );
            if ( slot != -1 )
            {
                vertexAttribs[ figureElement ].SetDefault();
                vertexAttribs[ figureElement ].SetNamePtr( figureAttribName[ i ] );
                vertexAttribs[ figureElement ].SetBufferIndex( m_pPrimitive->GetBufferIndex( static_cast<nn::vfx::Primitive::PrimitiveDataElementIndex>( i ) ) );
                vertexAttribs[ figureElement ].SetFormat( m_pPrimitive->GetFormat( static_cast<nn::vfx::Primitive::PrimitiveDataElementIndex>( i ) ) );
                vertexAttribs[ figureElement ].SetOffset( m_pPrimitive->GetOffset( static_cast<nn::vfx::Primitive::PrimitiveDataElementIndex>( i ) ) );
                idxElement++;
                figureElement++;
            }
        }

        figureBufferCount = m_pPrimitive->GetBufferCount();
        idxElement = figureBufferCount;
    }

    // パーティクルプロパティ
    propertyElement = 0;
    for ( int i = 0; i < propertyAttribCount; i++ )
    {
        slot = pGfxShader->GetInterfaceSlot( nn::gfx::ShaderStage_Vertex, nn::gfx::ShaderInterfaceType_Input, propertyAttribName[ i ] );
        if ( slot != -1 )
        {
            vertexAttribs[ figureElement + propertyElement ].SetDefault();
            vertexAttribs[ figureElement + propertyElement ].SetNamePtr( propertyAttribName[ i ] );
            vertexAttribs[ figureElement + propertyElement ].SetBufferIndex( idxElement );
            vertexAttribs[ figureElement + propertyElement ].SetFormat( nn::gfx::AttributeFormat_32_32_32_32_Float );
            vertexAttribs[ figureElement + propertyElement ].SetOffset( 0 );

            m_ParticlePropertyBufferSlot[ i ] = idxElement;

            propertyElement++;
            idxElement++;
        }
        else
        {
            m_ParticlePropertyBufferSlot[ i ] = InvalidValueId_AttributeId;
        }
    }

    // エミッタプラグイン( ストライプ等 )
    if ( figureElement == 0 && propertyElement == 0 )
    {
        for ( int i = 0; i < emitterPluginAttribCount; i++ )
        {
            slot = pGfxShader->GetInterfaceSlot( nn::gfx::ShaderStage_Vertex, nn::gfx::ShaderInterfaceType_Input, emitterPluginAttribName[ i ] );
            vertexAttribs[ idxElement ].SetDefault();
            if ( slot != -1 )
            {
                vertexAttribs[ idxElement ].SetNamePtr( emitterPluginAttribName[ i ] );
                vertexAttribs[ idxElement ].SetBufferIndex( 0 );
                vertexAttribs[ idxElement ].SetFormat( nn::gfx::AttributeFormat_32_32_32_32_Float );
                vertexAttribs[ idxElement ].SetOffset( i * sizeof( nn::util::Float4 ) );
                idxElement++;
                emitterPluginElement++;
            }
        }

        m_ParticlePropertyBufferSlot[ detail::ParticlePropertyAttributeBufferIndex_EmitterPlugin ] = 0;
    }


    // 頂点バッファ情報を設定
    nn::gfx::VertexBufferStateInfo bufferStateInfo[ 16 ];

    // パーティクル形状
    bufferStateInfo[ 0 ].SetDefault();
    if ( figureElement != 0 )
    {
        for ( int i = 0; i < figureBufferCount; i++ )
        {
            bufferStateInfo[ i ].SetStride( m_pPrimitive->GetStrideSize( i ) );
        }
    }


    if ( propertyElement != 0 )
    {
        for ( int i = 0; i < propertyElement; i++ )
        {
            bufferStateInfo[ i + figureBufferCount ].SetDefault();
            bufferStateInfo[ i + figureBufferCount ].SetStride( sizeof( nn::util::Float4 ) );
            bufferStateInfo[ i + figureBufferCount ].SetDivisor( 1 );
        }
    }

    int bufferStateCount = 0;
    if ( emitterPluginElement != 0 )
    {
        bufferStateInfo[ 0 ].SetStride( detail::EmitterPluginAttributeType_MaxAttributeCount * sizeof( nn::util::Float4 ) );
        bufferStateCount++;
    }

    // 頂点属性と頂点バッファ情報を頂点ステートに設定
    nn::gfx::VertexState::InfoType info;
    info.SetDefault();
    info.SetVertexAttributeStateInfoArray( vertexAttribs, figureElement + propertyElement + emitterPluginElement );
    info.SetVertexBufferStateInfoArray( bufferStateInfo, figureBufferCount + propertyElement + bufferStateCount );

    size_t memorySize = nn::gfx::VertexState::GetRequiredMemorySize( info );
    NN_SDK_ASSERT_NOT_NULL( m_pVertexStateBuffer );
    NN_SDK_ASSERT_ALIGNED( m_pVertexStateBuffer, nn::gfx::VertexState::RequiredMemoryInfo_Alignment );
    NN_SDK_ASSERT( memorySize <= m_VertexStateBufferSize );
    m_VertexState.SetMemory( m_pVertexStateBuffer, memorySize );
    m_VertexState.Initialize( pDevice, info, pGfxShader );

    return true;
}// NOLINT(readability/fn_size)

//---------------------------------------------------------------------------
//  頂点ステートを破棄します。
//---------------------------------------------------------------------------
void EmitterResource::FinalizeVertexState( nn::gfx::Device* pDevice ) NN_NOEXCEPT
{
    if ( m_IsInitialized )
    {
        m_VertexState.Finalize( pDevice );
        m_pVertexStateBuffer = nullptr;
    }
}

//---------------------------------------------------------------------------
//  エミッタリソースのセットアップを行います。
//---------------------------------------------------------------------------
bool EmitterResource::Setup( nn::vfx::Resource* pResource ) NN_NOEXCEPT
{
    if ( m_IsInitialized ) return true;

    nn::gfx::Device* pDevice = pResource->GetGfxDevice();

    detail::ShaderManager*        pShaderManager           = pResource->GetShaderManager();
    detail::ComputeShaderManager* pComputeShaderManager    = pResource->GetComputeShaderManager();

    // シェーダリソースが未セットアップ
    if ( !pShaderManager->IsInitialized() || !pComputeShaderManager->IsInitialized() )
    {
        return false;
    }

    // G3dプリミティブリソースをindexで検索して設定(ストライプを除く)
    if ( m_pResEmitter->ptcl.primitiveIndex != static_cast< uint64_t >( InvalidValueId_PrimitiveIndex ) &&
        ( m_EmitterPluginIndex == detail::EmitterPluginId_None || detail::EmitterPluginId_3 < m_EmitterPluginIndex ) )
    {
        NN_SDK_ASSERT_EQUAL( m_pPrimitive, nullptr );

        // 描画形状は、g3dプリミティブから情報を取得
        m_pPrimitive = pResource->GetG3dPrimitive( m_pResEmitter->ptcl.primitiveIndex );
    }

    // このタイミングでシェーダとプリミティブがセットアップ済みであれば、頂点ステートを作成する
    bool ret = UpdateShaderResource( pShaderManager, pComputeShaderManager );
    if ( !ret )
    {
        detail::OutputWarning( "ShaderBinary Bind Error.\n" );
        return false;
    }

    // 頂点ステートを初期化する( プリミティブとシェーダのセットアップ済みが必須 )
    if ( !InitializeVertexState( pDevice ) )
    {
        return false;
    }

    m_IsInitialized = true;

    return true;
}


//---------------------------------------------------------------------------
//  シェーダリソースを更新します。。
//---------------------------------------------------------------------------
bool EmitterResource::UpdateShaderResource( detail::ShaderManager* shaderManager, detail::ComputeShaderManager* computeShaderManager ) NN_NOEXCEPT
{
    m_pShader[ ShaderType_DefaultPath1 ] = nullptr;
    m_pShader[ ShaderType_DefaultPath2 ] = nullptr;

    m_pShader[ ShaderType_Normal ] = shaderManager->GetShader( m_pResEmitter->shader.shaderIndex );
    NN_SDK_ASSERT_NOT_NULL( m_pShader[ ShaderType_Normal ] );

    if ( m_pResEmitter->shader.userShaderIndex1 != static_cast< uint32_t >( InvalidValueId_ShaderId ) )
    {
        m_pShader[ ShaderType_DefaultPath1 ] = shaderManager->GetShader( m_pResEmitter->shader.userShaderIndex1 );
        if ( !m_pShader[ ShaderType_DefaultPath1 ] ) return false;
    }
    if ( m_pResEmitter->shader.userShaderIndex2 != static_cast< uint32_t >( InvalidValueId_ShaderId ) )
    {
        m_pShader[ ShaderType_DefaultPath2 ] = shaderManager->GetShader( m_pResEmitter->shader.userShaderIndex2 );
        if ( !m_pShader[ ShaderType_DefaultPath2 ] ) return false;
    }

    if ( m_pResEmitter->shader.computeShaderIndex != static_cast< uint32_t >( InvalidValueId_ShaderId ) )
    {
        m_pComputeShader = computeShaderManager->GetShader( m_pResEmitter->shader.computeShaderIndex );
    }

    return true;
}

//---------------------------------------------------------------------------
//  emitterDataに関連するパラメータを更新します。
//---------------------------------------------------------------------------
void EmitterResource::UpdateParams( nn::vfx::Resource* pResource ) NN_NOEXCEPT
{
    // テクスチャサンプラ―を取得
    if ( m_pResEmitter->textureSampler0.guid != static_cast< uint64_t >( InvalidValueId_TextureId ) )
    {
        m_TextureSampler[ TextureSlotId_0 ] = detail::TextureSampler::GetSamplerFromTable( &m_pResEmitter->textureSampler0 );
        NN_SDK_ASSERT_NOT_NULL( m_TextureSampler[ TextureSlotId_0 ] );
    }
    if ( m_pResEmitter->textureSampler1.guid != static_cast< uint64_t >( InvalidValueId_TextureId ) )
    {
        m_TextureSampler[ TextureSlotId_1 ] = detail::TextureSampler::GetSamplerFromTable( &m_pResEmitter->textureSampler1 );
        NN_SDK_ASSERT_NOT_NULL( m_TextureSampler[ TextureSlotId_1 ] );
    }
    if ( m_pResEmitter->textureSampler2.guid != static_cast< uint64_t >( InvalidValueId_TextureId ) )
    {
        m_TextureSampler[ TextureSlotId_2 ] = detail::TextureSampler::GetSamplerFromTable( &m_pResEmitter->textureSampler2 );
        NN_SDK_ASSERT_NOT_NULL( m_TextureSampler[ TextureSlotId_2 ] );
    }

    // 暫定処置
    // 定数バッファに FIXED Type の場合は、カラー/アルファ値を書き込む
    if ( m_pResEmitter->ptclColor.color0Type == detail::ParticleColorCalculationMode_FixedColor )
    {
        detail::Float3Copy( &m_pResEmitter->staticUbo.color0Anim.value[ 0 ], m_pResEmitter->ptclColor.color0 );
    }
    if ( m_pResEmitter->ptclColor.alpha0Type == detail::ParticleColorCalculationMode_FixedColor )
    {
        m_pResEmitter->staticUbo.alpha0Anim.value[ 0 ].x = m_pResEmitter->ptclColor.alpha0;
    }
    if ( m_pResEmitter->ptclColor.color1Type == detail::ParticleColorCalculationMode_FixedColor )
    {
        detail::Float3Copy( &m_pResEmitter->staticUbo.color1Anim.value[ 0 ], m_pResEmitter->ptclColor.color1 );
    }
    if ( m_pResEmitter->ptclColor.alpha1Type == detail::ParticleColorCalculationMode_FixedColor )
    {
        m_pResEmitter->staticUbo.alpha1Anim.value[ 0 ].x = m_pResEmitter->ptclColor.alpha1;
    }

    // 暫定処置
    // 繰り返し数からテクスチャスケールを設定する
    switch ( m_pResEmitter->textureAnim0.repeat )
    {
    case 0:
        m_pResEmitter->staticUbo.texShiftAnim0.uvScale.x = 1.0f;
        m_pResEmitter->staticUbo.texShiftAnim0.uvScale.y = 1.0f;
        break;
    case 1:
        m_pResEmitter->staticUbo.texShiftAnim0.uvScale.x = 2.0f;
        m_pResEmitter->staticUbo.texShiftAnim0.uvScale.y = 1.0f;
        break;
    case 2:
        m_pResEmitter->staticUbo.texShiftAnim0.uvScale.x = 1.0f;
        m_pResEmitter->staticUbo.texShiftAnim0.uvScale.y = 2.0f;
        break;
    case 3:
        m_pResEmitter->staticUbo.texShiftAnim0.uvScale.x = 2.0f;
        m_pResEmitter->staticUbo.texShiftAnim0.uvScale.y = 2.0f;
        break;
    default:
        break;
    }
    switch ( m_pResEmitter->textureAnim1.repeat )
    {
    case 0:
        m_pResEmitter->staticUbo.texShiftAnim1.uvScale.x = 1.0f;
        m_pResEmitter->staticUbo.texShiftAnim1.uvScale.y = 1.0f;
        break;
    case 1:
        m_pResEmitter->staticUbo.texShiftAnim1.uvScale.x = 2.0f;
        m_pResEmitter->staticUbo.texShiftAnim1.uvScale.y = 1.0f;
        break;
    case 2:
        m_pResEmitter->staticUbo.texShiftAnim1.uvScale.x = 1.0f;
        m_pResEmitter->staticUbo.texShiftAnim1.uvScale.y = 2.0f;
        break;
    case 3:
        m_pResEmitter->staticUbo.texShiftAnim1.uvScale.x = 2.0f;
        m_pResEmitter->staticUbo.texShiftAnim1.uvScale.y = 2.0f;
        break;
    default:
        break;
    }
    switch ( m_pResEmitter->textureAnim2.repeat )
    {
    case 0:
        m_pResEmitter->staticUbo.texShiftAnim2.uvScale.x = 1.0f;
        m_pResEmitter->staticUbo.texShiftAnim2.uvScale.y = 1.0f;
        break;
    case 1:
        m_pResEmitter->staticUbo.texShiftAnim2.uvScale.x = 2.0f;
        m_pResEmitter->staticUbo.texShiftAnim2.uvScale.y = 1.0f;
        break;
    case 2:
        m_pResEmitter->staticUbo.texShiftAnim2.uvScale.x = 1.0f;
        m_pResEmitter->staticUbo.texShiftAnim2.uvScale.y = 2.0f;
        break;
    case 3:
        m_pResEmitter->staticUbo.texShiftAnim2.uvScale.x = 2.0f;
        m_pResEmitter->staticUbo.texShiftAnim2.uvScale.y = 2.0f;
        break;
    default:
        break;
    }

    // UVスクロール値のキャンセル
    if ( !m_pResEmitter->textureAnim0.isScroll )
    {
        m_pResEmitter->staticUbo.texShiftAnim0.scrollInit.x = 0.0f;
        m_pResEmitter->staticUbo.texShiftAnim0.scrollInit.y = 0.0f;
        m_pResEmitter->staticUbo.texShiftAnim0.scrollAdd.x = 0.0f;
        m_pResEmitter->staticUbo.texShiftAnim0.scrollAdd.y = 0.0f;
        m_pResEmitter->staticUbo.texShiftAnim0.scrollInitRand.x = 0.0f;
        m_pResEmitter->staticUbo.texShiftAnim0.scrollInitRand.y = 0.0f;
    }
    if ( !m_pResEmitter->textureAnim1.isScroll )
    {
        m_pResEmitter->staticUbo.texShiftAnim1.scrollInit.x = 0.0f;
        m_pResEmitter->staticUbo.texShiftAnim1.scrollInit.y = 0.0f;
        m_pResEmitter->staticUbo.texShiftAnim1.scrollAdd.x = 0.0f;
        m_pResEmitter->staticUbo.texShiftAnim1.scrollAdd.y = 0.0f;
        m_pResEmitter->staticUbo.texShiftAnim1.scrollInitRand.x = 0.0f;
        m_pResEmitter->staticUbo.texShiftAnim1.scrollInitRand.y = 0.0f;
    }
    if ( !m_pResEmitter->textureAnim2.isScroll )
    {
        m_pResEmitter->staticUbo.texShiftAnim2.scrollInit.x = 0.0f;
        m_pResEmitter->staticUbo.texShiftAnim2.scrollInit.y = 0.0f;
        m_pResEmitter->staticUbo.texShiftAnim2.scrollAdd.x = 0.0f;
        m_pResEmitter->staticUbo.texShiftAnim2.scrollAdd.y = 0.0f;
        m_pResEmitter->staticUbo.texShiftAnim2.scrollInitRand.x = 0.0f;
        m_pResEmitter->staticUbo.texShiftAnim2.scrollInitRand.y = 0.0f;
    }

    if ( !m_pResEmitter->textureAnim0.isRotate )
    {
        m_pResEmitter->staticUbo.texShiftAnim0.rotAdd = 0.0f;
        m_pResEmitter->staticUbo.texShiftAnim0.rotInit = 0.0f;
        m_pResEmitter->staticUbo.texShiftAnim0.rotInitRand = 0.0f;
    }
    if ( !m_pResEmitter->textureAnim1.isRotate )
    {
        m_pResEmitter->staticUbo.texShiftAnim1.rotAdd = 0.0f;
        m_pResEmitter->staticUbo.texShiftAnim1.rotInit = 0.0f;
        m_pResEmitter->staticUbo.texShiftAnim1.rotInitRand = 0.0f;
    }
    if ( !m_pResEmitter->textureAnim2.isRotate )
    {
        m_pResEmitter->staticUbo.texShiftAnim2.rotAdd = 0.0f;
        m_pResEmitter->staticUbo.texShiftAnim2.rotInit = 0.0f;
        m_pResEmitter->staticUbo.texShiftAnim2.rotInitRand = 0.0f;
    }

    if ( !m_pResEmitter->textureAnim0.isScale )
    {
        m_pResEmitter->staticUbo.texShiftAnim0.scaleInit.x = 1.0f;
        m_pResEmitter->staticUbo.texShiftAnim0.scaleInit.y = 1.0f;
        m_pResEmitter->staticUbo.texShiftAnim0.scaleAdd.x = 0.0f;
        m_pResEmitter->staticUbo.texShiftAnim0.scaleAdd.y = 0.0f;
        m_pResEmitter->staticUbo.texShiftAnim0.scaleInitRand.x = 0.0f;
        m_pResEmitter->staticUbo.texShiftAnim0.scaleInitRand.y = 0.0f;
    }
    if ( !m_pResEmitter->textureAnim1.isScale )
    {
        m_pResEmitter->staticUbo.texShiftAnim1.scaleInit.x = 1.0f;
        m_pResEmitter->staticUbo.texShiftAnim1.scaleInit.y = 1.0f;
        m_pResEmitter->staticUbo.texShiftAnim1.scaleAdd.x = 0.0f;
        m_pResEmitter->staticUbo.texShiftAnim1.scaleAdd.y = 0.0f;
        m_pResEmitter->staticUbo.texShiftAnim1.scaleInitRand.x = 0.0f;
        m_pResEmitter->staticUbo.texShiftAnim1.scaleInitRand.y = 0.0f;
    }
    if ( !m_pResEmitter->textureAnim2.isScale )
    {
        m_pResEmitter->staticUbo.texShiftAnim2.scaleInit.x = 1.0f;
        m_pResEmitter->staticUbo.texShiftAnim2.scaleInit.y = 1.0f;
        m_pResEmitter->staticUbo.texShiftAnim2.scaleAdd.x = 0.0f;
        m_pResEmitter->staticUbo.texShiftAnim2.scaleAdd.y = 0.0f;
        m_pResEmitter->staticUbo.texShiftAnim2.scaleInitRand.x = 0.0f;
        m_pResEmitter->staticUbo.texShiftAnim2.scaleInitRand.y = 0.0f;
    }

    if ( !m_pResEmitter->ptcl.isRotateX )
    {
        m_pResEmitter->staticUbo.rotateInit.x = 0.0f;
        m_pResEmitter->staticUbo.rotateInitRand.x = 0.0f;
        m_pResEmitter->staticUbo.rotateAdd.x = 0.0f;
        m_pResEmitter->staticUbo.rotateAddRand.x = 0.0f;
    }
    if ( !m_pResEmitter->ptcl.isRotateY )
    {
        m_pResEmitter->staticUbo.rotateInit.y = 0.0f;
        m_pResEmitter->staticUbo.rotateInitRand.y = 0.0f;
        m_pResEmitter->staticUbo.rotateAdd.y = 0.0f;
        m_pResEmitter->staticUbo.rotateAddRand.y = 0.0f;
    }
    if ( !m_pResEmitter->ptcl.isRotateZ )
    {
        m_pResEmitter->staticUbo.rotateInit.z = 0.0f;
        m_pResEmitter->staticUbo.rotateInitRand.z = 0.0f;
        m_pResEmitter->staticUbo.rotateAdd.z = 0.0f;
        m_pResEmitter->staticUbo.rotateAddRand.z = 0.0f;
    }


    // テクスチャパタン サーフェイス
    // 2Dアレイテクスチャの配列数をここで取得
    if ( m_pResEmitter->textureAnim0.patternAnimType == detail::TexturePatternAnimationType_Surface )
    {
        const nn::gfx::ResTexture* resTexture = pResource->SearchResTexture( m_pResEmitter->textureSampler0.guid );
        if ( resTexture )
        {
            m_pResEmitter->staticUbo.texPtnAnim0.ptnTableNum = static_cast<float>( resTexture->GetTextureInfo()->GetArrayLength() );
        }
    }
    if ( m_pResEmitter->textureAnim1.patternAnimType == detail::TexturePatternAnimationType_Surface )
    {
        const nn::gfx::ResTexture* resTexture = pResource->SearchResTexture( m_pResEmitter->textureSampler1.guid );
        if ( resTexture )
        {
            m_pResEmitter->staticUbo.texPtnAnim1.ptnTableNum = static_cast<float>( resTexture->GetTextureInfo()->GetArrayLength() );
        }
    }
    if ( m_pResEmitter->textureAnim2.patternAnimType == detail::TexturePatternAnimationType_Surface )
    {
        const nn::gfx::ResTexture* resTexture = pResource->SearchResTexture( m_pResEmitter->textureSampler2.guid );
        if ( resTexture )
        {
            m_pResEmitter->staticUbo.texPtnAnim2.ptnTableNum = static_cast<float>( resTexture->GetTextureInfo()->GetArrayLength() );
        }
    }

    // 8キーアニメーション、未使用キーを最終キーの値でクランプ
    if ( m_pResEmitter->staticUbo.color0AnimKeynum > 0 )
    {
        for ( int i = m_pResEmitter->staticUbo.color0AnimKeynum; i < 8; i++ )
        {
            m_pResEmitter->staticUbo.color0Anim.value[ i ] = m_pResEmitter->staticUbo.color0Anim.value[ m_pResEmitter->staticUbo.color0AnimKeynum - 1 ];
            m_pResEmitter->staticUbo.color0Anim.value[ i ].w += i;
        }
    }
    if ( m_pResEmitter->staticUbo.color1AnimKeynum > 0 )
    {
        for ( int i = m_pResEmitter->staticUbo.color1AnimKeynum; i < 8; i++ )
        {
            m_pResEmitter->staticUbo.color1Anim.value[ i ] = m_pResEmitter->staticUbo.color1Anim.value[ m_pResEmitter->staticUbo.color1AnimKeynum - 1 ];
            m_pResEmitter->staticUbo.color1Anim.value[ i ].w += i;
        }
    }
    if ( m_pResEmitter->staticUbo.alpha0AnimKeyNum > 0 )
    {
        for ( int i = m_pResEmitter->staticUbo.alpha0AnimKeyNum; i < 8; i++ )
        {
            m_pResEmitter->staticUbo.alpha0Anim.value[ i ] = m_pResEmitter->staticUbo.alpha0Anim.value[ m_pResEmitter->staticUbo.alpha0AnimKeyNum - 1 ];
            m_pResEmitter->staticUbo.alpha0Anim.value[ i ].w += i;
        }
    }
    if ( m_pResEmitter->staticUbo.alpha1AnimKeyNum > 0 )
    {
        for ( int i = m_pResEmitter->staticUbo.alpha1AnimKeyNum; i < 8; i++ )
        {
            m_pResEmitter->staticUbo.alpha1Anim.value[ i ] = m_pResEmitter->staticUbo.alpha1Anim.value[ m_pResEmitter->staticUbo.alpha1AnimKeyNum - 1 ];
            m_pResEmitter->staticUbo.alpha1Anim.value[ i ].w += i;
        }
    }
    if ( m_pResEmitter->staticUbo.scaleAnimKeyNum > 0 )
    {
        for ( int i = m_pResEmitter->staticUbo.scaleAnimKeyNum; i < 8; i++ )
        {
            m_pResEmitter->staticUbo.scale8keyAnim.value[ i ] = m_pResEmitter->staticUbo.scale8keyAnim.value[ m_pResEmitter->staticUbo.scaleAnimKeyNum - 1 ];
            m_pResEmitter->staticUbo.scale8keyAnim.value[ i ].w += i;
        }
    }
    if ( m_pResEmitter->staticUbo.shaderAnimKeyNum > 0 )
    {
        for ( int i = m_pResEmitter->staticUbo.shaderAnimKeyNum; i < 8; i++ )
        {
            m_pResEmitter->staticUbo.shaderAnim.value[ i ] = m_pResEmitter->staticUbo.shaderAnim.value[ m_pResEmitter->staticUbo.shaderAnimKeyNum - 1 ];
            m_pResEmitter->staticUbo.shaderAnim.value[ i ].w += i;
        }
    }

    // 8キーアニメーションのループ対応
    if ( m_pResEmitter->ptcl.color0AnimIsLoop )
    {
        m_pResEmitter->staticUbo.color0AnimLoopRate = static_cast< float >( m_pResEmitter->ptcl.color0AnimLoopRate );
    }
    else
    {
        m_pResEmitter->staticUbo.color0AnimLoopRate = 0.0f;
    }
    m_pResEmitter->staticUbo.color0AnimIsLoopInitRandom = ( m_pResEmitter->ptcl.color0AnimIsLoopInitRandom ) ? 1.0f : 0.0f;

    //
    if ( m_pResEmitter->ptcl.alpha0AnimIsLoop )
    {
        m_pResEmitter->staticUbo.alpha0AnimLoopRate = static_cast< float >( m_pResEmitter->ptcl.alpha0AnimLoopRate );
    }
    else
    {
        m_pResEmitter->staticUbo.alpha0AnimLoopRate = 0.0f;
    }
    m_pResEmitter->staticUbo.alpha0AnimIsLoopInitRandom = ( m_pResEmitter->ptcl.alpha0AnimIsLoopInitRandom ) ? 1.0f : 0.0f;

    //
    if ( m_pResEmitter->ptcl.color1AnimIsLoop )
    {
        m_pResEmitter->staticUbo.color1AnimLoopRate = static_cast< float >( m_pResEmitter->ptcl.color1AnimLoopRate );
    }
    else
    {
        m_pResEmitter->staticUbo.color1AnimLoopRate = 0.0f;
    }
    m_pResEmitter->staticUbo.color1AnimIsLoopInitRandom = ( m_pResEmitter->ptcl.color1AnimIsLoopInitRandom ) ? 1.0f : 0.0f;

    //
    if ( m_pResEmitter->ptcl.alpha1AnimIsLoop )
    {
        m_pResEmitter->staticUbo.alpha1AnimLoopRate = static_cast< float >( m_pResEmitter->ptcl.alpha1AnimLoopRate );
    }
    else
    {
        m_pResEmitter->staticUbo.alpha1AnimLoopRate = 0.0f;
    }
    m_pResEmitter->staticUbo.alpha1AnimIsLoopInitRandom = ( m_pResEmitter->ptcl.alpha1AnimIsLoopInitRandom ) ? 1.0f : 0.0f;

    //
    if ( m_pResEmitter->ptcl.scaleAnimIsLoop )
    {
        m_pResEmitter->staticUbo.scaleAnimLoopRate = static_cast< float >( m_pResEmitter->ptcl.scaleAnimLoopRate );
    }
    else
    {
        m_pResEmitter->staticUbo.scaleAnimLoopRate = 0.0f;
    }
    m_pResEmitter->staticUbo.scaleAnimIsLoopInitRandom = ( m_pResEmitter->ptcl.scaleAnimIsLoopInitRandom ) ? 1.0f : 0.0f;


    // 暫定処置:
    // プリミティブ型のエミッタ形状以外の場合は、放出タイプを無効化する
    if ( !( m_pResEmitter->volume.volumeType == detail::EmitterVolumeType_CircleEquallyDivided ||
        m_pResEmitter->volume.volumeType == detail::EmitterVolumeType_LineEquallyDivided ||
        m_pResEmitter->volume.volumeType == detail::EmitterVolumeType_SphereEquallyDivided ||
        m_pResEmitter->volume.volumeType == detail::EmitterVolumeType_SphereEqually64Divided ||
        m_pResEmitter->volume.volumeType == detail::EmitterVolumeType_Primitive ) )
    {
        m_pResEmitter->volume.primEmitType = static_cast< uint32_t >( InvalidValueId_PrimitiveEmitter );
    }

    //「球(分割)」、「球(分割64)」対応
    if ( m_pResEmitter->volume.volumeType == detail::EmitterVolumeType_SphereEquallyDivided )
    {
        if ( m_pResEmitter->volume.primEmitType == detail::PrimitiveEmissionType_Unison )
        {
            // UI上のIndex値から放出レートにマッピング。
            const float pattern[] =
            {
                2, 3, 4, 6, 8, 12, 20, 32
            };
            m_pResEmitter->emission.rate = pattern[ m_pResEmitter->volume.volumeTblIndex ];
        }
        else
        {
            // それ以外の場合はMakerからの値をそのまま採用する
        }
    }
    else if ( m_pResEmitter->volume.volumeType == detail::EmitterVolumeType_SphereEqually64Divided )
    {
        if ( m_pResEmitter->volume.primEmitType == detail::PrimitiveEmissionType_Unison )
        {
            // UI上のIndex値を加算して放出レートに合わせる。
            m_pResEmitter->emission.rate = m_pResEmitter->volume.volumeTblIndex64;
        }
        else
        {
            // それ以外の場合はMakerからの値をそのまま採用する
        }
    }
    else if ( m_pResEmitter->volume.volumeType == detail::EmitterVolumeType_Primitive )
    {
        // 入力されたプリミティブの頂点数に合わせる。
        if ( m_pVolumePrimitive )
        {
            // 放出タイプで放出数も切り替える。
            switch ( m_pResEmitter->volume.primEmitType )
            {
            case 0:
                m_pResEmitter->emission.rate = static_cast< float >( m_pVolumePrimitive->GetVertexBufferArrayCount() );
                break;
            default:
                break;
            }
        }
        else
        {
            m_pResEmitter->emission.rate = 1.0;
        }
    }
    else if ( m_pResEmitter->volume.volumeType == detail::EmitterVolumeType_CircleEquallyDivided ||
        m_pResEmitter->volume.volumeType == detail::EmitterVolumeType_LineEquallyDivided )
    {
        if ( m_pResEmitter->volume.primEmitType == detail::PrimitiveEmissionType_Unison )
        {
            // それ以外：円等分割 or 線等分割 で、かつ一斉放出の場合、
            // 放出レートを1で上書き（EffectMaker上での固定がやや高コストなので、こちらでクッションしておく）
            m_pResEmitter->emission.rate = 1.0f;
        }
    }

    // シェーダフラグ
    detail::ShaderFlag shaderFlag;
    shaderFlag.Invalidate();
    shaderFlag.Initialize( this );

    // フラグを書き込む
    m_pResEmitter->staticUbo.flag[ 0 ] = shaderFlag.GetFlag0();
    m_pResEmitter->staticUbo.flag[ 1 ] = shaderFlag.GetFlag1();

    // カスタムフィールドフラグ
    if ( m_pFieldCustomData )
    {
        m_pResEmitter->staticUbo.flag[ 3 ] = m_pFieldCustomData->flags;
    }

    // リソース内エミッタマトリクス
    nn::util::Vector3fType scale;
    nn::util::Vector3fType scaleDef = NN_UTIL_VECTOR_3F_INITIALIZER( 1, 1, 1 );
    nn::util::Vector3fType rotateXyz;
    nn::util::Vector3fType translate;
    nn::util::VectorLoad( &scale, m_pResEmitter->emitter.scale.v );
    nn::util::VectorLoad( &rotateXyz, m_pResEmitter->emitter.rotate.v );
    nn::util::VectorLoad( &translate, m_pResEmitter->emitter.trans.v );

    nn::util::Matrix4x3fType matrixSrt;
    nn::util::Matrix4x3fType matrixRt;
    detail::MatrixCreateSrtXyz( &matrixSrt, scale, rotateXyz, translate );
    detail::MatrixCreateSrtXyz( &matrixRt, scaleDef, rotateXyz, translate );
    nn::util::FloatT4x3 aMatSrt;
    nn::util::FloatT4x3 aMatRt;
    nn::util::MatrixStore( &aMatSrt, matrixSrt );
    nn::util::MatrixStore( &aMatRt, matrixRt );

    m_ResMatrixSrt = matrixSrt;
    m_ResMatrixRt = matrixRt;

    // CPU参照用にコピー
#if defined( NN_BUILD_CONFIG_OS_COS )
    if ( m_pResEmitter->emitter.calcType == detail::EmitterCalculationMode_Cpu )
    {
        memcpy( m_ResEmitterStaticConstantBuffer, &m_pResEmitter->staticUbo, sizeof( detail::EmitterStaticUniformBlock ) );
    }
#endif

    nn::util::VectorLoad( &m_InitialRotate, m_pResEmitter->staticUbo.rotateInit.v );

    //----------------------------------------------
    // コンスタントバッファを更新
    //----------------------------------------------

    // エミッタ静的コンスタントバッファを更新
    {
        void* mappedPtr = m_pResourceConstantBuffer->Map( m_EmitterStaticBufferOffset );
        if ( mappedPtr )
        {
            detail::MemUtil::Copy( mappedPtr, m_pResEmitter, sizeof( detail::ResCommon ) + sizeof( detail::EmitterStaticUniformBlock ) );
            m_pResourceConstantBuffer->Unmap( m_EmitterStaticBufferOffset, sizeof( detail::ResCommon ) + sizeof( detail::EmitterStaticUniformBlock ) );
        }
    }

    // エミッタプラグイン コンスタントバッファを更新
    if ( m_pEmitterPluginData )
    {
        void* mappedPtr = m_pResourceConstantBuffer->Map( m_EmitterPluginConstantBufferOffset );
        if ( mappedPtr )
        {
            memcpy( mappedPtr, m_pEmitterPluginData, sizeof( nn::util::Float4 ) * 8 );
            m_pResourceConstantBuffer->Unmap( m_EmitterPluginConstantBufferOffset, sizeof( nn::util::Float4 ) * 8 );
        }
    }

    // カスタムシェーダ コンスタントバッファを更新
    if ( m_pCustomShaderParam )
    {
        void* mappedPtr = m_pResourceConstantBuffer->Map( m_CustomShaderConstantBufferOffset );
        if ( mappedPtr )
        {
            memcpy( mappedPtr, m_pCustomShaderParam, m_CustomShaderParamSize );
            m_pResourceConstantBuffer->Unmap( m_CustomShaderConstantBufferOffset, m_CustomShaderParamSize );
        }
    }

    // フィールド コンスタントバッファを更新
    if ( m_IsUseField )
    {
        void* mappedPtr = m_pResourceConstantBuffer->Map( m_FieldConstantBufferOffset );
        if ( mappedPtr )
        {
            detail::EmitterFieldConstantBuffer* fieldConstantBuffer = static_cast<detail::EmitterFieldConstantBuffer*>( mappedPtr );

            if ( m_pFieldGpuNoiseData )
            {
                detail::Float3Copy( &fieldConstantBuffer->randomParam, m_pFieldGpuNoiseData->randomVel );
                fieldConstantBuffer->randomParam.w = static_cast< float >( m_pFieldGpuNoiseData->blank );    // interval

                fieldConstantBuffer->randomParam1.x = m_pFieldGpuNoiseData->waveParamHzRate0;
                fieldConstantBuffer->randomParam1.y = m_pFieldGpuNoiseData->waveParamHzRate1;
                fieldConstantBuffer->randomParam1.z = m_pFieldGpuNoiseData->waveParamHzRate2;
                fieldConstantBuffer->randomParam1.w = m_pFieldGpuNoiseData->waveParamHzRate3;

                fieldConstantBuffer->randomParam2.x = m_pFieldGpuNoiseData->waveParam0;
                fieldConstantBuffer->randomParam2.y = m_pFieldGpuNoiseData->waveParam1;
                fieldConstantBuffer->randomParam2.z = m_pFieldGpuNoiseData->waveParam2;
                fieldConstantBuffer->randomParam2.w = m_pFieldGpuNoiseData->waveParam3;

                fieldConstantBuffer->randomParam3.x = static_cast< float >( m_pFieldGpuNoiseData->enableUnifiedPhase );
                fieldConstantBuffer->randomParam3.y = static_cast< float >( m_pFieldGpuNoiseData->enableDetailedOption );
                fieldConstantBuffer->randomParam3.z = static_cast< float >( m_pFieldGpuNoiseData->enableAirRegist );

                fieldConstantBuffer->randomParam4.x = m_pFieldGpuNoiseData->unifiedPhaseSpeed;
                fieldConstantBuffer->randomParam4.y = m_pFieldGpuNoiseData->unifiedPhaseDistribution;
            }
            if ( m_pFieldRandomSimpleData )
            {
                detail::Float3Copy( &fieldConstantBuffer->randomFe1Param, m_pFieldRandomSimpleData->randomVel );
                fieldConstantBuffer->randomFe1Param.w = static_cast< float >( m_pFieldRandomSimpleData->blank );
            }
            if ( m_pFieldPosAddData )
            {
                detail::Float3Copy( &fieldConstantBuffer->posAddParam, m_pFieldPosAddData->fieldPosAdd );
                fieldConstantBuffer->posAddParam.w = static_cast< float >( m_pFieldPosAddData->isFieldPosAddGlobal );
            }
            if ( m_pFieldMagnetData )
            {
                detail::Float3Copy( &fieldConstantBuffer->magnetParam, m_pFieldMagnetData->fieldMagnetPos );
                fieldConstantBuffer->magnetParam.w = m_pFieldMagnetData->fieldMagnetPower;
                fieldConstantBuffer->magnetParam1.x = static_cast< float >(m_pFieldMagnetData->isFollowEmitter);
#ifdef _VFX_NSDK01215_COMPATIBLE
#else
                fieldConstantBuffer->magnetParam1.y = static_cast< float >(m_pFieldMagnetData->fieldMagnetFlagX);
                fieldConstantBuffer->magnetParam1.z = static_cast< float >(m_pFieldMagnetData->fieldMagnetFlagY);
                fieldConstantBuffer->magnetParam1.w = static_cast< float >(m_pFieldMagnetData->fieldMangetFlagZ);
#endif
            }
            if ( m_pFieldConvergenceData )
            {
                detail::Float3Copy( &fieldConstantBuffer->convergenceParam, m_pFieldConvergenceData->fieldConvergencePos );
                fieldConstantBuffer->convergenceParam.w = m_pFieldConvergenceData->fieldConvergenceRatio;
                fieldConstantBuffer->convergenceParam1.x = static_cast< float >( m_pFieldConvergenceData->fieldConvergenceType );
            }
            if ( m_pFieldSpinData )
            {
                fieldConstantBuffer->spinParam.x = m_pFieldSpinData->fieldSpinRotate;
                fieldConstantBuffer->spinParam.y = m_pFieldSpinData->fieldSpinOuter;
                fieldConstantBuffer->spinParam.z = static_cast< float >( m_pFieldSpinData->fieldSpinAxis );
            }
            if ( m_pFieldCollisionData )
            {
                fieldConstantBuffer->collisionParam.x = ( m_pFieldCollisionData->fieldCollisionType * 2 ) + static_cast< float >( m_pFieldCollisionData->fieldCollisionIsWorld );
                fieldConstantBuffer->collisionParam.y = m_pFieldCollisionData->fieldCollisionCoef;
                fieldConstantBuffer->collisionParam.z = m_pFieldCollisionData->fieldCollisionRegist;
                fieldConstantBuffer->collisionParam.w = m_pFieldCollisionData->fieldCollisionCoord;
            }
            if ( m_pFieldCurlNoiseData )
            {
                detail::Float3Copy( &fieldConstantBuffer->curlNoiseParam, m_pFieldCurlNoiseData->fieldCurlNoiseInfluence );
                fieldConstantBuffer->curlNoiseParam.w = m_pFieldCurlNoiseData->fieldCurlNoiseScale;

                detail::Float3Copy( &fieldConstantBuffer->curlNoiseParam1, m_pFieldCurlNoiseData->fieldCurlNoiseSpeed );
                fieldConstantBuffer->curlNoiseParam1.w = m_pFieldCurlNoiseData->fieldCurlNoiseBase;

                fieldConstantBuffer->curlNoiseParam2.x = m_pFieldCurlNoiseData->isFieldCurlNoiseBaseRandom;
                fieldConstantBuffer->curlNoiseParam2.y = m_pFieldCurlNoiseData->isWorldCoordinate;
            }
            if ( m_pFieldCustomData )
            {
                fieldConstantBuffer->customFieldParam0.x = m_pFieldCustomData->value0;
                fieldConstantBuffer->customFieldParam0.y = m_pFieldCustomData->value1;
                fieldConstantBuffer->customFieldParam0.z = m_pFieldCustomData->value2;
                fieldConstantBuffer->customFieldParam0.w = m_pFieldCustomData->value3;

                fieldConstantBuffer->customFieldParam1.x = m_pFieldCustomData->value4;
                fieldConstantBuffer->customFieldParam1.y = m_pFieldCustomData->value5;
                fieldConstantBuffer->customFieldParam1.z = m_pFieldCustomData->value6;
                fieldConstantBuffer->customFieldParam1.w = m_pFieldCustomData->value7;
            }

            m_pResourceConstantBuffer->Unmap( m_FieldConstantBufferOffset, sizeof( detail::EmitterFieldConstantBuffer ) );
        }
    }

}// NOLINT(readability/fn_size)


namespace /* anonymous */ {

//--------------------------------------------------------------------------
//! @brief      バイナリからエミッタ時間アニメーションを取り出します.
//--------------------------------------------------------------------------
nn::vfx::detail::ResAnimEmitterKeyParamSet* GetEmitterAnimationTable( nn::vfx::detail::BinaryData* pData ) NN_NOEXCEPT
{
#if defined( VFX_BI_ENDIAN )
    uint8_t beforeFlipped = pData->flipped;
#endif

    nn::vfx::detail::ResAnimEmitterKeyParamSet* animBin = pData->GetBinaryDataWithFlip<nn::vfx::detail::ResAnimEmitterKeyParamSet>();

#if defined( VFX_BI_ENDIAN )
    if ( pData->endian != SystemEndianType && animBin->keyNum >= 2 && beforeFlipped != pData->flipped )
    {
        // キーが2個以上ある時は2個目以降をここでフリップ(本当は出力コードをいじりたいが……)
        nn::vfx::detail::EndianUtil::FlipArray( animBin->keyNum - 1, animBin->keyValue + 1 );
    }
#endif

    return animBin;
}

} // namespace /* anonymous */

//---------------------------------------------------------------------------
//  エミッタリソースのアドレス解決をします。
//---------------------------------------------------------------------------
void EmitterResource::ResolveBinaryData( nn::vfx::detail::BinaryData* pEmitterBinaryData ) NN_NOEXCEPT
{
    if ( !pEmitterBinaryData->GetSubData() ) return;

    detail::BinaryData* pSubData = pEmitterBinaryData->GetSubData();

    while ( pSubData )
    {
        switch ( pSubData->GetBinaryTag() )
        {
            // フィールド：ランダム（新方式）
        case VFX_MAKE_TAG( 'F', 'R', 'N', 'D' ):
            m_pFieldGpuNoiseData = pSubData->GetBinaryDataWithFlip<detail::ResFieldRandom>();
            m_IsUseField = true;
            break;
            // フィールド：ランダム（FE1互換）
        case VFX_MAKE_TAG( 'F', 'R', 'N', '1' ):
            m_pFieldRandomSimpleData = pSubData->GetBinaryDataWithFlip<detail::ResFieldRandomFe1>();
            m_IsUseField = true;
            break;
            // フィールド：磁力
        case VFX_MAKE_TAG( 'F', 'M', 'A', 'G' ):
            m_pFieldMagnetData = pSubData->GetBinaryDataWithFlip<detail::ResFieldMagnet>();
            m_IsUseField = true;
            break;
            // フィールド：スピン
        case VFX_MAKE_TAG( 'F', 'S', 'P', 'N' ):
            m_pFieldSpinData = pSubData->GetBinaryDataWithFlip<detail::ResFieldSpinData>();
            m_IsUseField = true;
            break;
            // フィールド：シンプルコリジョン
        case VFX_MAKE_TAG( 'F', 'C', 'O', 'L' ):
            m_pFieldCollisionData = pSubData->GetBinaryDataWithFlip<detail::ResFieldCollisionData>();
            m_IsUseField = true;
            break;
            // フィールド：収束
        case VFX_MAKE_TAG( 'F', 'C', 'O', 'V' ):
            m_pFieldConvergenceData = pSubData->GetBinaryDataWithFlip<detail::ResFieldConvergenceData>();
            m_IsUseField = true;
            break;
            // フィールド：カールノイズ
        case VFX_MAKE_TAG( 'F', 'C', 'L', 'N' ):
            m_pFieldCurlNoiseData = pSubData->GetBinaryDataWithFlip<detail::ResFieldCurlNoiseData>();
            m_IsUseField = true;
            break;
            // フィールド：位置に加算
        case VFX_MAKE_TAG( 'F', 'P', 'A', 'D' ):
            m_pFieldPosAddData = pSubData->GetBinaryDataWithFlip<detail::ResFieldPosAddData>();
            m_IsUseField = true;
            break;
            // フィールド：カスタム
        case VFX_MAKE_TAG( 'F', 'C', 'S', 'F' ):
            m_pFieldCustomData = pSubData->GetBinaryDataWithFlip<detail::ResFieldCustom>();
            m_IsUseField = true;
            break;

            // エミッタプラグイン / 連結式ストライプ
        case VFX_MAKE_TAG( 'E', 'P', '0', '1' ):
            m_pEmitterPluginData = pSubData->GetBinaryDataWithFlip<detail::ConnectionStripeSystem::ResourceType>();
            m_EmitterPluginIndex = detail::ConnectionStripeSystem::PluginId;
            break;
            // エミッタプラグイン / 履歴式ストライプ
        case VFX_MAKE_TAG( 'E', 'P', '0', '2' ):
            m_pEmitterPluginData = pSubData->GetBinaryDataWithFlip<detail::StripeSystem::ResourceType>();
            m_EmitterPluginIndex = detail::StripeSystem::PluginId;
            break;
            // エミッタプラグイン / 履歴式ストライプ2
        case VFX_MAKE_TAG( 'E', 'P', '0', '3' ):
            m_pEmitterPluginData = pSubData->GetBinaryDataWithFlip<detail::SuperStripeSystem::ResourceType>();
            m_EmitterPluginIndex = detail::SuperStripeSystem::PluginId;
            break;
            // エミッタプラグイン / 範囲内ループ
        case VFX_MAKE_TAG( 'E', 'P', '0', '4' ):
            m_pEmitterPluginData = pSubData->GetBinaryDataWithFlip<detail::AreaLoopSystem::ResourceType>();
            m_EmitterPluginIndex = detail::AreaLoopSystem::PluginId;
            break;

            // カスタムデータパラメータ
        case VFX_MAKE_TAG( 'C', 'U', 'D', 'P' ):
        {
            m_pCustomDataParam = reinterpret_cast< void* >( pSubData->GetBinaryData() );
            if ( pSubData->IsNeedFlip() )
            {
                uint32_t size = static_cast< uint32_t >( pSubData->GetBinarySize() - 32 );
                {
                    uint32_t* temp = reinterpret_cast< uint32_t* >( m_pCustomDataParam );
                    int count = static_cast< int >( size / 4 );
                    for ( int i = 0; i < count; i++ )
                    {
                        nn::vfx::detail::EndianUtil::Flip( temp );
                        temp++;
                    }
                }
                pSubData->flipped = static_cast< uint8_t >( detail::BinaryDataFlipFlag_Flipped );
            }
        }
        break;

        // カスタムアクションパラメータ
        case VFX_MAKE_TAG( 'C', 'A', 'D', 'P' ):
        {
            m_pCustomActionParam = pSubData->GetBinaryData();
            if ( pSubData->IsNeedFlip() )
            {
                uint32_t size = static_cast< uint32_t >( pSubData->GetBinarySize() - 32 );
                {
                    uint32_t* temp = reinterpret_cast< uint32_t* >( m_pCustomActionParam );
                    int count = static_cast< int >( size / 4 );
                    for ( int i = 0; i < count; i++ )
                    {
                        nn::vfx::detail::EndianUtil::Flip( temp );
                        temp++;
                    }
                }
                pSubData->flipped = static_cast< uint8_t >( detail::BinaryDataFlipFlag_Flipped );
            }
        }
        break;

        // カスタムシェーダパラメータ
        case VFX_MAKE_TAG( 'C', 'S', 'D', 'P' ):
        {
            m_pCustomShaderParam = pSubData->GetBinaryData();
            m_CustomShaderParamSize = pSubData->GetBinarySize() - 32;
        }
        break;

        // エミッタ時間アニメーション : パーティクルスケールアニメーション配列
        case VFX_MAKE_TAG( 'E', 'A', 'S', 'L' ):
        {
            m_EmitterAnimationArray[ detail::EmitterAnimationType_ParticleScale ] = GetEmitterAnimationTable( pSubData );
            m_IsUseEmitterAnim = true;
        }
        break;

        // エミッタ時間アニメーション : エミッタスケールアニメーション配列
        case VFX_MAKE_TAG( 'E', 'A', 'E', 'S' ):
        {
            m_EmitterAnimationArray[ detail::EmitterAnimationType_Scale ] = GetEmitterAnimationTable( pSubData );
            m_IsEmitterSrtAnim = true;
            m_IsUseEmitterAnim = true;
        }
        break;

        // エミッタ時間アニメーション : エミッタ回転アニメーション配列
        case VFX_MAKE_TAG( 'E', 'A', 'E', 'R' ):
        {
            m_EmitterAnimationArray[ detail::EmitterAnimationType_Rotate ] = GetEmitterAnimationTable( pSubData );
            m_IsEmitterSrtAnim = true;
            m_IsUseEmitterAnim = true;
        }
        break;

        // エミッタ時間アニメーション : エミッタ位置アニメーション配列
        case VFX_MAKE_TAG( 'E', 'A', 'E', 'T' ):
        {
            m_EmitterAnimationArray[ detail::EmitterAnimationType_Translate ] = GetEmitterAnimationTable( pSubData );
            m_IsEmitterSrtAnim = true;
            m_IsUseEmitterAnim = true;
        }
        break;

        // エミッタ時間アニメーション : 放出レート
        case VFX_MAKE_TAG( 'E', 'A', 'T', 'R' ):
        {
            m_EmitterAnimationArray[ detail::EmitterAnimationType_EmissionRate ] = GetEmitterAnimationTable( pSubData );
            m_IsUseEmitterAnim = true;
        }
        break;

        // エミッタ時間アニメーション : 初速／全方向初速
        case VFX_MAKE_TAG( 'E', 'A', 'O', 'V' ):
        {
            m_EmitterAnimationArray[ detail::EmitterAnimationType_AllDirectionalVelocity ] = GetEmitterAnimationTable( pSubData );
            m_IsUseEmitterAnim = true;
        }
        break;

        // エミッタ時間アニメーション : 初速／指定方向初速
        case VFX_MAKE_TAG( 'E', 'A', 'D', 'V' ):
        {
            m_EmitterAnimationArray[ detail::EmitterAnimationType_DesignatedDirectionalVelocity ] = GetEmitterAnimationTable( pSubData );
            m_IsUseEmitterAnim = true;
        }
        break;

        // エミッタ時間アニメーション : その他運動設定／重力
        case VFX_MAKE_TAG( 'E', 'A', 'G', 'V' ):
        {
            m_EmitterAnimationArray[ detail::EmitterAnimationType_GravityScale ] = GetEmitterAnimationTable( pSubData );
            m_IsUseEmitterAnim = true;
        }
        break;

        // エミッタ時間アニメーション : パーティクル寿命／寿命
        case VFX_MAKE_TAG( 'E', 'A', 'P', 'L' ):
        {
            m_EmitterAnimationArray[ detail::EmitterAnimationType_ParticleLife ] = GetEmitterAnimationTable( pSubData );
            m_IsUseEmitterAnim = true;
        }
        break;

        // エミッタ時間アニメーション : グローバルカラー0
        case VFX_MAKE_TAG( 'E', 'A', 'C', '0' ):
        {
            m_EmitterAnimationArray[ detail::EmitterAnimationType_Color0 ] = GetEmitterAnimationTable( pSubData );
            m_IsUseEmitterAnim = true;
        }
        break;

        // エミッタ時間アニメーション : グローバルカラー1
        case VFX_MAKE_TAG( 'E', 'A', 'C', '1' ):
        {
            m_EmitterAnimationArray[ detail::EmitterAnimationType_Color1 ] = GetEmitterAnimationTable( pSubData );
            m_IsUseEmitterAnim = true;
        }
        break;

        // エミッタ時間アニメーション : グローバルアルファ0
        case VFX_MAKE_TAG( 'E', 'A', 'A', '0' ):
        {
            m_EmitterAnimationArray[ detail::EmitterAnimationType_Alpha0 ] = GetEmitterAnimationTable( pSubData );
            m_IsUseEmitterAnim = true;
        }
        break;

        // エミッタ時間アニメーション : グローバルアルファ1
        case VFX_MAKE_TAG( 'E', 'A', 'A', '1' ):
        {
            m_EmitterAnimationArray[ detail::EmitterAnimationType_Alpha1 ] = GetEmitterAnimationTable( pSubData );
            m_IsUseEmitterAnim = true;
        }
        break;

        // エミッタ時間アニメーション : エミッタボリュームスケール
        case VFX_MAKE_TAG( 'E', 'A', 'S', 'S' ):
        {
            m_EmitterAnimationArray[ detail::EmitterAnimationType_EmitterVolumeScale ] = GetEmitterAnimationTable( pSubData );
            m_IsUseEmitterAnim = true;
        }
        break;

        default:
            break;
        }
        pSubData = pSubData->GetNextData();
    }
}// NOLINT(readability/fn_size)


} // namespace vfx
} // namespace nn
