﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/ui2d/detail/ui2d_Log.h>
#include <nn/nn_SdkAssert.h>
#if defined(NN_BUILD_CONFIG_OS_WIN)

#include <nn/nn_windows.h>
#include <cstdio>

#include <nn/ui2d/viewer/win/ui2d_ScreenShotWin.h>

#if defined(NN_UI2D_VIEWER_ENABLED)

namespace nn
{
namespace ui2d
{
namespace viewer
{

//------------------------------------------------------------------------------
// オフセットの切り上げ
//------------------------------------------------------------------------------
static ptrdiff_t AlignupOffset(ptrdiff_t offset, size_t alignment)
{
    return ((offset + alignment - 1) / alignment) * alignment;
}

size_t ScreenShot::GetRequiredMemoryPoolSize() const
{
    return m_width * m_height * ColorComponentSize;
}

size_t ScreenShot::GetRequiredMemoryPoolAlignment(nn::gfx::Device* pDevice) const
{
    size_t imageBufferSize = m_width * m_height * ColorComponentSize;

    nn::gfx::Buffer::InfoType info;
    info.SetDefault();
    info.SetSize(imageBufferSize);
    info.SetGpuAccessFlags(nn::gfx::GpuAccess_ColorBuffer);

    return nn::gfx::Buffer::GetBufferAlignment(pDevice, info);
}

void ScreenShot::Prepare(
    nn::gfx::Device* pDevice,
    nn::gfx::MemoryPool* pMemoryPool,
    ptrdiff_t* pMemoryPoolOffset,
    nn::AlignedAllocateFunctionWithUserData pAllocateFunction,
    void* pUserDataForAllocator)
{
    NN_SDK_ASSERT_NOT_NULL(pAllocateFunction);

    if (m_state == State_ReadyToTake)
    {
        NN_DETAIL_UI2D_WARN("already prepared\n");
        return;
    }

    m_pDevice      = pDevice;
    m_pMemoryPool  = pMemoryPool;

    size_t imageBufferSize = m_width * m_height * ColorComponentSize;
    {
        nn::gfx::Buffer::InfoType info;
        info.SetDefault();
        info.SetSize(imageBufferSize);
        info.SetGpuAccessFlags(nn::gfx::GpuAccess_ColorBuffer);

        *pMemoryPoolOffset = AlignupOffset(*pMemoryPoolOffset, nn::gfx::Buffer::GetBufferAlignment(pDevice, info));
        size_t size = info.GetSize();
        m_CaptureBuffer.Initialize(pDevice, info, pMemoryPool, *pMemoryPoolOffset, size);
    }

    m_Buffer = static_cast<uint8_t *>(pAllocateFunction(m_width * m_height * PixelSize, 1, pUserDataForAllocator));
    m_state = State_ReadyToTake;
}

void ScreenShot::PutCommandScreenShot(
    nn::gfx::CommandBuffer* pCommandBuffer, nn::gfx::Texture* pSrcTexture
    )
{
    if(GetState() == ScreenShot::State_ReadyToTake)
    {
        nn::gfx::TextureCopyRegion region;
        region.SetDefault();
        region.SetWidth(m_width);
        region.SetHeight(m_height);
        pCommandBuffer->CopyImageToBuffer(&m_CaptureBuffer, 0, pSrcTexture, region);
    }
}

void ScreenShot::Take(
    nn::FreeFunctionWithUserData pFreeFunction,
    void* pUserDataForAllocator)
{
    NN_SDK_ASSERT_NOT_NULL(pFreeFunction);

    if (m_state != State_ReadyToTake)
    {
        NN_DETAIL_UI2D_WARN("not ready to take\n");
        return;
    }

    // バッファオブジェクトの内容を、メモリにコピーします。
    {
        const uint8_t* pBuffer = static_cast<uint8_t*>(m_CaptureBuffer.Map());
        const ptrdiff_t stride = static_cast<ptrdiff_t>(4) * m_width;

        m_CaptureBuffer.InvalidateMappedRange(0, stride * m_height);

        int ySrc = 0;
        int yDst = 0;
        for(; ySrc < m_height; ySrc++, yDst++)
        {
            const uint8_t* pSrc = pBuffer + stride * ySrc;
            uint8_t* pDst = m_Buffer + stride * yDst;
            memcpy(pDst, pSrc, stride);
        }

        m_CaptureBuffer.Unmap();
    }

    BITMAPFILEHEADER fileHeader;
    memset(&fileHeader, 0, sizeof(fileHeader));
    memcpy(&fileHeader.bfType, "BM", 2);
    fileHeader.bfSize = sizeof(BITMAPFILEHEADER) + sizeof(BITMAPINFOHEADER) + m_width * m_height * PixelSize;
    fileHeader.bfOffBits = sizeof(BITMAPFILEHEADER) + sizeof(BITMAPINFOHEADER);

    BITMAPINFOHEADER infoHeader;
    memset(&infoHeader, 0, sizeof(infoHeader));
    infoHeader.biSize = sizeof(infoHeader);
    infoHeader.biWidth = m_width;
    infoHeader.biHeight = m_height;
    infoHeader.biPlanes = 1;
    infoHeader.biBitCount = PixelSize * 8;
    infoHeader.biCompression = BI_RGB;
    infoHeader.biSizeImage = m_width * m_height * PixelSize;
    infoHeader.biXPelsPerMeter = 3780; // 96dpi
    infoHeader.biYPelsPerMeter = 3780; // 96dpi
    infoHeader.biClrUsed = 0;
    infoHeader.biClrImportant = 0;

    for (int  y = 0; y < m_height; ++y)
    {
        for (int  x = 0; x < m_width; ++x)
        {
            uint8_t * pixel = m_Buffer + (y * m_width + x) * PixelSize;

            // BMPのピクセルは Blue, Green, Red, Reserved(0) の順
            uint8_t  r = pixel[0];
            uint8_t  b = pixel[2];
            pixel[0] = b;
            pixel[2] = r;
            pixel[3] = 0;
        }
    }

    FILE* fp = NULL;

    if (fopen_s(&fp, m_fileName, "wb") == 0)
    {
        fwrite(&fileHeader, sizeof(fileHeader), 1, fp);
        fwrite(&infoHeader, sizeof(infoHeader), 1, fp);
        fwrite(m_Buffer, 1, m_width * m_height * PixelSize, fp);

        fclose(fp);
    }
    else
    {
        NN_SDK_ASSERT(false, "can not create screenshot (%s).", m_fileName);
    }

    pFreeFunction(m_Buffer, pUserDataForAllocator);

    m_Buffer = NULL;
    m_state = State_Done;

    m_CaptureBuffer.Finalize(m_pDevice);
}

} // namespace viewer
} // namespace ui2d
} // namespace nn

#endif // NN_UI2D_VIEWER_ENABLED

#endif // NN_BUILD_CONFIG_OS_WIN
