﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NOMINMAX
#define NOMINMAX
#endif

#include <sstream>

#include <nn/util/util_Arithmetic.h>

#include <nn/nn_Macro.h>
#include <nn/util/util_Matrix.h>
#include <nn/util/util_StringUtil.h>
#include <nn/ui2d/detail/ui2d_Log.h>

#include <nn/ui2d/viewer/ui2d_Viewer.h>
#include <nn/font/font_ScalableFontUtil.h>

#include <nn/ui2d/viewer/ui2d_ToolConnector.h>

#if defined(NN_UI2D_VIEWER_ENABLED)

//--------------------------------------------------------------------------------------------------
// ミドルウェア情報
#include <nn/nn_Middleware.h>
#include <nn/nn_Version.h>

#define NN_DETAIL_XX_MACRO_TO_STRING(x) NN_DETAIL_XX_MACRO_TO_STRING_(x)
#define NN_DETAIL_XX_MACRO_TO_STRING_(x) #x

#define NW_UI2D_VIEWER_MIDDLEWARE_SYMBOL(buildOption) "NintendoWare_Ui2d_Viewer_For_Develop-" NN_DETAIL_XX_MACRO_TO_STRING(NN_NX_ADDON_VERSION_MAJOR) "_" \
NN_DETAIL_XX_MACRO_TO_STRING(NN_NX_ADDON_VERSION_MINOR) "_" NN_DETAIL_XX_MACRO_TO_STRING(NN_NX_ADDON_VERSION_MICRO) "-" #buildOption

namespace {
#if defined(NN_SDK_BUILD_DEBUG)
    NN_DEFINE_MIDDLEWARE(g_MiddlewareInfo, "Nintendo", NW_UI2D_VIEWER_MIDDLEWARE_SYMBOL(Debug));
#elif defined(NN_SDK_BUILD_DEVELOP)
    NN_DEFINE_MIDDLEWARE(g_MiddlewareInfo, "Nintendo", NW_UI2D_VIEWER_MIDDLEWARE_SYMBOL(Develop));
#elif defined(NN_SDK_BUILD_RELEASE)
    NN_DEFINE_MIDDLEWARE(g_MiddlewareInfo, "Nintendo", NW_UI2D_VIEWER_MIDDLEWARE_SYMBOL(Release));
#endif
}

//--------------------------------------------------------------------------------------------------

namespace
{

float StrToFloat_(const char* pStr)
{
    std::stringstream ss;
    ss << pStr;

    float outValue;
    ss >> outValue;

    return outValue;
}

// フォント名からフォントデータを探すコールバック関数です。
void* AcquireFontFunctionForComplexFont(size_t* pOutFontDataSize, const char* pFontName, nn::ui2d::ResType resType, void* pUserData)
{
    nn::ui2d::ResourceAccessor* pResAccsessor = reinterpret_cast<nn::ui2d::ResourceAccessor*>(pUserData);

    void* pFont = pResAccsessor->FindResourceByName(pOutFontDataSize, resType, pFontName);
    NN_SDK_ASSERT(pFont != NULL, "The font resource [%s] was not found.\n", pFontName);
    return pFont;
}

}

namespace nn
{
namespace ui2d
{
namespace viewer
{

static const float DrcWidth = 854.f;
static const float DrcHeight = 480.f;

static const size_t RecvBufferSize = 512;

#if defined(NN_BUILD_CONFIG_OS_WIN)
static const float DrcTouchWidth = 854.f;
static const float DrcTouchHeight = 480.f;
#endif

const Viewer::ViewSizeValue Viewer::ViewSizeValues[ViewSize_MaxViewSize] =
{
    {1280.f, 720.f},
    {1920.f, 1080.f},
    {1280.f, 720.f},
    {854.f, 480.f}
};

const char* Viewer::ViewSizeDescriptions[ViewSize_MaxViewSize] =
{
    "original",
    "FullHD",
    "HD",
    "DRC"
};

Viewer::Viewer()
: m_AnimateOnce(false)
, m_Layout(NULL)
, m_pComplexFontSet(NULL)
, m_ComplexFontCount(0)
, m_pTextureCache(NULL)
#if defined(NN_BUILD_CONFIG_OS_WIN)
, m_ResourceAccessor(*DirResourceAccessor::Create())
#endif
, m_pControlCreator(NULL)
, m_ViewSize(ViewSize_Original)
, m_MainViewportSize(Size::Create(0.0f, 0.0f))
, m_ProjectionMatrix(NN_UTIL_MATRIX_T4X4F_INITIALIZER(1,0,0,0, 0,1,0,0, 0,0,1,0, 0,0,0,1))
, m_CameraViewMatrix(NN_UTIL_MATRIX_T4X3F_INITIALIZER(1,0,0, 0,1,0, 0,0,1, 0,0,0))
, m_ProjectionMatrixDRC(NN_UTIL_MATRIX_T4X4F_INITIALIZER(1,0,0,0, 0,1,0,0, 0,0,1,0, 0,0,0,1))
, m_CameraViewMatrixDRC(NN_UTIL_MATRIX_T4X3F_INITIALIZER(1,0,0, 0,1,0, 0,0,1, 0,0,0))
, m_IsPerspectiveProjection(false)
, m_Fps(Fps_60)
, m_PerspectiveFovyDeg(40.f)
, m_PerspectiveNear(1.f)
, m_PerspectiveFar(10000.f)
, m_IsViewMtxDirty(false)
, m_IsSrgbWriteEnabled(true)
, m_IsHudVisible(true)
, m_LoadFunction(NULL)
, m_UnloadFunction(NULL)
, m_RegisterTextureViewSlot(NULL)
, m_UnregisterTextureViewSlot(NULL)
, m_pUserDataForDescriptorSlotAllocator(NULL)
, m_ScalableFontTextureCacheWidth(0)
, m_ScalableFontTextureCacheHeight(0)
, m_ScalableFontWorkMemorySize(0)
, m_ScalableFontNoPlotWorkMemorySize(0)
, m_GetUserShaderInformationFromUserDataCallback(NULL)
, m_PostCalculateCallback(NULL)
, m_PreReloadLayoutDataCallback(NULL)
, m_PostReloadLayoutDataCallback(NULL)
, m_pBackgroundImageFile(NULL)
, m_BackgroundImageSize(Size::Create(0.0f,0.0f))
{
    NN_USING_MIDDLEWARE(g_MiddlewareInfo);

    m_PreviewOption.reset();
    for (int i = 0; i < 3; ++i)
    {
        // 169 は LayoutEditor のデフォルト背景色。
        m_BackgroundColor[i] = 169.0f / 255.0f;
    }
    m_BackgroundColor[3] = 1.0f;
}

Viewer::~Viewer()
{
#if defined(NN_BUILD_CONFIG_OS_WIN)
    delete &m_ResourceAccessor;
#endif
}

void Viewer::Initialize(const InitializeArg& initializeArg)
{
    NN_SDK_ASSERT_NOT_NULL(initializeArg.pGraphicsResource);
    m_DrawInfo.SetGraphicsResource(initializeArg.pGraphicsResource);

    m_MainViewportSize = initializeArg.mainViewportSize;

    NN_SDK_ASSERT_NOT_NULL(initializeArg.loadFunction);
    NN_SDK_ASSERT_NOT_NULL(initializeArg.unloadFunction);
    m_LoadFunction = initializeArg.loadFunction;
    m_UnloadFunction = initializeArg.unloadFunction;

    NN_SDK_ASSERT_NOT_NULL(initializeArg.registerTextureViewSlot);
    NN_SDK_ASSERT_NOT_NULL(initializeArg.unregisterTextureViewSlot);
    m_RegisterTextureViewSlot = initializeArg.registerTextureViewSlot;
    m_UnregisterTextureViewSlot = initializeArg.unregisterTextureViewSlot;
    m_pUserDataForDescriptorSlotAllocator = initializeArg.pUserDataForDescriptorSlotAllocator;

    m_ScalableFontTextureCacheWidth = initializeArg.scalableFontTextureCacheWidth;
    m_ScalableFontTextureCacheHeight = initializeArg.scalableFontTextureCacheHeight;
    m_ScalableFontWorkMemorySize = initializeArg.scalableFontWorkMemorySize;
    m_ScalableFontNoPlotWorkMemorySize = initializeArg.scalableFontNoPlotWorkMemorySize;

    ToolConnector::Initialize(
        Layout::GetAllocateFunction(),
        Layout::GetFreeFunction(),
        Layout::GetUserDataForAllocator(),
        RecvBufferSize,
        RecvBufferSize,
        false,
        initializeArg.toolConnectorThreadPriority);

#if defined(NN_BUILD_CONFIG_OS_WIN)
    m_ScreenShot.SetSize(static_cast<uint32_t >(initializeArg.mainViewportSize.width), static_cast<uint32_t >(initializeArg.mainViewportSize.height));
#endif
}

void Viewer::Finalize(nn::gfx::Device* pDevice)
{
    ToolConnector::Finalize();
    FinalizeResources(pDevice);
}

void Viewer::FinalizeScalableFont(nn::gfx::Device* pDevice)
{
    if(m_pComplexFontSet != NULL)
    {
        for(int i = 0; i < m_ComplexFontCount; i++)
        {
            ComplexFontHelper::FinalizeComplexFontTree(
                pDevice,
                m_pComplexFontSet[i].pComplexFontTree,
                m_UnregisterTextureViewSlot,
                m_pUserDataForDescriptorSlotAllocator);
        }

        Layout::FreeMemory(m_pComplexFontSet);

        m_pComplexFontSet = NULL;
        m_ComplexFontCount = 0;
    }

    if (m_pTextureCache != NULL)
    {
        m_pTextureCache->UnregisterTextureViewFromDescriptorPool(m_UnregisterTextureViewSlot, m_pUserDataForDescriptorSlotAllocator);
        m_pTextureCache->Finalize(pDevice, Layout::GetFreeFunction(), nn::ui2d::Layout::GetUserDataForAllocator());
        Layout::DeleteObj(m_pTextureCache);
        m_pTextureCache = NULL;
    }
}

void Viewer::SetBackgroundColor(const char* pHexColorString)
{
    uint32_t    color;
    std::sscanf(pHexColorString, "0x%x", &color);

    m_BackgroundColor[0] = ((color & 0xFF000000) >> 24) / 255.0f;
    m_BackgroundColor[1] = ((color & 0x00FF0000) >> 16) / 255.0f;
    m_BackgroundColor[2] = ((color & 0x0000FF00) >> 8) / 255.0f;
    m_BackgroundColor[3] = ((color & 0x000000FF) >> 0) / 255.0f;
}

bool Viewer::ConvertHostToFsPath(char *outFsPath, const char *mountPoint, const char *inHostPath)
{
    NN_SDK_ASSERT_NOT_NULL(outFsPath);
    NN_SDK_ASSERT_NOT_NULL(mountPoint);
    NN_SDK_ASSERT_NOT_NULL(inHostPath);

    if ( std::strlen(inHostPath) <= 0 )
    {
        return false;
    }

    char part[PathLengthMax];

    if (0 == strncmp("/cygdrive/", inHostPath, 10) && isalpha(inHostPath[10]) && inHostPath[11] == '/')
    {
        // CYGWIN形式
        nn::util::SNPrintf(outFsPath, PathLengthMax, "%s/%c", mountPoint, inHostPath + 10);

        nn::util::Strlcpy(part, inHostPath + 11, PathLengthMax);
    }
    else if (isalpha(inHostPath[0]) && inHostPath[1] == ':')
    {
        // DOS形式
        nn::util::SNPrintf(outFsPath, PathLengthMax, "%s/%c", inHostPath[0]);
        nn::util::Strlcpy(part, inHostPath + 2, PathLengthMax);
    }
    else
    {
        // パスが不正。
        return false;
    }

    size_t partLen = std::strlen(part);
    for(size_t i = 0; i < partLen; i ++)
    {
        size_t length = strlen(outFsPath);
        NN_SDK_ASSERT(length < PathLengthMax - 1);

        outFsPath[length]        = part[i] == '\\' ? '/' : part[i];
        outFsPath[length + 1]    = NULL;
    }

    return true;
}

#if defined(NN_BUILD_CONFIG_OS_WIN)
namespace
{
    bool IsOptionExist_(int argc, const char** argv, const char* name)
    {
        for (int i = 0; i < argc; i++)
        {
            if (strcmp(argv[i], name) == 0)
            {
                return true;
            }
        }

        return false;
    }

    const char* GetOptionValue_(int argc, const char** argv, const char* name)
    {
        for (int i = 0; i < argc; i++)
        {
            if (strcmp(argv[i], name) == 0)
            {
                if (i + 1 < argc)
                {
                    return argv[i + 1];
                }
                else {
                    return nullptr;
                }
            }
        }

        return nullptr;
    }
}

bool Viewer::PreviewByCommandLineOption(nn::gfx::Device* pDevice, int argc, const char** argv)
{
    NN_UNUSED(argc);
    NN_UNUSED(argv);

    m_PreviewOption.reset();

    const char* pValue = nullptr;

    pValue = GetOptionValue_(argc, argv, "-m");
    if(pValue != nullptr && pValue[0] == 'c')
    {
        // モードを指定する
        m_PreviewOption.previewMode = PreviewMode_Control;
    }

    pValue = GetOptionValue_(argc, argv, "-n");
    if(pValue != nullptr)
    {
        // 複数レイアウトがあった場合のルートのレイアウト名を指定する
        nn::util::Strlcpy(m_PreviewOption.layoutName, pValue, PathLengthMax);
        std::strncat(m_PreviewOption.layoutName, ".bflyt", PathLengthMax - strlen(m_PreviewOption.layoutName) - 1);
    }

    pValue = GetOptionValue_(argc, argv, "-s");
    if(pValue != nullptr)
    {
        // スクリーンショットを撮影します。引数でファイル名を指定します。
        m_ScreenShot.SetFileName(pValue);
    }

    pValue = GetOptionValue_(argc, argv, "-f");
    if(pValue != nullptr)
    {
        // スクリーンショットを撮影する際のフレームを指定します。
        m_ScreenShot.SetFrame(std::stof(pValue));
    }

    if(IsOptionExist_(argc, argv, "-g"))
    {
        // sRGBライトを行い、ガンマ値を補正します。
        m_PreviewOption.isSrgbWriteEnabled = true;
    }

    pValue = GetOptionValue_(argc, argv, "-bg_color");
    if (pValue != nullptr)
    {
        SetBackgroundColor(pValue);
    }

    // スクリーンショットが有効なら必ずアニメーションモード
    if (m_ScreenShot.IsNeedToTake() && m_PreviewOption.previewMode == PreviewMode_Control)
    {
        NN_DETAIL_UI2D_WARN("preview mode must be anim when taking screen shot.\n");
        m_PreviewOption.previewMode = PreviewMode_Animation;
    }

    if (argc > 1)
    {
        // 最後の引数をファイル名と仮定します。
        nn::util::Strlcpy(m_PreviewOption.path, argv[argc - 1], PathLengthMax);
        StartPreview(pDevice, m_PreviewOption);
        return true;
    }
    else
    {
        return false;
    }
}

LRESULT Viewer::PreviewByMessageProcCallback(nn::gfx::Device* pDevice, HWND hWnd, UINT message, WPARAM wParam, LPARAM lParam)
{
    NN_UNUSED(hWnd);
    NN_UNUSED(wParam);

    switch(message)
    {
    case WM_COPYDATA:
        {
            PCOPYDATASTRUCT cds = NULL;
            LayoutProtocol  *protocol = NULL;
            char    *verb = NULL;

            cds = (PCOPYDATASTRUCT)lParam;
            protocol = (LayoutProtocol*)cds->lpData;
            verb = protocol->Verb;

            if(::strcmp(verb, "Open") == 0)
            {
                PreviewByCommandStr(pDevice, protocol->Data);
            }

            return 1;
        }
    default:
        // 何もしません。
        return 0;
    }
}
#endif

bool Viewer::PreviewByCommandStr(nn::gfx::Device* pDevice, const char* commandStr)
{
    // コントロール
    if (nn::util::Strncmp(commandStr, "ViewerCtrlCmd_:", sizeof("ViewerCtrlCmd_:") - 1) == 0)
    {
        return true;
    }


    m_PreviewOption.reset();

    ParseCommandStr(&m_PreviewOption, commandStr);

    StartPreview(pDevice, m_PreviewOption);

    return true;
}

void Viewer::DrawFirstFrameCaptureTextureAndUpdateLayout(nn::gfx::Device* pDevice, nn::gfx::CommandBuffer& commandBuffer, int constantBufferIndex)
{
    if (m_Layout)
    {
        m_DrawInfo.Map(constantBufferIndex);

        // キャプチャテクスチャの更新処理(描画処理)でアクセスされるため、呼び出し前に設定しておく。
        m_DrawInfo.SetGpuAccessBufferIndex(constantBufferIndex);

        float oldFrame = m_AnimationMgr.GetCurrentAnimationFrame();

        // 0 フレームに時間を戻してキャプチャテクスチャを作成する。
        m_AnimationMgr.SetCurrentAnimationFrame(0.0f);
        m_Layout->AnimateAndUpdateAnimFrame();

        m_DrawInfo.SetProjectionMtx(m_ProjectionMatrix);
        m_DrawInfo.SetViewMtx(m_CameraViewMatrix);

        // ドロップシャドウの静的レンダリングキャッシュを作成する際にアルファ値を無視してレンダリングする必要があるため
        // アルファが 0 でも Calculate 内の処理が行われるようにフラグを設定します。
        const bool alphaZeroPaneCalculated = m_DrawInfo.IsAlphaZeroPaneCalculated();
        m_DrawInfo.SetAlphaZeroPaneCalculated(true);

        m_Layout->Calculate(m_DrawInfo, true);

        // 初回更新キャプチャテクスチャを作成。
        m_Layout->DrawCaptureTexture(pDevice, m_DrawInfo, commandBuffer);
        // ドロップシャドウキャッシュを作成。
        m_Layout->DrawDropShadowStaticCache(pDevice, m_DrawInfo, commandBuffer);

        m_DrawInfo.SetAlphaZeroPaneCalculated(alphaZeroPaneCalculated);

        // 時間をもとに戻して通常更新処理。
        m_AnimationMgr.SetCurrentAnimationFrame(oldFrame);
        m_Layout->AnimateAndUpdateAnimFrame();

        m_Layout->Calculate(m_DrawInfo, true);

        if (m_PostCalculateCallback != NULL)
        {
            (*m_PostCalculateCallback)(*m_Layout);
        }
        m_AnimateOnce = true;
        m_IsViewMtxDirty = false;

        m_DrawInfo.Unmap();
    }
}

void Viewer::UpdateLayout(int constantBufferIndex)
{
    if (m_Layout)
    {
        m_DrawInfo.Map(constantBufferIndex);

        m_Layout->AnimateAndUpdateAnimFrame();
        m_DrawInfo.SetProjectionMtx(m_ProjectionMatrix);
        m_DrawInfo.SetViewMtx(m_CameraViewMatrix);
        m_Layout->Calculate(m_DrawInfo, m_IsViewMtxDirty);
        if (m_PostCalculateCallback != NULL)
        {
            (*m_PostCalculateCallback)(*m_Layout);
        }
        m_AnimateOnce = true;
        m_IsViewMtxDirty = false;

        m_DrawInfo.Unmap();

        m_DrawInfo.SetGpuAccessBufferIndex(constantBufferIndex);
    }
}

void Viewer::UpdateSystem(nn::gfx::Device* pDevice)
{
    uint8_t* message = ToolConnector::GetPacket();
    if (message)
    {
        PreviewByCommandStr(pDevice, reinterpret_cast<const char*>(message));
    }
}

#if defined(NN_BUILD_CONFIG_OS_WIN)
void Viewer::UpdateForScreenShot(
    int constantBufferIndex,
    nn::gfx::Device* pDevice,
    nn::gfx::MemoryPool* pMemoryPool,
    ptrdiff_t* pMemoryPoolOffset,
    nn::AlignedAllocateFunctionWithUserData pAllocateFunction, void* pUserDataForAllocator)
{
    if (m_Layout == NULL)
    {
        return;
    }

    if (m_ScreenShot.IsNeedToTake())
    {
        if (m_ScreenShot.GetState() == ScreenShot::State_NotReady)
        {
            m_DrawInfo.Map(constantBufferIndex);

            if (m_AnimationMgr.GetAnimator())
            {
                m_AnimationMgr.GetAnimator()->StopAt(m_ScreenShot.GetFrame());
            }
            m_Layout->Animate();
            m_Layout->Calculate(m_DrawInfo, m_IsViewMtxDirty);
            m_ScreenShot.Prepare(pDevice, pMemoryPool, pMemoryPoolOffset, pAllocateFunction, pUserDataForAllocator);

            m_AnimateOnce = true;
            m_IsViewMtxDirty = false;

            m_DrawInfo.Unmap();
            m_DrawInfo.SetGpuAccessBufferIndex(constantBufferIndex);
        }
        return;
    }
}
#endif

void Viewer::UpdateInputs(nn::gfx::Device* pDevice, const InputDeviceState& inputDeviceState)
{
    // HUD の表示・非表示きりかえ
    if(inputDeviceState.isTrigX)
    {
        SetHudVisible(!m_IsHudVisible);
    }

    if (m_PreviewOption.previewMode == PreviewMode_Control && m_Layout != NULL)
    {
        bool is_pointer_on = inputDeviceState.isMousePointerOn;
        if (is_pointer_on)
        {
            // ポインタの位置をレイアウトの空間に変換し、ButtonGroupのUpdateに与える
            nn::ui2d::Size viewSize = GetViewSize();
            nn::ui2d::Size layoutSize = m_Layout->GetLayoutSize();
            float widthRate = layoutSize.width / viewSize.width;
            float heightRate = layoutSize.height / viewSize.height;
            float x, y; // 画面上のポインタ位置。中心原点、右下が大きくなる座標系
            float viewport_width, viewport_heigth;
            bool is_touch;
            bool is_release;

            x = 2.0f * inputDeviceState.pointerX / m_MainViewportSize.width - 1.0f;
            y = 2.0f * inputDeviceState.pointerY / m_MainViewportSize.height - 1.0f;
            viewport_width = m_MainViewportSize.width;
            viewport_heigth = m_MainViewportSize.height;

            is_touch = inputDeviceState.isTrigMouseLButton;
            is_release = inputDeviceState.isReleaseMouseLButton;

            nn::util::Float2 pos = NN_UTIL_FLOAT_2_INITIALIZER( x * (viewport_width / 2.f) * widthRate, -y * (viewport_heigth / 2.f) * heightRate );
            m_ButtonGroup.Update(&pos, is_touch, is_release);
        }
        else
        {
            // ポインタが画面に入っていないときはnullを与えてもよい
            m_ButtonGroup.Update(NULL, false);
        }
    }
    else
    {
        // 再生停止・リセット
        if (inputDeviceState.isTrigA)
        {
            if (m_AnimationMgr.IsAnimationPlaying())
            {
                m_AnimationMgr.StopAnimation();
            }
            else
            {
                m_AnimationMgr.StartAnimation();
            }
        }
        else if (inputDeviceState.isTrigB)
        {
            m_AnimationMgr.ResetAnimation();
        }

        nn::ui2d::Animator* animator = m_AnimationMgr.GetAnimator();
        if (animator)
        {
            // 左右キーで、アニメーションフレームをコントロール
            if (inputDeviceState.isHoldLeft)
            {
                animator->StopAtCurrentFrame();
                if (animator->IsMinFrame())
                {
                    animator->StopAtEndFrame();
                }
                else
                {
                    animator->SetFrame(animator->GetFrame() - 1);
                }
            }
            else if (inputDeviceState.isHoldRight)
            {
                animator->StopAtCurrentFrame();
                if (animator->IsMaxFrame())
                {
                    animator->StopAtStartFrame();
                }
                else
                {
                    animator->SetFrame(animator->GetFrame() + 1);
                }
            }

            // 上下キーで、再生するアニメーションを切り替え
            if (inputDeviceState.isTrigUp)
            {
                if (m_AnimationMgr.GetCurrentAnimationNo() == 0)
                {
                    m_AnimationMgr.SetCurrentAnimationNo(pDevice, m_AnimationMgr.GetAnimationCount() - 1);
                }
                else
                {
                    m_AnimationMgr.SetCurrentAnimationNo(pDevice, m_AnimationMgr.GetCurrentAnimationNo() - 1);
                }
                m_AnimationMgr.StartAnimation();
            }
            else if (inputDeviceState.isTrigDown)
            {
                if (m_AnimationMgr.GetAnimationCount() <= m_AnimationMgr.GetCurrentAnimationNo() + 1)
                {
                    m_AnimationMgr.SetCurrentAnimationNo(pDevice, 0);
                }
                else
                {
                    m_AnimationMgr.SetCurrentAnimationNo(pDevice, m_AnimationMgr.GetCurrentAnimationNo() + 1);
                }
                m_AnimationMgr.StartAnimation();
            }
        }
    }
}// NOLINT(impl/function_size)

void Viewer::UpdateMenu()
{
    // 一時的にメニュー表示を廃止しています。
}

void Viewer::DrawCaptureTexture(nn::gfx::Device* pDevice, nn::gfx::CommandBuffer& commandBuffer)
{
    if (m_Layout != NULL)
    {
        m_DrawInfo.SetProjectionMtx(m_ProjectionMatrix);
        m_DrawInfo.SetViewMtx(m_CameraViewMatrix);

        m_Layout->DrawCaptureTexture(pDevice, m_DrawInfo, commandBuffer);
    }
}

void Viewer::DrawLayout(nn::gfx::CommandBuffer& commandBuffer)
{
    if (m_Layout != NULL && m_AnimateOnce)
    {
        m_DrawInfo.SetProjectionMtx(m_ProjectionMatrix);
        m_DrawInfo.SetViewMtx(m_CameraViewMatrix);

        m_Layout->Draw(m_DrawInfo, commandBuffer);
    }
}

void Viewer::DrawLayoutDRC(nn::gfx::CommandBuffer& commandBuffer)
{
    if (m_Layout != NULL && m_AnimateOnce)
    {
        m_DrawInfo.SetProjectionMtx(m_ProjectionMatrixDRC);
        m_DrawInfo.SetViewMtx(m_CameraViewMatrixDRC);
        m_Layout->Draw(m_DrawInfo, commandBuffer);
    }
}

void Viewer::DrawSystem(nn::gfx::CommandBuffer& commandBuffer)
{
    NN_UNUSED(commandBuffer);
}

#if defined(NN_BUILD_CONFIG_OS_WIN)
void Viewer::TakeScreenShotAfterCommandDone(
    nn::FreeFunctionWithUserData pFreeFunction, void* pUserDataForAllocator)
{
    if (m_ScreenShot.IsNeedToTake())
    {
        if (m_ScreenShot.GetState() == ScreenShot::State_ReadyToTake)
        {
            m_ScreenShot.Take(pFreeFunction, pUserDataForAllocator);
        }
    }
}
#endif

void Viewer::FinalizeResources(nn::gfx::Device* pDevice)
{
    // 背景画像リソースの開放処理
    if (m_pBackgroundImageFile != NULL)
    {
        m_pBackgroundImageFile->GetResTexture(0)->Finalize(pDevice);
        m_pBackgroundImageFile->Finalize(pDevice);
        m_pBackgroundImageFile = NULL;
    }

    if (m_Layout != NULL)
    {
        // レイアウトが読み込まれていたらスケーラブルフォントを解放する。
        // m_ResourceAccessor.Finalize でアーカイブファイルが解放されるよりも先に
        // fcpx が使っている ResFont を解放しなければならない。
        FinalizeScalableFont(pDevice);
    }

    // レイアウトリソースの解放
    if (m_Layout != NULL)
    {
        m_ButtonGroup.FreeAll();
        m_AnimationMgr.Finalize();
        m_Layout->Finalize(pDevice);
        nn::ui2d::Layout::DeleteObj(m_Layout);
        m_Layout = NULL;
    }

    // 既にアタッチされていたらDetach
    if (m_ResourceAccessor.IsAttached())
    {
#if defined(NN_BUILD_CONFIG_OS_WIN)
#else
        uint8_t * arc_data = reinterpret_cast<uint8_t *>(const_cast<void*>(m_ResourceAccessor.GetArchiveDataStart()));
#endif
        m_ResourceAccessor.UnregisterTextureViewFromDescriptorPool(m_UnregisterTextureViewSlot, m_pUserDataForDescriptorSlotAllocator);
        m_ResourceAccessor.Detach();
        m_ResourceAccessor.Finalize(pDevice);

#if defined(NN_BUILD_CONFIG_OS_WIN)
        m_UnloadFunction(NULL);
#else
        m_UnloadFunction(arc_data);
#endif
    }

}

void Viewer::StartPreview(nn::gfx::Device* pDevice, const PreviewOption& option)
{
    m_AnimateOnce = false;

    // 再生状態を継続するために以前の設定を残す
    bool isAnimationPlayingOld = true;
    float animationFrameOld = 0;
    char animationNameOld[nn::ui2d::AnimTagNameStrMax + 1] = {};

    if (m_Layout != NULL && m_AnimationMgr.GetAnimationFrameMax() != 0)
    {
        isAnimationPlayingOld = m_AnimationMgr.IsAnimationPlaying();
        animationFrameOld = m_AnimationMgr.GetCurrentAnimationFrame();
        if (m_AnimationMgr.GetCurrentAnimationNo() > 0)
        {
            nn::util::Strlcpy(animationNameOld, m_AnimationMgr.GetCurrentAnimationName(), nn::ui2d::AnimTagNameStrMax + 1);
        }
    }

    FinalizeResources(pDevice);

    if (m_PreReloadLayoutDataCallback != NULL)
    {
        (*m_PreReloadLayoutDataCallback)();
    }

#if defined(NN_BUILD_CONFIG_OS_WIN)
    m_ResourceAccessor.Attach(option.path);
#else
    {
        void* pData = m_LoadFunction(option.path, nn::ui2d::ArchiveResourceAlignment);

        // バイナリリソースのルートディレクトリを指定してリソースアクセサを生成。
        if ( ! m_ResourceAccessor.Attach(pData, "."))
        {
            NN_SDK_ASSERT(false, "can not attach layout archive.\n");
        }
    }
#endif

    m_Layout = nn::ui2d::Layout::AllocateAndConstruct<nn::ui2d::Layout>();

    {
        // スケーラブルフォント関連の初期化
        InitializeScalableFont(pDevice);

        const char* layoutName = NULL;
        // レイアウト名が指定されていないときは探す
        if (std::strlen(option.layoutName) <= 0)
        {
            layoutName = m_ResourceAccessor.FindFile(nn::ui2d::ResourceTypeLayout, NULL);
        }
        else
        {
            layoutName = option.layoutName;
        }
        const void* lytRes = m_ResourceAccessor.FindResourceByName(nn::ui2d::ResourceTypeLayout, layoutName);
        NN_SDK_ASSERT_NOT_NULL(lytRes);

        nn::ui2d::Layout::BuildOption buildOption;
        buildOption.SetDefault();
        buildOption.pGetUserShaderInformationFromUserDataCallback = m_GetUserShaderInformationFromUserDataCallback;

        BuildResultInformation buildResultInformation;
        buildResultInformation.SetDefault();

        if (m_PreviewOption.previewMode == PreviewMode_Animation)
        {
            m_Layout->Build(&buildResultInformation, pDevice, &m_ResourceAccessor, NULL, NULL, lytRes, buildOption);
            m_AnimationMgr.Setup(pDevice, m_Layout, &m_ResourceAccessor);

            if (m_PreviewOption.continueAnimation)
            {
                // 以前の状態を引き継ぐ
                int animationNo = m_AnimationMgr.GetAnimationNoByTagName(animationNameOld);
                if (animationNo < m_AnimationMgr.GetAnimationCount())
                {
                    m_AnimationMgr.SetCurrentAnimationNo(pDevice, animationNo);
                }

                m_AnimationMgr.SetCurrentAnimationFrame(animationFrameOld);
                if (isAnimationPlayingOld)
                {
                    m_AnimationMgr.StartAnimation();
                }
            }
            else
            {
                m_AnimationMgr.StartAnimation();
            }
        }
        else
        {
            if (m_pControlCreator == NULL)
            {
                nn::ui2d::DefaultControlCreator controlCreater(&m_ButtonGroup);
                m_Layout->Build(&buildResultInformation, pDevice, &m_ResourceAccessor, &controlCreater, NULL, lytRes, buildOption);
            }
            else
            {
                m_Layout->Build(&buildResultInformation, pDevice, &m_ResourceAccessor, m_pControlCreator, NULL, lytRes, buildOption);
            }
        }

        m_ResourceAccessor.RegisterTextureViewToDescriptorPool(m_RegisterTextureViewSlot, m_pUserDataForDescriptorSlotAllocator);

        SetViewSize(m_ViewSize);

        if (m_PostReloadLayoutDataCallback != NULL)
        {
            (*m_PostReloadLayoutDataCallback)(m_DrawInfo, buildResultInformation);
        }
    }

    // 背景画像の読み込み処理
    if (option.backgroundImageEnabled)
    {
        void* pTextureData = m_ResourceAccessor.FindResourceByName(nn::ui2d::ResourceTypeTexture, "__BgImage.bntx");

        m_pBackgroundImageFile = nn::gfx::ResTextureFile::ResCast(pTextureData);
        m_pBackgroundImageFile->Initialize(pDevice);
        m_pBackgroundImageFile->GetResTexture(0)->Initialize(pDevice);
        nn::gfx::ResTexture*    pResTexture = m_pBackgroundImageFile->GetResTexture(0);
        m_BackgroundImageSize.width = (float)pResTexture->GetTextureInfo()->GetWidth();
        m_BackgroundImageSize.height = (float)pResTexture->GetTextureInfo()->GetHeight();
        nn::gfx::TextureView*   pTextureView = pResTexture->GetTextureView();
        (*m_RegisterTextureViewSlot)(&m_BackgroundImageSlot, *pTextureView, m_pUserDataForDescriptorSlotAllocator);
    }

    m_Layout->ResetFirstFrameCaptureUpdatdFlag();
    m_IsSrgbWriteEnabled = option.isSrgbWriteEnabled != 0 ? true : false;
}// NOLINT(impl/function_size)

void Viewer::InitializeScalableFont(nn::gfx::Device* pDevice)
{
    // ResourceTypeComplexFont の 数を数える
    int complexFontCount = 0;
    {
        const char* pFileName = NULL;
        pFileName = m_ResourceAccessor.FindFile(nn::ui2d::ResourceTypeComplexFont, pFileName);
        while(pFileName != NULL)
        {
            complexFontCount++;
            pFileName = m_ResourceAccessor.FindFile(nn::ui2d::ResourceTypeComplexFont, pFileName);
        }
    }
    if (complexFontCount == 0)
    {
        return;
    }

    // バッファ確保
    m_ComplexFontCount = complexFontCount;
    m_pComplexFontSet = static_cast<ComplexFont*>(Layout::AllocateMemory(sizeof(ComplexFont) * complexFontCount));

    // 複合フォントのファイルを列挙して処理する
    int* pFontFaceHead = static_cast<int*>(Layout::AllocateMemory(sizeof(int) * complexFontCount));
    nn::font::TextureCache::InitializeArg* pArg = static_cast<nn::font::TextureCache::InitializeArg*>(Layout::AllocateMemory(sizeof(nn::font::TextureCache::InitializeArg)));
    pArg->SetDefault();
    {
        int index = 0;
        const char* pFileName = NULL;
        pFileName = m_ResourceAccessor.FindFile(nn::ui2d::ResourceTypeComplexFont, pFileName);
        while (pFileName != NULL)
        {
            const void* pCpxData = m_ResourceAccessor.FindResourceByName(nn::ui2d::ResourceTypeComplexFont, pFileName);

            pFontFaceHead[index] = ComplexFontHelper::SetupTextureCacheArg(
                pArg,
                AcquireFontFunctionForComplexFont,
                &m_ResourceAccessor,
                pCpxData);
            index++;
            pFileName = m_ResourceAccessor.FindFile(nn::ui2d::ResourceTypeComplexFont, pFileName);
        }
    }
    if (pArg->fontCount != 0)
    {
        NN_SDK_ASSERT(m_pTextureCache == NULL);
        m_pTextureCache = Layout::AllocateAndConstruct<nn::font::TextureCache>();
        pArg->workMemorySize = m_ScalableFontWorkMemorySize;
        pArg->noPlotWorkMemorySize = m_ScalableFontNoPlotWorkMemorySize;
        pArg->allocateFunction = Layout::GetAllocateFunction();
        pArg->textureCacheWidth = m_ScalableFontTextureCacheWidth;
        pArg->textureCacheHeight = m_ScalableFontTextureCacheHeight;
        m_pTextureCache->Initialize(pDevice, *pArg);
        m_pTextureCache->RegisterTextureViewToDescriptorPool(m_RegisterTextureViewSlot, m_pUserDataForDescriptorSlotAllocator);
    }
    {
        int index = 0;
        const char* pFileName = NULL;
        pFileName = m_ResourceAccessor.FindFile(nn::ui2d::ResourceTypeComplexFont, pFileName);
        while (pFileName != NULL)
        {
            const void* pCpxData = m_ResourceAccessor.FindResourceByName(nn::ui2d::ResourceTypeComplexFont, pFileName);

            nn::font::Font* pComplexFontTree = ComplexFontHelper::InitializeComplexFontTree(
                pDevice,
                m_RegisterTextureViewSlot,
                m_pUserDataForDescriptorSlotAllocator,
                m_pTextureCache,
                pFontFaceHead[index],
                AcquireFontFunctionForComplexFont,
                &m_ResourceAccessor,
                pCpxData);
            m_pComplexFontSet[index].pComplexFontTree = pComplexFontTree;

            // アーカイバに .fcpx として登録します。
            {
                char temp[FontNameMax];
                memset(temp, 0x0, sizeof(temp));

                const char* pDot = strchr(pFileName, '.');
                if (pDot != NULL)
                {
                    ptrdiff_t size = pDot - pFileName;
                    nn::util::Strlcpy(temp, pFileName, static_cast<int>(size + 1));
                    strncat(temp, ".fcpx", FontNameMax - strlen(pFileName) - 1);
                }
                else {
                    nn::util::Strlcpy(temp, pFileName, FontNameMax);
                    strncat(temp, ".fcpx", FontNameMax - strlen(temp) - 1);
                }

                m_ResourceAccessor.RegisterFont(temp, pComplexFontTree);
            }

            index++;
            pFileName = m_ResourceAccessor.FindFile(nn::ui2d::ResourceTypeComplexFont, pFileName);
        }
    }
    Layout::FreeMemory(pArg);
    Layout::FreeMemory(pFontFaceHead);
}

void Viewer::RegisterAndCompleteAllGlyphsForScalableFontTextureCache()
{
    if(m_Layout == NULL)
    {
        return;
    }

    for( int i = 0; i < m_ComplexFontCount; i++ )
    {
        nn::font::RegisterGlyphFromTextBoxRecursive(m_Layout->GetRootPane());
    }

    if (m_pTextureCache != NULL)
    {
        m_pTextureCache->UpdateTextureCache();
        m_pTextureCache->CompleteTextureCache();
    }
}

Viewer::OptionMode Viewer::GetOptionMode(const char* buf)
{
    if (std::strcmp(buf, "-m") == 0)
    {
        return OptionMode_PreviewMode;
    }
    else if (std::strcmp(buf, "-n") == 0)
    {
        return OptionMode_LayoutName;
    }
    else if (std::strcmp(buf, "-projection") == 0)
    {
        return OptionMode_PerspectiveProjection;
    }
    else if (std::strcmp(buf, "-fov") == 0)
    {
        return OptionMode_PerspectiveFovy;
    }
    else if (std::strcmp(buf, "-near") == 0)
    {
        return OptionMode_Near;
    }
    else if (std::strcmp(buf, "-far") == 0)
    {
        return OptionMode_Far;
    }
    else if (std::strcmp(buf, "-viewmode") == 0)
    {
        return OptionMode_ViewSize;
    }
    else if (std::strcmp(buf, "-bg_color") == 0)
    {
        return OptionMode_BackgroundColor;
    }
    else if (std::strcmp(buf, "-bg_image") == 0)
    {
        return OptionMode_BackgroundImage;
    }

    return OptionMode_None;
}

void Viewer::AnalyseOptionArgument(PreviewOption* option, OptionMode mode, const char* buf)
{
    switch(mode)
    {
    case OptionMode_PreviewMode:
        // モードを指定する
        if (buf[0] == 'c')
        {
            option->previewMode = PreviewMode_Control;
        }
        else
        {
            option->previewMode = PreviewMode_Animation;
        }
        break;
    case OptionMode_LayoutName:
        // レイアウト名を指定する
        nn::util::Strlcpy(option->layoutName, buf, PathLengthMax);
        std::strncat(option->layoutName, ".bflyt", PathLengthMax - strlen(option->layoutName) - 1);
        break;
    case OptionMode_PerspectiveProjection:
        m_IsPerspectiveProjection = std::strcmp(buf, "pers") == 0;
        break;
    case OptionMode_PerspectiveFovy:
        m_PerspectiveFovyDeg = StrToFloat_(buf);
        break;
    case OptionMode_Near:
        m_PerspectiveNear = StrToFloat_(buf);
        break;
    case OptionMode_Far:
        m_PerspectiveFar = StrToFloat_(buf);
        break;
    case OptionMode_ViewSize:
        if (std::strcmp(buf, "original") == 0)
        {
            m_ViewSize = ViewSize_Original;
        }
        else if (std::strcmp(buf, "fit_screen") == 0)
        {
            m_ViewSize = ViewSize_FitScreen;
        }
        break;
    case OptionMode_BackgroundColor:
        SetBackgroundColor(buf);
        break;
    case OptionMode_BackgroundImage:
        {
            option->backgroundImageEnabled = true;
        }
        break;
    case OptionMode_None:
    default:
        NN_SDK_ASSERT(false);
    }
}

void Viewer::ParseCommandStr(PreviewOption* option, const char* commandStr)
{
    OptionMode optionMode = OptionMode_None;
    const char* pCursor = commandStr;
    const char* pCursorNext = commandStr;

    char buf[PathLengthMax];

    while( pCursorNext != NULL)
    {
        pCursorNext = strchr(pCursor, ' ');

        if(pCursorNext == NULL)
        {
            // PCのファイルシステムにおける絶対パスが指定されます。
            nn::util::Strlcpy(option->path, pCursor, PathLengthMax);
            continue;
        }

        size_t copyLen = pCursorNext - pCursor;
        nn::util::Strlcpy(buf, pCursor, static_cast<int>(copyLen + 1));

        pCursor = pCursorNext + 1;

        if (optionMode == OptionMode_None)
        {
            if ((optionMode = GetOptionMode(buf)) != OptionMode_None)
            {
                continue;
            }
        }

        // 引数指定のないオプションを直接解析する。
        if (std::strcmp(buf, "-g") == 0)
        {
            option->isSrgbWriteEnabled = true;
            continue;
        }
        else if (std::strcmp(buf, "-is30fps") == 0)
        {
            m_Fps = Fps_30;
            continue;
        }
        else if (std::strcmp(buf, "-is60fps") == 0)
        {
            m_Fps = Fps_60;
            continue;
        }
        else if (std::strcmp(buf, "-continue-animation") == 0)
        {
            option->continueAnimation = true;
            continue;
        }

        if (optionMode != OptionMode_None)
        {
            AnalyseOptionArgument(option, optionMode, buf);
        }
        else
        {
            return ;
        }
        optionMode = OptionMode_None;
    }
}

nn::ui2d::Size Viewer::GetViewSize() const
{
    if (m_ViewSize == ViewSize_Original && m_Layout != NULL)
    {
        return m_Layout->GetLayoutSize();
    }
    else if (m_ViewSize == ViewSize_FitScreen)
    {
        return nn::ui2d::Size::Create(m_MainViewportSize.width, m_MainViewportSize.height);
    }
    else
    {
        return nn::ui2d::Size::Create(ViewSizeValues[m_ViewSize].x, ViewSizeValues[m_ViewSize].y);
    }
}

void Viewer::SetViewSize(ViewSize viewSize)
{
    m_ViewSize = viewSize;
    m_IsViewMtxDirty = true;

    // drawInfoへの行列の設定
    if (m_Layout != NULL)
    {
        nn::ui2d::Size viewSizeValue = GetViewSize();
        nn::ui2d::Size layoutSize = m_Layout->GetLayoutSize();
        float widthRate = layoutSize.width / viewSizeValue.width;
        float heightRate = layoutSize.height / viewSizeValue.height;
        float halfWidth = m_MainViewportSize.width / 2.f;
        float halfHeight = m_MainViewportSize.height / 2.f;
        static const float halfWidthDRC = 0.5f * DrcWidth;
        static const float halfHeightDRC = 0.5f * DrcHeight;
        if (m_IsPerspectiveProjection)
        {
            float fovy = nn::util::DegreeToRadian(m_PerspectiveFovyDeg);
            {
                float height = halfHeight * heightRate / nn::util::TanTable(nn::util::RadianToAngleIndex(fovy / 2.f));
                float aspect = (halfWidth * widthRate) / (halfHeight * heightRate);

                nn::util::MatrixPerspectiveFieldOfViewRightHanded(&m_ProjectionMatrix, fovy, aspect, m_PerspectiveNear, m_PerspectiveFar);

                nn::util::Vector3fType pos;
                nn::util::VectorSet(&pos, 0.f, 0.f, height);
                nn::util::Vector3fType up;
                nn::util::VectorSet(&up, 0.f, 1.f, 0.f);
                nn::util::Vector3fType target;
                nn::util::VectorSet(&target, 0.f, 0.f, 0.f);

                nn::util::MatrixLookAtRightHanded(&m_CameraViewMatrix, pos, target, up);
            }
            {
                float height = halfHeightDRC * heightRate / nn::util::TanTable(nn::util::RadianToAngleIndex(fovy / 2.f));
                float aspect = (halfWidthDRC * widthRate) / (halfHeightDRC * heightRate);

                nn::util::MatrixPerspectiveFieldOfViewRightHanded(&m_ProjectionMatrixDRC, fovy, aspect, m_PerspectiveNear, m_PerspectiveFar);

                nn::util::Vector3fType pos;
                nn::util::VectorSet(&pos, 0.f, 0.f, height);
                nn::util::Vector3fType up;
                nn::util::VectorSet(&up, 0.f, 1.f, 0.f);
                nn::util::Vector3fType target;
                nn::util::VectorSet(&target, 0.f, 0.f, 0.f);

                nn::util::MatrixLookAtRightHanded(&m_CameraViewMatrixDRC, pos, target, up);
            }
        }
        else
        {
            {
                float left = -halfWidth * widthRate;
                float right = halfWidth * widthRate;
                float top = halfHeight * heightRate;
                float bottom = -halfHeight * heightRate;
                float nnear = 0.0f;
                float ffar = 500.0f;

                nn::util::MatrixOrthographicOffCenterRightHanded(&m_ProjectionMatrix, left, right, bottom, top, nnear, ffar);

                float cx = (left + right) * 0.5f;
                float cy = (top + bottom) * 0.5f;
                nn::util::Vector3fType pos;
                nn::util::VectorSet(&pos, cx, cy, nnear);
                nn::util::Vector3fType up;
                nn::util::VectorSet(&up, 0.f, 1.f, 0.f);
                nn::util::Vector3fType target;
                nn::util::VectorSet(&target, cx, cy, nnear - 1);

                nn::util::MatrixLookAtRightHanded(&m_CameraViewMatrix, pos, target, up);
            }
            {
                float left = -halfWidthDRC * widthRate;
                float right = halfWidthDRC * widthRate;
                float top = halfHeightDRC * heightRate;
                float bottom = -halfHeightDRC * heightRate;
                float nnear = 0.0f;
                float ffar = 500.0f;

                nn::util::MatrixOrthographicOffCenterRightHanded(&m_ProjectionMatrixDRC, left, right, bottom, top, nnear, ffar);

                float cx = (left + right) * 0.5f;
                float cy = (top + bottom) * 0.5f;
                nn::util::Vector3fType pos;
                nn::util::VectorSet(&pos, cx, cy, nnear);
                nn::util::Vector3fType up;
                nn::util::VectorSet(&up, 0.f, 1.f, 0.f);
                nn::util::Vector3fType target;
                nn::util::VectorSet(&target, cx, cy, nnear - 1);

                nn::util::MatrixLookAtRightHanded(&m_CameraViewMatrixDRC, pos, target, up);
            }
        }
    }
}

//-------------------------------------------------------------------------
void Viewer::GetBackgroundColor(float* pBackgroundColor) const
{
    const float srgbGammaConst = 2.2f;

    if (IsSrgbWriteEnabled())
    {
        pBackgroundColor[0] = std::pow(m_BackgroundColor[0], srgbGammaConst);
        pBackgroundColor[1] = std::pow(m_BackgroundColor[1], srgbGammaConst);
        pBackgroundColor[2] = std::pow(m_BackgroundColor[2], srgbGammaConst);
    }
    else
    {
        pBackgroundColor[0] = m_BackgroundColor[0];
        pBackgroundColor[1] = m_BackgroundColor[1];
        pBackgroundColor[2] = m_BackgroundColor[2];
    }
    pBackgroundColor[3] = m_BackgroundColor[3];
}

//-------------------------------------------------------------------------
void Viewer::WriteHud(nn::font::TextWriter& writer) const
{
    if(!IsHudVisible())
    {
        return;
    }

    if (m_PreviewOption.previewMode == PreviewMode_Control)
    {
        writer.Printf("\n");
        writer.Printf(" Control Mode -------------------------------------------------------------------------\n");
        writer.Printf(" B:Reset  X:HideHUD\n");
        writer.Printf(" --------------------------------------------------------------------------------------\n");
    }else{
        writer.Printf("\n");
        writer.Printf(" Animation Mode -----------------------------------------------------------------------\n");
        writer.Printf(" A:Play/Stop  B:Reset  Left/Right:FrameControl  Up/Down:SelectAnimation  X:HideHUD\n");
        writer.Printf(" --------------------------------------------------------------------------------------\n");
        writer.Printf(" Frame:%3.0f /%3.0f\n", m_AnimationMgr.GetCurrentAnimationFrame(), m_AnimationMgr.GetAnimationFrameMax());
        writer.Printf("  Anim: %s%s\n", m_AnimationMgr.GetAnimationCount() > 0 ? m_AnimationMgr.GetCurrentAnimationName() : "none", m_AnimationMgr.IsCurrentAnimationLoop() ? "(Loop)" : "");
        writer.Printf("        %2d /%2d\n", m_AnimationMgr.GetCurrentAnimationNo() + 1, m_AnimationMgr.GetAnimationCount());
    }

    nn::ui2d::Size  showSize = GetViewSize();

    // FitScreen モードの時は画面にレイアウトデータを画面サイズに拡大して表示しているため
    // GetViewSize で取得されるサイズは画面の大きさになります。
    // サイズに常に画面サイズを表示するよりもレイアウトデータのサイズを表示したほうがユーザーにとって
    // 自然で有益なため、表示するサイズを置き換えます。
    if (m_ViewSize == ViewSize_FitScreen)
    {
        showSize = m_Layout->GetLayoutSize();
    }
    writer.Printf("  Size: %.0fx%.0f\n", showSize.width, showSize.height);
    writer.Printf("   FPS: %s\n", this->GetFps() == Fps_60 ? "60fps" : "30fps");
    writer.Printf(" Gamma: %s\n", this->IsSrgbWriteEnabled() ? "Linear" : "sRGB");
    writer.Printf("  Proj: %s\n", this->m_IsPerspectiveProjection ? "Perspective" : "Ortho");
    writer.Printf("  FOVY: %.0f (degree)\n", m_PerspectiveFovyDeg);
    writer.Printf("  Near: %.0f \n", m_PerspectiveNear);
    writer.Printf("   Far: %.0f \n", m_PerspectiveFar);
}

} // namespace viewer
} // namespace ui2d
} // namespace nn

#endif // NN_UI2D_VIEWER_ENABLED
