﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NOMINMAX
#define NOMINMAX
#endif

#include <sstream>
#include <nn/nn_Macro.h>

#include <nn/font/font_ScalableFontUtil.h>
#include <nn/ui2d/viewer/ui2d_ScalableFontManager.h>

namespace nn
{
namespace ui2d
{
namespace viewer
{

namespace
{
    //--------------------------------------------------------------------------------------------------
    // フォント名からフォントデータを探すコールバック関数です。
    void* AcquireFontFunctionForComplexFont2(size_t* pOutFontDataSize, const char* pFontName, nn::ui2d::ResType resType, void* pUserData)
    {
        nn::ui2d::IFindableResourceAccessor* pResAccsessor = reinterpret_cast<nn::ui2d::IFindableResourceAccessor*>(pUserData);

        void* pFont = pResAccsessor->FindData(pOutFontDataSize, resType, pFontName);
        NN_SDK_ASSERT(pFont != NULL, "The font resource [%s] was not found.\n", pFontName);
        return pFont;
    }
}

//---------------------------------------------------------------------------
void ScalableFontManager::InitializeComplexFontSet(nn::gfx::Device* pDevice, nn::ui2d::IFindableResourceAccessor& resourceAccessor, RegisterTextureViewSlot registerTextureViewSlot, void* pUserDataForDescriptorSlotAllocator)
{
    // ResourceTypeComplexFont の 数を数える
    int complexFontCount = 0;
    {
        const char* pFileName = NULL;
        pFileName = resourceAccessor.FindFile(nn::ui2d::ResourceTypeComplexFont, pFileName);
        while (pFileName != NULL)
        {
            complexFontCount++;
            pFileName = resourceAccessor.FindFile(nn::ui2d::ResourceTypeComplexFont, pFileName);
        }
    }

    if (complexFontCount == 0)
    {
        return;
    }

    // TextureCache
    {
        nn::font::TextureCache::InitializeArg* pArg = static_cast<nn::font::TextureCache::InitializeArg*>(Layout::AllocateMemory(sizeof(nn::font::TextureCache::InitializeArg)));
        {
            pArg->SetDefault();

            // 複合フォントのファイルを列挙して処理する
            {
                const char* pFileName = NULL;

                pFileName = resourceAccessor.FindFile(nn::ui2d::ResourceTypeComplexFont, pFileName);
                while (pFileName != NULL)
                {
                    const void* pCpxData = resourceAccessor.FindData(NULL, nn::ui2d::ResourceTypeComplexFont, pFileName);
                    (void)ComplexFontHelper::SetupTextureCacheArg(
                        pArg,
                        AcquireFontFunctionForComplexFont2,
                        &resourceAccessor,
                        pCpxData);

                    pFileName = resourceAccessor.FindFile(nn::ui2d::ResourceTypeComplexFont, pFileName);
                }
            }

            pArg->workMemorySize = m_WorkMemorySize;
            pArg->noPlotWorkMemorySize = m_NoPlotWorkMemorySize;
            pArg->allocateFunction = Layout::GetAllocateFunction();
            pArg->textureCacheWidth = m_TextureCacheWidth;
            pArg->textureCacheHeight = m_TextureCacheHeight;
        }

        m_pTextureCache = Layout::AllocateAndConstruct<nn::font::TextureCache>();
        m_pTextureCache->Initialize(pDevice, *pArg);
        m_pTextureCache->RegisterTextureViewToDescriptorPool(registerTextureViewSlot, pUserDataForDescriptorSlotAllocator);
        Layout::FreeMemory(pArg);
    }

    // 複合フォントの初期化。
    {
        m_ComplexFontCount = complexFontCount;
        m_pComplexFontSet = static_cast<ComplexFont*>(Layout::AllocateMemory(sizeof(ComplexFont) * complexFontCount));

        int index = 0;
        int fontFaceHead = 0;
        const char* pFileName = NULL;

        nn::font::TextureCache::InitializeArg* pArg = static_cast<nn::font::TextureCache::InitializeArg*>(Layout::AllocateMemory(sizeof(nn::font::TextureCache::InitializeArg)));
        pArg->SetDefault();

        pFileName = resourceAccessor.FindFile(nn::ui2d::ResourceTypeComplexFont, pFileName);
        while (pFileName != NULL)
        {
            const void* pCpxData = resourceAccessor.FindData(NULL, nn::ui2d::ResourceTypeComplexFont, pFileName);
            const int fontFaceCount = ComplexFontHelper::SetupTextureCacheArg(
                pArg,
                AcquireFontFunctionForComplexFont2,
                &resourceAccessor,
                pCpxData);

            fontFaceHead += fontFaceCount;

            nn::font::Font* pComplexFontTree = ComplexFontHelper::InitializeComplexFontTree(
                pDevice,
                registerTextureViewSlot,
                pUserDataForDescriptorSlotAllocator,
                m_pTextureCache,
                fontFaceHead,
                AcquireFontFunctionForComplexFont2,
                &resourceAccessor,
                pCpxData);

            m_pComplexFontSet[index].pComplexFontTree = pComplexFontTree;

            // リソースに .fcpx として登録します。⇒ 登録は別途行うことにする。
            {
                char* pName = m_pComplexFontSet[index].name;
                memset(pName, 0x0, FontNameMax);

                const char* pDot = strchr(pFileName, '.');
                if (pDot != NULL)
                {
                    ptrdiff_t size = pDot - pFileName;
                    nn::util::Strlcpy(pName, pFileName, static_cast<int>(size + 1));
                    strncat(pName, ".fcpx", FontNameMax - strlen(pFileName) - 1);
                }
                else {
                    nn::util::Strlcpy(pName, pFileName, FontNameMax);
                    strncat(pName, ".fcpx", FontNameMax - strlen(pName) - 1);
                }
            }

            index++;
            pFileName = resourceAccessor.FindFile(nn::ui2d::ResourceTypeComplexFont, pFileName);
        }
        Layout::FreeMemory(pArg);
    }
}

//---------------------------------------------------------------------------
void ScalableFontManager::FinalizeComplexFontSet(nn::gfx::Device* pDevice, UnregisterTextureViewSlot unregisterTextureViewSlot, void* pUserDataForDescriptorSlotAllocator)
{
    if (m_pComplexFontSet != NULL)
    {
        for (int i = 0; i < m_ComplexFontCount; i++)
        {
            ComplexFontHelper::FinalizeComplexFontTree(
                pDevice,
                m_pComplexFontSet[i].pComplexFontTree,
                unregisterTextureViewSlot,
                pUserDataForDescriptorSlotAllocator);
        }

        Layout::FreeMemory(m_pComplexFontSet);

        m_pComplexFontSet = NULL;
        m_ComplexFontCount = 0;
    }

    if (m_pTextureCache != NULL)
    {
        m_pTextureCache->UnregisterTextureViewFromDescriptorPool(unregisterTextureViewSlot, pUserDataForDescriptorSlotAllocator);
        m_pTextureCache->Finalize(pDevice, Layout::GetFreeFunction(), nn::ui2d::Layout::GetUserDataForAllocator());
        Layout::DeleteObj(m_pTextureCache);
        m_pTextureCache = NULL;
    }
}

} // namespace viewer
} // namespace ui2d
} // namespace nn
