﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/util/util_StringUtil.h>

#include <nn/ui2d/ui2d_TextureContainer.h>

#include <nn/ui2d/ui2d_Layout.h>
#include <nn/ui2d/ui2d_Util.h>

namespace nn
{
namespace ui2d
{

//----------------------------------------
TextureRefLink::TextureRefLink(TextureInfo* pTextureInfo, bool isOwned)
    :m_pTexInfo(pTextureInfo)
    ,m_IsOwned(isOwned)
{
    NN_SDK_ASSERT_NOT_NULL(pTextureInfo);
    m_Name[0] = '\0';
}

//----------------------------------------
TextureRefLink::~TextureRefLink()
{
}

//----------------------------------------
void TextureRefLink::Finalize(nn::gfx::Device* pDevice)
{
    if(m_IsOwned)
    {
        m_pTexInfo->Finalize(pDevice);
    }

    Layout::DeleteObj(m_pTexInfo);
}

//----------------------------------------
void
TextureRefLink::SetName(
    const char* pName)
{
    NN_SDK_ASSERT_NOT_NULL(pName);

    NN_SDK_ASSERT(std::strlen(pName) < TexImageNameMax, "Invalid texture name[%s]", pName);
    nn::util::Strlcpy(m_Name, pName, sizeof(m_Name));
}

//----------------------------------------
TextureContainer::~TextureContainer()
{
    NN_SDK_ASSERT(this->empty(), "TextureContainer must be empty before destruction");
}

//----------------------------------------
void
TextureContainer::Finalize(nn::gfx::Device* pDevice)
{
    while (!this->empty())
    {
        TextureRefLink* pLink = &this->front();
        pLink->Finalize(pDevice);
        this->erase(this->iterator_to(*pLink));
        Layout::DeleteObj(pLink);
    }
}

//----------------------------------------
ResourceTextureInfo*
TextureContainer::RegisterResourceTexture(const char* pResourceName)
{
    NN_SDK_ASSERT(pResourceName);

    // リソース用テクスチャとして初期化、アロケートする。
    ResourceTextureInfo* pResourceTextureInfo = Layout::AllocateAndConstruct<ResourceTextureInfo>();
    TextureRefLink* pLink = Layout::AllocateAndConstruct<TextureRefLink>(pResourceTextureInfo, true);
    if (pLink == NULL)
    {
        return NULL;
    }

    pLink->SetName(pResourceName);

    this->push_back(*pLink);

    return pResourceTextureInfo;
}

//----------------------------------------
PlacementTextureInfo*
TextureContainer::RegisterPlacementTexture(const char* pName, bool isOwned)
{
    NN_SDK_ASSERT(pName);

    // リソース用テクスチャとして初期化、アロケートする。
    PlacementTextureInfo* pPlacementTextureInfo = Layout::AllocateAndConstruct<PlacementTextureInfo>();

    TextureRefLink* pLink = Layout::AllocateAndConstruct<TextureRefLink>(pPlacementTextureInfo, isOwned);
    if (pLink == NULL)
    {
        return NULL;
    }

    pLink->SetName(pName);

    this->push_back(*pLink);

    return pPlacementTextureInfo;
}

//----------------------------------------
RenderTargetTextureInfo*
TextureContainer::RegisterRenderTargetTexture(const char* pName, bool isOwned)
{
    NN_SDK_ASSERT(pName);

    // RenderTarget 用テクスチャとして初期化、アロケートする。
    RenderTargetTextureInfo* pRenderTargetTextureInfo = Layout::AllocateAndConstruct<RenderTargetTextureInfo>();

    TextureRefLink* pLink = Layout::AllocateAndConstruct<TextureRefLink>(pRenderTargetTextureInfo, isOwned);
    if (pLink == NULL)
    {
        return NULL;
    }

    pLink->SetName(pName);

    this->push_back(*pLink);

    return pRenderTargetTextureInfo;
}

//----------------------------------------
void
TextureContainer::UnregisterTexture(TextureInfo* pTexInfo)
{
    NN_SDK_ASSERT_NOT_NULL(pTexInfo);

    for (iterator iter = this->begin(); iter != this->end(); ++iter)
    {
        if (&iter->GetTextureInfo() == pTexInfo)
        {
            this->erase(iter);
            Layout::DeleteObj(&(*iter));
            break;
        }
    }
}

//----------------------------------------

void
TextureContainer::RegisterTextureViewToDescriptorPool(RegisterTextureViewSlot regsiterTextureViewSlot, void* pUserData)
{
    for (iterator iter = this->begin(); iter != this->end(); ++iter)
    {
        // 所有していないものは処理しません。
        if(!iter->IsOwned())
        {
            continue;
        }

        // スロット登録済みのテクスチャはスキップする
        if(iter->GetTextureInfo().IsTextureDescriptorSlotReady())
        {
            continue;
        }

        regsiterTextureViewSlot(iter->GetTextureInfo().GetTextureDescriptorSlot(), *(iter->GetTextureInfo().GetTextureView()), pUserData);
    }
}

//----------------------------------------

void
TextureContainer::UnregisterTextureViewFromDescriptorPool(UnregisterTextureViewSlot unregsiterTextureViewSlot, void* pUserData)
{
    for (iterator iter = this->begin(); iter != this->end(); ++iter)
    {
        // 所有していないものは処理しません。
        if(!iter->IsOwned())
        {
            continue;
        }

        // スロットが未設定のテクスチャはスキップする
        if(!iter->GetTextureInfo().IsTextureDescriptorSlotReady())
        {
            continue;
        }

        unregsiterTextureViewSlot(iter->GetTextureInfo().GetTextureDescriptorSlot(), *(iter->GetTextureInfo().GetTextureView()), pUserData);
        iter->GetTextureInfo().ResetTextureDescriptorSlot();
    }
}

//----------------------------------------
const TextureInfo*
TextureContainer::FindTextureByName(const char* pName) const
{
    const_iterator endIter = this->end();
    for (const_iterator iter = this->begin(); iter != endIter; ++iter)
    {
        if (detail::EqualsTexImageName(pName, iter->GetResourceName()))
        {
            return &iter->GetTextureInfo();
        }
    }

    return NULL;
}

} // namespace nn::ui2d
} // namespace nn
