﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Macro.h>

#include <nn/util/util_Arithmetic.h>
#include <nn/util/util_StringUtil.h>

#include <nn/ui2d/ui2d_Common.h>

#include <nn/ui2d/ui2d_Pane.h>
#include <nn/ui2d/ui2d_ResourceAccessor.h>
#include <nn/ui2d/ui2d_DrawInfo.h>
#include <nn/ui2d/ui2d_GraphicsResource.h>
#include <nn/ui2d/ui2d_Layout.h>
#include <nn/ui2d/ui2d_Animation.h>
#include <nn/ui2d/ui2d_Material.h>
#include <nn/util/util_Matrix.h>
#include <nn/ui2d/ui2d_MaterialHelper.h>
#include <nn/ui2d/ui2d_PaneEffect.h>
#include <nn/ui2d/ui2d_StateMachine.h>

namespace nn
{
namespace ui2d
{
namespace detail
{


PaneBase::PaneBase()
{
}

PaneBase::~PaneBase()
{
}

} // namespace nn::ui2d::detail

namespace detail
{
namespace
{

inline void
MultMatrixSrtxST(
    nn::util::MatrixT4x3fType* __restrict pMtx,
    const nn::util::MatrixT4x3fType* __restrict pMtxParent,
    const nn::util::Float2* __restrict pScale,
    const nn::util::Float3* __restrict pTransform)
{
    //(Ma . Mc) . Mb
    //
    // = [ a00  a01  a02  a03 ] . [   1    0    0   cx ] . [ b00    0    0  b03 ]
    //   [                    ]   [                    ]   [                    ]
    //   [ a10  a11  a12  a13 ]   [   0    1    0   cy ]   [   0  b11    0  b13 ]
    //   [                    ]   [                    ]   [                    ]
    //   [ a20  a21  a22  a23 ]   [   0    0    1    0 ]   [   0    0    1  b23 ]
    //   [                    ]   [                    ]   [                    ]
    //   [   0    0    0    1 ]   [   0    0    0    1 ]   [   0    0    0    1 ]
    //
    // = [ a00*b00  a01*b11  a02  a02*b23 + a01*b13 + a00*b03 + a03 + a00 * cx + a01 * cy ]
    //   [                                                                                ]
    //   [ a10*b00  a11*b11  a12  a12*b23 + a11*b13 + a10*b03 + a13 + a10 * cx + a11 * cy ]
    //   [                                                                                ]
    //   [ a20*b00  a21*b11  a22  a22*b23 + a21*b13 + a20*b03 + a23                       ]
    //   [                                                                                ]
    //   [    0        0      0                              1                            ]

    const float b00 = pScale->v[0];
    const float b11 = pScale->v[1];
    const float b03 = pTransform->v[0];
    const float b13 = pTransform->v[1];
    const float b23 = pTransform->v[2];

    nn::util::FloatT4x3 a;
    nn::util::MatrixStore(&a, *pMtxParent);
    nn::util::FloatT4x3 out;

    out.m[0][0] = a.m[0][0] * b00;
    out.m[0][1] = a.m[0][1] * b11;
    out.m[0][2] = a.m[0][2];
    out.m[0][3] = a.m[0][2] * b23 + a.m[0][1] * b13 + a.m[0][0] * b03 + a.m[0][3];

    out.m[1][0] = a.m[1][0] * b00;
    out.m[1][1] = a.m[1][1] * b11;
    out.m[1][2] = a.m[1][2];
    out.m[1][3] = a.m[1][2] * b23 + a.m[1][1] * b13 + a.m[1][0] * b03 + a.m[1][3];

    out.m[2][0] = a.m[2][0] * b00;
    out.m[2][1] = a.m[2][1] * b11;
    out.m[2][2] = a.m[2][2];
    out.m[2][3] = a.m[2][2] * b23 + a.m[2][1] * b13 + a.m[2][0] * b03 + a.m[2][3];

    nn::util::MatrixLoad(pMtx, out);
}

// Scale、Rotate、Transの設定
inline void
MakeMatrixSrt(nn::util::MatrixT4x3fType& o, const nn::util::Float2& s, const float rotateZ, const nn::util::Float3& t)
{
    float sin, cos;

    const nn::util::AngleIndex index = nn::util::DegreeToAngleIndex(rotateZ);
    nn::util::SinCosTable(&sin, &cos, index);

    const float tmp0 = s.v[0] * cos;
    const float tmp1 = -s.v[1] * sin;
    const float tmp2 = s.v[0] * sin;
    const float tmp3 = s.v[1] * cos;

    nn::util::MatrixSet(
        &o,

        tmp0,
        tmp2,
        0.f,

        tmp1,
        tmp3,
        0.f,

        0.f,
        0.f,
        1.f,

        t.v[0],
        t.v[1],
        t.v[2]
    );
}

inline void
MakeMatrixSrt(nn::util::MatrixT4x3fType& o, const nn::util::Float2& s, const nn::util::Float3& r, const nn::util::Float3& t)
{
    float sinx, cosx;
    float siny, cosy;
    float sinz, cosz;

    nn::util::SinCosTable(&sinx, &cosx, nn::util::DegreeToAngleIndex(r.v[0]));
    nn::util::SinCosTable(&siny, &cosy, nn::util::DegreeToAngleIndex(r.v[1]));
    nn::util::SinCosTable(&sinz, &cosz, nn::util::DegreeToAngleIndex(r.v[2]));

    const float opt1 = cosx * cosz;
    const float opt2 = sinx * siny;
    const float opt3 = cosx * sinz;

    const float tmp00 = s.v[0] * (cosy * cosz);
    const float tmp10 = s.v[0] * (cosy * sinz);
    const float tmp20 = s.v[0] * (-siny);

    const float tmp01 = s.v[1] * ((opt2 * cosz) - (opt3));
    const float tmp11 = s.v[1] * ((opt2 * sinz) + (opt1));
    const float tmp21 = s.v[1] * ((sinx * cosy));

    const float tmp02 = (opt1 * siny) + (sinx * sinz);
    const float tmp12 = (opt3 * siny) - (sinx * cosz);
    const float tmp22 = (cosx * cosy);

    nn::util::MatrixSet(
        &o,

        tmp00,
        tmp10,
        tmp20,

        tmp01,
        tmp11,
        tmp21,

        tmp02,
        tmp12,
        tmp22,

        t.v[0],
        t.v[1],
        t.v[2]
    );
}

} // namespace nn::ui2d::detail::{anonymous}
} // namespace nn::ui2d::detail


Pane::Pane()
{
    InitializeParams();

    m_BasePosition =
          HorizontalPosition_Center
        + VerticalPosition_Center * VerticalPosition_MaxVerticalPosition;
    memset(m_Name, 0, sizeof(m_Name));
    memset(m_UserData, 0, sizeof(m_UserData));

    m_Translate.v[0] = 0.0f;
    m_Translate.v[1] = 0.0f;
    m_Translate.v[2] = 0.0f;
    m_Rotate.v[0] = 0.0f;
    m_Rotate.v[1] = 0.0f;
    m_Rotate.v[2] = 0.0f;
    m_Scale.v[0] = 1.0f;
    m_Scale.v[1] = 1.0f;
    m_Size.Set(0.f, 0.f);
    m_Alpha = std::numeric_limits<uint8_t>::max();
    m_GlbAlpha = m_Alpha;

    SetVisible(true);
    SetGlbMtxDirty();
}

Pane::Pane(
    const ResPane* pBlock,
    const BuildArgSet& buildArgSet)
{
    InitializeByResourceBlock(NULL, NULL, pBlock, buildArgSet);
}

Pane::Pane(
    BuildResultInformation* pOutBuildResultInformation,
    nn::gfx::Device* pDevice,
    const ResPane* pBlock,
    const BuildArgSet& buildArgSet)
{
    InitializeByResourceBlock(pOutBuildResultInformation, pDevice, pBlock, buildArgSet);
}

void Pane::InitializeByResourceBlock(
    BuildResultInformation* pOutBuildResultInformation,
    nn::gfx::Device* pDevice,
    const ResPane* pBlock,
    const BuildArgSet& buildArgSet)
{
    InitializeParams();

    const ResVec3* pTranslate = &pBlock->translate;
    const ResVec3* pRotate = &pBlock->rotate;
    const ResVec2* pScale = &pBlock->scale;
    const ResVec2* pSize = &pBlock->size;
    const uint8_t* pAlpha = &pBlock->alpha;
    const char* pUserData = pBlock->userData;

    // ペイン基本情報の上書きを行っている場合、そちらに置き換える
    if (buildArgSet.pOverridePartsPaneBasicInfo)
    {
        if (detail::TestBit(buildArgSet.overrideBasicUsageFlag, BasicOverrideUsageFlag_TranslateEnabled))
        {
            pTranslate = &buildArgSet.pOverridePartsPaneBasicInfo->translate;
        }
        if (detail::TestBit(buildArgSet.overrideBasicUsageFlag, BasicOverrideUsageFlag_RotateEnabled))
        {
            pRotate = &buildArgSet.pOverridePartsPaneBasicInfo->rotate;
        }
        if (detail::TestBit(buildArgSet.overrideBasicUsageFlag, BasicOverrideUsageFlag_ScaleEnabled))
        {
            pScale = &buildArgSet.pOverridePartsPaneBasicInfo->scale;
        }
        if (detail::TestBit(buildArgSet.overrideBasicUsageFlag, BasicOverrideUsageFlag_SizeEnabled))
        {
            pSize = &buildArgSet.pOverridePartsPaneBasicInfo->size;
        }
        if (detail::TestBit(buildArgSet.overrideBasicUsageFlag, BasicOverrideUsageFlag_AlphaEnabled))
        {
            pAlpha = &buildArgSet.pOverridePartsPaneBasicInfo->alpha;
        }
        if (detail::TestBit(buildArgSet.overrideBasicUsageFlag, BasicOverrideUsageFlag_BasicUserDataEnabled))
        {
            pUserData = buildArgSet.pOverridePartsPaneBasicInfo->userData;
        }
    }

    m_BasePosition = pBlock->basePosition;
    SetName(pBlock->name);
    SetUserData(pUserData);

    m_FlagEx = pBlock->flagEx;

    if(detail::TestBit(m_FlagEx, PaneFlagEx_ExtUserDataAnimationEnabled) && buildArgSet.pExtUserDataList != NULL)
    {
        // 拡張ユーザーデータをアニメーションさせるペインの場合はオリジナルの拡張ユーザーデータのコピーを作成してデータの書き換えに備える
        AllocateAndCopyAnimatedExtUserData(buildArgSet.pExtUserDataList);
    }
    else
    {
        SetExtUserDataList(buildArgSet.pExtUserDataList);
    }

    m_Translate.v[0] = pTranslate->x;
    m_Translate.v[1] = pTranslate->y;
    m_Translate.v[2] = pTranslate->z;
    m_Rotate.v[0] = pRotate->x;
    m_Rotate.v[1] = pRotate->y;
    m_Rotate.v[2] = pRotate->z;
    m_Scale.v[0] = pScale->x;
    m_Scale.v[1] = pScale->y;
    if (buildArgSet.pOverrideBuildResSet && (buildArgSet.magnify.v[0] != 1.f || buildArgSet.magnify.v[1] != 1.f))
    {
        // 部品を構築中、部品倍率を考慮する必要がある
        if (detail::TestBit(pBlock->flagEx, PaneFlagEx_IgnorePartsMagnify))
        {
            // 部品倍率を適用しない
            m_Size.width = pSize->x;
            m_Size.height = pSize->y;
        }
        else if (detail::TestBit(pBlock->flagEx, PaneFlagEx_PartsMagnifyAdjustToPartsBound))
        {
            // 部品倍率を部品の境界からの位置が一定になるように適用する
            float compH = (GetBasePositionX() == HorizontalPosition_Center ? 1.0f : 0.5f );
            float compV = (GetBasePositionY() == VerticalPosition_Center ? 1.0f : 0.5f );
            nn::util::Float2 global_scale;
            CalculateScaleFromPartsRoot(&global_scale, buildArgSet.pParentPane);
            float expectPartsSizeX = buildArgSet.partsSize.v[0] * buildArgSet.magnify.v[0];
            float expectPartsSizeY = buildArgSet.partsSize.v[1] * buildArgSet.magnify.v[1];
            float expandH = (expectPartsSizeX - buildArgSet.partsSize.v[0]) * compH;
            float expandV = (expectPartsSizeY - buildArgSet.partsSize.v[1]) * compV;
            m_Size.width = (pSize->x + expandH / (global_scale.v[0] * pScale->x));
            m_Size.height = (pSize->y + expandV / (global_scale.v[1] * pScale->y));
        }
        else
        {
            // 部品倍率を単純にスケール倍して適用する
            m_Size.width = pSize->x * buildArgSet.magnify.v[0];
            m_Size.height = pSize->y * buildArgSet.magnify.v[1];
        }
    }
    else
    {
        m_Size.width = pSize->x;
        m_Size.height = pSize->y;
    }
    m_Alpha = *pAlpha;
    m_GlbAlpha = m_Alpha;

    m_Flag = pBlock->flag;
#if ! defined(NN_SDK_BUILD_RELEASE)
    detail::SetBit(&m_FlagEx, PaneFlagEx_ViewerInvisible, IsViewerInvisible() || buildArgSet.isViewerInvisibleByParent);
#endif

    // 表示/非表示についてはここで上書きを行う
    if (buildArgSet.pOverrideBuildResSet)
    {
        if (detail::TestBit(buildArgSet.overrideBasicUsageFlag, BasicOverrideUsageFlag_VisibleEnabled))
        {
            SetVisible(detail::TestBit(buildArgSet.overrideBasicUsageFlag, BasicOverrideUsageFlag_VisibleValue));
        }
    }

    // システム用拡張ユーザーデータの登録順番に関する問題を回避するためペインエフェクトよりも先に他のシステム用拡張ユーザーデータを登録します。
    const SystemDataProceduralShape* pProceduralShapeData = static_cast<const SystemDataProceduralShape*>(FindSystemExtDataByType(PaneSystemDataType_ProceduralShape));
    if (pProceduralShapeData != NULL)
    {
        // 拡張ユーザーデータとして任意形状データを追加します。
        SystemDataProceduralShapeRuntimeInfo systemData;
        systemData.type = PaneSystemDataType_ProceduralShapeRuntimeInfo;

        AddDynamicSystemExtUserData(static_cast<PaneSystemDataType>(systemData.type), &systemData, sizeof(systemData));
    }

    // ペインエフェクトを初期化する
    size_t constantBufferRequiredSize = InitializePaneEffects(pDevice, buildArgSet);
    if (pOutBuildResultInformation != NULL)
    {
        pOutBuildResultInformation->requiredUi2dConstantBufferSize += constantBufferRequiredSize;
    }

    SetGlbMtxDirty();
}// NOLINT(impl/function_size)


size_t
Pane::InitializePaneEffects(nn::gfx::Device* pDevice, const BuildArgSet& buildArgSet)
{
    size_t additionalConstantBufferSize = 0;

    const SystemDataMaskTexture* pMaskData = static_cast<const SystemDataMaskTexture*>(FindSystemExtDataByType(PaneSystemDataType_MaskTexture));
    const SystemDataDropShadow* pDropShadowData = static_cast<const SystemDataDropShadow*>(FindSystemExtDataByType(PaneSystemDataType_DropShadow));

    if (pMaskData != NULL ||
        pDropShadowData != NULL)
    {
        NN_SDK_ASSERT(
            pDevice != NULL,
            "You must call 4 arguments constructor of the Pane class if the data using pane effect function.");

        detail::PaneEffect* pPaneEffect = Layout::AllocateAndConstruct<detail::PaneEffect>();

        // 拡張ユーザーデータとしてペインエフェクトデータを追加します。
        SystemDataPaneEffectRuntimeInfo systemData;
        systemData.type = PaneSystemDataType_PaneEffect;
        systemData.pPaneEffect = pPaneEffect;

        AddDynamicSystemExtUserData(static_cast<PaneSystemDataType>(systemData.type), &systemData, sizeof(systemData));

        // PaneEffect のシステムデータを追加するとマスクデータなどの位置がずれるため、最終的なポインタはシステムデータを追加した後に処理する。
        pPaneEffect->Initialize(pDevice, this, buildArgSet);
        // ペインエフェクトで使用するコンスタントバッファのサイズを追加します。
        additionalConstantBufferSize += pPaneEffect->GetRequiredConstantBufferSize(pDevice);
    }

    return additionalConstantBufferSize;
}

void
Pane::CopyImpl(const Pane& pane, nn::gfx::Device* pDevice, ResourceAccessor* pResAccessor, const char* pNewRootName, const Layout* pLayout)
{
    m_pParent = NULL;
    m_Translate = pane.m_Translate;
    m_Rotate = pane.m_Rotate;
    m_Scale = pane.m_Scale;
    m_Size = pane.m_Size;
    m_Flag = pane.m_Flag;
    m_Alpha = pane.m_Alpha;
    m_GlbAlpha = pane.m_GlbAlpha;
    m_BasePosition = pane.m_BasePosition;
    m_FlagEx = pane.m_FlagEx;
    m_SystemDataFlags = pane.m_SystemDataFlags;
    m_GlbMtx = pane.m_GlbMtx;
    m_pUserMtx = NULL;
    m_pExtUserDataList = NULL;

    SetName(pane.GetName());
    SetUserData(pane.GetUserData());
    ResetMtx();
    SetGlbMtxDirty();

    // 拡張ユーザーデータアニメーション用領域のコピー
    if (detail::TestBit(m_FlagEx, PaneFlagEx_ExtUserDataAnimationEnabled) ||
        detail::TestBit(m_FlagEx, PaneFlagEx_DynamicExtUserDataEnabled))
    {
        AllocateAndCopyAnimatedExtUserData(pane.m_pExtUserDataList);
    }
    else
    {
        m_pExtUserDataList = pane.m_pExtUserDataList;
    }

    const SystemDataPaneEffectRuntimeInfo* pSrcPaneEffectInfo = static_cast<const SystemDataPaneEffectRuntimeInfo*>(pane.FindSystemExtDataByType(PaneSystemDataType_PaneEffect));

    if (pSrcPaneEffectInfo != NULL)
    {
        detail::PaneEffect* pPaneEffect =
            Layout::AllocateAndConstruct<
            detail::PaneEffect,
            const detail::PaneEffect&,
            const Pane*,
            nn::gfx::Device*,
            ResourceAccessor*,
            const char*,
            const Layout*>(*(pSrcPaneEffectInfo->pPaneEffect), this, pDevice, pResAccessor, pNewRootName, pLayout);

        SystemDataPaneEffectRuntimeInfo* pInfo = static_cast<SystemDataPaneEffectRuntimeInfo*>(const_cast<void*>(FindSystemExtDataByType(PaneSystemDataType_PaneEffect)));
        pInfo->pPaneEffect = pPaneEffect;
    }
}

void
Pane::InitializeParams()
{
    m_pParent = 0;
    m_Flag = 0;
    m_FlagEx = 0;
    m_SystemDataFlags = 0;
    m_pExtUserDataList = 0;
    nn::util::MatrixIdentity(&m_GlbMtx);
    m_pUserMtx = NULL;
}

Pane::~Pane()
{
}

void
Pane::Finalize(nn::gfx::Device* pDevice)
{
    // ペインエフェクト用の拡張ユーザーデータを破棄します。
    if (IsPaneEffectEnabled())
    {
        SystemDataPaneEffectRuntimeInfo* pInfo = static_cast<SystemDataPaneEffectRuntimeInfo*>(const_cast<void*>(FindSystemExtDataByType(PaneSystemDataType_PaneEffect)));

        pInfo->pPaneEffect->Finalize(pDevice);
        Layout::FreeMemory(pInfo->pPaneEffect);
    }

    StateMachine* pStateMachine = GetStateMachine();
    if (pStateMachine != nullptr)
    {
        pStateMachine->Finalize();
        Layout::FreeMemory(pStateMachine);
    }

    // 子供Paneの後処理
    for (PaneList::iterator iter = m_ChildList.begin(); iter != m_ChildList.end();)
    {
        PaneList::iterator currIter = iter++;
        m_ChildList.erase(currIter);
        if (! (*currIter).IsUserAllocated())
        {
            Parts* pParts = DynamicCast<Parts*>(&(*currIter));
            if (pParts && pParts->GetLayout()) {
                // 部品ペインでレイアウトがある場合は、レイアウトをDeleteObjする。
                // 部品ペインは部品レイアウトのルートペインになってるので、一緒に消される
                Layout* pLayout = pParts->GetLayout();
                pLayout->Finalize(pDevice);
                Layout::DeleteObj(pLayout);
            } else {
                (*currIter).Finalize(pDevice);
                Layout::DeleteObj(&(*currIter));
            }
        }
    }

    //  アニメーション用にコピーした拡張ユーザーデータの領域を開放する
    if(detail::TestBit(m_FlagEx, PaneFlagEx_ExtUserDataAnimationEnabled) ||
        detail::TestBit(m_FlagEx, PaneFlagEx_DynamicExtUserDataEnabled))
    {
        Layout::FreeMemory(const_cast<ResExtUserDataList*>(m_pExtUserDataList));
    }
}

void
Pane::SetName(const char* pName)
{
    nn::util::Strlcpy(m_Name, pName, sizeof(m_Name));
}

void
Pane::SetUserData(const char* pUserData)
{
    nn::util::Strlcpy(m_UserData, pUserData, sizeof(m_UserData));
}

void
Pane::AppendChild(Pane* pChild)
{
    InsertChild(m_ChildList.end(), pChild);
}

void
Pane::PrependChild(Pane* pChild)
{
    InsertChild(m_ChildList.begin(), pChild);
}

void
Pane::InsertChild(
    Pane* pNext,
    Pane* pChild
)
{
    NN_SDK_ASSERT(pNext, "pNext must not be NULL for Pane[%s]", GetName());
    NN_SDK_ASSERT(pNext->m_pParent == this, "pNext->m_pParent must be this for Pane[%s]", GetName());

    InsertChild(m_ChildList.iterator_to(*pNext), pChild);
}

void
Pane::InsertChild(
    PaneList::iterator next,
    Pane* pChild
)
{
    NN_SDK_ASSERT(pChild, "pChild must be NULL for Pane[%s]", GetName());
    NN_SDK_ASSERT(pChild->m_pParent == 0, "pChild->m_pParent must be NULL for Pane[%s]", GetName());

    m_ChildList.insert(next, *pChild);
    pChild->m_pParent = this;
    // 追加された子ペインはグローバルマトリックスを計算し直す必要がある
    pChild->SetGlbMtxDirty();
}

void
Pane::RemoveChild(Pane* pChild)
{
    NN_SDK_ASSERT(pChild, "pChild must be NULL for Pane[%s]", GetName());
    NN_SDK_ASSERT(pChild->m_pParent == this, "pChild->m_pParent must be this for Pane[%s]", GetName());

    m_ChildList.erase(m_ChildList.iterator_to(*pChild));
    pChild->m_pParent = NULL;
}

const nn::font::Rectangle
Pane::GetPaneRect() const
{
    nn::font::Rectangle ret;
    const nn::util::Float2& basePt = GetVertexPos();

    ret.left = basePt.v[0];
    ret.top = basePt.v[1];
    ret.right = ret.left + m_Size.width;
    ret.bottom = ret.top - m_Size.height;

    return ret;
}

const nn::util::Unorm8x4
Pane::GetVertexColor(int  idx) const
{
    NN_UNUSED(idx);

    const nn::util::Unorm8x4 result = { { std::numeric_limits<uint8_t>::max(), std::numeric_limits<uint8_t>::max(), std::numeric_limits<uint8_t>::max(), std::numeric_limits<uint8_t>::max() } };
    return result;
}

void
Pane::SetVertexColor(
    int  idx,
    const nn::util::Unorm8x4& value
)
{
    NN_UNUSED(idx);
    NN_UNUSED(value);
}

uint8_t
Pane::GetColorElement(int  idx) const
{
    NN_SDK_ASSERT(idx < AnimTargetPaneColor_MaxAnimTargetPaneColor, "out of bounds: idx[%u] < AnimTargetPaneColor_MaxAnimTargetPaneColor for Pane[%s]", idx, GetName());

    switch (idx)
    {
    case AnimTargetPaneColor_PaneAlpha:
        return m_Alpha;
    default:
        return GetVertexColorElement(idx);
    }
}

void
Pane::SetColorElement(
    int  idx,
    uint8_t  value
)
{
    NN_SDK_ASSERT(idx < AnimTargetPaneColor_MaxAnimTargetPaneColor, "out of bounds: idx[%u] < AnimTargetPaneColor_MaxAnimTargetPaneColor for Pane[%s]", idx, GetName());

    switch (idx)
    {
    case AnimTargetPaneColor_PaneAlpha:
        m_Alpha = value;
        break;
    default:
        SetVertexColorElement(idx, value);
    }
}

uint8_t
Pane::GetVertexColorElement(int  idx) const
{
    NN_UNUSED(idx);
    return std::numeric_limits<uint8_t>::max();
}

void
Pane::SetVertexColorElement(
    int  idx,
    uint8_t  value)
{
    NN_UNUSED(idx);
    NN_UNUSED(value);
}

Pane* Pane::FindPaneByName(const char* pFindName, bool bRecursive)
{
    if (detail::EqualsResName(m_Name, pFindName))
    {
        return this;
    }

    if (bRecursive)
    {
        // 子供に一致するものが無いか検索
        PaneList::iterator endIter = m_ChildList.end();
        for (PaneList::iterator iter = m_ChildList.begin(); iter != endIter; ++iter)
        {
            if (Pane* pPane = (*iter).FindPaneByNameRecursive(pFindName))
            {
                return pPane;
            }
        }
    }

    return 0;
}

const Pane* Pane::FindPaneByName(const char* pFindName, bool bRecursive) const
{
    return const_cast<Pane*>(this)->FindPaneByName(pFindName, bRecursive);
}

Material* Pane::FindMaterialByName(const char* pFindName, bool bRecursive)
{
    int  nbMaterial = GetMaterialCount();
    for (int  idx = 0; idx < nbMaterial; ++idx)
    {
        Material* pMaterial = GetMaterial(idx);
        if (pMaterial)
        {
            if (detail::EqualsMaterialName(pMaterial->GetName(), pFindName))
            {
                return pMaterial;
            }
        }
    }

    if (bRecursive)
    {
        // 子供に一致するものが無いか検索
        PaneList::iterator endIter = m_ChildList.end();
        for (PaneList::iterator iter = m_ChildList.begin(); iter != endIter; ++iter)
        {
            if (Material* pMat = (*iter).FindMaterialByNameRecursive(pFindName))
            {
                return pMat;
            }
        }
    }

    return 0;
}

const Material* Pane::FindMaterialByName(const char* pFindName, bool bRecursive) const
{
    return const_cast<Pane*>(this)->FindMaterialByName(pFindName, bRecursive);
}


bool
Pane::IsConstantBufferUpdateNeeded() const
{
    // 非表示、もしくはアルファ値が 0 になって見えない状態のペインは
    // コンスタントバッファの確保と更新処理は不要なので行わない。
    const bool visibility = IsVisible() && GetGlobalAlpha() > 0;
    // ドロップシャドウの静的レンダリングキャッシュ作成時だけはアルファが 0 でもコンスタントバッファを作成して描画したい。
    const bool paneEffectSpecialCase = IsDropShadowEnabled() && GetPaneEffectInstance()->IsDropShadowCacheRenderingNeeded();

    return visibility || paneEffectSpecialCase;
}

void
Pane::Calculate(DrawInfo& drawInfo, CalculateContext& context, bool isDirtyParentMtx)
{
    if (! IsVisible() && ! context.isInvisiblePaneCalculateMtx)
    {
        // 非表示であることにより Calculate を中断した場合、非表示が解除された
        // ときに正しく自分及び子階層のグローバル行列が計算されるように、自分自身
        // をDirty状態にしておく
        if (isDirtyParentMtx)
        {
            SetGlbMtxDirty();
        }

        SetConstantBufferReady(false);
        return;
    }

    // アルファの計算
    if (context.isInfluenceAlpha && m_pParent)
    {
        m_GlbAlpha = uint8_t (m_Alpha * context.influenceAlpha);
    }
    else
    {
        m_GlbAlpha = m_Alpha;
    }

    if (m_GlbAlpha == 0 && IsInfluencedAlpha() && ! context.isAlphaZeroPaneCalculateMtx)
    {
        // 透明かつ子階層に透明度の影響を与えることにより Calculate を中断した場合、
        // 透明でなくなったときに正しく自分及び子階層のグローバル行列が計算されるように、
        // 自分自身をDirty状態にしておく
        if (isDirtyParentMtx)
        {
            SetGlbMtxDirty();
        }

        SetConstantBufferReady(false);
        return;
    }

    bool isGlbMtxDirty = (this->IsGlbMtxDirty() || isDirtyParentMtx);
    if (this->IsUserGlobalMtx())
    {
        // グローバルマトリックスは常にDirtyと見なす
        isGlbMtxDirty = true;
    }
    else
    {
        // 行列の計算
        if (this->IsUserMtx())
        {
            const nn::util::MatrixT4x3fType* pParentMtx = NULL;
            if (m_pParent)
            {
                pParentMtx = &m_pParent->m_GlbMtx;
            }
            else
            {
                pParentMtx = context.pViewMtx;
            }

            // グローバルマトリックスは常にDirtyと見なす
            isGlbMtxDirty = true;
            nn::util::MatrixMultiply(&m_GlbMtx, *m_pUserMtx, *pParentMtx);
        }
        else if (isGlbMtxDirty)
        {
            // ここで計算されるので、グローバルマトリックスはクリーンになる
            CleanGlbMtx();
            CalculateGlobalMatrixSelf(context);
        }
    }

    /*
        PANEFLAG_INFLUENCEDALPHAが立っていて、自分の透明度が255でないなら、
        CalculateContextに累積透明度(0.f - 1.f)をセットして、子供の計算に影響させる。
    */
    if (IsInfluencedAlpha() && m_Alpha != std::numeric_limits<uint8_t>::max())
    {
        const float crInfluenceAlpha = context.influenceAlpha;
        const bool bCrInfluenced = context.isInfluenceAlpha;
        const float invAlpha = 1.0f / std::numeric_limits<uint8_t>::max();
        context.influenceAlpha = crInfluenceAlpha * m_Alpha * invAlpha;
        context.isInfluenceAlpha = true;

        // 子供の行列計算
        PaneList::iterator endIter = m_ChildList.end();
        for (PaneList::iterator iter = m_ChildList.begin(); iter != endIter; ++iter)
        {
            (*iter).Calculate(drawInfo, context, isGlbMtxDirty);
        }

        // 変更した値を元に戻す
        context.influenceAlpha = crInfluenceAlpha;
        context.isInfluenceAlpha = bCrInfluenced;
    }
    else
    {
        // 子供の行列計算
        PaneList::iterator endIter = m_ChildList.end();
        for (PaneList::iterator iter = m_ChildList.begin(); iter != endIter; ++iter)
        {
            (*iter).Calculate(drawInfo, context, isGlbMtxDirty);
        }
    }

    if (IsPaneEffectEnabled())
    {
        SystemDataPaneEffectRuntimeInfo* pInfo = static_cast<SystemDataPaneEffectRuntimeInfo*>(const_cast<void*>(FindSystemExtDataByType(PaneSystemDataType_PaneEffect)));
        pInfo->pPaneEffect->Calculate(drawInfo);
    }

    SetConstantBufferReady(true);
    SetConstantBufferReadySelf(true);
#if !defined(NN_SDK_BUILD_RELEASE)
    SetCalculationFinishedSelf(true);
#endif
}// NOLINT(impl/function_size)

void
Pane::SetExtUserDataList(const ResExtUserDataList* pBlock)
{
    NN_SDK_ASSERT(m_pExtUserDataList == NULL || !detail::TestBit(m_FlagEx, PaneFlagEx_ExtUserDataAnimationEnabled),
        "You can't change the ExtUserDataList when the pane uses userdata animation.");

    m_pExtUserDataList = pBlock;
    m_SystemDataFlags = 0;

    // システム用拡張ユーザーデータのフラグを更新する。
    UpdateSystemExtDataFlag(m_pExtUserDataList);
}

void
Pane::AddDynamicSystemExtUserDataAllNewImpl(PaneSystemDataType dataType, void* pData, int dataSize)
{
    NN_UNUSED(dataType);

    // 空なので必要なデータを追加するだけ
    // システム用拡張ユーザーデータはユーザーデータのバイナリの中にさらに内容物に関するフォーマットを持っている。
    uint32_t allocateSize = sizeof(ResExtUserDataList) + sizeof(ResExtUserData)
                        + sizeof(ResSystemExtUserData) + sizeof(uint32_t) + dataSize;
    void*   pMemory = Layout::AllocateMemory(allocateSize);
    uint32_t    offset = 0;
    // ResExtUserDataList を初期化
    ResExtUserDataList* pResExtUserDataList = nn::util::BytePtr(pMemory).Get<ResExtUserDataList>();
    pResExtUserDataList->blockHeader.kind = DataBlockKindUserDataList;
    pResExtUserDataList->blockHeader.size = allocateSize;
    pResExtUserDataList->count = 1;
    offset += sizeof(ResExtUserDataList);

    // SystemExtUserData 用の ResExtUserData を初期化
    // 名前は使用しないため offset に 0 を設定している。
    new(nn::util::BytePtr(pMemory, offset).Get<void>()) ResExtUserData(0, sizeof(ResExtUserData), 1, ExtUserDataType_SystemData);
    offset += sizeof(ResExtUserData);

    ResSystemExtUserData* pSystemExtUserData = nn::util::BytePtr(pMemory, offset).Get<ResSystemExtUserData>();
    pSystemExtUserData->version = 0;
    pSystemExtUserData->count = 1;
    offset += sizeof(ResSystemExtUserData);

    // オフセットテーブルには ResSystemExtUserData の先頭からのオフセットが入る。
    uint32_t*   pOffset = nn::util::BytePtr(pMemory, offset).Get<uint32_t>();
    *pOffset = sizeof(ResSystemExtUserData) + sizeof(uint32_t);
    offset += sizeof(uint32_t);

    m_pExtUserDataList = pResExtUserDataList;

    detail::SetBit(&m_FlagEx, PaneFlagEx_DynamicExtUserDataEnabled, 1);

    memcpy(nn::util::BytePtr(pMemory, offset).Get<void>(), pData, dataSize);
}

void
Pane::AddDynamicSystemExtUserDataPartialImpl(PaneSystemDataType dataType, void* pData, int dataSize)
{
    NN_UNUSED(dataType);

    bool createNewSystemData = GetSystemExtDataFlag() == 0;

    // ResExtUserData が存在しないため ResExtUserData を作成して SystemExtUserData を設定する。
    uint32_t originalDataSize = m_pExtUserDataList->blockHeader.size;
    uint32_t increaseSize = sizeof(uint32_t) + dataSize;
    if (createNewSystemData)
    {
        increaseSize += sizeof(ResExtUserData) + sizeof(ResSystemExtUserData);
    }
    void* pMemory = Layout::AllocateMemory(originalDataSize + increaseSize);
    ResExtUserDataList* pResExtUserDataList = nn::util::BytePtr(pMemory).Get<ResExtUserDataList>();

    uint32_t    readOffset = 0;
    uint32_t    writeOffset = 0;

    // ResExtUserDataList をコピー
    memcpy(pResExtUserDataList, m_pExtUserDataList, sizeof(ResExtUserData));
    // 必要な部分を書き換え
    pResExtUserDataList->blockHeader.size = originalDataSize + increaseSize;
    pResExtUserDataList->count = m_pExtUserDataList->count;

    readOffset += sizeof(ResExtUserDataList);
    writeOffset += sizeof(ResExtUserDataList);

    if (createNewSystemData)
    {
        // ResExtUserData にシステムデータ用の領域が存在しないため新たに作成して追加する。

        // システムデータ用にResExtUserData が一つ増加する。
        ++(pResExtUserDataList->count);

        // SystemExtUserData 用の ResExtUserData を初期化
        // 名前は使用しないため offset に 0 を設定している。
        new(nn::util::BytePtr(pMemory, writeOffset).Get<void>()) ResExtUserData(0, sizeof(ResExtUserData) * pResExtUserDataList->count, 1, ExtUserDataType_SystemData);
        writeOffset += sizeof(ResExtUserData);

        // 既存の ResExtUserData をコピーする
        // システム用拡張ユーザーデータは一番最初に追加するため、その後ろにコピーされる既存の拡張ユーザーデータのオフセット値を調整する
        const ResExtUserData* pResExtUserData = nn::util::ConstBytePtr(m_pExtUserDataList, sizeof(*m_pExtUserDataList)).Get<const ResExtUserData>();
        ResExtUserData* pNewResExtUserData = nn::util::BytePtr(pMemory, writeOffset).Get<ResExtUserData>();

        uint32_t addOffset = sizeof(ResSystemExtUserData) + sizeof(uint32_t) + dataSize;

        for (int i = 0; i < m_pExtUserDataList->count; ++i)
        {
            // オフセットをずらしてコピーする
            new(&pNewResExtUserData[i]) ResExtUserData(pResExtUserData[i].GetNameOffset() + addOffset, pResExtUserData[i].GetDataOffset() + addOffset, pResExtUserData[i].GetCount(), static_cast<uint8_t>(pResExtUserData[i].GetType()));
            readOffset += sizeof(ResExtUserData);
            writeOffset += sizeof(ResExtUserData);
        }

        // ResSystemExtUserData を作成する
        ResSystemExtUserData* pSystemExtUserData = nn::util::BytePtr(pMemory, writeOffset).Get<ResSystemExtUserData>();
        pSystemExtUserData->version = 0;
        pSystemExtUserData->count = 1;
        writeOffset += sizeof(ResSystemExtUserData);

        // ResSystemExtUserData 内でのデータへのオフセットを書き込み
        // オフセットテーブルには ResSystemExtUserData の先頭からのオフセットが入る。
        uint32_t*   pOffset = nn::util::BytePtr(pMemory, writeOffset).Get<uint32_t>();
        *pOffset = sizeof(ResSystemExtUserData) + sizeof(uint32_t);
        writeOffset += sizeof(uint32_t);

        // 追加するシステム拡張ユーザーデータをコピーする
        memcpy(nn::util::BytePtr(pMemory, writeOffset).Get<void>(), pData, dataSize);
        writeOffset += dataSize;

        // 既存の拡張ユーザーデータをコピーする
        uint32_t copySize = m_pExtUserDataList->blockHeader.size - (sizeof(pResExtUserDataList) + sizeof(ResExtUserData) * m_pExtUserDataList->count);
        memcpy(nn::util::BytePtr(pMemory, writeOffset).Get<void>(), nn::util::ConstBytePtr(m_pExtUserDataList, readOffset).Get<const void>(), copySize);
    }
    else
    {
        const ResExtUserData* pResExtUserData = nn::util::ConstBytePtr(m_pExtUserDataList, readOffset).Get<const ResExtUserData>();
        ResExtUserData* pNewResExtUserData = nn::util::BytePtr(pMemory, writeOffset).Get<ResExtUserData>();

        int addOffset = sizeof(uint32_t) + dataSize;

        for (int i = 0; i < m_pExtUserDataList->count; ++i)
        {
            int dataOffset = 0;
            if (pResExtUserData[i].GetType() != ExtUserDataType_SystemData)
            {
                dataOffset = addOffset;
            }
            // オフセットをずらしてコピーする
            new(&pNewResExtUserData[i]) ResExtUserData(pResExtUserData[i].GetNameOffset() + addOffset, pResExtUserData[i].GetDataOffset() + dataOffset, pResExtUserData[i].GetCount(), static_cast<uint8_t>(pResExtUserData[i].GetType()));
            readOffset += sizeof(ResExtUserData);
            writeOffset += sizeof(ResExtUserData);
        }

        // 既存のシステム拡張ユーザデータのヘッダとオフセットテーブルをコピーする
        const ResSystemExtUserData* pSystemExtUserData = nn::util::ConstBytePtr(m_pExtUserDataList, readOffset).Get<const ResSystemExtUserData>();
        ResSystemExtUserData* pNewSystemExtUserData = nn::util::BytePtr(pMemory, writeOffset).Get<ResSystemExtUserData>();
        memcpy(pNewSystemExtUserData, pSystemExtUserData, sizeof(ResSystemExtUserData) + sizeof(uint32_t) * pSystemExtUserData->count);

        // 新たにシステム用拡張ユーザーデータを追加するので数を増やす
        ++(pNewSystemExtUserData->count);

        readOffset += (sizeof(ResSystemExtUserData) + sizeof(uint32_t) * pSystemExtUserData->count);
        writeOffset += (sizeof(ResSystemExtUserData));

        // 既存のオフセットテーブルの値を、テーブルの要素追加分を考慮して増やす。
        uint32_t*   pOffsetTable = nn::util::BytePtr(pMemory, writeOffset).Get<uint32_t>();

        for (int i = 0; i < pSystemExtUserData->count; ++i)
        {
            *pOffsetTable += sizeof(uint32_t);
            ++pOffsetTable;
            writeOffset += sizeof(uint32_t);
        }

        uint32_t copySize = 0;

        if (m_pExtUserDataList->count == 1)
        {
            // システムデータだけなのでブロック全体のサイズからコピーする元データのサイズを計算する。
            copySize = m_pExtUserDataList->blockHeader.size - (sizeof(ResExtUserDataList) + sizeof(ResExtUserData) + sizeof(ResSystemExtUserData) + sizeof(uint32_t) * pSystemExtUserData->count);
        }
        else
        {
            copySize = pResExtUserData[1].GetDataOffset() - pResExtUserData[0].GetDataOffset() - sizeof(ResSystemExtUserData) + sizeof(uint32_t) * pSystemExtUserData->count;
        }

        // システム用拡張ユーザーデータのデータオフセットテーブルの末尾に新たに追加するデータのオフセットを書き込む
        // オフセットテーブルには ResSystemExtUserData の先頭からのオフセットが入る。
        uint32_t*   pOffset = nn::util::BytePtr(pMemory, writeOffset).Get<uint32_t>();
        *pOffset = sizeof(ResSystemExtUserData) + sizeof(uint32_t) * pNewSystemExtUserData->count + copySize;
        writeOffset += sizeof(uint32_t);

        // 既存のシステムデータをコピー
        memcpy(nn::util::BytePtr(pMemory, writeOffset).Get<void>(), nn::util::ConstBytePtr(m_pExtUserDataList, readOffset).Get<const void>(), copySize);
        readOffset += copySize;
        writeOffset += copySize;

        // 新たに追加するデータをコピー
        memcpy(nn::util::BytePtr(pMemory, writeOffset).Get<void>(), pData, dataSize);
        writeOffset += dataSize;

        // 残りの拡張ユーザーデータをコピー
        if (m_pExtUserDataList->count > 1)
        {
            memcpy(nn::util::BytePtr(pMemory, writeOffset).Get<void>(), nn::util::ConstBytePtr(m_pExtUserDataList, readOffset).Get<const void>(), m_pExtUserDataList->blockHeader.size - readOffset);
        }
    }

    m_pExtUserDataList = reinterpret_cast<const ResExtUserDataList*>(pMemory);
}// NOLINT(impl/function_size)

void
Pane::AddDynamicSystemExtUserData(PaneSystemDataType dataType, void* pData, int dataSize)
{
    if (m_pExtUserDataList == NULL)
    {
        // ResExtUserDataList から含めてすべて新規作成
        AddDynamicSystemExtUserDataAllNewImpl(dataType, pData, dataSize);
    }
    else
    {
        ResExtUserDataList* pOriginalAllocatedMemory = NULL;

        if (detail::TestBit(m_FlagEx, PaneFlagEx_ExtUserDataAnimationEnabled) ||
            detail::TestBit(m_FlagEx, PaneFlagEx_DynamicExtUserDataEnabled))
        {
            // アニメーションしているか、動的な拡張ユーザーデータが追加されていて動的確保されたメモリ上に存在している。
            if (detail::TestBit(m_FlagEx, PaneFlagEx_DynamicExtUserDataEnabled))
            {
                // 同タイプのデータがすでに追加されていないか確認する。
                if (FindSystemExtDataByType(dataType) != NULL)
                {
                    // すでに存在した場合、Debug 版ではアサートして何も処理しない。
                    NN_SDK_ASSERT(false, "Same type of the SystemExtUserData is already existed[%d].", dataType);
                    return;
                }

                // 別タイプのシステム用拡張ユーザーデータなのでメモリ上のデータを編集して追加する。
            }

            pOriginalAllocatedMemory = const_cast<ResExtUserDataList*>(m_pExtUserDataList);
        }

        // 既存の ResExtUserDataList へ部分的に追加
        AddDynamicSystemExtUserDataPartialImpl(dataType, pData, dataSize);

        if (pOriginalAllocatedMemory)
        {
            Layout::FreeMemory(pOriginalAllocatedMemory);
        }
    }

    detail::SetBit(&m_FlagEx, PaneFlagEx_DynamicExtUserDataEnabled, 1);

    // システム用拡張ユーザーデータのフラグを更新する。
    UpdateSystemExtDataFlag(m_pExtUserDataList);
}



void
Pane::Draw(DrawInfo& drawInfo, nn::gfx::CommandBuffer& commandBuffer)
{
#if !defined(NN_SDK_BUILD_RELEASE)
    SetCalculationFinishedSelf(false);
#endif

    // Visible でない場合は描画をしない
    // Visible かつ、 IsConstantBufferReady を満たさない場合は描画をしない
    // PaneFlag_IsConstantBufferReady が false の場合、該当ペイン以下の子階層すべてが、false とみなして描画をスキップします。
    if (!IsDrawTreeReady())
    {
        return;
    }

    // IsConstantBufferReady の場合は描画する
    if (IsDrawSelfReady())
    {

        if (0 < GetGlobalAlpha()
#if ! defined(NN_SDK_BUILD_RELEASE)
             && !IsViewerInvisible()
#endif
            )
        {
            // ペインエフェクトが有効な場合はそれぞれのエフェクトの最終描画を行う
            if (IsPaneEffectEnabled())
            {
                SystemDataPaneEffectRuntimeInfo* pInfo = static_cast<SystemDataPaneEffectRuntimeInfo*>(const_cast<void*>(FindSystemExtDataByType(PaneSystemDataType_PaneEffect)));
                pInfo->pPaneEffect->Draw(drawInfo, commandBuffer);
            }
            else
            {
                DrawSelf(drawInfo, commandBuffer);
            }
        }
    }

    DrawChildren(drawInfo, commandBuffer);
}

void
Pane::DrawChildren(DrawInfo& drawInfo, nn::gfx::CommandBuffer& commandBuffer)
{
    // 子供の描画。子にアルファが影響する設定でかつ自分のアルファが0なら、子のアルファは0なので描画しない
    if ( ! IsInfluencedAlpha() || m_GlbAlpha != 0)
    {
        PaneList::iterator endIter = m_ChildList.end();
        for (PaneList::iterator iter = m_ChildList.begin(); iter != endIter; ++iter)
        {
            (*iter).Draw(drawInfo, commandBuffer);
        }
    }
}


void
Pane::DrawSelf(DrawInfo& drawInfo, nn::gfx::CommandBuffer& commandBuffer)
{
    NN_UNUSED(drawInfo);
    NN_UNUSED(commandBuffer);
}

void
Pane::BindAnimation(
    AnimTransform* pAnimTrans,
    bool bRecursive,
    bool bEnable
)
{
    NN_SDK_ASSERT(pAnimTrans, "pAnimTrans must not be NULL for Pane[%s]", GetName());

    pAnimTrans->BindPane(this, bRecursive);
    pAnimTrans->SetEnabled(bEnable);
}

bool
Pane::CompareCopiedInstanceTest(const Pane& target) const
{
    /*
    コピーコンストラクタでは親子関係はコピーされないため比較対象から外しています。
    ペインツリーを Clone すると構造も含めてコピーされます。
    if (m_pParent != target.m_pParent)
    {
        return false;
    }

    if (m_ChildList.size() != target.m_ChildList.size())
    {
        return false;
    }
    PaneList::const_iterator endIter = m_ChildList.end();
    PaneList::const_iterator iterOwn = m_ChildList.begin();
    PaneList::const_iterator iterTarget = target.m_ChildList.begin();
    for (; iterOwn != endIter; ++iterOwn, ++iterTarget)
    {
        if ((*iterOwn).CompareCopiedInstanceTest(*iterTarget) == false)
        {
            return false;
        }
    }
    */

    // 下記構造体はパディングが入らない前提で memcmp で比較する。
    if (memcmp(&m_Translate, &target.m_Translate, sizeof(nn::util::Float3)) != 0 ||
        memcmp(&m_Rotate, &target.m_Rotate, sizeof(nn::util::Float3)) != 0 ||
        memcmp(&m_Scale, &target.m_Scale, sizeof(nn::util::Float2)) != 0 ||
        memcmp(&m_Size, &target.m_Size, sizeof(Size)) != 0 ||
        memcmp(&m_GlbMtx, &target.m_GlbMtx, sizeof(nn::util::MatrixT4x3fType)) != 0)
    {
        return false;
    }

    /*
    コピーコンストラクタではフラグがコピーされた後に以下のフラグ操作が行われている。
    同値にならないこともあるので、双方に同様の操作を行い同値になるかチェックする。
    ResetMtx();
        m_Flag = detail::SetBit(m_Flag, PaneFlag_UserMatrix, false);
    SetGlbMtxDirty();
        detail::SetBit(&m_Flag, PaneFlag_IsGlobalMatrixDirty, true);
    */
    uint8_t flagOwn = m_Flag;
    uint8_t flagTarget = target.m_Flag;

    detail::SetBit(&flagOwn, PaneFlag_UserMatrix, false);
    detail::SetBit(&flagOwn, PaneFlag_IsGlobalMatrixDirty, true);
    detail::SetBit(&flagTarget, PaneFlag_UserMatrix, false);
    detail::SetBit(&flagTarget, PaneFlag_IsGlobalMatrixDirty, true);

    if (flagOwn != flagTarget ||
        m_Alpha != target.m_Alpha ||
        m_GlbAlpha != target.m_GlbAlpha ||
        m_BasePosition != target.m_BasePosition ||
        m_FlagEx != target.m_FlagEx)
    {
        return false;
    }

    // 拡張ユーザーデータはポインタをコピーしている。
    if (m_pUserMtx != target.m_pUserMtx ||
        m_pExtUserDataList != target.m_pExtUserDataList)
    {
        return false;
    }

    // アニメーション拡張ユーザーデータはディープコピー。
    if (detail::TestBit(m_FlagEx, PaneFlagEx_ExtUserDataAnimationEnabled))
    {
        if (target.m_pExtUserDataList == NULL)
        {
            return false;
        }

        uint32_t    animatedExtUserDataSize = m_pExtUserDataList->blockHeader.size;
        if (memcmp(m_pExtUserDataList, target.m_pExtUserDataList, animatedExtUserDataSize) != 0)
        {
            return false;
        }
    }

    if (strncmp(m_Name, target.m_Name, ResourceNameStrMax + 1) != 0 ||
        strncmp(m_UserData, target.m_UserData, ResourceNameStrMax + 1) != 0)
    {
        return false;
    }

    return true;
}

void
Pane::UnbindAnimation(
    AnimTransform* pAnimTrans,
    bool bRecursive
)
{
    UnbindAnimationSelf(pAnimTrans);

    // 再帰的にたどる
    if (bRecursive)
    {
        PaneList::iterator endIter = m_ChildList.end();
        for (PaneList::iterator iter = m_ChildList.begin(); iter != endIter; ++iter)
        {
            (*iter).UnbindAnimation(pAnimTrans, bRecursive);
        }
    }
}

void
Pane::UnbindAnimationSelf(AnimTransform* pAnimTrans)
{
    int  nbMaterial = GetMaterialCount();
    for (int  idx = 0; idx < nbMaterial; ++idx)
    {
        Material* pMaterial = GetMaterial(idx);
        if (pMaterial)
        {
            pAnimTrans->UnbindMaterial(pMaterial);
        }
    }

    pAnimTrans->UnbindPane(this);
}

void
Pane::LoadMtx(DrawInfo& drawInfo)
{
    drawInfo.SetModelViewMtx(this->GetGlobalMtx());
}

void Pane::CalculateGlobalMatrix(Pane::CalculateContext& context, bool isDirtyParentMtx)
{
    bool isGlobalMatrixDirty = (this->IsGlbMtxDirty() || isDirtyParentMtx);
    if (isGlobalMatrixDirty && !IsUserGlobalMtx())
    {
        // ここで計算されるので、グローバルマトリックスはクリーンになる
        CleanGlbMtx();
        CalculateGlobalMatrixSelf(context);
    }

    // 子供のグローバルマトリックスの初期化
    PaneList::iterator endIter = m_ChildList.end();
    for (PaneList::iterator iter = m_ChildList.begin(); iter != endIter; ++iter)
    {
        (*iter).CalculateGlobalMatrix(context, isGlobalMatrixDirty);
    }
}

detail::PaneEffect* Pane::GetPaneEffectInstance() const
{
    if (IsPaneEffectEnabled())
    {
        const SystemDataPaneEffectRuntimeInfo* pInfo = static_cast<const SystemDataPaneEffectRuntimeInfo*>(FindSystemExtDataByType(PaneSystemDataType_PaneEffect));
        return pInfo->pPaneEffect;
    }

    return NULL;
}

bool Pane::IsMaskEnabled() const
{
    if (IsPaneEffectEnabled())
    {
        const SystemDataPaneEffectRuntimeInfo* pInfo = static_cast<const SystemDataPaneEffectRuntimeInfo*>(FindSystemExtDataByType(PaneSystemDataType_PaneEffect));
        return pInfo->pPaneEffect->IsMaskFunctionEnabled();
    }

    return false;
}

bool Pane::IsDropShadowEnabled() const
{
    if (IsPaneEffectEnabled())
    {
        const SystemDataPaneEffectRuntimeInfo* pInfo = static_cast<const SystemDataPaneEffectRuntimeInfo*>(FindSystemExtDataByType(PaneSystemDataType_PaneEffect));
        return pInfo->pPaneEffect->IsDropShadowFunctionEnabled();
    }

    return false;
}


const nn::util::Float2
Pane::GetVertexPos() const
{
    nn::util::Float2 basePt = NN_UTIL_FLOAT_2_INITIALIZER(0, 0);

    switch (this->GetBasePositionX())
    {
    case HorizontalPosition_Left:
    default:                        basePt.v[0] = 0;                   break;
    case HorizontalPosition_Center: basePt.v[0] = - m_Size.width / 2;   break;
    case HorizontalPosition_Right:  basePt.v[0] = - m_Size.width;       break;
    }

    switch (this->GetBasePositionY())
    {
    case VerticalPosition_Top:
    default:                        basePt.v[1] = 0;                   break;
    case VerticalPosition_Center:   basePt.v[1] =   m_Size.height / 2;  break;
    case VerticalPosition_Bottom:   basePt.v[1] =   m_Size.height;      break;
    }

    return basePt;
}


Material*
Pane::GetMaterial() const
{
    if (GetMaterialCount() > 0)
    {
        return GetMaterial(0);
    }
    else
    {
        return 0;
    }
}

uint8_t
Pane::GetMaterialCount() const
{
    return 0;
}

Material*
Pane::GetMaterial(int  idx) const
{
    NN_UNUSED(idx);
    NN_DETAIL_UI2D_WARNING(idx < GetMaterialCount(), "idx >= GetMaterialCount() : %d >= %d", idx, GetMaterialCount());

    return 0;
}

uint16_t
Pane::GetExtUserDataCount() const
{
    if (! m_pExtUserDataList)
    {
        return 0;
    }

    if (GetSystemExtDataFlag() != 0)
    {
        return m_pExtUserDataList->count - 1;
    }
    else
    {
        return m_pExtUserDataList->count;
    }
}

const ResExtUserData*
Pane::GetExtUserDataArray() const
{
    if (!m_pExtUserDataList)
    {
        return 0;
    }

    // 最初の拡張ユーザーデータがシステムデータの場合はシステムデータを隠すため一つ後ろの拡張ユーザーデータを返す。
    if (GetSystemExtDataFlag() != 0)
    {
        // 要素数が 1 の時はシステム用ユーザーデータしか存在しないため NULL を返す。
        if (m_pExtUserDataList->count == 1)
        {
            return 0;
        }
        else
        {
            return nn::util::ConstBytePtr(m_pExtUserDataList, sizeof(*m_pExtUserDataList) + sizeof(ResExtUserData)).Get<const ResExtUserData>();
        }
    }
    else
    {
        return nn::util::ConstBytePtr(m_pExtUserDataList, sizeof(*m_pExtUserDataList)).Get<const ResExtUserData>();
    }
}

const ResExtUserData*
Pane::FindExtUserDataByName(const char* pName) const
{
    const ResExtUserData* pExtUserData = GetExtUserDataArray();
    if (! pExtUserData)
    {
        return 0;
    }

    for (int i = 0; i < m_pExtUserDataList->count; ++i, ++pExtUserData)
    {
        if (0 == std::strcmp(pName, pExtUserData->GetName()))
        {
            return pExtUserData;
        }
    }

    return 0;
}

ResExtUserData*
Pane::GetExtUserDataArrayForAnimation() const
{
    if (!detail::TestBit(m_FlagEx, PaneFlagEx_ExtUserDataAnimationEnabled))
    {
        return NULL;
    }

    return const_cast<ResExtUserData*>(GetExtUserDataArray());
}

ResExtUserData*
Pane::FindExtUserDataByNameForAnimation(const char* pName) const
{
    if (!detail::TestBit(m_FlagEx, PaneFlagEx_ExtUserDataAnimationEnabled))
    {
        return NULL;
    }

    return const_cast<ResExtUserData*>(FindExtUserDataByName(pName));
}

Pane* Pane::FindPaneByNameRecursive(const char* pFindName)
{
    if (detail::EqualsResName(m_Name, pFindName))
    {
        return this;
    }

    // 子供に一致するものが無いか検索
    PaneList::iterator endIter = m_ChildList.end();
    for (PaneList::iterator iter = m_ChildList.begin(); iter != endIter; ++iter)
    {
        if (Pane* pPane = (*iter).FindPaneByNameRecursive(pFindName))
        {
            return pPane;
        }
    }

    return NULL;
}

const Pane* Pane::FindPaneByNameRecursive(const char* pFindName) const
{
    return const_cast<Pane*>(this)->FindPaneByNameRecursive(pFindName);
}

Material* Pane::FindMaterialByNameRecursive(const char* pFindName)
{
    int  nbMaterial = GetMaterialCount();
    for (int  idx = 0; idx < nbMaterial; ++idx)
    {
        Material* pMaterial = GetMaterial(idx);
        if (pMaterial)
        {
            if (detail::EqualsMaterialName(pMaterial->GetName(), pFindName))
            {
                return pMaterial;
            }
        }
    }

    // 子供に一致するものが無いか検索
    PaneList::iterator endIter = m_ChildList.end();
    for (PaneList::iterator iter = m_ChildList.begin(); iter != endIter; ++iter)
    {
        if (Material* pMat = (*iter).FindMaterialByNameRecursive(pFindName))
        {
            return pMat;
        }
    }

    return 0;
}

const Material* Pane::FindMaterialByNameRecursive(const char* pFindName) const
{
    return const_cast<Pane*>(this)->FindMaterialByNameRecursive(pFindName);
}

void
Pane::UpdateSystemExtDataFlag(const ResExtUserDataList* pExtUserDataList)
{
    if (pExtUserDataList != NULL)
    {
        const ResExtUserData* pFirstData = nn::util::ConstBytePtr(m_pExtUserDataList, sizeof(*m_pExtUserDataList)).Get<const ResExtUserData>();
        if (pFirstData->GetType() == ExtUserDataType::ExtUserDataType_SystemData)
        {
            int systemDataCount = pFirstData->GetSystemDataCount();
            for (int i = 0; i < systemDataCount; ++i)
            {
                const uint32_t*   pType = static_cast<const uint32_t*>(pFirstData->GetSystemData(i));

                // 有効なデータにビットフラグを立てる。
                m_SystemDataFlags |= (1 << *pType);
            }
        }
    }
}

bool Pane::CheckInvisibleAndUpdateConstantBufferReady()
{
    if (!IsConstantBufferUpdateNeeded())
    {
        if (IsInfluencedAlpha())
        {
            SetConstantBufferReady(false);
        }
        else
        {
            SetConstantBufferReadySelf(false);
        }
        return true; // ペインは非表示状態
    }
    return false; // ペインは表示状態
}

void Pane::CalculateCaptureProjectionMatrix(nn::util::MatrixT4x4fType& mtx) const
{
    const nn::ui2d::Size  size = GetSize();
    const nn::util::Float2& basePt = GetVertexPos();

    nn::util::MatrixOrthographicOffCenterRightHanded(&mtx, basePt.x, basePt.x + size.width, basePt.y, basePt.y - size.height, 0.0f, 500.0f);
}

void Pane::UpdateMaterialConstantBufferForEffectCapture(const DrawInfo& drawInfo)
{
    nn::util::MatrixT4x4fType   projMtx;

    CalculateCaptureProjectionMatrix(projMtx);

    // キャプチャペインの子供ペインをキャプチャするテクスチャの原点付近へ描画するため
    //  親となっているキャプチャペインのグローバルマトリックスに単位行列を設定する。
    nn::util::MatrixT4x3fType glbMtx;

    detail::CalculateCaptureRootMatrix(glbMtx, drawInfo);

    int matCount = GetMaterialCount();

    for (int i = 0; i < matCount; ++i)
    {
        Material* pMat = GetMaterial(i);
        Material::ConstantBufferForVertexShader* pConstantBuffer = pMat->GetConstantBufferForVertexShader(drawInfo);

        nn::util::MatrixStore(reinterpret_cast<nn::util::FloatT4x4*>(pConstantBuffer->projection), projMtx);
        nn::util::MatrixStore(reinterpret_cast<nn::util::FloatT4x3*>(pConstantBuffer->modelView), glbMtx);

        // ドロップシャドウ静的レンダリング時に透明度の適用を最終キャプチャ結果描画時に遅延させるため
        // キャプチャ時の描画では透明度を 255 でレンダリングする。
        if (IsDropShadowEnabled())
        {
            const detail::PaneEffect* pPaneEffect = GetPaneEffectInstance();
            if (pPaneEffect != NULL &&
                pPaneEffect->IsDropShadowCacheRenderingNeeded())
            {
                pConstantBuffer->color[3] = 1.0f / 255.0f;
            }
        }
    }
}

void Pane::UpdateRenderStateForPaneEffectCapture(nn::gfx::CommandBuffer& commandBuffer, const DrawInfo& drawInfo)
{
    commandBuffer.SetBlendState(drawInfo.GetGraphicsResource()->GetPresetBlendState(nn::ui2d::PresetBlendStateId_OpaqueOrAlphaTest));
}

void Pane::CalculateScaleFromPartsRoot(nn::util::Float2* scale, Pane* pane) const
{
    scale->v[0] = 1.f;
    scale->v[1] = 1.f;
    while (pane && ! IsDerivedFrom<nn::ui2d::Parts>(pane)) {
        scale->v[0] = scale->v[0] * pane->GetScale().v[0];
        scale->v[1] = scale->v[1] * pane->GetScale().v[1];
        pane = pane->GetParent();
    }
}

void Pane::CalculateContext::SetDefault()
{
    pRectDrawer = NULL;
    pViewMtx = NULL;
    locationAdjustScale.v[0] = 1.f;
    locationAdjustScale.v[1] = 1.f;
    influenceAlpha = 1.f;
    isLocationAdjust = false;
    isInvisiblePaneCalculateMtx = false;
    isAlphaZeroPaneCalculateMtx = false;
    isInfluenceAlpha = false;
    pLayout = NULL;
}

void Pane::CalculateContext::Set(const DrawInfo& drawInfo, const Layout* pLayoutData)
{
    pRectDrawer = &drawInfo.GetGraphicsResource()->GetFontDrawer();
    pViewMtx = &drawInfo.GetViewMtx();
    locationAdjustScale = drawInfo.GetLocationAdjustScale();
    influenceAlpha = 1.f;
    isLocationAdjust = drawInfo.IsLocationAdjustEnabled();
    isInvisiblePaneCalculateMtx = drawInfo.IsInvisiblePaneCalculated();
    isAlphaZeroPaneCalculateMtx = drawInfo.IsAlphaZeroPaneCalculated();
    isInfluenceAlpha = false;
    pLayout = pLayoutData;
}

void
Pane::AllocateAndCopyAnimatedExtUserData(const ResExtUserDataList* pExtUserDataList)
{
    NN_SDK_ASSERT(
        pExtUserDataList != NULL && this->m_pExtUserDataList == NULL,
        "Invalid function call for Pane[%s]", GetName() );

    // 拡張ユーザーデータアニメーション用領域のコピー
    auto pDstExtUserDataList = static_cast<ResExtUserDataList*>(Layout::AllocateMemory(pExtUserDataList->blockHeader.size));
    memcpy(pDstExtUserDataList, pExtUserDataList, pExtUserDataList->blockHeader.size);

    this->m_pExtUserDataList = pDstExtUserDataList;

    // システム用拡張ユーザーデータのフラグを更新する。
    UpdateSystemExtDataFlag(m_pExtUserDataList);
}

void Pane::CalculateGlobalMatrixSelf(CalculateContext& context)
{
    const nn::util::MatrixT4x3fType* pParentMtx = NULL;
    if (m_pParent)
    {
        pParentMtx = &m_pParent->m_GlbMtx;
    }
    else
    {
        pParentMtx = context.pViewMtx;
    }

    nn::util::Float2 scale = this->GetScale();

    if (context.isLocationAdjust && this->IsLocationAdjust())
    {
        scale.v[0] *= context.locationAdjustScale.v[0];
        scale.v[1] *= context.locationAdjustScale.v[1];
    }

    nn::util::Float3 trans = m_Translate;

    // 親ペイン相対での原点位置を計算する
    switch (GetParentRelativePositionX()) {
    case HorizontalPosition_Left:
        trans.v[0] -= m_pParent->GetSize().width / 2;
        break;
    case HorizontalPosition_Right:
        trans.v[0] += m_pParent->GetSize().width / 2;
        break;
    default:
        break;
    }
    switch (GetParentRelativePositionY()) {
    case VerticalPosition_Top:
        trans.v[1] += m_pParent->GetSize().height / 2;
        break;
    case VerticalPosition_Bottom:
        trans.v[1] -= m_pParent->GetSize().height / 2;
        break;
    default:
        break;
    }

    if (m_Rotate.v[0] != 0 || m_Rotate.v[1] != 0)
    { // XY回転有り
        nn::util::MatrixT4x3fType mtx;
        detail::MakeMatrixSrt(mtx, scale, m_Rotate, trans);
        nn::util::MatrixMultiply(&m_GlbMtx, mtx, *pParentMtx);
    }
    else if (m_Rotate.v[2] != 0)
    { // Z回転有り
        nn::util::MatrixT4x3fType mtx;
        detail::MakeMatrixSrt(mtx, scale, m_Rotate.v[2], trans);
        nn::util::MatrixMultiply(&m_GlbMtx, mtx, *pParentMtx);
    }
    else
    {
        detail::MultMatrixSrtxST(&m_GlbMtx, pParentMtx, &scale, &trans);
    }
}
const SystemDataAlignmentExInfo* Pane::FindAlignmentExInfo() const
{
    if (detail::TestBit(GetSystemExtDataFlag(), PaneSystemDataType_AlignmentExInfo))
    {
        return static_cast<const SystemDataAlignmentExInfo*>(FindSystemExtDataByType(PaneSystemDataType_AlignmentExInfo));
    }

    return NULL;
}

bool Pane::IsAlignmentIgnore()
{
    const SystemDataAlignmentExInfo* pAlignmentExInfo = FindAlignmentExInfo();
    if (pAlignmentExInfo != NULL)
    {
        if (detail::TestBit(pAlignmentExInfo->flags, SystemDataAlignmentExInfo::AlignmentFlags::AlignmentFlags_AlignmentIgnore))
        {
            return true;
        }
    }

    return false;
}

bool Pane::IsAlignmentMarginEnabled()
{
    const SystemDataAlignmentExInfo* pAlignmentExInfo = FindAlignmentExInfo();
    if (pAlignmentExInfo != NULL)
    {
        if (detail::TestBit(pAlignmentExInfo->flags, SystemDataAlignmentExInfo::AlignmentFlags::AlignmentFlags_AlignmentMarginEnabled))
        {
            return true;
        }
    }

    return false;
}

bool Pane::IsAlignmentNullPane()
{
    const SystemDataAlignmentExInfo* pAlignmentExInfo = FindAlignmentExInfo();
    if (pAlignmentExInfo != NULL)
    {
        if (detail::TestBit(pAlignmentExInfo->flags, SystemDataAlignmentExInfo::AlignmentFlags::AlignmentFlags_AlignmentNullPane))
        {
            return true;
        }
    }

    return false;
}

float Pane::GetAlignmentMargin()
{
    const SystemDataAlignmentExInfo* pAlignmentExInfo = FindAlignmentExInfo();
    if (pAlignmentExInfo != NULL)
    {
        return pAlignmentExInfo->alignmentMargin;
    }

    return 0.0f;
}


} // namespace nn::ui2d
} // namespace nn
