﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/ui2d/ui2d_Layout.h>
#include <nn/ui2d/ui2d_ControlCreator.h>
#include <nn/ui2d/ui2d_DragButton.h>

namespace nn
{
namespace ui2d
{

DragButton::DragButton()
: AnimButton()
, m_pRootPane(NULL)
, m_IsMoveHorizontal(true)
, m_IsMoveVertical(true)
{
    m_DragStartPos.v[0] = 0.0f;
    m_DragStartPos.v[1] = 0.0f;
    m_DragStartTranslate.v[0] = 0.0f;
    m_DragStartTranslate.v[1] = 0.0f;

    SetDragMode(DragMode_OnDown);
}

void DragButton::Build(nn::gfx::Device* pDevice, ui2d::Layout* pLayout, const ui2d::ControlSrc& controlSrc)
{
    AnimButton::Build(pDevice, pLayout, controlSrc);

    // Releaseアニメーションの取得
    nn::ui2d::Animator* pAnimator = pLayout->CreateGroupAnimatorAuto(pDevice, controlSrc.FindFunctionalAnimName("Release"), false);
    NN_SDK_ASSERT(pAnimator != NULL, "cannot create CancelAnimator[%s] for Layout[%s]", controlSrc.FindFunctionalAnimName("Release"), pLayout->GetName());
    SetCancelAnimator(pAnimator);

    m_pRootPane = pLayout->GetRootPane();
}

void DragButton::InitializeDragPosition(const nn::util::Float2& pos)
{
    m_DragStartPos = pos;
    const nn::util::Float3& vector = m_pRootPane->GetTranslate();
    m_DragStartTranslate.v[0] = vector.v[0];
    m_DragStartTranslate.v[1] = vector.v[1];
}

void DragButton::UpdateDragPosition(const nn::util::Float2* pPos)
{
    if (pPos) {
        nn::util::Float3 newPos;
        if (m_IsMoveHorizontal) {
            newPos.v[0] = m_DragStartTranslate.v[0] + (pPos->v[0] - m_DragStartPos.v[0]);
        } else {
            newPos.v[0] = m_DragStartTranslate.v[0];
        }
        if (m_IsMoveVertical) {
            newPos.v[1] = m_DragStartTranslate.v[1] + (pPos->v[1] - m_DragStartPos.v[1]);
        } else {
            newPos.v[1] = m_DragStartTranslate.v[1];
        }
        newPos.v[2] = m_pRootPane->GetTranslate().v[2];
        m_pRootPane->SetTranslate(newPos);
    }
}

bool DragButton::ProcessOff()
{
    // STATE_DOWN及びSTATE_DOWN_IDLE中のoffアクションを無視するようにします。
    // また、STATE_CANCEL中のoffアクションは待機するようにします。
    const State currentState = GetState();
    switch (currentState) {
    case State_OffIdle:
        return true;
    case State_On:
        ChangeState(State_Off);
        StartOff();
        return true;
    case State_Off:
        return true;
    case State_OnIdle:
        ChangeState(State_Off);
        StartOff();
        return true;
    case State_Down:
        return true;
    case State_DownIdle:
        return true;
    case State_Cancel:
        return false;
    default:
        NN_SDK_ASSERT(false, "Unknown State.");
        return true;
    };
}

bool DragButton::ProcessCancel()
{
    // Cancelアクションを受け付けるようにします。
    const State currentState = GetState();
    switch (currentState) {
    case State_OffIdle:
        return true;
    case State_On:
        return true;
    case State_Off:
        return true;
    case State_OnIdle:
        return true;
    case State_Down:
        return false;
    case State_DownIdle:
        ChangeState(State_Cancel);
        StartCancel();
        return true;
    case State_Cancel:
        return false;
    default:
        NN_SDK_ASSERT(false, "Unknown State.");
        return true;
    };
}

void DragButton::FinishCancel()
{
    // STATE_CANCELの後は、デフォルトのSTATE_OFF_IDLEではなくSTATE_ON_IDLEに移行します。
    ChangeState(State_OnIdle);
}

} // namespace nn::ui2d
} // namespace nn
