﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/ui2d/ui2d_Common.h>
#include <nn/ui2d/ui2d_Pane.h>
#include <nn/ui2d/ui2d_Group.h>
#include <nn/ui2d/ui2d_Material.h>
#include <nn/ui2d/ui2d_Animation.h>
#include <nn/ui2d/ui2d_Layout.h>
#include <nn/ui2d/ui2d_ResourceAccessor.h>
#include <nn/ui2d/ui2d_Util.h>
#include <nn/ui2d/ui2d_TextBox.h>
#include <nn/ui2d/ui2d_Window.h>
#include <nn/ui2d/detail/ui2d_Log.h>
#include <nn/ui2d/ui2d_StateMachine.h>

#include <nn/perf.h>

namespace
{
const float FrameTolerance = 0.001F;
}

namespace nn
{
namespace ui2d
{
namespace
{

#if defined(NN_DETAIL_ENABLE_SDK_ASSERT)
// アサート有効なら、条件を満たした場合アサートに失敗して停止する。後続のブロックは処理されない。
#define NN_UI2D_FAILSAFE_IF(exp) if(exp) { NN_SDK_ASSERT(#exp); } if(NN_STATIC_CONDITION(false))
#else
#define NN_UI2D_FAILSAFE_IF(exp) if(exp)
#endif

/*---------------------------------------------------------------------------*
  @brief float 型の値を許容値付きで比較します。

  @param[in] a 比較する値です。
  @param[in] b 比較する値です。
  @param[in] tolerance 許容値です。

  @return ２つの値の差の絶対値が許容値未満なら true を返します。
 *---------------------------------------------------------------------------*/
inline bool
RIsSame(
    const float a,
    const float b,
    const float tolerance = 1.0e-5F
)
{
    float c = a - b;
    return (-tolerance < c && c < tolerance);
}

/*---------------------------------------------------------------------------*
  @brief ステップ形式カーブの特定フレームでの値を取得します。

  @param[in] frame 値を取得するフレームです。
  @param[in] pKeys ステップ形式キーの配列です。
  @param[in] keySize キー数です。

  @return 値です。
 *---------------------------------------------------------------------------*/
uint16_t
GetStepCurveValue(
    float frame,
    const ResStepKey* pKeys,
    uint32_t  keySize
)
{
    NN_SDK_ASSERT(keySize > 0, "out of bounds: keySize[%u] > 0", keySize);

    if (keySize == 1 || frame <= pKeys[0].frame)
    {
        return pKeys[0].value;
    }
    else if (frame >= pKeys[keySize - 1].frame)
    {
        return pKeys[keySize - 1].value;
    }

    int ikeyL = 0;
    int ikeyR = (int)keySize - 1;
    while (ikeyL != ikeyR - 1 && ikeyL != ikeyR)
    {
        int ikeyCenter = (ikeyL + ikeyR) / 2;
        const ResStepKey& centerKey = pKeys[ikeyCenter];
        if (frame < centerKey.frame)
        {
            ikeyR = ikeyCenter;
        }
        else
        {
            ikeyL = ikeyCenter;
        }
    }

    if (RIsSame(frame, pKeys[ikeyR].frame, FrameTolerance))
    {
        return pKeys[ikeyR].value;
    }
    else
    {
        return pKeys[ikeyL].value;
    }
}

void
AnimatePaneSrt(
    Pane* pPane,
    const ResAnimationInfo* pAnimInfo,
    const uint32_t* pAnimTargetOffsets,
    float frame
)
{
    for (int i = 0; i < pAnimInfo->count; ++i)
    {
        const ResAnimationTarget* pAnimTarget = nn::util::ConstBytePtr(pAnimInfo, pAnimTargetOffsets[i]).Get<ResAnimationTarget>();
        NN_SDK_ASSERT(pAnimTarget->target < AnimTargetPane_MaxAnimTargetPane);

        float value;
        if (pAnimTarget->curveType == AnimCurve_ParameterizedAnim)
        {
            const ResParameterizedAnim* pAnim = nn::util::ConstBytePtr(pAnimTarget, pAnimTarget->keysOffset).Get<ResParameterizedAnim>();
            float current = pPane->GetSrtElement(pAnimTarget->target);
            value = GetParameterizedAnimValue(frame, current, pAnim);
        }
        else
        {
            NN_SDK_ASSERT(pAnimTarget->curveType == AnimCurve_Hermite);   // 現時点ではHERMITEのみ
            const ResHermiteKey* pKeys = nn::util::ConstBytePtr(pAnimTarget, pAnimTarget->keysOffset).Get<ResHermiteKey>();
            value = GetHermiteCurveValue(frame, pKeys, pAnimTarget->keyCount);
        }
        pPane->SetSrtElement(pAnimTarget->target, value);
    }
}

void
AnimateVisibility(
    Pane* pPane,
    const ResAnimationInfo* pAnimInfo,
    const uint32_t* pAnimTargetOffsets,
    float frame
)
{
    for (int i = 0; i < pAnimInfo->count; ++i)
    {
        const ResAnimationTarget* pAnimTarget = nn::util::ConstBytePtr(pAnimInfo, pAnimTargetOffsets[i]).Get<ResAnimationTarget>();

        NN_SDK_ASSERT(pAnimTarget->target < AnimTargetVisibility_MaxAnimTargetVisibility);
        NN_SDK_ASSERT(pAnimTarget->curveType == AnimCurve_Step);

        const ResStepKey* pKeys = nn::util::ConstBytePtr(pAnimTarget, pAnimTarget->keysOffset).Get<ResStepKey>();
        pPane->SetVisible(0 != GetStepCurveValue(frame, pKeys, pAnimTarget->keyCount));
    }
}

void
AnimateVertexColor(
    Pane* pPane,
    const ResAnimationInfo* pAnimInfo,
    const uint32_t* pAnimTargetOffsets,
    float frame
)
{
    for (int i = 0; i < pAnimInfo->count; ++i)
    {
        const ResAnimationTarget* pAnimTarget = nn::util::ConstBytePtr(pAnimInfo, pAnimTargetOffsets[i]).Get<ResAnimationTarget>();

        NN_SDK_ASSERT(pAnimTarget->target < AnimTargetPaneColor_MaxAnimTargetPaneColor);
        NN_SDK_ASSERT(pAnimTarget->curveType == AnimCurve_Hermite);   // 現時点ではHERMITEのみ

        const ResHermiteKey* pKeys = nn::util::ConstBytePtr(pAnimTarget, pAnimTarget->keysOffset).Get<ResHermiteKey>();
        float value = GetHermiteCurveValue(frame, pKeys, pAnimTarget->keyCount);
        value += 0.5f;  // 四捨五入のため
        uint8_t  u8Val = static_cast<uint8_t >(std::min(std::max(value, 0.f), 255.f));
        pPane->SetColorElement(pAnimTarget->target, u8Val);
    }
}

void
AnimatePerCharacterTransform(
    Pane* pPane,
    const ResAnimationInfo* pAnimInfo,
    const uint32_t* pAnimTargetOffsets,
    float frame
)
{
    for (int i = 0; i < pAnimInfo->count; ++i)
    {
        const ResAnimationTarget* pAnimTarget = nn::util::ConstBytePtr(pAnimInfo, pAnimTargetOffsets[i]).Get<ResAnimationTarget>();

        NN_SDK_ASSERT(pAnimTarget->target < AnimTargetPerCharacterTransform_MaxAnimTargetPerCharacterTransform);
        NN_SDK_ASSERT(pAnimTarget->curveType == AnimCurve_Hermite);   // 現時点ではHERMITEのみ

        const ResHermiteKey* pKeys = nn::util::ConstBytePtr(pAnimTarget, pAnimTarget->keysOffset).Get<ResHermiteKey>();
        float value = GetHermiteCurveValue(frame, pKeys, pAnimTarget->keyCount);
        static_cast<TextBox*>(pPane)->SetPerCharacterTransform(pAnimTarget->target, value);
    }
}

void
AnimateWindow(
    Pane* pPane,
    const ResAnimationInfo* pAnimInfo,
    const uint32_t* pAnimTargetOffsets,
    float frame
)
{
    for (int i = 0; i < pAnimInfo->count; ++i)
    {
        const ResAnimationTarget* pAnimTarget = nn::util::ConstBytePtr(pAnimInfo, pAnimTargetOffsets[i]).Get<ResAnimationTarget>();

        NN_SDK_ASSERT_MINMAX(pAnimTarget->target, AnimTargetWindow_FrameTop, AnimTargetWindow_FrameRight);
        NN_SDK_ASSERT(pAnimTarget->curveType == AnimCurve_Hermite);

        const ResHermiteKey* pKeys = nn::util::ConstBytePtr(pAnimTarget, pAnimTarget->keysOffset).Get<ResHermiteKey>();
        float value = GetHermiteCurveValue(frame, pKeys, pAnimTarget->keyCount);
        static_cast<Window*>(pPane)->SetWindowFrameSize(pAnimTarget->target, static_cast<int>(value));
    }
}

void
AnimateMaskTexSrt(
    Pane* pPane,
    const ResAnimationInfo* pAnimInfo,
    const uint32_t* pAnimTargetOffsets,
    float frame
)
{
    for (int i = 0; i < pAnimInfo->count; ++i)
    {
        const ResAnimationTarget* pAnimTarget = nn::util::ConstBytePtr(pAnimInfo, pAnimTargetOffsets[i]).Get<ResAnimationTarget>();

        NN_SDK_ASSERT(pAnimTarget->target < AnimTargetMaskTexSrt_MaxAnimTargetMask);

        const ResHermiteKey* pKeys = nn::util::ConstBytePtr(pAnimTarget, pAnimTarget->keysOffset).Get<ResHermiteKey>();
        float value = GetHermiteCurveValue(frame, pKeys, pAnimTarget->keyCount);
        pPane->SetMaskTexSrtElement(pAnimTarget->target, value);
    }
}

void
AnimateDropShadow(
    Pane* pPane,
    const ResAnimationInfo* pAnimInfo,
    const uint32_t* pAnimTargetOffsets,
    float frame
)
{
    for (int i = 0; i < pAnimInfo->count; ++i)
    {
        const ResAnimationTarget* pAnimTarget = nn::util::ConstBytePtr(pAnimInfo, pAnimTargetOffsets[i]).Get<ResAnimationTarget>();

        NN_SDK_ASSERT(pAnimTarget->target < AnimTargetDropShadow_MaxAnimTargetDropShadow);

        const ResHermiteKey* pKeys = nn::util::ConstBytePtr(pAnimTarget, pAnimTarget->keysOffset).Get<ResHermiteKey>();
        float value = GetHermiteCurveValue(frame, pKeys, pAnimTarget->keyCount);
        pPane->SetDropShadowElement(pAnimTarget->target, value);
    }
}

void
AnimateProceduralShape(
    Pane* pPane,
    const ResAnimationInfo* pAnimInfo,
    const uint32_t* pAnimTargetOffsets,
    float frame
)
{
    for (int i = 0; i < pAnimInfo->count; ++i)
    {
        const ResAnimationTarget* pAnimTarget = nn::util::ConstBytePtr(pAnimInfo, pAnimTargetOffsets[i]).Get<ResAnimationTarget>();

        NN_SDK_ASSERT(pAnimTarget->target < AnimTargetProceduralShape_MaxAnimTargetProceduralShape);

        const ResHermiteKey* pKeys = nn::util::ConstBytePtr(pAnimTarget, pAnimTarget->keysOffset).Get<ResHermiteKey>();
        float value = GetHermiteCurveValue(frame, pKeys, pAnimTarget->keyCount);
        pPane->SetProceduralShapeElement(pAnimTarget->target, value);
    }
}

void
AnimateStateMachine(
    Pane* pPane,
    const ResAnimationInfo* pAnimInfo,
    const uint32_t* pAnimTargetOffsets,
    float frame
)
{
    for (int targetIdx = 0; targetIdx < pAnimInfo->count; ++targetIdx)
    {
        const ResAnimationTarget* pAnimTarget = nn::util::ConstBytePtr(pAnimInfo, pAnimTargetOffsets[targetIdx]).Get<ResAnimationTarget>();

        NN_SDK_ASSERT(pAnimTarget->target < AnimTargetStateMachineEvent_MaxAnimTargetStateMachineEvent);

        NN_SDK_ASSERT(pAnimTarget->curveType == AnimCurve_ParameterizedAnim);

        const ResParameterizedAnim* pAnim = nn::util::ConstBytePtr(pAnimTarget, pAnimTarget->keysOffset).Get<ResParameterizedAnim>();

        const uint16_t count = pAnim->parameterizedAnimCount;
        for (uint16_t paramIdx = 0; paramIdx < count; paramIdx++)
        {
            const uint32_t* pOffset = nn::util::ConstBytePtr(pAnim, sizeof(*pAnim) + sizeof(uint32_t) * paramIdx).Get<uint32_t>();
            nn::ui2d::ResParameterizedAnimParameter* pParameter = nn::util::BytePtr(const_cast<ResParameterizedAnim*>(pAnim), *pOffset).Get<nn::ui2d::ResParameterizedAnimParameter>();
            if (!IsFrameInRange(frame, pParameter->offset, pParameter->duration))
            {
                continue;
            }

            // 無効なイベント
            if (pParameter->pEvent->pParam1 == nullptr && pParameter->pEvent->pParam2 == nullptr)
            {
                return;
            }

            nn::ui2d::Parts* pParts = nn::ui2d::DynamicCast<nn::ui2d::Parts*, nn::ui2d::Pane>(pPane);
            NN_SDK_ASSERT_NOT_NULL(pParts);

            nn::ui2d::StateMachine* pStateMachine = pParts->GetLayout()->GetStateMachine();
            if (pStateMachine != nullptr)
            {
                pStateMachine->PostEvent(
                    nn::ui2d::StateMachineEventKind_StateChangeRequested,
                    reinterpret_cast<uint64_t>(pParameter->pEvent->pParam1),
                    reinterpret_cast<uint64_t>(pParameter->pEvent->pParam2),
                    0);

                // 一回発火したら無効にする。
                pParameter->pEvent->pParam1 = nullptr;
                pParameter->pEvent->pParam2 = nullptr;
            }
        }
    }
}

void
AnimateMaterialColor(
    Material* pMaterial,
    const ResAnimationInfo* pAnimInfo,
    const uint32_t* pAnimTargetOffsets,
    float frame
)
{
    for (int i = 0; i < pAnimInfo->count; ++i)
    {
        const ResAnimationTarget* pAnimTarget = nn::util::ConstBytePtr(pAnimInfo, pAnimTargetOffsets[i]).Get<ResAnimationTarget>();

        bool    byteData = true;

        // float データ用のカーブは AnimTargetMatColor の最大数より大きな値で
        // AnimTargetMatColorFloat として定義されている。
        if (pAnimTarget->target > AnimTargetMatColor_MaxAnimTargetMatColor)
        {
            byteData = false;
        }

        float value;
        if (pAnimTarget->curveType == AnimCurve_ParameterizedAnim)
        {
            const ResParameterizedAnim* pAnim = nn::util::ConstBytePtr(pAnimTarget, pAnimTarget->keysOffset).Get<ResParameterizedAnim>();
            float current = static_cast<float>(pMaterial->GetColorElement(pAnimTarget->target));
            value = GetParameterizedAnimValue(frame, current, pAnim);
        }
        else
        {
            const ResHermiteKey* pKeys = nn::util::ConstBytePtr(pAnimTarget, pAnimTarget->keysOffset).Get<ResHermiteKey>();
            value = GetHermiteCurveValue(frame, pKeys, pAnimTarget->keyCount);
        }

        if (byteData)
        {
            value += 0.5f;  // 四捨五入のため
            uint8_t  val = static_cast<uint8_t>(std::min(std::max(value, 0.f), 255.f));
            pMaterial->SetColorElement(pAnimTarget->target, val);
        }
        else
        {
            pMaterial->SetColorElementFloat(pAnimTarget->target, value);
        }
    }
}

void
AnimateTextureSrt(
    Material* pMaterial,
    const ResAnimationInfo* pAnimInfo,
    const uint32_t* pAnimTargetOffsets,
    float frame
)
{
    for (int i = 0; i < pAnimInfo->count; ++i)
    {
        const ResAnimationTarget* pAnimTarget = nn::util::ConstBytePtr(pAnimInfo, pAnimTargetOffsets[i]).Get<ResAnimationTarget>();
        const ResHermiteKey* pKeys = nn::util::ConstBytePtr(pAnimTarget, pAnimTarget->keysOffset).Get<ResHermiteKey>();
        pMaterial->SetTexSrtElement(pAnimTarget->id, pAnimTarget->target, GetHermiteCurveValue(frame, pKeys, pAnimTarget->keyCount));
    }
}

void
AnimateTexturePattern(
    Material* pMaterial,
    const ResAnimationInfo* pAnimInfo,
    const uint32_t* pAnimTargetOffsets,
    float frame,
    const TextureInfo** texInfos
)
{
    for (int j = 0; j < pAnimInfo->count; ++j)
    {
        const ResAnimationTarget* pAnimTarget = nn::util::ConstBytePtr(pAnimInfo, pAnimTargetOffsets[j]).Get<ResAnimationTarget>();
        const ResStepKey* pKeys = nn::util::ConstBytePtr(pAnimTarget, pAnimTarget->keysOffset).Get<ResStepKey>();
        const uint16_t  fileIdx = GetStepCurveValue(frame, pKeys, pAnimTarget->keyCount);

        if (texInfos[fileIdx]->IsValid())
        {
            pMaterial->SetTexMap(pAnimTarget->id, texInfos[fileIdx]);
        }
    }
}

void
AnimateIndirectSrt(
    Material*                 pMaterial,
    const ResAnimationInfo*   pAnimInfo,
    const uint32_t*           pAnimTargetOffsets,
    float                     frame
)
{
    for (int i = 0; i < pAnimInfo->count; ++i)
    {
        const ResAnimationTarget* pAnimTarget = nn::util::ConstBytePtr(pAnimInfo, pAnimTargetOffsets[i]).Get<ResAnimationTarget>();
        const ResHermiteKey* pKeys = nn::util::ConstBytePtr(pAnimTarget, pAnimTarget->keysOffset).Get<ResHermiteKey>();
        pMaterial->SetIndirectSrtElement(pAnimTarget->target, GetHermiteCurveValue(frame, pKeys, pAnimTarget->keyCount));
    }
}

void
AnimateAlphaCompare(
    Material* pMaterial,
    const ResAnimationInfo* pAnimInfo,
    const uint32_t* pAnimTargetOffsets,
    float frame
)
{
    for (int i = 0; i < pAnimInfo->count; ++i)
    {
        const ResAnimationTarget* pAnimTarget = nn::util::ConstBytePtr(pAnimInfo, pAnimTargetOffsets[i]).Get<ResAnimationTarget>();
        const ResHermiteKey* pKeys = nn::util::ConstBytePtr(pAnimTarget, pAnimTarget->keysOffset).Get<ResHermiteKey>();
        float value = GetHermiteCurveValue(frame, pKeys, pAnimTarget->keyCount);
        float val = std::min(std::max(value, 0.f), 1.f);
        ResAlphaCompare compare(pMaterial->GetAlphaCompare().GetFunc(), val);
        pMaterial->SetAlphaCompare(compare);
    }
}

void
AnimateFontShadow(
    Material*                   pMaterial,
    const ResAnimationInfo*   pAnimInfo,
    const uint32_t*           pAnimTargetOffsets,
    float                         frame
)
{
    for (int i = 0; i < pAnimInfo->count; ++i)
    {
        const ResAnimationTarget* pAnimTarget = nn::util::ConstBytePtr(pAnimInfo, pAnimTargetOffsets[i]).Get<ResAnimationTarget>();
        const ResHermiteKey* pKeys = nn::util::ConstBytePtr(pAnimTarget, pAnimTarget->keysOffset).Get<ResHermiteKey>();
        float value = GetHermiteCurveValue(frame, pKeys, pAnimTarget->keyCount);
        value += 0.5f;  // 四捨五入のため
        uint8_t  value8 = static_cast<uint8_t >(std::min(std::max(value, 0.f), 255.f));
        pMaterial->SetFontShadowParameterElement(pAnimTarget->target, value8);
    }
}

void
AnimateExtUserData(
    ResExtUserData* pExtUserData,
    const ResAnimationInfo* pAnimInfo,
    const uint32_t* pAnimTargetOffsets,
    float frame
)
{
    for (int i = 0; i < pAnimInfo->count; ++i)
    {
        const ResAnimationTarget* pAnimTarget = nn::util::ConstBytePtr(pAnimInfo, pAnimTargetOffsets[i]).Get<ResAnimationTarget>();

        NN_SDK_ASSERT(pAnimTarget->curveType == AnimCurve_Hermite);   // 現時点ではHERMITEのみ

        const ResHermiteKey* pKeys = nn::util::ConstBytePtr(pAnimTarget, pAnimTarget->keysOffset).Get<ResHermiteKey>();

        switch(pExtUserData->GetType())
        {
        case ExtUserDataType_Int:
            pExtUserData->WriteIntValue(static_cast<int32_t>(static_cast<int32_t>(GetHermiteCurveValue(frame, pKeys, pAnimTarget->keyCount))), pAnimTarget->id);
            break;
        case ExtUserDataType_Float:
            pExtUserData->WriteFloatValue(static_cast<float>(GetHermiteCurveValue(frame, pKeys, pAnimTarget->keyCount)), pAnimTarget->id);
            break;
        case ExtUserDataType_String:
            NN_SDK_ASSERT(false, "ExtUserDataAnimation can not be used at ExtUserDataType_String.");
            break;
        default:
            NN_SDK_ASSERT(false, "Unknown ExtUserDataType type.");
            break;
        }
    }
}

const uint32_t* GetAnimInfoOffsets(const ResAnimationContent& animContent)
{
    if (animContent.type == AnimContentType_ExtUserData)
    {
        // 拡張ユーザーデータアニメーションの場合のみ、AnimContent が持っている AnimationInfo が特殊なフォーマットになっている。
        // AnimContent の直後に AnimationInfo と拡張ユーザーデータ名の文字列テーブルへのオフセットテーブルが書き込まれている。
        // ここではたどって必要な情報を取得する。
        const uint32_t* tableOffsets = nn::util::ConstBytePtr(&animContent, sizeof(animContent)).Get<uint32_t>();
        return nn::util::ConstBytePtr(&animContent, tableOffsets[0]).Get<uint32_t>();
    }
    else
    {
        return nn::util::ConstBytePtr(&animContent, sizeof(animContent)).Get<uint32_t>();
    }
}

const char* GetExtUserDataTargetName(const ResAnimationContent& animContent)
{
    if (animContent.type == AnimContentType_ExtUserData)
    {
        // 拡張ユーザーデータアニメーションの場合のみ、AnimContent が持っている AnimationInfo が特殊なフォーマットになっている。
        // AnimContent の直後に AnimationInfo と拡張ユーザーデータ名の文字列テーブルへのオフセットテーブルが書き込まれている。
        // ここではたどって必要な情報を取得する。
        const uint32_t* tableOffsets = nn::util::ConstBytePtr(&animContent, sizeof(animContent)).Get<uint32_t>();
        const uint32_t* fileNameOffsets = nn::util::ConstBytePtr(&animContent, tableOffsets[1]).Get<uint32_t>();
        return detail::GetStrTableStr(fileNameOffsets, 0);
    }

    return NULL;
}


} // namespace nn::ui2d::{no-name}

AnimTransform::AnimTransform()
: m_pRes(0)
, m_Frame(0)
, m_IsEnabled(true)
{
}

AnimTransform::~AnimTransform()
{
}

uint16_t
AnimTransform::GetFrameSize() const
{
    return m_pRes->frameSize;
}

void
AnimTransform::UpdateFrame(float progressFrame)
{
    NN_UNUSED(progressFrame);
}

void
AnimTransform::SetEnabled(bool bEnable)
{
    m_IsEnabled = bEnable;
}

bool
AnimTransform::IsLoopData() const
{
    return m_pRes->loop != 0;
}

bool
AnimTransform::IsWaitData() const
{
    return m_pRes->frameSize == 0;
}

AnimTransformBasic::AnimTransformBasic()
: m_pTextures(0)
, m_pBindPairs(0)
, m_BindPairCount(0)
, m_BindPairCountMax(0)
{
}

AnimTransformBasic::~AnimTransformBasic()
{
    Layout::DeletePrimArray(m_pBindPairs);
    Layout::DeletePrimArray(m_pTextures);
}

void
AnimTransformBasic::SetResource(
    nn::gfx::Device* pDevice,
    ResourceAccessor* pResAccessor,
    const ResAnimationBlock* pRes
)
{
    NN_SDK_ASSERT_NOT_NULL(pRes);
    SetResource(pDevice, pResAccessor, pRes, pRes->animContCount);
}

void
AnimTransformBasic::SetResource(
    nn::gfx::Device* pDevice,
    ResourceAccessor* pResAccessor,
    const ResAnimationBlock*  pRes,
    uint16_t  animNum
)
{
    NN_SDK_ASSERT(m_pTextures == 0);
    NN_SDK_ASSERT(m_pBindPairs == 0);
    NN_SDK_ASSERT_NOT_NULL(pRes);

    this->SetAnimResource(pRes);
    m_pTextures = 0;

     if (pRes->fileCount > 0)
    {
        NN_SDK_ASSERT_NOT_NULL(pResAccessor);

        m_pTextures = Layout::NewArray<const TextureInfo*>(pRes->fileCount);
        if (m_pTextures)
        {
            const uint32_t* fileNameOffsets = nn::util::ConstBytePtr(pRes, sizeof(*pRes)).Get<uint32_t>();

            for (int i = 0; i < pRes->fileCount; ++i)
            {
                const char *const fileName = detail::GetStrTableStr(fileNameOffsets, i);

                // テクスチャオブジェクトをロードする。
                m_pTextures[i] = pResAccessor->AcquireTexture(pDevice, fileName);
            }
        }
    }

    m_pBindPairs = Layout::NewArray<BindPair>(animNum);
    if (m_pBindPairs)
    {
        m_BindPairCountMax = animNum;
    }
}

void
AnimTransformBasic::BindPane(
    Pane* pPane,
    bool bRecursive
)
{
    NN_SDK_ASSERT_NOT_NULL(pPane);

    const ResAnimationBlock* pRes = this->GetAnimResource();

    const uint32_t *const pAnimContentOffsets = nn::util::ConstBytePtr(pRes, pRes->animContOffsetsOffset).Get<uint32_t>();
    for (uint16_t  i = 0; i < pRes->animContCount; ++i)
    {
        const ResAnimationContent& animContent = *nn::util::ConstBytePtr(pRes, pAnimContentOffsets[i]).Get<ResAnimationContent>();
        switch(animContent.type)
        {
        case AnimContentType_Pane:
        case AnimContentType_StateMachine:
            if (Pane *const pFindPane = pPane->FindPaneByName(animContent.name, bRecursive))
            {
                if (! CheckBindAnimationDoubly(pFindPane, &animContent))
                {
                    if (! BindPaneImpl(pFindPane, &animContent))
                    {
                        break;
                    }
                }
            }
            break;
        case AnimContentType_Material:
            if (Material *const pFindMat = pPane->FindMaterialByName(animContent.name, bRecursive))
            {
                if (! CheckBindAnimationDoubly(pFindMat, &animContent))
                {
                    if (! BindMaterialImpl(pFindMat, &animContent))
                    {
                        break;
                    }
                }
            }
            break;
        case AnimContentType_ExtUserData:
            if (Pane *const pFindPane = pPane->FindPaneByName(animContent.name, bRecursive))
            {
                if (!BindExtUserDataToPane(pFindPane, animContent))
                {
                    break;
                }
            }
            break;
        default:
            NN_SDK_ASSERT(false, "Unknown AnimContentType type.");
            break;
        }
    }
}

void
AnimTransformBasic::BindGroup(
    Group* pGroup
)
{
    NN_SDK_ASSERT_NOT_NULL(pGroup);

    const ResAnimationBlock* pRes = this->GetAnimResource();

    const uint32_t *const pAnimContentOffsets = nn::util::ConstBytePtr(pRes, pRes->animContOffsetsOffset).Get<uint32_t>();
    for (uint16_t  i = 0; i < pRes->animContCount; ++i)
    {
        const ResAnimationContent& animContent = *nn::util::ConstBytePtr(pRes, pAnimContentOffsets[i]).Get<ResAnimationContent>();

        switch(animContent.type)
        {
        case AnimContentType_Pane:
        case AnimContentType_StateMachine:
        {
                // グループに所属しているペインの中から探す
                PaneLinkList::iterator paneEnd = pGroup->GetPaneList().end();
                Pane* targetPane = NULL;
                for (PaneLinkList::iterator iter = pGroup->GetPaneList().begin(); iter != paneEnd; ++iter)
                {
                    if (detail::EqualsResName(animContent.name, iter->pTarget->GetName()))
                    {
                        targetPane = iter->pTarget;
                        break;
                    }
                }
                if (targetPane) {
                    if (! CheckBindAnimationDoubly(targetPane, &animContent))
                    {
                        if (! BindPaneImpl(targetPane, &animContent))
                        {
                            break;
                        }
                    }
                }
            }
            break;
        case AnimContentType_Material:
            {
                // グループに所属しているペインの中からマテリアルを探す
                PaneLinkList& paneList = pGroup->GetPaneList();
                PaneLinkList::iterator paneEnd = paneList.end();
                Material* pFindMat = NULL;
                for (PaneLinkList::iterator iter = paneList.begin(); iter != paneEnd; ++iter)
                {
                    pFindMat = iter->pTarget->FindMaterialByName(animContent.name, false);
                    if (pFindMat)
                    {
                        break;
                    }
                }
                if (pFindMat) {
                    if (! CheckBindAnimationDoubly(pFindMat, &animContent))
                    {
                        if (! BindMaterialImpl(pFindMat, &animContent))
                        {
                            break;
                        }
                    }
                }
            }
            break;
        case AnimContentType_ExtUserData:
            {
                // グループに所属しているペインの中から探す
                PaneLinkList::iterator paneEnd = pGroup->GetPaneList().end();
                Pane* targetPane = NULL;
                for (PaneLinkList::iterator iter = pGroup->GetPaneList().begin(); iter != paneEnd; ++iter)
                {
                    if (detail::EqualsResName(animContent.name, iter->pTarget->GetName()))
                    {
                        targetPane = iter->pTarget;
                        break;
                    }
                }

                if (!BindExtUserDataToPane(targetPane, animContent))
                {
                    break;
                }
            }
            break;
        default:
            NN_SDK_ASSERT(false, "Unknown AnimContentType type.");
            break;
        }
    }
}

void
AnimTransformBasic::BindMaterial(
    Material* pMaterial
)
{
    NN_SDK_ASSERT_NOT_NULL(pMaterial);

    const ResAnimationBlock* pRes = this->GetAnimResource();

    const uint32_t *const pAnimContentOffsets = nn::util::ConstBytePtr(pRes, pRes->animContOffsetsOffset).Get<uint32_t>();
    for (uint32_t  i = 0; i < pRes->animContCount; ++i)
    {
        const ResAnimationContent& animContent = *nn::util::ConstBytePtr(pRes, pAnimContentOffsets[i]).Get<ResAnimationContent>();
        if (animContent.type == AnimContentType_Material)
        {
            if (detail::EqualsMaterialName(pMaterial->GetName(), animContent.name))
            {
                if (! CheckBindAnimationDoubly(pMaterial, &animContent))
                {
                    if (! BindMaterialImpl(pMaterial, &animContent))
                    {
                        break;
                    }
                }
            }
        }
    }
}

void AnimTransformBasic::ForceBindPane(
    Pane* pDstPane,
    const Pane* pSrcPane)
{
    NN_SDK_ASSERT_NOT_NULL(pDstPane);
    NN_SDK_ASSERT_NOT_NULL(pSrcPane);

    const ResAnimationBlock* pRes = this->GetAnimResource();

    const uint32_t *const pAnimContentOffsets = nn::util::ConstBytePtr(pRes, pRes->animContOffsetsOffset).Get<uint32_t>();
    for (uint16_t  i = 0; i < pRes->animContCount; ++i)
    {
        const ResAnimationContent& animContent = *nn::util::ConstBytePtr(pRes, pAnimContentOffsets[i]).Get<ResAnimationContent>();
        switch(animContent.type)
        {
        case AnimContentType_Pane:
            // src側の名前と一致したanimContentをdst側にバインドする
            if (detail::EqualsResName(pSrcPane->GetName(), animContent.name))
            {
                if (! CheckBindAnimationDoubly(pDstPane, &animContent))
                {
                    if (! BindPaneImpl(pDstPane, &animContent))
                    {
                        return;
                    }
                }
            }
            break;
        case AnimContentType_Material:
            {
                uint32_t  nbMaterial = pSrcPane->GetMaterialCount();
                for (uint32_t  idx = 0; idx < nbMaterial; ++idx)
                {
                    const Material* pSrcMaterial = pSrcPane->GetMaterial(idx);
                    if (pSrcMaterial)
                    {
                        // src側の名前と一致したanimContentをdst側にバインドする。
                        // バインドするマテリアルの対象は、同じインデックスのマテリアルとする。
                        if (detail::EqualsMaterialName(pSrcMaterial->GetName(), animContent.name))
                        {
                            Material* pDstMaterial = pDstPane->GetMaterial(idx);
                            if (pDstMaterial)
                            {
                                if (! BindMaterialImpl(pDstMaterial, &animContent))
                                {
                                    return;
                                }
                            }
                        }
                    }
                }
            }
            break;
        case AnimContentType_ExtUserData:
            // src側の名前と一致したanimContentをdst側にバインドする
            if (detail::EqualsResName(pSrcPane->GetName(), animContent.name))
            {
                if (!BindExtUserDataToPane(pDstPane, animContent))
                {
                    return;
                }
            }
            break;
        default:
            NN_SDK_ASSERT(false, "Unknown AnimContentType type.");
            break;
        }
    }
}


bool
AnimTransformBasic::BindExtUserDataToPane(Pane* pPane, const ResAnimationContent& animContent)
{
    if (pPane) {
        const char* const pExtUserDataName = GetExtUserDataTargetName(animContent);
        const ResExtUserData* pUserData = pPane->FindExtUserDataByName(pExtUserDataName);
        if (pUserData != NULL)
        {
            // bind
            if (! CheckBindAnimationDoubly(pUserData, &animContent))
            {
                return BindExtUserDataImpl(const_cast<ResExtUserData*>(pUserData), &animContent);
            }
        }
    }

    return true;
}

void
AnimTransformBasic::UnbindPane(const Pane* pPane)
{
    int end = m_BindPairCount;
    for (int i = 0; i < end; i++)
    {
        BindPair* pPair = &m_pBindPairs[i];
        if (pPair->target == pPane)
        {
            EraseBindPair(i);
            break;
        }

        uint32_t nbExtUserData = pPane->GetExtUserDataCount();
        for (uint32_t extUserDataIndex = 0; extUserDataIndex < nbExtUserData; ++extUserDataIndex)
        {
            const ResExtUserData* pUserData = &(pPane->GetExtUserDataArray()[extUserDataIndex]);
            if (pPair->target == pUserData)
            {
                EraseBindPair(i);
                break;
            }
        }
    }

    uint32_t  nbMaterial = pPane->GetMaterialCount();
    for (uint32_t  idx = 0; idx < nbMaterial; ++idx)
    {
        Material* pMaterial = pPane->GetMaterial(idx);
        if (pMaterial)
        {
            UnbindMaterial(pMaterial);
        }
    }
}

void
AnimTransformBasic::UnbindGroup(const Group* pGroup)
{
    // const Group* から PaneLinkList を取得する方法がないので、const_cast をしています。
    PaneLinkList& paneList = const_cast<Group*>(pGroup)->GetPaneList();
    PaneLinkList::iterator endIter = paneList.end();
    for (PaneLinkList::iterator it = paneList.begin(); it != endIter; ++it)
    {
        UnbindPane(it->pTarget);
    }
}

void
AnimTransformBasic::UnbindMaterial(const Material* pMaterial)
{
    int end = m_BindPairCount;
    for (int i = 0; i < end; i++)
    {
        BindPair* pPair = &m_pBindPairs[i];
        if (pPair->target == pMaterial)
        {
            EraseBindPair(i);
            break;
        }
    }
}

void
AnimTransformBasic::UnbindAll()
{
    m_BindPairCount = 0;
}

//----------------------------------------
bool
AnimTransformBasic::BindPaneImpl(
    Pane* pTarget,
    const ResAnimationContent* pAnimContent
)
{
    NN_SDK_ASSERT_NOT_NULL(pTarget);

    if (m_BindPairCount >= m_BindPairCountMax)
    {
        NN_DETAIL_UI2D_ERROR("all BindPair used.");
        return false;
    }

    BindPair* pPair = &m_pBindPairs[m_BindPairCount];
    m_BindPairCount++;

    pPair->target = pTarget;
    pPair->animCont = pAnimContent;

    return true;
}

//----------------------------------------
bool
AnimTransformBasic::BindMaterialImpl(
    Material* pTarget,
    const ResAnimationContent* pAnimContent
)
{
    NN_SDK_ASSERT_NOT_NULL(pTarget);

    if (m_BindPairCount >= m_BindPairCountMax)
    {
        NN_DETAIL_UI2D_ERROR("all BindPair used.");
        return false;
    }

    bool isBinding = true;

    if (pAnimContent != NULL)
    {
        const uint32_t* animInfoOffsets = GetAnimInfoOffsets(*pAnimContent);
        for (int j = 0; j < pAnimContent->count; ++j)
        {
            const ResAnimationInfo* animInfo = nn::util::ConstBytePtr(pAnimContent, animInfoOffsets[j]).Get<ResAnimationInfo>();
            const uint32_t* pAnimTargetOffsets = nn::util::ConstBytePtr(animInfo, sizeof(*animInfo)).Get<uint32_t>();

            for (int k = 0; k < animInfo->count; ++k)
            {
                const ResAnimationTarget* pAnimTarget = nn::util::ConstBytePtr(animInfo, pAnimTargetOffsets[k]).Get<ResAnimationTarget>();

                switch (animInfo->kind)
                {
                case AnimationTypeMaterialColor:
                    NN_UI2D_FAILSAFE_IF(AnimTargetMatColorFloat_MaxAnimTargetMatColor <= pAnimTarget->target) { isBinding = false; }
                    NN_UI2D_FAILSAFE_IF(pAnimTarget->curveType != AnimCurve_Hermite) { isBinding = false; }   // 現時点ではHERMITEのみ
                    break;
                case AnimationTypeTextureSrt:
                    NN_UI2D_FAILSAFE_IF(TexMapMax <= pAnimTarget->id) { isBinding = false; }
                    NN_UI2D_FAILSAFE_IF(pTarget->GetTexSrtCap() <= pAnimTarget->id) { isBinding = false; }
                    NN_UI2D_FAILSAFE_IF(AnimTargetTexSrt_MaxAnimTargetTexSrt <= pAnimTarget->target) { isBinding = false; }
                    NN_UI2D_FAILSAFE_IF(pAnimTarget->curveType != AnimCurve_Hermite) { isBinding = false; }   // 現時点ではHERMITEのみ
                    break;
                case AnimationTypeTexturePattern:
                    NN_UI2D_FAILSAFE_IF(m_pTextures == NULL) { isBinding = false; }
                    NN_UI2D_FAILSAFE_IF(pTarget->GetTexMapCount() <= pAnimTarget->id) { isBinding = false; }
                    NN_UI2D_FAILSAFE_IF(pAnimTarget->curveType != AnimCurve_Step) { isBinding = false; }              // 現時点ではSTEPのみ
                    NN_UI2D_FAILSAFE_IF(pAnimTarget->target != AnimTargetTexPattern_Image) { isBinding = false; }    // 現状ではimageのみ

                    for (int fileIdx = 0; fileIdx < this->GetAnimResource()->fileCount; ++fileIdx)
                    {
                        NN_UI2D_FAILSAFE_IF(!m_pTextures[fileIdx]->IsValid()) { isBinding = false; break; }
                    }
                    break;
                case AnimationTypeIndirectSrt:
                    NN_UI2D_FAILSAFE_IF(!pTarget->IsIndirectParameterCap()) { isBinding = false; }
                    NN_UI2D_FAILSAFE_IF(AnimTargetIndirectSrt_MaxAnimTargetIndirectSrt <= pAnimTarget->target) { isBinding = false; }
                    NN_UI2D_FAILSAFE_IF(pAnimTarget->curveType != AnimCurve_Hermite) { isBinding = false; }   // 現時点ではHERMITEのみ
                    break;
                case AnimationTypeAlphaCompare:
                    NN_UI2D_FAILSAFE_IF(!pTarget->IsAlphaCompareCap()) { isBinding = false; }
                    NN_UI2D_FAILSAFE_IF(AnimTargetAlphaCompare_MaxAnimTargetAlphaCompare <= pAnimTarget->target) { isBinding = false; }
                    NN_UI2D_FAILSAFE_IF(pAnimTarget->curveType != AnimCurve_Hermite) { isBinding = false; }   // 現時点ではHERMITEのみ
                    break;
                case AnimationTypeFontShadow:
                    NN_UI2D_FAILSAFE_IF(!pTarget->IsFontShadowParameterCap()) { isBinding = false; }
                    NN_UI2D_FAILSAFE_IF(AnimTargetFontShadow_MaxAnimTargetFontShadow <= pAnimTarget->target) { isBinding = false; }
                    NN_UI2D_FAILSAFE_IF(pAnimTarget->curveType != AnimCurve_Hermite) { isBinding = false; }   // 現時点ではHERMITEのみ
                    break;
                default:
                    NN_DETAIL_UI2D_ERROR("Invalid animation type. ");
                    break;
                }
            }
        }
    }

    if (isBinding)
    {
        BindPair* pPair = &m_pBindPairs[m_BindPairCount];
        m_BindPairCount++;

        pPair->target = pTarget;
        pPair->animCont = pAnimContent;
    }

    return true;
}

//----------------------------------------
bool
AnimTransformBasic::BindExtUserDataImpl(
    ResExtUserData* pTarget,
    const ResAnimationContent* pAnimContent
)
{
    NN_SDK_ASSERT_NOT_NULL(pTarget);

    if (m_BindPairCount >= m_BindPairCountMax)
    {
        NN_DETAIL_UI2D_ERROR("all BindPair used.");
        return false;
    }

    BindPair* pPair = &m_pBindPairs[m_BindPairCount];
    m_BindPairCount++;

    pPair->target = pTarget;
    pPair->animCont = pAnimContent;

    return true;
}

void
AnimTransformBasic::Animate()
{
    if (IsEnabled())
    {
        int end = m_BindPairCount;
        for (int i = 0; i < end; i++)
        {
            BindPair* pPair = &m_pBindPairs[i];
            switch(pPair->animCont->type)
            {
            case AnimContentType_Pane:
            case AnimContentType_StateMachine:
                AnimatePaneImpl(reinterpret_cast<Pane*>(pPair->target), pPair->animCont);
                break;
            case AnimContentType_Material:
                AnimateMaterialImpl(reinterpret_cast<Material*>(pPair->target), pPair->animCont);
                break;
            case AnimContentType_ExtUserData:
                AnimateExtUserDataImpl(reinterpret_cast<ResExtUserData*>(pPair->target), pPair->animCont);
                break;
            default:
                NN_SDK_ASSERT(false, "Unknown AnimContentType type.");
                break;
            }
        }
    }
}

void
AnimTransformBasic::AnimatePane(Pane* pPane)
{
    if (IsEnabled())
    {
        int end = m_BindPairCount;
        for (int i = 0; i < end; i++)
        {
            BindPair* pPair = &m_pBindPairs[i];
            if (pPair->target == pPane)
            {
                AnimatePaneImpl(pPane, pPair->animCont);

                uint32_t  nbMaterial = pPane->GetMaterialCount();
                for (uint32_t  idx = 0; idx < nbMaterial; ++idx)
                {
                    Material* pMaterial = pPane->GetMaterial(idx);
                    if (pMaterial)
                    {
                        AnimateMaterial(pMaterial);
                    }
                }
                break;
            }
        }
    }
}

void
AnimTransformBasic::AnimateMaterial(Material* pMaterial)
{
    if (IsEnabled())
    {
        int end = m_BindPairCount;
        for (int i = 0; i < end; i++)
        {
            BindPair* pPair = &m_pBindPairs[i];
            if (pPair->target == pMaterial)
            {
                AnimateMaterialImpl(pMaterial, pPair->animCont);
                break;
            }
        }
    }
}

void
AnimTransformBasic::AnimatePaneImpl(Pane* pPane, const ResAnimationContent* pAnimContent)
{
    NN_PERF_AUTO_MEASURE_INDEX_NAME(0, "nn::ui2d::AnimTransformBasic::Animate_Pane");

    const uint32_t* animInfoOffsets = GetAnimInfoOffsets(*pAnimContent);
    for (int i = 0; i < pAnimContent->count; ++i)
    {
        const ResAnimationInfo* pAnimInfo = nn::util::ConstBytePtr(pAnimContent, animInfoOffsets[i]).Get<ResAnimationInfo>();
        const uint32_t* pAnimTargetOffsets = nn::util::ConstBytePtr(pAnimInfo, sizeof(*pAnimInfo)).Get<uint32_t>();

        switch (pAnimInfo->kind)
        {
        case AnimationTypePaneSrt:
            AnimatePaneSrt(pPane, pAnimInfo, pAnimTargetOffsets, this->GetFrame());
            break;
        case AnimationTypeVisibility:
            AnimateVisibility(pPane, pAnimInfo, pAnimTargetOffsets, this->GetFrame());
            break;
        case AnimationTypeVertexColor:
            AnimateVertexColor(pPane, pAnimInfo, pAnimTargetOffsets, this->GetFrame());
            break;
        case AnimationTypePerCharacterTransform:
            AnimatePerCharacterTransform(pPane, pAnimInfo, pAnimTargetOffsets, this->GetFrame());
            break;
        case AnimationTypeWindow:
            AnimateWindow(pPane, pAnimInfo, pAnimTargetOffsets, this->GetFrame());
            break;
        case AnimationTypeMaskTexSrt:
            AnimateMaskTexSrt(pPane, pAnimInfo, pAnimTargetOffsets, this->GetFrame());
            break;
        case AnimationTypeDropShadow:
            AnimateDropShadow(pPane, pAnimInfo, pAnimTargetOffsets, this->GetFrame());
            break;
        case AnimationTypeProceduralShape:
            AnimateProceduralShape(pPane, pAnimInfo, pAnimTargetOffsets, this->GetFrame());
            break;
        case AnimationTypeStateMachine:
            AnimateStateMachine(pPane, pAnimInfo, pAnimTargetOffsets, this->GetFrame());
            break;
        default:
            NN_DETAIL_UI2D_ERROR("Invalid animation type. ");
            break;
        }
    }
}

void
AnimTransformBasic::AnimateMaterialImpl(Material* pMaterial, const ResAnimationContent* pAnimContent)
{
    NN_PERF_AUTO_MEASURE_INDEX_NAME(0, "nn::ui2d::AnimTransformBasic::Animate_Material");

    const uint32_t* animInfoOffsets = GetAnimInfoOffsets(*pAnimContent);
    for (int i = 0; i < pAnimContent->count; ++i)
    {
        const ResAnimationInfo* pAnimInfo = nn::util::ConstBytePtr(pAnimContent, animInfoOffsets[i]).Get<ResAnimationInfo>();
        const uint32_t* pAnimTargetOffsets = nn::util::ConstBytePtr(pAnimInfo, sizeof(*pAnimInfo)).Get<uint32_t>();

        switch (pAnimInfo->kind)
        {
        case AnimationTypeMaterialColor:
            AnimateMaterialColor(pMaterial, pAnimInfo, pAnimTargetOffsets, this->GetFrame());
            break;
        case AnimationTypeTextureSrt:
            AnimateTextureSrt(pMaterial, pAnimInfo, pAnimTargetOffsets, this->GetFrame());
            break;
        case AnimationTypeTexturePattern:
            AnimateTexturePattern(pMaterial, pAnimInfo, pAnimTargetOffsets, this->GetFrame(), m_pTextures);
            break;
        case AnimationTypeIndirectSrt:
            AnimateIndirectSrt(pMaterial, pAnimInfo, pAnimTargetOffsets, this->GetFrame());
            break;
        case AnimationTypeAlphaCompare:
            AnimateAlphaCompare(pMaterial, pAnimInfo, pAnimTargetOffsets, this->GetFrame());
            break;
        case AnimationTypeFontShadow:
            AnimateFontShadow(pMaterial, pAnimInfo, pAnimTargetOffsets, this->GetFrame());
            break;
        default:
            NN_DETAIL_UI2D_ERROR("Invalid animation type. ");
            break;
        }
    }
}

void
AnimTransformBasic::AnimateExtUserDataImpl(ResExtUserData* pExtUserData, const ResAnimationContent* pAnimContent)
{
    const uint32_t* animInfoOffsets = GetAnimInfoOffsets(*pAnimContent);
    for (int i = 0; i < pAnimContent->count; ++i)
    {
        const ResAnimationInfo* pAnimInfo = nn::util::ConstBytePtr(pAnimContent, animInfoOffsets[i]).Get<ResAnimationInfo>();
        const uint32_t* pAnimTargetOffsets = nn::util::ConstBytePtr(pAnimInfo, sizeof(*pAnimInfo)).Get<uint32_t>();

        AnimateExtUserData(pExtUserData, pAnimInfo, pAnimTargetOffsets, this->GetFrame());
    }
}



bool
AnimTransformBasic::CheckBindAnimationDoubly(const void* target, const ResAnimationContent* pAnimContent) const
{
#if defined(NN_SDK_BUILD_RELEASE)

    NN_UNUSED(target);
    NN_UNUSED(pAnimContent);

#else
    int end = m_BindPairCount;
    for (int i = 0; i < end; i++)
    {
        NN_SDK_ASSERT(m_pBindPairs[i].target != target || m_pBindPairs[i].animCont != pAnimContent,
            "already bind animation[%s].", pAnimContent->name);
    }

#endif  // #if !defined(NN_SDK_BUILD_RELEASE)

    return false;
}

void
AnimTransformBasic::EraseBindPair(int pos)
{
    if (pos + 1 < m_BindPairCount)
    {
        // m_BindPairAry内の順番はアニメーションの挙動には影響しないため、配列の空きを詰める処理は、
        // 一番後ろの要素を持ってくるだけでよい
        m_pBindPairs[pos] = m_pBindPairs[m_BindPairCount - 1];
    }
    m_BindPairCount -= 1;
}

AnimResource::AnimResource()
{
    Initialize();
}

bool
AnimResource::CheckResource() const
{
    if (!m_pResBlock)
    {
        NN_DETAIL_UI2D_ERROR("Animation resource is not set.");
        return false;
    }
    return true;
}

void
AnimResource::Set(const void* pAnimResBuf)
{
    NN_SDK_ASSERT_NOT_NULL(pAnimResBuf);

    Initialize();

    const nn::font::detail::BinaryFileHeader *const pFileHeader = static_cast<const nn::font::detail::BinaryFileHeader*>(pAnimResBuf);

    if (!nn::font::detail::IsValidBinaryFile(pFileHeader, FileSignatureFlan, BinaryFileFormatVersion))
    {
        NN_SDK_ASSERT(false, "not valid layout animation file.");
    }

    m_pFileHeader = pFileHeader;

    const nn::font::detail::BinaryBlockHeader* pDataBlockHead = nn::util::ConstBytePtr(m_pFileHeader, m_pFileHeader->headerSize).Get<nn::font::detail::BinaryBlockHeader>();
    for (int i = 0; i < m_pFileHeader->dataBlocks; ++i)
    {
        uint32_t kind = pDataBlockHead->kind;
        switch (kind)
        {
        case DataBlockKindPaneAnimTag:
            m_pTagBlock = reinterpret_cast<const ResAnimationTagBlock*>(pDataBlockHead);
            break;

        case DataBlockKindPaneAnimShare:
            m_pShareBlock = reinterpret_cast<const ResAnimationShareBlock*>(pDataBlockHead);
            break;

        case DataBlockKindPaneAnimInfo:
            m_pResBlock = reinterpret_cast<const ResAnimationBlock*>(pDataBlockHead);
            break;
        default:
            // 何もしません。
            break;
        }

        // 次のブロック位置へ
        pDataBlockHead = nn::util::ConstBytePtr(pDataBlockHead, pDataBlockHead->size).Get<nn::font::detail::BinaryBlockHeader>();
    }

    NN_DETAIL_UI2D_WARNING(m_pResBlock != NULL, "Animation resource is empty.");
}

void
AnimResource::Initialize()
{
    m_pFileHeader = 0;
    m_pResBlock = 0;
    m_pTagBlock = 0;
    m_pShareBlock = 0;
}

uint16_t
AnimResource::GetTagOrder() const
{
    if (! m_pTagBlock)
    {
        return uint16_t (-1);
    }

    return m_pTagBlock->tagOrder;
}

const char*
AnimResource::GetTagName() const
{
    if (! m_pTagBlock)
    {
        return 0;
    }

    return nn::util::ConstBytePtr(m_pTagBlock, m_pTagBlock->nameOffset).Get<const char>();
}

uint16_t
AnimResource::GetGroupCount() const
{
    if (! m_pTagBlock)
    {
        return 0;
    }

    return m_pTagBlock->groupCount;
}

const ResAnimationGroupRef*
AnimResource::GetGroupArray() const
{
    if (! m_pTagBlock)
    {
        return 0;
    }

    const ResAnimationGroupRef *const groups = nn::util::ConstBytePtr(m_pTagBlock, m_pTagBlock->groupsOffset).Get<const ResAnimationGroupRef>();

    return groups;

}

const ResExtUserDataList*
AnimResource::GetExtUserDataList() const
{
    if (! m_pTagBlock)
    {
        return NULL;
    }

    if(m_pTagBlock->userDataListOffset == 0)
    {
        return NULL;
    }

    const ResExtUserDataList *const list = nn::util::ConstBytePtr(m_pTagBlock, m_pTagBlock->userDataListOffset).Get<const ResExtUserDataList>();

    return list;

}

bool
AnimResource::IsDescendingBind() const
{
    if (! m_pTagBlock)
    {
        return false;
    }

    return detail::TestBit(m_pTagBlock->flag, AnimTagFlag_DescendingBind);
}

uint16_t
AnimResource::GetAnimationShareInfoCount() const
{
    if (! m_pShareBlock)
    {
        return 0;
    }

    return m_pShareBlock->shareCount;
}

const ResAnimationShareInfo*
AnimResource::GetAnimationShareInfoArray() const
{
    if (! m_pShareBlock)
    {
        return 0;
    }

    return nn::util::ConstBytePtr(m_pShareBlock, m_pShareBlock->animShareInfoOffset).Get<const ResAnimationShareInfo>();
}

uint16_t
AnimResource::CalculateAnimationCount(
    Pane* pPane,
    bool bRecursive
) const
{
    if (! CheckResource())
    {
        return 0;
    }

    uint16_t  linkNum = 0;

    const uint32_t *const pAnimContentOffsets = nn::util::ConstBytePtr(m_pResBlock, m_pResBlock->animContOffsetsOffset).Get<uint32_t>();
    for (uint16_t  i = 0; i < m_pResBlock->animContCount; ++i)
    {
        const ResAnimationContent& animContent = *nn::util::ConstBytePtr(m_pResBlock, pAnimContentOffsets[i]).Get<ResAnimationContent>();
        if (animContent.type == AnimContentType_Pane)
        {
            if (pPane->FindPaneByName(animContent.name, bRecursive))
            {
                ++linkNum;
            }
        }
        else
        {
            if (pPane->FindMaterialByName(animContent.name, bRecursive))
            {
                ++linkNum;
            }
        }
    }

    return linkNum;
}

uint16_t
AnimResource::CalculateAnimationCount(Material* pMaterial) const
{
    if (! CheckResource())
    {
        return 0;
    }

    uint16_t  linkNum = 0;

    const uint32_t *const pAnimContentOffsets = nn::util::ConstBytePtr(m_pResBlock, m_pResBlock->animContOffsetsOffset).Get<uint32_t>();
    for (uint16_t  i = 0; i < m_pResBlock->animContCount; ++i)
    {
        const ResAnimationContent& animContent = *nn::util::ConstBytePtr(m_pResBlock, pAnimContentOffsets[i]).Get<ResAnimationContent>();
        if (animContent.type == AnimContentType_Material)
        {
            if (detail::EqualsMaterialName(pMaterial->GetName(), animContent.name))
            {
                ++linkNum;
                break;  // 同じ名前の AnimationContent が複数存在することはないため、
                // 見つかったらすぐに抜ける。
            }
        }
    }

    return linkNum;
}

uint16_t
AnimResource::CalculateAnimationCount(
    Group* pGroup,
    bool bRecursive
) const
{
    NN_SDK_ASSERT_NOT_NULL(pGroup);

    uint16_t  linkNum = 0;

    PaneLinkList& paneList = pGroup->GetPaneList();
    PaneLinkList::iterator endIter = paneList.end();
    for (PaneLinkList::iterator iter = paneList.begin(); iter != endIter; ++iter)
    {
        linkNum += CalculateAnimationCount(iter->pTarget, bRecursive);
    }

    return linkNum;
}

namespace detail
{

AnimPaneTree::AnimPaneTree()
{
    Initialize();
}

AnimPaneTree::AnimPaneTree(
    Pane* pTargetPane,
    const AnimResource& animRes)
{
    Initialize();
    Set(pTargetPane, animRes);
}

const ResAnimationContent*
AnimPaneTree::FindAnimContent(
    const ResAnimationBlock* pAnimBlock,
    const char* pAnimContentName,
    uint8_t  animContentType
)
{
    NN_SDK_ASSERT_NOT_NULL(pAnimBlock);

    const uint32_t *const pAnimContentOffsets = nn::util::ConstBytePtr(pAnimBlock, pAnimBlock->animContOffsetsOffset).Get<uint32_t>();
    for (uint16_t  i = 0; i < pAnimBlock->animContCount; ++i)
    {
        const ResAnimationContent *const pAnimCont = nn::util::ConstBytePtr(pAnimBlock, pAnimContentOffsets[i]).Get<ResAnimationContent>();
        if (pAnimCont->type == animContentType &&
            detail::EqualsMaterialName(pAnimCont->name, pAnimContentName))
        {
            return pAnimCont;
        }
    }

    return NULL;
}

void
AnimPaneTree::Initialize()
{
    m_LinkCount = 0;

    m_pPaneAnimContent = NULL;
    m_AnimMatCnt = 0;
    for (uint8_t  i = 0; i < MaterialCountMax; ++i)
    {
        m_pMatAnimContents[i] = NULL;
    }
}

void
AnimPaneTree::Set(
    Pane* pTargetPane,
    const AnimResource& animRes
)
{
    NN_SDK_ASSERT_NOT_NULL(pTargetPane);

    uint16_t  linkNum = 0;
    const ResAnimationBlock* pAnimBlock = animRes.GetResourceBlock();

    const ResAnimationContent* pAnimContent = FindAnimContent(pAnimBlock, pTargetPane->GetName(), AnimContentType_Pane);
    if (pAnimContent != NULL)
    {
        ++linkNum;
    }

    // マテリアルの処理
    const uint8_t  animMatCnt = pTargetPane->GetMaterialCount();
    NN_SDK_ASSERT(animMatCnt <= MaterialCountMax);
    const ResAnimationContent* animMatIdxs[MaterialCountMax];

    for (uint8_t  i = 0; i < animMatCnt; ++i)
    {
        animMatIdxs[i] = FindAnimContent(pAnimBlock, pTargetPane->GetMaterial(i)->GetName(), AnimContentType_Material);
        if (animMatIdxs[i] != NULL)
        {
            ++linkNum;
        }
    }

    // アニメーション対象がひとつも無い
    if (linkNum == 0)
    {
        return;
    }

    m_AnimRes = animRes;
    m_pPaneAnimContent = pAnimContent;
    m_AnimMatCnt = animMatCnt;
    for (uint8_t  i = 0; i < animMatCnt; ++i)
    {
        m_pMatAnimContents[i] = animMatIdxs[i];
    }
    m_LinkCount = linkNum;
}

AnimTransform*
AnimPaneTree::Bind(
    nn::gfx::Device* pDevice,
    Layout* pLayout,
    Pane* pTargetPane,
    ResourceAccessor* pResAccessor
) const
{
    NN_SDK_ASSERT_NOT_NULL(pLayout);
    NN_SDK_ASSERT_NOT_NULL(pTargetPane);

    AnimTransformBasic *const pAnimTrans = static_cast<AnimTransformBasic*>(pLayout->CreateAnimTransformBasic());
    pAnimTrans->SetResource(pDevice, pResAccessor, m_AnimRes.GetResourceBlock(), m_LinkCount);

    // ペインのアニメーションのバインド
    if (m_pPaneAnimContent != NULL)
    {
        pAnimTrans->BindPaneImpl(pTargetPane, m_pPaneAnimContent);
    }

    // マテリアルアニメーションのバインド
    // ※マテリアルの個数が共有元ペインと等しいとは限りません。
    const uint32_t  animMatMax = std::min(m_AnimMatCnt, pTargetPane->GetMaterialCount());
    for (uint32_t  i = 0; i < animMatMax; ++i)
    {
        if (m_pMatAnimContents[i] != NULL)
        {
            Material *const pMaterial = pTargetPane->GetMaterial(i);
            NN_SDK_ASSERT_NOT_NULL(pMaterial);
            if (! pAnimTrans->BindMaterialImpl(pMaterial, m_pMatAnimContents[i]))
            {
                break;
            }
        }
    }

    return pAnimTrans;
}

} // namespace nn::ui2d::detail
} // namespace nn::ui2d
} // namespace nn
