﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/ui2d/ui2d_Alignment.h>

#include <nn/ui2d/ui2d_TextBox.h>
#include <nn/ui2d/ui2d_Window.h>
#include <nn/ui2d/ui2d_Parts.h>
#include <nn/ui2d/ui2d_Picture.h>

namespace nn
{
namespace ui2d
{

namespace {

class AlignInfo
{
public:
    AlignInfo()
    {
        Length = 0.0f;
        Margin = 0.0f;
        CenterOffset = 0.0f;
        pPane = NULL;
    }

    float Length;
    float Margin;
    float CenterOffset;
    Pane *pPane;
};

void DoHorizontalExpand_(AlignInfo *pInfo, float width, bool isLeftEdge)
{
    NN_SDK_ASSERT_NOT_NULL(pInfo->pPane);

    Size size = pInfo->pPane->GetSize();
    size.Set(width, size.height);
    pInfo->pPane->SetSize(size);

    const uint8_t basePositionX = pInfo->pPane->GetBasePositionX();
    nn::util::Float3 position = pInfo->pPane->GetTranslate();
    if (isLeftEdge)
    {
        if (basePositionX == HorizontalPosition_Center)
        {
            pInfo->pPane->SetTranslate(NN_UTIL_FLOAT_3_INITIALIZER(position.x - (width / 2.0f), position.y, position.z));
        }
        else if (basePositionX == HorizontalPosition_Left)
        {
            pInfo->pPane->SetTranslate(NN_UTIL_FLOAT_3_INITIALIZER(position.x - width, position.y, position.z));
        }
    }
    else
    {
        if (basePositionX == HorizontalPosition_Center)
        {
            pInfo->pPane->SetTranslate(NN_UTIL_FLOAT_3_INITIALIZER(position.x + (width / 2.0f), position.y, position.z));
        }
        else if (basePositionX == HorizontalPosition_Right)
        {
            pInfo->pPane->SetTranslate(NN_UTIL_FLOAT_3_INITIALIZER(position.x + width, position.y, position.z));
        }
    }
}

void DoVerticalExpand_(AlignInfo *pInfo, float height, bool isTopEdge)
{
    NN_SDK_ASSERT_NOT_NULL(pInfo->pPane);

    Size size = pInfo->pPane->GetSize();
    size.Set(size.width, height);
    pInfo->pPane->SetSize(size);

    const uint8_t basePositionY = pInfo->pPane->GetBasePositionY();
    nn::util::Float3 position = pInfo->pPane->GetTranslate();
    if (isTopEdge)
    {
        if (basePositionY == VerticalPosition_Center)
        {
            pInfo->pPane->SetTranslate(NN_UTIL_FLOAT_3_INITIALIZER(position.x, position.y + (height / 2.0f), position.z));
        }
        else if (basePositionY == VerticalPosition_Top)
        {
            pInfo->pPane->SetTranslate(NN_UTIL_FLOAT_3_INITIALIZER(position.x, position.y + height, position.z));
        }
    }
    else
    {
        if (basePositionY == VerticalPosition_Center)
        {
            pInfo->pPane->SetTranslate(NN_UTIL_FLOAT_3_INITIALIZER(position.x, position.y - (height / 2.0f), position.z));
        }
        else if (basePositionY == VerticalPosition_Bottom)
        {
            pInfo->pPane->SetTranslate(NN_UTIL_FLOAT_3_INITIALIZER(position.x, position.y - height, position.z));
        }
    }
}

void MesurePartsHorizontalBound_(Pane *pPane, float transX, float &left, float &right)
{
    // 非表示なら、サイズ計算の対象にしません。透明度ゼロは、計算対象になる点に注意してください。
    if (!pPane->IsVisible())
    {
        // これ以上処理が必要なペイン
        return;
    }

    if (!pPane->IsAlignmentIgnore())
    {
        float pane_width = pPane->GetSize().width;

        // Translateの計算
        transX += pPane->GetTranslate().x;

        // 範囲を拡張する
        float min = std::numeric_limits<float>::max();
        float max = std::numeric_limits<float>::min();

        const TextBox* pTextBox = DynamicCast<const TextBox*>(pPane);
        if (pTextBox)
        {
            float width = pTextBox->GetSize().width;
            const HorizontalPosition textPositionX = pTextBox->GetTextPositionX();
            if (textPositionX == HorizontalPosition_Right)
            {
                // テキストが右揃え
                max = transX + pane_width / 2.0f;
                min = max - width;
            }
            else if (textPositionX == HorizontalPosition_Left)
            {
                // テキストが左揃え
                min = transX - pane_width / 2.0f;
                max = min + width;
            }
            else
            {
                // テキストが中央揃え
                min = transX - width / 2.0f;
                max = transX + width / 2.0f;
            }
        }
        else
        {
            const Picture* pPicture = DynamicCast<const Picture*>(pPane);
            const Window* pWindow = DynamicCast<const Window*>(pPane);
            if (pPicture || pWindow)
            {
                float scalex = pPane->GetScale().x;
                min = transX - (pane_width / 2.0f * std::abs(scalex));
                max = transX + (pane_width / 2.0f * std::abs(scalex));
            }
        }

        if (min < max)
        {
            // ペインの基準位置を反映する
            const uint8_t basePositionX = pPane->GetBasePositionX();
            if (basePositionX == HorizontalPosition_Left)
            {
                min += pane_width / 2.0f;
                max += pane_width / 2.0f;
            }
            else if (basePositionX == HorizontalPosition_Right)
            {
                min -= pane_width / 2.0f;
                max -= pane_width / 2.0f;
            }
        }

        left = std::min(left, min);
        right = std::max(right, max);
    }

    // 再帰的に呼び出し
    PaneList::iterator endIter = pPane->GetChildList().end();
    for (PaneList::iterator iter = pPane->GetChildList().begin(); iter != endIter; ++iter)
    {
        MesurePartsHorizontalBound_(&(*iter), transX, left, right);
    }
}

void MesurePartsVerticalBound_(Pane *pPane, float transY, float &top, float &bottom)
{
    // 非表示なら、サイズ計算の対象にしません。透明度ゼロは、計算対象になる点に注意してください。
    if (!pPane->IsVisible())
    {
        // これ以上処理が必要なペイン
        return;
    }

    if (!pPane->IsAlignmentIgnore())
    {
        float pane_height = pPane->GetSize().height;

        // Translateの計算
        transY += pPane->GetTranslate().y;

        // 範囲を拡張する
        float min = std::numeric_limits<float>::max();
        float max = std::numeric_limits<float>::min();

        const TextBox* pTextBox = DynamicCast<const TextBox*>(pPane);
        if (pTextBox)
        {
            float height = pTextBox->GetSize().height;
            const VerticalPosition textPositionY = pTextBox->GetTextPositionY();
            if (textPositionY == VerticalPosition_Top)
            {
                // テキストが上揃え
                max = transY + pane_height / 2.0f;
                min = max - height;
            }
            else if (textPositionY == VerticalPosition_Bottom)
            {
                // テキストが下揃え
                min = transY - pane_height / 2.0f;
                max = min + height;
            }
            else
            {
                // テキストが中央揃え
                min = transY - height / 2.0f;
                max = transY + height / 2.0f;
            }
        }
        else
        {
            const Picture* pPicture = DynamicCast<const Picture*>(pPane);
            const Window* pWindow = DynamicCast<const Window*>(pPane);
            if (pPicture || pWindow)
            {
                float scaley = pPane->GetScale().y;
                min = transY - (pane_height / 2.0f * std::abs(scaley));
                max = transY + (pane_height / 2.0f * std::abs(scaley));
            }
        }

        if (min < max)
        {
            // ペインの基準位置を反映する
            const uint8_t basePositionY = pPane->GetBasePositionY();
            if (basePositionY == VerticalPosition_Bottom)
            {
                min += pane_height / 2.0f;
                max += pane_height / 2.0f;
            }
            else if (basePositionY == VerticalPosition_Top)
            {
                min -= pane_height / 2.0f;
                max -= pane_height / 2.0f;
            }
        }

        top = std::max(top, max);
        bottom = std::min(bottom, min);
    }

    // 再帰的に呼び出し
    PaneList::iterator endIter = pPane->GetChildList().end();
    for (PaneList::iterator iter = pPane->GetChildList().begin(); iter != endIter; ++iter)
    {
        MesurePartsVerticalBound_(&(*iter), transY, top, bottom);
    }
}

void MesurePartsHorizontalBoundFromRoot_(Pane *pPane, float transX, float &left, float &right)
{
    NN_UNUSED(transX);

    PaneList::iterator endIter = pPane->GetChildList().end();
    for (PaneList::iterator iter = pPane->GetChildList().begin(); iter != endIter; ++iter)
    {
        MesurePartsHorizontalBound_(&(*iter), 0.0f, left, right);
    }
}

void MesurePartsVerticalBoundFromRoot_(Pane *pPane, float transY, float &top, float &bottom)
{
    NN_UNUSED(transY);

    PaneList::iterator endIter = pPane->GetChildList().end();
    for (PaneList::iterator iter = pPane->GetChildList().begin(); iter != endIter; ++iter)
    {
        MesurePartsVerticalBound_(&(*iter), 0.0f, top, bottom);
    }
}

float CalcStringWidth(TextBox *pTextBox)
{
    NN_SDK_ASSERT_NOT_NULL(pTextBox);
    const nn::font::Rectangle rect = pTextBox->GetTextDrawRect();
    return rect.right - rect.left;
}

float CalcStringHeight(TextBox *pTextBox)
{
    NN_SDK_ASSERT_NOT_NULL(pTextBox);
    const nn::font::Rectangle rect = pTextBox->GetTextDrawRect();
    return rect.top - rect.bottom;
}

bool MakeHorizontalAlignInfo_(Pane *pPane, AlignInfo &info)
{
    if (pPane->IsAlignmentIgnore())
    {
        return false;
    }

    // ペインの種類によって処理が変わる
    // 部品の場合は計算
    // null の場合は配下を計算
    // 計算の結果、幅がゼロなら 非表示状態とみなす
    const Alignment* pParentAlignment = DynamicCast<const Alignment*>(pPane->GetParent());
    NN_SDK_ASSERT_NOT_NULL(pParentAlignment);

    info.pPane = pPane;
    info.Length = pPane->GetSize().width;
    info.Margin = 0.0f;
    info.CenterOffset = 0.0f;

    // 注意：透明度ゼロ（childPane.Transparency == 0）の場合は、幅計算の対象となります。
    bool isVisible = pPane->IsVisible();

    const Parts* pParts = DynamicCast<const Parts*>(pPane);
    if (pParts)
    {
        float left = std::numeric_limits<float>::max();
        float right = std::numeric_limits<float>::min();
        MesurePartsHorizontalBoundFromRoot_(pPane, 0.0f, left, right);

        info.Length = right - left;
        info.CenterOffset = (right + left) / 2.0f;
        if (info.Length == 0.0f)
        {
            isVisible = false;
        }
    }
    else if (pPane->IsAlignmentNullPane() && !pPane->GetChildList().empty())
    {
        // 子供を持つ、Null ペインは（部品ペインのように）配下のペインのサイズ総和をサイズとする
        float left = std::numeric_limits<float>::max();
        float right = std::numeric_limits<float>::min();
        MesurePartsHorizontalBoundFromRoot_(pPane, 0.0f, left, right);

        info.Length = right - left;
        info.CenterOffset = (right + left) / 2.0f;
        if (info.Length == 0.0f)
        {
            isVisible = false;
        }
    }
    else
    {
        TextBox* pTextBox = DynamicCast<TextBox*>(pPane);
        if (pTextBox)
        {
            const float paneWidth = pTextBox->GetSize().width;
            const float calcWidth = CalcStringWidth(pTextBox);
            info.Length = calcWidth;

            float diffTextW = -(paneWidth / 2.0f) + (calcWidth / 2.0f);
            const HorizontalPosition textPositionX = pTextBox->GetTextPositionX();
            if (textPositionX == HorizontalPosition_Right)
            {
                info.CenterOffset = -diffTextW;
            }
            else if (textPositionX == HorizontalPosition_Left)
            {
                info.CenterOffset = diffTextW;
            }
        }
        else
        {
            info.Length = pPane->GetSize().width * std::abs(pPane->GetScale().x);
        }

        // CenterOffset
        {
            // ペインの基準位置を反映する
            const uint8_t basePositionX = pPane->GetBasePositionX();
            if (basePositionX == HorizontalPosition_Left)
            {
                info.CenterOffset += pPane->GetSize().width / 2.0f;
            }
            else if (basePositionX == HorizontalPosition_Right)
            {
                info.CenterOffset -= pPane->GetSize().width / 2.0f;
            }
        }
    }

    // info.Margin
    {
        // 表示されている場合は、マージンを設定します。
        if (isVisible)
        {
            if(pPane->IsAlignmentMarginEnabled())
            {
                info.Margin = info.pPane->GetAlignmentMargin();
            }
            else
            {
                info.Margin = pParentAlignment->GetDefaultMargin();
            }
        }
        else
        {
            info.Margin = 0.0f;
        }
    }

    return isVisible;
}

bool MakeVerticalAlignInfo_(Pane *pPane, AlignInfo &info)
{
    if (pPane->IsAlignmentIgnore())
    {
        return false;
    }

    // ペインの種類によって処理が変わる
    // 部品の場合は計算
    // null の場合は配下を計算
    // 計算の結果、幅がゼロなら 非表示状態とみなす
    const Alignment* pParentAlignment = DynamicCast<const Alignment*>(pPane->GetParent());
    NN_SDK_ASSERT_NOT_NULL(pParentAlignment);

    info.pPane = pPane;
    info.Length = pPane->GetSize().height;
    info.Margin = 0.0f;
    info.CenterOffset = 0.0f;

    // 注意：透明度ゼロ（childPane.Transparency == 0）の場合は、幅計算の対象となります。
    bool isVisible = pPane->IsVisible();

    const Parts* pParts = DynamicCast<const Parts*>(pPane);
    if (pParts)
    {
        float top = std::numeric_limits<float>::min();
        float bottom = std::numeric_limits<float>::max();
        MesurePartsVerticalBoundFromRoot_(pPane, 0.0f, top, bottom);

        info.Length = top - bottom;
        info.CenterOffset = (top + bottom) / 2.0f;
        if (info.Length == 0.0f)
        {
            isVisible = false;
        }
    }
    else if (pPane->IsAlignmentNullPane() && !pPane->GetChildList().empty())
    {
        // 子供を持つ、Null ペインは（部品ペインのように）配下のペインのサイズ総和をサイズとする
        float top = std::numeric_limits<float>::min();
        float bottom = std::numeric_limits<float>::max();
        MesurePartsVerticalBoundFromRoot_(pPane, 0.0f, top, bottom);

        info.Length = top - bottom;
        info.CenterOffset = (top + bottom) / 2.0f;
        if (info.Length == 0.0f)
        {
            isVisible = false;
        }
    }
    else
    {
        TextBox* pTextBox = DynamicCast<TextBox*>(pPane);
        if (pTextBox)
        {
            const float paneHeight = pTextBox->GetSize().height;
            const float calcHeight = CalcStringHeight(pTextBox);
            info.Length = calcHeight;

            float diffTextH = -(paneHeight / 2.0f) + (calcHeight / 2.0f);
            const VerticalPosition textPositionY = pTextBox->GetTextPositionY();
            if (textPositionY == VerticalPosition_Top)
            {
                info.CenterOffset = -diffTextH;
            }
            else if (textPositionY == VerticalPosition_Bottom)
            {
                info.CenterOffset = diffTextH;
            }
        }
        else
        {
            info.Length = pPane->GetSize().height * std::abs(pPane->GetScale().y);
        }

        // CenterOffset
        {
            // ペインの基準位置を反映する
            const uint8_t basePositionY = pPane->GetBasePositionY();
            if (basePositionY == VerticalPosition_Top)
            {
                info.CenterOffset -= pPane->GetSize().height / 2.0f;
            }
            else if (basePositionY == VerticalPosition_Bottom)
            {
                info.CenterOffset += pPane->GetSize().height / 2.0f;
            }
        }
    }

    // info.Margin
    {
        // 表示されている場合は、マージンを設定します。
        if (isVisible)
        {
            if(pPane->IsAlignmentMarginEnabled())
            {
                info.Margin = info.pPane->GetAlignmentMargin();
            }
            else
            {
                info.Margin = pParentAlignment->GetDefaultMargin();
            }
        }
        else
        {
            info.Margin = 0.0f;
        }
    }

    return isVisible;
}

bool IsHorizontalExpandTarget_(Alignment *pAlignment, Pane *pChild)
{
    if (pAlignment->GetChildList().empty())
    {
        return false;
    }

    switch (pAlignment->GetHorizontalAlignment())
    {
        case HorizontalAlignment_Left: return pChild == &(*pAlignment->GetChildList().rbegin());
        case HorizontalAlignment_Center: return pChild == &(*pAlignment->GetChildList().begin()) || pChild == &(*pAlignment->GetChildList().rbegin());
        case HorizontalAlignment_Right: return pChild == &(*pAlignment->GetChildList().begin());
        default: return false;
    }
}

bool IsVerticalExpandTarget_(Alignment *pAlignment, Pane *pChild)
{
    if (pAlignment->GetChildList().empty())
    {
        return false;
    }

    switch (pAlignment->GetVerticalAlignment())
    {
        case VerticalAlignment_Top: return pChild == &(*pAlignment->GetChildList().rbegin());
        case VerticalAlignment_Center: return pChild == &(*pAlignment->GetChildList().begin()) || pChild == &(*pAlignment->GetChildList().rbegin());
        case VerticalAlignment_Bottom: return pChild == &(*pAlignment->GetChildList().begin());
        default: return false;
    }
}

} // namespace

//----------------------------------------
Alignment::Alignment()
: Base()
, m_HorizontalAlignment(0)
, m_DefaultMergine(0.0f)
, m_IsExtendEdgeEnabled(false)
, m_Flag(0)
{
}

//----------------------------------------
Alignment::~Alignment()
{
}

//----------------------------------------
Alignment::Alignment(const ResAlignment* pBlock, const BuildArgSet& buildArgSet)
: Base(pBlock, buildArgSet)
{
    m_HorizontalAlignment = pBlock->horizontalAlignment;
    m_DefaultMergine = pBlock->defaultMergine;
    m_IsExtendEdgeEnabled = pBlock->isExtendEdgeEnabled;
    m_Flag = 0;

    // 整列（水平）、または整列（垂直）を設定する。
    const bool bAlignmentDirection = detail::GetBits(pBlock->flag, ResAlignment::PosAlignmentDirection, ResAlignment::LenAlignmentDirection) != 0;
    detail::SetBit(&m_Flag, Flags_AlignmentDirection, bAlignmentDirection);

    RequestAlignment();
}

//----------------------------------------
Alignment::Alignment(const Alignment& alignment)
: Base(alignment)
{
    m_HorizontalAlignment = alignment.m_HorizontalAlignment;
    m_DefaultMergine = alignment.m_DefaultMergine;
    m_IsExtendEdgeEnabled = alignment.m_IsExtendEdgeEnabled;
    m_Flag = alignment.m_Flag;
}

//----------------------------------------
float Alignment::GetDefaultMargin() const
{
    return m_DefaultMergine;
}

//----------------------------------------

int Alignment::GetHorizontalAlignment() const
{
    return m_HorizontalAlignment;
}

//----------------------------------------

int Alignment::GetVerticalAlignment() const
{
    return m_VerticalAlignment;
}

//----------------------------------------
void Alignment::RequestAlignment()
{
    detail::SetBit(&m_Flag, Flags_RecalculationRequest, 1);
}

//----------------------------------------
void
Alignment::Calculate(DrawInfo& drawInfo, Pane::CalculateContext& context, bool isDirtyParentMtx)
{
    if(detail::TestBit(m_Flag, Flags_RecalculationRequest))
    {
        detail::SetBit(&m_Flag, Flags_RecalculationRequest, 0);

        Pane::CalculateGlobalMatrix(context);

        // 整列を行う
        MakeAlignment();
    }

    Pane::Calculate(drawInfo, context, isDirtyParentMtx);

    LoadMtx(drawInfo);
}

//----------------------------------------
void Alignment::MakeForwardHorizontalAlignment()
{
    float totalWidth = 0.0f;
    AlignInfo firstInfo;
    AlignInfo lastInfo;
    bool bFirst = false;

    float leftX = m_HorizontalAlignment == HorizontalAlignment_Center ? 0.0f: - GetSize().width / 2.0f;

    PaneList::iterator endIter = GetChildList().end();
    PaneList::iterator startIter = GetChildList().begin();
    for (PaneList::iterator iter = startIter; iter != endIter; ++iter)
    {

        AlignInfo info;
        if (!MakeHorizontalAlignInfo_(&(*iter), info))
        {
            continue;
        }

        if (m_IsExtendEdgeEnabled)
        {
            // 端を延長するモードの場合、延長する対象のペインのサイズは、ここでは0と見なす。
            // ただしマージンだけは計算する
            if (IsHorizontalExpandTarget_(this, &(*iter)))
            {
                info.Length = 0.0f;
                info.CenterOffset = 0.0f;
            }
        }

        if (!bFirst)
        {
            //=== 中央揃えと左揃え
            // 左端のマージンは考慮しないので引く
            totalWidth -= info.Margin;

            firstInfo = info;
            bFirst = true;
        }
        else
        {
            leftX += info.Margin;
        }

        float newX = leftX + info.Length / 2.0f - info.CenterOffset;

        nn::util::Float3 position;
        position = (*iter).GetTranslate();
        (*iter).SetTranslate(NN_UTIL_FLOAT_3_INITIALIZER(newX, position.y, position.z));

        totalWidth += info.Length + info.Margin;
        leftX += info.Length;
        lastInfo = info;
    }

    if (m_HorizontalAlignment == HorizontalAlignment_Center)
    {
        for (PaneList::iterator iter = startIter; iter != endIter; ++iter)
        {
            if (!(*iter).IsVisible())
            {
                continue;
            }

            // 整列ペインの処理から除外する場合は補正は行わない
            if ((*iter).IsAlignmentIgnore())
            {
                continue;
            }

            nn::util::Float3 v = (*iter).GetTranslate();
            float centerLeftX = -totalWidth / 2.0f;
            (*iter).SetTranslate(NN_UTIL_FLOAT_3_INITIALIZER(v.x + centerLeftX, v.y, v.z));
        }
    }

    // ExtendEdge
    if (m_IsExtendEdgeEnabled)
    {
        if (m_HorizontalAlignment == HorizontalAlignment_Left)
        {
            // 右端の要素を延長
            float width = GetSize().width - totalWidth;
            DoHorizontalExpand_(&lastInfo, width, false);
        }
        else if (m_HorizontalAlignment == HorizontalAlignment_Center)
        {
            // 左端及び右端の要素を延長
            float width = (GetSize().width - totalWidth) / 2.0f;
            DoHorizontalExpand_(&lastInfo, width, false);
            DoHorizontalExpand_(&firstInfo, width, true);
        }
    }
}

//----------------------------------------
void Alignment::MakeReverseHorizontalAlignment()
{
    float totalWidth = 0.0f;
    AlignInfo firstInfo;
    AlignInfo lastInfo;
    bool bLast = false;

    // 右揃えでも、「最初の子ペインが一番左になる」のは同じ。
    // このため、逆順にイテレートして置いていく。
    float rightX = GetSize().width / 2.0f;

    PaneList::reverse_iterator rendIter = GetChildList().rend();
    PaneList::reverse_iterator rstartIter = GetChildList().rbegin();
    for (PaneList::reverse_iterator iter = rstartIter; iter != rendIter; ++iter)
    {
        AlignInfo info;
        if (!MakeHorizontalAlignInfo_(&(*iter), info))
        {
            continue;
        }

        if (m_IsExtendEdgeEnabled)
        {
            // 端を延長するモードの場合、延長する対象のペインのサイズは、ここでは0と見なす。
            // ただしマージンだけは計算する
            if (IsHorizontalExpandTarget_(this, &(*iter)))
            {
                info.Length = 0.0f;
                info.CenterOffset = 0.0f;
            }
        }

        if (!bLast)
        {
            //=== 右揃え
            // 右端のマージンは考慮しないので引く
            totalWidth -= info.Margin;

            bLast = true;
        }
        else
        {
            rightX -= info.Margin;
        }

        float newX = rightX - info.Length / 2.0f - info.CenterOffset;

        const nn::util::Float3& position = (*iter).GetTranslate();
        (*iter).SetTranslate(NN_UTIL_FLOAT_3_INITIALIZER(newX, position.y, position.z));

        totalWidth += info.Length + info.Margin;
        rightX -= info.Length;
        firstInfo = info;
    }

    // ExtendEdge
    if (m_IsExtendEdgeEnabled)
    {
        // 左端の要素を延長
        float width = GetSize().width - totalWidth;
        DoHorizontalExpand_(&firstInfo, width, true);
    }
}

//----------------------------------------
void Alignment::MakeForwardVerticalAlignment()
{
    float totalHeight = 0.0f;
    AlignInfo firstInfo;
    AlignInfo lastInfo;
    bool bFirst = false;

    float upY = m_VerticalAlignment == VerticalAlignment_Center ? 0.0f: GetSize().height / 2.0f;

    PaneList::iterator endIter = GetChildList().end();
    PaneList::iterator startIter = GetChildList().begin();
    for (PaneList::iterator iter = startIter; iter != endIter; ++iter)
    {

        AlignInfo info;
        if (!MakeVerticalAlignInfo_(&(*iter), info))
        {
            continue;
        }

        if (m_IsExtendEdgeEnabled)
        {
            // 端を延長するモードの場合、延長する対象のペインのサイズは、ここでは0と見なす。
            // ただしマージンだけは計算する
            if (IsVerticalExpandTarget_(this, &(*iter)))
            {
                info.Length = 0.0f;
                info.CenterOffset = 0.0f;
            }
        }

        if (!bFirst)
        {
            //=== 中央揃えと上揃え
            // 上端のマージンは考慮しないので引く
            totalHeight -= info.Margin;

            firstInfo = info;
            bFirst = true;
        }
        else
        {
            upY -= info.Margin;
        }

        float newY = upY - info.Length / 2.0f - info.CenterOffset;

        nn::util::Float3 position;
        position = (*iter).GetTranslate();
        (*iter).SetTranslate(NN_UTIL_FLOAT_3_INITIALIZER(position.x, newY, position.z));

        totalHeight += info.Length + info.Margin;
        upY -= info.Length;
        lastInfo = info;
    }

    if (m_VerticalAlignment == VerticalAlignment_Center)
    {
        for (PaneList::iterator iter = startIter; iter != endIter; ++iter)
        {
            if (!(*iter).IsVisible())
            {
                continue;
            }

            // 整列ペインの処理から除外する場合は補正は行わない
            if ((*iter).IsAlignmentIgnore())
            {
                continue;
            }

            nn::util::Float3 v = (*iter).GetTranslate();
            float centerTopY = totalHeight / 2.0f;
            (*iter).SetTranslate(NN_UTIL_FLOAT_3_INITIALIZER(v.x, v.y + centerTopY, v.z));
        }
    }

    // ExtendEdge
    if (m_IsExtendEdgeEnabled)
    {
        if (m_VerticalAlignment == VerticalAlignment_Top)
        {
            // 下端の要素を延長
            float height = GetSize().height - totalHeight;
            DoVerticalExpand_(&lastInfo, height, false);
        }
        else if (m_VerticalAlignment == HorizontalAlignment_Center)
        {
            // 上端及び下端の要素を延長
            float height = (GetSize().height - totalHeight) / 2.0f;
            DoVerticalExpand_(&lastInfo, height, false);
            DoVerticalExpand_(&firstInfo, height, true);
        }
    }
}

//----------------------------------------
void Alignment::MakeReverseVerticalAlignment()
{
    float totalHeight = 0.0f;
    AlignInfo firstInfo;
    AlignInfo lastInfo;
    bool bLast = false;

    // 下揃えでも、「最初の子ペインが一番上になる」のは同じ。
    // このため、逆順にイテレートして置いていく。
    float bottomY = -GetSize().height / 2.0f;

    PaneList::reverse_iterator rendIter = GetChildList().rend();
    PaneList::reverse_iterator rstartIter = GetChildList().rbegin();
    for (PaneList::reverse_iterator iter = rstartIter; iter != rendIter; ++iter)
    {
        AlignInfo info;
        if (!MakeVerticalAlignInfo_(&(*iter), info))
        {
            continue;
        }

        if (m_IsExtendEdgeEnabled)
        {
            // 端を延長するモードの場合、延長する対象のペインのサイズは、ここでは0と見なす。
            // ただしマージンだけは計算する
            if (IsVerticalExpandTarget_(this, &(*iter)))
            {
                info.Length = 0.0f;
                info.CenterOffset = 0.0f;
            }
        }

        if (!bLast)
        {
            //=== 下揃え
            // 下端のマージンは考慮しないので引く
            totalHeight -= info.Margin;

            bLast = true;
        }
        else
        {
            bottomY += info.Margin;
        }

        float newY = bottomY + info.Length / 2.0f - info.CenterOffset;

        const nn::util::Float3& position = (*iter).GetTranslate();
        (*iter).SetTranslate(NN_UTIL_FLOAT_3_INITIALIZER(position.x, newY, position.z));

        totalHeight += info.Length + info.Margin;
        bottomY += info.Length;
        firstInfo = info;
    }

    // ExtendEdge
    if (m_IsExtendEdgeEnabled)
    {
        // 上端の要素を延長
        float height = GetSize().height - totalHeight;
        DoVerticalExpand_(&firstInfo, height, true);
    }
}

//----------------------------------------
void
Alignment::MakeHorizontalAlignment()
{
    if (m_HorizontalAlignment == HorizontalAlignment_Center ||
        m_HorizontalAlignment == HorizontalAlignment_Left)
    {
        MakeForwardHorizontalAlignment();
    }
    else
    {
        MakeReverseHorizontalAlignment();
    }
}

//----------------------------------------
void
Alignment::MakeVerticalAlignment()
{
    if (m_VerticalAlignment == VerticalAlignment_Center ||
        m_VerticalAlignment == VerticalAlignment_Top)
    {
        MakeForwardVerticalAlignment();
    }
    else
    {
        MakeReverseVerticalAlignment();
    }
}

//----------------------------------------
void
Alignment::MakeAlignment()
{
    // ペインの種類によって処理が変わる
    // 部品の場合は計算
    // null の場合は配下を計算
    // 計算の結果、幅がゼロなら 非表示状態とみなす

    if (IsHorizontalAlignment())
    {
        MakeHorizontalAlignment();
    }
    else
    {
        MakeVerticalAlignment();
    }
}

//----------------------------------------
bool
Alignment::CompareCopiedInstanceTest(const Alignment& target) const
{
    if (m_HorizontalAlignment != target.m_HorizontalAlignment ||
        m_DefaultMergine != target.m_DefaultMergine ||
        m_IsExtendEdgeEnabled != target.m_IsExtendEdgeEnabled ||
        m_Flag != target.m_Flag)
    {
        return false;
    }

    return true;
}

//----------------------------------------
bool
Alignment::IsHorizontalAlignment() const
{
    return detail::TestBit(m_Flag, Flags_AlignmentDirection) == false;
}

//----------------------------------------
bool
Alignment::IsVerticalAlignment() const
{
    return detail::TestBit(m_Flag, Flags_AlignmentDirection) == true;
}


} // namespace nn::ui2d
} // namespace nn

