﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "../../../precompiled.h"

#include <climits>
#include <nn/spy/detail/fnd/os/spyfnd_Thread.h>

namespace nn {
namespace spy {
namespace detail {
namespace fnd {

NN_DEFINE_STATIC_CONSTANT(const int Thread::DefaultThreadPriority);
NN_DEFINE_STATIC_CONSTANT(const int Thread::MinThreadPriority);
NN_DEFINE_STATIC_CONSTANT(const int Thread::MaxThreadPriority);

//---------------------------------------------------------------------------
Thread::RunArgs::RunArgs() NN_NOEXCEPT
    : name("")
    , stack(NULL)
    , stackSize(0)
    , affinityMask(Thread::AffinityMask_CoreDefault)
    , priority(Thread::DefaultThreadPriority)
    , param(NULL)
    , handler(NULL)
{ }

//---------------------------------------------------------------------------
bool
Thread::RunArgs::IsValid() const NN_NOEXCEPT
{
    if(stack == NULL)
    {
        return false;
    }

    if(stackSize == 0)
    {
        return false;
    }

    if(priority > MaxThreadPriority)
    {
        return false;
    }

    if(handler == NULL)
    {
        return false;
    }

    return true;
}

//---------------------------------------------------------------------------
Thread::~Thread() NN_NOEXCEPT
{
}

//---------------------------------------------------------------------------
bool
Thread::Run(const RunArgs& args) NN_NOEXCEPT
{
    if(!args.IsValid())
    {
        NN_ABORT("invalid arguments.\n");
        return false;
    }

    m_Param = args.param;
    m_Handler = args.handler;

    if(!Create(m_Handle, args))
    {
        return false;
    }

    SetName(args.name);

    if(args.affinityMask != AffinityMask_CoreDefault)
    {
        SetAffinityMask(args.affinityMask);
    }

    m_Priority = args.priority;

    Resume();

    return true;
}

//---------------------------------------------------------------------------
void
Thread::WaitForExit() NN_NOEXCEPT
{
    Join();
}

//---------------------------------------------------------------------------
void
Thread::Release() NN_NOEXCEPT
{
    switch(m_State)
    {
    case State_Exited:
        break;

    case State_Running:
        NN_ABORT("must not Release() for running thread.\n");
        return;

    default:
        return;
    }

    Detach();
    SetState(State_Released);
}

//---------------------------------------------------------------------------
int
Thread::GetPriority() const NN_NOEXCEPT
{
    return m_Priority;
}

//---------------------------------------------------------------------------
Thread::State
Thread::GetState() const NN_NOEXCEPT
{
    return static_cast<State>(m_State);
}

//---------------------------------------------------------------------------
void
Thread::SetState(State value) NN_NOEXCEPT
{
    m_State = value;
}

//---------------------------------------------------------------------------
void
Thread::OnRun() NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_State == State_NotRun || m_State == State_Released);
    SetState(State_Running);
}

//---------------------------------------------------------------------------
void
Thread::OnExit() NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_State == State_Running);
    SetState(State_Exited);
}

} // namespace nn::spy::detail::fnd
} // namespace nn::spy::detail
} // namespace nn::spy
} // namespace nn
