﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>

#include <nn/util/util_FormatString.h>
#include <nn/ige/ige_BufferWriter.h>
#include <nn/ige/ige_TextBox.h>

namespace nn { namespace ige {

typedef util::BitPack32::Field<0, 1, bool> FileDropField;

TextBox::TextBox(char* buffer, size_t size, TextBoxOption option) NN_NOEXCEPT
    : Control(&m_Node, option)
    , m_Size(size)
    , m_Pack()
{
    NN_SDK_REQUIRES_NOT_NULL(buffer);
    NN_SDK_REQUIRES(0 < size);

    NodeOption nodeOption;
    nodeOption.pMetadataFunction = GenerateMetadata;
    nodeOption.pUserData = this;
    nodeOption.enableAutoNotify = false;

    SetFileDrop(option.isFileDropEnabled);

    m_Node.Initialize(buffer, static_cast<int>(m_Size), nodeOption);
}

void TextBox::GenerateMetadata(BufferWriter* pWriter, const Node& node) NN_NOEXCEPT
{
    auto& self = *node.GetUserData<TextBox>();
    WriteAttribute(pWriter, "type", "TextBox");
    self.WriteCommonMetadata(pWriter);

    WriteAttribute(pWriter, "fileDrop", self.m_Pack.Get<FileDropField>() ? "true" : "false");

    {
        char buffer[16] = {};
        ::nn::util::SNPrintf(buffer, NN_ARRAY_SIZE(buffer), "%d", self.m_Size);
        WriteAttribute(pWriter, "size", buffer);
    }
}

bool TextBox::IsFileDropEnabled() const NN_NOEXCEPT
{
    return m_Pack.Get<FileDropField>();
}

void TextBox::SetFileDrop(bool isFileDropEnabled) NN_NOEXCEPT
{
    m_Pack.Set<FileDropField>(isFileDropEnabled);
}

}} // namespace nn::ige
