﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/util/util_Endian.h>

namespace nn { namespace ige { namespace detail {

//! @briefprivate パケットバージョンです。
struct Version
{
    //! @briefprivate   マイクロバージョンです。
    uint8_t micro;

    //! @briefprivate   マイナーバージョンです。
    uint8_t minor;

    //! @briefprivate   メジャーバージョンです。
    uint16_t major;

    //! @briefprivate   現在のバージョンに対して対象バージョンに互換性があるかどうかを取得します。
    //! @param[in]      majorVersion    メジャーバージョンです。
    //! @param[in]      minorVersion    マイナーバージョンです。
    //! @param[in]      microVersion    マイクロバージョンです。
    //! @return         バージョンが有効であれば true を返します。
    bool IsValid(int majorVersion, int minorVersion, int microVersion) const NN_NOEXCEPT
    {
        NN_UNUSED( microVersion);
        return util::LoadLittleEndian( &this->major ) == majorVersion && this->minor <= minorVersion;
    }

    //! @briefprivate   現在のバージョンに対して対象バージョンに互換性があるかどうかを取得します。
    //! @param[in]      target  対象のバージョンです。
    //! @return         バージョンが有効であれば true を返します。
    bool IsValid(Version target) const NN_NOEXCEPT
    {
        return IsValid(target.major, target.minor, target.micro);
    }

    //! @briefprivate   バージョンを設定します。
    //! @param[in]      packedVersion   バージョンを示す値です。
    void SetPacked( uint32_t packedVersion )
    {
        util::StoreLittleEndian( &this->major, static_cast< uint16_t >( packedVersion >> 16 ) );
        this->minor = static_cast< uint8_t >( ( packedVersion >> 8 ) & 0xFF );
        this->micro = static_cast< uint8_t >( packedVersion & 0xFF );
    }

    //! @briefprivate   バージョンに相当する値を取得します。
    //! @return         バージョンに相当する値を返します。
    uint32_t GetPacked() const
    {
        return static_cast< uint32_t >( util::LoadLittleEndian( &this->major ) << 16 | minor << 8 | micro );
    }

    static Version CurrentVersion()
    {
        Version version;
        version.major = 0;
        version.minor = 0;
        version.micro = 0;
        return version;
    }
};

}}} // namespace nn::ige::detail
