﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include "../g3d_ViewerDetailDefine.h"



#include "../g3d_Allocator.h"

namespace nn { namespace g3d { namespace viewer {

namespace detail {

template<typename TCharType>
class DynamicLengthStringBase
{
public:
    static const TCharType NULL_CHAR = static_cast<TCharType>(0);
    static const size_t MAXIMU_LENGTH = 1024; // 1024文字まで

    explicit DynamicLengthStringBase(Allocator* allocator) NN_NOEXCEPT
        : m_pString(nullptr)
        , m_StringLength(0)
        , m_pAllocator(allocator)
    {
        NN_G3D_VIEWER_ASSERT_NOT_NULL(allocator);
    }

    ~DynamicLengthStringBase() NN_NOEXCEPT
    {
        Clear();
        m_pAllocator = nullptr;
    }

    void Clear() NN_NOEXCEPT
    {
        if (m_pString != nullptr)
        {
            m_pAllocator->Free(m_pString);
            m_pString = nullptr;
        }
    }

    ViewerResult Assign(const TCharType* str) NN_NOEXCEPT
    {
        NN_G3D_VIEWER_ASSERT_NOT_NULL(m_pAllocator);
        size_t length = CalculateStrLength(str);
        if (length == 0)
        {
            return ViewerResult_InvalidArgument;
        }

        size_t bufferSize = (length + 1) * sizeof(TCharType);
        TCharType* stringBuffer = static_cast<TCharType*>(m_pAllocator->Allocate(bufferSize, nn::g3d::detail::Alignment_Default, AllocateType_DynamicBuffer));
        if (stringBuffer == nullptr)
        {
            return ViewerResult_MemoryAllocationFailed;
        }

        if (m_pString != nullptr)
        {
            m_pAllocator->Free(m_pString);
            m_pString = nullptr;
        }

        m_pString = stringBuffer;
        std::memcpy(m_pString, str, (length * sizeof(TCharType)));

        m_pString[length] = DynamicLengthStringBase<TCharType>::NULL_CHAR;
        m_StringLength = length;

        return ViewerResult_Success;
    }

    int Compare( const DynamicLengthStringBase<TCharType>& rhs) const NN_NOEXCEPT
    {
        NN_G3D_VIEWER_ASSERT_NOT_NULL(m_pAllocator);
        // アドレスがいっしょなので同一
        if (GetStr() == rhs.GetStr())
        {
            return 0;
        }

        size_t length = GetLength();
        if (length < rhs.GetLength())
        {
            length = rhs.GetLength();
        }

        for (size_t i = 0; i <= length; ++i)
        {
            if (m_pString[i] == NULL_CHAR && rhs.m_pString[i] == NULL_CHAR)
            {
                return 0;
            }
            else if (m_pString[i] == NULL_CHAR)
            {
                return -1;
            }
            else if (rhs.m_pString[i] == NULL_CHAR)
            {
                return 1;
            }
            else if (m_pString[i] < rhs.m_pString[i])
            {
                return -1;
            }
            else if (m_pString[i] > rhs.m_pString[i])
            {
                return 1;
            }
        }
        return 0;
    }

    const TCharType* GetStr() const NN_NOEXCEPT
    {
        return m_pString;
    }
    size_t GetLength() const NN_NOEXCEPT
    {
        return m_StringLength;
    }

private:
    inline size_t CalculateStrLength(const TCharType* str) const NN_NOEXCEPT;

    TCharType* m_pString;
    size_t m_StringLength;
    Allocator* m_pAllocator;
};

template<typename TCharType>
size_t DynamicLengthStringBase<TCharType>::CalculateStrLength(const TCharType* str) const NN_NOEXCEPT
{
    NN_G3D_VIEWER_ASSERT_NOT_NULL(m_pAllocator);
    NN_G3D_VIEWER_ASSERT_NOT_NULL(str);

    size_t length = 0;
    while (NN_STATIC_CONDITION(1))
    {
        if ( length > MAXIMU_LENGTH || str[length] == NULL_CHAR)
        {
            break;
        }
        ++length;
    }
    if (length > MAXIMU_LENGTH)
    {
        NN_G3D_WARNING(0, "string is too long.");
        return 0;
    }
    return length;
}

typedef DynamicLengthStringBase<char> DynamicLengthString;

} // namespace detail

}}} // namespace nn::g3d::edit::ut


