﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "g3d_EditShaderArchive.h"

#include "../g3d_Allocator.h"
#include "g3d_EditShadingModel.h"
#include "g3d_EditShaderProgram.h"
#include "../util/g3d_ViewerUtility.h"
#include "../g3d_ViewerKeyManager.h"

namespace nn { namespace g3d { namespace viewer { namespace detail {

void UpdateProgram(nn::gfx::Device* pDevice, ResShadingModel* pResShadingModel, int programIndex) NN_NOEXCEPT
{
    NN_UNUSED(pDevice);
    EditShadingModel* pEditShadingModel = static_cast<EditShadingModel*>(pResShadingModel->ToData().pCallbackParam.Get());
    NN_G3D_VIEWER_ASSERT_NOT_NULL_DETAIL(pEditShadingModel, "%s\n", NN_G3D_VIEWER_RES_NAME(pResShadingModel, GetName()));

    EditShaderProgram* pEditShaderProgram = pEditShadingModel->GetEditShaderProgram(programIndex);
    NN_G3D_VIEWER_ASSERT_NOT_NULL_DETAIL(pEditShaderProgram, "%s\n", NN_G3D_VIEWER_RES_NAME(pResShadingModel, GetName()));

    const EditShaderArchive* pEditShaderArchive = pEditShadingModel->GetEditShaderArchive();
    NN_G3D_VIEWER_ASSERT_NOT_NULL_DETAIL(pEditShaderArchive, "%s\n", NN_G3D_VIEWER_RES_NAME(pResShadingModel, GetName()));

    if (!pEditShaderProgram->GetSendInfoCallbackFlag())
    {
        // 3DEditor から送られてきたプログラムと指し替わっているのでオリジナルのコールバックを呼んで初期化する
        pEditShaderArchive->CallOriginalUpdateCallback(pResShadingModel, programIndex);
        return;
    }

    ResShaderProgram* pResShaderProgram = pResShadingModel->GetShaderProgram(programIndex);
    NN_G3D_VIEWER_ASSERT_NOT_NULL_DETAIL(pResShaderProgram, "%s\n", NN_G3D_VIEWER_RES_NAME(pResShadingModel, GetName()));

    bool isProgramInitialized = pResShaderProgram->IsInitialized();
    if (!isProgramInitialized)
    {
        // 初期化されていない差し替え前のプログラムはオリジナルのコールバックを呼んで初期化する
        // そうしないと数フレームの間、プログラムが初期化されていない状態になってしまう
        pEditShaderArchive->CallOriginalUpdateCallback(pResShadingModel, programIndex);
    }

    pEditShaderProgram->ClearOptionInfo();

    const Bit32* staticKey = pResShadingModel->GetStaticKey(programIndex);
    int staticOptionCount = pResShadingModel->GetStaticOptionCount();

    for (int i = 0; i < staticOptionCount; ++i)
    {
        ResShaderOption* pResShaderOption = pResShadingModel->GetStaticOption(i);
        const char* optionId = pResShaderOption->GetName();
        int choiceIndex = pResShaderOption->ReadStaticKey(staticKey);
        const char* choice = pResShaderOption->GetChoiceName(choiceIndex);
        bool result = pEditShaderProgram->PushOptionInfo(optionId, choice);
        NN_G3D_VIEWER_ASSERT_DETAIL(result, "%s\n", NN_G3D_VIEWER_RES_NAME(pResShadingModel, GetName()));
    }

    const Bit32* dynamicKey = pResShadingModel->GetDynamicKey(programIndex);
    int dynamicOptionCount = pResShadingModel->GetDynamicOptionCount() - 1; // 最後の1つはシステム予約

    for (int i = 0; i < dynamicOptionCount; ++i)
    {
        ResShaderOption* pResShaderOption = pResShadingModel->GetDynamicOption(i);
        const char* optionId = pResShaderOption->GetName();
        int choiceIndex = pResShaderOption->ReadDynamicKey(dynamicKey);
        const char* choice = pResShaderOption->GetChoiceName(choiceIndex);
        bool result = pEditShaderProgram->PushOptionInfo(optionId, choice);
        NN_G3D_VIEWER_ASSERT_DETAIL(result, "%s\n", NN_G3D_VIEWER_RES_NAME(pResShadingModel, GetName()));
    }

    {
        bool result = pEditShaderProgram->MakeModifiedShaderProgramPacket();
        NN_G3D_VIEWER_ASSERT_DETAIL(result, "%s\n", NN_G3D_VIEWER_RES_NAME(pResShadingModel, GetName()));
    }

    pResShaderProgram->ToData().flag &= ~ResShaderProgram::Flag_UpdateRequired;
    pEditShaderProgram->SetSendInfoCallbackFlag(false);
}

EditShaderArchive::EditShaderArchive(nn::gfx::Device* pDevice, Allocator* allocator, ResShaderArchive* resShaderArchive) NN_NOEXCEPT
    : DeviceDependentObj(pDevice, allocator)
    , m_pTargetResShaderArchive(resShaderArchive)
{
    NN_G3D_VIEWER_ASSERT_NOT_NULL_DETAIL(allocator, "%s\n", NN_G3D_VIEWER_RES_NAME(resShaderArchive, GetName()));
    NN_G3D_VIEWER_ASSERT_NOT_NULL(resShaderArchive);

    m_pOriginalUpdateProgramCallback = m_pTargetResShaderArchive->GetUpdateProgramCallback();
}

bool
EditShaderArchive::Attach() NN_NOEXCEPT
{
    NN_G3D_VIEWER_ASSERT_NOT_NULL(m_pTargetResShaderArchive);

    int shadingModelCount = m_pTargetResShaderArchive->GetShadingModelCount();
    if (shadingModelCount > 0)
    {
        size_t bufferSize = FixedSizeArray<EditShadingModel>::CalculateBufferSize(shadingModelCount);
        void* buffer = m_pAllocator->Allocate(bufferSize, nn::g3d::detail::Alignment_Default, AllocateType_EditObj);
        if (buffer == nullptr) // バッファ確保失敗
        {
            return false;
        }

        m_ShadingModelArray.SetArrayBuffer(buffer, shadingModelCount);
        bool isFailed = false;
        for(int shadingModelIndex = 0, end = m_ShadingModelArray.GetCount(); shadingModelIndex < end; ++shadingModelIndex)
        {
            nn::g3d::ResShadingModel* pShadingModel = m_pTargetResShaderArchive->GetShadingModel(shadingModelIndex);
            EditShadingModel* pEditShadingModel = m_ShadingModelArray.UnsafeAt(shadingModelIndex);
            ViewerKeyType key = ViewerKeyManager::GetInstance().FindKey(m_pTargetResShaderArchive);
            NN_G3D_VIEWER_ASSERT_VALID_KEY(key);
            new (pEditShadingModel) EditShadingModel(
                m_pDevice,
                m_pAllocator, key, shadingModelIndex, pShadingModel, this);
            isFailed |= !pEditShadingModel->Init();
        }

        // 初期化失敗のものが一つでもあった場合は、インスタンスを破棄して失敗にする。
        if (isFailed)
        {
            Detach();
            return false;
        }
    }
    return true;
}

void
EditShaderArchive::Detach() NN_NOEXCEPT
{
    ResetToOriginal();
    Destroy();

    // UpdateProgramCallback をオリジナルのものに戻す
    if (m_pOriginalUpdateProgramCallback != m_pTargetResShaderArchive->GetUpdateProgramCallback())
    {
        m_pTargetResShaderArchive->SetUpdateProgramCallback(m_pOriginalUpdateProgramCallback);
    }
}

void
EditShaderArchive::ResetToOriginal() NN_NOEXCEPT
{
    int shadingModelSize = m_pTargetResShaderArchive->GetShadingModelCount();
    for (int i = 0; i < shadingModelSize; ++i)
    {
        EditShadingModel* editShadingModel = m_ShadingModelArray.UnsafeAt(i);
        editShadingModel->ResetToOriginal();
    }
}

void
EditShaderArchive::Destroy() NN_NOEXCEPT
{
    DestroyShadingModels();
}

void
EditShaderArchive::UpdateShaderProgram(int shadingModelIndex, int shaderProgramIndex, ResShaderArchive* pResShaderArchive) NN_NOEXCEPT
{
    NN_G3D_VIEWER_ASSERT_DETAIL(shadingModelIndex >= 0, "%s\n", NN_G3D_VIEWER_RES_NAME(pResShaderArchive, GetName()));
    if (shadingModelIndex >= m_ShadingModelArray.GetCount())
    {
        return;// 範囲外なので処理しない
    }

    EditShadingModel* editShadingModel = m_ShadingModelArray.UnsafeAt(shadingModelIndex);
    editShadingModel->UpdateShadingModel(pResShaderArchive);
    EditShaderProgram* editShaderProgram = editShadingModel->GetEditShaderProgram(shaderProgramIndex);
    if (editShaderProgram == nullptr)
    {
        return;
    }
    editShaderProgram->UpdateShaderProgram(pResShaderArchive);
}

void
EditShaderArchive::UpdateShadingModels(int shadingModelIndices[], int indexCount) NN_NOEXCEPT
{
    m_pTargetResShaderArchive->SetUpdateProgramCallback(UpdateProgram);
    for (int i = 0; i < indexCount; ++i)
    {
        ResShadingModel* resShadingModel = m_pTargetResShaderArchive->GetShadingModel(shadingModelIndices[i]);
        EditShadingModel* editShadingModel = m_ShadingModelArray.UnsafeAt(shadingModelIndices[i]);
        int shaderProgramSize = resShadingModel->GetShaderProgramCount();
        for (int j = 0; j < shaderProgramSize; ++j)
        {
            ResShaderProgram* resShaderProgram = resShadingModel->GetShaderProgram(j);
            resShaderProgram->ToData().flag |= ResShaderProgram::Flag_UpdateRequired;
            editShadingModel->GetEditShaderProgram(j)->SetSendInfoCallbackFlag(true);
        }
    }
}

void
EditShaderArchive::DestroyShadingModels() NN_NOEXCEPT
{
    for(int i = 0, end = m_ShadingModelArray.GetCount(); i < end; ++i)
    {
        EditShadingModel* editShadingModel = m_ShadingModelArray.UnsafeAt(i);
        editShadingModel->Destroy();
    }

    if (void* buffer = m_ShadingModelArray.GetArrayBufferPtr())
    {
        m_ShadingModelArray.Clear();
        m_pAllocator->Free(buffer);
    }
}

void
EditShaderArchive::SendModifiedShaderPrograms(EditSocketBase* pSocket) NN_NOEXCEPT
{
    ScopedLock scopedLock(m_ShadingModelArray);
    for (int shadingModelIndex = 0, shadingModelCount = m_ShadingModelArray.GetCount();
        shadingModelIndex < shadingModelCount; ++shadingModelIndex)
    {
        EditShadingModel* shadingModel = m_ShadingModelArray.UnsafeAt(shadingModelIndex);
        shadingModel->SendModifiedShaderPrograms(pSocket);
    }
}

}}}} // namespace nn::g3d::viewer::detail

