﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "g3d_EditShapeObj.h"

#include <nn/g3d/g3d_ModelObj.h>
#include <nn/g3d/g3d_ShapeObj.h>
#include <nn/g3d/g3d_ResFile.h>
#include "../g3d_Allocator.h"
#include "../util/g3d_ViewerUtility.h"

namespace nn { namespace g3d { namespace viewer { namespace detail {

EditShapeObj::EditShapeObj(
    nn::gfx::Device* pDevice,
    Allocator* pAllocator,
    int index,
    nn::g3d::ModelObj* pOwnerModelObj,
    nn::g3d::ShapeObj* pShapeObj) NN_NOEXCEPT
    : BlinkingObj(pDevice, pAllocator)
    , m_Index(index)
    , m_pOwnerModelObj(pOwnerModelObj)
    , m_pShapeObj(pShapeObj)
    , m_ViewDependent(false)
    , m_WorkBuffer(pAllocator, ShapeObj::Alignment_Buffer, AllocateType_DynamicBuffer)
    , m_WorkUserAreaBuffer(pAllocator, ShapeObj::Alignment_Buffer, AllocateType_DynamicBuffer)
    , m_WorkMeshArrayBuffer(pAllocator, ShapeObj::Alignment_Buffer, AllocateType_DynamicBuffer)
    , m_WorkSubMeshBoundingArrayBuffer(pAllocator, ShapeObj::Alignment_Buffer, AllocateType_DynamicBuffer)
    , m_BlockBufferObjectManager(UniformBlockBufferingCount, pAllocator)
    , m_ShapeBlockArray(pAllocator, ShapeObj::Alignment_Buffer, ShapeBlock())
    , m_CurrentForceMeshLodLevel(ForceMeshLodDisabled)
    , m_MemoryPoolManager(nullptr)
    , m_OriginalResVertexAttrFlags(pAllocator, "InvalidAttribute", 0)
{
    NN_G3D_VIEWER_ASSERT_NOT_NULL_DETAIL(pShapeObj, "%s\n", NN_G3D_VIEWER_RES_NAME(pOwnerModelObj->GetResource(), GetName()));
    nn::g3d::ResShape* pResShape = const_cast<nn::g3d::ResShape*>(pShapeObj->GetResource());
    NN_G3D_VIEWER_ASSERT_NOT_NULL_DETAIL(pResShape, "%s\n", NN_G3D_VIEWER_RES_NAME(pOwnerModelObj->GetResource(), GetName()));
    m_pOriginalResShape = pResShape;

    m_pOriginalBuffer = pShapeObj->GetBufferPtr();
    m_OriginalViewCount = pShapeObj->GetViewCount();
    m_OriginalIsViewDependent = pShapeObj->IsViewDependent();
    m_OriginalIsBounding = pShapeObj->GetBounding() != nullptr;
    m_OriginalBufferingCount = pShapeObj->GetBufferingCount();

    m_pOriginalBlockMemoryPool = pShapeObj->GetMemoryPoolPtr();
    m_OriginalBlockMemoryPoolOffset = pShapeObj->GetMemoryPoolOffset();
    m_OriginalBlockBufferSize = pShapeObj->CalculateBlockBufferSize(m_pDevice);
    if (pShapeObj->GetUserAreaSize() > 0)
    {
        // ユーザエリアサイズが指定されていてワークバッファが確保できない場合はアサートで落とす。
        bool result = m_WorkUserAreaBuffer.Resize(pShapeObj->GetUserAreaSize());
        NN_G3D_VIEWER_ASSERT_DETAIL(result, "%s\n", NN_G3D_VIEWER_RES_NAME(pOwnerModelObj->GetResource(), GetName()));
    }

    m_pOriginalMeshArray = &pResShape->ToData().pMeshArray.Get()->ToData();
    m_pMeshArray = m_pOriginalMeshArray;

    m_pOriginalSubMeshBoundingArray = pResShape->ToData().pSubMeshBoundingArray.Get();
    m_pSubMeshBoundingArray = m_pOriginalSubMeshBoundingArray;

    m_pOriginalBlockBufferArray = pShapeObj->GetShapeBlockArray();
    m_OriginalShapeBlockCount = pShapeObj->GetShapeBlockCount();

    {
        void* buffer = m_pAllocator->Allocate(sizeof(MemoryPoolManager), nn::DefaultAlignment, AllocateType_Other);
        NN_G3D_VIEWER_ASSERT_NOT_NULL(buffer);
        m_MemoryPoolManager = new (buffer) MemoryPoolManager(UniformBlockBufferingCount, pDevice, pAllocator);
    }

    // 動的頂点バッファーの頂点属性を記録しておく
    {
        m_pOriginalDynamicVertexBufferPtr = pShapeObj->GetDynamicVertexBufferArray();

        if (m_pOriginalDynamicVertexBufferPtr)
        {
            // インデックスで管理するとシェーダーのリアルタイム編集によって順番が変更される可能性があるため、属性名で管理
            ResVertex* pResVertex = pResShape->GetVertex();
            for (int vertexAttrIndex = 0, vertexAttrCount = pResVertex->GetVertexAttrCount(); vertexAttrIndex < vertexAttrCount; ++vertexAttrIndex)
            {
                ResVertexAttr* pResVertexAttr = pResVertex->GetVertexAttr(vertexAttrIndex);
                nn::util::BinString* attributeName = pResVertexAttr->ToData().pName.Get();
                char* pBuffer = static_cast<char*>(m_pAllocator->Allocate(attributeName->GetLength(), NN_ALIGNOF(char), AllocateType_Other));
                memcpy(pBuffer, attributeName->GetData(), attributeName->GetLength());
                m_OriginalResVertexAttrFlags.Register(pBuffer, pResVertexAttr->ToData().flag);
            }
        }
    }
}

EditShapeObj::~EditShapeObj() NN_NOEXCEPT
{
    // 頂点属性フラグを破棄
    {
        for (int resVertexAttrFlagIndex = 0, resVertexAttrFlagCount = m_OriginalResVertexAttrFlags.GetCount(); resVertexAttrFlagIndex < resVertexAttrFlagCount; ++resVertexAttrFlagIndex)
        {
            const char* pBuffer = m_OriginalResVertexAttrFlags.GetKey(resVertexAttrFlagIndex);
            NN_G3D_VIEWER_ASSERT_NOT_NULL(pBuffer);
            m_pAllocator->Free(const_cast<char*>(pBuffer));
        }
    }
    m_MemoryPoolManager->~MemoryPoolManager();
    m_pAllocator->Free(m_MemoryPoolManager);
    m_MemoryPoolManager = nullptr;
    m_pOriginalDynamicVertexBufferPtr = nullptr;

    for (int bufferIndex = 0, bufferCount = m_BlockBufferObjectManager.GetBufferCount(); bufferIndex < bufferCount; ++bufferIndex)
    {
        if (m_BlockBufferObjectManager.GetBufferSize(bufferIndex) > 0)
        {
            nn::gfx::Buffer* pShapeBlockArray = m_BlockBufferObjectManager.GetBuffer<nn::gfx::Buffer>(bufferIndex);
            size_t bufferSize = m_BlockBufferObjectManager.GetCurrentBufferSize();
            int arrayLength = static_cast<int>( bufferSize / sizeof(nn::gfx::Buffer));
            FinalizeBuffers(m_pDevice, pShapeBlockArray, arrayLength);
        }
    }
}

void
EditShapeObj::ResetToOriginal() NN_NOEXCEPT
{
    SaveShapeBlockData();

    ResetLodLevel();
    m_pOriginalResShape->ToData().pMeshArray.Set(static_cast<ResMesh*>(m_pOriginalMeshArray));
    m_pOriginalResShape->ToData().pSubMeshBoundingArray.Set(static_cast<nn::g3d::Bounding*>(m_pOriginalSubMeshBoundingArray));

    if (m_OriginalBlockBufferSize > 0)
    {
        // Shape Uniform Blocks
        FinalizeBuffers(m_pDevice, m_pOriginalBlockBufferArray, m_OriginalShapeBlockCount * m_OriginalBufferingCount);

        // DynamicVertexBuffer
        if (m_pOriginalDynamicVertexBufferPtr)
        {
            const ResVertex* pResVertex = m_pOriginalResShape->GetVertex();
            for (int vertexBufferIndex = 0, vertexBufferCount = pResVertex->GetVertexBufferCount(); vertexBufferIndex < vertexBufferCount; ++vertexBufferIndex)
            {
                if (!m_pOriginalDynamicVertexBufferPtr[vertexBufferIndex])
                {
                    continue;
                }
                FinalizeBuffers(m_pDevice, m_pOriginalDynamicVertexBufferPtr[vertexBufferIndex], m_OriginalBufferingCount);
            }
        }
    }

    InitializeShapeObj(m_pOriginalResShape, m_OriginalIsViewDependent, m_pOriginalBuffer, 0);

    NN_G3D_VIEWER_ASSERT(m_OriginalBlockBufferSize == m_pShapeObj->CalculateBlockBufferSize(m_pDevice));
    m_pShapeObj->SetupBlockBuffer(m_pDevice, m_pOriginalBlockMemoryPool, m_OriginalBlockMemoryPoolOffset, m_OriginalBlockBufferSize);

    LoadShapeBlockData();
}

void
EditShapeObj::SetLodLevel(int level) NN_NOEXCEPT
{
    ResetLodLevel();

    nn::g3d::ResShape* pResShape = const_cast<nn::g3d::ResShape*>(m_pShapeObj->GetResource());
    int meshCount = pResShape->GetMeshCount();
    if (level < 0 || level >= meshCount)
    {
        NN_G3D_WARNING(0, "Lod level out of range. 0 <= %d < %d\n", level, meshCount);
        return;
    }

    // 置き換え前の SubMeshBoundingArray は、メッシュ置き換え前に取得しておきます。
    int subMeshBoundingIndex = 0;
    for (int meshIndex = 0; meshIndex < level; ++meshIndex)
    {
        // LOD 毎に、SubMeshBounding + ShapeBounding 分だけ Bounding がある
        int shapeBoundingCount = 1;
        subMeshBoundingIndex += pResShape->GetSubMeshCount(meshIndex) + shapeBoundingCount;
    }
    nn::g3d::Bounding* pSrcSubMeshBoundingArray = &pResShape->ToData().pSubMeshBoundingArray.Get()[subMeshBoundingIndex];


    m_CurrentForceMeshLodLevel = level;

    ResMeshData* pUpdatedMeshArray = m_WorkMeshArrayBuffer.GetWorkBufferPtr<ResMeshData>();
    const ResMeshData& targetLodResMeshData = (&pResShape->ToData().pMeshArray.Get()->ToData())[level];

    // 全てのメッシュを特定の LOD レベルのメッシュに置き換えます。
    for (int meshIndex = 0; meshIndex < meshCount; ++meshIndex)
    {
        pUpdatedMeshArray[meshIndex] = targetLodResMeshData;
        pUpdatedMeshArray[meshIndex].pSubMeshArray.Set(const_cast<ResSubMesh*>(targetLodResMeshData.pSubMeshArray.Get()));
        pUpdatedMeshArray[meshIndex].pIndexBuffer.Set(const_cast<nn::gfx::Buffer*>(targetLodResMeshData.pIndexBuffer.Get()));
    }

    pResShape->ToData().pMeshArray.Set(reinterpret_cast<ResMesh*>(pUpdatedMeshArray));


    // 全ての SubMeshBounding を特定の LOD レベルの SubMeshBounding に置き換えます。
    // シェイプ LOD の表示切り替え用に SubMeshBoundingArray のバッファを作成
    nn::g3d::Bounding* pUpdatedSubMeshBoundingArray = m_WorkSubMeshBoundingArrayBuffer.GetWorkBufferPtr<nn::g3d::Bounding>();

    subMeshBoundingIndex = 0;
    for (int meshIndex = 0; meshIndex < meshCount; ++meshIndex)
    {
        // LOD 毎に、SubMeshBounding + ShapeBounding 分だけ Bounding がある
        int shapeBoundingCount = 1;
        int boundingCount = pResShape->GetSubMeshCount(meshIndex) + shapeBoundingCount;
        for (int boundingIndex = 0; boundingIndex < boundingCount; ++boundingIndex)
        {
            pUpdatedSubMeshBoundingArray[subMeshBoundingIndex + boundingIndex] = pSrcSubMeshBoundingArray[boundingIndex];
        }
        subMeshBoundingIndex += boundingCount;
    }

    pResShape->ToData().pSubMeshBoundingArray.Set(reinterpret_cast<nn::g3d::Bounding*>(pUpdatedSubMeshBoundingArray));

    // SubMeshBounding 更新のため、ShapeObj を再初期化します。
    {
        SaveShapeBlockData();

        // ShapeObj の再初期化
        {
            bool successResizeObjBuffer = ReinitBuffer(pResShape);
            NN_G3D_VIEWER_ASSERT_DETAIL(successResizeObjBuffer, "Insufficient memory.");
            SwitchBlockBuffer();

            InitializeShapeObj(pResShape, m_ViewDependent, m_WorkBuffer.GetWorkBufferPtr(), m_WorkBuffer.GetSize());

            ReplaceBlockBuffer();
        }

        // ユニフォームブロックの再セットアップ
        {
            m_MemoryPoolManager->SwitchCurrentBuffer();
            size_t memoryPoolSize = CalculateBlockBufferSize(m_ViewDependent, pResShape);
            if (memoryPoolSize > 0)
            {
                bool successAllocateMemoryPool = m_MemoryPoolManager->ResizeMemoryPoolBuffer(memoryPoolSize);
                // TODO: RuntimeErrorCode で結果を返すようにする
                NN_G3D_VIEWER_ASSERT_DETAIL(successAllocateMemoryPool, "Insufficient memory.");
            }

            size_t memoryPoolAlignment = GetBlockBufferAlignment();
            ptrdiff_t memoryPoolOffset = m_MemoryPoolManager->AllocateMemoryPool(memoryPoolSize, memoryPoolAlignment);
            NN_G3D_VIEWER_ASSERT(memoryPoolOffset != nn::gfx::util::MemoryPoolAllocator::InvalidOffset);

            bool successSetupBlockBuffer = m_pShapeObj->SetupBlockBuffer(
                m_pDevice, m_MemoryPoolManager->GetMemoryPool(), memoryPoolOffset, memoryPoolSize);
            NN_G3D_VIEWER_ASSERT(successSetupBlockBuffer);
        }

        LoadShapeBlockData();
    }
}

void EditShapeObj::SwitchBlockBuffer() NN_NOEXCEPT
{
    m_BlockBufferObjectManager.SwitchCurrentBuffer();
    if (m_BlockBufferObjectManager.GetCurrentBufferSize() > 0)
    {
        // 過去に使っていたバッファーを使い回すので一度終了処理する
        nn::gfx::Buffer* pOldShapeBlockArray = m_BlockBufferObjectManager.GetCurrentBuffer<nn::gfx::Buffer>();
        size_t oldShapeBlockArraySize = m_BlockBufferObjectManager.GetCurrentBufferSize();
        int arrayLength = static_cast<int>(oldShapeBlockArraySize / sizeof(nn::gfx::Buffer));
        FinalizeBuffers(m_pDevice, pOldShapeBlockArray, arrayLength);
    }
}

// BlockBuffer を BlockBufferObjectManager のものに差し替える。
void EditShapeObj::ReplaceBlockBuffer() NN_NOEXCEPT
{
    int blockBufferCount = CalculateBlockBufferArrayLength(m_ViewDependent);
    int dynamicVertexBufferCount = CalculateDynamicVertexBufferCount();
    bool successResizeBufferArray = m_BlockBufferObjectManager.ResizeCurrentBuffer(sizeof(nn::gfx::Buffer) * (blockBufferCount + dynamicVertexBufferCount));
    NN_G3D_VIEWER_ASSERT_DETAIL(successResizeBufferArray, "Insufficient memory.");

    nn::gfx::Buffer* pBufferArray = m_BlockBufferObjectManager.GetCurrentBuffer<nn::gfx::Buffer>();
    m_pShapeObj->SetShapeBlockArray(pBufferArray);
    for (int bufferIndex = 0; bufferIndex < blockBufferCount; ++bufferIndex)
    {
        new(pBufferArray) nn::gfx::Buffer();
        ++pBufferArray;
    }

    // DynamicVertexBuffer
    nn::gfx::Buffer** pDynamicVertexBufferPtr = m_pShapeObj->GetDynamicVertexBufferArray();
    if (!pDynamicVertexBufferPtr)
    {
        return;
    }

    for (int vertexBufferIndex = 0, vertexBufferCount = m_pShapeObj->GetVertexBufferCount(); vertexBufferIndex < vertexBufferCount; ++vertexBufferIndex)
    {
        if (!pDynamicVertexBufferPtr[vertexBufferIndex])
        {
            continue;
        }

        pDynamicVertexBufferPtr[vertexBufferIndex] = pBufferArray;
        for (int bufferIndex = 0; bufferIndex < m_OriginalBufferingCount; ++bufferIndex)
        {
            new(pBufferArray) nn::gfx::Buffer();
            ++pBufferArray;
        }
    }
}

// 動的頂点バッファーの情報をオリジナルを元に設定
void EditShapeObj::WriteDynamicVertexBufferIndexFromOriginal(ResVertex* pResVertex) NN_NOEXCEPT
{
    for (int vertexAttrIndex = 0, vertexAttrCount = pResVertex->GetVertexAttrCount(); vertexAttrIndex < vertexAttrCount; ++vertexAttrIndex)
    {
        ResVertexAttr* pResVertexAttr = pResVertex->GetVertexAttr(vertexAttrIndex);
        const char* attrName = pResVertexAttr->GetName();
        if (!m_OriginalResVertexAttrFlags.IsKeyRegistered(attrName))
        {
            continue;
        }

        nn::Bit32 flag = m_OriginalResVertexAttrFlags.FindValue(attrName);
        if (flag & ResVertexAttr::Flag_ContainDynamicVertexBuffer)
        {
            pResVertexAttr->SetDynamicVertexAttrEnabled();
        }
        else
        {
            pResVertexAttr->SetDynamicVertexAttrDisabled();
        }
    }
}

void
EditShapeObj::ResetLodLevel() NN_NOEXCEPT
{
    nn::g3d::ResShape* resShape = const_cast<nn::g3d::ResShape*>(m_pShapeObj->GetResource());
    resShape->ToData().pMeshArray.Set(static_cast<ResMesh*>(m_pMeshArray));
    resShape->ToData().pSubMeshBoundingArray.Set(static_cast<nn::g3d::Bounding*>(m_pSubMeshBoundingArray));
    m_CurrentForceMeshLodLevel = ForceMeshLodDisabled;
}

bool
EditShapeObj::ReinitBuffer(nn::g3d::ResShape* pResShape) NN_NOEXCEPT
{
    ShapeObj::Builder builder(pResShape);
    builder.ViewCount(m_pShapeObj->GetViewCount());

    if (m_ViewDependent)
    {
        builder.ViewDependent();
    }
    else
    {
        builder.ViewIndependent();
    }

    builder.BufferingCount(m_pShapeObj->GetBufferingCount());

    // ユーザエリアが指定されている場合はサイズを指定する。
    if (m_pShapeObj->GetUserAreaSize() > 0)
    {
        builder.UserAreaSize(m_pShapeObj->GetUserAreaSize());
    }

    if (m_pOwnerModelObj->GetBounding())
    {
        builder.SetBoundingEnabled();
    }

    // 動的頂点バッファーの情報をオリジナルを元に設定
    WriteDynamicVertexBufferIndexFromOriginal(pResShape->GetVertex());

    builder.CalculateMemorySize();
    size_t size = builder.GetWorkMemorySize();
    return m_WorkBuffer.Resize(size);
}

size_t
EditShapeObj::CalculateBlockBufferSize(bool isViewDependent, const ResShape* pResShape) const NN_NOEXCEPT
{
    size_t totalSize = 0ULL;

    // Shape Uniform Blocks
    {
        nn::gfx::Buffer::InfoType info;
        info.SetDefault();
        info.SetSize(sizeof(ShapeBlock));
        info.SetGpuAccessFlags(nn::gfx::GpuAccess_ConstantBuffer);
        size_t alignment = nn::gfx::Buffer::GetBufferAlignment(m_pDevice, info);

        int blockCount = isViewDependent ? m_OriginalViewCount : 1;
        totalSize += nn::util::align_up(sizeof(ShapeBlock), alignment) * blockCount * m_OriginalBufferingCount;
    }

    // DynamicVertexBuffer
    {
        // オリジナルの頂点属性から動的頂点バッファーのインデックスを収集
        const ResVertex* pResVertex = pResShape->GetVertex();
        ResVertex::DynamicVertexBufferMask dynamicVertexBufferMask;
        dynamicVertexBufferMask.Reset();

        for (int vertexAttrIndex = 0, vertexAttrCount = pResVertex->GetVertexAttrCount(); vertexAttrIndex < vertexAttrCount; ++vertexAttrIndex)
        {
            const ResVertexAttr* pResVertexAttr = pResVertex->GetVertexAttr(vertexAttrIndex);
            const char* attrName = pResVertexAttr->GetName();
            if (!m_OriginalResVertexAttrFlags.IsKeyRegistered(attrName))
            {
                continue;
            }

            nn::Bit8 flag = m_OriginalResVertexAttrFlags.FindValue(attrName);
            if ((flag & ResVertexAttr::Flag_ContainDynamicVertexBuffer) != ResVertexAttr::Flag_ContainDynamicVertexBuffer)
            {
                continue;
            }

            dynamicVertexBufferMask.Set(pResVertexAttr->GetBufferIndex());
        }

        nn::gfx::Buffer::InfoType bufferInfo;
        for (int vertexBufferIndex = 0, vertexBufferCount = pResVertex->GetVertexBufferCount(); vertexBufferIndex < vertexBufferCount; ++vertexBufferIndex)
        {
            if (!dynamicVertexBufferMask.Test(vertexBufferIndex))
            {
                continue;
            }

            const nn::gfx::Buffer::InfoType* pBufferInfo = pResVertex->GetVertexBufferInfo(vertexBufferIndex);
            memcpy(&bufferInfo, pBufferInfo, sizeof(nn::gfx::Buffer::InfoType));
            bufferInfo.SetGpuAccessFlags(bufferInfo.GetGpuAccessFlags() | nn::gfx::GpuAccess_UnorderedAccessBuffer);

            totalSize += nn::util::align_up(bufferInfo.GetSize(), nn::gfx::Buffer::GetBufferAlignment(m_pDevice, bufferInfo)) * m_OriginalBufferingCount;
        }
    }

    return totalSize;
}

size_t
EditShapeObj::GetBlockBufferAlignment() const NN_NOEXCEPT
{
    nn::gfx::Buffer::InfoType info;
    info.SetDefault();
    info.SetSize(sizeof(ShapeBlock));
    info.SetGpuAccessFlags(nn::gfx::GpuAccess_ConstantBuffer);

    return nn::gfx::Buffer::GetBufferAlignment(m_pDevice, info);
}

void
EditShapeObj::SetupShapeInstance(
    nn::g3d::ResShape* pResShape,
    nn::gfx::MemoryPool* pMemoryPool,
    ptrdiff_t memoryPoolOffset,
    size_t memoryPoolSize) NN_NOEXCEPT
{
    SaveShapeBlockData();

    UpdateLodLevel(pResShape);

    InitializeShapeObj(pResShape, m_ViewDependent, m_WorkBuffer.GetWorkBufferPtr(), m_WorkBuffer.GetSize());

    ReplaceBlockBuffer();

    size_t reqiredMemoryPoolSize = m_pShapeObj->CalculateBlockBufferSize(m_pDevice);
    NN_G3D_VIEWER_ASSERT(memoryPoolSize == reqiredMemoryPoolSize);

    m_pShapeObj->SetupBlockBuffer(m_pDevice, pMemoryPool, memoryPoolOffset, memoryPoolSize);

    LoadShapeBlockData();
}

void EditShapeObj::SaveShapeBlockData() NN_NOEXCEPT
{
    m_ShapeBlockArray.Resize(m_pShapeObj->GetViewCount() * m_pShapeObj->GetBufferingCount());
    m_ShapeBlockArray.Clear();
    for (int bufferIndex = 0, bufferingCount = m_pShapeObj->GetBufferingCount(); bufferIndex < bufferingCount; ++bufferIndex)
    {
        for (int viewIndex = 0, viewCount = m_pShapeObj->GetViewCount(); viewIndex < viewCount; ++viewIndex)
        {
            nn::gfx::Buffer& shapeBlock = *m_pShapeObj->GetShapeBlock(viewIndex, bufferIndex);
            ShapeBlock* pShpBuffer = shapeBlock.Map<ShapeBlock>();
            m_ShapeBlockArray.PushBack(*pShpBuffer);
            shapeBlock.Unmap();
        }
    }
}

void EditShapeObj::LoadShapeBlockData() NN_NOEXCEPT
{
    NN_G3D_VIEWER_ASSERT(m_ShapeBlockArray.GetCount() == m_pShapeObj->GetViewCount() * m_pShapeObj->GetBufferingCount());
    for (int bufferIndex = 0, bufferingCount = m_pShapeObj->GetBufferingCount(); bufferIndex < bufferingCount; ++bufferIndex)
    {
        for (int viewIndex = 0, viewCount = m_pShapeObj->GetViewCount(); viewIndex < viewCount; ++viewIndex)
        {
            nn::gfx::Buffer& shapeBlock = *m_pShapeObj->GetShapeBlock(viewIndex, bufferIndex);
            ShapeBlock* pShpBuffer = shapeBlock.Map<ShapeBlock>();
            Copy32<false>(pShpBuffer, &m_ShapeBlockArray[bufferIndex * viewCount + viewIndex], sizeof(ShapeBlock) >> 2);
            shapeBlock.FlushMappedRange(0, sizeof(ShapeBlock));
            shapeBlock.Unmap();
        }
    }
}

void
EditShapeObj::InitializeShapeObj(nn::g3d::ResShape* pResShape,
    bool isViewDependent,
    void* pWorkBuffer,
    size_t workBufferSize) NN_NOEXCEPT
{
    ShapeObj::Builder builder(pResShape);
    builder.ViewCount(m_OriginalViewCount);

    if (isViewDependent)
    {
        builder.ViewDependent();
    }
    else
    {
        builder.ViewIndependent();
    }

    if (m_OriginalIsBounding)
    {
        builder.SetBoundingEnabled();
    }

    builder.BufferingCount(m_OriginalBufferingCount);

    // ユーザエリアが指定されている場合は、値をワークバッファに一時退避
    if (m_pShapeObj->GetUserAreaSize() > 0)
    {
        builder.UserAreaSize(m_pShapeObj->GetUserAreaSize());
        Copy32<false>(m_WorkUserAreaBuffer.GetWorkBufferPtr(), m_pShapeObj->GetUserArea(), static_cast<int>(m_pShapeObj->GetUserAreaSize() >> 2));
    }

    // 動的頂点バッファーの情報をオリジナルを元に設定
    WriteDynamicVertexBufferIndexFromOriginal(pResShape->GetVertex());

    builder.CalculateMemorySize();
    size_t bufferSize = builder.GetWorkMemorySize();
    NN_G3D_VIEWER_ASSERT((workBufferSize == 0) || (bufferSize == workBufferSize));
    NN_UNUSED(workBufferSize);

    bool isShapeAnimCalculateEnabled = m_pShapeObj->IsShapeAnimCalculationEnabled();
    void* userPtr = m_pShapeObj->GetUserPtr();// UserPtr を保持するために一時退避
    pResShape->Cleanup(m_pDevice);
    pResShape->Setup(m_pDevice);

    bool success = builder.Build(m_pShapeObj, pWorkBuffer, bufferSize);
    NN_G3D_VIEWER_ASSERT(success);
    m_pShapeObj->SetUserPtr(userPtr);
    if (isShapeAnimCalculateEnabled)
    {
        m_pShapeObj->SetShapeAnimCalculationEnabled();
    }

    // ユーザエリアが指定されている場合は、事前に退避していた値をコピーして戻す。
    if (m_pShapeObj->GetUserAreaSize() > 0)
    {
        Copy32<false>(m_pShapeObj->GetUserArea(), m_WorkUserAreaBuffer.GetWorkBufferPtr(), static_cast<int>(m_pShapeObj->GetUserAreaSize() >> 2));
    }
}

void
EditShapeObj::UpdateLodLevel(nn::g3d::ResShape* pResShape) NN_NOEXCEPT
{
    // シェイプ LOD の表示切り替え用に MeshArray のバッファを作成
    m_pMeshArray = pResShape->ToData().pMeshArray.Get();
    size_t meshArraySize = sizeof(ResMeshData) * pResShape->GetMeshCount();
    bool result = m_WorkMeshArrayBuffer.Resize(meshArraySize);
    NN_G3D_VIEWER_ASSERT_DETAIL(result, "%s\n", NN_G3D_VIEWER_RES_NAME(m_pOwnerModelObj->GetResource(), GetName()));

    // シェイプ LOD の表示切り替え用に SubMeshBoundingArray のバッファを作成
    m_pSubMeshBoundingArray = pResShape->ToData().pSubMeshBoundingArray.Get();
    int maxSubMeshCount = 0;
    for (int meshIndex = 0; meshIndex < pResShape->GetMeshCount(); ++meshIndex)
    {
        int count = pResShape->GetSubMeshCount(meshIndex);
        if (count > maxSubMeshCount)
        {
            maxSubMeshCount = count;
        }
    }
    size_t boundingArraySize = sizeof(nn::g3d::Bounding) * (maxSubMeshCount * pResShape->GetMeshCount() + pResShape->GetMeshCount());
    result = m_WorkSubMeshBoundingArrayBuffer.Resize(boundingArraySize);
    NN_G3D_VIEWER_ASSERT_DETAIL(result, "%s\n", NN_G3D_VIEWER_RES_NAME(m_pOwnerModelObj->GetResource(), GetName()));
}

int EditShapeObj::CalculateDynamicVertexBufferCount() const NN_NOEXCEPT
{
    const ResVertex* pResVertex = m_pShapeObj->GetResVertex();
    ResVertex::DynamicVertexBufferMask dynamicVertexBufferMask;
    pResVertex->CalculateDynamicVertexBufferIndex(&dynamicVertexBufferMask);
    return dynamicVertexBufferMask.CountPopulation() * m_OriginalBufferingCount;
}

void EditShapeObj::Show() NN_NOEXCEPT
{
    const ResShape* pResShape = m_pShapeObj->GetResource();
    int meshCount = pResShape->GetMeshCount();
    NN_G3D_VIEWER_ASSERT_DETAIL(meshCount == m_pOriginalResShape->GetMeshCount(), "%s\n", NN_G3D_VIEWER_RES_NAME(m_pOwnerModelObj->GetResource(), GetName()));

    int currentLodLevel = m_CurrentForceMeshLodLevel;
    ResetLodLevel();

    for (int meshIndex = 0; meshIndex < meshCount; ++meshIndex)
    {
        const ResMesh* pResMesh = pResShape->GetMesh(meshIndex);
        const ResMesh* pOriginalResMesh = m_pOriginalResShape->GetMesh(meshIndex);
        int subMeshCount = pResMesh->GetSubMeshCount();
        NN_G3D_VIEWER_ASSERT_DETAIL(subMeshCount == pOriginalResMesh->GetSubMeshCount(), "%s\n", NN_G3D_VIEWER_RES_NAME(m_pOwnerModelObj->GetResource(), GetName()));
        for (int subMeshIndex = 0; subMeshIndex < subMeshCount; ++subMeshIndex)
        {
            ResSubMesh* pResSubMesh = const_cast<ResSubMesh*>(pResMesh->GetSubMesh(subMeshIndex));
            const ResSubMesh* pOriginalResSubMesh = pOriginalResMesh->GetSubMesh(subMeshIndex);
            pResSubMesh->ToData().count = pOriginalResSubMesh->GetCount();
            pResSubMesh->ToData().offset = static_cast<uint32_t>(pOriginalResSubMesh->GetOffset());
        }
    }

    if (currentLodLevel != ForceMeshLodDisabled)
    {
        SetLodLevel(currentLodLevel);
    }
}

void EditShapeObj::Hide() NN_NOEXCEPT
{
    const ResShape* pResShape = m_pShapeObj->GetResource();
    int meshCount = pResShape->GetMeshCount();
    for (int meshIndex = 0; meshIndex < meshCount; ++meshIndex)
    {
        ResMesh* pResMesh = const_cast<ResMesh*>(pResShape->GetMesh(meshIndex));
        int subMeshCount = pResMesh->GetSubMeshCount();
        NN_G3D_VIEWER_ASSERT(subMeshCount > 0);
        ResSubMesh* pFirstSubMesh = pResMesh->GetSubMesh(0);
        uint32_t firstSubMeshOffset = static_cast<uint32_t>(pFirstSubMesh->GetOffset());

        for (int subMeshIndex = 0; subMeshIndex < subMeshCount; ++subMeshIndex)
        {
            ResSubMesh* pResSubMesh = pResMesh->GetSubMesh(subMeshIndex);

            // 本当は0にしたいが、低レイヤーの描画APIがサイズ0の入力を正常に処理するか保証がないので、
            // 1ポリゴンだけ残す
            pResSubMesh->ToData().count = 3;
            // サブメッシュ範囲指定描画で破綻しないよう、先頭サブメッシュのオフセットに書き換える
            pResSubMesh->ToData().offset = firstSubMeshOffset;
        }
    }
}

bool EditShapeObj::IsVisibleByDefault() const NN_NOEXCEPT
{
    return true;
}

}}}} // namespace nn::g3d::edit::detail


