﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "g3d_ResourceManager.h"

#include "g3d_ViewerKeyManager.h"
#include "anim\g3d_EditAnimObj.h"
#include "anim\g3d_EditSceneAnimObj.h"
#include "anim/g3d_EditSkeletalAnimObj.h"
#include "anim/g3d_EditTexPatternAnimObj.h"
#include "anim/g3d_EditShaderParamAnimObj.h"
#include "anim/g3d_EditVisibilityAnimObj.h"
#include "anim/g3d_EditShapeAnimObj.h"
#include "anim/g3d_EditMaterialAnimObj.h"
#include "util\g3d_DynamicLengthString.h"
#include "util\g3d_ResourceEditUtility.h"
#include "shader\g3d_EditShaderArchive.h"
#include "g3d_ResourceDestructionScheduler.h"

#include <nn/g3d/g3d_ResFile.h>
#include <nn/g3d/g3d_ResShader.h>
#include <nn/diag/text/diag_SdkTextG3dviewer.h>

using namespace nn::g3d::viewer::detail;

namespace {
    template<typename TDeviceDependentObjType>
    TDeviceDependentObjType* FindDeviceDependentObj(
        nn::g3d::viewer::detail::SynchronizedDynamicPtrArray<TDeviceDependentObjType>& DeviceDependentObjArray,
        nn::g3d::ResFile* pResFile) NN_NOEXCEPT
    {
        int editAnimObjCount = DeviceDependentObjArray.GetCount();
        for (int animIndex = 0; animIndex < editAnimObjCount; ++animIndex)
        {
            TDeviceDependentObjType* pObj = DeviceDependentObjArray.UnsafeAt(animIndex);
            if (pObj->IsResFileBound(pResFile))
            {
                return pObj;
            }
        }

        return nullptr;
    }

    nn::gfx::ResTextureFile* SetupBufferAsResTextureFile(void* buffer, CallbackCaller& callback) NN_NOEXCEPT
    {
        nn::gfx::ResTextureFileData* pResTextureFileData = reinterpret_cast<nn::gfx::ResTextureFileData*>(buffer);
        NN_UNUSED(pResTextureFileData);
        NN_G3D_VIEWER_DEBUG_PRINT(
            "Setup ResTextureFile: name = %s, size = %zu, alignment = %zu\n",
            pResTextureFileData->fileHeader.GetFileName() != nullptr ? pResTextureFileData->fileHeader.GetFileName().data() : "nullptr",
            pResTextureFileData->fileHeader.GetFileSize(),
            pResTextureFileData->fileHeader.GetAlignment());

        nn::gfx::ResTextureFile* pResTextureFile = nn::gfx::ResTextureFile::ResCast(buffer);
        NN_G3D_VIEWER_ASSERT_NOT_NULL(pResTextureFile);

        {
            nn::g3d::viewer::TextureFileLoadedArg texLoadedArg;
            texLoadedArg.pResTextureFile = pResTextureFile;
            callback.Call(nn::g3d::viewer::CallbackType_TextureFileLoaded, &texLoadedArg);
        }

        return pResTextureFile;
    }

    nn::g3d::BindResult BindAnimation(
        nn::g3d::viewer::detail::EditModelObj* pEditModelObj,
        nn::g3d::viewer::detail::EditAnimObj* pEditAnimObj,
        float currentFrame,
        float previewStep) NN_NOEXCEPT
    {
        nn::g3d::BindResult result;

        bool success = pEditModelObj->AddBoundAnim(pEditAnimObj);
        if (!success)
        {
            result.SetFailureBit();
            return result;
        }

        result = pEditAnimObj->BindModelObj(pEditModelObj);
        pEditAnimObj->SetFrame(pEditModelObj, currentFrame);
        pEditAnimObj->SetStep(previewStep);
        return result;
    }

    int CompareSameModelObj(const EditModelObj* pEditModelObj, const nn::g3d::ModelObj* pFindModelObj) NN_NOEXCEPT
    {
        if (pEditModelObj->GetAttachedModelObjCount() == 0)
        {
            return -1;
        }

        for (int index = 0, count = pEditModelObj->GetAttachedModelObjCount(); index < count; ++index)
        {
            const nn::g3d::ModelObj* pModelObj = pEditModelObj->GetTargetModelObj(index);
            if (pModelObj == pFindModelObj)
            {
                return 0;
            }
        }

        return -1;
    }

    int CompareSameKey(const EditModelObj* a, const ViewerKeyType* b) NN_NOEXCEPT
    {
        if (a->GetKey() < *b)
        {
            return -1;
        }
        else if (a->GetKey() > *b)
        {
            return 1;
        }
        return 0;
    }

    int CompareSameResShaderArchive(const EditShaderArchive* a, const nn::g3d::ResShaderArchive* b) NN_NOEXCEPT
    {
        if (a->GetTargetResShaderArchive() < b)
        {
            return -1;
        }
        else if (a->GetTargetResShaderArchive() > b)
        {
            return 1;
        }
        return 0;
    }

    int CompareSameKey(const EditAnimObj* a, const ViewerKeyType* b) NN_NOEXCEPT
    {
        if (a->GetResFileKey() < *b)
        {
            return -1;
        }
        else if (a->GetResFileKey() > *b)
        {
            return 1;
        }
        return 0;
    }

    int CompareSameKey(const EditSceneAnimObj* a, const ViewerKeyType* b) NN_NOEXCEPT
    {
        if (a->GetResFileKey() < *b)
        {
            return -1;
        }
        else if (a->GetResFileKey() > *b)
        {
            return 1;
        }
        return 0;
    }

    void ExecuteSceneAnimBoundCallback(
        nn::g3d::viewer::detail::EditSceneAnimObj* editAnimObj, nn::g3d::viewer::detail::CallbackCaller* pCallback) NN_NOEXCEPT
    {
        NN_G3D_VIEWER_ASSERT_NOT_NULL(editAnimObj);
        NN_G3D_VIEWER_ASSERT_NOT_NULL(pCallback);

        nn::g3d::viewer::SceneAnimBoundArg arg;

        arg.pCameraAnimObjs = editAnimObj->GetCameraAnimObjPtrArray().GetArrayBufferPtr();
        arg.cameraAnimObjCount = editAnimObj->GetCameraAnimObjPtrArray().GetCount();

        arg.pLightAnimObjs = editAnimObj->GetLightAnimObjPtrArray().GetArrayBufferPtr();
        arg.lightAnimObjCount = editAnimObj->GetLightAnimObjPtrArray().GetCount();

        arg.pFogAnimObjs = editAnimObj->GetFogAnimObjPtrArray().GetArrayBufferPtr();
        arg.fogAnimObjCount = editAnimObj->GetFogAnimObjPtrArray().GetCount();

        pCallback->Call(nn::g3d::viewer::CallbackType_SceneAnimBound, &arg);
        editAnimObj->SetBindFlag(true);
    }

    void ExecuteSceneAnimUnboundCallback(
        nn::g3d::viewer::detail::EditSceneAnimObj* editAnimObj, nn::g3d::viewer::detail::CallbackCaller* pCallback) NN_NOEXCEPT
    {
        NN_G3D_VIEWER_ASSERT_NOT_NULL(editAnimObj);
        NN_G3D_VIEWER_ASSERT_NOT_NULL(pCallback);

        nn::g3d::viewer::SceneAnimUnboundArg arg;

        arg.pCameraAnimObjs = editAnimObj->GetCameraAnimObjPtrArray().GetArrayBufferPtr();
        arg.cameraAnimObjCount = editAnimObj->GetCameraAnimObjPtrArray().GetCount();

        arg.pLightAnimObjs = editAnimObj->GetLightAnimObjPtrArray().GetArrayBufferPtr();
        arg.lightAnimObjCount = editAnimObj->GetLightAnimObjPtrArray().GetCount();

        arg.pFogAnimObjs = editAnimObj->GetFogAnimObjPtrArray().GetArrayBufferPtr();
        arg.fogAnimObjCount = editAnimObj->GetFogAnimObjPtrArray().GetCount();

        pCallback->Call(nn::g3d::viewer::CallbackType_SceneAnimUnbound, &arg);
        editAnimObj->SetBindFlag(false);
    }

    template<typename TTarget>
    void ForceBindTextureIfBindable(
        TTarget* pTarget,
        const nn::gfx::ResTextureFile* pResTextureFile,
        ViewerKeyType textureKey,
        CallbackCaller& callback) NN_NOEXCEPT
    {
        // テクスチャのバインド情報を持っている場合は、バインド情報に含まれるテクスチャだけをバインドする
        bool isTextureBindable = pTarget->ExaminesTextureBindable(textureKey);
        if (!isTextureBindable)
        {
            return;
        }

        pTarget->ForceBindTexture(pResTextureFile, callback);
    }
}

class ResourceManager::AnimResourceManager : public DeviceDependentObj
{
public:
    AnimResourceManager(nn::gfx::Device* pDevice, Allocator* pAllocator, CallbackCaller* pCallback, ResourceDestructionScheduler* pDestrcutionScheduler) NN_NOEXCEPT
        : DeviceDependentObj(pDevice, pAllocator)
        , m_IsAnimationPlaying(false)
        , m_Frame(0.f)
        , m_FrameStep(1.f)
        , m_PreviewFrameStepRate(1.f)
        , m_PlayPolicy(EDIT_PLAY_POLICY_AUTO)
        , m_StartFrame(0.0f)
        , m_FrameCount(0.f)
        , m_IsLoopAnim(false)
        , m_EditAnimArray(pAllocator, nn::g3d::detail::Alignment_Default)
        , m_EditSceneAnimArray(pAllocator, nn::g3d::detail::Alignment_Default)
        , m_AnimResFileInfoArray(pAllocator, nn::g3d::detail::Alignment_Default, ResourceInfo())
        , m_pViewerCallback(pCallback)
        , m_pDestructionScheduler(pDestrcutionScheduler)
    {
    }

    int GetResourceCount() NN_NOEXCEPT
    {
        return m_AnimResFileInfoArray.GetCount() + m_EditAnimArray.GetCount() + m_EditSceneAnimArray.GetCount();
    }

    EditAnimObj* AddEditAnimObj(nn::g3d::ResFile* pResFile, FileDataKind kind) NN_NOEXCEPT
    {
        EditAnimObj* pEditAnimObj = nullptr;
        switch (kind)
        {
        case FILEDATA_SHADER_PARAM_ANIM:
            pEditAnimObj = AddEditShaderParamAnimObj(pResFile);
            break;

        case FILEDATA_MAT_COLOR_ANIM:
            pEditAnimObj = AddEditColorAnimObj(pResFile);
            break;

        case FILEDATA_TEXTURE_SRT_ANIM:
            pEditAnimObj = AddEditTexSrtAnimObj(pResFile);
            break;

        case FILEDATA_TEXTURE_PATTERN_ANIM:
            pEditAnimObj = AddEditTexPatternAnimObj(pResFile);
            break;

        case FILEDATA_SKELETAL_ANIM:
            pEditAnimObj = AddEditSkeletalAnimObj(pResFile);
            break;

        case FILEDATA_BONE_VISIBILITY_ANIM:
            pEditAnimObj = AddEditBoneVisibilityAnimObj(pResFile);
            break;

        case FILEDATA_MAT_VISIBILITY_ANIM:
            pEditAnimObj = AddEditMatVisibilityAnimObj(pResFile);
            break;

        case FILEDATA_SHAPE_ANIM:
            pEditAnimObj = AddEditShapeAnimObj(pResFile);
            break;

        case FILEDATA_MATERIAL_ANIM:
            pEditAnimObj = AddEditMaterialAnimObj(pResFile);
            break;

        default:
            NN_UNEXPECTED_DEFAULT;
        }

        return pEditAnimObj;
    }

    EditSceneAnimObj* AddEditSceneAnimObj(nn::g3d::ResFile* targetResFile) NN_NOEXCEPT
    {
        NN_G3D_VIEWER_ASSERT_NOT_NULL(targetResFile);
        NN_G3D_VIEWER_ASSERT_DETAIL(targetResFile->GetSceneAnimCount() > 0, "%s\n", NN_G3D_VIEWER_RES_NAME(targetResFile, GetName()));
        nn::g3d::ResSceneAnim* resAnim = targetResFile->GetSceneAnim(0);
        NN_G3D_VIEWER_ASSERT_NOT_NULL_DETAIL(resAnim, "%s\n", NN_G3D_VIEWER_RES_NAME(targetResFile, GetName()));

        void* buffer = m_pAllocator->Allocate(sizeof(EditSceneAnimObj), nn::g3d::detail::Alignment_Default, AllocateType_EditObj);
        NN_G3D_VIEWER_ASSERT_NOT_NULL_DETAIL(buffer, "%s\n", NN_G3D_VIEWER_RES_NAME(targetResFile, GetName()));//今は止める

        EditSceneAnimObj* editAnimObj = new(buffer) EditSceneAnimObj(m_pAllocator, targetResFile, resAnim);
        {
            bool result = editAnimObj->Attach();
            NN_G3D_VIEWER_ASSERT_DETAIL(result, "%s\n", NN_G3D_VIEWER_RES_NAME(targetResFile, GetName()));//今は止める
            result = editAnimObj->Setup();
            NN_G3D_VIEWER_ASSERT_DETAIL(result, "%s\n", NN_G3D_VIEWER_RES_NAME(targetResFile, GetName()));//今は止める
        }

        bool success = m_EditSceneAnimArray.PushBack(editAnimObj);
        if (!success)
        {
            return nullptr;
        }

        // アニメーションが追加されたのでグローバルプレイポリシーの再設定を行う
        ResetPlayPolicy();

        return editAnimObj;
    }

    Iter<EditAnimObj*> DeleteEditAnimObj(EditAnimObj* pDeleteTargetEditAnim) NN_NOEXCEPT
    {
        NN_G3D_VIEWER_ASSERT_NOT_NULL(pDeleteTargetEditAnim);

        ScopedLock scopedLock(m_EditAnimArray);

        int index = m_EditAnimArray.IndexOf(pDeleteTargetEditAnim);
        NN_G3D_VIEWER_ASSERT(index >= 0);

        pDeleteTargetEditAnim->~EditAnimObj();
        m_pAllocator->Free(pDeleteTargetEditAnim);

        return m_EditAnimArray.Erase(m_EditAnimArray.Begin() + index);
    }

    void DeleteEditAnimObj(ViewerKeyType resFileKey) NN_NOEXCEPT
    {
        EditAnimObj* pDeleteTargetEditAnim = FindEditAnimObj(resFileKey);
        while (pDeleteTargetEditAnim != nullptr)
        {
            NN_G3D_VIEWER_DEBUG_PRINT("Deleting...\n");
            pDeleteTargetEditAnim->PrintAnimInfo();
            DeleteEditAnimObj(pDeleteTargetEditAnim);
            pDeleteTargetEditAnim = FindEditAnimObj(resFileKey);
        }
    }

    void DeleteEditSceneAnimObj(ViewerKeyType key) NN_NOEXCEPT
    {
        EditSceneAnimObj* pDeleteTargetEditAnim = FindEditSceneAnimObj(key);
        if (pDeleteTargetEditAnim == nullptr)
        {
            return;
        }

        DeleteEditSceneAnimObj(pDeleteTargetEditAnim);
    }

    Iter<EditSceneAnimObj*> DeleteEditSceneAnimObj(EditSceneAnimObj* pDeleteTargetEditAnim) NN_NOEXCEPT
    {
        ScopedLock scopedLock(m_EditSceneAnimArray);

        // EditSceneAnimObj の中で Update した方が良いかも
        if (m_pViewerCallback != nullptr && pDeleteTargetEditAnim->IsBound())
        {
            ExecuteSceneAnimUnboundCallback(pDeleteTargetEditAnim, m_pViewerCallback);
        }

        pDeleteTargetEditAnim->Detach();

        int index = m_EditSceneAnimArray.IndexOf(pDeleteTargetEditAnim);
        NN_G3D_VIEWER_ASSERT(index >= 0);
        pDeleteTargetEditAnim->~EditSceneAnimObj();
        m_pAllocator->Free(pDeleteTargetEditAnim);
        return m_EditSceneAnimArray.Erase(m_EditSceneAnimArray.Begin() + index);
    }

    void BindSceneAnimations(const BindAnimArg& arg) NN_NOEXCEPT
    {
        for (int i = 0, count = static_cast<int>(arg.animationKeySize); i < count; ++i)
        {
            EditSceneAnimObj* editAnimObj = FindEditSceneAnimObj(arg.animationKeys[i]);
            if (editAnimObj != nullptr)
            {
                ExecuteSceneAnimBoundCallback(editAnimObj, m_pViewerCallback);
            }
        }
    }

    void UnbindSceneAnimations(const BindAnimArg& arg) NN_NOEXCEPT
    {
        for (int i = 0, count = static_cast<int>(arg.animationKeySize); i < count; ++i)
        {
            EditSceneAnimObj* editAnimObj = FindEditSceneAnimObj(arg.animationKeys[i]);
            if (editAnimObj != nullptr)
            {
                ExecuteSceneAnimUnboundCallback(editAnimObj, m_pViewerCallback);
            }
        }
    }

    void DestructOrScheduleDestructAllResources() NN_NOEXCEPT
    {
        // アニメーション
        {
            for (Iter<EditAnimObj*> iter = m_EditAnimArray.Begin(), end = m_EditAnimArray.End();
                iter != end; ++iter)
            {
                iter = DeleteEditAnimObj(*iter);
            }
        }

        // シーン
        {
            for (Iter<EditSceneAnimObj*> iter = m_EditSceneAnimArray.Begin(), end = m_EditSceneAnimArray.End();
                iter != end; ++iter)
            {
                EditSceneAnimObj* pEditAnimObj = *iter;
                iter = DeleteEditSceneAnimObj(pEditAnimObj);
            }
        }

        for (Iter<ResourceInfo> iter = m_AnimResFileInfoArray.Begin(), end = m_AnimResFileInfoArray.End();
            iter != end; ++iter)
        {
            ResourceInfo resFileInfo = *iter;
            m_pDestructionScheduler->Register(resFileInfo);
            iter = m_AnimResFileInfoArray.Erase(iter);
        }
    }

    EditAnimObj* FindEditAnimObj(ViewerKeyType resFileKey) const NN_NOEXCEPT
    {
        return m_EditAnimArray.Find<ViewerKeyType>(&resFileKey, CompareSameKey);
    }

    bool HasEditAnim() const NN_NOEXCEPT
    {
        return m_EditAnimArray.GetCount() > 0;
    }

    bool HasEditSceneAnim() const NN_NOEXCEPT
    {
        return m_EditSceneAnimArray.GetCount() > 0;
    }

    int GetEditAnimCount() const NN_NOEXCEPT
    {
        return m_EditAnimArray.GetCount();
    }

    int GetEditSceneAnimCount() const NN_NOEXCEPT
    {
        return m_EditSceneAnimArray.GetCount();
    }

    float GetEditAnimFrameCount(int animIndex) const NN_NOEXCEPT
    {
        NN_G3D_VIEWER_ASSERT_INDEX_BOUNDS(animIndex, GetEditAnimCount());
        return m_EditAnimArray.At(animIndex)->GetFrameCount();
    }

    const char* GetEditAnimName(int animIndex) const NN_NOEXCEPT
    {
        NN_G3D_VIEWER_ASSERT_INDEX_BOUNDS(animIndex, GetEditAnimCount());
        EditAnimObj* pEditAnimObj = m_EditAnimArray.At(animIndex);
        return pEditAnimObj->GetName();
    }

    ViewerAnimKind GetViewerAnimKind(int animIndex) const NN_NOEXCEPT
    {
        NN_G3D_VIEWER_ASSERT_INDEX_BOUNDS(animIndex, GetEditAnimCount());
        EditAnimObj* pEditAnimObj = m_EditAnimArray.At(animIndex);
        return pEditAnimObj->GetAnimKind();
    }

    nn::g3d::ResFile* FindAnimResFile(ViewerKeyType resFileKey) NN_NOEXCEPT
    {
        for (int i = 0, end = m_AnimResFileInfoArray.GetCount(); i < end; ++i)
        {
            ResourceInfo info = m_AnimResFileInfoArray[i];
            if (resFileKey == ViewerKeyManager::GetInstance().FindKey(info.pResFile))
            {
                return info.pResFile;
            }
        }

        return nullptr;
    }

    ResourceInfo* FindAnimResFileInfo(ViewerKeyType resFileKey) NN_NOEXCEPT
    {
        for (int i = 0, end = m_AnimResFileInfoArray.GetCount(); i < end; ++i)
        {
            ResourceInfo* pInfo = &m_AnimResFileInfoArray[i];
            if (resFileKey == ViewerKeyManager::GetInstance().FindKey(pInfo->pResFile))
            {
                return pInfo;
            }
        }

        return nullptr;
    }

    void ScheduleDestroyAnimResFile(ViewerKeyType resFileKey) NN_NOEXCEPT
    {
        for (Iter<ResourceInfo> iter = m_AnimResFileInfoArray.Begin(), end = m_AnimResFileInfoArray.End(); iter != end; ++iter)
        {
            ResourceInfo* pResFileInfo = &(*iter);
            if (pResFileInfo->resFileKey == resFileKey)
            {
                m_pDestructionScheduler->Register(*pResFileInfo);
                m_AnimResFileInfoArray.Erase(iter);
                return;
            }
        }
    }

    EditSceneAnimObj* FindEditSceneAnimObj(ViewerKeyType key) const NN_NOEXCEPT
    {
        return m_EditSceneAnimArray.Find<ViewerKeyType>(&key, CompareSameKey);
    }

    SynchronizedDynamicPtrArray<EditAnimObj>& GetEditAnimArray()
    {
        return m_EditAnimArray;
    }

    SynchronizedDynamicPtrArray<EditSceneAnimObj>& GetEditSceneAnimArray()
    {
        return m_EditSceneAnimArray;
    }

    DynamicArray<ResourceInfo>& GetAnimResFileInfoArray()
    {
        return m_AnimResFileInfoArray;
    }

    void PrintAllEditAnimObjs() NN_NOEXCEPT
    {
        for (int i = 0, end = m_EditAnimArray.GetCount(); i < end; ++i)
        {
            EditAnimObj* pEditAnimObj = m_EditAnimArray.UnsafeAt(i);
            NN_G3D_VIEWER_DEBUG_PRINT("[%d] : key = %d\n", i, pEditAnimObj->GetResFileKey());
            pEditAnimObj->PrintAnimInfo();
        }
    }

    void PrintEditAnimObjs(ViewerKeyType resFileKey, ViewerKeyType boundModelKey) NN_NOEXCEPT
    {
        for (int animIndex = 0, animCount = m_EditAnimArray.GetCount(); animIndex < animCount; ++animIndex)
        {
            EditAnimObj* pEditAnimObj = m_EditAnimArray.UnsafeAt(animIndex);
            if (pEditAnimObj->GetResFileKey() == resFileKey)
            {
                for (int modelIndex = 0, boundModelObjCount = pEditAnimObj->GetBoundEditModelObjCount(); modelIndex < boundModelObjCount; ++modelIndex)
                {
                    EditModelObj* pBoundEditModelObj = pEditAnimObj->GetBoundEditModelObj(modelIndex);
                    if (pBoundEditModelObj == nullptr || pBoundEditModelObj->GetKey() != boundModelKey)
                    {
                        continue;
                    }

                    NN_G3D_VIEWER_DEBUG_PRINT("[%d] : key = %d\n", animIndex, pEditAnimObj->GetResFileKey());
                    pEditAnimObj->PrintAnimInfo();
                    return;
                }
            }
        }
    }

    void PrintAllResFiles() NN_NOEXCEPT
    {
        for (int resFileIndex = 0, end = m_AnimResFileInfoArray.GetCount(); resFileIndex < end; ++resFileIndex)
        {
            ResourceInfo info = m_AnimResFileInfoArray[resFileIndex];
            NN_UNUSED(info);
            NN_G3D_VIEWER_DEBUG_PRINT("AnimResFileInfoArray[%d] : resFileKey = %d, name = %s (%s)\n",
                resFileIndex, info.resFileKey, GetResName(info.pResFile, info.kind), GetFileDataKindString(info.kind));
        }
    }

    void EditAnimCurve(const EditAnimCurveArg& arg) NN_NOEXCEPT
    {
        EditAnimObj* targetEditAnim = FindEditAnimObj(arg.animationKey);
        if (targetEditAnim == nullptr)
        {
            return;
        }

        switch (arg.animationKind)
        {
        case EDIT_TARGET_MODEL_ANIMATION_SHADER_PARAM_CURVE:
            {
                EditShaderParamAnimObj* editAnim = static_cast<EditShaderParamAnimObj*>(targetEditAnim);
                EditShaderParamAnimCurve(editAnim, arg);
            }
            break;
        case EDIT_TARGET_MODEL_ANIMATION_TEX_PATTERN_CURVE:
            {
                EditTexPatternAnimObj* editAnim = static_cast<EditTexPatternAnimObj*>(targetEditAnim);
                EditTexPatternAnimCurve(editAnim, arg);
            }
            break;
        case EDIT_TARGET_MODEL_ANIMATION_BONE_VISIBILITY_CURVE:
            {
                EditMatVisibilityAnimObj* editAnim = static_cast<EditMatVisibilityAnimObj*>(targetEditAnim);
                EditMatVisibilityAnimCurve(editAnim, arg);
            }
            break;
        case EDIT_TARGET_MODEL_ANIMATION_MAT_VISIBILITY_CURVE:
            {
                EditBoneVisibilityAnimObj* editAnim = static_cast<EditBoneVisibilityAnimObj*>(targetEditAnim);
                EditBoneVisibilityAnimCurve(editAnim, arg);
            }
            break;
        case EDIT_TARGET_MODEL_ANIMATION_SHAPE_CURVE:
            {
                EditShapeAnimObj* editAnim = static_cast<EditShapeAnimObj*>(targetEditAnim);
                EditShapeAnimCurve(editAnim, arg);
            }
            break;
        case EDIT_TARGET_MODEL_ANIMATION_MATERIAL_CURVE:
            {
                EditMaterialAnimObj* pEditAnim = static_cast<EditMaterialAnimObj*>(targetEditAnim);
                EditMaterialAnimCurve(pEditAnim, arg);
            }
            break;
        default:
            NN_G3D_VIEWER_LOG("Unknown animation kind(=%x)\n", arg.animationKind);
            break;
        }
    }

    void EditSceneAnimCurve(const EditAnimCurveArg& arg) NN_NOEXCEPT
    {
        EditSceneAnimObj* editSceneAnim = FindEditSceneAnimObj(arg.animationKey);
        if (editSceneAnim == nullptr)
        {
            return;
        }

        switch (arg.animationKind)
        {
        case EDIT_TARGET_SCENE_ANIMATION_CAMERA_CURVE:
            {
                EditCameraAnimCurve(editSceneAnim, arg);
            }
            break;
        case EDIT_TARGET_SCENE_ANIMATION_LIGHT_CURVE:
            {
                EditLightAnimCurve(editSceneAnim, arg);
            }
            break;
        case EDIT_TARGET_SCENE_ANIMATION_FOG_CURVE:
            {
                EditFogAnimCurve(editSceneAnim, arg);
            }
            break;
        default:
            NN_G3D_VIEWER_LOG("Unknown scene animation kind(=%x)\n", arg.animationKind);
            break;
        }
    }

    void ClearCalculateFlagModelAnims() NN_NOEXCEPT
    {
        int animCount = m_EditAnimArray.GetCount();
        for (int i = 0; i < animCount; ++i)
        {
            EditAnimObj* animObj = m_EditAnimArray.UnsafeAt(i);
            NN_G3D_VIEWER_ASSERT_NOT_NULL(animObj);
            animObj->ClearCalculateFlag();
        }
    }

    void Reset() NN_NOEXCEPT
    {
        m_IsAnimationPlaying = false;
    }

    bool IsAnimationPlaying() const NN_NOEXCEPT
    {
        return m_IsAnimationPlaying;
    }

    void SetAnimPlaying(bool enable) NN_NOEXCEPT
    {
        m_IsAnimationPlaying = enable;

        {
            int size = m_EditAnimArray.GetCount();
            for (int i = 0; i < size; ++i)
            {
                EditAnimObj* editAnim = m_EditAnimArray.UnsafeAt(i);
                editAnim->SetFrameForAllBoundModels(m_Frame);
                editAnim->SetStep(GetPreviewStep());
            }
        }

        {
            int size = m_EditSceneAnimArray.GetCount();
            for (int i = 0; i < size; ++i)
            {
                EditSceneAnimObj* editSceneAnim = m_EditSceneAnimArray.UnsafeAt(i);
                editSceneAnim->SetFrame(m_Frame);
                editSceneAnim->SetStep(GetPreviewStep());
            }
        }
    }


    bool IsLoopAnim() const NN_NOEXCEPT
    {
        return m_IsLoopAnim;
    }

    void SetFrame(float frame) NN_NOEXCEPT
    {
        m_Frame = frame;
        SetAnimPlaying(false);// フレームを設定した場合は、アニメーションを止める
    }

    float GetFrameStep() const NN_NOEXCEPT
    {
        return m_FrameStep;
    }

    float GetPreviewFrameStepRate() const NN_NOEXCEPT
    {
        return m_PreviewFrameStepRate;
    }

    void SetStartFrame(float startFrame) NN_NOEXCEPT
    {
        m_StartFrame = startFrame;
    }

    float GetStartFrame() const NN_NOEXCEPT
    {
        return m_StartFrame;
    }

    void SetFrameCount(float frameCount) NN_NOEXCEPT
    {
        m_FrameCount = frameCount;
    }

    float GetFrameCount() const NN_NOEXCEPT
    {
        return m_FrameCount;
    }

    void SetFrameStep(float frameStep) NN_NOEXCEPT
    {
        m_FrameStep = frameStep;
        SetPreviewStep();
    }

    void SetPreviewFrameStepRate(float rate) NN_NOEXCEPT
    {
        m_PreviewFrameStepRate = rate;
        SetPreviewStep();
    }

    void CalculateSceneAnimation()
    {
        // シーンアニメーションの計算
        int size = m_EditSceneAnimArray.GetCount();
        if (!m_IsAnimationPlaying)
        {
            for (int i = 0; i < size; ++i)
            {
                EditSceneAnimObj* editAnim = m_EditSceneAnimArray.UnsafeAt(i);

                // バインドされていない場合は、スキップ
                if (!editAnim->IsBound())
                {
                    continue;
                }
                editAnim->SetFrame(m_Frame);
            }
        }

        for (int i = 0; i < size; ++i)
        {
            EditSceneAnimObj* editAnim = m_EditSceneAnimArray.UnsafeAt(i);

            // バインドされていない場合は、スキップ
            if (!editAnim->IsBound())
            {
                continue;
            }
            editAnim->Calculate();
            ApplySceneAnimRequestedArg arg;

            arg.pCameraAnimObjs = editAnim->GetCameraAnimObjPtrArray().GetArrayBufferPtr();
            arg.cameraAnimObjCount = editAnim->GetCameraAnimObjPtrArray().GetCount();

            arg.pLightAnimObjs = editAnim->GetLightAnimObjPtrArray().GetArrayBufferPtr();
            arg.lightAnimObjCount = editAnim->GetLightAnimObjPtrArray().GetCount();

            arg.pFogAnimObjs = editAnim->GetFogAnimObjPtrArray().GetArrayBufferPtr();
            arg.fogAnimObjCount = editAnim->GetFogAnimObjPtrArray().GetCount();

            m_pViewerCallback->Call(CallbackType_ApplySceneAnimRequested, &arg);
        }
    }

    void UpdateFrame() NN_NOEXCEPT
    {
        if (m_IsAnimationPlaying)
        {
            float endFrame = m_StartFrame + m_FrameCount;
            if (m_IsLoopAnim)
            {
                m_Frame = AnimFrameCtrl::PlayLoop(m_Frame + GetPreviewStep(), m_StartFrame, endFrame, nullptr);
            }
            else
            {
                m_Frame = AnimFrameCtrl::PlayOneTime(m_Frame + GetPreviewStep(), m_StartFrame, endFrame, nullptr);
                if (m_Frame >= endFrame)
                {
                    m_IsAnimationPlaying = false;
                }
            }
        }
    }

    float GetFrame() const NN_NOEXCEPT
    {
        return m_Frame;
    }

    void BindAnimations(EditModelObj* pEditModelObj, const BindAnimArg& arg) NN_NOEXCEPT
    {
        // バインド時に、PlayPolicy を設定
        SetPlayPolicy(m_PlayPolicy);

        for (int animKeyIndex = 0, count = static_cast<int>(arg.animationKeySize); animKeyIndex < count; ++animKeyIndex)
        {
            int animResFileKey = arg.animationKeys[animKeyIndex];
            bool isAlreadyBound = IsBound(pEditModelObj, animResFileKey);
            if (isAlreadyBound)
            {
                // アニメーションが既にモデルにバインド済みであれば無視
                continue;
            }

            EditAnimObj* pEditAnimObj = FindEditAnimObj(animResFileKey);
            if (pEditAnimObj == nullptr)
            {
                // 見つからなければ無視
                continue;
            }

            BindResult result = BindAnimation(pEditModelObj, pEditAnimObj, m_Frame, GetPreviewStep());
            if (!result.IsBound())
            {
                // バインドに失敗しても無視するが、正しくバインドできない不具合に備えてエラーログを出しておく
                NN_G3D_VIEWER_ERROR_LOG(
                    NN_TEXT_G3DVIEWER("モデル \"%s\" へのアニメーション \"%s\" のバインドに失敗しました。\n"),
                    pEditModelObj->GetName(), pEditAnimObj->GetName());

                // TODO: 作業中のデザイナが気づけるようにユーザメッセージ機能を応用して警告ログを出すようにするのを検討する
            }
        }

        // 元からバインドされているものもAutoの場合は変わる可能性があるので全体を設定しなおす。
        ResetPlayPolicy();
    }

    bool IsBound(EditModelObj* pEditModelObj, ViewerKeyType animResFileKey) const NN_NOEXCEPT
    {
        const EditAnimObj* pEditAnimObj = FindEditAnimObj(animResFileKey);
        if (pEditAnimObj == nullptr)
        {
            return false;
        }

        if (pEditModelObj->GetAttachedModelObjCount() == 0)
        {
            return false;
        }

        return pEditAnimObj->IsModelBound(pEditModelObj);
    }

    void ResetPlayPolicy() NN_NOEXCEPT
    {
        m_IsLoopAnim = CanLoopAnim();
        SetAnimLoopFlag(m_IsLoopAnim);
    }

    void ReloadTextureResource(const nn::gfx::ResTextureFile* pOldResTextureFile, const nn::gfx::ResTextureFile* pResTextureFile)
    {
        for (int index = 0, animCount = m_EditAnimArray.GetCount(); index < animCount; ++index)
        {
            EditAnimObj* pEditAnimObj = m_EditAnimArray.UnsafeAt(index);
            switch (pEditAnimObj->GetAnimKind())
            {
            case ViewerAnimKind_TexturePatternAnim:
                {
                    EditTexPatternAnimObj* pTexPatternAnimObj = static_cast<EditTexPatternAnimObj*>(pEditAnimObj);
                    pTexPatternAnimObj->ReloadTexture(pOldResTextureFile, pResTextureFile, *m_pViewerCallback);
                }
                break;
            case ViewerAnimKind_MaterialAnim:
                {
                    EditMaterialAnimObj* pMatAnimObj = static_cast<EditMaterialAnimObj*>(pEditAnimObj);
                    pMatAnimObj->ReloadTexture(pOldResTextureFile, pResTextureFile, *m_pViewerCallback);
                }
                break;
            default:
                break;
            }
        }
    }

    void SetRetargetingHostModel(EditModelObj* pRetargetHostEditModelObj, ViewerKeyType animResFileKey)
    {
        for (int animIndex = 0, animCount = m_EditAnimArray.GetCount(); animIndex < animCount; ++animIndex)
        {
            EditAnimObj* pEditAnimObj = m_EditAnimArray.UnsafeAt(animIndex);
            if (pEditAnimObj->GetResFileKey() == animResFileKey)
            {
                if (pEditAnimObj->GetAnimKind() == ViewerAnimKind_SkeletalAnim)
                {
                    EditSkeletalAnimObj* pEditSkeletalAnimObj = static_cast<EditSkeletalAnimObj*>(pEditAnimObj);
                    for (int modelIndex = 0, modelCount = pRetargetHostEditModelObj->GetAttachedModelObjCount(); modelIndex < modelCount; ++modelIndex)
                    {
                        pEditSkeletalAnimObj->SetRetargetingHostModel(pRetargetHostEditModelObj->GetTargetModelObj(modelIndex));
                    }
                }
            }
        }
    }

    void UnsetRetargetingHostModel(ViewerKeyType animResFileKey) NN_NOEXCEPT
    {
        for (int animIndex = 0, end = m_EditAnimArray.GetCount(); animIndex < end; ++animIndex)
        {
            EditAnimObj* pEditAnimObj = m_EditAnimArray.UnsafeAt(animIndex);
            if (pEditAnimObj->GetResFileKey() == animResFileKey)
            {
                if (pEditAnimObj->GetAnimKind() == ViewerAnimKind_SkeletalAnim)
                {
                    EditSkeletalAnimObj* pEditSkeletalAnimObj = static_cast<EditSkeletalAnimObj*>(pEditAnimObj);
                    pEditSkeletalAnimObj->UnsetRetargetingHostModel();
                }
            }
        }
    }

    void SetPlayMotionMirroringEnabled(ViewerKeyType animResFileKey, bool isEnabled)
    {
        for (int animIndex = 0, animCount = m_EditAnimArray.GetCount(); animIndex < animCount; ++animIndex)
        {
            EditAnimObj* pEditAnimObj = m_EditAnimArray.UnsafeAt(animIndex);
            if (pEditAnimObj->GetResFileKey() == animResFileKey)
            {
                if (pEditAnimObj->GetAnimKind() == ViewerAnimKind_SkeletalAnim)
                {
                    EditSkeletalAnimObj* pEditSkeletalAnimObj = static_cast<EditSkeletalAnimObj*>(pEditAnimObj);
                    pEditSkeletalAnimObj->SetPlayMotionMirroringEnabled(isEnabled);
                }
            }
        }
    }

    bool ExaminesTextureFileUsed(const nn::gfx::ResTextureFile* pResTextureFile) const NN_NOEXCEPT
    {
        for (int index = 0, animCount = m_EditAnimArray.GetCount(); index < animCount; ++index)
        {
            EditAnimObj* pEditAnimObj = m_EditAnimArray.UnsafeAt(index);
            switch (pEditAnimObj->GetAnimKind())
            {
            case ViewerAnimKind_TexturePatternAnim:
                {
                    EditTexPatternAnimObj* pTexPatternAnimObj = static_cast<EditTexPatternAnimObj*>(pEditAnimObj);
                    if (pTexPatternAnimObj->ExaminesTextureFileUsed(pResTextureFile))
                    {
                        return true;
                    }
                }
                break;
            case ViewerAnimKind_MaterialAnim:
                {
                    EditMaterialAnimObj* pMatAnimObj = static_cast<EditMaterialAnimObj*>(pEditAnimObj);
                    if (pMatAnimObj->ExaminesTextureFileUsed(pResTextureFile))
                    {
                        return true;
                    }
                }
                break;
            default:
                break;
            }
        }

        return false;
    }

    int FindActiveEditAnimIndex(const EditModelObj* pEditModelObj, int boundAnimIndex) const NN_NOEXCEPT
    {
        int boundCount = 0;
        int count = m_EditAnimArray.GetCount();
        int index = -1;
        for (int i = 0; i < count; ++i)
        {
            EditAnimObj* pEditAnim = m_EditAnimArray.UnsafeAt(i);
            for (int animIndex = 0, end = pEditAnim->GetBoundEditModelObjCount(); animIndex < end; ++animIndex)
            {
                if (pEditModelObj == pEditAnim->GetBoundEditModelObj(animIndex))
                {
                    if (boundCount == boundAnimIndex)
                    {
                        index = i;
                        break;
                    }
                    ++boundCount;
                    break;
                }
            }
        }
        return index;
    }

    RuntimeErrorCode ReloadEditSceneAnimObj(EditSceneAnimObj* pEditSceneAnimObj, nn::g3d::ResFile* pResFile) NN_NOEXCEPT
    {
        NN_G3D_VIEWER_ASSERT_NOT_NULL(pResFile);

        bool isBound = pEditSceneAnimObj->IsBound();
        if (m_pViewerCallback && isBound)
        {
            ExecuteSceneAnimUnboundCallback(pEditSceneAnimObj, m_pViewerCallback);
        }

        bool success = pEditSceneAnimObj->ReloadSceneAnimObj(pResFile);
        if (!success)
        {
            return RUNTIME_ERROR_CODE_UNKNOWN_ERROR;
        }

        if (m_pViewerCallback && isBound)
        {
            ExecuteSceneAnimBoundCallback(pEditSceneAnimObj, m_pViewerCallback);
        }

        // アニメーションが更新されたのでグローバルプレイポリシーの再設定を行う
        ResetPlayPolicy();

        return RUNTIME_ERROR_CODE_NO_ERROR;
    }

    void SetPlayPolicy(EditPlayPolicyKind kind) NN_NOEXCEPT
    {
        // 同じ値なら処理しない
        if (kind == m_PlayPolicy)
        {
            return;
        }
        m_PlayPolicy = kind;
        ResetPlayPolicy();
    }

private:
    EditAnimObj*
        AddEditShaderParamAnimObj(nn::g3d::ResFile* targetResFile) NN_NOEXCEPT
    {
        NN_G3D_VIEWER_ASSERT_NOT_NULL(targetResFile);

        void* buffer = m_pAllocator->Allocate(sizeof(EditShaderParamAnimObj), nn::g3d::detail::Alignment_Default, AllocateType_EditObj);
        NN_G3D_VIEWER_ASSERT_NOT_NULL_DETAIL(buffer, "%s\n", NN_G3D_VIEWER_RES_NAME(targetResFile, GetName()));//今は止める

        EditShaderParamAnimObj* editAnimObj = new (buffer) EditShaderParamAnimObj(m_pAllocator, targetResFile);
        {
            bool result = editAnimObj->CreateDataForEditingAnimCurve();
            NN_G3D_VIEWER_ASSERT_DETAIL(result, "%s\n", NN_G3D_VIEWER_RES_NAME(targetResFile, GetName()));//今は止める
        }

        if (m_EditAnimArray.PushBack(editAnimObj))
        {
            return editAnimObj;
        }
        return nullptr;
    }

    EditAnimObj*
        AddEditColorAnimObj(nn::g3d::ResFile* targetResFile) NN_NOEXCEPT
    {
        NN_G3D_VIEWER_ASSERT_NOT_NULL(targetResFile);

        void* buffer = m_pAllocator->Allocate(sizeof(EditColorAnimObj), nn::g3d::detail::Alignment_Default, AllocateType_EditObj);
        NN_G3D_VIEWER_ASSERT_NOT_NULL_DETAIL(buffer, "%s\n", NN_G3D_VIEWER_RES_NAME(targetResFile, GetName()));//今は止める

        EditColorAnimObj* editAnimObj = new (buffer) EditColorAnimObj(m_pAllocator, targetResFile);
        {
            bool result = editAnimObj->CreateDataForEditingAnimCurve();
            NN_G3D_VIEWER_ASSERT_DETAIL(result, "%s\n", NN_G3D_VIEWER_RES_NAME(targetResFile, GetName()));//今は止める
        }

        if (m_EditAnimArray.PushBack(editAnimObj))
        {
            return editAnimObj;
        }
        return nullptr;
    }

    EditAnimObj*
        AddEditTexSrtAnimObj(nn::g3d::ResFile* targetResFile) NN_NOEXCEPT
    {
        NN_G3D_VIEWER_ASSERT_NOT_NULL(targetResFile);

        void* buffer = m_pAllocator->Allocate(sizeof(EditTexSrtAnimObj), nn::g3d::detail::Alignment_Default, AllocateType_EditObj);
        NN_G3D_VIEWER_ASSERT_NOT_NULL_DETAIL(buffer, "%s\n", NN_G3D_VIEWER_RES_NAME(targetResFile, GetName()));//今は止める

        EditTexSrtAnimObj* editAnimObj = new(buffer) EditTexSrtAnimObj(m_pAllocator, targetResFile);
        {
            bool result = editAnimObj->CreateDataForEditingAnimCurve();
            NN_G3D_VIEWER_ASSERT_DETAIL(result, "%s\n", NN_G3D_VIEWER_RES_NAME(targetResFile, GetName()));//今は止める
        }

        if (m_EditAnimArray.PushBack(editAnimObj))
        {
            return editAnimObj;
        }
        return nullptr;
    }

    EditAnimObj*
        AddEditSkeletalAnimObj(nn::g3d::ResFile* targetResFile) NN_NOEXCEPT
    {
        NN_G3D_VIEWER_ASSERT_NOT_NULL(targetResFile);

        void* buffer = m_pAllocator->Allocate(sizeof(EditSkeletalAnimObj), nn::g3d::detail::Alignment_Default, AllocateType_EditObj);
        NN_G3D_VIEWER_ASSERT_NOT_NULL_DETAIL(buffer, "%s\n", NN_G3D_VIEWER_RES_NAME(targetResFile, GetName()));//今は止める

        EditSkeletalAnimObj* editAnimObj = new (buffer) EditSkeletalAnimObj(m_pAllocator, targetResFile);
        {
            bool result = editAnimObj->CreateDataForEditingAnimCurve();
            NN_G3D_VIEWER_ASSERT_DETAIL(result, "%s\n", NN_G3D_VIEWER_RES_NAME(targetResFile, GetName()));//今は止める
        }

        if (m_EditAnimArray.PushBack(editAnimObj))
        {
            return editAnimObj;
        }
        return nullptr;
    }

    EditAnimObj*
        AddEditTexPatternAnimObj(nn::g3d::ResFile* targetResFile) NN_NOEXCEPT
    {
        NN_G3D_VIEWER_ASSERT_NOT_NULL(targetResFile);

        void* buffer = m_pAllocator->Allocate(sizeof(EditTexPatternAnimObj), nn::g3d::detail::Alignment_Default, AllocateType_EditObj);
        NN_G3D_VIEWER_ASSERT_NOT_NULL_DETAIL(buffer, "%s\n", NN_G3D_VIEWER_RES_NAME(targetResFile, GetName()));//今は止める

        EditTexPatternAnimObj* editAnimObj = new(buffer) EditTexPatternAnimObj(m_pAllocator, targetResFile);
        {
            bool result = editAnimObj->CreateDataForEditingAnimCurve();
            NN_G3D_VIEWER_ASSERT_DETAIL(result, "%s\n", NN_G3D_VIEWER_RES_NAME(targetResFile, GetName()));//今は止める
        }

        if (m_EditAnimArray.PushBack(editAnimObj))
        {
            return editAnimObj;
        }
        return nullptr;
    }

    EditAnimObj* AddEditMaterialAnimObj(nn::g3d::ResFile* pTargetResFile) NN_NOEXCEPT
    {
        NN_G3D_VIEWER_ASSERT_NOT_NULL(pTargetResFile);

        void* buffer = m_pAllocator->Allocate(sizeof(EditMaterialAnimObj), nn::g3d::detail::Alignment_Default, AllocateType_EditObj);
        NN_G3D_VIEWER_ASSERT_NOT_NULL_DETAIL(buffer, "%s\n", NN_G3D_VIEWER_RES_NAME(pTargetResFile, GetName()));//今は止める

        EditMaterialAnimObj* pEditAnimObj = new (buffer) EditMaterialAnimObj(m_pAllocator, pTargetResFile);
        {
            bool result = pEditAnimObj->CreateDataForEditingAnimCurve();
            NN_G3D_VIEWER_ASSERT_DETAIL(result, "%s\n", NN_G3D_VIEWER_RES_NAME(pTargetResFile, GetName()));//今は止める
        }

        if (m_EditAnimArray.PushBack(pEditAnimObj))
        {
            return pEditAnimObj;
        }
        return nullptr;
    }

    EditAnimObj*
        AddEditBoneVisibilityAnimObj(nn::g3d::ResFile* targetResFile) NN_NOEXCEPT
    {
        NN_G3D_VIEWER_ASSERT_NOT_NULL(targetResFile);

        void* buffer = m_pAllocator->Allocate(sizeof(EditBoneVisibilityAnimObj), nn::g3d::detail::Alignment_Default, AllocateType_EditObj);
        NN_G3D_VIEWER_ASSERT_NOT_NULL_DETAIL(buffer, "%s\n", NN_G3D_VIEWER_RES_NAME(targetResFile, GetName()));//今は止める

        EditBoneVisibilityAnimObj* editAnimObj = new(buffer) EditBoneVisibilityAnimObj(m_pAllocator, targetResFile);
        {
            bool result = editAnimObj->CreateDataForEditingAnimCurve();
            NN_G3D_VIEWER_ASSERT_DETAIL(result, "%s\n", NN_G3D_VIEWER_RES_NAME(targetResFile, GetName()));//今は止める
        }

        if (m_EditAnimArray.PushBack(editAnimObj))
        {
            return editAnimObj;
        }
        return nullptr;
    }

    EditAnimObj*
        AddEditMatVisibilityAnimObj(nn::g3d::ResFile* targetResFile) NN_NOEXCEPT
    {
        NN_G3D_VIEWER_ASSERT_NOT_NULL(targetResFile);

        void* buffer = m_pAllocator->Allocate(sizeof(EditMatVisibilityAnimObj), nn::g3d::detail::Alignment_Default, AllocateType_EditObj);
        NN_G3D_VIEWER_ASSERT_NOT_NULL_DETAIL(buffer, "%s\n", NN_G3D_VIEWER_RES_NAME(targetResFile, GetName()));//今は止める

        EditMatVisibilityAnimObj* editAnimObj = new(buffer) EditMatVisibilityAnimObj(m_pAllocator, targetResFile);
        {
            bool result = editAnimObj->CreateDataForEditingAnimCurve();
            NN_G3D_VIEWER_ASSERT_DETAIL(result, "%s\n", NN_G3D_VIEWER_RES_NAME(targetResFile, GetName()));//今は止める
        }

        if (m_EditAnimArray.PushBack(editAnimObj))
        {
            return editAnimObj;
        }
        return nullptr;
    }

    EditAnimObj*
        AddEditShapeAnimObj(nn::g3d::ResFile* targetResFile) NN_NOEXCEPT
    {
        NN_G3D_VIEWER_ASSERT_NOT_NULL(targetResFile);

        void* buffer = m_pAllocator->Allocate(sizeof(EditShapeAnimObj), nn::g3d::detail::Alignment_Default, AllocateType_EditObj);
        NN_G3D_VIEWER_ASSERT_NOT_NULL_DETAIL(buffer, "%s\n", NN_G3D_VIEWER_RES_NAME(targetResFile, GetName()));//今は止める

        EditShapeAnimObj* editAnimObj = new (buffer) EditShapeAnimObj(m_pAllocator, targetResFile);
        {
            bool result = editAnimObj->CreateDataForEditingAnimCurve();
            NN_G3D_VIEWER_ASSERT_DETAIL(result, "%s\n", NN_G3D_VIEWER_RES_NAME(targetResFile, GetName()));//今は止める
        }

        if (m_EditAnimArray.PushBack(editAnimObj))
        {
            return editAnimObj;
        }
        return nullptr;
    }


    void EditShaderParamAnimCurve(EditShaderParamAnimObj* editAnimObj, const EditAnimCurveArg& arg) NN_NOEXCEPT
    {
        NN_G3D_VIEWER_ASSERT_NOT_NULL(editAnimObj);

        AnimCurveValue* value = static_cast<AnimCurveValue*>(arg.value);
        const nn::g3d::ResAnimCurve* resAnimCurve = static_cast<const nn::g3d::ResAnimCurve*>(arg.curveData);
        editAnimObj->EditCurve(
            value->materialIndex,
            value->curveIndex,
            resAnimCurve);
    }

    void EditTexPatternAnimCurve(EditTexPatternAnimObj* editAnimObj, const EditAnimCurveArg& arg) NN_NOEXCEPT
    {
        NN_G3D_VIEWER_ASSERT_NOT_NULL(editAnimObj);

        AnimCurveValue* value = static_cast<AnimCurveValue*>(arg.value);
        const nn::g3d::ResAnimCurve* resAnimCurve = static_cast<const nn::g3d::ResAnimCurve*>(arg.curveData);
        editAnimObj->EditCurve(
            value->materialIndex,
            value->curveIndex,
            resAnimCurve);
    }

    void EditMaterialAnimCurve(EditMaterialAnimObj* pEditAnimObj, const EditAnimCurveArg& arg) NN_NOEXCEPT
    {
        NN_G3D_VIEWER_ASSERT_NOT_NULL(pEditAnimObj);

        AnimCurveValue* value = static_cast<AnimCurveValue*>(arg.value);
        const nn::g3d::ResAnimCurve* resAnimCurve = static_cast<const nn::g3d::ResAnimCurve*>(arg.curveData);
        pEditAnimObj->EditCurve(
            value->materialIndex,
            value->curveIndex,
            resAnimCurve);
    }

    void EditMatVisibilityAnimCurve(EditMatVisibilityAnimObj* editAnimObj, const EditAnimCurveArg& arg) NN_NOEXCEPT
    {
        NN_G3D_VIEWER_ASSERT_NOT_NULL(editAnimObj);

        AnimCurveValue* value = static_cast<AnimCurveValue*>(arg.value);
        const nn::g3d::ResAnimCurve* resAnimCurve = static_cast<const nn::g3d::ResAnimCurve*>(arg.curveData);
        editAnimObj->EditCurve(
            value->curveIndex,
            resAnimCurve);
    }

    void EditBoneVisibilityAnimCurve(EditBoneVisibilityAnimObj* editAnimObj, const EditAnimCurveArg& arg) NN_NOEXCEPT
    {
        NN_G3D_VIEWER_ASSERT_NOT_NULL(editAnimObj);

        AnimCurveValue* value = static_cast<AnimCurveValue*>(arg.value);
        const nn::g3d::ResAnimCurve* resAnimCurve = static_cast<const nn::g3d::ResAnimCurve*>(arg.curveData);
        editAnimObj->EditCurve(
            value->curveIndex,
            resAnimCurve);
    }

    void EditShapeAnimCurve(EditShapeAnimObj* editAnimObj, const EditAnimCurveArg& arg) NN_NOEXCEPT
    {
        NN_G3D_VIEWER_ASSERT_NOT_NULL(editAnimObj);

        AnimCurveValue* value = static_cast<AnimCurveValue*>(arg.value);
        const nn::g3d::ResAnimCurve* resAnimCurve = static_cast<const nn::g3d::ResAnimCurve*>(arg.curveData);
        editAnimObj->EditCurve(
            value->vertexShapeIndex,
            value->curveIndex,
            resAnimCurve);
    }


    void EditCameraAnimCurve(EditSceneAnimObj* editAnimObj, const EditAnimCurveArg& arg) NN_NOEXCEPT
    {
        NN_G3D_VIEWER_ASSERT_NOT_NULL(editAnimObj);

        AnimCurveValue* value = static_cast<AnimCurveValue*>(arg.value);
        const nn::g3d::ResAnimCurve* resAnimCurve = static_cast<const nn::g3d::ResAnimCurve*>(arg.curveData);
        editAnimObj->EditCameraAnimCurve(
            value->cameraIndex,
            value->curveIndex,
            resAnimCurve);
    }

    void EditLightAnimCurve(EditSceneAnimObj* editAnimObj, const EditAnimCurveArg& arg) NN_NOEXCEPT
    {
        NN_G3D_VIEWER_ASSERT_NOT_NULL(editAnimObj);

        AnimCurveValue* value = static_cast<AnimCurveValue*>(arg.value);
        const nn::g3d::ResAnimCurve* resAnimCurve = static_cast<const nn::g3d::ResAnimCurve*>(arg.curveData);
        editAnimObj->EditLightAnimCurve(
            value->lightIndex,
            value->curveIndex,
            resAnimCurve);
    }

    void EditFogAnimCurve(EditSceneAnimObj* editAnimObj, const EditAnimCurveArg& arg) NN_NOEXCEPT
    {
        NN_G3D_VIEWER_ASSERT_NOT_NULL(editAnimObj);

        AnimCurveValue* value = static_cast<AnimCurveValue*>(arg.value);
        const nn::g3d::ResAnimCurve* resAnimCurve = static_cast<const nn::g3d::ResAnimCurve*>(arg.curveData);
        editAnimObj->EditFogAnimCurve(
            value->fogIndex,
            value->curveIndex,
            resAnimCurve);
    }


    bool CanLoopAnim() const NN_NOEXCEPT
    {
        bool hasLoopAnim = false;
        {
            int size = m_EditAnimArray.GetCount();
            for (int i = 0; i < size; ++i)
            {
                EditAnimObj* editAnim = m_EditAnimArray.UnsafeAt(i);
                if (editAnim->HasBoundModel())
                {
                    hasLoopAnim |= editAnim->IsLoopAnim();
                }
            }
        }

        // シーンアニメーションでループ設定があるものがあるか探す
        {
            int size = m_EditSceneAnimArray.GetCount();
            for (int i = 0; i < size; ++i)
            {
                EditSceneAnimObj* editSceneAnim = m_EditSceneAnimArray.UnsafeAt(i);
                hasLoopAnim |= editSceneAnim->ContainsLoopAnim();
            }
        }

        bool isLoopAnim = false;
        if (m_PlayPolicy == EDIT_PLAY_POLICY_AUTO)
        {
            if (hasLoopAnim)
            {
                isLoopAnim = true;
            }
        }
        else if (m_PlayPolicy == EDIT_PLAY_POLICY_LOOP)
        {
            isLoopAnim = true;
        }
        return isLoopAnim;
    }

    void SetAnimLoopFlag(bool enable) NN_NOEXCEPT
    {
        {
            int count = m_EditAnimArray.GetCount();
            for (int i = 0; i < count; ++i)
            {
                EditAnimObj* editAnim = m_EditAnimArray.UnsafeAt(i);
                editAnim->SetPlayPolicy(enable);
                editAnim->SetStep(GetPreviewStep());
            }
        }

        {
            int count = m_EditSceneAnimArray.GetCount();
            for (int i = 0; i < count; ++i)
            {
                EditSceneAnimObj* editSceneAnim = m_EditSceneAnimArray.UnsafeAt(i);
                editSceneAnim->SetPlayPolicy(enable);
                editSceneAnim->SetStep(GetPreviewStep());
            }
        }
    }

    void SetPreviewStep() NN_NOEXCEPT
    {
        {
            int size = m_EditAnimArray.GetCount();
            for (int i = 0; i < size; ++i)
            {
                EditAnimObj* editAnim = m_EditAnimArray.UnsafeAt(i);
                editAnim->SetStep(GetPreviewStep());
            }
        }

        {
            int size = m_EditSceneAnimArray.GetCount();
            for (int i = 0; i < size; ++i)
            {
                EditSceneAnimObj* editSceneAnim = m_EditSceneAnimArray.UnsafeAt(i);
                editSceneAnim->SetStep(GetPreviewStep());
            }
        }
    }

    float GetPreviewStep() const NN_NOEXCEPT
    {
        return m_FrameStep * m_PreviewFrameStepRate;
    }

private:
    bool m_IsAnimationPlaying;
    float m_Frame;
    float m_FrameStep;
    float m_PreviewFrameStepRate;
    EditPlayPolicyKind m_PlayPolicy;
    float m_StartFrame;
    float m_FrameCount;
    bool m_IsLoopAnim;

    SynchronizedDynamicPtrArray<EditAnimObj> m_EditAnimArray;
    SynchronizedDynamicPtrArray<EditSceneAnimObj> m_EditSceneAnimArray;

    // アニメーションは他のリソースと違ってモデルにバインドされたタイミングで
    // Obj が作られるので、取り回しの違いの都合で別管理にする
    DynamicArray<ResourceInfo> m_AnimResFileInfoArray;

    CallbackCaller* m_pViewerCallback;

    ResourceDestructionScheduler* m_pDestructionScheduler;
};

ResourceManager::ResourceManager(nn::gfx::Device* pDevice, Allocator* pAllocator, CallbackCaller* pCallbackCaller) NN_NOEXCEPT
    : DeviceDependentObj(pDevice, pAllocator)
    , m_EditModelArray(pAllocator, nn::g3d::detail::Alignment_Default)
    , m_EditShaderArchiveArray(pAllocator, nn::g3d::detail::Alignment_Default)
    , m_ResFileInfoArray(pAllocator, nn::g3d::detail::Alignment_Default, ResourceInfo())
    , m_pViewerCallback(pCallbackCaller)
    , m_pAnimResourceManager(nullptr)
{
    {
        void* pBuffer = m_pAllocator->Allocate(sizeof(ResourceDestructionScheduler), nn::DefaultAlignment, AllocateType_Other);
        NN_G3D_VIEWER_ASSERT_NOT_NULL_DETAIL(pBuffer, "Insufficient memory");
        m_pDestructionScheduler = new (pBuffer) ResourceDestructionScheduler(pDevice, pAllocator, pCallbackCaller);
    }

    {
        void* pBuffer = m_pAllocator->Allocate(sizeof(AnimResourceManager), nn::DefaultAlignment, AllocateType_Other);
        NN_G3D_VIEWER_ASSERT_NOT_NULL_DETAIL(pBuffer, "Insufficient memory");
        m_pAnimResourceManager = new (pBuffer) AnimResourceManager(pDevice, pAllocator, pCallbackCaller, m_pDestructionScheduler);
    }
}

ResourceManager::~ResourceManager() NN_NOEXCEPT
{
    m_pAnimResourceManager->~AnimResourceManager();
    m_pAllocator->Free(m_pAnimResourceManager);

    m_pDestructionScheduler->~ResourceDestructionScheduler();
    m_pAllocator->Free(m_pDestructionScheduler);
}

int ResourceManager::GetResourceCount() const NN_NOEXCEPT
{
    return m_ResFileInfoArray.GetCount()
        + m_EditModelArray.GetCount()
        + m_EditShaderArchiveArray.GetCount()
        + m_pAnimResourceManager->GetResourceCount()
        + m_pDestructionScheduler->GetResourceCount();
}


Iter<EditModelObj*> ResourceManager::ScheduleDestructEditModelObj(EditModelObj* pEditModelObj) NN_NOEXCEPT
{
    NN_G3D_VIEWER_REQUIRES_NOT_NULL(pEditModelObj);
    NN_G3D_VIEWER_LOG("Delete Target Model Name: \"%s\"\n", pEditModelObj->GetResModel()->GetName());

    ScopedLock scopedLock(m_EditModelArray);

    int index = m_EditModelArray.IndexOf(pEditModelObj);
    NN_G3D_VIEWER_ASSERT_DETAIL(index >= 0, "%s\n", NN_G3D_VIEWER_RES_NAME(pEditModelObj->GetResModel(), GetName()));

    // バインドされているEditAnimObj をアンバインド
    ForceUnbindAnimations(pEditModelObj);

    // 状態を初期状態へ戻す
    pEditModelObj->ResetToOriginal();

    // 遅延破棄
    ResourceInfo resourceInfo(pEditModelObj);
    m_pDestructionScheduler->Register(resourceInfo);
    return m_EditModelArray.Erase(m_EditModelArray.Begin() + index);
}

void ResourceManager::ScheduleDestructionAll() NN_NOEXCEPT
{
    // モデル
    {
        for (Iter<EditModelObj*> iter = m_EditModelArray.Begin(), end = m_EditModelArray.End();
            iter != end; ++iter)
        {
            iter = ScheduleDestructEditModelObj(*iter);
        }
    }

    // シェーダ
    {
        for (Iter<EditShaderArchive*> iter = m_EditShaderArchiveArray.Begin(), end = m_EditShaderArchiveArray.End();
            iter != end; ++iter)
        {
            iter = DeleteEditShaderArchive(*iter);
        }
    }

    // リソースファイル
    {
        for (Iter<ResourceInfo> iter = m_ResFileInfoArray.Begin(), end = m_ResFileInfoArray.End(); iter != end; ++iter)
        {
            ResourceInfo* pInfo = &*iter;
            m_pDestructionScheduler->Register(*pInfo);
            iter = m_ResFileInfoArray.Erase(iter);
        }
    }

    m_pAnimResourceManager->DestructOrScheduleDestructAllResources();
}

Iter<EditAnimObj*> ResourceManager::DeleteEditAnimObj(EditAnimObj* pDeleteTargetEditAnim) NN_NOEXCEPT
{
    NN_G3D_VIEWER_ASSERT_NOT_NULL(pDeleteTargetEditAnim);

    // 対象アニメーションがバインドされているモデルからアニメーションのバインドを外す
    while (pDeleteTargetEditAnim->GetBoundEditModelObjCount() > 0)
    {
        EditModelObj* pEditModelObj = pDeleteTargetEditAnim->GetBoundEditModelObj(0);
        UnbindAnimFromEditModelObj(pDeleteTargetEditAnim, pEditModelObj);
    }

    return m_pAnimResourceManager->DeleteEditAnimObj(pDeleteTargetEditAnim);
}

Iter<EditShaderArchive*> ResourceManager::DeleteEditShaderArchive(EditShaderArchive* pTarget) NN_NOEXCEPT
{
    NN_G3D_VIEWER_REQUIRES_NOT_NULL(pTarget);

    ScopedLock scopedLock(m_EditShaderArchiveArray);

    ResShaderArchive* pTargetResShaderArchive = pTarget->GetTargetResShaderArchive();
    pTarget->Detach();
    int index = m_EditShaderArchiveArray.IndexOf(pTarget);
    NN_G3D_VIEWER_ASSERT(index >= 0);

    pTarget->~EditShaderArchive();
    m_pAllocator->Free(pTarget);

    ShaderDetachedArg shaderDetachedArg;
    shaderDetachedArg.pShaderArchive = pTargetResShaderArchive;
    m_pViewerCallback->Call(CallbackType_ShaderDetached, &shaderDetachedArg);

    ViewerKeyManager::GetInstance().Unregister(pTargetResShaderArchive);

    return m_EditShaderArchiveArray.Erase(m_EditShaderArchiveArray.Begin() + index);
}

void ResourceManager::DeleteEditShaderArchive(ViewerKeyType key) NN_NOEXCEPT
{
    ResShaderArchive* pTargetResShaderArchive = ViewerKeyManager::GetInstance().FindData<ResShaderArchive>(key);
    EditShaderArchive* pDeleteEditShaderArchive = FindEditShaderArchive(pTargetResShaderArchive);
    if (pDeleteEditShaderArchive != nullptr)
    {
        DeleteEditShaderArchive(pDeleteEditShaderArchive);
    }
    else
    {
        // アタッチがキャンセルされた場合は pDeleteEditShaderArchive は nullptr
        ViewerKeyManager::GetInstance().Unregister(key);
    }
}

void ResourceManager::TryDestroyUnreferencedResources() NN_NOEXCEPT
{
    m_pDestructionScheduler->UpdateSchedule();
}

int ResourceManager::FindResFileInfoIndex(ViewerKeyType resFileKey) NN_NOEXCEPT
{
    for (int index = 0, end = m_ResFileInfoArray.GetCount(); index < end; ++index)
    {
        ResourceInfo* pInfo = &m_ResFileInfoArray[index];
        if (resFileKey == pInfo->resFileKey)
        {
            return index;
        }
    }

    return -1;
}

void ResourceManager::UnbindAnimFromEditModelObj(EditAnimObj* pEditAnimObj, EditModelObj* pTargetEditModelObj) NN_NOEXCEPT
{
    NN_G3D_VIEWER_REQUIRES_NOT_NULL(pEditAnimObj);
    NN_G3D_VIEWER_REQUIRES_NOT_NULL(pTargetEditModelObj);

    pEditAnimObj->UnbindModelObj(pTargetEditModelObj);

    bool removed = pTargetEditModelObj->RemoveBoundAnim(pEditAnimObj);
    NN_UNUSED(removed);

    // モデルからバインド情報が消えているのに、アニメーションにだけ残っていた場合、内部実装の不具合なので警告しておく
    NN_G3D_WARNING(removed, "inconsistency of animation binding state was detected");

    if ((pEditAnimObj->GetAnimKind() == ViewerAnimKind_MaterialAnim)
        || (pEditAnimObj->GetAnimKind() == ViewerAnimKind_TexturePatternAnim))
    {
        // テクスチャパターンアニメーションがアンバインドされた後はモデルに割り当てられたテクスチャを再バインドする
        ForceBindExistingTextures(pTargetEditModelObj);
    }
}

void ResourceManager::UnbindAnimation(EditModelObj* pTargetEditModelObj, ViewerKeyType animResFileKey) NN_NOEXCEPT
{
    EditAnimObj* pBoundEditAnimObj = FindBoundEditAnimObj(pTargetEditModelObj->GetKey(), animResFileKey);
    NN_G3D_VIEWER_ASSERT_NOT_NULL_DETAIL(pBoundEditAnimObj, "anim key %d not found", animResFileKey);
    UnbindAnimFromEditModelObj(pBoundEditAnimObj, pTargetEditModelObj);
}

void ResourceManager::UnbindAnimations(const BindAnimArg& arg) NN_NOEXCEPT
{
    EditModelObj* pEditModelObj = FindEditModelObj(arg.modelKey);
    if (pEditModelObj == nullptr)
    {
        return;
    }

    for (int animKeyIndex = 0, count = static_cast<int>(arg.animationKeySize); animKeyIndex < count; ++animKeyIndex)
    {
        ViewerKeyType animResFileKey = arg.animationKeys[animKeyIndex];
        NN_G3D_VIEWER_DEBUG_PRINT("Unbinding anim key %d from model %s\n", animResFileKey, pEditModelObj->GetName());
        UnbindAnimation(pEditModelObj, animResFileKey);
    }
}

void ResourceManager::ForceUnbindAnimations(EditModelObj* pEditModelObj) NN_NOEXCEPT
{
    NN_G3D_VIEWER_ASSERT_NOT_NULL(pEditModelObj);

    while (0 != pEditModelObj->GetBoundAnimCount())
    {
        UnbindAnimFromEditModelObj(pEditModelObj->GetBoundAnimAt(0), pEditModelObj);
    }
}

EditModelObj* ResourceManager::FindEditModelObj(ViewerKeyType key) NN_NOEXCEPT
{
    return m_EditModelArray.Find<ViewerKeyType>(&key, CompareSameKey);
}

EditModelObj* ResourceManager::FindEditModelObj(const nn::g3d::ModelObj* pTargetModelObj) NN_NOEXCEPT
{
    return m_EditModelArray.Find<ModelObj>(pTargetModelObj, CompareSameModelObj);
}

EditShaderArchive* ResourceManager::FindEditShaderArchive(const nn::g3d::ResShaderArchive* pResShaderArchive) NN_NOEXCEPT
{
    return m_EditShaderArchiveArray.Find<ResShaderArchive>(pResShaderArchive, CompareSameResShaderArchive);
}

EditAnimObj* ResourceManager::FindEditAnimObj(ViewerKeyType resFileKey) NN_NOEXCEPT
{
    return m_pAnimResourceManager->FindEditAnimObj(resFileKey);
}

const EditAnimObj* ResourceManager::FindEditAnimObj(ViewerKeyType resFileKey) const NN_NOEXCEPT
{
    return m_pAnimResourceManager->FindEditAnimObj(resFileKey);
}

const EditShaderArchive* ResourceManager::FindEditShaderArchive(ViewerKeyType key) const NN_NOEXCEPT
{
    const ResShaderArchive* pResShaderArchive = ViewerKeyManager::GetInstance().FindData<nn::g3d::ResShaderArchive>(key);
    if (pResShaderArchive == nullptr)
    {
        return nullptr;
    }

    return m_EditShaderArchiveArray.Find<ResShaderArchive>(pResShaderArchive, CompareSameResShaderArchive);
}

EditShaderArchive* ResourceManager::FindEditShaderArchive(ViewerKeyType key) NN_NOEXCEPT
{
    const ResShaderArchive* pResShaderArchive = ViewerKeyManager::GetInstance().FindData<nn::g3d::ResShaderArchive>(key);
    if (pResShaderArchive == nullptr)
    {
        return nullptr;
    }

    return m_EditShaderArchiveArray.Find<ResShaderArchive>(pResShaderArchive, CompareSameResShaderArchive);
}

EditSceneAnimObj* ResourceManager::FindEditSceneAnimObj(ViewerKeyType resFileKey) NN_NOEXCEPT
{
    return m_pAnimResourceManager->FindEditSceneAnimObj(resFileKey);
}

template<typename TTarget>
void ResourceManager::ForceBindExistingTexturesImpl(TTarget* pTarget) NN_NOEXCEPT
{
    for (int resFileInfoIndex = 0, count = m_ResFileInfoArray.GetCount(); resFileInfoIndex < count; ++resFileInfoIndex)
    {
        ResourceInfo resFileInfo = m_ResFileInfoArray[resFileInfoIndex];
        if (resFileInfo.kind != FILEDATA_TEXTURE)
        {
            continue;
        }

        ForceBindTextureIfBindable<TTarget>(pTarget, resFileInfo.pResTextureFile, resFileInfo.resFileKey, *m_pViewerCallback);
    }
}

void ResourceManager::ForceBindExistingTextures(EditModelObj* pEditModelObj) NN_NOEXCEPT
{
    ForceBindExistingTexturesImpl<EditModelObj>(pEditModelObj);
}

void ResourceManager::ForceBindExistingTextures(EditAnimObj* pEditAnimObj) NN_NOEXCEPT
{
    ViewerAnimKind kind = pEditAnimObj->GetAnimKind();
    switch (kind)
    {
    case ViewerAnimKind_MaterialAnim:
        ForceBindExistingTexturesImpl<EditMaterialAnimObj>(static_cast<EditMaterialAnimObj*>(pEditAnimObj));
        return;
    case ViewerAnimKind_TexturePatternAnim:
        ForceBindExistingTexturesImpl<EditTexPatternAnimObj>(static_cast<EditTexPatternAnimObj*>(pEditAnimObj));
        return;
    default:
        return;
    }
}

void ResourceManager::ForceBindExistingTexturesForAllResources() NN_NOEXCEPT
{
    for (auto iter = m_EditModelArray.Begin(), end = m_EditModelArray.End();
        iter != end; ++iter)
    {
        EditModelObj* pEditModelObj = *iter;
        ForceBindExistingTextures(pEditModelObj);
    }

    for (auto iter = m_pAnimResourceManager->GetEditAnimArray().Begin(), end = m_pAnimResourceManager->GetEditAnimArray().End();
        iter != end; ++iter)
    {
        EditAnimObj* pEditAnimObj = *iter;
        ForceBindExistingTextures(pEditAnimObj);
    }
}

void ResourceManager::ForceBindTextureForAllObjs(const nn::gfx::ResTextureFile* pResTextureFile) NN_NOEXCEPT
{
    ViewerKeyType textureKey = ViewerKeyManager::GetInstance().FindKey(pResTextureFile);
    for (int index = 0, modelCount = m_EditModelArray.GetCount(); index < modelCount; ++index)
    {
        EditModelObj* pEditModelObj = m_EditModelArray.UnsafeAt(index);
        ForceBindTextureIfBindable<EditModelObj>(pEditModelObj, pResTextureFile, textureKey, *m_pViewerCallback);
    }

    for (auto iter = m_pAnimResourceManager->GetEditAnimArray().Begin(), end = m_pAnimResourceManager->GetEditAnimArray().End();
        iter != end; ++iter)
    {
        EditAnimObj* pEditAnimObj = *iter;
        switch (pEditAnimObj->GetAnimKind())
        {
        case ViewerAnimKind_MaterialAnim:
            {
                EditMaterialAnimObj* pCastedAnimObj = static_cast<EditMaterialAnimObj*>(pEditAnimObj);
                ForceBindTextureIfBindable<EditMaterialAnimObj>(pCastedAnimObj, pResTextureFile, textureKey, *m_pViewerCallback);
            }
            break;
        case ViewerAnimKind_TexturePatternAnim:
            {
                EditTexPatternAnimObj* pCastedAnimObj = static_cast<EditTexPatternAnimObj*>(pEditAnimObj);
                ForceBindTextureIfBindable<EditTexPatternAnimObj>(pCastedAnimObj, pResTextureFile, textureKey, *m_pViewerCallback);
            }
            break;
        default:
            break;
        }
    }
}

void ResourceManager::BindExistingTextures(nn::g3d::ResFile* pResFile) NN_NOEXCEPT
{
    // 存在しないテクスチャにユーザがダミーテクスチャを割り当てられるように最初に空でバインドコールバックを呼ぶ
    nn::g3d::viewer::detail::BindTexture(pResFile, nullptr, *m_pViewerCallback);

    for (int resFileInfoIndex = 0, count = m_ResFileInfoArray.GetCount(); resFileInfoIndex < count; ++resFileInfoIndex)
    {
        ResourceInfo resFileInfo = m_ResFileInfoArray[resFileInfoIndex];
        if (resFileInfo.kind != FILEDATA_TEXTURE)
        {
            continue;
        }

        nn::g3d::viewer::detail::ForceBindTexture(pResFile, resFileInfo.pResTextureFile, *m_pViewerCallback);
    }
}

RuntimeErrorCode ResourceManager::UpdateTextureBindings(TextureBindingBlock* pBlock) NN_NOEXCEPT
{
    {
        EditModelObj* pEditModelObj = FindEditModelObj(pBlock->bindTargetKey);
        if (pEditModelObj != nullptr)
        {
            pEditModelObj->UpdateTextureBindings(pBlock);
            ForceBindExistingTextures(pEditModelObj);
            return RUNTIME_ERROR_CODE_NO_ERROR;
        }
    }

    {
        EditAnimObj* pEditAnimObj = m_pAnimResourceManager->FindEditAnimObj(pBlock->bindTargetKey);
        if (pEditAnimObj != nullptr)
        {
            switch (pEditAnimObj->GetAnimKind())
            {
            case ViewerAnimKind_MaterialAnim:
                {
                    EditMaterialAnimObj* pCastedAnimObj = static_cast<EditMaterialAnimObj*>(pEditAnimObj);
                    pCastedAnimObj->UpdateTextureBindings(pBlock);
                    ForceBindExistingTextures(pCastedAnimObj);
                }
                break;
            case ViewerAnimKind_TexturePatternAnim:
                {
                    EditTexPatternAnimObj* pCastedAnimObj = static_cast<EditTexPatternAnimObj*>(pEditAnimObj);
                    pCastedAnimObj->UpdateTextureBindings(pBlock);
                    ForceBindExistingTextures(pCastedAnimObj);
                }
                break;
            default:
                NN_UNEXPECTED_DEFAULT;
            }

            return RUNTIME_ERROR_CODE_NO_ERROR;
        }
    }

    return RUNTIME_ERROR_CODE_TARGET_MODEL_NOT_FOUND;
}


EditAnimObj* ResourceManager::FindBoundEditAnimObj(ViewerKeyType modelKey, ViewerKeyType boundAnimationResFileKey) NN_NOEXCEPT
{
    EditModelObj* pEditModelObj = FindEditModelObj(modelKey);
    if (pEditModelObj == nullptr)
    {
        return nullptr;
    }

    int boundAnimCount = pEditModelObj->GetBoundAnimCount();
    for (int i = 0; i < boundAnimCount; ++i)
    {
        EditAnimObj* pIterEditAnimObj = pEditModelObj->GetBoundAnimAt(i);
        ViewerKeyType animKey = pIterEditAnimObj->GetResFileKey();
        if (animKey == boundAnimationResFileKey)
        {
            return pIterEditAnimObj;
        }
    }

    return nullptr;
}

void ResourceManager::UnbindTexturesFromAllObj(ViewerKeyType resFileKey) NN_NOEXCEPT
{
    int textureIndex = FindResFileInfoIndex(resFileKey);
    NN_G3D_VIEWER_ASSERT_DETAIL(textureIndex != -1, NN_TEXT("3DEditor から送信されていないテクスチャのアンロードが実行されました: resFileKey = %d"), resFileKey);

    ResourceInfo resFileInfo = m_ResFileInfoArray[textureIndex];
    NN_G3D_VIEWER_ASSERT(resFileInfo.kind == FILEDATA_TEXTURE);
    for (int index = 0, modelCount = m_EditModelArray.GetCount(); index < modelCount; ++index)
    {
        EditModelObj* pEditModelObj = m_EditModelArray.UnsafeAt(index);
        pEditModelObj->UnbindTexture(resFileInfo.pResTextureFile, *m_pViewerCallback);
    }

    for (auto iter = m_pAnimResourceManager->GetEditAnimArray().Begin(), end = m_pAnimResourceManager->GetEditAnimArray().End();
        iter != end; ++iter)
    {
        EditAnimObj* pEditAnimObj = *iter;
        switch (pEditAnimObj->GetAnimKind())
        {
        case ViewerAnimKind_TexturePatternAnim:
            {
                EditTexPatternAnimObj* pTexPatternAnimObj = static_cast<EditTexPatternAnimObj*>(pEditAnimObj);
                pTexPatternAnimObj->UnbindTexture(resFileInfo.pResTextureFile, *m_pViewerCallback);
            }
            break;
        case ViewerAnimKind_MaterialAnim:
            {
                EditMaterialAnimObj* pMatAnimObj = static_cast<EditMaterialAnimObj*>(pEditAnimObj);
                pMatAnimObj->UnbindTexture(resFileInfo.pResTextureFile, *m_pViewerCallback);
            }
            break;
        default:
            break;
        }
    }
}


void ResourceManager::EditMaterials(const EditMaterialArg& arg) NN_NOEXCEPT
{
    EditModelObj* editModel = FindEditModelObj(arg.key);
    if (editModel == nullptr)
    {
        return;
    }

    if (arg.editTargetKind == EDIT_TARGET_MATERIAL_VISIBILITY)
    {
        const EditSimpleValue* editValue = static_cast<const EditSimpleValue*>(arg.value);
        if (!editValue->bValue)
        {
            TargetSelectedArg selectTargetArg;
            selectTargetArg.pModelObj = editModel->GetTargetModelObj(0);
            selectTargetArg.targetKind = TargetSelectedArg::TargetKind_Material;
            selectTargetArg.index = arg.index;
            selectTargetArg.indexCount = arg.indexCount;

            editModel->BreakMaterialBlinking(selectTargetArg);
        }
    }

    bool success = nn::g3d::viewer::detail::EditMaterials(m_pDevice, editModel, arg);
    for (int modelIndex = 0, modelCount = editModel->GetAttachedModelObjCount(); modelIndex < modelCount; ++modelIndex)
    {
        ModelObj* pTargetModelObj = editModel->GetTargetModelObj(modelIndex);
        if (success)
        {
            switch (arg.editTargetKind)
            {
            case EDIT_TARGET_MATERIAL_SAMPLER_NAME:
            case EDIT_TARGET_MATERIAL_SAMPLER_TEXTURE_REF:
            case EDIT_TARGET_MATERIAL_SAMPLER_WRAP_U:
            case EDIT_TARGET_MATERIAL_SAMPLER_WRAP_V:
            case EDIT_TARGET_MATERIAL_SAMPLER_WRAP_W:
            case EDIT_TARGET_MATERIAL_SAMPLER_MAG_FILTER:
            case EDIT_TARGET_MATERIAL_SAMPLER_MIN_FILTER:
            case EDIT_TARGET_MATERIAL_SAMPLER_MIP_FILTER:
            case EDIT_TARGET_MATERIAL_SAMPLER_MAX_ANISO:
            case EDIT_TARGET_MATERIAL_SAMPLER_MIN_LOD:
            case EDIT_TARGET_MATERIAL_SAMPLER_MAX_LOD:
            case EDIT_TARGET_MATERIAL_SAMPLER_LOD_BIAS:
                {
                    SamplerParamUpdatedArg samplerParamUpdatedArg;
                    samplerParamUpdatedArg.pModelObj = pTargetModelObj;
                    m_pViewerCallback->Call(CallbackType_SamplerParamUpdated, &samplerParamUpdatedArg);
                }
                break;

            default:
                break;
            }
        }
    }
}

void ResourceManager::EditBones(const EditBoneArg& arg) NN_NOEXCEPT
{
    EditModelObj* editModel = FindEditModelObj(arg.modelKey);
    if (editModel == nullptr)
    {
        return;
    }

    for (int modelIndex = 0, modelCount = editModel->GetAttachedModelObjCount(); modelIndex < modelCount; ++modelIndex)
    {
        nn::g3d::viewer::detail::EditBones(editModel->GetTargetModelObj(modelIndex), arg);
    }
}

void ResourceManager::UpdateBlink() NN_NOEXCEPT
{
    int size = m_EditModelArray.GetCount();
    for (int i = 0; i < size; ++i)
    {
        EditModelObj* pEditModel = m_EditModelArray.UnsafeAt(i);
        pEditModel->UpdateMaterialBlinking();
        pEditModel->UpdateShapeBlinking();
    }
}

void ResourceManager::StartBlinking(const nn::g3d::viewer::TargetSelectedArg& arg) NN_NOEXCEPT
{
    EditModelObj* editModel = FindSameModelObj(arg.pModelObj);
    if (editModel == nullptr)
    {
        return;
    }

    switch (arg.targetKind)
    {
    case nn::g3d::viewer::TargetSelectedArg::TargetKind_Model:
        {
            editModel->StartModelBlinking();
        }

        return;
    case nn::g3d::viewer::TargetSelectedArg::TargetKind_Shape:
        {
            editModel->StartShapeBlinking(arg);
        }

        return;
    case nn::g3d::viewer::TargetSelectedArg::TargetKind_Material:
        {
            editModel->StartMaterialBlinking(arg);
        }

        return;
    default:
        return;
    }
}

EditAnimObj* ResourceManager::AddEditAnimObj(nn::g3d::ResFile* pResFile, FileDataKind kind) NN_NOEXCEPT
{
    return m_pAnimResourceManager->AddEditAnimObj(pResFile, kind);
}

void ResourceManager::EditAnimCurve(const EditAnimCurveArg& arg) NN_NOEXCEPT
{
    return m_pAnimResourceManager->EditAnimCurve(arg);
}

void ResourceManager::PrintAllEditModelObjs() NN_NOEXCEPT
{
    for (int modelIndex = 0, modelCount = m_EditModelArray.GetCount(); modelIndex < modelCount; ++modelIndex)
    {
        EditModelObj* pEditModelObj = m_EditModelArray.UnsafeAt(modelIndex);
        for (int objIndex = 0, objCount = pEditModelObj->GetAttachedModelObjCount(); objIndex < objCount; ++objIndex)
        {
            NN_G3D_VIEWER_DEBUG_PRINT("[%d][%d] : key = %d\n", modelIndex, objIndex, pEditModelObj->GetModelObjKey(objIndex));
        }
    }
}

void ResourceManager::PrintAllEditAnimObjs() NN_NOEXCEPT
{
    m_pAnimResourceManager->PrintAllEditAnimObjs();
}

void ResourceManager::PrintEditAnimObjs(ViewerKeyType resFileKey, ViewerKeyType boundModelKey) NN_NOEXCEPT
{
    m_pAnimResourceManager->PrintEditAnimObjs(resFileKey, boundModelKey);
}

void ResourceManager::PrintAllResFiles() NN_NOEXCEPT
{
    for (int resFileInfoIndex = 0, end = m_ResFileInfoArray.GetCount(); resFileInfoIndex < end; ++resFileInfoIndex)
    {
        ResourceInfo info = m_ResFileInfoArray[resFileInfoIndex];
        switch (info.kind)
        {
        case FILEDATA_MODEL:
        case FILEDATA_SCENE_ANIM:
            NN_G3D_VIEWER_DEBUG_PRINT("ResFileInfoArray[%d] : resFileKey = %d, name = %s (%s)\n",
                resFileInfoIndex, info.resFileKey, GetResName(info.pResFile, info.kind), GetFileDataKindString(info.kind));
            break;
        case FILEDATA_SHADER_ARCHIVE:
            NN_G3D_VIEWER_DEBUG_PRINT("ResFileInfoArray[%d] : resFileKey = %d, name = %s (%s)\n",
                resFileInfoIndex, info.resFileKey, info.pResShaderArchive->GetName(), GetFileDataKindString(info.kind));
            break;
        case FILEDATA_TEXTURE:
            NN_G3D_VIEWER_DEBUG_PRINT("ResFileInfoArray[%d] : resFileKey = %d, name = %s (%s)\n",
                resFileInfoIndex, info.resFileKey, info.pResTextureFile->GetResTexture(0)->GetName(), GetFileDataKindString(info.kind));
            break;
        case FILEDATA_EDIT_MODEL_OBJ:
            NN_G3D_VIEWER_DEBUG_PRINT("ResFileInfoArray[%d] : address = %s (%s)\n",
                resFileInfoIndex, info.pEditModelObj, GetFileDataKindString(info.kind));
            break;
        default:
            NN_UNEXPECTED_DEFAULT;
        }
    }

    m_pAnimResourceManager->PrintAllResFiles();
}

EditSceneAnimObj*
ResourceManager::AddEditSceneAnimObj(nn::g3d::ResFile* pTargetResFile) NN_NOEXCEPT
{
    return m_pAnimResourceManager->AddEditSceneAnimObj(pTargetResFile);
}

Iter<EditSceneAnimObj*> ResourceManager::DeleteEditSceneAnimObj(EditSceneAnimObj* pDeleteTargetEditAnim) NN_NOEXCEPT
{
    return m_pAnimResourceManager->DeleteEditSceneAnimObj(pDeleteTargetEditAnim);
}

void ResourceManager::DeleteEditSceneAnimObj(ViewerKeyType key) NN_NOEXCEPT
{
    m_pAnimResourceManager->DeleteEditSceneAnimObj(key);
}

void ResourceManager::EditSceneAnimCurve(const EditAnimCurveArg& arg) NN_NOEXCEPT
{
    m_pAnimResourceManager->EditSceneAnimCurve(arg);
}

EditModelObj*
ResourceManager::FindSameModelObj(const nn::g3d::ModelObj* modelObj) const NN_NOEXCEPT
{
    NN_G3D_VIEWER_ASSERT_NOT_NULL(modelObj);
    return m_EditModelArray.Find<nn::g3d::ModelObj>(modelObj, CompareSameModelObj);
}

EditShaderArchive*
ResourceManager::FindSameResShaderArchive(const nn::g3d::ResShaderArchive* resShaderArchive) const NN_NOEXCEPT
{
    NN_G3D_VIEWER_ASSERT_NOT_NULL(resShaderArchive);
    return m_EditShaderArchiveArray.Find<ResShaderArchive>(resShaderArchive, CompareSameResShaderArchive);
}

void ResourceManager::ClearCalculateFlagModelAnims() NN_NOEXCEPT
{
    m_pAnimResourceManager->ClearCalculateFlagModelAnims();
}

bool ResourceManager::IsBound(EditModelObj* pEditModelObj, ViewerKeyType animResFileKey) const NN_NOEXCEPT
{
    return m_pAnimResourceManager->IsBound(pEditModelObj, animResFileKey);
}

void ResourceManager::BindSceneAnimations(const BindAnimArg& arg) NN_NOEXCEPT
{
    m_pAnimResourceManager->BindSceneAnimations(arg);
}

void ResourceManager::UnbindSceneAnimations(const BindAnimArg& arg) NN_NOEXCEPT
{
    m_pAnimResourceManager->UnbindSceneAnimations(arg);
}

void ResourceManager::CalculateAnimations() NN_NOEXCEPT
{
    if ((m_pAnimResourceManager->GetEditAnimCount() + m_pAnimResourceManager->GetEditSceneAnimCount()) == 0)
    {
        return;
    }

    // モデルアニメーションの計算
    {
        int size = m_EditModelArray.GetCount();
        for (int i = 0; i < size; ++i)
        {
            EditModelObj* pEditModelObj = m_EditModelArray.UnsafeAt(i);
            int animCount = pEditModelObj->GetBoundAnimCount();
            for (int j = 0; j < animCount; ++j)
            {
                EditAnimObj* editAnim = pEditModelObj->GetBoundAnimAt(j);

                // CalculateSkeletalAnimations で計算済みの場合は処理をスキップ
                // バインド対象が設定されていないので処理をスキップ
                if (editAnim->IsCalculated(pEditModelObj) || !editAnim->HasBoundModel())
                {
                    continue;
                }

                if (editAnim->IsPaused(pEditModelObj))
                {
                    editAnim->InvalidateContext(pEditModelObj);
                    editAnim->SetFrame(pEditModelObj, editAnim->GetPauseFrame(pEditModelObj));
                }
                else if (!m_pAnimResourceManager->IsAnimationPlaying())
                {
                    editAnim->InvalidateContext(pEditModelObj);
                    editAnim->SetFrame(pEditModelObj, m_pAnimResourceManager->GetFrame());
                }

                editAnim->Calculate(pEditModelObj);
                editAnim->ApplyAnimTo(pEditModelObj);
            }
        }
    }

    m_pAnimResourceManager->CalculateSceneAnimation();
    m_pAnimResourceManager->UpdateFrame();
}

void ResourceManager::CalculateSkeletalAnimations(const nn::g3d::ModelObj* pModelObj) NN_NOEXCEPT
{
    if (m_pAnimResourceManager->GetEditAnimCount() == 0)
    {
        return;
    }

    NN_G3D_VIEWER_ASSERT_NOT_NULL(pModelObj);
    EditModelObj* pEditModelObj = FindSameModelObj(pModelObj);
    if (pEditModelObj == nullptr)
    {
        return;
    }

    // モデルアニメーションの計算
    {
        int size = pEditModelObj->GetBoundAnimCount();
        for (int i = 0; i < size; ++i)
        {
            EditAnimObj* pEditAnim = pEditModelObj->GetBoundAnimAt(i);
            bool isSkeletalAnim = pEditAnim->GetAnimKind() == ViewerAnimKind_SkeletalAnim;

            // スケルタルアニメーションでない場合は処理をスキップ
            // バインド対象が設定されていないので処理をスキップ
            if (!isSkeletalAnim || !pEditAnim->HasBoundModel())
            {
                continue;
            }

            if (pEditAnim->IsPaused(pEditModelObj))
            {
                pEditAnim->InvalidateContext(pEditModelObj);
                pEditAnim->SetFrame(pEditModelObj, pEditAnim->GetPauseFrame(pEditModelObj));
            }
            else if (!m_pAnimResourceManager->IsAnimationPlaying())
            {
                pEditAnim->InvalidateContext(pEditModelObj);
                pEditAnim->SetFrame(pEditModelObj, m_pAnimResourceManager->GetFrame());
            }

            pEditAnim->Calculate(pEditModelObj);
            pEditAnim->ApplyAnimTo(pEditModelObj);
        }
    }
}

void ResourceManager::Reset() NN_NOEXCEPT
{
    m_pAnimResourceManager->Reset();
}


void ResourceManager::BindAnimations(const BindAnimArg& arg) NN_NOEXCEPT
{
    EditModelObj* pEditModelObj = FindEditModelObj(arg.modelKey);
    NN_G3D_VIEWER_ASSERT_NOT_NULL(pEditModelObj);

    m_pAnimResourceManager->BindAnimations(pEditModelObj, arg);
}

void ResourceManager::UpdateEditShaderArchive(ViewerKeyType key, int shadingModelIndices[], int indexCount) NN_NOEXCEPT
{
    EditShaderArchive* editShaderArchive = FindEditShaderArchive(key);
    if (editShaderArchive != nullptr)
    {
        editShaderArchive->UpdateShadingModels(shadingModelIndices, indexCount);
    }
}

void ResourceManager::SendModifiedShaderPrograms(EditSocketBase* pSocket) NN_NOEXCEPT
{
    ScopedLock scopedLock(m_EditShaderArchiveArray);

    NN_G3D_VIEWER_ASSERT_NOT_NULL(pSocket);
    int shaderArchiveCount = m_EditShaderArchiveArray.GetCount();
    for (int i = 0; i < shaderArchiveCount; ++i)
    {
        EditShaderArchive* editShaderArchive = m_EditShaderArchiveArray.UnsafeAt(i);
        editShaderArchive->SendModifiedShaderPrograms(pSocket);
    }
}

void ResourceManager::DeleteEditAnimObjBindingForAllModels(EditAnimObj* pTarget) NN_NOEXCEPT
{
    for (int modelIndex = 0, modelEnd = m_EditModelArray.GetCount(); modelIndex < modelEnd; ++modelIndex)
    {
        EditModelObj* pEditModelObj = m_EditModelArray.At(modelIndex);
        UnbindAnimFromEditModelObj(pTarget, pEditModelObj);
    }
}


RuntimeErrorCode ResourceManager::ReloadAnimResource(ViewerKeyType resFileKey, nn::g3d::ResFile* pResFile, FileDataKind kind) NN_NOEXCEPT
{
    NN_G3D_VIEWER_ASSERT_NOT_NULL(pResFile);
    ResourceInfo* pResFileInfo = m_pAnimResourceManager->FindAnimResFileInfo(resFileKey);
    NN_G3D_VIEWER_ASSERT_NOT_NULL_DETAIL(pResFileInfo,
        NN_TEXT("ロードされていないアニメーションをリロードするコマンドが 3DEditor から送信されました: resFileKey = %d"),
        resFileKey);
    NN_G3D_VIEWER_ASSERT_NOT_NULL(pResFileInfo->pResFile);

    nn::g3d::ResFile* pOldResAnimFile = pResFileInfo->pResFile;

    // アニメーションをバインド済みのモデルを新しいリソースでバインドし直す
    EditAnimObj* pEditAnimObj = FindEditAnimObj(resFileKey);
    NN_G3D_VIEWER_ASSERT_NOT_NULL(pEditAnimObj);
    {
        bool success = pEditAnimObj->ReloadResource(pResFile);
        if (!success)
        {
            // リロードに失敗した場合はEditAnimObjのモデルバインド情報は消えるので、
            // すべてのモデルから該当のEditAnimObjのバインド情報を消す
            DeleteEditAnimObjBindingForAllModels(pEditAnimObj);
            return RUNTIME_ERROR_CODE_UNKNOWN_ERROR;
        }
    }

    // リソース情報を新しいデータで上書き
    pResFileInfo->pResFile = pResFile;
    ViewerKeyManager::GetInstance().UpdateData(resFileKey, pResFile);

    // 古いリソースは遅延破棄
    {
        ViewerKeyType oldResFileKey = ViewerKeyManager::GetInstance().Register(pOldResAnimFile, ViewerKeyManager::DataType_AnimResFile);
        m_pDestructionScheduler->Register(ResourceInfo(pOldResAnimFile, pResFileInfo->kind, oldResFileKey));
    }

    // アニメーションが更新されたのでグローバルプレイポリシーの再設定を行う
    m_pAnimResourceManager->ResetPlayPolicy();

    // リロード後に未使用のリソースがあったら解放する
    ScheduleDestroyUnusedResFiles();

    if (kind == FILEDATA_TEXTURE_PATTERN_ANIM ||
        kind == FILEDATA_MATERIAL_ANIM)
    {
        // バインドしているモデルが古いテクスチャパターンアニメーションのテクスチャを参照している場合があるので
        // バインドしている各モデルのサンプラのテクスチャ参照を元に戻す
        for (int boundModelIndex = 0, boundModelCount = pEditAnimObj->GetBoundEditModelObjCount(); boundModelIndex < boundModelCount; ++boundModelIndex)
        {
            EditModelObj* pEditModelObj = pEditAnimObj->GetBoundEditModelObj(boundModelIndex);
            RuntimeErrorCode errorCode = pEditModelObj->ResetTextureRefToOriginal();
            if (errorCode != RUNTIME_ERROR_CODE_NO_ERROR)
            {
                return errorCode;
            }
        }
    }

    return RUNTIME_ERROR_CODE_NO_ERROR;
}

RuntimeErrorCode ResourceManager::ReloadTextureResource(ViewerKeyType resFileKey, nn::gfx::ResTextureFile* pResTextureFile) NN_NOEXCEPT
{
    NN_G3D_VIEWER_ASSERT_NOT_NULL(pResTextureFile);
    int resFileInfoIndex = FindResFileInfoIndex(resFileKey);
    NN_G3D_VIEWER_ASSERT_DETAIL(resFileInfoIndex != -1,
        NN_TEXT("ロードされていないテクスチャをリロードするコマンドが 3DEditor から送信されました: resFileKey = %d"),
        resFileKey);

    ResourceInfo resFileInfo = m_ResFileInfoArray[resFileInfoIndex];
    NN_G3D_VIEWER_ASSERT(resFileInfo.kind == FILEDATA_TEXTURE);
    NN_G3D_VIEWER_ASSERT_NOT_NULL(resFileInfo.pResTextureFile);

    // バインド済みのテクスチャを新しいテクスチャでリバインドする
    nn::gfx::ResTextureFile* pOldResTextureFile = resFileInfo.pResTextureFile;
    {
        for (int index = 0, modelCount = m_EditModelArray.GetCount(); index < modelCount; ++index)
        {
            EditModelObj* pEditModelObj = m_EditModelArray.UnsafeAt(index);
            pEditModelObj->ReloadTexture(pOldResTextureFile, pResTextureFile, *m_pViewerCallback);
        }

        m_pAnimResourceManager->ReloadTextureResource(pOldResTextureFile, pResTextureFile);
    }

    // リソース情報を新しいデータで上書き
    m_ResFileInfoArray[resFileInfoIndex].pResTextureFile = pResTextureFile;
    ViewerKeyManager::GetInstance().UpdateData(resFileKey, pResTextureFile);

    // 古いリソースは遅延破棄をするために再度別のキーで登録
    {
        ViewerKeyType oldResFileKey = ViewerKeyManager::GetInstance().Register(pOldResTextureFile, ViewerKeyManager::DataType_ResTextureFile);
        m_pDestructionScheduler->Register(ResourceInfo(pOldResTextureFile, oldResFileKey));
    }

    return RUNTIME_ERROR_CODE_NO_ERROR;
}

void ResourceManager::ScheduleDestroyAnimResFile(ViewerKeyType resFileKey) NN_NOEXCEPT
{
    m_pAnimResourceManager->ScheduleDestroyAnimResFile(resFileKey);
}

bool ResourceManager::IsShaderArchiveUsed(const nn::g3d::ResShaderArchive* pShaderArchive) const NN_NOEXCEPT
{
    for (int i = 0, modelCount = m_EditModelArray.GetCount(); i < modelCount; ++i)
    {
        EditModelObj* pEditModel = m_EditModelArray.UnsafeAt(i);
        if (pEditModel->IsResShaderArchiveUsed(pShaderArchive))
        {
            return true;
        }
    }

    for (int shaderArchiveIndex = 0, shaderCount = m_EditShaderArchiveArray.GetCount();
        shaderArchiveIndex < shaderCount; ++shaderArchiveIndex)
    {
        EditShaderArchive* pEditShaderArchive = m_EditShaderArchiveArray.UnsafeAt(shaderArchiveIndex);
        if (pEditShaderArchive->GetTargetResShaderArchive() == pShaderArchive)
        {
            return true;
        }

        for (int shadingModelIndex = 0, shadingModelCount = pEditShaderArchive->GetEditShadingModelCount();
            shadingModelIndex < shadingModelCount; ++shadingModelIndex)
        {
            EditShadingModel* pShadingModel = pEditShaderArchive->GetEditShadingModel(shadingModelIndex);
            for (int shaderProgramIndex = 0, shaderProgramCount = pShadingModel->GetEditShaderProgramCount();
                shaderProgramIndex < shaderProgramCount; ++shaderProgramIndex)
            {
                EditShaderProgram* pShaderProgram = pShadingModel->GetEditShaderProgram(shaderProgramIndex);
                if (pShaderProgram->GetUpdateResShaderArchive() == pShaderArchive)
                {
                    return true;
                }
            }
        }
    }

    return false;
}

bool ResourceManager::ExaminesTextureFileUsed(const nn::gfx::ResTextureFile* pResTextureFile) const NN_NOEXCEPT
{
    for (int index = 0, modelCount = m_EditModelArray.GetCount(); index < modelCount; ++index)
    {
        EditModelObj* pEditModelObj = m_EditModelArray.UnsafeAt(index);
        if (pEditModelObj->ExaminesTextureFileUsed(pResTextureFile))
        {
            return true;
        }
    }

    return m_pAnimResourceManager->ExaminesTextureFileUsed(pResTextureFile);
}

bool ResourceManager::IsResFileUsedForModel(nn::g3d::ResFile* pResFile) NN_NOEXCEPT
{
    EditModelObj* pUsedEditModelObj = FindDeviceDependentObj(m_EditModelArray, pResFile);
    if (pUsedEditModelObj != nullptr)
    {
        return true;
    }

    for (auto iter = m_ResFileInfoArray.Begin(), end = m_ResFileInfoArray.End(); iter != end; ++iter)
    {
        ResourceInfo resFileInfo = *iter;
        if (resFileInfo.kind == FILEDATA_EDIT_MODEL_OBJ)
        {
            bool isBound = resFileInfo.pEditModelObj->IsResFileBound(pResFile);
            if (isBound)
            {
                return true;
            }
        }
    }

    return false;
}

void ResourceManager::ScheduleDestroyUnusedResFiles() NN_NOEXCEPT
{
    for (Iter<ResourceInfo> iter = m_ResFileInfoArray.Begin(),
        end = m_ResFileInfoArray.End(); iter != end; ++iter)
    {
        ResourceInfo* pInfo = &*iter;
        switch (pInfo->kind)
        {
        case FILEDATA_MODEL:
        case FILEDATA_SCENE_ANIM:
            {
                ResFile* pResFile = pInfo->pResFile;
                if (pResFile)
                {
                    EditAnimObj* pUsedEditAnimObj = FindDeviceDependentObj(m_pAnimResourceManager->GetEditAnimArray(), pResFile);
                    EditSceneAnimObj* pUsedEditSceneAnimObj = FindDeviceDependentObj(m_pAnimResourceManager->GetEditSceneAnimArray(), pResFile);
                    bool isResFileUsedForModel = IsResFileUsedForModel(pResFile);
                    if (pUsedEditAnimObj == nullptr &&
                        pUsedEditSceneAnimObj == nullptr &&
                        !isResFileUsedForModel)
                    {
                        m_pDestructionScheduler->Register(*pInfo);
                        iter = m_ResFileInfoArray.Erase(iter);
                    }
                }
            }
            break;
        case FILEDATA_SHADER_ARCHIVE:
            {
                if (pInfo->pResShaderFile)
                {
                    // 未使用シェーダの削除
                    if (!IsShaderArchiveUsed(pInfo->pResShaderArchive))
                    {
                        m_pDestructionScheduler->Register(*pInfo);
                        iter = m_ResFileInfoArray.Erase(iter);
                    }
                }
            }
            break;
        case FILEDATA_EDIT_MODEL_OBJ:
        case FILEDATA_TEXTURE:
        case FILEDATA_SHADER_PARAM_ANIM:
        case FILEDATA_MAT_COLOR_ANIM:
        case FILEDATA_TEXTURE_SRT_ANIM:
        case FILEDATA_TEXTURE_PATTERN_ANIM:
        case FILEDATA_SKELETAL_ANIM:
        case FILEDATA_BONE_VISIBILITY_ANIM:
        case FILEDATA_MAT_VISIBILITY_ANIM:
        case FILEDATA_SHAPE_ANIM:
        case FILEDATA_MATERIAL_ANIM:
            {
                // 3DEditor からアンロードコマンドが送られてきたときに削除
            }
            break;
        default:
            NN_UNEXPECTED_DEFAULT;
        }
    }
}

RuntimeErrorCode ResourceManager::SetRetargetingHostModel(ViewerKeyType animResFileKey, ViewerKeyType retargetHostModelObjKey) NN_NOEXCEPT
{
    EditModelObj* pRetargetHostEditModelObj = FindEditModelObj(retargetHostModelObjKey);
    if (pRetargetHostEditModelObj == nullptr)
    {
        return RUNTIME_ERROR_CODE_RETARGET_HOST_MODEL_NOT_FOUND;
    }

    m_pAnimResourceManager->SetRetargetingHostModel(pRetargetHostEditModelObj, animResFileKey);

    return RUNTIME_ERROR_CODE_NO_ERROR;
}

void ResourceManager::UnsetRetargetingHostModel(ViewerKeyType animResFileKey) NN_NOEXCEPT
{
    m_pAnimResourceManager->UnsetRetargetingHostModel(animResFileKey);
}

void ResourceManager::SetPlayMotionMirroringEnabled(ViewerKeyType animResFileKey, bool isEnabled) NN_NOEXCEPT
{
    m_pAnimResourceManager->SetPlayMotionMirroringEnabled(animResFileKey, isEnabled);
}

EditModelObj*  ResourceManager::FindUnattachedEditModelObj(ViewerKeyType resModelKey) NN_NOEXCEPT
{
    for (int i = 0, end = m_EditModelArray.GetCount(); i < end; ++i)
    {
        EditModelObj* pModelObj = m_EditModelArray.UnsafeAt(i);
        bool isAttached = pModelObj->GetAttachedModelObjCount() > 0;
        if (pModelObj->GetResModelKey() == resModelKey && !isAttached)
        {
            return pModelObj;
        }
    }

    return nullptr;
}

int ResourceManager::FindModelAnimBoundCount(const nn::g3d::ModelObj* modelObj) const NN_NOEXCEPT
{
    EditModelObj* editModelObj = m_EditModelArray.Find<nn::g3d::ModelObj>(modelObj, CompareSameModelObj);
    return editModelObj->GetBoundAnimCount();
}


nn::g3d::ResFile* ResourceManager::LoadResFileWithoutCopyAndRegister(void** pFileBuffer, FileDataKind kind) NN_NOEXCEPT
{
    ResFile* pResFile = SetupBufferAsResFile(m_pDevice, *pFileBuffer);
    *pFileBuffer = nullptr;
    ResourceInfo info(pResFile, kind);
    m_ResFileInfoArray.PushBack(info);

    switch (kind)
    {
    case FILEDATA_MODEL:
        ViewerKeyManager::GetInstance().Register(pResFile, ViewerKeyManager::DataType_ModelResFile);
        break;
    case FILEDATA_SHADER_PARAM_ANIM:
    case FILEDATA_MAT_COLOR_ANIM:
    case FILEDATA_TEXTURE_SRT_ANIM:
    case FILEDATA_TEXTURE_PATTERN_ANIM:
    case FILEDATA_SKELETAL_ANIM:
    case FILEDATA_BONE_VISIBILITY_ANIM:
    case FILEDATA_MAT_VISIBILITY_ANIM:
    case FILEDATA_SHAPE_ANIM:
    case FILEDATA_MATERIAL_ANIM:
        ViewerKeyManager::GetInstance().Register(pResFile, ViewerKeyManager::DataType_AnimResFile);
        break;
    case FILEDATA_SCENE_ANIM:
        ViewerKeyManager::GetInstance().Register(pResFile, ViewerKeyManager::DataType_SceneAnimResFile);
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }

    return pResFile;
}

nn::g3d::ResShaderArchive* ResourceManager::LoadResShaderArchiveWithoutCopyAndRegister(void** pFileBuffer) NN_NOEXCEPT
{
    nn::g3d::ResShaderFile* pResShaderFile = static_cast<nn::g3d::ResShaderFile*>(*pFileBuffer);
    NN_G3D_VIEWER_ASSERT(nn::g3d::ResShaderFile::IsValid(pResShaderFile));
    ResShaderArchive* pResShaderArchive = nn::g3d::ResShaderFile::ResCast(pResShaderFile)->GetResShaderArchive();
    pResShaderArchive->Setup(m_pDevice);

    ResourceInfo info(pResShaderFile, pResShaderArchive);
    m_ResFileInfoArray.PushBack(info);

    ViewerKeyManager::GetInstance().Register(pResShaderFile, ViewerKeyManager::DataType_ResShaderFile);

    *pFileBuffer = nullptr;

    return pResShaderArchive;
}

nn::gfx::ResTextureFile* ResourceManager::LoadResTextureFileWithoutCopyAndRegister(void** pFileBuffer) NN_NOEXCEPT
{
    nn::gfx::ResTextureFile* pResTextureFile = LoadResTextureFileWithoutCopy(pFileBuffer);

    ViewerKeyType resFileKey = ViewerKeyManager::GetInstance().Register(pResTextureFile, ViewerKeyManager::DataType_ResTextureFile);

    ResourceInfo info(pResTextureFile, resFileKey);
    m_ResFileInfoArray.PushBack(info);

    return pResTextureFile;
}

nn::gfx::ResTextureFile* ResourceManager::LoadResTextureFileWithoutCopy(void** pFileBuffer) NN_NOEXCEPT
{
    nn::gfx::ResTextureFile* pResTextureFile = SetupBufferAsResTextureFile(*pFileBuffer, *m_pViewerCallback);
    *pFileBuffer = nullptr;
    return pResTextureFile;
}

nn::g3d::ResFile* ResourceManager::LoadAnimResFileWithoutCopyAndRegister(void** pFileBuffer, FileDataKind kind) NN_NOEXCEPT
{
    ResFile* pResFile = LoadAnimResFileWithoutCopy(pFileBuffer);
    ViewerKeyType resFileKey = ViewerKeyManager::GetInstance().Register(pResFile, ViewerKeyManager::DataType_AnimResFile);
    ResourceInfo info(pResFile, kind, resFileKey);
    m_pAnimResourceManager->GetAnimResFileInfoArray().PushBack(info);

    return pResFile;
}

nn::g3d::ResFile* ResourceManager::LoadAnimResFileWithoutCopy(void** pFileBuffer) NN_NOEXCEPT
{
    ResFile* pResFile = SetupBufferAsResFile(m_pDevice, *pFileBuffer);
    *pFileBuffer = nullptr;
    return pResFile;
}

nn::g3d::ResFile* ResourceManager::SetupBufferAsResFile(nn::gfx::Device* pDevice, void* buffer) NN_NOEXCEPT
{
    NN_G3D_VIEWER_ASSERT(nn::g3d::ResFile::IsValid(buffer));

    nn::g3d::ResFileData* pResFileData = reinterpret_cast<nn::g3d::ResFileData*>(buffer);
    NN_UNUSED(pResFileData);
    NN_G3D_VIEWER_DEBUG_PRINT(
        "Setup ResFile: name = %s, size = %zu, alignment = %zu\n",
        pResFileData->fileHeader.GetFileName() != nullptr ? pResFileData->fileHeader.GetFileName().data() : "nullptr",
        pResFileData->fileHeader.GetFileSize(),
        pResFileData->fileHeader.GetAlignment());

    // リソースファイル内のポインタ情報をセットアップする
    nn::g3d::ResFile* pResFile = nn::g3d::ResFile::ResCast(buffer);

    NN_G3D_VIEWER_ASSERT_NOT_NULL(pResFile);

    // edit 側で管理する ResFile なので、ここでSetupを行う
    pResFile->Setup(pDevice);

    BindExistingTextures(pResFile);

    return pResFile;
}

int ResourceManager::FindActiveEditAnimIndex(const nn::g3d::ModelObj* pModelObj, int boundAnimIndex) const NN_NOEXCEPT
{
    EditModelObj* pEditModelObj = FindSameModelObj(pModelObj);
    return m_pAnimResourceManager->FindActiveEditAnimIndex(pEditModelObj, boundAnimIndex);
}

void ResourceManager::UnloadResTextureFile(ViewerKeyType resFileKey) NN_NOEXCEPT
{
    UnbindTexturesFromAllObj(resFileKey);

    int textureIndex = FindResFileInfoIndex(resFileKey);
    NN_G3D_VIEWER_ASSERT_DETAIL(textureIndex != -1, NN_TEXT("3DEditor から送信されていないテクスチャのアンロードが実行されました: resFileKey = %d"), resFileKey);

    ResourceInfo* pResFileInfo = &m_ResFileInfoArray[textureIndex];
    NN_G3D_VIEWER_ASSERT(pResFileInfo->kind == FILEDATA_TEXTURE);

    m_pDestructionScheduler->Register(*pResFileInfo);
    m_ResFileInfoArray.EraseByIndex(textureIndex);
}

RuntimeErrorCode ResourceManager::ReloadEditSceneAnimObj(EditSceneAnimObj* pEditSceneAnimObj, nn::g3d::ResFile* pResFile) NN_NOEXCEPT
{
    NN_G3D_VIEWER_ASSERT_NOT_NULL(pResFile);

    RuntimeErrorCode result = m_pAnimResourceManager->ReloadEditSceneAnimObj(pEditSceneAnimObj, pResFile);
    if (result != RUNTIME_ERROR_CODE_NO_ERROR)
    {
        return result;
    }

    // リロード後に未使用のリソースがあったら解放する
    ScheduleDestroyUnusedResFiles();

    return RUNTIME_ERROR_CODE_NO_ERROR;
}

int ResourceManager::GetEditAnimCount() const NN_NOEXCEPT
{
    return m_pAnimResourceManager->GetEditAnimCount();
}
float ResourceManager::GetEditAnimFrameCount(int animIndex) const NN_NOEXCEPT
{
    return m_pAnimResourceManager->GetEditAnimFrameCount(animIndex);
}
const char* ResourceManager::GetEditAnimName(int animIndex) const NN_NOEXCEPT
{
    return m_pAnimResourceManager->GetEditAnimName(animIndex);
}
ViewerAnimKind ResourceManager::GetViewerAnimKind(int animIndex) const NN_NOEXCEPT
{
    return m_pAnimResourceManager->GetViewerAnimKind(animIndex);
}

bool ResourceManager::IsAnimationPlaying() const NN_NOEXCEPT
{
    return m_pAnimResourceManager->IsAnimationPlaying();
}
void ResourceManager::SetAnimPlaying(bool enable) NN_NOEXCEPT
{
    m_pAnimResourceManager->SetAnimPlaying(enable);
}
bool ResourceManager::IsLoopAnim() const NN_NOEXCEPT
{
    return m_pAnimResourceManager->IsLoopAnim();
}
float ResourceManager::GetFrame() const NN_NOEXCEPT
{
    return m_pAnimResourceManager->GetFrame();
}
float ResourceManager::GetFrameStep() const NN_NOEXCEPT
{
    return m_pAnimResourceManager->GetFrameStep();
}
void ResourceManager::SetFrameStep(float frameStep) NN_NOEXCEPT
{
    m_pAnimResourceManager->SetFrameStep(frameStep);
}
float ResourceManager::GetPreviewFrameStepRate() const NN_NOEXCEPT
{
    return m_pAnimResourceManager->GetPreviewFrameStepRate();
}
void ResourceManager::SetPreviewFrameStepRate(float rate) NN_NOEXCEPT
{
    m_pAnimResourceManager->SetPreviewFrameStepRate(rate);
}
float ResourceManager::GetFrameCount() const NN_NOEXCEPT
{
    return m_pAnimResourceManager->GetFrameCount();
}
void ResourceManager::SetFrame(float frame) NN_NOEXCEPT
{
    m_pAnimResourceManager->SetFrame(frame);
}

void ResourceManager::SetStartFrame(float startFrame) NN_NOEXCEPT
{
    m_pAnimResourceManager->SetStartFrame(startFrame);
}

void ResourceManager::SetFrameCount(float frameCount) NN_NOEXCEPT
{
    m_pAnimResourceManager->SetFrameCount(frameCount);
}

void ResourceManager::SetPlayPolicy(EditPlayPolicyKind kind) NN_NOEXCEPT
{
    m_pAnimResourceManager->SetPlayPolicy(kind);
}

void ResourceManager::DeleteEditAnimObj(ViewerKeyType key) NN_NOEXCEPT
{
    m_pAnimResourceManager->DeleteEditAnimObj(key);
}

RuntimeErrorCode ResourceManager::AttachModelObjs(EditModelObj* pTarget, nn::g3d::ResFile* pAttachModelResFile, nn::g3d::ModelObj** pModelObjs, int modelObjCount) NN_NOEXCEPT
{
    NN_G3D_REQUIRES(modelObjCount > 0, "");
    ViewerKeyType modelObjKey = ViewerKeyManager::GetInstance().FindKey(pModelObjs[0]);
    bool isModelObjAlreadyAttached = FindEditModelObj(modelObjKey) != nullptr;
    if (isModelObjAlreadyAttached)
    {
        return RUNTIME_ERROR_CODE_DUPLICATE_MODEL_OBJ_KEY;
    }

    return pTarget->AttachModelObjs(pAttachModelResFile, pModelObjs, modelObjCount);
}


RuntimeErrorCode ResourceManager::CreateEditModelObj(
    EditModelObj** pOut, nn::g3d::ResFile* pOriginalResFile) NN_NOEXCEPT
{
    NN_G3D_VIEWER_ASSERT_NOT_NULL(pOriginalResFile);
    NN_G3D_VIEWER_ASSERT_DETAIL(pOriginalResFile->GetModelCount() > 0, "%s\n", NN_G3D_VIEWER_RES_NAME(pOriginalResFile, GetName()));

    nn::g3d::ResModel* pOriginalResModel = pOriginalResFile->GetModel(0);
    NN_G3D_VIEWER_ASSERT_NOT_NULL_DETAIL(pOriginalResModel, "%s\n", NN_G3D_VIEWER_RES_NAME(pOriginalResFile, GetName()));

    void* buffer = m_pAllocator->Allocate(sizeof(EditModelObj), nn::g3d::detail::Alignment_Default, AllocateType_EditObj);
    if (buffer == nullptr)
    {
        return RUNTIME_ERROR_CODE_INSUFFICIENT_MEMORY;
    }

    EditModelObj* pEditModelObj = new (buffer) EditModelObj(
        m_pDevice, m_pAllocator, pOriginalResModel, m_pViewerCallback);
    NN_G3D_VIEWER_ASSERT_NOT_NULL(pEditModelObj);

    bool result = AddEditModelObj(pEditModelObj);
    NN_G3D_VIEWER_ASSERT(result);

    *pOut = pEditModelObj;
    return RUNTIME_ERROR_CODE_NO_ERROR;
}

RuntimeErrorCode ResourceManager::CreateAttachedEditModelObj(
    nn::g3d::ModelObj** pAttachTargetModelObjs, int modelObjCount, nn::g3d::ResFile* pFirstLoadResFile) NN_NOEXCEPT
{
    NN_G3D_VIEWER_ASSERT_NOT_NULL(pAttachTargetModelObjs);
    NN_G3D_VIEWER_ASSERT_NOT_NULL(pFirstLoadResFile);
    NN_G3D_VIEWER_ASSERT(modelObjCount > 0);

    nn::g3d::ResModel* pAttachTargetResModel = const_cast<nn::g3d::ResModel*>(pAttachTargetModelObjs[0]->GetResource());
    NN_G3D_VIEWER_ASSERT_NOT_NULL(pAttachTargetResModel);

    void* buffer = m_pAllocator->Allocate(sizeof(EditModelObj), nn::g3d::detail::Alignment_Default, AllocateType_EditObj);
    if (buffer == nullptr)
    {
        return RUNTIME_ERROR_CODE_INSUFFICIENT_MEMORY;
    }

    EditModelObj* pEditModelObj = new (buffer) EditModelObj(
        m_pDevice, m_pAllocator, pAttachTargetResModel, m_pViewerCallback);
    NN_G3D_VIEWER_ASSERT_NOT_NULL_DETAIL(pEditModelObj, "%s\n", NN_G3D_VIEWER_RES_NAME(pAttachTargetResModel, GetName()));

    bool result = AddEditModelObj(pEditModelObj);
    NN_G3D_VIEWER_ASSERT_DETAIL(result, "%s\n", NN_G3D_VIEWER_RES_NAME(pAttachTargetResModel, GetName()));

    RuntimeErrorCode errorCode = this->AttachModelObjs(pEditModelObj, pFirstLoadResFile, pAttachTargetModelObjs, modelObjCount);
    if (errorCode != RUNTIME_ERROR_CODE_NO_ERROR)
    {
        ScheduleDestructEditModelObj(pEditModelObj);
        return errorCode;
    }

    ForceBindExistingTextures(pEditModelObj);

    return RUNTIME_ERROR_CODE_NO_ERROR;
}


EditShaderArchive* ResourceManager::CreateAttachEditShaderArchive(nn::g3d::ResShaderArchive* pTargetResShaderArchive) NN_NOEXCEPT
{
    NN_G3D_VIEWER_ASSERT_NOT_NULL(pTargetResShaderArchive);
    void* buffer = m_pAllocator->Allocate(sizeof(EditShaderArchive), nn::g3d::detail::Alignment_Default, AllocateType_EditObj);
    if (buffer == nullptr)
    {
        return nullptr;
    }

    EditShaderArchive* editShaderArchive =
        new(buffer) EditShaderArchive(m_pDevice, m_pAllocator, pTargetResShaderArchive);
    return editShaderArchive;
}
