﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include "g3d_DeviceDependentObj.h"
#include "g3d_ResourceManager.h"

namespace nn { namespace g3d { namespace viewer { namespace detail {

//! @brief リソースの遅延破棄を管理するためのクラスです。
class ResourceDestructionScheduler : DeviceDependentObj
{
    NN_DISALLOW_COPY(ResourceDestructionScheduler);
public:
    ResourceDestructionScheduler(nn::gfx::Device* pDevice, Allocator* pAllocator, CallbackCaller* pCallbackCaller) NN_NOEXCEPT
        : DeviceDependentObj(pDevice, pAllocator)
        , m_pCallbackCaller(pCallbackCaller)
        , m_DestructionScheduledResources(pAllocator, nn::g3d::detail::Alignment_Default, DestructionInfo(ResourceInfo(), ResourceType_Unknown))
    {
    }

    //! @brief 遅延破棄対象として登録します。
    void Register(ResourceInfo resourceInfo) NN_NOEXCEPT;

    //! @brief 遅延破棄対象のスケジュールを更新します。更新により遅延破棄可能になった場合、すぐに破棄されます。
    void UpdateSchedule() NN_NOEXCEPT;

    int GetResourceCount() const NN_NOEXCEPT
    {
        return m_DestructionScheduledResources.GetCount();
    }

private:
    enum ResourceType
    {
        ResourceType_Unknown,
        ResourceType_ResFile,
        ResourceType_ResTextureFile,
        ResourceType_ResShaderFile,
        ResourceType_EditModelObj,
    };

    struct DestructionInfo
    {
        ResourceInfo resourceInfo;
        ResourceType type;

        // リソース破棄を遅延させるためのリソース破棄までのフレーム数
        int destroyFrameCount;

        explicit DestructionInfo(ResourceInfo resourceInfo, ResourceType type) NN_NOEXCEPT
            : resourceInfo(resourceInfo)
            , type(type)
        {
            const int DelayDestructionFrameCount = UniformBlockBufferingCount;
            destroyFrameCount = DelayDestructionFrameCount;
        }

        //! @brief リソースの破棄が予定されているかどうかを取得します。
        bool IsDestructionScheduled() const NN_NOEXCEPT
        {
            return destroyFrameCount >= 0;
        }

        //! @brief リソースの破棄予約状態を更新します。
        void UpdateDestructionSchedule() NN_NOEXCEPT
        {
            if (destroyFrameCount <= 0)
            {
                return;
            }

            --destroyFrameCount;
        }

        //! @brief リソースが破棄可能かどうかを取得します。
        bool IsDestructible() const NN_NOEXCEPT
        {
            return destroyFrameCount == 0;
        }
    };

private:
    ResourceType ConvertToResourceType(const ResourceInfo& info) NN_NOEXCEPT;
    void DeleteResFile(nn::g3d::ResFile* pResFile) NN_NOEXCEPT;
    void DeleteResTextureFile(nn::gfx::ResTextureFile* pResTextureFile) NN_NOEXCEPT;
    void DeleteResShaderFile(nn::g3d::ResShaderFile* pResShaderFile) NN_NOEXCEPT;
    void DeleteEditModelObj(EditModelObj* pDeleteEditModel) NN_NOEXCEPT;

private:
    CallbackCaller* m_pCallbackCaller;

    // 破棄予定のリソース
    DynamicArray<DestructionInfo> m_DestructionScheduledResources;
};

}}}}
