﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "g3d_EditCommandManager.h"



#include "g3d_Allocator.h"
#include "util/g3d_ResourceEditUtility.h"
#include "util/g3d_ViewerUtility.h"
#include "model/g3d_EditModelObj.h"
#include "anim/g3d_EditSkeletalAnimObj.h"
#include "anim/g3d_EditTexPatternAnimObj.h"
#include "anim/g3d_EditShaderParamAnimObj.h"
#include "anim/g3d_EditVisibilityAnimObj.h"
#include "anim/g3d_EditShapeAnimObj.h"
#include "anim/g3d_EditMaterialAnimObj.h"
#include "g3d_EditSocket.h"
#include "anim/g3d_EditSceneAnimObj.h"
#include "shader/g3d_EditShaderArchive.h"
#include "anim/g3d_EditAnimObj.h"
#include "command/g3d_NotifyMessageCommand.h"
#include "command/g3d_AttachCommand.h"
#include "command/g3d_LayoutModelCommand.h"
#include "command/g3d_FileLoadedCommand.h"
#include "command/g3d_SendRenderInfoDefinitionCommand.h"
#include "g3d_ResourceManager.h"

#include <nn/g3d/viewer/g3d_ViewerServer.h>

#include <nn/g3d/g3d_ModelObj.h>
#include <nn/g3d/g3d_ResFile.h>
#include <nn/g3d/g3d_ResShader.h>
#include <nn/diag/text/diag_SdkTextG3dviewer.h>
#include <nn/nn_SdkText.h>

using namespace nn::g3d::viewer::detail;

namespace {
    const char* GetErrorMessage(RuntimeErrorCode errorCode)
    {
        switch (errorCode)
        {
        case RUNTIME_ERROR_CODE_INVALID_MATERIAL_COUNT:        return NN_TEXT_G3DVIEWER("モデルのマテリアル数がランタイム側のリソースと異なります。");
        case RUNTIME_ERROR_CODE_ATTACH_CANCELED:               return NN_TEXT_G3DVIEWER("アタッチが中断されました。");
        case RUNTIME_ERROR_CODE_LOAD_FILE_FAILED:              return NN_TEXT_G3DVIEWER("3DEditor から送られてきたファイルの読み込み処理に失敗しました。接続先アプリケーションの実装の問題、もしくはメモリ不足などが原因の可能性があります。");
        case RUNTIME_ERROR_CODE_INSUFFICIENT_MEMORY:           return NN_TEXT_G3DVIEWER("メモリ不足です。");
        case RUNTIME_ERROR_CODE_BIND_ANIM_FAILED:              return NN_TEXT_G3DVIEWER("アニメーションのバインドに失敗しました。");
        case RUNTIME_ERROR_CODE_RETARGET_HOST_MODEL_NOT_FOUND: return NN_TEXT_G3DVIEWER("リターゲティングホストモデルがランタイム側に読み込まれていません。");
        case RUNTIME_ERROR_CODE_DUPLICATE_MODEL_OBJ_KEY:       return NN_TEXT_G3DVIEWER("重複したモデルキーが検知されました。接続先アプリケーション、もしくはランタイムライブラリの問題です。");
        case RUNTIME_ERROR_CODE_INVALID_SHADER_DETECTED:       return NN_TEXT_G3DVIEWER("不正なシェーダが検出されました。アタッチしたシェーダがアプリケーションでロードされているシェーダと異なる構造を持っている場合などにこのエラーが発生します。アタッチしたシェーダが間違っていないか、アプリケーションでロードしているシェーダアーカイブと 3DEditor にロードしているシェーダ定義がどちらも最新であるかどうかを確認して下さい。");
        case RUNTIME_ERROR_CODE_INVALID_MODEL_ATTACHED:        return NN_TEXT_G3DVIEWER("アタッチしたモデル中間ファイルが正しくありません。アプリケーション側でアタッチ対象に登録したモデルが、3DEditor でアタッチしたモデル中間ファイルから変換されたものであるかどうかを確認して下さい。");
        case RUNTIME_ERROR_CODE_OPEN_FILE_FAILED:              return NN_TEXT_G3DVIEWER("3DEditor が作成したバイナリファイルのオープンに失敗しました。詳細はアプリケーションのログに出力されているファイルオープンのリザルトコードを参照してください。");
        case RUNTIME_ERROR_CODE_TARGET_MODEL_NOT_FOUND:        return NN_TEXT_G3DVIEWER("3DEditor から指定されたキーに該当するモデルがランタイムライブラリに登録されていません。3DEditor、もしくはランタイムライブラリの不具合です。");
        case RUNTIME_ERROR_CODE_INVALID_SHAPE_COUNT:           return NN_TEXT_G3DVIEWER("モデルのシェイプ数がランタイム側のリソースと異なります。");
        case RUNTIME_ERROR_CODE_INVALID_MESH_COUNT:            return NN_TEXT_G3DVIEWER("モデルの LOD 数がランタイム側のリソースと異なります。");
        case RUNTIME_ERROR_CODE_INVALID_SUBMESH_COUNT:         return NN_TEXT_G3DVIEWER("モデルのサブメッシュ数がランタイム側のリソースと異なります。");
        case RUNTIME_ERROR_CODE_UNKNOWN_ERROR:                 return NN_TEXT_G3DVIEWER("不明なエラーです。");
        default:
            NN_UNEXPECTED_DEFAULT;
        }
    }
}

namespace nn { namespace g3d { namespace viewer { namespace detail {

    EditCommandManager::EditCommandManager(nn::gfx::Device* pDevice, Allocator* pAllocator, CallbackCaller* pCallbackCaller) NN_NOEXCEPT
        : m_IsWriteStarted(false)
        , m_pDevice(pDevice)
        , m_pAllocator(pAllocator)
        , m_pViewerCallback(pCallbackCaller)
        , m_CodePage(0)
        , m_CommandQueue(pAllocator)
        , m_AttachCommandQueue(pAllocator)
        , m_AttachingModelObjArray(pAllocator, nn::g3d::detail::Alignment_Default)
        , m_pResourceManager(nullptr)
    {
        NN_SDK_REQUIRES_NOT_NULL(pDevice);
        NN_SDK_REQUIRES_NOT_NULL(pAllocator);

        // TODO: ViewerKeyManager が static でなくなったらメンバにする
        ViewerKeyManager::Initialize(m_pAllocator);

        {
            void* pBuffer = m_pAllocator->Allocate(sizeof(ResourceManager), nn::DefaultAlignment, AllocateType_Other);
            NN_SDK_ASSERT_NOT_NULL(pBuffer);
            m_pResourceManager = new (pBuffer) ResourceManager(m_pDevice, m_pAllocator, m_pViewerCallback);
        }
    }

    EditCommandManager::~EditCommandManager() NN_NOEXCEPT
    {
        ViewerKeyManager::Finalize();

        ClearEditCommandManager();
        m_AttachingModelObjArray.Destroy();

        m_pResourceManager->~ResourceManager();
        m_pAllocator->Free(m_pResourceManager);

        m_pAllocator = nullptr;
    }

    void EditCommandManager::ForceDeleteAll() NN_NOEXCEPT
    {
        while (m_pResourceManager->GetResourceCount() > 0)
        {
            DeleteAll();
            TryDestroyUnreferencedResources();
        }

        // この時点で本来はすべてのキーが登録削除されているはずなので、あったら内部警告を出しておく
        for (int keyIndex = 0, keyCount = ViewerKeyManager::GetInstance().GetRegisteredKeyCount();
            keyIndex < keyCount; ++keyIndex)
        {
            ViewerKeyType key = ViewerKeyManager::GetInstance().GetKey(keyIndex);
            NN_UNUSED(key);
            NN_G3D_VIEWER_INTERNAL_WARNING(
                "Key %d(%s) has not been unregistered, force to unregister.\n",
                key, ViewerKeyManager::GetDataTypeAsString(ViewerKeyManager::GetInstance().FindDataType(key)));
        }

        ViewerKeyManager::GetInstance().UnregisterAllKeys();
    }

    void EditCommandManager::DeleteAll() NN_NOEXCEPT
    {
        for (Iter<ModelObj*> iter = m_AttachingModelObjArray.Begin(), end = m_AttachingModelObjArray.End();
            iter != end; ++iter)
        {
            iter = m_AttachingModelObjArray.Erase(iter);
        }

        m_pResourceManager->ScheduleDestructionAll();
    }

    void EditCommandManager::TryDestroyUnreferencedResources() NN_NOEXCEPT
    {
        m_pResourceManager->TryDestroyUnreferencedResources();
    }

    void EditCommandManager::ScheduleDestructEditModelObj(ViewerKeyType key) NN_NOEXCEPT
    {
        EditModelObj* pDeleteEditModel = m_pResourceManager->FindEditModelObj(key);
        if (pDeleteEditModel != nullptr)
        {
            m_pResourceManager->ScheduleDestructEditModelObj(pDeleteEditModel);
            return;
        }
        else
        {
            // EditModelObj が見つからない場合はアタッチ中の可能性あり
            ModelObj* pModelObj = ViewerKeyManager::GetInstance().FindData<nn::g3d::ModelObj>(key);
            if (pModelObj && IsAttaching(pModelObj))
            {
                const ResModel* pResModel = pModelObj->GetResource();
                NN_G3D_VIEWER_LOG("Canceled Attaching Model: \"%s\"\n", pResModel->GetName());
                RemoveAttachingModelObjs(pResModel);
                bool success;
                success = ViewerKeyManager::GetInstance().Unregister(key);
                NN_G3D_VIEWER_ASSERT_DETAIL(success, "Tried to unregister the key which has not been registered");
                success = ViewerKeyManager::GetInstance().Unregister(pResModel);
                NN_G3D_VIEWER_ASSERT_DETAIL(success, "Tried to unregister the key which has not been registered");
                return;
            }
        }
    }

    AttachCommand* EditCommandManager::FindSameAttachResModelCommand(const ResModel* pResModel) NN_NOEXCEPT
    {
        for (int i = 0, commandCount = m_AttachCommandQueue.GetCount(); i < commandCount; ++i)
        {
            AttachCommand* pCommand = m_AttachCommandQueue[i];
            if (pCommand->GetResModel() == pResModel)
            {
                return pCommand;
            }
        }

        return nullptr;
    }

    ViewerResult EditCommandManager::QueueAttachModel(nn::g3d::ModelObj** pTargetModels, int modelCount, const char* path) NN_NOEXCEPT
    {
        NN_G3D_VIEWER_REQUIRES_NOT_NULL(pTargetModels);

        AttachCommand* pExistingCommand = FindSameAttachResModelCommand(pTargetModels[0]->GetResource());
        if (pExistingCommand)
        {
            // 同じリソースを持つモデルがアタッチ予約中であれば、そこに追加する
            for (int modelIndex = 0; modelIndex < modelCount; ++modelIndex)
            {
                pExistingCommand->PushBack(pTargetModels[modelIndex]);
            }
        }
        else
        {
            // 見つからなければ新規登録
            const ResModel* pResModel = pTargetModels[0]->GetResource();
            ViewerKeyType resModelKey = ViewerKeyManager::GetInstance().FindKey(pResModel);
            if (resModelKey == InvalidKey)
            {
                // 登録されていない場合(ロードでなく、アタッチの場合)はここで新規にキーを登録する
                // ロードの場合は、ロードされたタイミングでキーが登録されている
                resModelKey = ViewerKeyManager::GetInstance().Register(pResModel, ViewerKeyManager::DataType_ResModel);
            }

            void* commandBuffer = m_pAllocator->Allocate(sizeof(AttachCommand), NN_ALIGNOF(AttachCommand), AllocateType_Communication);
            AttachCommand* command = new (commandBuffer) AttachCommand(
                this, m_pAllocator, Alignment_Default, resModelKey, ATTACH_MODEL, path);
            for (int modelIndex = 0; modelIndex < modelCount; ++modelIndex)
            {
                command->PushBack(pTargetModels[modelIndex]);
            }

            command->SetResModel(pResModel);

            bool success = m_AttachCommandQueue.PushBack(command);
            if (!success)
            {
                command->~AttachCommand();
                m_pAllocator->Free(commandBuffer);
                return ViewerResult_CommandQueueFull;
            }
        }

        for (int modelIndex = 0; modelIndex < modelCount; ++modelIndex)
        {
            m_AttachingModelObjArray.PushBack(pTargetModels[modelIndex]);
        }

        return ViewerResult_Success;
    }

    ViewerResult EditCommandManager::QueueDetachModel(const nn::g3d::ModelObj* pTargetModelObj) NN_NOEXCEPT
    {
        NN_G3D_VIEWER_ASSERT_NOT_NULL(pTargetModelObj);

        int index = m_AttachingModelObjArray.IndexOf(pTargetModelObj);
        if (index >= 0)
        {
            // アタッチ中の場合はアタッチをキャンセル
            RemoveAttachingModelObj(pTargetModelObj);
            return ViewerResult_Success;
        }

        // 多重削除防止のため指定モデルが存在しない場合は処理を抜ける
        EditModelObj* editModelObj = m_pResourceManager->FindEditModelObj(pTargetModelObj);
        if (editModelObj == nullptr)
        {
            return ViewerResult_NotAttached;
        }

        bool success = QueueDeleteEditModelObj(editModelObj);
        if (!success)
        {
            return ViewerResult_CommandQueueFull;
        }

        return ViewerResult_Success;
    }

    bool EditCommandManager::QueueDeleteEditModelObj(EditModelObj* pEditModelObj) NN_NOEXCEPT
    {
        const ResModel* pResModel = pEditModelObj->GetTargetModelObj(0)->GetResource();
        void* buffer = m_pAllocator->Allocate(sizeof(AttachCommand), NN_ALIGNOF(AttachCommand), AllocateType_Communication);
        ViewerKeyType key = ViewerKeyManager::GetInstance().FindKey(pResModel);
        NN_G3D_VIEWER_ASSERT_VALID_KEY(key);
        AttachCommand* command = new (buffer) AttachCommand(
            this, m_pAllocator, Alignment_Default,
            key, DETACH_MODEL, nullptr);
        command->PushBack(pEditModelObj->GetTargetModelObj(0));

        command->SetResModel(pResModel);
        return m_AttachCommandQueue.PushBack(command);
    }

    ViewerResult EditCommandManager::QueueAttachShaderArchive(nn::g3d::ResShaderArchive* pResShaderArchive, const char* path) NN_NOEXCEPT
    {
        NN_G3D_VIEWER_ASSERT_NOT_NULL(pResShaderArchive);
        ViewerKeyType shaderArchiveKey = ViewerKeyManager::GetInstance().Register(pResShaderArchive, ViewerKeyManager::DataType_ResShaderArchive);
        void* commandBuffer = m_pAllocator->Allocate(sizeof(AttachCommand), NN_ALIGNOF(AttachCommand), AllocateType_Communication);
        AttachCommand* command = new (commandBuffer) AttachCommand(
            this, m_pAllocator, Alignment_Default, shaderArchiveKey, ATTACH_SHADER_ARCHIVE, path);

        command->SetResShaderArchive(pResShaderArchive);

        if (!m_AttachCommandQueue.PushBack(command))
        {
            return ViewerResult_CommandQueueFull;
        }

        return ViewerResult_Success;
    }

    ViewerResult EditCommandManager::QueueDetachShaderArchive(const nn::g3d::ResShaderArchive* pResShaderArchive) NN_NOEXCEPT
    {
        NN_G3D_VIEWER_ASSERT_NOT_NULL(pResShaderArchive);

        // 多重削除防止のため指定シェーダアーカイブが存在しない場合は処理を抜ける
        EditShaderArchive* editShaderArchive = m_pResourceManager->FindEditShaderArchive(pResShaderArchive);
        if (editShaderArchive == nullptr)
        {
            return ViewerResult_NotAttached;
        }

        if (!QueueDeleteEditShaderArchive(editShaderArchive))
        {
            return ViewerResult_CommandQueueFull;
        }

        return ViewerResult_Success;
    }

    bool EditCommandManager::QueueDeleteEditShaderArchive(EditShaderArchive* editShaderArchive) NN_NOEXCEPT
    {
        ResShaderArchive* pResShaderArchive = editShaderArchive->GetTargetResShaderArchive();
        void* buffer = m_pAllocator->Allocate(sizeof(AttachCommand), NN_ALIGNOF(AttachCommand), AllocateType_Communication);
        ViewerKeyType key = ViewerKeyManager::GetInstance().FindKey(pResShaderArchive);
        NN_G3D_VIEWER_ASSERT_VALID_KEY(key);
        AttachCommand* command = new (buffer) AttachCommand(
            this, m_pAllocator, Alignment_Default,
            key,
            DETACH_SHADER_ARCHIVE, nullptr);
        command->SetResShaderArchive(pResShaderArchive);
        return m_AttachCommandQueue.PushBack(command);
    }

    void EditCommandManager::Reset() NN_NOEXCEPT
    {
        m_pResourceManager->Reset();
        m_IsWriteStarted = false;

        // キューに積まれたコマンドを破棄
        while (m_AttachCommandQueue.GetCount() > 0)
        {
            AttachCommand* pAttachCommand = m_AttachCommandQueue.PopFront();
            pAttachCommand->~AttachCommand();
            m_pAllocator->Free(pAttachCommand);
        }

        while (m_CommandQueue.GetCount() > 0)
        {
            IViewerCommand* pViewerCommand = m_CommandQueue.PopFront();
            pViewerCommand->~IViewerCommand();
            m_pAllocator->Free(pViewerCommand);
        }
    }

    void EditCommandManager::ClearEditCommandManager() NN_NOEXCEPT
    {
        DeleteAll();
        Reset();
    }

    AttachCommand*
        EditCommandManager::FindSameAttachModelObjQueue(const ModelObj* modelObj) const NN_NOEXCEPT
    {
        for (int commandIndex = 0, commandCount = m_AttachCommandQueue.GetCount(); commandIndex < commandCount; ++commandIndex)
        {
            AttachCommand* const pCommand = m_AttachCommandQueue[commandIndex];
            if (pCommand->GetKind() != ATTACH_MODEL)
            {
                continue;
            }

            for (int i = 0; i < pCommand->GetModelObjCount(); ++i)
            {
                if (pCommand->GetModelObj(i) == modelObj)
                {
                    return pCommand;
                }
            }
        }

        return nullptr;
    }

    AttachCommand*
        EditCommandManager::FindSameAttachShaderArchiveQueue(const ResShaderArchive* resShaderArchive) const NN_NOEXCEPT
    {
        for (int commandIndex = 0, commandCount = m_AttachCommandQueue.GetCount(); commandIndex < commandCount; ++commandIndex)
        {
            AttachCommand* const pCommand = m_AttachCommandQueue[commandIndex];
            if (pCommand->GetKind() != ATTACH_SHADER_ARCHIVE)
            {
                continue;
            }

            if (pCommand->GetResShaderArchive() == resShaderArchive)
            {
                return pCommand;
            }
        }

        return nullptr;
    }

    void EditCommandManager::ExecuteThreadUnsafeCommands() NN_NOEXCEPT
    {
        ExecuteAttachCommandQueueThreadUnsafe();
    }

    void EditCommandManager::ExecuteThreadSafeCommands(EditSocketBase* pSocket) NN_NOEXCEPT
    {
        NN_G3D_VIEWER_ASSERT_NOT_NULL(pSocket);

        ExecuteAttachCommandQueue(pSocket);
        ExecuteCommandQueueImpl(pSocket);
    }

    bool EditCommandManager::AddModelFileLoadedQueue(
        ViewerKeyType resFileKey, ViewerKeyType resModelKey, ViewerKeyType firstLoadedModelFileKey, ViewerKeyType toolKey) NN_NOEXCEPT
    {
        void* pBuffer = m_pAllocator->Allocate(sizeof(FileLoadedCommand), NN_ALIGNOF(FileLoadedCommand), AllocateType_Communication);
        NN_G3D_VIEWER_ASSERT_NOT_NULL_DETAIL(pBuffer, "insufficient memory");
        FileLoadedCommand* pCommand = new (pBuffer) FileLoadedCommand(resModelKey, resFileKey, firstLoadedModelFileKey, toolKey);
        return m_CommandQueue.PushBack(pCommand);
    }

    bool EditCommandManager::AddFileLoadedQueue(
        FileDataKind kind,
        ViewerKeyType resFileKey,
        ViewerKeyType toolKey) NN_NOEXCEPT
    {
        void* pBuffer = m_pAllocator->Allocate(sizeof(FileLoadedCommand), NN_ALIGNOF(FileLoadedCommand), AllocateType_Communication);
        NN_G3D_VIEWER_ASSERT_NOT_NULL_DETAIL(pBuffer, "insufficient memory");
        FileLoadedCommand* pCommand = new (pBuffer) FileLoadedCommand(resFileKey, toolKey, kind);
        return m_CommandQueue.PushBack(pCommand);
    }

    bool EditCommandManager::AddSceneAnimFileLoadedQueue(const EditSceneAnimObj* pEditSceneAnimObj, ViewerKeyType toolKey) NN_NOEXCEPT
    {
        NN_G3D_VIEWER_ASSERT_NOT_NULL(pEditSceneAnimObj);
        void* pBuffer = m_pAllocator->Allocate(sizeof(FileLoadedCommand), NN_ALIGNOF(FileLoadedCommand), AllocateType_Communication);
        NN_G3D_VIEWER_ASSERT_NOT_NULL_DETAIL(pBuffer, "insufficient memory");
        FileLoadedCommand* pCommand = new (pBuffer) FileLoadedCommand(pEditSceneAnimObj->GetResFileKey(), toolKey, FILEDATA_SCENE_ANIM);
        return m_CommandQueue.PushBack(pCommand);
    }

    bool EditCommandManager::AddModelFileReloadedQueue(const EditModelObj* pEditModelObj, ViewerKeyType resFileKey) NN_NOEXCEPT
    {
        NN_G3D_VIEWER_ASSERT_NOT_NULL(pEditModelObj);

        void* pBuffer = m_pAllocator->Allocate(sizeof(FileLoadedCommand), NN_ALIGNOF(FileLoadedCommand), AllocateType_Communication);
        NN_G3D_VIEWER_ASSERT_NOT_NULL_DETAIL(pBuffer, "insufficient memory");
        ViewerKeyType modelObjKey = ViewerKeyManager::GetInstance().FindKey(pEditModelObj->GetTargetModelObj(0));
        NN_G3D_VIEWER_ASSERT_VALID_KEY(modelObjKey);
        ViewerKeyType originalResFileKey = ViewerKeyManager::GetInstance().FindKey(pEditModelObj->GetOriginalResFileManagedByUser());
        FileLoadedCommand* pCommand = new (pBuffer) FileLoadedCommand(
            pEditModelObj->GetResModelKey(),
            modelObjKey,
            resFileKey,
            originalResFileKey,
            0);
        return m_CommandQueue.PushBack(pCommand);
    }

    bool EditCommandManager::AddAnimFileReloadedQueue(FileDataKind kind, ViewerKeyType newResFileKey, ViewerKeyType oldResFileKey) NN_NOEXCEPT
    {
        void* pBuffer = m_pAllocator->Allocate(sizeof(FileLoadedCommand), NN_ALIGNOF(FileLoadedCommand), AllocateType_Communication);
        NN_G3D_VIEWER_ASSERT_NOT_NULL_DETAIL(pBuffer, "insufficient memory");
        FileLoadedCommand* pCommand = new (pBuffer) FileLoadedCommand(
            oldResFileKey,
            newResFileKey,
            0,
            kind);
        return m_CommandQueue.PushBack(pCommand);
    }

    bool EditCommandManager::AddSceneAnimFileReloadedQueue(const EditSceneAnimObj* editAnimObj, ViewerKeyType resFileKey) NN_NOEXCEPT
    {
        NN_G3D_VIEWER_ASSERT_NOT_NULL(editAnimObj);

        void* pBuffer = m_pAllocator->Allocate(sizeof(FileLoadedCommand), NN_ALIGNOF(FileLoadedCommand), AllocateType_Communication);
        NN_G3D_VIEWER_ASSERT_NOT_NULL_DETAIL(pBuffer, "insufficient memory");
        FileLoadedCommand* pCommand = new (pBuffer) FileLoadedCommand(
            resFileKey,
            editAnimObj->GetResFileKey(),
            0,
            FILEDATA_SCENE_ANIM);
        return m_CommandQueue.PushBack(pCommand);
    }

    bool EditCommandManager::AddTextureFileReloadedQueue(ViewerKeyType newResFileKey, ViewerKeyType oldResFileKey) NN_NOEXCEPT
    {
        void* pBuffer = m_pAllocator->Allocate(sizeof(FileLoadedCommand), NN_ALIGNOF(FileLoadedCommand), AllocateType_Communication);
        NN_G3D_VIEWER_ASSERT_NOT_NULL_DETAIL(pBuffer, "insufficient memory");
        FileLoadedCommand* pCommand = new (pBuffer) FileLoadedCommand(
            oldResFileKey,
            newResFileKey,
            0,
            FILEDATA_TEXTURE);
        return m_CommandQueue.PushBack(pCommand);
    }

    ViewerResult EditCommandManager::AddModelLayoutQueue(ViewerKeyType modelKey,
        const nn::util::Vector3fType& scale,
        const nn::util::Vector3fType& rotate,
        const nn::util::Vector3fType& translate) NN_NOEXCEPT
    {
        void* pBuffer = m_pAllocator->Allocate(sizeof(LayoutModelCommand), NN_ALIGNOF(LayoutModelCommand), AllocateType_Communication);
        NN_G3D_VIEWER_ASSERT_NOT_NULL_DETAIL(pBuffer, "insufficient memory");
        LayoutModelCommand* pCommand = new (pBuffer) LayoutModelCommand(modelKey, scale, rotate, translate);
        if (!m_CommandQueue.PushBack(pCommand))
        {
            return ViewerResult_CommandQueueFull;
        }

        return ViewerResult_Success;
    }



    RuntimeErrorCode EditCommandManager::ReloadAnimResource(ViewerKeyType resFileKey, nn::g3d::ResFile* pResFile, FileDataKind kind) NN_NOEXCEPT
    {
        RuntimeErrorCode result = m_pResourceManager->ReloadAnimResource(resFileKey, pResFile, kind);
        if (result != RUNTIME_ERROR_CODE_NO_ERROR)
        {
            return result;
        }

        // TODO: 3DEditorが同期待ちしているので暫定的に同じResFileKeyでリロード完了通知を送る
        //       3DEditorの同期待ちをなくしたらリロード完了通知をなくす
        {
            bool success = AddAnimFileReloadedQueue(
                kind,
                resFileKey,
                resFileKey);
            NN_G3D_VIEWER_ASSERT(success);
        }

        return RUNTIME_ERROR_CODE_NO_ERROR;
    }

    RuntimeErrorCode EditCommandManager::ReloadTextureResource(ViewerKeyType resFileKey, nn::gfx::ResTextureFile* pResTextureFile) NN_NOEXCEPT
    {
        return m_pResourceManager->ReloadTextureResource(resFileKey, pResTextureFile);
    }

    RuntimeErrorCode EditCommandManager::ReloadEditSceneAnimObj(ViewerKeyType resFileKey, nn::g3d::ResFile* pResFile) NN_NOEXCEPT
    {
        NN_G3D_VIEWER_ASSERT_NOT_NULL(pResFile);
        EditSceneAnimObj* pEditSceneAnimObj = m_pResourceManager->FindEditSceneAnimObj(resFileKey);
        RuntimeErrorCode result = m_pResourceManager->ReloadEditSceneAnimObj(pEditSceneAnimObj, pResFile);
        if (result != RUNTIME_ERROR_CODE_NO_ERROR)
        {
            return result;
        }

        bool success = AddSceneAnimFileReloadedQueue(
            pEditSceneAnimObj,
            resFileKey);
        NN_G3D_VIEWER_ASSERT(success);

        return RUNTIME_ERROR_CODE_NO_ERROR;
    }

    void EditCommandManager::ExecuteAttachCommandQueueThreadUnsafe() NN_NOEXCEPT
    {
        ScopedLock scopedLock(m_AttachCommandQueue);

        // この関数を実行する所でチェックしているので、pSocket の nullptr チェックは行わない
        // Attach*** の遅延処理
        for (int commandIndex = 0, queueSize = m_AttachCommandQueue.GetCount(); commandIndex < queueSize; ++commandIndex)
        {
            AttachCommand* pCommand = m_AttachCommandQueue[commandIndex];
            NN_G3D_VIEWER_ASSERT_NOT_NULL(pCommand);

            switch (pCommand->GetExecuteState())
            {
            case AttachCommand::ExecuteState_WaitPreprocess:
                {
                    pCommand->ExecutePreprocess();
                }
                break;
            case AttachCommand::ExecuteState_WaitExecute:
                {
                    // コマンド実行処理はメインスレッドと非同期で処理できるので
                    // 別スレッド化可能な関数で行う
                }
                break;
            case AttachCommand::ExecuteState_WaitPostprocess:
                {
                    pCommand->ExecutePostprocess();
                }
                break;
            case AttachCommand::ExecuteState_Finish:
                {
                    pCommand->~AttachCommand();
                    m_pAllocator->Free(pCommand);
                    m_AttachCommandQueue.EraseAt(commandIndex);
                }
                return;
            case AttachCommand::ExecuteState_ImpossibleToContinue:
                {
                    // エラーになり得るのはロードモデルのときだけ
                    ModelObj* pModelObj = pCommand->GetModelObj(0);
                    NN_G3D_VIEWER_ASSERT_NOT_NULL(pModelObj);
                    ViewerKeyType modelObjKey = ViewerKeyManager::GetInstance().FindKey(pModelObj);
                    ScheduleDestructEditModelObj(modelObjKey);

                    // 該当するロード完了通知コマンドを削除
                    {
                        int fileLoadedCommandIndex = FindFileLoadedCommandIndex(pCommand->GetKey());
                        NN_G3D_VIEWER_ASSERT(fileLoadedCommandIndex >= 0);
                        IViewerCommand* pFileLoadedCommand = m_CommandQueue[fileLoadedCommandIndex];
                        m_CommandQueue.EraseAt(fileLoadedCommandIndex);
                        pFileLoadedCommand->~IViewerCommand();
                        m_pAllocator->Free(pFileLoadedCommand);
                    }

                    pCommand->~AttachCommand();
                    m_pAllocator->Free(pCommand);
                    m_AttachCommandQueue.EraseAt(commandIndex);
                    QueueErrorCommand(pCommand->GetLastError());
                }
                return;
            default:
                break;
            }
        }
    } // NOLINT (readability/fn_size)

    void EditCommandManager::ExecuteAttachCommandQueue(EditSocketBase* pSocket) NN_NOEXCEPT
    {
        ScopedLock scopedLock(m_AttachCommandQueue);

        // この関数を実行する所でチェックしているので、pSocket の nullptr チェックは行わない
        // Attach*** の遅延処理
        int waitExecuteCommandCount;
        do
        {
            if (m_IsWriteStarted)
            {
                if (pSocket->IsWriting())
                {
                    // 送信処理を書き込み中であれば、処理を抜ける
                    return;
                }

                // 処理が終わっていれば、このままキューに実行待ちコマンドがなくなるまで処理
                pSocket->ResetWriteFlag();
                m_IsWriteStarted = false;
            }

            for (int commandIndex = 0, queueSize = m_AttachCommandQueue.GetCount(); commandIndex < queueSize; ++commandIndex)
            {
                AttachCommand* pCommand = m_AttachCommandQueue[commandIndex];
                NN_G3D_VIEWER_ASSERT_NOT_NULL(pCommand);
                CommandResult result = pCommand->Execute(pSocket);
                if (result == CommandResult_Success)
                {
                    m_IsWriteStarted = true;
                }
            }

            // 通信処理が必要なアタッチコマンド数を調べる
            waitExecuteCommandCount = 0;
            for (int commandIndex = 0, queueSize = m_AttachCommandQueue.GetCount(); commandIndex < queueSize; ++commandIndex)
            {
                AttachCommand* pCommand = m_AttachCommandQueue[commandIndex];
                if (pCommand->GetExecuteState() == AttachCommand::ExecuteState_WaitExecute)
                {
                    ++waitExecuteCommandCount;
                }
            }
        } while ( 0 < waitExecuteCommandCount && pSocket->IsConnected() );
    }

    void EditCommandManager::ExecuteCommandQueueImpl(EditSocketBase* pSocket) NN_NOEXCEPT
    {
        NN_G3D_VIEWER_ASSERT_NOT_NULL(pSocket);

        ScopedLock scopedLock(m_CommandQueue);
        int queueSize;
        do
        {
            if (m_IsWriteStarted)
            {
                if (pSocket->IsWriting())
                {
                    // 送信処理を書き込み中であれば、処理を抜ける
                    return;
                }

                // 処理が終わっていれば、このままキューがなくなるまで処理
                pSocket->ResetWriteFlag();
                m_IsWriteStarted = false;
            }

            queueSize = m_CommandQueue.GetCount();
            int skippedCommandCount = 0;
            for (int idx = 0; idx < queueSize; ++idx)
            {
                IViewerCommand* pCommand = m_CommandQueue[idx];
                NN_G3D_VIEWER_ASSERT_NOT_NULL(pCommand);
                CommandResult result = pCommand->Execute(pSocket);
                if (result == CommandResult_Success)
                {
                    m_CommandQueue.EraseAt(idx);

                    pCommand->~IViewerCommand();
                    m_pAllocator->Free(pCommand);

                    m_IsWriteStarted = true;
                    break;
                }
                else if (result == CommandResult_Skipped)
                {
                    ++skippedCommandCount;
                }
            }

            if (skippedCommandCount == queueSize)
            {
                // 全てのコマンドがスキップされたら(他の処理待ちになっていたら)処理を抜ける
                return;
            }

        } while ( (queueSize > 0) && pSocket->IsConnected() );
    }

    int EditCommandManager::GetModelAnimBoundCount(const ModelObj* modelObj) const NN_NOEXCEPT
    {
        return m_pResourceManager->FindModelAnimBoundCount(modelObj);
    }

    int EditCommandManager::GetActiveEditAnimIndex(const ModelObj* pModelObj, int boundAnimIndex) const NN_NOEXCEPT
    {
        return m_pResourceManager->FindActiveEditAnimIndex(pModelObj, boundAnimIndex);
    }

    bool EditCommandManager::QueueErrorCommand(RuntimeErrorCode errorCode) NN_NOEXCEPT
    {
        void* pBuffer = m_pAllocator->Allocate(sizeof(NotifyMessageCommand), nn::DefaultAlignment, AllocateType_Communication);
        NN_G3D_VIEWER_ASSERT_NOT_NULL(pBuffer);

        // NN_TEXT を使うと UTF-8 になってしまうので、強制指定する
        const int utf8CodePage = 65001;
        NotifyMessageCommand* pCommand = new (pBuffer) NotifyMessageCommand(
            m_pAllocator,
            GetErrorMessage(errorCode),
            utf8CodePage,
            MessageType_RuntimeError,
            ViewerServer::SendUserMessageArg::MessageDestination_Dialog);

        return m_CommandQueue.PushBack(pCommand);
    }

    SendRenderInfoDefinitionCommand* EditCommandManager::QueueSendRenderInfoDefinitionCommand(CommandMonitor* pCommandMonitor) NN_NOEXCEPT
    {
        void* pBuffer = m_pAllocator->Allocate(sizeof(SendRenderInfoDefinitionCommand), nn::DefaultAlignment, AllocateType_Communication);
        NN_G3D_VIEWER_ASSERT_NOT_NULL(pBuffer);
        SendRenderInfoDefinitionCommand* pCommand = new (pBuffer) SendRenderInfoDefinitionCommand(m_pAllocator, pCommandMonitor);
        bool success = m_CommandQueue.PushBack(pCommand);
        if (!success)
        {
            pCommand->~SendRenderInfoDefinitionCommand();
            m_pAllocator->Free(pCommand);
            return nullptr;
        }

        return pCommand;
    }

    ViewerResult EditCommandManager::QueueNotifyMessageCommand(
        const nn::util::string_view& message,
        ViewerServer::SendUserMessageArg::MessageType messageType,
        ViewerServer::SendUserMessageArg::MessageDestination messageDestination) NN_NOEXCEPT
    {
        void* pCommandBuffer = m_pAllocator->Allocate(sizeof(NotifyMessageCommand), DefaultAlignment, AllocateType_Communication);
        if (pCommandBuffer == nullptr)
        {
            return ViewerResult_MemoryAllocationFailed;
        }

        MessageType packetMessageType;
        switch (messageType)
        {
        case ViewerServer::SendUserMessageArg::MessageType_Info:
            packetMessageType = MessageType_UserInfo;
            break;
        case ViewerServer::SendUserMessageArg::MessageType_Warning:
            packetMessageType = MessageType_UserWarning;
            break;
        case ViewerServer::SendUserMessageArg::MessageType_Error:
            packetMessageType = MessageType_UserError;
            break;
        default:
            NN_UNEXPECTED_DEFAULT;
        }

        NotifyMessageCommand* pNotifyMessageCommand = new (pCommandBuffer) NotifyMessageCommand(
            m_pAllocator,
            message,
            m_CodePage,
            packetMessageType,
            messageDestination);

        if (!m_CommandQueue.PushBack(pNotifyMessageCommand))
        {
            return ViewerResult_CommandQueueFull;
        }

        return ViewerResult_Success;
    }


    void EditCommandManager::SetRetargetingHostModel(ViewerKeyType animResFileKey, ViewerKeyType retargetHostModelObjKey) NN_NOEXCEPT
    {
        RuntimeErrorCode result = m_pResourceManager->SetRetargetingHostModel(animResFileKey, retargetHostModelObjKey);
        if (result != RUNTIME_ERROR_CODE_NO_ERROR)
        {
            this->QueueErrorCommand(result);
            return;
        }
    }

    void EditCommandManager::UnsetRetargetingHostModel(ViewerKeyType animResFileKey) NN_NOEXCEPT
    {
        m_pResourceManager->UnsetRetargetingHostModel(animResFileKey);
    }

    void EditCommandManager::SetPlayMotionMirroringEnabled(ViewerKeyType animResFileKey, bool isEnabled) NN_NOEXCEPT
    {
        m_pResourceManager->SetPlayMotionMirroringEnabled(animResFileKey, isEnabled);
    }

    void EditCommandManager::UnloadResTextureFile(ViewerKeyType resFileKey) NN_NOEXCEPT
    {
        m_pResourceManager->UnloadResTextureFile(resFileKey);
    }

    ModelObj* EditCommandManager::RemoveAttachingModelObj(const ResModel* pResModel) NN_NOEXCEPT
    {
        for (Iter<ModelObj*> iter = m_AttachingModelObjArray.Begin(), end = m_AttachingModelObjArray.End(); iter != end; ++iter)
        {
            ModelObj* pModelObj = *iter;
            if (pModelObj->GetResource() == pResModel)
            {
                m_AttachingModelObjArray.Erase(iter);
                return pModelObj;
            }
        }

        return nullptr;
    }

    void EditCommandManager::RemoveAttachingModelObjs(const ResModel* pResModel) NN_NOEXCEPT
    {
        for (Iter<ModelObj*> iter = m_AttachingModelObjArray.Begin(), end = m_AttachingModelObjArray.End(); iter != end; ++iter)
        {
            ModelObj* pModelObj = *iter;
            if (pModelObj->GetResource() == pResModel)
            {
                iter = m_AttachingModelObjArray.Erase(iter);
            }
        }
    }

    bool EditCommandManager::IsShaderArchiveAttaching(const nn::g3d::ResShaderArchive* pResShaderArchive) const NN_NOEXCEPT
    {
        NN_G3D_VIEWER_ASSERT_NOT_NULL(pResShaderArchive);
        if (FindSameAttachShaderArchiveQueue(pResShaderArchive) != nullptr)
        {
            return true;
        }

        bool isKeyRegistered = ViewerKeyManager::GetInstance().FindKey(pResShaderArchive) != InvalidKey;
        return !m_pResourceManager->HasShaderArchive(pResShaderArchive)
            && isKeyRegistered;
    }

}}}}

