﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/g3d/g3d_SkeletalAnimObj.h>
#include <algorithm>
#include <nn/g3d/detail/g3d_Perf.h>
#include <nn/g3d/g3d_SkeletonObj.h>
#include <nn/g3d/g3d_ModelObj.h>

#include <cfloat>

NN_PRAGMA_PUSH_WARNINGS
NN_DISABLE_WARNING_DEPRECATED_DECLARATIONS
namespace nn { namespace g3d {

namespace {

inline
void CopySTAndFlag(LocalMtx& local, Bit32 flag, const nn::util::Vector3fType& scale, const nn::util::Vector3fType& translate)
{
    local.flag &= ~ResBone::Mask_Transform;
    local.flag |= flag & ResBone::Mask_Transform;
    local.scale = scale;
    MatrixSetTranslate(&local.mtxRT, translate);
    //local.mtxRT.SetT(translate);
}

template <typename ConvRot>
struct LerpRot
{
    static void Blend(
        BoneAnimBlendResult* pTarget, const BoneAnimResult* pResult, float weight)
    {
        // rotate
        nn::util::Vector3fType axes[2];
        ConvRot::Convert(axes, pResult);

#if defined( NN_BUILD_CONFIG_TOOLCHAIN_GHS )
        const f32x2* pAxes = reinterpret_cast<const f32x2*>(axes);
        f32x2* pTargetAxes = reinterpret_cast<f32x2*>(pTarget->axes);
        pTargetAxes[0] = __PS_MADDS0F(pAxes[0], weight, pTargetAxes[0]);
        pTargetAxes[1] = __PS_MADDS0F(pAxes[1], weight, pTargetAxes[1]);
        pTargetAxes[2] = __PS_MADDS0F(pAxes[2], weight, pTargetAxes[2]);
#else
        VectorMultiply(&axes[0], axes[0], weight);
        VectorAdd(&pTarget->axes[0], pTarget->axes[0], axes[0]);
        //axes[0].Mul(axes[0], weight);
        //pTarget->axes[0].Add(pTarget->axes[0], axes[0]);

        VectorMultiply(&axes[1], axes[1], weight);
        VectorAdd(&pTarget->axes[1], pTarget->axes[1], axes[1]);
        //axes[1].Mul(axes[1], weight);
        //pTarget->axes[1].Add(pTarget->axes[1], axes[1]);
#endif
    }
};

struct SlerpEuler
{
    static void Blend(
        BoneAnimBlendResult* pTarget, const BoneAnimResult* pResult, float weight)
    {
        // rotate
        nn::util::Matrix4x3fType mtx;
        nn::util::Vector3fType euler;
        VectorLoad(&euler, pResult->euler);
        MatrixSetRotateXyz(&mtx, euler);
        //mtx.SetR(pResult->euler);
        if (pTarget->weight == 0.0f)
        {
            QuaternionFromMatrix(&pTarget->rotate, mtx);
            //pTarget->rotate.Set(mtx);
        }
        else
        {
            nn::util::Vector4fType quat;
            QuaternionFromMatrix(&quat, mtx);
            float t = weight / (pTarget->weight + weight);
            QuaternionSlerp(&pTarget->rotate, pTarget->rotate, quat, t);
            /*quat.Set(mtx);
            float t = weight * Math::Rcp(pTarget->weight + weight);
            pTarget->rotate.Slerp(pTarget->rotate, quat, t);*/
        }
    }
};

struct SlerpQuat
{
    static void Blend(
        BoneAnimBlendResult* pTarget, const BoneAnimResult* pResult, float weight)
    {
        // rotate
        if (pTarget->weight == 0.0f)
        {
            VectorLoad(&pTarget->rotate, pResult->quat);
            //pTarget->rotate = pResult->quat;
        }
        else
        {
            nn::util::Vector4fType quat;
            VectorLoad(&quat, pResult->quat);
            float t = weight / (pTarget->weight + weight);
            QuaternionSlerp(&pTarget->rotate, pTarget->rotate, quat, t);
            //float t = weight * Math::Rcp(pTarget->weight + weight);
            //pTarget->rotate.Slerp(pTarget->rotate, pResult->quat, t);
        }
    }
};

enum ConvertValue
{
    ConvertValue_Rotate    = 0x1 << 0,
    ConvertValue_Translate = 0x1 << 1,
    ConvertValue_Both      = ConvertValue_Rotate | ConvertValue_Translate
};

template <ConvertValue value>
static void ClearAnimResultValue(BoneAnimResult* pResult, const ResBoneAnim* pResBoneAnim, const ResBone* pResBone)
{
    if (NN_STATIC_CONDITION(value & ConvertValue_Rotate))
    {
        if (pResBone->GetMirroringState() != nn::g3d::ResBone::MirroringState_Side)
        {
            // scale の base が存在するかで、rotate の初期値のオフセットが変わる
            // バイナリーに rotate のオフセットを入れたいが、5.0.0 ではバイナリーに差分が出ては困るので以下のように自前で計算している
            if (pResBoneAnim->ToData().flag & ResBoneAnim::Flag_BaseScale)
            {
                std::memcpy(&pResult->euler, pResBoneAnim->ToData().pBaseValueArray.Get() + 3, sizeof(pResult->euler));
            }
            else
            {
                std::memcpy(&pResult->euler, pResBoneAnim->ToData().pBaseValueArray.Get(), sizeof(pResult->euler));
            }
        }
    }
    if (NN_STATIC_CONDITION(value & ConvertValue_Translate))
    {
        std::memcpy(&pResult->translate, pResBoneAnim->ToData().pBaseValueArray.Get() + pResBoneAnim->ToData().beginBaseTranslate, sizeof(pResult->translate));
    }
}

template <ResSkeleton::MirroringMode mode, ConvertValue value>
static void CalculateMirroring(BoneAnimResult* pResult, Bit32 state)
{
    switch (state)
    {
    case nn::g3d::ResBone::MirroringState_CenterPreRotate:
        {
            if (NN_STATIC_CONDITION(value & ConvertValue_Rotate))
            {
                pResult->euler.y = -pResult->euler.y;
                pResult->euler.z = -pResult->euler.z;
            }
            if (NN_STATIC_CONDITION(value & ConvertValue_Translate))
            {
                pResult->translate.x = -pResult->translate.x;
            }
        }
        break;
    case nn::g3d::ResBone::MirroringState_Side:
        {
            if (NN_STATIC_CONDITION(value & ConvertValue_Translate))
            {
                pResult->translate.x = -pResult->translate.x;
                pResult->translate.y = -pResult->translate.y;
                pResult->translate.z = -pResult->translate.z;
            }
        }
        break;
    case nn::g3d::ResBone::MirroringState_SideRoot:
        {
            if (NN_STATIC_CONDITION(mode == ResSkeleton::MirroringMode_X))
            {
                if (NN_STATIC_CONDITION(value & ConvertValue_Rotate))
                {
                    pResult->euler.x = pResult->euler.x + nn::util::FloatPi;
                    pResult->euler.y = -pResult->euler.y;
                    pResult->euler.z = -pResult->euler.z;
                }
                if (NN_STATIC_CONDITION(value & ConvertValue_Translate))
                {
                    pResult->translate.x = -pResult->translate.x;
                }
            }
            else if (NN_STATIC_CONDITION(mode == ResSkeleton::MirroringMode_XY))
            {
                if (NN_STATIC_CONDITION(value & ConvertValue_Rotate))
                {
                    pResult->euler.y = pResult->euler.y + nn::util::FloatPi;
                    pResult->euler.z = -pResult->euler.z;
                }
                if (NN_STATIC_CONDITION(value & ConvertValue_Translate))
                {
                    pResult->translate.y = -pResult->translate.y;
                }
            }
            else if (NN_STATIC_CONDITION(mode == ResSkeleton::MirroringMode_XZ))
            {
                if (NN_STATIC_CONDITION(value & ConvertValue_Rotate))
                {
                    pResult->euler.z = pResult->euler.z + nn::util::FloatPi;
                }
                if (NN_STATIC_CONDITION(value & ConvertValue_Translate))
                {
                    pResult->translate.z = -pResult->translate.z;
                }
            }
        }
        break;
    case nn::g3d::ResBone::MirroringState_CenterRotate:
        {
            if (NN_STATIC_CONDITION(mode == ResSkeleton::MirroringMode_XY))
            {
                if (NN_STATIC_CONDITION(value & ConvertValue_Rotate))
                {
                    pResult->euler.x = -pResult->euler.x;
                    pResult->euler.z = nn::util::FloatPi - pResult->euler.z;
                }
                if (NN_STATIC_CONDITION(value & ConvertValue_Translate))
                {
                    pResult->translate.x = -pResult->translate.x;
                }
            }
            else if (NN_STATIC_CONDITION(mode == ResSkeleton::MirroringMode_XZ))
            {
                if (NN_STATIC_CONDITION(value & ConvertValue_Rotate))
                {
                    pResult->euler.x = nn::util::FloatPi - pResult->euler.x;
                    pResult->euler.z = nn::util::FloatPi - pResult->euler.z;
                }
                if (NN_STATIC_CONDITION(value & ConvertValue_Translate))
                {
                    pResult->translate.x = -pResult->translate.x;
                }
            }
        }
        break;
    case nn::g3d::ResBone::MirroringState_CenterPostRotate:
        {
            if (NN_STATIC_CONDITION(mode == ResSkeleton::MirroringMode_XY))
            {
                if (NN_STATIC_CONDITION(value & ConvertValue_Rotate))
                {
                    pResult->euler.x = -pResult->euler.x;
                    pResult->euler.z = -pResult->euler.z;
                }
                if (NN_STATIC_CONDITION(value & ConvertValue_Translate))
                {
                    pResult->translate.y = -pResult->translate.y;
                }
            }
            else if (NN_STATIC_CONDITION(mode == ResSkeleton::MirroringMode_XZ))
            {
                if (NN_STATIC_CONDITION(value & ConvertValue_Rotate))
                {
                    pResult->euler.x = -pResult->euler.x;
                    pResult->euler.y = -pResult->euler.y;
                }
                if (NN_STATIC_CONDITION(value & ConvertValue_Translate))
                {
                    pResult->translate.z = -pResult->translate.z;
                }
            }
        }
    default:
        break;
    }
} // NOLINT

static void(*const s_pFuncCalculateMirror[])(BoneAnimResult* pResult, Bit32 state) =
{
    &CalculateMirroring<ResSkeleton::MirroringMode_X,  ConvertValue_Rotate>,
    &CalculateMirroring<ResSkeleton::MirroringMode_X,  ConvertValue_Translate>,
    &CalculateMirroring<ResSkeleton::MirroringMode_X,  ConvertValue_Both>,
    &CalculateMirroring<ResSkeleton::MirroringMode_XY, ConvertValue_Rotate>,
    &CalculateMirroring<ResSkeleton::MirroringMode_XY, ConvertValue_Translate>,
    &CalculateMirroring<ResSkeleton::MirroringMode_XY, ConvertValue_Both>,
    &CalculateMirroring<ResSkeleton::MirroringMode_XZ, ConvertValue_Rotate>,
    &CalculateMirroring<ResSkeleton::MirroringMode_XZ, ConvertValue_Translate>,
    &CalculateMirroring<ResSkeleton::MirroringMode_XZ, ConvertValue_Both>,
};

class QuatToQuat
{
public:

    static bool Convert(nn::util::Vector4fType* pQuat, const BoneAnimBlendResult* pResult) NN_NOEXCEPT
    {
        *pQuat = pResult->rotate;
        return true;
    }

    static bool Convert(nn::util::Vector4fType* pQuat, const BoneAnimResult* pResult) NN_NOEXCEPT
    {
        nn::util::VectorLoad(pQuat, pResult->quat);
        return true;
    }
};

inline
void NormalizeResult(BoneAnimBlendResult *pResult, float weight)
{
    float rcpWeight = 1 / weight;
    //float rcpWeight = Math::Rcp(weight);
#if defined( NN_BUILD_CONFIG_TOOLCHAIN_GHS )
    f32x2* pScaleTrans = reinterpret_cast<f32x2*>(&pResult->scale);
    pScaleTrans[0] = __PS_MULS0F(pScaleTrans[0], rcpWeight);
    pScaleTrans[1] = __PS_MULS0F(pScaleTrans[1], rcpWeight);
    pScaleTrans[2] = __PS_MULS0F(pScaleTrans[2], rcpWeight);
#else
    VectorMultiply(&pResult->scale, pResult->scale, rcpWeight);
    VectorMultiply(&pResult->translate, pResult->translate, rcpWeight);
    //pResult->scale.Mul(pResult->scale, rcpWeight);
    //pResult->translate.Mul(pResult->translate, rcpWeight);
#endif
    // rotate 成分は正規化不要。
    // axes -> mtx 時、もしくは quat ブレンド時に正規化されている。
}

// 0.1% 未満の誤差は無視する
inline
bool WeightEqualsZero(float weight)
{
    return Math::Abs(weight) < 0.001f;
}

inline
bool WeightEqualsOne(float weight)
{
    return Math::Abs(weight - 1.0f) < 0.001f;
}

NN_FORCEINLINE
void BlendTranslateAndFlag(BoneAnimBlendResult* pTarget, const BoneAnimResult* pResult, float weight)
{
    // translate
    nn::util::Vector3fType translate;
    VectorLoad(&translate, pResult->translate);
    VectorMultiply(&translate, translate, weight);
    VectorAdd(&pTarget->translate, pTarget->translate, translate);

    // flag and weight
    pTarget->flag |= ~pResult->flag; // ClearResult() の最適化のため反転して論理和をとる。
    pTarget->weight += weight;
}

struct InterpolateST
{
    static void BlendAndFlag(BoneAnimBlendResult* pTarget, const BoneAnimResult* pResult, float weight) NN_NOEXCEPT
    {
#if defined( NN_BUILD_CONFIG_TOOLCHAIN_GHS )
        f32x2* pTargetScaleTrans = reinterpret_cast<f32x2*>(&pTarget->scale);
        const f32x2* pScaleTrans = reinterpret_cast<const f32x2*>(&pResult->scale);
        pTargetScaleTrans[0] = __PS_MADDS0F(pScaleTrans[0], weight, pTargetScaleTrans[0]);
        pTargetScaleTrans[1] = __PS_MADDS0F(pScaleTrans[1], weight, pTargetScaleTrans[1]);
        pTargetScaleTrans[2] = __PS_MADDS0F(pScaleTrans[2], weight, pTargetScaleTrans[2]);

        // flag and weight
        pTarget->flag |= ~pResult->flag; // ClearResult() の最適化のため反転して論理和をとる。
        pTarget->weight += weight;
#else
        // scale
        nn::util::Vector3fType scale;
        VectorLoad(&scale, pResult->scale);
        VectorMultiply(&scale, scale, weight);
        VectorAdd(&pTarget->scale, pTarget->scale, scale);

        BlendTranslateAndFlag(pTarget, pResult, weight);
#endif
    }
};

struct AdditiveST
{
    static void BlendAndFlag(BoneAnimBlendResult* pTarget, const BoneAnimResult* pResult, float weight) NN_NOEXCEPT
    {
        // scale
        {
            nn::util::Float3 resultScale;
            resultScale.x = std::pow(pResult->scale.x, weight);
            resultScale.y = std::pow(pResult->scale.y, weight);
            resultScale.z = std::pow(pResult->scale.z, weight);

            if (WeightEqualsZero(pTarget->weight))
            {
                nn::util::VectorSet(&pTarget->scale, 1.0f, 1.0f, 1.0f);
            }

            nn::util::Vector3fType scale;
            VectorLoad(&scale, resultScale);
            VectorMultiply(&pTarget->scale, pTarget->scale, scale);
        }

        BlendTranslateAndFlag(pTarget, pResult, weight);
    }
};

} // anonymous namespace

//--------------------------------------------------------------------------------------------------

// メンバ関数ポインターはサイズの解釈が一定でないためヘッダに出さない
class SkeletalAnimObj::Impl
{
public:
    static void (SkeletalAnimObj::* const s_pFuncClearImpl[])(const ResSkeleton* pSkeleton);
    static void (SkeletalAnimObj::* const s_pFuncCalculateImpl[])();
    static void (SkeletalAnimObj::* const s_pFuncApplyToImpl[])(SkeletonObj* pSkeletonObj) const;
};

void (SkeletalAnimObj::* const SkeletalAnimObj::Impl::s_pFuncClearImpl[])(const ResSkeleton* pSkeleton) =
{
    &SkeletalAnimObj::ClearImpl<false, false>,
    &SkeletalAnimObj::ClearImpl<false, true>,
    &SkeletalAnimObj::ClearImpl<true, false>,
    &SkeletalAnimObj::ClearImpl<true, true>,
};

void (SkeletalAnimObj::* const SkeletalAnimObj::Impl::s_pFuncCalculateImpl[])() =
{
    &SkeletalAnimObj::CalculateImpl<false, false>,
    &SkeletalAnimObj::CalculateImpl<false, true>,
    &SkeletalAnimObj::CalculateImpl<true, false>,
    &SkeletalAnimObj::CalculateImpl<true, true>,
};

void (SkeletalAnimObj::* const SkeletalAnimObj::Impl::s_pFuncApplyToImpl[])(SkeletonObj* pSkeleton) const =
{
    &SkeletalAnimObj::ApplyToImpl<QuatToMtx>,
    &SkeletalAnimObj::ApplyToImpl<EulerToMtx>,
};

void SkeletalAnimObj::InitializeArgument::CalculateMemorySize() NN_NOEXCEPT
{
    int bindCount = std::max(GetMaxBoneCount(), GetMaxBoneAnimCount());
    int animCount = GetMaxBoneAnimCount();
    int curveCount = GetMaxCurveCount();

    for (int blockIndex = 0; blockIndex < MemoryBlockIndex_End; ++blockIndex)
    {
        m_MemoryBlock[blockIndex].Initialize();
    }

    // サイズ計算
    m_MemoryBlock[MemoryBlockIndex_ResultBuffer].SetSizeBy<BoneAnimResult>(1,  animCount);
    m_MemoryBlock[MemoryBlockIndex_BindTable].SetSizeBy<Bit32>(1, bindCount);
    size_t size = IsContextEnabled() ? nn::util::align_up(sizeof(AnimFrameCache) * curveCount, Alignment_Default) : 0;
    m_MemoryBlock[MemoryBlockIndex_FrameCacheArray].SetSize(size);
    size = IsRetargetingEnabled() ? sizeof(nn::util::Vector4fType) * GetMaxBoneAnimCount() : 0;
    m_MemoryBlock[MemoryBlockIndex_RetargetingPreCalculateBuffer].SetAlignment(MatrixVectorAlignment);
    m_MemoryBlock[MemoryBlockIndex_RetargetingPreCalculateBuffer].SetSize(size);

    m_WorkMemory.Initialize();
    for (int blockIndex = 0; blockIndex < MemoryBlockIndex_End; ++blockIndex)
    {
        m_WorkMemory.Append(&m_MemoryBlock[blockIndex]);
    }
}

bool SkeletalAnimObj::Initialize(const InitializeArgument& arg, void* pBuffer, size_t bufferSize) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(pBuffer != NULL || bufferSize == 0);
    NN_SDK_REQUIRES(IsAligned(pBuffer, Alignment_Buffer));

    if (!arg.IsMemoryCalculated())
    {
        return false;
    }
    if (arg.GetWorkMemorySize() > bufferSize)
    {
        // バッファーが必要なサイズに満たない場合は失敗。
        return false;
    }

    int bindCount = std::max(arg.GetMaxBoneCount(), arg.GetMaxBoneAnimCount());
    int curveCount = arg.GetMaxCurveCount();

    // メンバの初期化。
    SetBufferPtr(pBuffer);
    // フレーム関係はリソース設定時に初期化
    m_pRes = NULL;
    GetBindTable().Initialize(arg.GetBuffer<Bit32>(pBuffer, InitializeArgument::MemoryBlockIndex_BindTable), bindCount);
    GetContext().Initialize(arg.GetBuffer<AnimFrameCache>(pBuffer, InitializeArgument::MemoryBlockIndex_FrameCacheArray), curveCount);
    SetResultBuffer(arg.GetBuffer(pBuffer, InitializeArgument::MemoryBlockIndex_ResultBuffer));
    m_MaxBoneAnim = arg.GetMaxBoneAnimCount();
    m_pRetargetingPreCalculateArray = arg.GetBuffer<nn::util::Vector4fType>(pBuffer, InitializeArgument::MemoryBlockIndex_RetargetingPreCalculateBuffer);
    return true;
}

void SkeletalAnimObj::SetResource(const ResSkeletalAnim* pRes) NN_NOEXCEPT
{
    NN_G3D_REQUIRES(IsAcceptable(pRes) == true,                           NN_G3D_RES_GET_NAME(pRes, GetName()));
    NN_G3D_REQUIRES(pRes->GetBoneAnimCount() <= GetBindTable().GetSize(), NN_G3D_RES_GET_NAME(pRes, GetName()));

    m_pRes = pRes;
    m_pBoneAnimArray = pRes->ToData().pBoneAnimArray.Get();

    SetTargetUnbound();

    bool loop = (pRes->ToData().flag & AnimFlag_PlayPolicyLoop) != 0;
    ResetFrameCtrl(pRes->GetFrameCount(), loop);
    GetBindTable().SetAnimCount(pRes->GetBoneAnimCount());
    GetContext().SetCurveCount(pRes->GetCurveCount());
}

BindResult SkeletalAnimObj::Bind(const ResSkeleton* pSkeleton) NN_NOEXCEPT
{
    NN_G3D_REQUIRES(IsAcceptable(pSkeleton) == true, NN_G3D_RES_GET_NAME(m_pRes, GetName()));
    m_pResSkeleton = pSkeleton;
    m_Flag &= ~SkeletalAnimObj::Flag_RetargetingEnabled;
    m_Flag &= ~SkeletalAnimObj::Flag_MirroringEnabled;
    BindResult result = BindImpl(pSkeleton);
    SkeletalAnimObj::ClearResult(pSkeleton);
    return result;
}

BindResult SkeletalAnimObj::Bind(const SkeletonObj* pSkeletonObj) NN_NOEXCEPT
{
    NN_G3D_REQUIRES(pSkeletonObj != NULL, NN_G3D_RES_GET_NAME(m_pRes, GetName()));
    return Bind(pSkeletonObj->GetResource());
}

BindResult SkeletalAnimObj::Bind(const ResModel* pModel) NN_NOEXCEPT
{
    NN_G3D_REQUIRES(pModel != NULL, NN_G3D_RES_GET_NAME(m_pRes, GetName()));
    return Bind(pModel->GetSkeleton());
}

BindResult SkeletalAnimObj::Bind(const ModelObj* pModelObj) NN_NOEXCEPT
{
    NN_G3D_REQUIRES(pModelObj != NULL, NN_G3D_RES_GET_NAME(m_pRes, GetName()));
    return Bind(pModelObj->GetSkeleton()->GetResource());
}

BindResult SkeletalAnimObj::Bind(const BindArgument& arg) NN_NOEXCEPT
{
    // スケルトンリソースとアニメーションリソースをバインド
    const ResSkeleton* pSkeleton = arg.GetResource();
    const ResSkeleton* pHostSkeleton = arg.GetHostResource();
    NN_G3D_REQUIRES(pSkeleton != NULL, NN_G3D_RES_GET_NAME(m_pRes, GetName()));
    NN_G3D_REQUIRES(IsAcceptable(pSkeleton) == true, NN_G3D_RES_GET_NAME(m_pRes, GetName()));

    m_pResSkeleton = pSkeleton;
    BindResult result = BindImpl(pSkeleton);

    if (arg.IsMirroringEnabled())
    {
        NN_SDK_ASSERT(arg.GetResource()->HasMirroringInfo() == true, "The mirroring data does not exist in the model. Resource name : %s", NN_G3D_RES_GET_NAME(m_pRes, GetName()));
        NN_SDK_ASSERT(m_pRes->GetRotateMode() == ResSkeletalAnim::RotateMode_EulerXyz, "The rotate mode must be EulerXyz. Resource name : %s", NN_G3D_RES_GET_NAME(m_pRes, GetName()));
        m_Flag |= SkeletalAnimObj::Flag_MirroringEnabled;
    }
    else
    {
        m_Flag &= ~SkeletalAnimObj::Flag_MirroringEnabled;
    }

    if (arg.IsRetargetingEnabled() && pSkeleton != pHostSkeleton)
    {
        NN_G3D_REQUIRES(pHostSkeleton != NULL, NN_G3D_RES_GET_NAME(m_pRes, GetName()));
        m_Flag |= SkeletalAnimObj::Flag_RetargetingEnabled;
        result = InitRetargeting(pSkeleton, pHostSkeleton);
    }
    else
    {
        m_Flag &= ~SkeletalAnimObj::Flag_RetargetingEnabled;
    }

    // ClearResult の前に Retargeting と Mirroring のフラグを確定しておく
    SkeletalAnimObj::ClearResult(pSkeleton);
    return result;
}

BindResult SkeletalAnimObj::BindFast(const BindArgument& arg) NN_NOEXCEPT
{
    const ResSkeleton* pSkeleton = arg.GetResource();
    const ResSkeleton* pHostSkeleton = arg.GetHostResource();
    NN_G3D_REQUIRES(pSkeleton != NULL, NN_G3D_RES_GET_NAME(m_pRes, GetName()));
    NN_G3D_REQUIRES(IsAcceptable(pSkeleton) == true, NN_G3D_RES_GET_NAME(m_pRes, GetName()));
    NN_G3D_REQUIRES(pSkeleton == m_pRes->GetBindSkeleton(), NN_G3D_RES_GET_NAME(m_pRes, GetName()));

    m_pResSkeleton = pSkeleton;
    BindResult result;
    AnimBindTable& bindTable = GetBindTable();
    bindTable.ClearAll(pSkeleton->GetBoneCount());
    bindTable.BindAll(m_pRes->ToData().pBindIndexArray.Get());
    SetTargetBound();

    if (arg.IsMirroringEnabled())
    {
        NN_SDK_ASSERT(arg.GetResource()->HasMirroringInfo() == true, "The mirroring data does not exist in the model. Resource name : %s", NN_G3D_RES_GET_NAME(m_pRes, GetName()));
        NN_SDK_ASSERT(m_pRes->GetRotateMode() == ResSkeletalAnim::RotateMode_EulerXyz, "The rotate mode must be EulerXyz. Resource name : %s", NN_G3D_RES_GET_NAME(m_pRes, GetName()));
        m_Flag |= SkeletalAnimObj::Flag_MirroringEnabled;
    }
    else
    {
        m_Flag &= ~SkeletalAnimObj::Flag_MirroringEnabled;
    }

    if (arg.IsRetargetingEnabled() && pSkeleton != pHostSkeleton)
    {
        NN_G3D_REQUIRES(pHostSkeleton != NULL, NN_G3D_RES_GET_NAME(m_pRes, GetName()));
        m_Flag |= SkeletalAnimObj::Flag_RetargetingEnabled;
        result = InitRetargeting(pSkeleton, pHostSkeleton);
    }
    else
    {
        m_Flag &= ~SkeletalAnimObj::Flag_RetargetingEnabled;
    }

    // ClearResult の前に Retargeting と Mirroring のフラグを確定しておく
    SkeletalAnimObj::ClearResult(pSkeleton);
    result.SetSuccessBit();
    return result;
}

BindResult SkeletalAnimObj::Bind(const ResSkeleton* pSkeleton, const ResSkeleton* pHostSkeleton) NN_NOEXCEPT
{
    NN_G3D_REQUIRES(IsAcceptable(pSkeleton) == true, NN_G3D_RES_GET_NAME(m_pRes, GetName()));
    if (pSkeleton == pHostSkeleton)
    {
        return Bind(pSkeleton);
    }
    else
    {
        m_pResSkeleton = pSkeleton;
        m_Flag |= SkeletalAnimObj::Flag_RetargetingEnabled;
        BindImpl(pSkeleton);
        BindResult result = InitRetargeting(pSkeleton, pHostSkeleton);
        SkeletalAnimObj::ClearResult(pSkeleton);
        return result;
    }
}

BindResult SkeletalAnimObj::Bind(const SkeletonObj* pSkeletonObj, const SkeletonObj* pHostSkeletonObj) NN_NOEXCEPT
{
    NN_G3D_REQUIRES(pSkeletonObj != NULL,     NN_G3D_RES_GET_NAME(m_pRes, GetName()));
    NN_G3D_REQUIRES(pHostSkeletonObj != NULL, NN_G3D_RES_GET_NAME(m_pRes, GetName()));
    return Bind(pSkeletonObj->GetResource(), pHostSkeletonObj->GetResource());
}

BindResult SkeletalAnimObj::Bind(const ResModel* pModel, const ResModel* pHostModel) NN_NOEXCEPT
{
    NN_G3D_REQUIRES(pModel != NULL,     NN_G3D_RES_GET_NAME(m_pRes, GetName()));
    NN_G3D_REQUIRES(pHostModel != NULL, NN_G3D_RES_GET_NAME(m_pRes, GetName()));
    return Bind(pModel->GetSkeleton(), pHostModel->GetSkeleton());
}

BindResult SkeletalAnimObj::Bind(const ModelObj* pModelObj, const ModelObj* pHostModelObj) NN_NOEXCEPT
{
    NN_G3D_REQUIRES(pModelObj != NULL,     NN_G3D_RES_GET_NAME(m_pRes, GetName()));
    NN_G3D_REQUIRES(pHostModelObj != NULL, NN_G3D_RES_GET_NAME(m_pRes, GetName()));
    return Bind(pModelObj->GetSkeleton()->GetResource(), pHostModelObj->GetSkeleton()->GetResource());
}

void SkeletalAnimObj::BindFast(const ResSkeleton* pSkeleton) NN_NOEXCEPT
{
    NN_G3D_REQUIRES(IsAcceptable(pSkeleton) == true,        NN_G3D_RES_GET_NAME(m_pRes, GetName()));
    NN_G3D_REQUIRES(pSkeleton == m_pRes->GetBindSkeleton(), NN_G3D_RES_GET_NAME(m_pRes, GetName()));

    m_Flag &= ~SkeletalAnimObj::Flag_RetargetingEnabled;
    m_Flag &= ~SkeletalAnimObj::Flag_MirroringEnabled;

    AnimBindTable& bindTable = GetBindTable();
    bindTable.ClearAll(pSkeleton->GetBoneCount());
    bindTable.BindAll(m_pRes->ToData().pBindIndexArray.Get());
    m_pResSkeleton = pSkeleton;

    SetTargetBound();
    SkeletalAnimObj::ClearResult(pSkeleton);
}

void SkeletalAnimObj::BindFast(const ResModel* pModel) NN_NOEXCEPT
{
    NN_G3D_REQUIRES(pModel != NULL, NN_G3D_RES_GET_NAME(m_pRes, GetName()));
    BindFast(pModel->GetSkeleton());
}

BindResult SkeletalAnimObj::BindFast(const ResSkeleton* pSkeleton, const ResSkeleton* pHostSkeleton) NN_NOEXCEPT
{
    NN_G3D_REQUIRES(IsAcceptable(pSkeleton) == true,        NN_G3D_RES_GET_NAME(m_pRes, GetName()));
    NN_G3D_REQUIRES(pSkeleton == m_pRes->GetBindSkeleton(), NN_G3D_RES_GET_NAME(m_pRes, GetName()));
    NN_G3D_REQUIRES(pHostSkeleton != NULL,                  NN_G3D_RES_GET_NAME(m_pRes, GetName()));

    BindResult result;
    if (pSkeleton == pHostSkeleton)
    {
        BindFast(pSkeleton);
        result.SetSuccessBit();
    }
    else
    {
        m_Flag |= SkeletalAnimObj::Flag_RetargetingEnabled;

        m_pResSkeleton = pSkeleton;
        AnimBindTable& bindTable = GetBindTable();
        bindTable.ClearAll(pSkeleton->GetBoneCount());
        bindTable.BindAll(m_pRes->ToData().pBindIndexArray.Get());

        SetTargetBound();

        result = InitRetargeting(pSkeleton, pHostSkeleton);
        SkeletalAnimObj::ClearResult(pSkeleton);
    }

    return result;
}

BindResult SkeletalAnimObj::BindFast(const ResModel* pModel, const ResModel* pHostModel) NN_NOEXCEPT
{
    NN_G3D_REQUIRES(pModel != NULL,     NN_G3D_RES_GET_NAME(m_pRes, GetName()));
    NN_G3D_REQUIRES(pHostModel != NULL, NN_G3D_RES_GET_NAME(m_pRes, GetName()));

    return BindFast(pModel->GetSkeleton(), pHostModel->GetSkeleton());
}

void SkeletalAnimObj::ClearResult() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pRes);

    BoneAnimResult* pResultArray = GetResultArray();

    // バインドされていなくても初期化してしまう。
    for (int idxAnim = 0, animCount = GetAnimCount(); idxAnim < animCount; ++idxAnim)
    {
        const ResBoneAnim* pBoneAnim = GetBoneAnim(idxAnim);
        BoneAnimResult* pResult = &pResultArray[idxAnim];
        pBoneAnim->Initialize(pResult, NULL);
    }
}

void SkeletalAnimObj::ClearResult(const ResSkeleton* pSkeleton) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pRes);
    NN_G3D_REQUIRES(IsTargetBound() == true, NN_G3D_RES_GET_NAME(m_pRes, GetName()));
    NN_G3D_REQUIRES(pSkeleton != NULL,       NN_G3D_RES_GET_NAME(m_pRes, GetName()));

    int funcIndex = ((m_Flag & Flag_MirroringEnabled) != 0) << 1 | ((m_Flag & Flag_RetargetingEnabled) != 0);
    (this->*Impl::s_pFuncClearImpl[funcIndex])(pSkeleton);
}

void SkeletalAnimObj::SetBindFlag(const ResSkeleton* pSkeleton, int boneIndex, BindFlag flag) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pRes);
    NN_G3D_REQUIRES(IsTargetBound() == true, NN_G3D_RES_GET_NAME(m_pRes, GetName()));
    NN_G3D_REQUIRES(pSkeleton != NULL,       NN_G3D_RES_GET_NAME(m_pRes, GetName()));

    Bit32 flagRaw = CreateFlagValue<Bit32>(
        Bit32(flag), AnimBindTable::Shift_Flag, AnimBindTable::Mask_Flag);
    AnimBindTable& bindTable = GetBindTable();
    int branchEndIndex = pSkeleton->GetBranchEndIndex(boneIndex);
    do
    {
        int idxBoneAnim = bindTable.GetAnimIndex(boneIndex);
        if (idxBoneAnim != AnimBindTable::Flag_NotBound)
        {
            // バインドされているもののみ変更する。
            bindTable.SetBindFlagRaw(idxBoneAnim, flagRaw);
        }
    } while (++boneIndex < branchEndIndex);
}

void SkeletalAnimObj::Calculate() NN_NOEXCEPT
{
    NN_G3D_PERF_LEVEL1("SkeletalAnimObj::Calculate");
    NN_SDK_REQUIRES_NOT_NULL(m_pRes);
    NN_G3D_REQUIRES(IsTargetBound() == true, NN_G3D_RES_GET_NAME(m_pRes, GetName()));

    if (IsFrameChanged())
    {
        int funcIndex = ((m_Flag & Flag_MirroringEnabled) != 0) << 1 | ((m_Flag & Flag_RetargetingEnabled) != 0);
        (this->*Impl::s_pFuncCalculateImpl[funcIndex])();
        UpdateLastFrame();
    }
}

void SkeletalAnimObj::ApplyTo(ModelObj* pModelObj) const NN_NOEXCEPT
{
    NN_G3D_REQUIRES(pModelObj != NULL, NN_G3D_RES_GET_NAME(m_pRes, GetName()));
    ApplyTo(pModelObj->GetSkeleton());
}

void SkeletalAnimObj::ApplyTo(SkeletonObj* pSkeletonObj) const NN_NOEXCEPT
{
    NN_G3D_PERF_LEVEL1("SkeletalAnimObj::ApplyTo");
    NN_SDK_REQUIRES_NOT_NULL(m_pRes);
    NN_G3D_REQUIRES(IsTargetBound() == true, NN_G3D_RES_GET_NAME(m_pRes, GetName()));
    NN_G3D_REQUIRES(pSkeletonObj != NULL,    NN_G3D_RES_GET_NAME(m_pRes, GetName()));

    int funcIndex = m_pRes->GetRotateMode() >> ResSkeletalAnim::Shift_Rot;
    (this->*Impl::s_pFuncApplyToImpl[funcIndex])(pSkeletonObj);
}

//--------------------------------------------------------------------------------------------------

BindResult SkeletalAnimObj::BindImpl(const ResSkeleton* pSkeleton) NN_NOEXCEPT
{
    NN_G3D_REQUIRES(IsAcceptable(pSkeleton) == true, NN_G3D_RES_GET_NAME(m_pRes, GetName()));

    BindResult result;
    AnimBindTable& bindTable = GetBindTable();
    bindTable.ClearAll(pSkeleton->GetBoneCount());

    for (int idxAnim = 0, animCount = bindTable.GetAnimCount(); idxAnim < animCount; ++idxAnim)
    {
        const char* name = GetBoneAnim(idxAnim)->GetName();
        int idxTarget = pSkeleton->FindBoneIndex(name);
        if (idxTarget >= 0)
        {
            bindTable.Bind(idxAnim, idxTarget);
            result.SetSuccessBit();
        }
        else
        {
            result.SetFailureBit();
        }
    }

    SetTargetBound();
    return result;
}

template <bool mirroring, bool retargeting>
void SkeletalAnimObj::ClearImpl(const ResSkeleton* pSkeleton) NN_NOEXCEPT
{
    BoneAnimResult* pResultArray = GetResultArray();
    const AnimBindTable& bindTable = GetBindTable();

    int mirroringModeFuncOffset = 0;
    NN_UNUSED(mirroringModeFuncOffset);
    if (NN_STATIC_CONDITION(mirroring))
    {
        mirroringModeFuncOffset = (m_pResSkeleton->GetMirroringMode() >> (ResSkeleton::Shift_MirroringMode)) * 3;
    }

    for (int idxBoneAnim = 0, boneAnimCount = bindTable.GetAnimCount();
        idxBoneAnim < boneAnimCount; ++idxBoneAnim)
    {
        int idxTarget = bindTable.GetTargetIndex(idxBoneAnim);
        if (idxTarget != AnimBindTable::Flag_NotBound)
        {
            const ResBoneAnim* pBoneAnim = GetBoneAnim(idxBoneAnim);
            const ResBone* pBone = pSkeleton->GetBone(idxTarget);
            BoneAnimResult* pResult = &pResultArray[idxBoneAnim];

            // ミラーリング再生する場合は、ボーンアニメーションをミラー先の Result と ResBone で初期化する
            int mirroredBoneIndex = -1;
            NN_UNUSED(mirroredBoneIndex);
            if (NN_STATIC_CONDITION(mirroring))
            {
                mirroredBoneIndex = m_pResSkeleton->GetMirroredBoneIndex(idxTarget);
                if (mirroredBoneIndex >= 0)
                {
                    pBone = pSkeleton->GetBone(mirroredBoneIndex);
                    pResult = &pResultArray[bindTable.GetAnimIndex(mirroredBoneIndex)];
                }
            }

            pBoneAnim->Initialize(pResult, pBone);

            // ベース値があってカーブを持たないボーンはここで事前にリターゲティング計算しておく
            if (NN_STATIC_CONDITION(retargeting))
            {
                // ベース値があってカーブを持つボーンは Calculate() で毎回計算されるため事前計算不要
                // ベース値が無いボーンはモデルのバインド値が適用されるためアニメーション計算しないので事前計算不要
                if (!((pBoneAnim->ToData().flag ^ ResBoneAnim::Flag_BaseTranslate) & (ResBoneAnim::Flag_BaseTranslate | ResBoneAnim::Mask_CurveTranslate)))
                {
                    VectorQuaternionRotateXyz(&pResult->translate, m_pRetargetingPreCalculateArray[idxBoneAnim], pResult->translate);
                }
            }

            // Rotate もしくは Translate のベース値が存在してカーブを持たないボーンはここで事前にミラーリングの計算をしておく
            if (NN_STATIC_CONDITION(mirroring))
            {
                if (mirroredBoneIndex >= 0)
                {
                    if (!((pBoneAnim->ToData().flag ^ ResBoneAnim::Flag_BaseRotate) & (ResBoneAnim::Flag_BaseRotate | ResBoneAnim::Mask_CurveRotate)) &&
                        !((pBoneAnim->ToData().flag ^ ResBoneAnim::Flag_BaseTranslate) & (ResBoneAnim::Flag_BaseTranslate | ResBoneAnim::Mask_CurveTranslate)))
                    {
                        (s_pFuncCalculateMirror[mirroringModeFuncOffset + 2])(pResult, pBone->GetMirroringState());
                    }
                    else if (!((pBoneAnim->ToData().flag ^ ResBoneAnim::Flag_BaseRotate) & (ResBoneAnim::Flag_BaseRotate | ResBoneAnim::Mask_CurveRotate)))
                    {
                        (s_pFuncCalculateMirror[mirroringModeFuncOffset + 0])(pResult, pBone->GetMirroringState());
                    }
                    else if (!((pBoneAnim->ToData().flag ^ ResBoneAnim::Flag_BaseTranslate) & (ResBoneAnim::Flag_BaseTranslate | ResBoneAnim::Mask_CurveTranslate)))
                    {
                        (s_pFuncCalculateMirror[mirroringModeFuncOffset + 1])(pResult, pBone->GetMirroringState());
                    }
                }
            }
        }
    }
}

template <bool mirroring, bool retargeting>
void SkeletalAnimObj::CalculateImpl() NN_NOEXCEPT
{
    float frame = GetFrameCtrl().GetFrame();
    BoneAnimResult* pResultArray = GetResultArray();
    const AnimBindTable& bindTable = GetBindTable();
    AnimContext& context = GetContext();

    int mirroringModeFuncOffset = 0;
    NN_UNUSED(mirroringModeFuncOffset);
    if (NN_STATIC_CONDITION(mirroring))
    {
        mirroringModeFuncOffset = (m_pResSkeleton->GetMirroringMode() >> (ResSkeleton::Shift_MirroringMode)) * 3;
    }

    if (context.IsFrameCacheValid())
    {
        int contextCount = 0;
        for (int idxBoneAnim = 0, boneAnimCount = bindTable.GetAnimCount();
            idxBoneAnim < boneAnimCount; ++idxBoneAnim)
        {
            const ResBoneAnim* pBoneAnim = GetBoneAnim(idxBoneAnim);
            int idxContext = contextCount;
            contextCount += pBoneAnim->GetCurveCount();

            if (!bindTable.IsCalculateEnabled(idxBoneAnim))
            {
                continue;
            }

            BoneAnimResult* pResult = &pResultArray[idxBoneAnim];

            // ミラーリングされるボーンの場合。ミラーリングされるボーンかつリターゲティングが有効の場合もここで処理する
            if (NN_STATIC_CONDITION(mirroring))
            {
                // ミラーリング再生するボーンではミラー先ボーンアニメ Result にアニメ結果を格納するように参照結果を変更する
                int boneIndex = bindTable.GetTargetIndex(idxBoneAnim);
                int mirroredBoneIndex = m_pResSkeleton->GetMirroredBoneIndex(boneIndex);
                if (mirroredBoneIndex >= 0)
                {
                    int boneAnimResultIndex = bindTable.GetAnimIndex(mirroredBoneIndex);
                    pResult = &pResultArray[boneAnimResultIndex];
                    if ((pBoneAnim->ToData().flag & ResBoneAnim::Mask_CurveTranslate) && (pBoneAnim->ToData().flag & ResBoneAnim::Mask_CurveRotate))
                    {
                        ClearAnimResultValue<ConvertValue_Both>(pResult, pBoneAnim, m_pResSkeleton->GetBone(mirroredBoneIndex));
                        pBoneAnim->Evaluate(pResult, frame, context.GetFrameCacheArray(idxContext));
                        if (NN_STATIC_CONDITION(retargeting))
                        {
                            VectorQuaternionRotateXyz(&pResult->translate, m_pRetargetingPreCalculateArray[idxBoneAnim], pResult->translate);
                        }
                        (s_pFuncCalculateMirror[mirroringModeFuncOffset + 2])(pResult, m_pResSkeleton->GetBone(mirroredBoneIndex)->GetMirroringState());
                    }
                    else if (pBoneAnim->ToData().flag & ResBoneAnim::Mask_CurveRotate)
                    {
                        ClearAnimResultValue<ConvertValue_Rotate>(pResult, pBoneAnim, m_pResSkeleton->GetBone(mirroredBoneIndex));
                        pBoneAnim->Evaluate(pResult, frame, context.GetFrameCacheArray(idxContext));
                        (s_pFuncCalculateMirror[mirroringModeFuncOffset + 0])(pResult, m_pResSkeleton->GetBone(mirroredBoneIndex)->GetMirroringState());
                    }
                    else if (pBoneAnim->ToData().flag & ResBoneAnim::Mask_CurveTranslate)
                    {
                        ClearAnimResultValue<ConvertValue_Translate>(pResult, pBoneAnim, m_pResSkeleton->GetBone(mirroredBoneIndex));
                        pBoneAnim->Evaluate(pResult, frame, context.GetFrameCacheArray(idxContext));
                        if (NN_STATIC_CONDITION(retargeting))
                        {
                            VectorQuaternionRotateXyz(&pResult->translate, m_pRetargetingPreCalculateArray[idxBoneAnim], pResult->translate);
                        }
                        (s_pFuncCalculateMirror[mirroringModeFuncOffset + 1])(pResult, m_pResSkeleton->GetBone(mirroredBoneIndex)->GetMirroringState());
                    }
                    else
                    {
                        pBoneAnim->Evaluate(pResult, frame, context.GetFrameCacheArray(idxContext));
                    }
                    continue;
                }
            }

            // ミラーリングボーンではなくリターゲッティングが有効な場合
            if (NN_STATIC_CONDITION(retargeting))
            {
                if (pBoneAnim->ToData().flag & ResBoneAnim::Mask_CurveTranslate)
                {
                    ClearAnimResultValue<ConvertValue_Translate>(pResult, pBoneAnim, NULL);
                    pBoneAnim->Evaluate(pResult, frame, context.GetFrameCacheArray(idxContext));
                    VectorQuaternionRotateXyz(&pResult->translate, m_pRetargetingPreCalculateArray[idxBoneAnim], pResult->translate);
                    continue;
                }
            }

            // ミラーリングされるボーンではなくリターゲッティングが無効な場合は単純にカーブからアニメーション値を読み込む
            pBoneAnim->Evaluate(pResult, frame, context.GetFrameCacheArray(idxContext));
        }
    }
    else
    {
        for (int idxBoneAnim = 0, boneAnimCount = bindTable.GetAnimCount();
            idxBoneAnim < boneAnimCount; ++idxBoneAnim)
        {
            if (!bindTable.IsCalculateEnabled(idxBoneAnim))
            {
                continue;
            }
            const ResBoneAnim* pBoneAnim = GetBoneAnim(idxBoneAnim);
            BoneAnimResult* pResult = &pResultArray[idxBoneAnim];

            if (NN_STATIC_CONDITION(mirroring))
            {
                int boneIndex = bindTable.GetTargetIndex(idxBoneAnim);
                int mirroredBoneIndex = m_pResSkeleton->GetMirroredBoneIndex(boneIndex);
                if (mirroredBoneIndex >= 0)
                {
                    int boneAnimResultIndex = bindTable.GetAnimIndex(mirroredBoneIndex);
                    pResult = &pResultArray[boneAnimResultIndex];
                    if ((pBoneAnim->ToData().flag & ResBoneAnim::Mask_CurveTranslate) && (pBoneAnim->ToData().flag & ResBoneAnim::Mask_CurveRotate))
                    {
                        ClearAnimResultValue<ConvertValue_Both>(pResult, pBoneAnim, m_pResSkeleton->GetBone(mirroredBoneIndex));
                        pBoneAnim->Evaluate(pResult, frame);
                        if (NN_STATIC_CONDITION(retargeting))
                        {
                            VectorQuaternionRotateXyz(&pResult->translate, m_pRetargetingPreCalculateArray[idxBoneAnim], pResult->translate);
                        }
                        (s_pFuncCalculateMirror[mirroringModeFuncOffset + 2])(pResult, m_pResSkeleton->GetBone(mirroredBoneIndex)->GetMirroringState());
                    }
                    else if (pBoneAnim->ToData().flag & ResBoneAnim::Mask_CurveRotate)
                    {
                        ClearAnimResultValue<ConvertValue_Rotate>(pResult, pBoneAnim, m_pResSkeleton->GetBone(mirroredBoneIndex));
                        pBoneAnim->Evaluate(pResult, frame);
                        (s_pFuncCalculateMirror[mirroringModeFuncOffset + 0])(pResult, m_pResSkeleton->GetBone(mirroredBoneIndex)->GetMirroringState());
                    }
                    else if (pBoneAnim->ToData().flag & ResBoneAnim::Mask_CurveTranslate)
                    {
                        ClearAnimResultValue<ConvertValue_Translate>(pResult, pBoneAnim, m_pResSkeleton->GetBone(mirroredBoneIndex));
                        pBoneAnim->Evaluate(pResult, frame);
                        if (NN_STATIC_CONDITION(retargeting))
                        {
                            VectorQuaternionRotateXyz(&pResult->translate, m_pRetargetingPreCalculateArray[idxBoneAnim], pResult->translate);
                        }
                        (s_pFuncCalculateMirror[mirroringModeFuncOffset + 1])(pResult, m_pResSkeleton->GetBone(mirroredBoneIndex)->GetMirroringState());
                    }
                    else
                    {
                        pBoneAnim->Evaluate(pResult, frame);
                    }
                    continue;
                }
            }
            if (NN_STATIC_CONDITION(retargeting))
            {
                if (pBoneAnim->ToData().flag & ResBoneAnim::Mask_CurveTranslate)
                {
                    ClearAnimResultValue<ConvertValue_Translate>(pResult, pBoneAnim, NULL);
                    pBoneAnim->Evaluate(pResult, frame);
                    VectorQuaternionRotateXyz(&pResult->translate, m_pRetargetingPreCalculateArray[idxBoneAnim], pResult->translate);
                    continue;
                }
            }
            pBoneAnim->Evaluate(pResult, frame);
        }
    }
} // NOLINT

template <typename ConvRot>
void SkeletalAnimObj::ApplyToImpl(SkeletonObj* pSkeletonObj) const NN_NOEXCEPT
{
    LocalMtx* pLocalMtxArray = pSkeletonObj->GetLocalMtxArray();
    const BoneAnimResult* pResultArray = GetResultArray();
    const AnimBindTable& bindTable = GetBindTable();

    for (int idxBoneAnim = 0, boneAnimCount = bindTable.GetAnimCount();
        idxBoneAnim < boneAnimCount; ++idxBoneAnim)
    {
        if (!bindTable.IsApplyEnabled(idxBoneAnim))
        {
            continue;
        }

        int idxBone = bindTable.GetTargetIndex(idxBoneAnim);
        LocalMtx& local = pLocalMtxArray[idxBone];
        const BoneAnimResult* pResult = &pResultArray[idxBoneAnim];
        nn::util::Vector3fType scale, translate;
        VectorLoad(&scale, pResult->scale);
        VectorLoad(&translate, pResult->translate);
        CopySTAndFlag(local, pResult->flag, scale, translate);
        //CopySTAndFlag(local, pResult->flag, pResult->scale, pResult->translate);
        ConvRot::Convert(&local.mtxRT, pResult);
    }
}

BindResult SkeletalAnimObj::InitRetargeting(
    const ResSkeleton* pTargetSkeleton, const ResSkeleton* pHostSkeleton) NN_NOEXCEPT
{
    NN_G3D_REQUIRES(pTargetSkeleton != NULL,                 NN_G3D_RES_GET_NAME(m_pRes, GetName()));
    NN_G3D_REQUIRES(pHostSkeleton != NULL,                   NN_G3D_RES_GET_NAME(m_pRes, GetName()));
    NN_G3D_REQUIRES(m_pRetargetingPreCalculateArray != NULL, NN_G3D_RES_GET_NAME(m_pRes, GetName()));

    BindResult result;
    AnimBindTable& bindTable = GetBindTable();

    for (int idxAnim = 0, animCount = bindTable.GetAnimCount(); idxAnim < animCount; ++idxAnim)
    {
        nn::util::Vector4fType& preCalcResult = m_pRetargetingPreCalculateArray[idxAnim];
        VectorSet(&preCalcResult, 0.0f, 0.0f, 0.0f, 1.0f);
        //preCalcResult.Identity();
        const ResBoneAnim* pBoneAnim = GetBoneAnim(idxAnim);
        const char* name = pBoneAnim->GetName();
        int idxTarget = bindTable.GetTargetIndex(idxAnim);
        int idxHost = pHostSkeleton->FindBoneIndex(name);
        if (idxHost >= 0 && idxTarget != AnimBindTable::Flag_NotBound)
        {
            result.SetSuccessBit();

            // ホストからターゲットへの回転・スケールを作る。
            nn::util::Vector3fType host;
            VectorLoad(&host, pHostSkeleton->GetBone(idxHost)->GetTranslate());
            float lenHostSq = VectorLengthSquared(host);
            //float lenHostSq = Vec3::LengthSq(host);
            if (lenHostSq == 0.0f)
            {
                continue;
            }

            nn::util::Vector3fType normalHost;
            //float lenHostInv = Math::RSqrt(lenHostSq);
            float lenHostInv = 1 / std::sqrt(lenHostSq);
            VectorMultiply(&normalHost, host, lenHostInv);
            //normalHost.Mul(host, lenHostInv);
            nn::util::Vector3fType normalTarget;
            VectorLoad(&normalTarget, pTargetSkeleton->GetBone(idxTarget)->GetTranslate());
            float lenSq = VectorLengthSquared(normalTarget);
            if (lenSq == .0f)
            {
                continue;
            }

            float lenTarget = lenSq / std::sqrt(lenSq);
            VectorNormalize(&normalTarget, normalTarget);
            //float lenTarget = normalTarget.Normalize(pTargetSkeleton->GetBone(idxTarget)->GetTranslate());

            float r = 2.0f * (VectorDot(normalHost, normalTarget) + 1.0f); // == (2cos(θ/2))^2
            float scale_sqrt = std::sqrt(lenTarget * lenHostInv);
            //float r = 2.0f * (Vec3::Dot(normalHost, normalTarget) + 1.0f); // == (2cos(θ/2))^2
            //float scale_sqrt = Math::Sqrt(lenTarget * lenHostInv);
            nn::util::Vector3fType axis;
            if (r < FLT_EPSILON)
            {
                // ぴったり反対を向いている場合は、てきとうな軸との外積を回転軸として半回転させる。
                if (VectorGetX(normalHost) > VectorGetY(normalHost))
                    //if (normalHost.x > normalHost.y)
                {
                    VectorSet(&axis, -VectorGetZ(normalHost), 0.0f, VectorGetX(normalHost));
                    //axis.Set(-normalHost.z, 0.0f, normalHost.x);
                }
                else
                {
                    VectorSet(&axis, 0.0f, 0.0f, -VectorGetY(normalHost));
                    //axis.Set(0.0f, VectorGetZ(normalHost), -normalHost.y);
                }
                float scale = nn::util::SinTable(nn::util::RadianToAngleIndex(nn::util::FloatPi * 0.5f)) * scale_sqrt / VectorLengthSquared(axis);
                //float scale = Math::Sin(Math::Pi() * 0.5f) * scale_sqrt * Math::RSqrt(Vec3::LengthSq(axis));
                VectorSet(&preCalcResult,
                    VectorGetX(axis) * scale,
                    VectorGetY(axis) * scale,
                    VectorGetZ(axis) * scale,
                    nn::util::CosTable(nn::util::RadianToAngleIndex(nn::util::FloatPi * 0.5f)) * scale_sqrt);
                //preCalcResult.Set(axis.x * scale, axis.y * scale, axis.z * scale,
                    //Math::Cos(Math::Pi() * 0.5f) * scale_sqrt);
            }
            else
            {
                float r_sqrt = nn::util::Rsqrt(r);
                float scale = scale_sqrt * r_sqrt;
                VectorCross(&axis, normalHost, normalTarget);
                VectorSet(&preCalcResult,
                    VectorGetX(axis) * scale,
                    VectorGetY(axis) * scale,
                    VectorGetZ(axis) * scale,
                    scale_sqrt * 0.5f * r * r_sqrt);
                //axis.Cross(normalHost, normalTarget);
                //preCalcResult.Set(axis.x * scale, axis.y * scale, axis.z * scale, scale_sqrt * 0.5f * r * r_sqrt);
            }
        }
        else
        {
            result.SetFailureBit();
        }
    }

    return result;
}

//--------------------------------------------------------------------------------------------------

class SkeletalAnimBlender::Impl
{
public:
    static void (SkeletalAnimBlender::* const s_pFuncBlend[])(SkeletalAnimObj*, float);
    static void (SkeletalAnimBlender::* const s_pFuncBlendResult[])(BoneAnimBlendResult*, const BoneAnimResult&, int, float);
    static void (SkeletalAnimBlender::* const s_pFuncCalculateBoneAnimDiff[])(BoneAnimResult*, const BoneAnimResult&, const BoneAnimResult&);
    static void (SkeletalAnimBlender::* const s_pFuncConvertValidResult[])(BoneAnimResult*, const BoneAnimBlendResult&, const  BoneAnimResult&);
    static void (SkeletalAnimBlender::* const s_pFuncApplyTo[])(SkeletonObj*) const;
    static void (SkeletalAnimBlender::* const s_pFuncConvertResultRotate[])();
};

void (SkeletalAnimBlender::* const SkeletalAnimBlender::Impl::s_pFuncBlend[])(SkeletalAnimObj*, float) =
{
    &SkeletalAnimBlender::BlendImpl<LerpRot<QuatToAxes>, false>,
    &SkeletalAnimBlender::BlendImpl<LerpRot<EulerToAxes>, false>,
    &SkeletalAnimBlender::BlendImpl<SlerpQuat, false>,
    &SkeletalAnimBlender::BlendImpl<SlerpEuler, false>,
    &SkeletalAnimBlender::BlendImpl<LerpRot<QuatToAxes>, true>,
    &SkeletalAnimBlender::BlendImpl<LerpRot<EulerToAxes>, true>,
    &SkeletalAnimBlender::BlendImpl<SlerpQuat, true>,
    &SkeletalAnimBlender::BlendImpl<SlerpEuler, true>,
};

void (SkeletalAnimBlender::* const SkeletalAnimBlender::Impl::s_pFuncCalculateBoneAnimDiff[])(BoneAnimResult*, const BoneAnimResult&, const BoneAnimResult&)
{
    &SkeletalAnimBlender::CalculateBoneAnimDiff<QuatToQuat,QuatToQuat>,
    &SkeletalAnimBlender::CalculateBoneAnimDiff<QuatToQuat,EulerToQuat>,
    &SkeletalAnimBlender::CalculateBoneAnimDiff<EulerToQuat,QuatToQuat>,
    &SkeletalAnimBlender::CalculateBoneAnimDiff<EulerToQuat,EulerToQuat>,
};

void (SkeletalAnimBlender::* const SkeletalAnimBlender::Impl::s_pFuncConvertValidResult[])(BoneAnimResult*, const BoneAnimBlendResult&, const BoneAnimResult&)
{
    &SkeletalAnimBlender::ConvertValidResult<AxesToQuat>,
    &SkeletalAnimBlender::ConvertValidResult<QuatToQuat>,
};

void (SkeletalAnimBlender::* const SkeletalAnimBlender::Impl::s_pFuncBlendResult[])(BoneAnimBlendResult*, const BoneAnimResult&, int, float) =
{
    &SkeletalAnimBlender::BlendResultImpl<LerpRot<QuatToAxes>>,
    &SkeletalAnimBlender::BlendResultImpl<LerpRot<EulerToAxes>>,
    &SkeletalAnimBlender::BlendResultImpl<SlerpQuat>,
    &SkeletalAnimBlender::BlendResultImpl<SlerpEuler>,
};

void (SkeletalAnimBlender::* const SkeletalAnimBlender::Impl::s_pFuncApplyTo[])(SkeletonObj*) const =
{
    &SkeletalAnimBlender::ApplyToImpl<AxesToMtx, BlendMode_Interpolate>,
    &SkeletalAnimBlender::ApplyToImpl<AxesToMtx, BlendMode_Add>,
    &SkeletalAnimBlender::ApplyToImpl<QuatToMtx, BlendMode_Interpolate>,
    &SkeletalAnimBlender::ApplyToImpl<QuatToMtx, BlendMode_Add>
};

void (SkeletalAnimBlender::* const SkeletalAnimBlender::Impl::s_pFuncConvertResultRotate[])() =
{
    &SkeletalAnimBlender::ConvertResultRotate<AxesToQuat>,
    &SkeletalAnimBlender::ConvertResultRotate<QuatToAxes>,
};

void SkeletalAnimBlender::InitializeArgument::CalculateMemorySize() NN_NOEXCEPT
{
    int targetCount = GetMaxBoneCount();

    for (int blockIndex = 0; blockIndex < MemoryBlockIndex_End; ++blockIndex)
    {
        m_MemoryBlock[blockIndex].Initialize();
    }

    m_MemoryBlock[MemoryBlockIndex_ResultBuffer].SetSizeBy<BoneAnimBlendResult>(1, targetCount);

    m_WorkMemory.Initialize();
    for (int blockIndex = 0; blockIndex < MemoryBlockIndex_End; ++blockIndex)
    {
        m_WorkMemory.Append(&m_MemoryBlock[blockIndex]);
    }
}

bool SkeletalAnimBlender::Initialize(const InitializeArgument& arg, void* pBuffer, size_t bufferSize) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(pBuffer != NULL || bufferSize == 0);
    NN_SDK_REQUIRES(IsAligned(pBuffer, Alignment_Buffer));

    if (!arg.IsMemoryCalculated())
    {
        return false;
    }
    if (arg.GetWorkMemorySize() > bufferSize)
    {
        // バッファーが必要なサイズに満たない場合は失敗。
        return false;
    }

    m_pBufferPtr = pBuffer;

    // メンバの初期化。
    m_pResultBuffer = arg.GetBuffer<void>(pBuffer, InitializeArgument::MemoryBlockIndex_ResultBuffer);
    m_MaxBone = static_cast<uint16_t>(arg.GetMaxBoneCount());
    m_BoneCount = m_MaxBone;

    return true;
}

void SkeletalAnimBlender::ClearResult() NN_NOEXCEPT
{
    NN_G3D_PERF_LEVEL1("SkeletalAnimBlender::ClearResult");

    m_Flag |= Flag_Normalized;

    size_t resultSize = nn::util::align_up(sizeof(BoneAnimBlendResult) * m_BoneCount, Alignment_Buffer);
    memset(m_pResultBuffer, 0, resultSize);
}

void SkeletalAnimBlender::Blend(SkeletalAnimObj* pAnimObj, float weight) NN_NOEXCEPT
{
    NN_G3D_PERF_LEVEL1("SkeletalAnimBlender::Blend");
    NN_SDK_REQUIRES_NOT_NULL(pAnimObj);

    if (WeightEqualsZero(weight))
    {
        return;
    }

    m_Flag &= ~Flag_Normalized;

    pAnimObj->Calculate();

    Bit32 rotateMode = pAnimObj->GetResource()->GetRotateMode();
    int funcIndex = (m_Flag & Flag_InterpolateMask) | (rotateMode >> ResBone::Shift_Rot);
    if( m_pCallback != NULL )
    {
        const int callbackFlag = 0x4;
        funcIndex |= callbackFlag;
    }

    (this->*Impl::s_pFuncBlend[funcIndex])(pAnimObj, weight);
}

void SkeletalAnimBlender::Blend(SkeletalAnimObj* pSrcAnimObj, SkeletalAnimObj* pReferenceAnimObj, float weight) NN_NOEXCEPT
{
    NN_G3D_PERF_LEVEL1("SkeletalAnimBlender::Blend");
    NN_SDK_REQUIRES_NOT_NULL(pSrcAnimObj);
    NN_SDK_REQUIRES_NOT_NULL(pReferenceAnimObj);
    NN_SDK_REQUIRES_EQUAL(GetBlendMode(), BlendMode_Add);

    if (WeightEqualsZero(weight))
    {
        return;
    }

    m_Flag &= ~Flag_Normalized;

    pSrcAnimObj->Calculate();
    pReferenceAnimObj->Calculate();

    BlendDiffAnim(pSrcAnimObj, pReferenceAnimObj, weight);
}

void SkeletalAnimBlender::BlendResult(const BoneAnimResult& boneAnimResult, int boneIndex, float weight) NN_NOEXCEPT
{
    NN_G3D_PERF_LEVEL1("SkeletalAnimBlender::BlendResult");

    if (WeightEqualsZero(weight))
    {
        return;
    }

    m_Flag &= ~Flag_Normalized;

    Bit32 rotateMode = (boneAnimResult.flag & ResBone::Mask_Rot);
    int funcIndex = (m_Flag & Flag_InterpolateMask) | (rotateMode >> ResBone::Shift_Rot);

    // ブレンドウェイトの更新処理です。
    float blendWeight = weight;
    if (m_pCallback != NULL)
    {
        ICalculateBlendWeightCallback::CallbackArg callbackArg(boneIndex, blendWeight);
        m_pCallback->Exec(callbackArg);
        blendWeight = callbackArg.GetBlendWeight();
    }

    (this->*Impl::s_pFuncBlendResult[funcIndex])(GetResultArray(), boneAnimResult, boneIndex, blendWeight);
}

void SkeletalAnimBlender::ApplyTo(SkeletonObj* pSkeletonObj) const NN_NOEXCEPT
{
    NN_G3D_PERF_LEVEL1("SkeletalAnimBlender::ApplyTo");
    // NORMALIZED が立っている場合は BLEND_ADD 扱いとして正規化処理をスキップする。
    int funcIndex = (m_Flag & (Flag_BlendMask | Flag_InterpolateMask)) | ((m_Flag & Flag_Normalized) >> 2);
    (this->*Impl:: s_pFuncApplyTo[funcIndex])(pSkeletonObj);
    m_Flag |= Flag_Normalized;
}

void SkeletalAnimBlender::BeginBlend(Bit32 flag) NN_NOEXCEPT
{
    InterpolateMode prevInterpolateMode = GetInterpolateMode();
    bool isInterpolateModeChanged = prevInterpolateMode != static_cast<InterpolateMode>(flag & Flag_InterpolateMask);
    if (isInterpolateModeChanged)
    {
        // 補間モードが変更されていた場合には、累積している回転値の変換を行う
        (this->*Impl::s_pFuncConvertResultRotate[static_cast<int>(prevInterpolateMode) >> Shift_Interpolate])();
    }

    m_Flag = (flag & (Flag_BlendMask | Flag_InterpolateMask));
    m_Flag |= Flag_Blending;
}

void SkeletalAnimBlender::EndBlend() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(m_Flag & Flag_Blending);
    BoneAnimBlendResult* pResultArray = GetResultArrayMutable();
    for (int boneIndex = 0; boneIndex < m_BoneCount; ++boneIndex)
    {
        BoneAnimBlendResult* pResult = &pResultArray[boneIndex];
        float& weight = pResult->weight;
        if (WeightEqualsZero(weight))
        {
            continue;
        }

        if ((GetBlendMode() != BlendMode_Add) && !WeightEqualsOne(weight))
        {
            NormalizeResult(pResult, weight);
        }

        weight = 1.0f;
    }

    m_Flag |= Flag_Normalized;
    m_Flag ^= Flag_Blending;
}

//--------------------------------------------------------------------------------------------------

template <typename BlendRot, bool useCallback>
void SkeletalAnimBlender::BlendImpl(SkeletalAnimObj* pAnimObj, float weight)
{
    const BoneAnimResult* pResultArray = pAnimObj->GetResultArray();
    BoneAnimBlendResult* pTargetArray = GetResultArray();
    const AnimBindTable& bindTable = pAnimObj->GetBindTable();

    for (int idxAnim = 0, animCount = bindTable.GetAnimCount(); idxAnim < animCount; ++idxAnim)
    {
        if (!bindTable.IsApplyEnabled(idxAnim))
        {
            continue;
        }

        int idxTarget = bindTable.GetTargetIndex(idxAnim);

        // ブレンドウェイトの更新処理です。
        float blendWeight = weight;
        if (NN_STATIC_CONDITION(useCallback))
        {
            ICalculateBlendWeightCallback::CallbackArg callbackArg(pAnimObj, idxTarget, blendWeight);
            m_pCallback->Exec(callbackArg);
            blendWeight = callbackArg.GetBlendWeight();
        }
        BlendResultImpl<BlendRot>(pTargetArray, pResultArray[idxAnim], idxTarget, blendWeight);
    }
}

void SkeletalAnimBlender::BlendDiffAnim(SkeletalAnimObj* pSrcAnimObj, SkeletalAnimObj* pReferenceAnimObj, float weight) NN_NOEXCEPT
{
    NN_SDK_ASSERT_EQUAL(GetBlendMode(), BlendMode_Add);
    NN_SDK_ASSERT_NOT_NULL(pSrcAnimObj);
    NN_SDK_ASSERT_NOT_NULL(pReferenceAnimObj);

    BoneAnimResult result;

    const AnimBindTable& bindTable = pSrcAnimObj->GetBindTable();
    const AnimBindTable& referenceBindTable = pReferenceAnimObj->GetBindTable();
    const BoneAnimResult* pTargetResult = pSrcAnimObj->GetResultArray();
    const BoneAnimResult* pReferenceResult = pReferenceAnimObj->GetResultArray();

    int targetRotateMode = (pSrcAnimObj->GetResource()->GetRotateMode() & ResBone::Mask_Rot) >> ResSkeletalAnim::Shift_Rot;
    int referenceRotateMode = (pReferenceAnimObj->GetResource()->GetRotateMode() & ResBone::Mask_Rot) >> ResSkeletalAnim::Shift_Rot;
    int diffFuncIndex = (targetRotateMode << 1) | referenceRotateMode;

    int convertFuncIndex = (m_Flag & Flag_InterpolateMask) >> Shift_Interpolate;
    BoneAnimBlendResult* blendResultArray = GetResultArray();

    for (int boneAnimIndex = 0, boneAnimCount = bindTable.GetAnimCount(); boneAnimIndex < boneAnimCount; ++boneAnimIndex)
    {
        if (!bindTable.IsApplyEnabled(boneAnimIndex))
        {
            continue;
        }

        int boneIndex = bindTable.GetTargetIndex(boneAnimIndex);

        BoneAnimBlendResult& blendResult = blendResultArray[boneIndex];

        // 加算対象が存在しないので BlendResult() でバインドポーズを累積値に埋めておく。
        // t * (1 - w) + td * w を実現するためには、t が必要
        if (WeightEqualsZero(blendResult.weight))
        {
            const ResSkeleton* pResSkeleton = pSrcAnimObj->GetResSkeleton();
            NN_SDK_ASSERT_NOT_NULL(pResSkeleton);

            const nn::g3d::ResBoneData& resBoneData = pResSkeleton->GetBone(boneIndex)->ToData();

            BoneAnimResult bindPose;
            {
                bindPose.flag = resBoneData.flag;
                bindPose.quat = resBoneData.rotate.quat; // quat と euler は共用体なのでサイズが大きい変数を選択
                bindPose.scale = resBoneData.scale;
                bindPose.translate = resBoneData.translate;
            }
            BlendResult(bindPose, boneIndex, 1.0f);
            blendResult.weight = .0f;
        }

        BoneAnimResult referenceBoneAnimResult;
        int referenceAnimIndex = referenceBindTable.GetAnimIndex(boneIndex);
        {
            // 基準となるアニメーションに値が 存在しない/有効でない 場合には、バインドポーズからの差分となる。
            if (!referenceBindTable.IsEnabled(referenceAnimIndex))
            {
                const ResSkeleton* pResSkeleton = pSrcAnimObj->GetResSkeleton();
                NN_SDK_ASSERT_NOT_NULL(pResSkeleton);

                const nn::g3d::ResBoneData& resBoneData = pResSkeleton->GetBone(boneIndex)->ToData();
                referenceBoneAnimResult.flag = resBoneData.flag;
                referenceBoneAnimResult.quat = resBoneData.rotate.quat; // quat と euler は共用体なのでサイズが大きい変数を選択
                referenceBoneAnimResult.scale = resBoneData.scale;
                referenceBoneAnimResult.translate = resBoneData.translate;
            }
            else
            {
                referenceBoneAnimResult = pReferenceResult[referenceAnimIndex];
            }
        }

        // リザルトの回転値にはどのバリエーションでもクォータニオンで格納する
        (this->*Impl::s_pFuncCalculateBoneAnimDiff[diffFuncIndex])(&result, pTargetResult[boneAnimIndex], referenceBoneAnimResult);

        // ブレンド方式に応じて、差分を有効な値に変換
        (this->*Impl::s_pFuncConvertValidResult[convertFuncIndex])(&result, blendResult, result);

        // 以下の補間式が成り立つようにウェイトや累積値を加工しブレンド
        // blendResult * (1.0 - weight) + (blendResult * Diff) * weight
        if (IsSlerpEnabled())
        {
            float prevWeight = blendResult.weight;
            blendResult.weight = 1.0f - weight;

            // ブレンド
            BlendResult(result, boneIndex, weight);

            blendResult.weight = prevWeight + weight;
        }
        else
        {
            nn::util::VectorMultiply(&blendResult.axes[0], blendResult.axes[0], 1.0f - weight);
            nn::util::VectorMultiply(&blendResult.axes[1], blendResult.axes[1], 1.0f - weight);

            // ブレンド
            BlendResult(result, boneIndex, weight);
        }
    }
}

template<typename TargetConvRot, typename ReferenceConvRot>
void SkeletalAnimBlender::CalculateBoneAnimDiff(BoneAnimResult* pDiffAnim, const BoneAnimResult& targetBone, const BoneAnimResult& referenceBone)
{
    // scale
    {
        nn::util::Vector3fType scale, invReferenceScale;
        nn::util::VectorLoad(&invReferenceScale, referenceBone.scale);
        NN_SDK_REQUIRES(!nn::util::VectorIsZero(invReferenceScale));

        nn::util::VectorSetX(&invReferenceScale, 1.0f / referenceBone.scale.x);
        nn::util::VectorSetY(&invReferenceScale, 1.0f / referenceBone.scale.y);
        nn::util::VectorSetZ(&invReferenceScale, 1.0f / referenceBone.scale.z);

        nn::util::VectorLoad(&scale, targetBone.scale);

        nn::util::VectorMultiply(&scale, scale, invReferenceScale);
        nn::util::VectorStore(&pDiffAnim->scale, scale);
    }

    // rotate
    {
        nn::util::Vector4fType targetQuaternion, referenceQuaternion;

        TargetConvRot::Convert(&targetQuaternion, &targetBone);
        ReferenceConvRot::Convert(&referenceQuaternion, &referenceBone);
        nn::util::QuaternionDivide(&targetQuaternion, targetQuaternion, referenceQuaternion);
        nn::util::VectorStore(&pDiffAnim->quat, targetQuaternion);

        pDiffAnim->flag = targetBone.flag;
        pDiffAnim->flag &= ~ResBone::Mask_Rot;
        pDiffAnim->flag |= (ResBone::RotateMode_Quat << ResBone::Shift_Rot) & ResBone::Mask_Rot;
    }

    // translate
    {
        nn::util::Vector3fType translate, referenceTranslate;
        nn::util::VectorLoad(&translate, targetBone.translate);
        nn::util::VectorLoad(&referenceTranslate, referenceBone.translate);
        nn::util::VectorSubtract(&translate, translate, referenceTranslate);
        nn::util::VectorStore(&pDiffAnim->translate, translate);
    }
}

template<typename BlendRot>
void SkeletalAnimBlender::ConvertValidResult(BoneAnimResult* pConvertedResult, const BoneAnimBlendResult& blendResult, const  BoneAnimResult& diffResult)
{
    NN_SDK_ASSERT_EQUAL(GetBlendMode(), BlendMode_Add);

    nn::util::Vector4fType quaternion = NN_UTIL_VECTOR_4F_INITIALIZER(.0f, .0f, .0f, 1.0f);
    bool isConverted = BlendRot::Convert(&quaternion, &blendResult);
    if (!isConverted)
    {
        NN_G3D_WARNING(isConverted, "Faild to convert blended rotate.");
    }

    nn::util::Vector4fType diffQuaternion;
    nn::util::VectorLoad(&diffQuaternion, diffResult.quat);
    nn::util::QuaternionMultiply(&quaternion, quaternion, diffQuaternion);

    nn::util::VectorStore(&pConvertedResult->quat, quaternion);
}

template<typename BlendRot>
void SkeletalAnimBlender::ConvertResultRotate()
{
    BoneAnimBlendResult* pResultArray = GetResultArrayMutable();
    for (int boneIndex = 0; boneIndex < m_BoneCount; ++boneIndex)
    {
        BoneAnimBlendResult* pResult = &pResultArray[boneIndex];
        float weight = pResult->weight;
        if (WeightEqualsZero(weight))
        {
            continue;
        }

        bool isConverted = BlendRot::Convert(pResult, pResult);
        NN_G3D_WARNING(isConverted, "Faild to convert blended rotate.(boneIndex=%d)", boneIndex);
        NN_UNUSED(isConverted);
    }
}

template <typename BlendRot>
void SkeletalAnimBlender::BlendResultImpl(BoneAnimBlendResult* pTargetArray, const BoneAnimResult& boneResult, int boneIndex, float weight)
{
    BoneAnimBlendResult* pTarget = &pTargetArray[boneIndex];
    BlendRot::Blend(pTarget, &boneResult, weight);

    // rotate で pTarget->weight を使用するので注意。
    if (GetBlendMode() == BlendMode_Interpolate)
    {
        InterpolateST::BlendAndFlag(pTarget, &boneResult, weight);
    }
    else
    {
        AdditiveST::BlendAndFlag(pTarget, &boneResult, weight);
    }
}

template <typename ConvRot, SkeletalAnimBlender::BlendMode mode>
void SkeletalAnimBlender::ApplyToImpl(SkeletonObj* pSkeletonObj) const
{
    NN_SDK_REQUIRES_NOT_NULL(pSkeletonObj);
    BoneAnimBlendResult* pResultArray = GetResultArrayMutable();
    LocalMtx* pLocalMtxArray = pSkeletonObj->GetLocalMtxArray();
    for (int idxBone = 0; idxBone < m_BoneCount; ++idxBone)
    {
        BoneAnimBlendResult* pResult = &pResultArray[idxBone];
        float weight = pResult->weight;
        if (WeightEqualsZero(weight))
        {
            continue;
        }

        if (NN_STATIC_CONDITION(mode != BlendMode_Add) && !WeightEqualsOne(weight))
        {
            NormalizeResult(pResult, weight);
        }

        LocalMtx& local = pLocalMtxArray[idxBone];
        CopySTAndFlag(local, ~pResult->flag, pResult->scale, pResult->translate);
        bool isNormalized = ConvRot::Convert(&local.mtxRT, pResult);
        NN_G3D_WARNING(isNormalized, "Failed to normalize blended matrix at \"%hs(index=%d)\".", pSkeletonObj->GetBoneName(idxBone), idxBone);
        NN_UNUSED(isNormalized);
    }
}

}} // namespace nn::g3d

NN_PRAGMA_POP_WARNINGS
