﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/g3d/g3d_ShaderUtility.h>
#include <nn/g3d/g3d_ResShader.h>
#include <nn/g3d/g3d_ResMaterial.h>
#include <nn/g3d/g3d_ResShape.h>
#include <nn/g3d/g3d_MaterialObj.h>
#include <nn/g3d/g3d_ShaderObj.h>
#include <algorithm>

namespace nn { namespace g3d {

void ShaderUtility::InitializeShaderKey(Bit32* pKey, int keyLength,
    const ResShadingModel* pResShadingModel, const ResShaderAssign* pResShaderAssign,
    bool materialPass) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(pResShadingModel != NULL);
    NN_G3D_REQUIRES(pResShaderAssign != NULL, NN_G3D_RES_GET_NAME(pResShadingModel, GetName()));
    NN_SDK_REQUIRES(keyLength >= pResShadingModel->GetKeyLength());
    // Release 版で未使用
    NN_UNUSED( keyLength );
    NN_UNUSED( materialPass );

    // シェーダーキーをまずデフォルトで初期化します。
    int staticKeyLength = pResShadingModel->GetStaticKeyLength();
    pResShadingModel->WriteDefaultStaticKey(pKey);
    pResShadingModel->WriteDefaultDynamicKey(pKey + staticKeyLength);

    // モデルに記録された静的なシェーダーオプションで上書きします。
    for (int idxOption = 0, optionCount = pResShaderAssign->GetShaderOptionCount();
        idxOption < optionCount; ++idxOption)
    {
        const char* optionName = pResShaderAssign->GetShaderOptionId(idxOption);
        if (const ResShaderOption* pOption = pResShadingModel->FindStaticOption(optionName))
        {
            const char* optionValue = pResShaderAssign->GetShaderOption(idxOption);
            int choice = pOption->FindChoiceIndex(optionValue);
            if (choice < 0)
            {
                // モデルが持っているオプションの値がシェーダーに存在しないので、
                // モデルが最新のシェーダーと関連付けられていない可能性があります。
                NN_G3D_WARNING(!materialPass, "Unknown choice. (%s)", optionValue);
                continue;
            }
            pOption->WriteStaticKey(pKey, choice);
        }
        else
        {
            // モデルが持っているオプションの名前がシェーダーに存在しません。
            // モデルが最新のシェーダーと関連付けられていない可能性があります。
            NN_G3D_WARNING(!materialPass, "Unknown option. (%s)", optionName);
        }
    }
}

void ShaderUtility::InitializeShaderKey(
    ShadingModelObj* pShadingModelObj, const ResShaderAssign* pResShaderAssign,
    bool materialPass /*= true*/) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(pShadingModelObj != NULL);
    NN_G3D_REQUIRES(pResShaderAssign != NULL, NN_G3D_RES_GET_NAME(pShadingModelObj->GetResource(), GetName()));
    // Release 版で未使用
    NN_UNUSED( materialPass );

    // シェーダーキーをまずデフォルトで初期化します。
    pShadingModelObj->ClearStaticKey();

    // モデルに記録された静的なシェーダーオプションで上書きします。
    for (int idxAssignOption = 0, assignOptionCount = pResShaderAssign->GetShaderOptionCount();
        idxAssignOption < assignOptionCount; ++idxAssignOption)
    {
        const char* optionName = pResShaderAssign->GetShaderOptionId(idxAssignOption);
        int idxOption = pShadingModelObj->FindStaticOptionIndex(optionName);
        if (idxOption >= 0)
        {
            const ResShaderOption* pOption = pShadingModelObj->GetStaticOption(idxOption);
            const char* optionValue = pResShaderAssign->GetShaderOption(idxAssignOption);
            int choice = pOption->FindChoiceIndex(optionValue);
            if (choice < 0)
            {
                // モデルが持っているオプションの値がシェーダーに存在しないので、
                // モデルが最新のシェーダーと関連付けられていない可能性があります。
                NN_G3D_WARNING(!materialPass, "Unknown choice. (%s)", optionValue);
                continue;
            }
            pShadingModelObj->WriteStaticKey(idxOption, choice);
        }
        else
        {
            // モデルが持っているオプションの名前がシェーダーに存在しません。
            // モデルが最新のシェーダーと関連付けられていない可能性があります。
            NN_G3D_WARNING(!materialPass, "Unknown option. (%s)", optionName);
        }
    }
}

void ShaderUtility::BindShaderParam(
    ResMaterial* pResMaterial, const ResShadingModel* pResShadingModel) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(pResMaterial != NULL);
    NN_G3D_REQUIRES(pResShadingModel != NULL, NN_G3D_RES_GET_NAME(pResMaterial, GetName()));
    int idxBlock = pResShadingModel->GetSystemBlockIndex(ResUniformBlockVar::Type_Material);
    if (idxBlock == -1)
    {
        pResMaterial->SetRawParamSize(0);
        return;
    }

    const ResUniformBlockVar* pBlockVar = pResShadingModel->GetUniformBlock(idxBlock);
    pResMaterial->SetRawParamSize(pBlockVar->GetSize()); // UniformBlock 全体のサイズを設定します。
    for (int idxParam = 0, paramCount = pResMaterial->GetShaderParamCount();
        idxParam < paramCount; ++idxParam)
    {
        // シェーダーパラメータの id でシェーダーからルックアップし、オフセットを初期化します。
        ResShaderParam* pResShaderParam = pResMaterial->GetShaderParam(idxParam);
        const nn::g3d::ResUniformVar* pUniformVar = pBlockVar->FindUniform(pResShaderParam->GetId());
        pResShaderParam->SetOffset(pUniformVar ? pUniformVar->GetOffset() : -1);
    }
}

void ShaderUtility::InitializeShaderParam(
    MaterialObj* pMaterialObj, const ResShadingModel* pResShadingModel) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(pMaterialObj != NULL);
    NN_G3D_REQUIRES(pResShadingModel != NULL, NN_G3D_RES_GET_NAME(pMaterialObj->GetResource(), GetName()));
    NN_G3D_REQUIRES(pMaterialObj->IsBlockBufferValid(), NN_G3D_RES_GET_NAME(pMaterialObj->GetResource(), GetName()));

    int idxBlock = pResShadingModel->GetSystemBlockIndex(ResUniformBlockVar::Type_Material);
    if (idxBlock == -1)
    {
        return;
    }

    const ResUniformBlockVar* pBlock = pResShadingModel->GetUniformBlock(idxBlock);
    NN_G3D_ASSERT(pMaterialObj->GetMaterialBlockSize() >= pBlock->GetSize(), NN_G3D_RES_GET_NAME(pResShadingModel, GetName()));

    for (int idxBuffer = 0, bufferCount = pMaterialObj->GetBufferingCount();
        idxBuffer < bufferCount; ++idxBuffer)
    {
        void* pData = pMaterialObj->GetMaterialBlock(idxBuffer)->Map();
        memcpy(pData, pBlock->GetDefault(), pBlock->GetSize());
        pMaterialObj->GetMaterialBlock(idxBuffer)->FlushMappedRange(0, pMaterialObj->GetMaterialBlockSize());
        pMaterialObj->GetMaterialBlock(idxBuffer)->Unmap();
    }
}

}} // namespace nn::g3d
