﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/g3d/g3d_MaterialAnimObj.h>
#include <algorithm>
#include <nn/g3d/detail/g3d_Perf.h>
#include <nn/g3d/g3d_ModelObj.h>

namespace nn { namespace g3d {

void MaterialAnimObj::InitializeArgument::CalculateMemorySize() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsValid());

    int bindCount = std::max(GetMaxMaterialCount(), GetMaxMaterialAnimCount());
    int curveCount = GetMaxCurveCount();
    int paramAnimCount = GetMaxParamAnimCount();

    for (int blockIndex = 0; blockIndex < MemoryBlockIndex_End; ++blockIndex)
    {
        m_MemoryBlock[blockIndex].Initialize();
    }

    // momeryblockのアライメントのデフォルトは8
    m_MemoryBlock[MemoryBlockIndex_ResultBuffer].SetSizeBy<Bit32>(1,  curveCount);
    m_MemoryBlock[MemoryBlockIndex_BindTable].SetSizeBy<Bit32>(1, bindCount);
    m_MemoryBlock[MemoryBlockIndex_SubBindTable].SetSizeBy<uint16_t>(4, paramAnimCount);
    size_t size = IsContextEnabled() ? nn::util::align_up(sizeof(AnimFrameCache) * curveCount, Alignment_Default) : 0;
    m_MemoryBlock[MemoryBlockIndex_FrameCacheArray].SetSize(size);
    m_MemoryBlock[MemoryBlockIndex_TextureTable].SetSizeBy<nn::gfx::TextureView*>(Alignment_Default, GetMaxTextureCount());
    m_MemoryBlock[MemoryBlockIndex_TextureDescriptorSlotTable].SetSizeBy<nn::gfx::DescriptorSlot>(Alignment_Default, GetMaxTextureCount());

    m_WorkMemory.Initialize();
    for (int blockIndex = 0; blockIndex < MemoryBlockIndex_End; ++blockIndex)
    {
        m_WorkMemory.Append(&m_MemoryBlock[blockIndex]);
    }
}

bool MaterialAnimObj::Initialize(const InitializeArgument& arg, void* pBuffer, size_t bufferSize) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(pBuffer != NULL || bufferSize == 0);
    NN_SDK_REQUIRES(IsAligned(pBuffer, Alignment_Buffer));

    if (!arg.IsMemoryCalculated() || !arg.IsValid())
    {
        return false;
    }
    if (arg.GetWorkMemorySize() > bufferSize)
    {
        // バッファーが必要なサイズに満たない場合は失敗。
        return false;
    }

    int bindCount = std::max(arg.GetMaxMaterialCount(), arg.GetMaxMaterialAnimCount());
    int curveCount = arg.GetMaxCurveCount();

    // メンバの初期化。
    SetBufferPtr(pBuffer);
    // フレーム関係はリソース設定時に初期化
    m_pRes = NULL;
    GetBindTable().Initialize(arg.GetBuffer<Bit32>(pBuffer, InitializeArgument::MemoryBlockIndex_BindTable), bindCount);
    GetContext().Initialize(arg.GetBuffer<AnimFrameCache>(pBuffer, InitializeArgument::MemoryBlockIndex_FrameCacheArray), curveCount);
    SetResultBuffer(arg.GetBuffer(pBuffer, InitializeArgument::MemoryBlockIndex_ResultBuffer));
    m_MaxMaterialAnimCount = arg.GetMaxMaterialAnimCount();
    m_MaxSubBindCount      = arg.GetMaxParamAnimCount();
    m_MaxTextureCount      = arg.GetMaxTextureCount();
    m_MaxCurveCount        = arg.GetMaxCurveCount();
    m_pSubBindIndexArray   = arg.GetBuffer<uint16_t>(pBuffer, InitializeArgument::MemoryBlockIndex_SubBindTable);
    m_ppTextureArray       = arg.GetBuffer<const nn::gfx::TextureView*>(pBuffer, InitializeArgument::MemoryBlockIndex_TextureTable);
    m_pTextureDescriptorSlotArray = arg.GetBuffer<nn::gfx::DescriptorSlot>(pBuffer, InitializeArgument::MemoryBlockIndex_TextureDescriptorSlotTable);

    return true;
}

void MaterialAnimObj::SetResource(const ResMaterialAnim* pRes) NN_NOEXCEPT
{
    NN_G3D_REQUIRES(IsAcceptable(pRes) == true, NN_G3D_RES_GET_NAME(pRes, GetName()));

    m_pRes = pRes;
    m_pMatAnimArray = pRes->ToData().pPerMaterialAnimArray.Get();

    SetTargetUnbound();

    bool loop = (pRes->ToData().flag & AnimFlag_PlayPolicyLoop) != 0;
    ResetFrameCtrl(pRes->GetFrameCount(), loop);
    GetBindTable().SetAnimCount(pRes->GetPerMaterialAnimCount());
    GetContext().SetCurveCount(pRes->GetCurveCount());

    ClearTexture();
}

BindResult MaterialAnimObj::Bind(const ResModel* pModel) NN_NOEXCEPT
{
    NN_G3D_REQUIRES(pModel               != NULL, NN_G3D_RES_GET_NAME(m_pRes, GetName()));
    NN_G3D_REQUIRES(IsAcceptable(pModel) == true, NN_G3D_RES_GET_NAME(m_pRes, GetName()));

    BindResult result;
    AnimBindTable& bindTable = GetBindTable();
    bindTable.ClearAll(pModel->GetMaterialCount());

    int subBindTableIndex = 0;
    for (int idxAnim = 0, animCount = bindTable.GetAnimCount(); idxAnim < animCount; ++idxAnim)
    {
        const ResPerMaterialAnim* pMatAnim = GetPerMaterialAnim(idxAnim);
        const char* pName = pMatAnim->ToData().pName.Get()->GetData();
        int idxTarget = pModel->FindMaterialIndex(pName);
        if (idxTarget >= 0)
        {
            // ターゲットのマテリアルが存在し、ビジビリティアニメがあるなら、成功ビットを立てておく。
            // ビジビリティアニメに関しては、バインド出来たと言えるので。
            if (pMatAnim->IsVisibilityAnimEnabled())
            {
                result.SetSuccessBit();
            }

            const ResMaterial* pMaterial = pModel->GetMaterial(idxTarget);
            BindResult subResult = SubBind(pMatAnim, pMaterial, subBindTableIndex);
            subBindTableIndex += (pMatAnim->GetShaderParamAnimCount() + pMatAnim->GetTexturePatternAnimCount());
            if (subResult.IsBound() || pMatAnim->IsVisibilityAnimEnabled())
            {
                // 1つも関連付けられなかった場合はバインドしない。
                // ビジビリティアニメーションはSubBindを必要としないため、ビジビリティアニメーションが存在する場合は無条件でバインドする。
                bindTable.Bind(idxAnim, idxTarget);
            }
            result |= subResult;
        }
        else
        {
            result.SetFailureBit();
        }
    }

    SetTargetBound();
    return result;
}

BindResult MaterialAnimObj::Bind(const ModelObj* pModel) NN_NOEXCEPT
{
    NN_G3D_REQUIRES(pModel != NULL, NN_G3D_RES_GET_NAME(m_pRes, GetName()));

    return Bind(pModel->GetResource());
}

void MaterialAnimObj::BindFast(const ResModel* pModel) NN_NOEXCEPT
{
    NN_G3D_REQUIRES(IsAcceptable(pModel) == true,      NN_G3D_RES_GET_NAME(m_pRes, GetName()));
    NN_G3D_REQUIRES(pModel == m_pRes->GetBindModel(), NN_G3D_RES_GET_NAME(m_pRes, GetName()));

    AnimBindTable& bindTable = GetBindTable();
    bindTable.ClearAll(pModel->GetMaterialCount());
    bindTable.BindAll(m_pRes->ToData().pBindIndexArray.Get());

    // MaterialAnim のバインド
    int subBindTableIndex = 0;
    for (int idxAnim = 0, animCount = bindTable.GetAnimCount(); idxAnim < animCount; ++idxAnim)
    {
        int idxTarget = bindTable.GetTargetIndex(idxAnim);
        if (idxTarget < AnimBindTable::Flag_NotBound)
        {
            const ResPerMaterialAnim* pMatAnim = GetPerMaterialAnim(idxAnim);
            BindResult result = SubBindFast(pMatAnim, subBindTableIndex);
            subBindTableIndex += (pMatAnim->GetShaderParamAnimCount() + pMatAnim->GetTexturePatternAnimCount());
            if (result.IsMissed())
            {
                // 1つも関連付けられなかった場合はバインド解除する。
                bindTable.Unbind(idxAnim, idxTarget);
            }
        }
    }

    SetTargetBound();
}

BindResult MaterialAnimObj::SubBind(
    const ResPerMaterialAnim* pMatAnim, const ResMaterial* pMaterial, int beginSubBindTableIndex) NN_NOEXCEPT
{
    BindResult result;
    const ResShaderParamAnimInfo* pShaderParamAnimInfo = pMatAnim->ToData().pShaderParamAnimInfoArray.Get();
    int beginIndex = beginSubBindTableIndex;

    // シェーダーパラメータの処理
    for (int idxParamAnim = 0, paramAnimCount = pMatAnim->GetShaderParamAnimCount();
        idxParamAnim < paramAnimCount; ++idxParamAnim, ++pShaderParamAnimInfo)
    {
        const char* pName = pShaderParamAnimInfo->pName.Get()->GetData();
        int idxTarget = pMaterial->FindShaderParamIndex(pName);
        m_pSubBindIndexArray[beginIndex + idxParamAnim] = static_cast<uint16_t>(idxTarget);
        if (idxTarget >= 0)
        {
            result.SetSuccessBit();
        }
        else
        {
            result.SetFailureBit();
        }
    }

    beginIndex += pMatAnim->GetShaderParamAnimCount();

    // テクスチャーパターンの処理
    const ResTexturePatternAnimInfo* pTexturePatternAnimInfo = pMatAnim->ToData().pTexturePatternAnimInfoArray.Get();
    for (int idxPatternAnim = 0, patternAnimCount = pMatAnim->GetTexturePatternAnimCount();
        idxPatternAnim < patternAnimCount; ++idxPatternAnim, ++pTexturePatternAnimInfo)
    {
        const char* pName = pTexturePatternAnimInfo->pName.Get()->GetData();
        int idxTarget = pMaterial->FindSamplerIndex(pName);
        m_pSubBindIndexArray[beginIndex + idxPatternAnim] = static_cast<uint16_t>(idxTarget);
        if (idxTarget >= 0)
        {
            result.SetSuccessBit();
        }
        else
        {
            result.SetFailureBit();
        }
    }

    return result;
}

BindResult MaterialAnimObj::SubBindFast(const ResPerMaterialAnim* pMatAnim, int beginSubBindTableIndex) NN_NOEXCEPT
{
    BindResult result;
    const ResShaderParamAnimInfo* pShaderParamAnimInfo = pMatAnim->ToData().pShaderParamAnimInfoArray.Get();
    int beginIndex = beginSubBindTableIndex;

    // シェーダーパラメータの処理
    for (int idxParamAnim = 0, paramAnimCount = pMatAnim->GetShaderParamAnimCount();
        idxParamAnim < paramAnimCount; ++idxParamAnim, ++pShaderParamAnimInfo)
    {
        m_pSubBindIndexArray[beginIndex + idxParamAnim] = pShaderParamAnimInfo->subbindIndex;
        if (pShaderParamAnimInfo->subbindIndex != AnimFlag_NotBound)
        {
            result.SetSuccessBit();
        }
        else
        {
            result.SetFailureBit();
        }
    }

    beginIndex += pMatAnim->GetShaderParamAnimCount();

    // テクスチャーパターンの処理
    const ResTexturePatternAnimInfo* pTexturePatternAnimInfo = pMatAnim->ToData().pTexturePatternAnimInfoArray.Get();
    for (int idxPatAnim = 0, patternAnimCount = pMatAnim->GetTexturePatternAnimCount();
        idxPatAnim < patternAnimCount; ++idxPatAnim, ++pTexturePatternAnimInfo)
    {
        m_pSubBindIndexArray[beginIndex + idxPatAnim] = pTexturePatternAnimInfo->subbindIndex;
        if (pTexturePatternAnimInfo->subbindIndex != -1)
        {
            result.SetSuccessBit();
        }
        else
        {
            result.SetFailureBit();
        }
    }

    return result;
}

void MaterialAnimObj::Calculate() NN_NOEXCEPT
{
    NN_G3D_PERF_LEVEL1("MaterialAnimObj::Calculate");
    NN_SDK_REQUIRES_NOT_NULL(m_pRes);
    NN_G3D_REQUIRES(IsTargetBound() == true, NN_G3D_RES_GET_NAME(m_pRes, GetName()));

    if (IsFrameChanged())
    {
        float frame = GetFrameCtrl().GetFrame();
        const AnimBindTable& bindTable = GetBindTable();
        AnimContext& context = GetContext();
        int materialAnimCount = bindTable.GetAnimCount();
        int subBindTableIndex = 0;
        if (context.IsFrameCacheValid())
        {
            for (int idxMatAnim = 0; idxMatAnim < materialAnimCount; ++idxMatAnim)
            {
                const ResPerMaterialAnim* pMatAnim = GetPerMaterialAnim(idxMatAnim);

                if (!bindTable.IsCalculateEnabled(idxMatAnim))
                {
                    continue;
                }

                // シェーダーパラメータの処理
                int beginCurve = pMatAnim->ToData().beginShaderParamCurveIndex;
                if (beginCurve != AnimFlag_NotBound)
                {
                    void* pShaderParamResult = AddOffset(GetResultBuffer(), sizeof(float) * beginCurve);
                    pMatAnim->EvaluateShaderParamAnim<true>(pShaderParamResult,
                                                            frame,
                                                            &m_pSubBindIndexArray[subBindTableIndex],
                                                            context.GetFrameCacheArray(beginCurve));
                }

                subBindTableIndex += pMatAnim->GetShaderParamAnimCount();

                // テクスチャーパターンの処理
                beginCurve = pMatAnim->ToData().beginTexturePatternCurveIndex;
                if (beginCurve != AnimFlag_NotBound)
                {
                    int32_t* pTexturePatternResult = AddOffset<int32_t>(GetResultBuffer(), sizeof(int32_t) * beginCurve);
                    pMatAnim->EvaluateTexturePatternAnim<true>(pTexturePatternResult,
                                                               frame,
                                                               &m_pSubBindIndexArray[subBindTableIndex],
                                                               context.GetFrameCacheArray(beginCurve));
                }

                subBindTableIndex += pMatAnim->GetTexturePatternAnimCount();

                // ビジビリティ処理
                if (pMatAnim->IsVisibilityCurveAnimEnabled())
                {
                    beginCurve = pMatAnim->ToData().beginVisibilityCurveIndex;
                    int32_t* pVisibilityResult = AddOffset<int32_t>(GetResultBuffer(), sizeof(int32_t) * beginCurve);
                    pMatAnim->EvaluateVisibilityAnim<true>(pVisibilityResult, frame, context.GetFrameCacheArray(beginCurve));
                }
            }
        }
        else
        {
            for (int idxMatAnim = 0; idxMatAnim < materialAnimCount; ++idxMatAnim)
            {
                const ResPerMaterialAnim* pMatAnim = GetPerMaterialAnim(idxMatAnim);

                if (!bindTable.IsCalculateEnabled(idxMatAnim))
                {
                    continue;
                }

                // シェーダーパラメータの処理
                int beginCurve = pMatAnim->ToData().beginShaderParamCurveIndex;
                if (beginCurve != AnimFlag_NotBound)
                {
                    void* pShaderParamResult = AddOffset(GetResultBuffer(), sizeof(float) * beginCurve);
                    pMatAnim->EvaluateShaderParamAnim<false>(pShaderParamResult,
                                                             frame,
                                                             &m_pSubBindIndexArray[subBindTableIndex],
                                                             NULL);
                }

                subBindTableIndex += pMatAnim->GetShaderParamAnimCount();

                // テクスチャーパターンの処理
                beginCurve = pMatAnim->ToData().beginTexturePatternCurveIndex;
                if (beginCurve != AnimFlag_NotBound)
                {
                    int32_t* pTexturePatternResult = AddOffset<int32_t>(GetResultBuffer(), sizeof(int32_t) * beginCurve);
                    pMatAnim->EvaluateTexturePatternAnim<false>(pTexturePatternResult,
                                                                frame,
                                                                &m_pSubBindIndexArray[subBindTableIndex],
                                                                NULL);
                }

                subBindTableIndex += pMatAnim->GetTexturePatternAnimCount();

                // ビジビリティ処理
                if (pMatAnim->IsVisibilityCurveAnimEnabled())
                {
                    beginCurve = pMatAnim->ToData().beginVisibilityCurveIndex;
                    int32_t* pVisibilityResult = AddOffset<int32_t>(GetResultBuffer(), sizeof(int32_t) * beginCurve);
                    pMatAnim->EvaluateVisibilityAnim<false>(pVisibilityResult, frame, NULL);
                }
            }
        }

        UpdateLastFrame();
    }
}

void MaterialAnimObj::ApplyTo(ModelObj* pModelObj) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pRes);
    NN_G3D_REQUIRES(IsTargetBound() == true, NN_G3D_RES_GET_NAME(m_pRes, GetName()));
    NN_G3D_REQUIRES(pModelObj != NULL,       NN_G3D_RES_GET_NAME(m_pRes, GetName()));

    const AnimBindTable& bindTable = GetBindTable();
    int subBindTableIndex = 0;
    for (int idxMatAnim = 0, materialAnimCount = bindTable.GetAnimCount();
        idxMatAnim < materialAnimCount; ++idxMatAnim)
    {
        if (!bindTable.IsApplyEnabled(idxMatAnim))
        {
            continue;
        }

        int idxMat = bindTable.GetTargetIndex(idxMatAnim);
        MaterialObj* pMaterial = pModelObj->GetMaterial(idxMat);
        const ResPerMaterialAnim* pMatAnim = GetPerMaterialAnim(idxMatAnim);
        ApplyTo(pMaterial, pMatAnim, subBindTableIndex);
        subBindTableIndex += (pMatAnim->GetShaderParamAnimCount() + pMatAnim->GetTexturePatternAnimCount());

        // ビジビリティの処理。モデルに対して行うのでここで行う。
        if (pMatAnim->IsVisibilityAnimEnabled())
        {
            int32_t visibleValue;
            // カーブ評価の結果を使用
            if (pMatAnim->IsVisibilityCurveAnimEnabled())
            {
                int beginCurve = pMatAnim->ToData().beginVisibilityCurveIndex;
                visibleValue = *AddOffset<int32_t>(GetResultBuffer(), sizeof(int32_t) * beginCurve);
            }
            // 定数値を使用
            else
            {
                const ResAnimConstant* pConstant = pMatAnim->GetConstant(pMatAnim->ToData().visibilityConstantIndex);
                visibleValue = pConstant->GetInt();
            }
            pModelObj->SetMaterialVisible(idxMat, visibleValue ? true : false);
        }
    }
}

void MaterialAnimObj::ApplyTo(MaterialObj* pMaterialObj, const ResPerMaterialAnim* pMatAnim, int beginSubBindTableIndex) const NN_NOEXCEPT
{
    int subBindTableIndex = beginSubBindTableIndex;

    // シェーダーパラメータの処理
    // カーブの評価結果
    int beginCurve = pMatAnim->ToData().beginShaderParamCurveIndex;
    const Bit32* pShaderParamResult = AddOffset<Bit32>(GetResultBuffer(), sizeof(float) * beginCurve);

    // 書き込み先と Param の情報
    const ResShaderParamAnimInfo* pShaderParamInfo = pMatAnim->ToData().pShaderParamAnimInfoArray.Get();
    int paramAnimCount = pMatAnim->GetShaderParamAnimCount();
    for (int idxParamAnim = 0; idxParamAnim < paramAnimCount; ++idxParamAnim, ++pShaderParamInfo, ++subBindTableIndex)
    {
        int subBindIndex = m_pSubBindIndexArray[subBindTableIndex];
        if (subBindIndex == AnimFlag_NotBound)
        {
            continue;
        }

        void* pSrcParam = pMaterialObj->EditShaderParam(subBindIndex);

        // カーブ
        for (int idxCurve = pShaderParamInfo->beginCurve,
            endCurve = idxCurve + pShaderParamInfo->floatCurveCount + pShaderParamInfo->intCurveCount;
            idxCurve < endCurve; ++idxCurve)
        {
            const ResAnimCurve* pCurve = pMatAnim->GetCurve(idxCurve);
            uint32_t targetOffset = pCurve->ToData().targetOffset;
            Bit32* pTarget = AddOffset<Bit32>(pSrcParam, targetOffset);
            *pTarget = pShaderParamResult[idxCurve];
        }

        // 定数
        for (int idxConstant = pShaderParamInfo->beginConstant,
            endConstant = idxConstant + pShaderParamInfo->constantCount;
            idxConstant < endConstant; ++idxConstant)
        {
            const ResAnimConstant* pConstant = pMatAnim->GetConstant(idxConstant);
            uint32_t targetOffset = pConstant->ToData().targetOffset;
            // 単純なコピーなので float も s32 としてコピーする。
            int32_t* pTarget = AddOffset<int32_t>(pSrcParam, targetOffset);
            *pTarget = pConstant->GetInt();
        }
    }

    // テクスチャーパターンの処理
    // カーブの評価結果
    beginCurve = pMatAnim->ToData().beginTexturePatternCurveIndex;
    const int32_t* pTexturePatternAnimResult = AddOffset<int32_t>(GetResultBuffer(), sizeof(int32_t) * beginCurve);

    // 書き込み先と Texture の情報
    const ResTexturePatternAnimInfo* pTexturePatternInfo = pMatAnim->ToData().pTexturePatternAnimInfoArray.Get();
    int patternAnimCount = pMatAnim->GetTexturePatternAnimCount();
    int resultIndex = 0;
    for (int idxPatAnim = 0; idxPatAnim < patternAnimCount; ++idxPatAnim, ++pTexturePatternInfo, ++subBindTableIndex)
    {
        int subBindIndex = m_pSubBindIndexArray[subBindTableIndex];
        if (subBindIndex == AnimFlag_NotBound)
        {
            continue;
        }

        if (pTexturePatternInfo->curveIndex != AnimFlag_NotBound)
        {
            int32_t index = pTexturePatternAnimResult[resultIndex];
            // relative_repeat で範囲外のインデクスを指そうとした場合はクランプします
            if (index < 0)
            {
                index = 0;
            }
            else if (m_pRes->ToData().textureCount <= index)
            {
                index = m_pRes->ToData().textureCount - 1;
            }
            TextureRef textureRef(m_ppTextureArray[index], m_pTextureDescriptorSlotArray[index]);
            pMaterialObj->SetTexture(subBindIndex, textureRef);
            ++resultIndex;
        }
        else
        {
            const ResAnimConstant* pConstant = pMatAnim->GetConstant(pTexturePatternInfo->constantIndex);
            int index = pConstant->GetInt();
            TextureRef textureRef(m_ppTextureArray[index], m_pTextureDescriptorSlotArray[index]);
            pMaterialObj->SetTexture(subBindIndex, textureRef);
        }
    }

    return;
}

void MaterialAnimObj::RevertTo(ModelObj* pModelObj) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pRes);
    NN_G3D_REQUIRES(IsTargetBound(), NN_G3D_RES_GET_NAME(m_pRes, GetName()));
    NN_G3D_REQUIRES(pModelObj != NULL, NN_G3D_RES_GET_NAME(m_pRes, GetName()));

    const AnimBindTable& bindTable = GetBindTable();
    int subBindTableIndex = 0;
    for (int animIndex = 0, animCount = bindTable.GetAnimCount(); animIndex < animCount; ++animIndex)
    {
        // Apply フラグが有効になっていないものは値を書き換えていないとみなしスキップ
        if (!bindTable.IsApplyEnabled(animIndex))
        {
            continue;
        }

        int materialIndex = bindTable.GetTargetIndex(animIndex);
        MaterialObj* pMaterialObj = pModelObj->GetMaterial(materialIndex);
        const ResPerMaterialAnim* pMatAnim = GetPerMaterialAnim(animIndex);

        // シェーダーパラメータとテクスチャパターンのリバート
        RevertTo(pMaterialObj, pMatAnim, subBindTableIndex);
        subBindTableIndex += (pMatAnim->GetShaderParamAnimCount() + pMatAnim->GetTexturePatternAnimCount());

        // マテリアルビジビリティのリバート
        if (pMatAnim->IsVisibilityAnimEnabled())
        {
            const ResMaterial* pResMaterial = pMaterialObj->GetResource();
            bool isVisibled = pResMaterial->ToData().flag & ResMaterial::Flag_Visibility ? true : false;
            pModelObj->SetMaterialVisible(materialIndex, isVisibled);
        }
    }
}

void MaterialAnimObj::RevertTo(MaterialObj* pMaterialObj, const ResPerMaterialAnim* pMatAnim, int beginSubBindTableIndex) const NN_NOEXCEPT
{
    int subBindTableIndex = beginSubBindTableIndex;
    const nn::g3d::ResMaterial* pResMaterial = pMaterialObj->GetResource();

    // 書き込み先と Param の情報
    const ResShaderParamAnimInfo* pShaderParamInfo = pMatAnim->ToData().pShaderParamAnimInfoArray.Get();
    for (int paramAnimIndex = 0, paramAnimCount = pMatAnim->GetShaderParamAnimCount();
        paramAnimIndex < paramAnimCount; ++paramAnimIndex, ++pShaderParamInfo, ++subBindTableIndex)
    {
        int subBindIndex = m_pSubBindIndexArray[subBindTableIndex];
        if (subBindIndex == AnimFlag_NotBound)
        {
            continue;
        }

        void* pSrcParam = pMaterialObj->EditShaderParam(subBindIndex);
        const void* pResSrcParam = pResMaterial->GetSrcParam(subBindIndex);

        // カーブ
        for (int curveIndex = pShaderParamInfo->beginCurve, curveCount = curveIndex + pShaderParamInfo->floatCurveCount + pShaderParamInfo->intCurveCount;
            curveIndex < curveCount; ++curveIndex)
        {
            const ResAnimCurve* pCurve = pMatAnim->GetCurve(curveIndex);
            uint32_t targetOffset = pCurve->ToData().targetOffset;
            Bit32* pTarget = AddOffset<Bit32>(pSrcParam, targetOffset);

            const Bit32* pValue = AddOffset<Bit32>(pResSrcParam, targetOffset);

            *pTarget = *pValue;
        }

        // 定数
        for (int constantIndex = pShaderParamInfo->beginConstant, constantCount = constantIndex + pShaderParamInfo->constantCount;
            constantIndex < constantCount; ++constantIndex)
        {
            const ResAnimConstant* pConstant = pMatAnim->GetConstant(constantIndex);
            uint32_t targetOffset = pConstant->ToData().targetOffset;

            int32_t* pTarget = AddOffset<int32_t>(pSrcParam, targetOffset);
            const int32_t* pValue = AddOffset<int32_t>(pResSrcParam, targetOffset);

            *pTarget = *pValue;
        }
    }

    // テクスチャパターン
    // 書き込み先と Texture の情報
    const ResTexturePatternAnimInfo* pTexturePatternInfo = pMatAnim->ToData().pTexturePatternAnimInfoArray.Get();
    for (int patternAnimIndex = 0, patternAnimCount = pMatAnim->GetTexturePatternAnimCount();
        patternAnimIndex < patternAnimCount; ++patternAnimIndex, ++pTexturePatternInfo, ++subBindTableIndex)
    {
        int subBindIndex = m_pSubBindIndexArray[subBindTableIndex];
        if (subBindIndex == AnimFlag_NotBound)
        {
            continue;
        }

        const nn::gfx::TextureView* pTextureView = pResMaterial->GetTexture(subBindIndex);
        const nn::gfx::DescriptorSlot slot = pResMaterial->GetTextureDescriptorSlot(subBindIndex);
        TextureRef textureRef(pTextureView, slot);
        pMaterialObj->SetTexture(subBindIndex, textureRef);
    }
}

}} // namespace nn::g3d
