﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_SdkLog.h>

#include <nn/font/font_RectDrawer.h>
#include <nn/font/font_DispStringBuffer.h>
#include <nn/font/font_ResourceFormat.h>
#include <nn/perf.h>

//--------------------------------------------------------------------------------------------------
// ミドルウェア情報
#include <nn/nn_Middleware.h>
#include <nn/nn_Version.h>

#define NN_DETAIL_XX_MACRO_TO_STRING(x) NN_DETAIL_XX_MACRO_TO_STRING_(x)
#define NN_DETAIL_XX_MACRO_TO_STRING_(x) #x

#define NW_MIDDLEWARE_SYMBOL(buildOption) "NintendoWare_Font-" NN_DETAIL_XX_MACRO_TO_STRING(NN_NX_ADDON_VERSION_MAJOR) "_" \
NN_DETAIL_XX_MACRO_TO_STRING(NN_NX_ADDON_VERSION_MINOR) "_" NN_DETAIL_XX_MACRO_TO_STRING(NN_NX_ADDON_VERSION_MICRO) "-" #buildOption

namespace {
#if defined(NN_SDK_BUILD_DEBUG)
    NN_DEFINE_MIDDLEWARE(g_MiddlewareInfo, "Nintendo", NW_MIDDLEWARE_SYMBOL(Debug));
#elif defined(NN_SDK_BUILD_DEVELOP)
    NN_DEFINE_MIDDLEWARE(g_MiddlewareInfo, "Nintendo", NW_MIDDLEWARE_SYMBOL(Develop));
#elif defined(NN_SDK_BUILD_RELEASE)
    NN_DEFINE_MIDDLEWARE(g_MiddlewareInfo, "Nintendo", NW_MIDDLEWARE_SYMBOL(Release));
#endif
}
//--------------------------------------------------------------------------------------------------


#if defined(NN_BUILD_CONFIG_OS_WIN)
#ifndef NOMINMAX
#define NOMINMAX
#endif
#include <nn/nn_Windows.h>
#endif

namespace nn
{
namespace font
{
namespace
{
#if defined(NN_BUILD_CONFIG_SPEC_NX)

#if(NN_GFX_IS_TARGET_VK)
    #include "..\..\..\Resources\NnFontShaders\font_BuildinShader_Vk_Ir_Source.h"
#elif (NN_GFX_IS_TARGET_GL)
    #define NN_DETAIL_FONT_INCLUDE_GL_BUILDINSHADER
#elif defined(NN_BUILD_CONFIG_OS_WIN)
    #include "..\..\..\Resources\NnFontShaders\font_BuildinShader_Nvn_Binary_Ir.h"
#elif defined(NN_BUILD_CONFIG_OS_HORIZON)
    #include "..\..\..\Resources\NnFontShaders\font_BuildinShader_Nvn_Binary.h"
#endif

#elif defined(NN_BUILD_CONFIG_OS_WIN)
    #define NN_DETAIL_FONT_INCLUDE_GL_BUILDINSHADER
#else
    static uint8_t FontBuildinShader[1] = { 0 };
#endif

#if defined(NN_DETAIL_FONT_INCLUDE_GL_BUILDINSHADER)
    // 同じヘッダファイルの include を複数書くと cpplint に引っかかるためマクロを立てて回避
    #include "..\..\..\Resources\NnFontShaders\font_BuildinShader_Gl_Source.h"
#endif

    void* RoundUpPointer(void* ptr, size_t alignment)
    {
        return nn::util::BytePtr(ptr).AlignUp(alignment).Get();
    }

    static const uint64_t InvalidDescriptorSlotValue = 0xFFFFFFFFFFFFFFFF;

    void ResetDescriptorSlot_(nn::gfx::DescriptorSlot& slot)
    {
        slot.ToData()->value = InvalidDescriptorSlotValue;
    }

    void SetCommonSampler_(nn::gfx::Sampler::InfoType* pSamplerInfo)
    {
        NN_SDK_ASSERT_NOT_NULL(pSamplerInfo);

        pSamplerInfo->SetDefault();
        pSamplerInfo->SetAddressU(nn::gfx::TextureAddressMode_Repeat);
        pSamplerInfo->SetAddressV(nn::gfx::TextureAddressMode_Repeat);
        pSamplerInfo->SetFilterMode(nn::gfx::FilterMode_MinLinear_MagLinear_MipPoint);
    }
}

//----------------------------------------
size_t RectDrawer::GetWorkBufferAlignment()
{
    nn::util::BinaryFileHeader* pHeader = reinterpret_cast<nn::util::BinaryFileHeader*>(FontBuildinShader);
    return pHeader->GetAlignment();
}

//----------------------------------------
size_t  RectDrawer::GetWorkBufferSize(nn::gfx::Device* pDevice, uint32_t  charCount)
{
    NN_UNUSED(charCount);

    size_t  vertexStateMemorySize = 0;
    {
        nn::gfx::VertexState::InfoType info;
        info.SetDefault();
        nn::gfx::VertexAttributeStateInfo attr;
        {
            attr.SetDefault();
            attr.SetNamePtr("aVertex");
            attr.SetBufferIndex(0);
            attr.SetFormat(nn::gfx::AttributeFormat_32_32_32_Float);
            attr.SetOffset(offsetof(detail::Vertex, xyz));
        }
        nn::gfx::VertexBufferStateInfo buffers[1];
        {
            buffers[0].SetDefault();
            buffers[0].SetStride(sizeof(detail::Vertex));
        }
        info.SetVertexAttributeStateInfoArray(&attr, 1);
        info.SetVertexBufferStateInfoArray(buffers, detail::GetArrayLength(buffers));
        vertexStateMemorySize = nn::gfx::VertexState::GetRequiredMemorySize(info);
    }

    // MemoryPool 前に配置するもののサイズ、ビルトインシェーダーに後続して配置する MemoryPool のための、アラインメント調整を含めたサイズ
    const size_t totalVertexStateMemorySize = nn::util::align_up(vertexStateMemorySize, nn::gfx::VertexState::RequiredMemoryInfo_Alignment) * ShaderVariation_ShaderQuantity;

    nn::gfx::MemoryPoolInfo memPoolInfo;
    memPoolInfo.SetDefault();
    const size_t MemoryPoolAlignment = nn::gfx::MemoryPool::GetPoolMemoryAlignment(pDevice, memPoolInfo);
    const size_t preMemoryPoolSize = nn::util::align_up(sizeof(FontBuildinShader) + totalVertexStateMemorySize, MemoryPoolAlignment);

    const size_t memoryPoolSize = CalculateMemoryPoolSize(pDevice, charCount);

    return preMemoryPoolSize + memoryPoolSize;
}

//----------------------------------------
size_t  RectDrawer::CalculateMemoryPoolSize(nn::gfx::Device* pDevice, uint32_t  charCount)
{
    NN_UNUSED(charCount);

    // アライメントにあわせて配置位置やサイズを調整
    const size_t vertexBufferSize = sizeof(detail::Vertex) * 4;
    const size_t indexBufferSize = sizeof(uint16_t) * IndexCountByLetter;
    const size_t blackWhiteInterBufferSize = sizeof(ShaderParamBlackWhiteInterpolation);

    const size_t  alignedOffsetForVertexBuffer = 0;
    const size_t  alignedOffsetForIndexBuffer = nn::util::align_up(alignedOffsetForVertexBuffer + vertexBufferSize, detail::GetIndexBufferAlignment(pDevice));
    const size_t  alignedOffsetForEnabledBuffer = nn::util::align_up(alignedOffsetForIndexBuffer + indexBufferSize, detail::GetConstantBufferAlignment(pDevice));
    const size_t  alignedOffsetForDisabledBuffer = nn::util::align_up(alignedOffsetForEnabledBuffer + blackWhiteInterBufferSize, detail::GetConstantBufferAlignment(pDevice));

    nn::gfx::MemoryPoolInfo memPoolInfo;
    memPoolInfo.SetDefault();

    const size_t memoryPoolSize = nn::util::align_up(alignedOffsetForDisabledBuffer + blackWhiteInterBufferSize, nn::gfx::MemoryPool::GetPoolMemorySizeGranularity(pDevice, memPoolInfo));

    return memoryPoolSize;
}

//----------------------------------------
size_t RectDrawer::CalculateMemoryPoolAlignment(nn::gfx::Device* pDevice)
{
    // メモリープールで、先頭に配置する頂点バッファのアライメント
    return detail::GetVertexBufferAlignment(pDevice);
}

//----------------------------------------
RectDrawer::RectDrawer()
: m_pResShaderFile(NULL)
, m_CodeType(nn::gfx::ShaderCodeType_Binary)
, m_CharCountMax(0)
, m_WorkMemory(NULL)
{
}

//----------------------------------------
RectDrawer::~RectDrawer()
{
    NN_SDK_ASSERT(!IsInitialized());
    NN_SDK_ASSERT(m_DescriptorSlotForSampler.ToData()->value == InvalidDescriptorSlotValue, "You must call UnregisterSamplerFromDescriptorPool() before destructor.");
}

//----------------------------------------
void RectDrawer::Finalize(nn::gfx::Device* pDevice)
{
    if (m_CharCountMax > 0)
    {
        FinalizeIfNecessary(m_ShaderParamBlackWhiteInterpolationDisabledBuffer, pDevice);
        FinalizeIfNecessary(m_ShaderParamBlackWhiteInterpolationEnabledBuffer, pDevice);
        FinalizeIfNecessary(m_VertexBuffer, pDevice);
        FinalizeIfNecessary(m_IndexBuffer, pDevice);

        FinalizeIfNecessary(m_MemoryPoolForBuffers, pDevice);

        FinalizeIfNecessary(m_Sampler, pDevice);

        for (int idx = 0; idx < ShaderVariation_ShaderQuantity; ++idx)
        {
            FinalizeIfNecessary(m_VertexStates[idx], pDevice);
        }

        nn::gfx::ResShaderContainer* pShaderContainer = m_pResShaderFile->GetShaderContainer();
        NN_SDK_ASSERT_NOT_NULL(pShaderContainer);

        for (int idx = 0; idx < ShaderVariation_ShaderQuantity; ++idx)
        {
            nn::gfx::ResShaderVariation* pVariation = pShaderContainer->GetResShaderVariation(idx);
            pVariation->GetResShaderProgram(m_CodeType)->Finalize(pDevice);
        }

        pShaderContainer->Finalize(pDevice);

        m_CharCountMax = 0;
    }
}

//----------------------------------------
void RectDrawer::RegisterSamplerToDescriptorPool(RegisterSamplerSlot pRegisterSamplerSlot, void* pUserData)
{
    NN_SDK_ASSERT(nn::gfx::IsInitialized(m_Sampler), "You must call RegisterSamplerToDescriptorPool() before you use GetDescriptorSlotForSampler().");
    pRegisterSamplerSlot(&m_DescriptorSlotForSampler, m_Sampler, pUserData);
}

//----------------------------------------
void RectDrawer::UnregisterSamplerFromDescriptorPool(UnregisterSamplerSlot pUnregisterSamplerSlot, void* pUserData)
{
    pUnregisterSamplerSlot(&m_DescriptorSlotForSampler, m_Sampler, pUserData);
    ResetDescriptorSlot_(m_DescriptorSlotForSampler);
}

//----------------------------------------

void RectDrawer::AcquireCommonSamplerSlot(AcquireSamplerSlot pAcquireSamplerSlot, void* pUserData)
{
    NN_SDK_ASSERT(nn::gfx::IsInitialized(m_Sampler), "You must call RegisterSamplerToDescriptorPool() before you use GetDescriptorSlotForSampler().");

    nn::gfx::Sampler::InfoType info;
    SetCommonSampler_(&info);

    pAcquireSamplerSlot(&m_DescriptorSlotForSampler, info, pUserData);
}

//----------------------------------------

void RectDrawer::ReleaseCommonSamplerSlot(ReleaseSamplerSlot pReleaseSamplerSlot, void* pUserData)
{
    nn::gfx::Sampler::InfoType info;
    SetCommonSampler_(&info);

    pReleaseSamplerSlot(&m_DescriptorSlotForSampler, info, pUserData);
    ResetDescriptorSlot_(m_DescriptorSlotForSampler);
}

//----------------------------------------
void RectDrawer::Draw(
    nn::gfx::CommandBuffer& commandBuffer,
    const DispStringBuffer& dispStringBuffer) const
{
    NN_PERF_AUTO_MEASURE_INDEX_NAME(0, "nn::font::RectDrawer::Draw");

    NN_SDK_ASSERT(dispStringBuffer.IsInitialized());
    NN_SDK_ASSERT_NOT_NULL(dispStringBuffer.m_pConstantBuffer);

    if (!dispStringBuffer.IsInitialized())
    {
        return;
    }

    // 使用メモリ計測モード時は描画は行わない。
    if (dispStringBuffer.m_pConstantBuffer->IsAnalyseMode())
    {
        return;
    }

    if (m_CharCountMax < dispStringBuffer.GetCharCount())
    {
        NN_SDK_ASSERT(false, "charCount[%d] is over m_CharCountMax[%d]",
            dispStringBuffer.GetCharCount(), m_CharCountMax);
        return;
    }

    const DispStringBuffer::VertexBufferData& vertexBufferData = dispStringBuffer.m_VertexBufferData;

    nn::gfx::GpuAddress constantBufferGpuAddr;
    nn::gfx::GpuAddress perCharacterParamsGpuAddr;

    constantBufferGpuAddr = *dispStringBuffer.m_pConstantBuffer->GetGpuAddress();
    constantBufferGpuAddr.Offset(dispStringBuffer.m_ConstantBufferOffset);
    perCharacterParamsGpuAddr = *dispStringBuffer.m_pConstantBuffer->GetGpuAddress();
    perCharacterParamsGpuAddr.Offset(dispStringBuffer.m_PerCharacterParamOffset);

    // フチ二回描画設定の場合はコンスタントバッファと描画フラグを変えてもう一度描画する。
    if (dispStringBuffer.m_DoubleDrawnBorder)
    {
        uint32_t flags = dispStringBuffer.m_DrawContentFlags;

        // 最初はフチありで下地を書く。
        flags &= ~ConstantBufferAdditionalContent::Flags_InvisibleBorder;

        AddDrawCommand(
            commandBuffer,
            vertexBufferData,
            flags,
            constantBufferGpuAddr,
            perCharacterParamsGpuAddr);

        // 次に中の文字だけを書く。
        // 二回目描画時のコンスタントバッファは DispStringBuffer::ShaderParam サイズ分ずらしたところにおいてある。
        size_t  constantBufferSize = nn::util::align_up(sizeof(DispStringBuffer::ShaderParam), dispStringBuffer.m_pConstantBuffer->GetBufferAlignment());
        constantBufferGpuAddr.Offset(constantBufferSize);
        flags |= ConstantBufferAdditionalContent::Flags_InvisibleBorder;

        AddDrawCommand(
            commandBuffer,
            vertexBufferData,
            flags,
            constantBufferGpuAddr,
            perCharacterParamsGpuAddr);
    }
    else
    {
        AddDrawCommand(
            commandBuffer,
            vertexBufferData,
            dispStringBuffer.m_DrawContentFlags,
            constantBufferGpuAddr,
            perCharacterParamsGpuAddr);
    }
}

//----------------------------------------
void RectDrawer::AddDrawCommand(
    nn::gfx::CommandBuffer& commandBuffer,
    const DispStringBuffer::VertexBufferData& vertexBufferData,
    const uint32_t contentDrawFlags,
    const nn::gfx::GpuAddress& constantBufferGpuAddr,
    const nn::gfx::GpuAddress& perCharacterParamsGpuAddr) const
{
    ShaderVariation lastShader = ShaderVariation_Invalid;
    uint32_t  textureUseInfoPos = vertexBufferData.textureUseInfoPos;
    const detail::TextureUseInfo* pTextureUseInfos = vertexBufferData.textureUseInfos;

    nn::gfx::GpuAddress perCharacterParamsGpuAddrCurrent = perCharacterParamsGpuAddr;

    for ( uint32_t  p = 0; p < textureUseInfoPos; p++ )
    {
        const detail::TextureUseInfo& info = pTextureUseInfos[p];
        if (info.useCount == 0) { continue; }

        ShaderVariation shaderVariation = ShaderVariation_NormalShader;

        // 文字単位の変換が必要な場合はシェーダーを変更する。
        if (contentDrawFlags & ConstantBufferAdditionalContent::Flags_PerCharacterTransformEnabled)
        {
            shaderVariation = ShaderVariation_NormalShaderWithPerCharacterTransform;
            if (info.bitFlags & detail::BitFlags_BorderEffectEnabled)
            {
                shaderVariation = (contentDrawFlags & ConstantBufferAdditionalContent::Flags_InvisibleBorder)
                    ? ShaderVariation_InvisibleBorderShaderWithPerCharacterTransform
                    : ShaderVariation_BorderShaderWithPerCharacterTransform;
            }
        }
        else
        {
            if (info.bitFlags & detail::BitFlags_BorderEffectEnabled)
            {
                shaderVariation = (contentDrawFlags & ConstantBufferAdditionalContent::Flags_InvisibleBorder)
                    ? ShaderVariation_InvisibleBorderShader
                    : ShaderVariation_BorderShader;
            }
        }

        ShaderVariation shader = shaderVariation;
        if (shader != lastShader)
        {
            //shader->Activate(commandBuffer);
            nn::gfx::ResShaderVariation* pVariation = m_pResShaderFile->GetShaderContainer()->GetResShaderVariation(shader);
            commandBuffer.SetShader(pVariation->GetResShaderProgram(m_CodeType)->GetShader(), nn::gfx::ShaderStageBit_All);
            commandBuffer.SetVertexState(&m_VertexStates[shaderVariation]);

            nn::gfx::GpuAddress vertexBufferAddr;
            m_VertexBuffer.GetGpuAddress(&vertexBufferAddr);
            commandBuffer.SetVertexBuffer(m_VertexShaderSlots[shader], vertexBufferAddr, sizeof(detail::Vertex), sizeof(detail::Vertex) * 4);
            commandBuffer.SetConstantBuffer(m_VertexShaderSlots[shader], nn::gfx::ShaderStage_Vertex, constantBufferGpuAddr, sizeof(DispStringBuffer::ShaderParam));
            commandBuffer.SetConstantBuffer(m_PixelShaderSlots[shader], nn::gfx::ShaderStage_Pixel, constantBufferGpuAddr, sizeof(DispStringBuffer::ShaderParam));

            lastShader = shader;
        }

        // パラメータを可変長配列として扱うため Shader Storage Buffer として設定する。
        size_t  perCharacterParamsSize = sizeof(detail::VertexShaderCharAttribute);
        if (contentDrawFlags & ConstantBufferAdditionalContent::Flags_PerCharacterTransformEnabled)
        {
            perCharacterParamsSize = sizeof(detail::VertexShaderCharAttributeWithTransform);
        }
        perCharacterParamsSize *= info.useCount;
        if (contentDrawFlags & ConstantBufferAdditionalContent::Flags_ShadowEnabled)
        {
            perCharacterParamsSize *= 2;
        }

        commandBuffer.SetUnorderedAccessBuffer(m_VertexShaderPerCharacterParamsSlots[shader], nn::gfx::ShaderStage_Vertex, perCharacterParamsGpuAddrCurrent, perCharacterParamsSize);
        // テクスチャを変えて描画するごとに参照する文字データを位置をずらしていく。
        perCharacterParamsGpuAddrCurrent.Offset(perCharacterParamsSize);


        NN_SDK_ASSERT(info.pTexObj->IsDescriptorSlotForTextureReady(), "You must call ResFont::RegisterTextureViewToDescriptorPool() before you use the font.");
        NN_SDK_ASSERT(m_DescriptorSlotForSampler.ToData()->value != InvalidDescriptorSlotValue, "You must call RegisterSamplerToDescriptorPool() before you use.");

        commandBuffer.SetTextureAndSampler(m_TextureSlots[shader], nn::gfx::ShaderStage_Pixel, info.pTexObj->GetDescriptorSlotForTexture(), m_DescriptorSlotForSampler);

        {
            nn::gfx::GpuAddress gpuAddr;
            if (info.bitFlags & detail::BitFlags_ColorBlackWhiteInterpolationEnabled)
            {
                m_ShaderParamBlackWhiteInterpolationEnabledBuffer.GetGpuAddress(&gpuAddr);
            }
            else
            {
                m_ShaderParamBlackWhiteInterpolationDisabledBuffer.GetGpuAddress(&gpuAddr);
            }
            commandBuffer.SetConstantBuffer(m_BlackWhiteInterpolationSlots[shader], nn::gfx::ShaderStage_Pixel, gpuAddr, sizeof(ShaderParamBlackWhiteInterpolation));
        }

        nn::gfx::GpuAddress indexBufferAddr;
        m_IndexBuffer.GetGpuAddress(&indexBufferAddr);

        // インスタンシング描画
        int instanceCount = info.useCount;
        if (contentDrawFlags & ConstantBufferAdditionalContent::Flags_ShadowEnabled)
        {
            // 影が有効なときは影の分も描画するためインスタンス数を２倍する。
            instanceCount *= 2;
        }
        commandBuffer.DrawIndexed(nn::gfx::PrimitiveTopology_TriangleList, nn::gfx::IndexFormat_Uint16, indexBufferAddr, IndexCountByLetter, 0, instanceCount, 0);
    }
}

//----------------------------------------
void RectDrawer::CreateIndices(uint16_t * indices, uint32_t  charCount)
{
    for( uint32_t  i = 0; i < charCount; i++ )
    {
        uint32_t  indexOffset = i * IndexCountByLetter;
        uint32_t  vertexOffset = i * detail::VertexCountByLetter;
        indices[indexOffset + 0] = static_cast<uint16_t >(vertexOffset + 0);
        indices[indexOffset + 1] = static_cast<uint16_t >(vertexOffset + 2);
        indices[indexOffset + 2] = static_cast<uint16_t >(vertexOffset + 3);
        indices[indexOffset + 3] = static_cast<uint16_t >(vertexOffset + 3);
        indices[indexOffset + 4] = static_cast<uint16_t >(vertexOffset + 1);
        indices[indexOffset + 5] = static_cast<uint16_t >(vertexOffset + 0);
    }
}

//----------------------------------------
bool RectDrawer::Initialize(nn::gfx::Device* pDevice, void* pWorkBuffer, uint32_t charCount, nn::gfx::MemoryPool* pExternalMemoryPool, ptrdiff_t externalMemoryPoolOffset, size_t externalMemoryPoolSize)
{
    NN_USING_MIDDLEWARE(g_MiddlewareInfo);

    NN_SDK_ASSERT(pExternalMemoryPool == NULL || nn::util::is_aligned(externalMemoryPoolOffset, CalculateMemoryPoolAlignment(pDevice)));

    NN_SDK_ASSERT(!IsInitialized());
    if (IsInitialized())
    {
        return false;
    }

    m_WorkMemory = pWorkBuffer;

    // シェーダーリソースのアライメント調整
    {
        NN_SDK_ASSERT(sizeof(FontBuildinShader) / sizeof(FontBuildinShader[0]) > 1);

        // sizeof(FontBuildinShader) + GetShaderAlignment() 分 pWorkBuffer を使います。
        // ただし、GetShaderAlignment() で整列されている想定なので、アライメント調整は起こらないはず。
        const size_t sdrAlign = GetWorkBufferAlignment();
        NN_SDK_ASSERT(nn::util::BytePtr(pWorkBuffer).IsAligned(sdrAlign),
            "RectDrawer::Initialize() pWorkBuffer must be aligned by %d !", sdrAlign);

        const size_t sdrSize = sizeof(FontBuildinShader);
        void* pShader = RoundUpPointer(pWorkBuffer, sdrAlign);
        pWorkBuffer = nn::util::BytePtr(pShader).Advance(sdrSize).Get();

        std::memcpy(pShader, FontBuildinShader, sizeof(FontBuildinShader));

        m_pResShaderFile = nn::gfx::ResShaderFile::ResCast(pShader);
    }

    nn::gfx::ResShaderContainer* pContainer = m_pResShaderFile->GetShaderContainer();
    NN_SDK_ASSERT_NOT_NULL(pContainer);
    pContainer->Initialize(pDevice);

    // バリエーション数がランタイムの想定と一致しているかアサート
    NN_SDK_ASSERT(m_pResShaderFile->GetShaderContainer()->GetShaderVariationCount() == ShaderVariation_ShaderQuantity);

    // 試しにバイナリで初期化してみる
    {
        nn::gfx::ResShaderVariation* pVariation = pContainer->GetResShaderVariation(0);
        NN_SDK_ASSERT_NOT_NULL(pVariation);

        nn::gfx::ResShaderProgram* pBinaryProgram = pVariation->GetResShaderProgram(nn::gfx::ShaderCodeType_Binary);
        m_CodeType = nn::gfx::ShaderCodeType_Binary;

        nn::gfx::ShaderInitializeResult shaderResult = pBinaryProgram != NULL ? pBinaryProgram->Initialize(pDevice) : nn::gfx::ShaderInitializeResult_SetupFailed;
        if (shaderResult != nn::gfx::ShaderInitializeResult_Success)
        {
            // 失敗したら中間表現で初期化する
            nn::gfx::ResShaderProgram* pIrProgram = pVariation->GetResShaderProgram(nn::gfx::ShaderCodeType_Ir);
            if (pIrProgram)
            {
                m_CodeType = nn::gfx::ShaderCodeType_Ir;
                shaderResult = pIrProgram->Initialize(pDevice);
            }
        }
        if (shaderResult != nn::gfx::ShaderInitializeResult_Success)
        {
            // 失敗したらソースで初期化する
            nn::gfx::ResShaderProgram* pSourceProgram = pVariation->GetResShaderProgram(nn::gfx::ShaderCodeType_Source);
            NN_SDK_ASSERT_NOT_NULL(pSourceProgram);
            m_CodeType = nn::gfx::ShaderCodeType_Source;
            shaderResult = pSourceProgram->Initialize(pDevice);
        }
        NN_SDK_ASSERT(shaderResult == nn::gfx::ShaderInitializeResult_Success);
    }

    // 残りのバリエーションを初期化
    for (int i = 1; i < ShaderVariation_ShaderQuantity; i++)
    {
        nn::gfx::ResShaderVariation* pVariation = pContainer->GetResShaderVariation(i);
        NN_SDK_ASSERT_NOT_NULL(pVariation);
        nn::gfx::ShaderInitializeResult shaderResult = pVariation->GetResShaderProgram(m_CodeType)->Initialize(pDevice);
        NN_SDK_ASSERT(shaderResult == nn::gfx::ShaderInitializeResult_Success);
        NN_UNUSED(pVariation);
        NN_UNUSED(shaderResult);
    }

    for (int i = 0; i < ShaderVariation_ShaderQuantity; i++)
    {
        nn::gfx::Shader* pVertexShader = GetVertexShader(i);
        nn::gfx::Shader* pPixelShader = GetPixelShader(i);
        m_VertexShaderSlots[i] = pVertexShader->GetInterfaceSlot(nn::gfx::ShaderStage_Vertex, nn::gfx::ShaderInterfaceType_ConstantBuffer, "ShaderParam");
        m_VertexShaderPerCharacterParamsSlots[i] = pVertexShader->GetInterfaceSlot(nn::gfx::ShaderStage_Vertex, nn::gfx::ShaderInterfaceType_UnorderedAccessBuffer, "PerCharacterParamBlock");
        m_PixelShaderSlots[i] = pPixelShader->GetInterfaceSlot(nn::gfx::ShaderStage_Pixel, nn::gfx::ShaderInterfaceType_ConstantBuffer, "ShaderParam");
        m_BlackWhiteInterpolationSlots[i] = pPixelShader->GetInterfaceSlot(nn::gfx::ShaderStage_Pixel, nn::gfx::ShaderInterfaceType_ConstantBuffer, "ShaderParamBlackWhiteInterpolation");

        m_TextureSlots[i] = pPixelShader->GetInterfaceSlot(nn::gfx::ShaderStage_Pixel, nn::gfx::ShaderInterfaceType_Sampler, "uTextureSrc");
    }

    // 頂点ステート
    {
        nn::gfx::VertexState::InfoType info;
        info.SetDefault();
        nn::gfx::VertexAttributeStateInfo attr;
        {
            attr.SetDefault();
            attr.SetNamePtr("aVertex");
            attr.SetBufferIndex(0);
            attr.SetFormat(nn::gfx::AttributeFormat_32_32_32_Float);
            attr.SetOffset(offsetof(detail::Vertex, xyz));
        }
        nn::gfx::VertexBufferStateInfo buffers[1];
        {
            buffers[0].SetDefault();
            buffers[0].SetStride(sizeof(detail::Vertex));
        }
        info.SetVertexAttributeStateInfoArray(&attr, 1);
        info.SetVertexBufferStateInfoArray(buffers, detail::GetArrayLength(buffers));
        void*   pMemory;
        size_t  memorySize = nn::gfx::VertexState::GetRequiredMemorySize(info);

        for (int i = 0; i < ShaderVariation_ShaderQuantity; i++)
        {
            pMemory = RoundUpPointer(pWorkBuffer, nn::gfx::VertexState::RequiredMemoryInfo_Alignment);
            m_VertexStates[i].SetMemory(pMemory, memorySize);
            m_VertexStates[i].Initialize(pDevice, info, GetVertexShader(i));
            pWorkBuffer = nn::util::BytePtr(pMemory).Advance(memorySize).Get();
        }
    }

    // サンプラー作成、及び設定
    {
        nn::gfx::Sampler::InfoType info;
        SetCommonSampler_(&info);
        m_Sampler.Initialize(pDevice, info);

        ResetDescriptorSlot_(m_DescriptorSlotForSampler);
    }

    // アライメントにあわせて配置位置やサイズを調整
    nn::gfx::MemoryPoolInfo memPoolInfo;
    memPoolInfo.SetDefault();
    const size_t memoryPoolAlignment = nn::gfx::MemoryPool::GetPoolMemoryAlignment(pDevice, memPoolInfo);
    const size_t memorySizeGranularity = nn::gfx::MemoryPool::GetPoolMemorySizeGranularity(pDevice, memPoolInfo);

    const size_t vertexBufferSize = sizeof(detail::Vertex) * 4;
    const size_t indexBufferSize = sizeof(uint16_t) * IndexCountByLetter;
    const size_t blackWhiteInterBufferSize = sizeof(ShaderParamBlackWhiteInterpolation);

    const size_t  alignedOffsetForVertexBuffer = pExternalMemoryPool != NULL ? externalMemoryPoolOffset : 0;
    const size_t  alignedOffsetForIndexBuffer = nn::util::align_up(alignedOffsetForVertexBuffer + vertexBufferSize, detail::GetIndexBufferAlignment(pDevice));
    const size_t  alignedOffsetForEnabledBuffer = nn::util::align_up(alignedOffsetForIndexBuffer + indexBufferSize, detail::GetConstantBufferAlignment(pDevice));
    const size_t  alignedOffsetForDisabledBuffer = nn::util::align_up(alignedOffsetForEnabledBuffer + blackWhiteInterBufferSize, detail::GetConstantBufferAlignment(pDevice));

    const size_t memoryPoolSize = nn::util::align_up(
        alignedOffsetForDisabledBuffer + blackWhiteInterBufferSize - externalMemoryPoolOffset, memorySizeGranularity);

    NN_SDK_ASSERT(pExternalMemoryPool == NULL || (memoryPoolSize <= externalMemoryPoolSize));
    NN_UNUSED(externalMemoryPoolSize);

    // メモリプール作成(外部から指定されなかった場合は作成します)
    nn::gfx::MemoryPool*    pCurrentMemoryPoolForBuffers = pExternalMemoryPool;
    if(pCurrentMemoryPoolForBuffers == NULL)
    {
        void* pPoolMemory = RoundUpPointer(pWorkBuffer, memoryPoolAlignment);
        pWorkBuffer = nn::util::BytePtr(pPoolMemory).Advance(memoryPoolSize).Get();

        memPoolInfo.SetPoolMemory(pPoolMemory, memoryPoolSize);
        memPoolInfo.SetMemoryPoolProperty(nn::gfx::MemoryPoolProperty_CpuUncached | nn::gfx::MemoryPoolProperty_GpuCached);

        m_MemoryPoolForBuffers.Initialize(pDevice, memPoolInfo);

        pCurrentMemoryPoolForBuffers = &m_MemoryPoolForBuffers;
    }

    // 頂点バッファ作成
    {
        nn::gfx::Buffer::InfoType vertexBufferInfo;
        vertexBufferInfo.SetDefault();
        vertexBufferInfo.SetSize(vertexBufferSize);
        vertexBufferInfo.SetGpuAccessFlags(nn::gfx::GpuAccess_VertexBuffer);
        m_VertexBuffer.Initialize(pDevice, vertexBufferInfo, pCurrentMemoryPoolForBuffers, alignedOffsetForVertexBuffer, vertexBufferSize);

        detail::Vertex* pVertex = m_VertexBuffer.Map<detail::Vertex>();
        {
            nn::util::Unorm8x4  color;
            for (uint32_t i = 0; i < 4; ++i)
            {
                color.v[i] = 255;
            }
            pVertex[0].Set(0.0f, 0.0f, 0.0f, color, 0.0f, 1.0f, 0.0f);
            pVertex[1].Set(1.0f, 0.0f, 0.0f, color, 1.0f, 1.0f, 0.0f);
            pVertex[2].Set(0.0f, 1.0f, 0.0f, color, 0.0f, 0.0f, 0.0f);
            pVertex[3].Set(1.0f, 1.0f, 0.0f, color, 1.0f, 0.0f, 0.0f);
        }
        m_VertexBuffer.Unmap();
    }

    // インデックスバッファ作成
    {
        nn::gfx::Buffer::InfoType indexBufferInfo;
        indexBufferInfo.SetDefault();
        indexBufferInfo.SetSize(indexBufferSize);
        indexBufferInfo.SetGpuAccessFlags(nn::gfx::GpuAccess_IndexBuffer);
        m_IndexBuffer.Initialize(pDevice, indexBufferInfo, pCurrentMemoryPoolForBuffers, alignedOffsetForIndexBuffer, indexBufferSize);

        uint16_t* pBuffer = m_IndexBuffer.Map<uint16_t>();
        {
            CreateIndices(pBuffer, 1);
        }
        m_IndexBuffer.Unmap();
    }

    // シェーダパラメータ・白黒補間(有効)
    {
        nn::gfx::Buffer::InfoType info;
        info.SetDefault();
        info.SetSize(blackWhiteInterBufferSize);
        info.SetGpuAccessFlags(nn::gfx::GpuAccess_ConstantBuffer);
        m_ShaderParamBlackWhiteInterpolationEnabledBuffer.Initialize(pDevice, info, pCurrentMemoryPoolForBuffers, alignedOffsetForEnabledBuffer, blackWhiteInterBufferSize);

        ShaderParamBlackWhiteInterpolation* pParam = m_ShaderParamBlackWhiteInterpolationEnabledBuffer.Map<ShaderParamBlackWhiteInterpolation>();
        m_ShaderParamBlackWhiteInterpolationEnabledBuffer.InvalidateMappedRange(0, info.GetSize());
        {
            pParam->flags = pParam->Flags_BlackWhiteInterpolation;
            SwapEndian(pParam->flags);
        }
        m_ShaderParamBlackWhiteInterpolationEnabledBuffer.Unmap();
    }

    // シェーダパラメータ・白黒補間(無効)
    {
        nn::gfx::Buffer::InfoType info;
        info.SetDefault();
        info.SetSize(blackWhiteInterBufferSize);
        info.SetGpuAccessFlags(nn::gfx::GpuAccess_ConstantBuffer);
        m_ShaderParamBlackWhiteInterpolationDisabledBuffer.Initialize(pDevice, info, pCurrentMemoryPoolForBuffers, alignedOffsetForDisabledBuffer, blackWhiteInterBufferSize);

        ShaderParamBlackWhiteInterpolation* pParam = m_ShaderParamBlackWhiteInterpolationDisabledBuffer.Map<ShaderParamBlackWhiteInterpolation>();
        m_ShaderParamBlackWhiteInterpolationDisabledBuffer.InvalidateMappedRange(0, info.GetSize());
        {
            pParam->flags = pParam->Flags_None;
            SwapEndian(pParam->flags);
        }
        m_ShaderParamBlackWhiteInterpolationDisabledBuffer.Unmap();
    }

    m_CharCountMax = charCount;
    return true;
}// NOLINT(impl/function_size)

//----------------------------------------
nn::gfx::Shader* RectDrawer::GetVertexShader(int variation) const
{
    return m_pResShaderFile->GetShaderContainer()->GetResShaderVariation(variation)->GetResShaderProgram(m_CodeType)->GetShader();
}

//----------------------------------------

nn::gfx::Shader* RectDrawer::GetPixelShader(int variation) const
{
    return m_pResShaderFile->GetShaderContainer()->GetResShaderVariation(variation)->GetResShaderProgram(m_CodeType)->GetShader();
}

}   // namespace font
}   // namespace nn

