﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/atk/viewer/detail/util/atk_PatriciaDictionary.h>

#ifdef NN_ATK_CONFIG_ENABLE_DEV

#include <algorithm>

namespace nn {
namespace atk {
namespace viewer {
namespace detail {

namespace {

//----------------------------------------------------------
uint32_t
GetBit(const char* str, size_t len, uint32_t refBit) NN_NOEXCEPT
{
    uint32_t wd = (refBit >> 3);
    uint32_t pos = (refBit & 7);
    // 参照ビットが name の外の場合は0
    return wd < len ? (str[wd] >> pos) & 0x1 : 0;
}

//----------------------------------------------------------
uint32_t
GetBit(const ResName* name, uint32_t refBit) NN_NOEXCEPT
{
    return GetBit(name->GetName(), name->GetLength(), refBit);
}

} // anonymous namespace


//----------------------------------------------------------
PatriciaDictionaryData*
PatriciaDictionary::CreateData(void* buffer, size_t length, int maxNodeCount) NN_NOEXCEPT
{
    if(buffer == NULL || length < sizeof(PatriciaDictionaryData))
    {
        return NULL;
    }

    PatriciaDictionaryData* data = reinterpret_cast<PatriciaDictionaryData*>(buffer);
    data->dataCount = 1;                                    // ルートノード 1つだけ。
    data->size    = static_cast<uint32_t>(GetRequiredSize(maxNodeCount));

    if(data->size > length)
    {
        NN_SDK_ASSERT(false, "out of memory.\n");
        return NULL;
    }

    // 0 番目はルートノードなので、+1 個分初期化する。
    for(auto index = 0; index <= maxNodeCount; ++index)
    {
        PatriciaDictionary::ResetNode(data->node[index]);
    }

    return data;
}

//----------------------------------------------------------
void*
PatriciaDictionary::Find(const char* str) const NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(this);
    NN_SDK_ASSERT_NOT_NULL(str);
    Node* pNode = FindNode(str, std::strlen(str));
    if (pNode)
    {
        return pNode->ofsData.to_ptr();
    }

    return NULL;
}

//----------------------------------------------------------
void*
PatriciaDictionary::Find(const ResName* name) const NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(this);
    NN_SDK_ASSERT_NOT_NULL(name);
    Node* pNode = FindNode(name);
    if (pNode)
    {
        return pNode->ofsData.to_ptr();
    }

    return NULL;
}

//----------------------------------------------------------
void*
PatriciaDictionary::Find(const char* str, size_t len) const NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(this);
    {
        NN_SDK_ASSERT_NOT_NULL(str);
        Node* pNode = FindNode(str, len);
        if (pNode)
        {
            return pNode->ofsData.to_ptr();
        }
    }

    return NULL;
}

//----------------------------------------------------------
int
PatriciaDictionary::FindIndex(const char* str) const NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(this);
    NN_SDK_ASSERT_NOT_NULL(str);
    const Node* pNode = FindNode(str, std::strlen(str));
    if (pNode)
    {
        return static_cast<int>(std::distance(this->node + 1, pNode));
    }

    return -1;
}

//----------------------------------------------------------
int
PatriciaDictionary::FindIndex(const char* str, size_t len) const NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(this);
    NN_SDK_ASSERT_NOT_NULL(str);
    const Node* pNode = FindNode(str, len);
    if (pNode)
    {
        return static_cast<int>(std::distance(this->node + 1, pNode));
    }

    return -1;
}

//----------------------------------------------------------
int
PatriciaDictionary::FindIndex(const ResName* name) const NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(this);
    NN_SDK_ASSERT_NOT_NULL(name);
    const Node* pNode = FindNode(name);
    if (pNode)
    {
        return static_cast<int>(std::distance(this->node + 1, pNode));
    }

    return -1;
}

//----------------------------------------------------------
PatriciaDictionary::BuildResult
PatriciaDictionary::Build() NN_NOEXCEPT
{
    // dataCount と各ノードの ofsName が正しく設定されている状態から辞書を再構築する。
    // size と各ノードの ofsData は変更しない。

    NN_SDK_ASSERT_NOT_NULL(this);

    Node* pNode = this->node;
    Node& root = pNode[0];
    root.idxLeft = root.idxRight = 0;
    root.refBit = ~0u;

    // 長さを取得できるように一時的に長さ0の ResName を設定する。
    // 64ビット版でも利用できるように、スタックではなく PatriciaDictionaryData::size の領域を借りる。
    ResNameData* empty = reinterpret_cast<ResNameData*>(&this->size);
    ResName::LengthType len = empty->len;
    empty->len = 0;
    root.ofsName.set_ptr(empty->str);

    for (auto index = 1, end = this->dataCount; index <= end; ++index)
    {
        Node& current = pNode[index];

        if(current.ofsName.to_ptr() == NULL)
        {
            continue;
        }

        const ResName* name = current.ofsName.GetResName();
        NN_SDK_ASSERT_NOT_NULL(name);
        uint32_t refBit = 0;

        // 1度最後まで辿ってビット比較対象の文字列を得る。
        {
            const Node* parent = &root;
            const Node* child = &pNode[parent->idxLeft];

            NN_SDK_ASSERT_NOT_NULL(child);

            while (parent->refBit > child->refBit)
            {
                // child->refBit目が1なら右のリンクを、0なら左のリンクを辿る。
                // 上のノードに戻ったらループを終了する。
                parent = child;
                child = GetBit(name, child->refBit) ?
                    &pNode[child->idxRight] : &pNode[child->idxLeft];
            }

            refBit = static_cast<uint32_t>(std::max(
                name->GetLength(), child->ofsName.GetResName()->GetLength())) * CHAR_BIT;
            while (GetBit(child->ofsName.GetResName(), refBit) == GetBit(name, refBit))
            {
                // 戻った先のノードのエントリと違うビットを探す
                // refBit ビット目で判断することにする。
                if (refBit == 0)
                {
                    return BuildResult_ErrorNameDuplication; // 名前が重複している場合は失敗。
                }
                --refBit;
            }
            current.refBit = refBit;
        }

        // 挿入
        {
            Node* parent = &root;
            Node* child = &pNode[parent->idxLeft];

            while (parent->refBit > child->refBit && child->refBit > refBit)
            {
                parent = child;
                child = GetBit(name, child->refBit) ?
                    &pNode[child->idxRight] : &pNode[child->idxLeft];
            }
            // この時点で、parent と child の間にノードを挿入することが確定している。

            uint16_t idxCurrent = static_cast<uint16_t>(index);
            uint16_t idxChild = static_cast<uint16_t>(std::distance(pNode, child));

            if (GetBit(name, current.refBit))
            {
                // 当該ビットが1の場合は右にt(自身)をくっつける(current -> (current, child))。
                current.idxLeft = idxChild;
                current.idxRight = idxCurrent;
            }
            else
            {
                // 当該ビットが0の場合は左にt(自身)をくっつける(current -> (child, current))。
                current.idxLeft = idxCurrent;
                current.idxRight = idxChild;
            }

            // parent のどちらの子供にするか決める。
            if (GetBit(name, parent->refBit))
            {
                parent->idxRight = idxCurrent;
            }
            else
            {
                parent->idxLeft  = idxCurrent;
            }
        }
    }

    root.ofsName.set_ptr(NULL);
    empty->len = len;
    return BuildResult_Success;
}

//----------------------------------------------------------
PatriciaDictionary::Node*
PatriciaDictionary::FindNode(const char* str, size_t len) const NN_NOEXCEPT
{
#ifndef NN_ATK_EDIT_DIC_LINEAR
    const Node* pNode = this->node;
    const Node* parent = &pNode[0];
    const Node* child = &pNode[parent->idxLeft];

    while(parent->refBit > child->refBit)
    {
        parent = child;
        // ビットが1なら右、0なら左をたどる
        child = GetBit(str, len, child->refBit) ? &pNode[child->idxRight] : &pNode[child->idxLeft];
    }

    if (child->ofsName.offset && child->ofsName.GetResName()->Equals(str, len))
    {
        return const_cast<Node*>(child);
    }
    else
    {
        return NULL;
    }
#else
    for (auto idx = 1, end = GetDataCount(); idx <= end; ++idx)
    {
        const Node& pNode = this->node[idx];
        if (child->ofsName.offset && pNode.ofsName.GetResName()->Equals(str, len))
        {
            return const_cast<Node*>(&pNode);
        }
    }
    return NULL;
#endif // NN_ATK_EDIT_DIC_LINEAR
}

//----------------------------------------------------------
PatriciaDictionary::Node*
PatriciaDictionary::FindNode(const ResName* name) const NN_NOEXCEPT
{
#ifndef NN_ATK_EDIT_DIC_LINEAR
    const Node* pNode = this->node;
    const Node* parent = &pNode[0];
    const Node* child = &pNode[parent->idxLeft];

    while(parent->refBit > child->refBit)
    {
        parent = child;
        // ビットが1なら右、0なら左をたどる
        child = GetBit(name, child->refBit) ? &pNode[child->idxRight] : &pNode[child->idxLeft];
    }

    if (child->ofsName.offset && child->ofsName.GetResName()->Equals(name))
    {
        return const_cast<Node*>(child);
    }
    else
    {
        return NULL;
    }
#else
    for (auto idx = 1, end = GetDataCount(); idx <= end; ++idx)
    {
        const Node& pNode = this->node[idx];
        if (child->ofsName.offset && pNode.ofsName.GetResName()->Equals(name))
        {
            return const_cast<Node*>(&pNode);
        }
    }
    return NULL;
#endif // NN_ATK_EDIT_DIC_LINEAR
}

} // namespace nn::atk::viewer::detail
} // namespace nn::atk::viewer
} // namespace nn::atk
} // namespace nn

#endif // NN_ATK_CONFIG_ENABLE_DEV
